/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#if TARGET_API_Win32 || TARGET_API_Win32_Console
#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{

void _Init(CLIPFORMAT f, DWORD m, FORMATETC& fmtetc)
{
	fmtetc.cfFormat = f;
	fmtetc.ptd = 0;
	fmtetc.dwAspect = DVASPECT_CONTENT;
	fmtetc.lindex = -1;
	fmtetc.tymed = m; 
}
void _Init(STGMEDIUM& stgmed)
{
	stgmed.tymed = 0;    // will be init by the other side
	stgmed.hGlobal = 0;	 // union with pstm (so both set to 0 here)
	stgmed.pUnkForRelease = 0;
}

static bool _FillFatData_TEXT(HGLOBAL h,
							  DragDropType::FatData& fatData )
{
	if (h == NULL)
		return false;
	char* txt= (char*) ::GlobalLock(h); // stgmed.hGlobal
	if (txt == NULL)
		return false;
	String s(String::From_c_str(txt));
	::GlobalUnlock(h); // stgmed.hGlobal

	fatData.text_ASCII.push_back(s);
	return true;
}
static bool _FillFatData_HDROP(HDROP fdr,
							   DragDropType::FatData& fatData )
{
	// method 1
	if (fdr == NULL)
		return false;
	uint32 n = ::DragQueryFile(fdr, -1U, NULL, 0);
	for(uint32 k=0; k<n; ++k)
	{
		uint32 len = ::DragQueryFile(fdr, k, NULL, 0);
		if (len == 0)
			continue;
		String fname;
		fname.resize(len);
		::DragQueryFile(fdr, k, fname.begin(), len+1);
		// use fname
		fatData.fileNames.push_back(FileName::FromPathNameOS(fname));
	}

	/* method 2
		DROPFILES* dp = (DROPFILES*)::GlobalLock(stgmed.hGlobal);
		if (dp != NULL)
		{
			char* fnm = dp->pFiles + (char*) dp;
			if (!dp->fWide) // else it's a unicode string ...
			{
				for(;;)
				{
					String fname(String::From_c_str(fnm));
					if (fname.isEmpty())
						break;
					fatData.text_ASCII.push_back(fname);
					fnm += 1+fname.length();
				}
			}
			::GlobalUnlock(stgmed.hGlobal);
		}
	*/
	return n > 0;
}

static bool _FillFatData_HBITMAP(HBITMAP handle,
								 DragDropType::FatData& fatData )
{
	if (handle == NULL)
		return false;

	DIBSECTION dbSect; // now get the DIB section's information
	::GetObject(handle,sizeof(dbSect),&dbSect);

	sint32 dx = dbSect.dsBm.bmWidth;
	sint32 dy = dbSect.dsBm.bmHeight;	 if (dy<0) dy = -dy;
	OffscreenImage image(dx, dy, OffscreenImage::RGBA32);

	Graphics g1(image);
	Graphics g2;

    HBITMAP oldHandle = (HBITMAP)::SelectObject(g2.GetDC(), handle);
	if (0 != ::BitBlt(g1.GetDC(), 0,0, dx,dy, g2.GetDC(), 0,0,SRCCOPY))
		fatData.image = image; // store the image
	::SelectObject(g2.GetDC(), oldHandle);

	return fatData.image != 0;
}
static bool _FillFatData_DIB(HGLOBAL h,
							 DragDropType::FatData& fatData )
{
	if (h == NULL)
		return false;
	BITMAPINFO* bmp= (BITMAPINFO*) ::GlobalLock(h);
	if (bmp == NULL)
		return false;

	sint32 dx = bmp->bmiHeader.biWidth;
	sint32 dy = bmp->bmiHeader.biHeight;	 
	if (dy<0) dy = -dy;
	
	OffscreenImage image(dx, dy, OffscreenImage::RGBA32);
	
	{   // transfer the image
		Graphics g1(image);
		if (0 != ::SetDIBits(g1.GetDC(),image.data->handle, 
							0,dy, 
							(WORD)bmp->bmiHeader.biSize+(uint8*)bmp,
							bmp, DIB_RGB_COLORS))
		{
			fatData.image = image; // store the image
		}
	}				
	
	::GlobalUnlock(h);
	return fatData.image != 0;
}
static HGLOBAL _StoreFatDataTo_DIB(DragDropType::FatData& fatData)
{
	if (fatData.image == 0)
		return NULL;
	
	Graphics g1(fatData.image);

	// load the info for the bitmap
	BITMAPINFO bmp;
	bmp.bmiHeader.biSize = sizeof(bmp.bmiHeader);
	bmp.bmiHeader.biSizeImage = 0; // paranoid ?!
	bmp.bmiHeader.biBitCount = 0; // need to let them fill this
	if (0 == ::GetDIBits(g1.GetDC(), fatData.image.data->handle,
							0, 0, NULL, &bmp, DIB_RGB_COLORS ))
		return NULL;
    // it's the size of the image without the header
	// biSizeImage = ((((biWidth * biBitCount) +31) &~31) >>3) *biHeight;

	// ToDo? bugfix ...
    HGLOBAL hGLOBAL = ::GlobalAlloc(GMEM_MOVEABLE, 
    			3*sizeof(RGBQUAD)+ bmp.bmiHeader.biSize + bmp.bmiHeader.biSizeImage);
    if (hGLOBAL == NULL)
    	return NULL;					
	BITMAPINFO* bmpBuffer= (BITMAPINFO*)::GlobalLock(hGLOBAL);
		//(BITMAPINFO*)::GlobalLock(hGLOBAL);
	if (bmpBuffer == NULL)
	{
		::GlobalFree(hGLOBAL);
		return NULL;
	}
	// copy the structure
	bmpBuffer->bmiHeader = bmp.bmiHeader;
	int lines = bmp.bmiHeader.biHeight;
	if (lines < 0) lines= -lines;
	// load the content of the bitmap
	// ToDo? bugfix ...
	uint8* bits = (3*sizeof(RGBQUAD)+(WORD)bmp.bmiHeader.biSize)+ (uint8*)bmpBuffer;
	if (0 == ::GetDIBits(g1.GetDC(), fatData.image.data->handle,
						0, (WORD)lines, bits, (BITMAPINFO*)bmpBuffer, DIB_RGB_COLORS ))
	{
		::GlobalFree(hGLOBAL);
		return NULL;
	}
	// ToDo? bugfix ...
	bmpBuffer->bmiHeader.biSize += 3*sizeof(RGBQUAD);
 	::GlobalUnlock(hGLOBAL);
	return hGLOBAL; 
}

static HBITMAP _StoreFatDataTo_HBITMAP(DragDropType::FatData& fatData)
{
	if (fatData.image == 0) 
		return NULL;
	Graphics g1(fatData.image);
	Size2D z(fatData.image.GetSize());
	HBITMAP hBITMAP = ::CreateCompatibleBitmap(g1.GetDC(),z.w, z.h); 
    {
		Graphics g2;
	    HBITMAP oldHandle = (HBITMAP)::SelectObject(g2.GetDC(), hBITMAP);
		::BitBlt(g2.GetDC(), 0,0, z.w,z.h, g1.GetDC(), 0,0,SRCCOPY);
		::SelectObject(g2.GetDC(), oldHandle);
	}
	return hBITMAP;
}

static uint32  _StoreFatDataTo_Text_Size(DragDropType::FatData& fatData)
{
	// join all the textpieces into one big string, separator is \r\n
	uint32 z = 0;
	std::vector<String>::iterator b = fatData.text_ASCII.begin();
	std::vector<String>::iterator e = fatData.text_ASCII.end();
	if (b == e)
		return 0;
	std::vector<String>::iterator p;

	for(p=b; p!=e; ++p)
	{
		z += p->length(); // for the text in the piece
		z += std::count(p->begin(), p->end(), '\n'); // for each LF we add a CR
	}
	z += 2*(e-b); // for the CRLF between the pieces
	z += 1; // for the ASCIIZ at the end
	return z;
}
static HGLOBAL _StoreFatDataTo_Text_Plain(DragDropType::FatData& fatData, HGLOBAL hTEXT)
{
	if (hTEXT == NULL)
		return NULL;
	std::vector<String>::iterator b = fatData.text_ASCII.begin();
	std::vector<String>::iterator e = fatData.text_ASCII.end();
	if (b == e)
		return NULL;

	char* txt= (char*) ::GlobalLock(hTEXT); 
	// fill the content
	bool havesep = true;
	std::vector<String>::iterator p;
	for(p=b; p!=e; ++p)
	{
		char* sb = p->begin();
		char* se = p->end();
		char* st;
		for(st=sb; st!=se; )
		{
			char *sn = std::find(st, se, '\n');
			txt = std::copy(st, sn, txt);
			if (sn != se)
			{
				*txt ++ = '\r';
				*txt ++ = '\n';
				st = sn+1;
				havesep = true;
			}
			else
			{
				st = sn;
				havesep = false;
			}
		}
		if ((p+1 != e) && (!havesep))
		{
			*txt ++ = '\r';
			*txt ++ = '\n';
			havesep = true;
		}
	}				  
	*txt ++ = 0;

	::GlobalUnlock(hTEXT);	
	return hTEXT; 
}
static HGLOBAL _StoreFatDataTo_Text_Plain(DragDropType::FatData& fatData)
{
	uint32 z = _StoreFatDataTo_Text_Size(fatData);
	HGLOBAL hTEXT = ::GlobalAlloc(GMEM_MOVEABLE, z);
    return _StoreFatDataTo_Text_Plain(fatData, hTEXT);
}

static HGLOBAL _StoreFatDataTo_HDROP(DragDropType::FatData& fatData, HGLOBAL h)
{
	std::vector<FileName>::iterator b = fatData.fileNames.begin();
	std::vector<FileName>::iterator e = fatData.fileNames.end();
	if (b == e)
		return NULL;

	// compute needed size
	uint32 z = sizeof(DROPFILES);
	std::vector<FileName>::iterator p;
	for(p=b; p!=e; ++p)
	{						 
		String s(p->GetFullPathNameOS());
		if (s.isEmpty()) // to avoid false double ASCIIZ inside the sequence
			continue;
		z += 1+s.length(); // for the text in the piece + ASCIIZ
	}
	++z; // one more final asciiZ
	
	HGLOBAL hDROP = 0;
	if (h != 0)
	{
		uint32 za = ::GlobalSize(h);
		if (z > za)
			return NULL;
		hDROP = h;
	}
	else
	{
		hDROP = ::GlobalAlloc(GMEM_MOVEABLE, z);
	}
	
	// now fill the space
	DROPFILES* dp = (DROPFILES*)::GlobalLock(hDROP);
	if (dp == NULL)
	{
		if (h == 0)
			::GlobalFree(hDROP);
		return NULL;
	}
	// fill the header
	dp->pFiles = sizeof(*dp);
   	dp->pt.x = 0;
   	dp->pt.y = 0;
    dp->fNC  = 0;
	dp->fWide= 0; // ASCII , not unicode
	// fill the filenames
	char* fnm = dp->pFiles + (char*) dp;
	for(p=b; p!=e; ++p)
	{
		String s(p->GetFullPathNameOS());
		if (s.isEmpty()) // to avoid false double ASCIIZ inside the sequence
			continue;
		fnm = std::copy(s.begin(),s.end(),fnm);
		*fnm++ = 0;
	}
	*fnm++ = 0; // final ASCIIZ
	
	::GlobalUnlock(hDROP);
	return hDROP;
}

#if 0
#pragma mark -
#pragma mark DropTargetWin32
#endif


DropTargetWin32::DropTargetWin32(Window::_Data* wd)
: refcount(1) // we assume we are already owned by wData ( no need to ask it to AddRef)
, wData(wd)
{
	ASSERT(wd != 0);
}
DropTargetWin32::~DropTargetWin32()
{
}

HRESULT STDMETHODCALLTYPE DropTargetWin32::QueryInterface( 
    REFIID riid,
    void __RPC_FAR *__RPC_FAR *ppvObject)
{
	if (riid == IID_IUnknown)
	{
		*ppvObject = (void*)(IUnknown*) this;
		++refcount; // we add the refcount for the requestor
		return NOERROR;
	}
	if (riid == IID_IDropTarget)
	{
		*ppvObject = (void*)(IDropTarget*) this;
		++refcount; // we add the refcount for the requestor
		return NOERROR;
	}
#ifdef DEBUG
	UnitTest::Log("!!! DropTargetWin32::QueryInterface unknown interface");
#endif
	// we do not support any other interface
	*ppvObject = NULL;
	return E_NOINTERFACE;
}

ULONG STDMETHODCALLTYPE DropTargetWin32::AddRef()
{
	return ++refcount;
}

ULONG STDMETHODCALLTYPE DropTargetWin32::Release()
{
	uint32 r = --refcount;
	if (r == 0)
		delete this;
	return r;
}


HRESULT STDMETHODCALLTYPE DropTargetWin32::DragEnter( 
    /* [unique][in] */ IDataObject __RPC_FAR *pDataObj,
    /* [in] */ DWORD grfKeyState,
    /* [in] */ POINTL screenPoint,
    /* [out][in] */ DWORD __RPC_FAR *pdwEffect)
{
	if (pDataObj == NULL) return E_INVALIDARG;

	#if 0
		if (pDataObj != 0)
			ScanData_Test(*pDataObj);
	#endif

	// determine and remember the drop types available from the pDataObj
	try	{ DetermineWhatSourceCanDrop(*pDataObj, whatSourceCanDrop); }
    catch(...)  { return E_UNEXPECTED; }

	// for the rest it's identical to the DragOver case 
	return DragOver(grfKeyState, screenPoint, pdwEffect);
}
    
HRESULT STDMETHODCALLTYPE DropTargetWin32::DragOver( 
    /* [in] */ DWORD grfKeyState,
    /* [in] */ POINTL screenPoint,
    /* [out][in] */ DWORD __RPC_FAR *pdwEffect)
{
	// we simulate a drop, but not for real, allowed effects will be changed
	// to tell us what would have happened
	return Drop(0, grfKeyState, screenPoint, pdwEffect);
}
    
HRESULT STDMETHODCALLTYPE DropTargetWin32::DragLeave( )
{
	whatSourceCanDrop.clear();
	return S_OK;
}
    
HRESULT STDMETHODCALLTYPE DropTargetWin32::Drop( 
    /* [unique][in] */ IDataObject __RPC_FAR *pDataObj,
    /* [in] */ DWORD grfKeyState,
    /* [in] */ POINTL screenPoint,
    /* [out][in] */ DWORD __RPC_FAR *pdwEffect)
{
	try
	{
    	DWORD allowedEffects = *pdwEffect 
    			& ( DROPEFFECT_MOVE|DROPEFFECT_COPY|DROPEFFECT_LINK);

    	// Further trim the allowed effects according to the key modifiers 
    	// if (grfKeyState & MK_LBUTTON)  the key state is only set during enter,over
    	// but not during drop ... ?!
    	{	
	    	if (grfKeyState & MK_SHIFT)
	    		allowedEffects &= DROPEFFECT_MOVE;
	    	else if (grfKeyState & MK_CONTROL)
	    		allowedEffects &= DROPEFFECT_COPY;
	    	else if (grfKeyState & MK_ALT)
	    		allowedEffects &= DROPEFFECT_LINK;
		}    		

    	if ((allowedEffects != DROPEFFECT_NONE) && (pDataObj != 0))
	    	DetermineWhatSourceCanDrop(*pDataObj, whatSourceCanDrop);

		// we do a real or a simulated drop (depending on pDataObj != 0), 
		// allowed effects will be changed to tell us what happened
    	// pDataObj->AddRef(); // to be safe
    	if (allowedEffects != DROPEFFECT_NONE)
    		DoTheDrop(pDataObj, whatSourceCanDrop, allowedEffects, &screenPoint); 
    	// pDataObj->Release(); // we're done with this obj
    	
    	#if DEBUG
    		if (pDataObj != 0)
				ScanData_Test(*pDataObj);
		#endif

		// tell the system how we really feel about this drop situation
		*pdwEffect = allowedEffects;

		return S_OK;
    }
    catch(...)
    {
    	return E_UNEXPECTED;
    }
}

void DropTargetWin32::DetermineWhatSourceCanDrop(IDataObject &cargo, 
   										   		 DragDropType::TypeSet& whatDrop)
{
	whatDrop.clear();
	
	FORMATETC fmtetc;
	IEnumFORMATETC* enumerator = NULL;
	HRESULT hr = cargo.EnumFormatEtc(DATADIR_GET , &enumerator);
	if (hr != S_OK)
	{
#if DEBUG
	UnitTest::Log("DropTargetWin32: calling EnumFormatEtc failed");
#endif		
		// they do not provide a format enumerator , we have to assume they 
		// can provide everything
		whatDrop.insert(&DragDropType::Image_Native);
		whatDrop.insert(&DragDropType::FileName);
		whatDrop.insert(&DragDropType::Text_Plain);
		return;
	}
	if (enumerator == NULL)
		return;

	for(;;)
	{
		fmtetc.ptd = NULL; // make sure it's init to null
		if (S_OK != enumerator->Next(1,&fmtetc,NULL))
			break;
		if (fmtetc.ptd != NULL) // we must clean this
			::CoTaskMemFree (fmtetc.ptd);
		fmtetc.ptd = NULL; 
		
		DragDropType::TypeID dropType = 0;

		switch(fmtetc.cfFormat)
		{
		case CF_DIB: 
		case CF_BITMAP: 
			dropType = &DragDropType::Image_Native; 
			break;
		case CF_HDROP:  
			dropType = &DragDropType::FileName; 
			break;
		case CF_TEXT:   
			dropType = &DragDropType::Text_Plain; 
			break;
		default:
			break;
		}; // case
		if (dropType != 0)
			whatDrop.insert(dropType);
	} //for

	enumerator->Release();
	enumerator = NULL;
} //  DetermineWhatSourceCanDrop

void DropTargetWin32::DoTheDrop(
			   	IDataObject 		  *cargo,
				DragDropType::TypeSet& availableTypes, 
			   	DragDropType::Effect&  effect,
			   	POINTL*		 		   screenPoint )
{
	if (availableTypes.empty() || (wData == 0))
	{
    	effect = 0;
    	return;
	}
	bool simulate = (cargo == 0);
    DragDropType::FatData 	fatData;
	
	if (!simulate)
	{ // we actually have to fill the fat data
		DragDropType::TypeSet orig;
		orig.swap(availableTypes);
		DragDropType::TypeSet::iterator b = orig.begin(); 
		DragDropType::TypeSet::iterator e = orig.end(); 
		for( ; b!=e; ++b)
		{
			if (FillFatData(*b, fatData, *cargo))
				availableTypes.insert(*b);// only keep the types that actually worked
		}
	}
	if (availableTypes.empty())
	{
    	effect = 0;
    	return;
	}
	wData->WindowDropEvent(availableTypes, effect, screenPoint, simulate, fatData);	
}

bool DropTargetWin32::FillFatData(DragDropType::TypeID tid, 
								  DragDropType::FatData& fatData,
								  IDataObject& cargo )
{
	if (tid == &DragDropType::Text_Plain)
		return FillFatData_Text_Plain(cargo, fatData);
	else if (tid == &DragDropType::FileName)
		return FillFatData_FileName(cargo, fatData);
	else if (tid == &DragDropType::Image_Native)
		return FillFatData_Image_Native(cargo, fatData);
	return false;
}

bool DropTargetWin32::FillFatData_Text_Plain(IDataObject& cargo,
								  DragDropType::FatData& fatData )
{
	FORMATETC fmtetc;
	STGMEDIUM stgmed;

	_Init(CF_TEXT, TYMED_HGLOBAL | TYMED_ISTREAM, fmtetc);
	_Init(stgmed);
	if (SUCCEEDED(cargo.GetData(&fmtetc,&stgmed)))
	{
		if (stgmed.tymed == TYMED_ISTREAM)
		{
			String s;
			s.reserve(64000);
			for(;;)
			{
				char buf[1024];
				uint32 len = 0;
				HRESULT hr = stgmed.pstm->Read(buf, sizeof(buf), &len);
				if (hr != S_OK)
					break;
				if (len == 0)
					break;
				if (buf[len-1] == 0)
					--len;
				s.append(String(buf, buf+len));
			}
			s.compact();
			fatData.text_ASCII.push_back(s);
		}
		else if (stgmed.tymed == TYMED_HGLOBAL)
		{
			_FillFatData_TEXT(stgmed.hGlobal, fatData);
		}
		::ReleaseStgMedium(&stgmed);
		return !fatData.text_ASCII.empty();
	}

	_Init(CF_OEMTEXT, TYMED_HGLOBAL, fmtetc);
	_Init(stgmed); 
	if (SUCCEEDED(cargo.GetData(&fmtetc,&stgmed)))
	{
		_FillFatData_TEXT(stgmed.hGlobal,fatData);
		::ReleaseStgMedium(&stgmed);
		return !fatData.text_ASCII.empty();
	}
	return false;
}

bool DropTargetWin32::FillFatData_FileName(IDataObject& cargo,
								  DragDropType::FatData& fatData )
{
	FORMATETC fmtetc;
	STGMEDIUM stgmed;

	_Init(CF_HDROP, TYMED_HGLOBAL, fmtetc); _Init(stgmed); // files
	if (SUCCEEDED(cargo.GetData(&fmtetc,&stgmed)))
	{ // DROPFILES structure followed by a double null-terminated list of file names
		bool r = false;
		HDROP fdr= (HDROP) ::GlobalLock(stgmed.hGlobal);
		if (fdr != NULL)
		{
			r = _FillFatData_HDROP(fdr,fatData);
			::GlobalUnlock(stgmed.hGlobal);
		}
		::ReleaseStgMedium(&stgmed);
		return r;
	}
	return false;
}

bool DropTargetWin32::FillFatData_Image_Native(IDataObject& cargo,
								  DragDropType::FatData& fatData )
{
	FORMATETC fmtetc;
	STGMEDIUM stgmed;
	_Init(CF_BITMAP, TYMED_GDI, fmtetc); _Init(stgmed); 
	if (SUCCEEDED(cargo.GetData(&fmtetc,&stgmed)))
	{
		bool r1 = _FillFatData_HBITMAP(stgmed.hBitmap, fatData);
		::ReleaseStgMedium(&stgmed);
		if (r1)
			return true;
	}
	_Init(CF_DIB, TYMED_HGLOBAL, fmtetc); _Init(stgmed);
	if (SUCCEEDED(cargo.GetData(&fmtetc,&stgmed)))
	{ // BITMAPINFO structure followed by the bitmap bits
		bool r2 = _FillFatData_DIB(stgmed.hGlobal, fatData);
		::ReleaseStgMedium(&stgmed);
		if (r2)
			return true;
	}
	return false;
}



#if DEBUG
void DropTargetWin32::ScanData_Test(IDataObject& cargo)
{
	FORMATETC fmtetc;
	char fmtName[256];

	IEnumFORMATETC* enumerator = NULL;
	if (S_OK != cargo.EnumFormatEtc(DATADIR_GET , &enumerator))
		return;
	if (enumerator == NULL)
		return;
	for(;;)
	{
		fmtetc.ptd = NULL; // make sure it's init to null
		if (S_OK != enumerator->Next(1,&fmtetc,NULL))
			break;
		if (fmtetc.ptd != NULL) // we must clean this
			::CoTaskMemFree (fmtetc.ptd);
		fmtetc.ptd = NULL; 
		if (fmtetc.cfFormat > CF_MAX)
		{   // format is out of predefined bounds, may be a dynamic registered format
			int len = ::GetClipboardFormatName(fmtetc.cfFormat, fmtName, sizeof(fmtName));
			if (len >= sizeof(fmtName))
				len = sizeof(fmtName)-1;
			fmtName[len] = 0;			  // safety first
			// ToDo: based on the string name of what can be dropped, determine the association
			//       if we ever plan to support fancy drop types or we make our own
			UnitTest::Log("Drop medium=%d fmt='%s'(%d)", fmtetc.tymed, fmtName, fmtetc.cfFormat);
			if (0 != (fmtetc.tymed & (TYMED_HGLOBAL | TYMED_ISTREAM | TYMED_FILE | TYMED_ISTORAGE)))
			{   // I can read this
				if (0 == ::strcmp(fmtName, "Rich Text Format"))
					FillData_Test(fmtetc.cfFormat, cargo);
				else if (0 == ::strcmp(fmtName, "Internet Message (rfc822/rfc1522)"))
					FillData_Test(fmtetc.cfFormat, cargo);
				else if (0 == ::strcmp(fmtName, "Outlook Express Folder"))
					FillData_Test(fmtetc.cfFormat, cargo);
				else if (0 == ::strcmp(fmtName, CFSTR_FILECONTENTS))
					FillData_Test(fmtetc.cfFormat, cargo);// FILEGROUPDESCRIPTOR* <- GlobalLock 
			}
		}
		else
		{
			fmtName[0] = 0;
			UnitTest::Log("Drop medium=%d fmt=%d", fmtetc.tymed, fmtetc.cfFormat);
		}
    }
    enumerator->Release();
    enumerator = NULL;
}

void DropTargetWin32::FillData_Test(CLIPFORMAT fmt, IDataObject& cargo)
{
	FORMATETC fmtetc;
	STGMEDIUM stgmed;
	_Init(fmt, TYMED_ISTREAM | TYMED_HGLOBAL | TYMED_FILE | TYMED_ISTORAGE, fmtetc);
	_Init(stgmed); 
	if (SUCCEEDED(cargo.GetData(&fmtetc,&stgmed)))
	{
		if (stgmed.tymed == TYMED_ISTREAM)
		{   // read from IStream stgmed.pstm
			uint32 z = 0;
			for(;;)
			{
				char bufr[800];
				uint32 len = 0;
				HRESULT hr = stgmed.pstm->Read(bufr, sizeof(bufr)-1, &len);
				if (hr != S_OK)
					break;
				if (len == 0)
					break;
				z+=len;
				// just so we can print this
				bufr[len] = 0;
				UnitTest::Log(bufr);
			}
			UnitTest::Log("================= Stream EOF reached (%d bytes)=========",z);
		}
		else if (stgmed.tymed == TYMED_HGLOBAL)
		{
			char* txt= (char*) ::GlobalLock(stgmed.hGlobal); 
			SIZE_T z = ::GlobalSize(stgmed.hGlobal);
			if (txt != NULL)
			{
				char bufr[800];
				::strncpy(bufr, txt, sizeof(bufr));
				bufr[sizeof(bufr)-1] = 0;
				UnitTest::Log(bufr);
				::GlobalUnlock(stgmed.hGlobal); // stgmed.hGlobal
			}
			UnitTest::Log("================= HGLOBAL Quote complete (%d bytes)=====",z);
		}
		else if (stgmed.tymed == TYMED_FILE)
		{   // LPOLESTR unicode filename stgmed.lpszFileName	
			UnitTest::Log("The drop is in '%S' (unicode filename)", stgmed.lpszFileName);
		}
		else if (stgmed.tymed == TYMED_ISTORAGE)
		{   // read from IStorage stgmed.pstg
			// for compound object streams (like a filesystem within a file)
			UnitTest::Log("The drop is in an IStorage interface");
			IEnumSTATSTG* storEnum = 0;
			if (S_OK == stgmed.pstg->EnumElements(0,0,0, &storEnum))
			{
				STATSTG storEntry;
				for(;;)
				{
					if (storEnum == NULL) break;
					if (S_OK != storEnum->Next(1,&storEntry,NULL))
						break;
					if (storEntry.pwcsName == 0)
						break;
					UnitTest::Log("\tItem='%S' size=%d", storEntry.pwcsName, storEntry.cbSize);
					::CoTaskMemFree(storEntry.pwcsName); // our responsibility
				}
			}
		}
		::ReleaseStgMedium(&stgmed);
		return;
	}
} //fill Test

#endif  // DEBUG


bool Window::PasteFromClipboard(DragDropType::TypeSet& availableTypes, 
							    bool 				   onlySimulate,
							    DragDropType::FatData& fatData )
{
	if (0 == ::OpenClipboard(data->hwnd))
		return false;
	availableTypes.clear();
	CLIPFORMAT fmt = 0;
	for (;;)
	{
		fmt = ::EnumClipboardFormats(fmt);
		if (fmt == 0)
			break;
		DragDropType::TypeID dropType = 0;
		switch(fmt)
		{
		case CF_BITMAP: 
			dropType = &DragDropType::Image_Native; 
			if (!onlySimulate && (fatData.image == 0))
			{
				HANDLE h = ::GetClipboardData(CF_BITMAP);
				if (!_FillFatData_HBITMAP((HBITMAP)h, fatData))
					dropType = 0;
			}
			break;
		case CF_DIB: 
			dropType = &DragDropType::Image_Native; 
			if (!onlySimulate && (fatData.image == 0))
			{
				HANDLE h = ::GetClipboardData(CF_DIB);
				if (!_FillFatData_DIB((HGLOBAL)h, fatData))
					dropType = 0;
			}
			break;
		case CF_HDROP:  
			dropType = &DragDropType::FileName; 
			if (!onlySimulate)
			{
				HANDLE h = ::GetClipboardData(CF_HDROP);
				if (!_FillFatData_HDROP((HDROP)h, fatData))
					dropType = 0;
			}
			break;
		case CF_TEXT:   
			dropType = &DragDropType::Text_Plain; 
			if (!onlySimulate)
			{
				HANDLE h = ::GetClipboardData(CF_TEXT);
				if (!_FillFatData_TEXT((HGLOBAL)h, fatData))
					dropType = 0;
			}
			break;
		}
		if (dropType == 0)
			continue;
		availableTypes.insert(dropType);
	}
	::CloseClipboard();
	return !availableTypes.empty();
}



bool Window::CopyToClipboard(DragDropType::TypeSet& availableTypes, 
						     DragDropType::FatData& fatData )
{
	if (availableTypes.empty())
		return false;
	if (0 == ::OpenClipboard(data->hwnd))
		return false;
	if (0 == ::EmptyClipboard())
		return false; // only what we copy now is to be pasted
	
	DragDropType::TypeSet::iterator e=availableTypes.end();
    bool success= false;

	HGLOBAL hDIB = NULL;
	HBITMAP hBITMAP = NULL;
	if (  (availableTypes.find(&DragDropType::Image_Native) != e) )
	{
		hDIB = _StoreFatDataTo_DIB(fatData);
		if (hDIB != NULL)
			success |= (NULL != ::SetClipboardData(CF_DIB, hDIB));

		// HBITMAP is good to copy for my own process but it won't 
		// cross process boundaries
		hBITMAP = _StoreFatDataTo_HBITMAP(fatData);
		if (hBITMAP != NULL)
			success |= (NULL != ::SetClipboardData(CF_BITMAP, hBITMAP));
	}	

	HGLOBAL hTEXT = NULL;
	if (  (availableTypes.find(&DragDropType::Text_Plain) != e) )
	{  
		hTEXT = _StoreFatDataTo_Text_Plain(fatData);
		if (hTEXT != NULL)
			success |= (NULL != ::SetClipboardData(CF_TEXT, hTEXT));
	}

	::CloseClipboard();
// 	DO NOT DELETE the handle if(hDIB != NULL) ::GlobalFree(hDIB); // no longer needed ?!
//	DO NOT DELETE            if(hTEXT != NULL) ::GlobalFree(hTEXT); // no longer needed ?!
	return success;
}


// Transferring Shell Objects with Drag-and-Drop and the Clipboard
// Dragging and Dropping Shell Objects Asynchronously => IASyncOperation
// Creating Context Menu Handlers
// struct FILEDESCRIPTOR
// CF_FILEDESCRIPTOR  ::RegisterClipboardFormat(CFSTR_FILEDESCRIPTOR)
// CF_DROPLOCATION ::RegisterClipboardFormat(CFSTR_FILECONTENTS)
	




#if 0
#pragma mark -
#pragma mark DragSourceWin32
#endif
class DragSourceFormatEnum;
class DragSourceDataObject;

class DragSourceWin32 : public IDropSource  
{
private:
	uint32 refcount;      // ole refcount 
public:
	DragSourceDataObject*  dataObject;
	/*
		maybe this vector should be allocated with 
		  if (FAILED(CoGetMalloc(MEMCTX_TASK, &pIMalloc)))
		    return NULL;
		  pFmt = (LPFORMATETC)pIMalloc->Alloc(size);
		  pIMalloc->Release();
		  
		or with
		  ::CoTaskMemAlloc(size), 
		  ::CoTaskMemFree(p)
	*/
	std::vector<FORMATETC> availableFormats;  		  // win32 format for transfer
	std::vector<DragDropType::TypeID> availableTypes; // our own logical types
	DragDropType::FatData  fatData;
public:
	DragSourceWin32();
	~DragSourceWin32();

	void DoDND(DragDropType::Effect& effect);
	void Set( DragDropType::FatData&  fatData, 
			  DragDropType::TypeSet& typeset);

	bool StoreFatData(uint32 typeIndex, STGMEDIUM& medium);

	bool StoreFatData_Text_Plain_IStream_Push(STGMEDIUM& medium);
	bool StoreFatData_Text_Plain_IStream_Pull(STGMEDIUM& medium);
	bool StoreFatData_Text_Plain_HGLOBAL(STGMEDIUM& medium);
	bool StoreFatData_FileName( STGMEDIUM& medium);
	bool StoreFatData_DIB( STGMEDIUM& medium );
	bool StoreFatData_HBITMAP( STGMEDIUM& medium );

public: // IUnknown
    virtual HRESULT STDMETHODCALLTYPE QueryInterface( 
        REFIID riid,
        void __RPC_FAR *__RPC_FAR *ppvObject);
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();

public: // IDropSource
    virtual HRESULT STDMETHODCALLTYPE QueryContinueDrag( 
        BOOL fEscapePressed,
        DWORD grfKeyState);
    virtual HRESULT STDMETHODCALLTYPE GiveFeedback( 
       DWORD dwEffect);
}; // class DragSourceWin32


#if 0
#pragma mark DragSourceFormatEnum
#endif

class DragSourceFormatEnum : public IEnumFORMATETC
{
private:
	uint32 refcount;      // ole refcount 
	DragSourceWin32* dragSource;
	bool		forWriting; 
	uint32		currentPos;
public:
	DragSourceFormatEnum(DragSourceWin32* ds, bool forWriting);
	~DragSourceFormatEnum();
	
public: // IUnknown
    virtual HRESULT STDMETHODCALLTYPE QueryInterface( 
        REFIID riid,
        void __RPC_FAR *__RPC_FAR *ppvObject);
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();
public: // IEnumFORMATETC
    virtual /* [local] */ HRESULT STDMETHODCALLTYPE Next( 
        /* [in] */ ULONG celt,
        /* [length_is][size_is][out] */ FORMATETC __RPC_FAR *rgelt,
        /* [out] */ ULONG __RPC_FAR *pceltFetched);
    
    virtual HRESULT STDMETHODCALLTYPE Skip( 
        /* [in] */ ULONG celt);
    
    virtual HRESULT STDMETHODCALLTYPE Reset( void);
    
    virtual HRESULT STDMETHODCALLTYPE Clone( 
        /* [out] */ IEnumFORMATETC __RPC_FAR *__RPC_FAR *ppenum);
};

#if 0
#pragma mark DragSourceDataObject
#endif

class DragSourceDataObject : public IDataObject
{
private:
	DragSourceWin32* dragSource;
public:
	DragSourceDataObject(DragSourceWin32* ds);
	~DragSourceDataObject();
	
public: // IUnknown
    virtual HRESULT STDMETHODCALLTYPE QueryInterface( 
        REFIID riid,
        void __RPC_FAR *__RPC_FAR *ppvObject);
    virtual ULONG STDMETHODCALLTYPE AddRef();
    virtual ULONG STDMETHODCALLTYPE Release();

public: // IDataObject
    virtual /* [local] */ HRESULT STDMETHODCALLTYPE GetData( 
        /* [unique][in] */ FORMATETC __RPC_FAR *pformatetcIn,
        /* [out] */ STGMEDIUM __RPC_FAR *pmedium);
    virtual /* [local] */ HRESULT STDMETHODCALLTYPE GetDataHere( 
        /* [unique][in] */ FORMATETC __RPC_FAR *pformatetc,
        /* [out][in] */ STGMEDIUM __RPC_FAR *pmedium);
    virtual HRESULT STDMETHODCALLTYPE QueryGetData( 
        /* [unique][in] */ FORMATETC __RPC_FAR *pformatetc);
    virtual HRESULT STDMETHODCALLTYPE GetCanonicalFormatEtc( 
        /* [unique][in] */ FORMATETC __RPC_FAR *pformatectIn,
        /* [out] */ FORMATETC __RPC_FAR *pformatetcOut);
    virtual /* [local] */ HRESULT STDMETHODCALLTYPE SetData( 
        /* [unique][in] */ FORMATETC __RPC_FAR *pformatetc,
        /* [unique][in] */ STGMEDIUM __RPC_FAR *pmedium,
        /* [in] */ BOOL fRelease);
    virtual HRESULT STDMETHODCALLTYPE EnumFormatEtc( 
        /* [in] */ DWORD dwDirection,
        /* [out] */ IEnumFORMATETC __RPC_FAR *__RPC_FAR *ppenumFormatEtc);
    virtual HRESULT STDMETHODCALLTYPE DAdvise( 
        /* [in] */ FORMATETC __RPC_FAR *pformatetc,
        /* [in] */ DWORD advf,
        /* [unique][in] */ IAdviseSink __RPC_FAR *pAdvSink,
        /* [out] */ DWORD __RPC_FAR *pdwConnection);
    virtual HRESULT STDMETHODCALLTYPE DUnadvise( 
        /* [in] */ DWORD dwConnection);
    virtual HRESULT STDMETHODCALLTYPE EnumDAdvise( 
        /* [out] */ IEnumSTATDATA __RPC_FAR *__RPC_FAR *ppenumAdvise);
}; // class DataObject	



DragSource::DragSource()
: _data(0)
{
	_data = new DragSourceWin32();
	_data->AddRef();
}
DragSource::~DragSource()
{
	_data->Release();
	_data = 0;
}
void DragSource::DoDND(DragDropType::Effect& effect)
{
	_data->DoDND(effect);
}
void DragSource::Set( DragDropType::FatData&  fatData, 
					  DragDropType::TypeSet& typeset)
{
	_data->Set(fatData, typeset);
}


DragSourceWin32::DragSourceWin32()
: refcount(0) 
, dataObject(0)
{
#if DEBUG
UnitTest::Log("Created %08X DragSourceWin32",this);
#endif
	dataObject = new DragSourceDataObject(this);
}
DragSourceWin32::~DragSourceWin32()
{
	delete dataObject;
	dataObject = 0;
#if DEBUG
UnitTest::Log("Deleted %08X DragSourceWin32",this);
#endif
}

HRESULT STDMETHODCALLTYPE DragSourceWin32::QueryInterface( 
    REFIID riid,
    void __RPC_FAR *__RPC_FAR *ppvObject)
{
	if (riid == IID_IUnknown)
	{
		*ppvObject = (void*)(IUnknown*) this;
		++refcount; // we add the refcount for the requestor
		return NOERROR;
	}
	if (riid == IID_IDropSource)
	{
		*ppvObject = (void*)(IDropSource*) this;
		++refcount; // we add the refcount for the requestor
		return NOERROR;
	}
	if (riid == IID_IDataObject)
	{
		*ppvObject = (void*)(IDataObject*) this;
		++refcount; // we add the refcount for the requestor
		return NOERROR;
	}
#ifdef DEBUG
	UnitTest::Log("!!! DragSourceWin32::QueryInterface unknown interface");
#endif
	// we do not support any other interface
	*ppvObject = NULL;
	return E_NOINTERFACE;
}

ULONG STDMETHODCALLTYPE DragSourceWin32::AddRef()
{
	return ++refcount;
}

ULONG STDMETHODCALLTYPE DragSourceWin32::Release()
{
	uint32 r = --refcount;
	if (r == 0)
		delete this;
	return r;
}

HRESULT STDMETHODCALLTYPE DragSourceWin32::QueryContinueDrag( 
    BOOL fEscapePressed,
    DWORD grfKeyState)
{
	if (fEscapePressed)
		return DRAGDROP_S_CANCEL;
	if ((grfKeyState & (MK_LBUTTON | MK_RBUTTON)) == (MK_LBUTTON | MK_RBUTTON))
		return DRAGDROP_S_CANCEL;
	if ((grfKeyState & (MK_LBUTTON | MK_RBUTTON)) == 0)
		return DRAGDROP_S_DROP;
	return S_OK; // continue DND
}

HRESULT STDMETHODCALLTYPE DragSourceWin32::GiveFeedback( 
   DWORD /*dwEffect*/)
{
	return DRAGDROP_S_USEDEFAULTCURSORS;
}

void DragSourceWin32::DoDND(DragDropType::Effect& availEffect)
{
	IDataObject* cargo  =  dataObject; 
	IDropSource* source =  this;
	AddRef(); // grab this and hold on to it
	try
	{
		DWORD effect = 0;
		// this is a stupid SYNCHRONOUS, BLOCKING call, 
		// if you have some networking agents working in the GUI thread 
		// it is perfectly possible for them to time out during this operation
		HRESULT hr = ::DoDragDrop(cargo, source, availEffect, &effect );
		if (hr == DRAGDROP_S_DROP)
		{
			availEffect = effect;
		}
		else if (hr == DRAGDROP_S_CANCEL)
		{   // operation was canceled
			availEffect = 0;
		}
		else
		{	// error during dnd
			availEffect = 0;
		}
	}
	catch(...)
	{
		availEffect = 0;
	}
	Release();
}

void DragSourceWin32::Set( DragDropType::FatData& fatData, 
					  	   DragDropType::TypeSet& typeset )
{
	FORMATETC fmt;
	DragDropType::TypeSet::iterator e = typeset.end();
    DragDropType::TypeSet::iterator w;

	w = typeset.find(&DragDropType::Text_Plain);
	if (w != e)
	{
		_Init(CF_TEXT, TYMED_HGLOBAL, fmt);
		availableTypes.push_back(*w);
		availableFormats.push_back(fmt);
	}

	w = typeset.find(&DragDropType::FileName);
	if (w != e)
	{
		_Init(CF_HDROP, TYMED_HGLOBAL, fmt);
		availableTypes.push_back(*w);
		availableFormats.push_back(fmt);
	}

	w = typeset.find(&DragDropType::Image_Native);
	if (w != e)
	{
		_Init(CF_DIB, TYMED_HGLOBAL, fmt);
		availableTypes.push_back(*w);
		availableFormats.push_back(fmt);

		_Init(CF_BITMAP, TYMED_GDI, fmt);
		availableTypes.push_back(*w);
		availableFormats.push_back(fmt);
	}

    if (!availableTypes.empty())
		this->fatData = fatData;
}

bool DragSourceWin32::StoreFatData(uint32 typeIndex, STGMEDIUM& medium)
{
	if (typeIndex >= availableTypes.size())
		return false;
	DragDropType::TypeID tid = availableTypes[typeIndex];
    FORMATETC& fmt = availableFormats[typeIndex];
    
	if ((tid == &DragDropType::Text_Plain) && 
		(0 != (medium.tymed & TYMED_ISTREAM)) &&
	 	(fmt.cfFormat == CF_TEXT))
	{
		if (medium.pstm != 0)
		{
			if (StoreFatData_Text_Plain_IStream_Push(medium))
			{
				if (medium.tymed != TYMED_ISTREAM)
					medium.tymed = TYMED_ISTREAM;
				return true;
			}
		}
		else
		{
			if (StoreFatData_Text_Plain_IStream_Pull(medium))
			{
				if (medium.tymed != TYMED_ISTREAM)
					medium.tymed = TYMED_ISTREAM;
				return true;
			}
		}
	}
	if ((tid == &DragDropType::Text_Plain) && 
	 	(0 != (medium.tymed & TYMED_HGLOBAL)) &&
		(fmt.cfFormat == CF_TEXT))
	{
		if (StoreFatData_Text_Plain_HGLOBAL(medium))
		{
			if (medium.tymed != TYMED_HGLOBAL)
				medium.tymed = TYMED_HGLOBAL;
			return true;
		}
	}
	if ((tid == &DragDropType::FileName) && 
		(0 != (medium.tymed & TYMED_HGLOBAL)) &&
		(fmt.cfFormat == CF_HDROP))
	{
		if (StoreFatData_FileName(medium))
		{
			if (medium.tymed != TYMED_HGLOBAL)
				medium.tymed = TYMED_HGLOBAL;
			return true;
		}
	}
	if ((tid == &DragDropType::Image_Native) && 
		(0 != (medium.tymed & TYMED_HGLOBAL)) &&
		(fmt.cfFormat == CF_DIB))
	{
		if (StoreFatData_DIB(medium))
		{
			if (medium.tymed != TYMED_HGLOBAL)
				medium.tymed = TYMED_HGLOBAL;
			return true;
		}
	}
	if ((tid == &DragDropType::Image_Native) && 
		(0 != (medium.tymed & TYMED_GDI)) &&
		(fmt.cfFormat == CF_BITMAP))
	{
		if (StoreFatData_HBITMAP(medium))
		{
			if (medium.tymed != TYMED_GDI)
				medium.tymed = TYMED_GDI;
			return true;
		}
	}
	return false;
}

bool DragSourceWin32::StoreFatData_Text_Plain_IStream_Push( STGMEDIUM& /*medium*/ )
{
	return false;  //ToDo
}
bool DragSourceWin32::StoreFatData_Text_Plain_IStream_Pull( STGMEDIUM& /*medium*/ )
{
	return false;  //ToDo
}
bool DragSourceWin32::StoreFatData_Text_Plain_HGLOBAL( STGMEDIUM& medium )
{
	if (medium.hGlobal != 0)
	{
		uint32 z = _StoreFatDataTo_Text_Size(fatData);
		uint32 az=  ::GlobalSize(medium.hGlobal);
		if (z <= az)
		    if (medium.hGlobal == _StoreFatDataTo_Text_Plain(fatData, medium.hGlobal))
		    	return true;
	}
	else
	{
		uint32 z = _StoreFatDataTo_Text_Size(fatData);
		medium.hGlobal= ::GlobalAlloc(GMEM_MOVEABLE, z);
	    if (medium.hGlobal == _StoreFatDataTo_Text_Plain(fatData, medium.hGlobal))
	    	return true;
	    ::GlobalFree(medium.hGlobal);
	    medium.hGlobal = NULL;
	}
	return false;
}
bool DragSourceWin32::StoreFatData_FileName( STGMEDIUM& medium )
{
	HGLOBAL h = _StoreFatDataTo_HDROP(fatData, medium.hGlobal);
	if (h != 0)
	{
		if (medium.hGlobal != h)
			medium.hGlobal = h;
		return true;
	}
	return false; 
}
bool DragSourceWin32::StoreFatData_DIB( STGMEDIUM& medium )
{
	if (medium.hGlobal != 0) 
	{   // push
		return false;
	}
	else
	{   // pull
		medium.hGlobal = _StoreFatDataTo_DIB(fatData);
		return (medium.hGlobal != 0);
	}
}
bool DragSourceWin32::StoreFatData_HBITMAP( STGMEDIUM& medium )
{
	if (medium.hGlobal != 0) 
	{   // push
		return false;
	}
	else
	{   // pull
		medium.hGlobal = _StoreFatDataTo_HBITMAP(fatData);
		return (medium.hGlobal != 0);
	}
}



DragSourceDataObject::DragSourceDataObject(DragSourceWin32* ds)
:dragSource(ds)
{
#if DEBUG
UnitTest::Log("Created %08X DragSourceDataObject",this);
#endif
}
DragSourceDataObject::~DragSourceDataObject()
{
#if DEBUG
UnitTest::Log("Deleted %08X DragSourceDataObject",this);
#endif
}

HRESULT STDMETHODCALLTYPE DragSourceDataObject::QueryInterface( 
    REFIID riid,
    void __RPC_FAR *__RPC_FAR *ppvObject)
{
	if (riid == IID_IUnknown)
	{
		*ppvObject = (void*)(IUnknown*) this;
		AddRef(); // we add the refcount for the requestor
		return NOERROR;
	}
	if (riid == IID_IDataObject)
	{
		*ppvObject = (void*)(IDataObject*) this;
		AddRef(); // we add the refcount for the requestor
		return NOERROR;
	}
#if DEBUG
	static IID IID_IdentityUnmarshal = {0x0000001B,0x0000,0x0000,0xC0,0,0,0,0,0,0,0x46};
	if (riid == IID_IEnumFORMATETC)
		UnitTest::Log("!!! DragSourceDataObject::QueryInterface IID_IEnumFORMATETC");
	else if (riid == IID_IMarshal)
		UnitTest::Log("!!! DragSourceDataObject::QueryInterface IID_IMarshal");
	else if (riid == IID_IStdMarshalInfo)
		UnitTest::Log("!!! DragSourceDataObject::QueryInterface IID_IStdMarshalInfo");
	else if (riid == IID_IExternalConnection)
		UnitTest::Log("!!! DragSourceDataObject::QueryInterface IID_IExternalConnection");
	else if (riid == IID_IdentityUnmarshal)
		UnitTest::Log("!!! DragSourceDataObject::QueryInterface IID_IdentityUnmarshal");
	else
	{					
		//__declspec(uuid(x))		  
		//__uuidof(IEnumFORMATETC)
		// IID_IEnumFORMATETC 00000103-0000-0000-C000-000000000046
		UnitTest::Log("!!! DragSourceDataObject::QueryInterface unknown %08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X"
			,0[(uint32*)&riid]
			,2[(uint16*)&riid]
			,3[(uint16*)&riid]
			,8[(uint8*)&riid]
			,9[(uint8*)&riid]
			,10[(uint8*)&riid]
			,11[(uint8*)&riid]
			,12[(uint8*)&riid]
			,13[(uint8*)&riid]
			,14[(uint8*)&riid]
			,15[(uint8*)&riid]
			);
	}
#endif
	// we do not support any other interface
	*ppvObject = NULL;
	return E_NOINTERFACE;
}

ULONG STDMETHODCALLTYPE DragSourceDataObject::AddRef()
{
	return dragSource->AddRef();
}

ULONG STDMETHODCALLTYPE DragSourceDataObject::Release()
{
	return dragSource->Release();
}

/* [local] */ HRESULT STDMETHODCALLTYPE DragSourceDataObject::GetData( 
    /* [unique][in] */ FORMATETC __RPC_FAR *pformatetc,
    /* [out] */ STGMEDIUM __RPC_FAR *pmedium)
{
#if DEBUG
	UnitTest::Log("incoming DragSourceDataObject::GetData this=%08X",this);
#endif		
	if (pformatetc == NULL)
		return E_INVALIDARG;
	if (pmedium == NULL)
		return E_INVALIDARG;
	if (pformatetc->lindex != -1)
		return DV_E_LINDEX;

	// this will force the medium members to be allocated
	_Init(*pmedium); 
	// we do not want to delete this medium, let the caller do it
	// he will call ReleaseStgMedium and therefore 
	// free the HGLOBAL, release the IStream...
	pmedium->pUnkForRelease = NULL; 

	// let's search through the vector of formats for one that matches
	std::vector<FORMATETC>& fmts = dragSource->availableFormats;
	std::vector<FORMATETC>::iterator b = fmts.begin();
	std::vector<FORMATETC>::iterator e = fmts.end();
	for (; b!=e; ++b)
	{
		if (b->cfFormat != pformatetc->cfFormat)
			continue;
		if (0 == (b->tymed & pformatetc->tymed))
			return DV_E_TYMED;
		break; // found it
	}	
	if (b==e)
		return DV_E_FORMATETC;
	
	// this may be more than one medium type, the final choice later
    pmedium->tymed = b->tymed & pformatetc->tymed;
    
    if (!dragSource->StoreFatData(b-fmts.begin(), *pmedium))
       return E_FAIL;

	return S_OK;
}

/* [local] */ HRESULT STDMETHODCALLTYPE DragSourceDataObject::GetDataHere( 
    /* [unique][in] */ FORMATETC __RPC_FAR *pformatetc,
    /* [out][in] */ STGMEDIUM __RPC_FAR *pmedium)
{
	// THIS IS PUSH TECHNOLOGY ... the client instead of reading from the stream
	// expects us to write to it ... 

	if (pformatetc == NULL)
		return E_INVALIDARG;
	if (pmedium == NULL)
		return E_INVALIDARG;
	// the supplied type should be compatible with the request
	// and since the medium is preallocated it better be HGLOBAL
	if (0 == (pformatetc->tymed & pmedium->tymed))
		return E_INVALIDARG; 
	if (pformatetc->lindex != -1)
		return DV_E_LINDEX;
	// the medium inside STGMEDIUM is already allocated by the caller
    // we do not touch pmedium, it is an input for us, here we check if the caller indeed
    // filled the medium, since it's a union all we need to check is hGlobal
	if (0 == pmedium->hGlobal)
		return E_INVALIDARG; 

	// let's search through the vector of formats for one that matches
	std::vector<FORMATETC>& fmts = dragSource->availableFormats;
	std::vector<FORMATETC>::iterator b = fmts.begin();
	std::vector<FORMATETC>::iterator e = fmts.end();
	for (; b!=e; ++b)
	{
		if (b->cfFormat != pformatetc->cfFormat)
			continue;
		if (0 == (b->tymed & pmedium->tymed))
			return DV_E_TYMED;
		break; //found it 
	}	
	if (b == e)
		return DV_E_FORMATETC;

    if (!dragSource->StoreFatData(b-fmts.begin(), *pmedium))
       return E_FAIL;

	return S_OK;
}

HRESULT STDMETHODCALLTYPE DragSourceDataObject::QueryGetData( 
    /* [unique][in] */ FORMATETC __RPC_FAR *pformatetc)
{
	if (pformatetc == NULL)
		return E_INVALIDARG;
	// let's search through the vector of formats for one that matches
	std::vector<FORMATETC>& fmts = dragSource->availableFormats;
	std::vector<FORMATETC>::iterator b = fmts.begin();
	std::vector<FORMATETC>::iterator e = fmts.end();
	for (; b!= e; ++b)
	{
		if (b->cfFormat != pformatetc->cfFormat)
			continue;
		if (0 == (b->tymed & pformatetc->tymed))
			return DV_E_TYMED;
		return S_OK;   // found it
	}	
	return DV_E_FORMATETC;
}

HRESULT STDMETHODCALLTYPE DragSourceDataObject::GetCanonicalFormatEtc( 
    /* [unique][in] */ FORMATETC __RPC_FAR *pformatectIn,
    /* [out] */ FORMATETC __RPC_FAR *pformatetcOut)
{
	if (pformatectIn == NULL)
		return E_INVALIDARG;
	if (pformatetcOut == NULL)
		return E_INVALIDARG;
	
	// let's search through the vector of formats for one that matches
	// and fill the output with that match
	std::vector<FORMATETC>& fmts = dragSource->availableFormats;
	std::vector<FORMATETC>::iterator b = fmts.begin();
	std::vector<FORMATETC>::iterator e = fmts.end();
	for (; b!= e; ++b)
	{
		if (b->cfFormat != pformatectIn->cfFormat)
			continue;
		if (b->tymed == pformatectIn->tymed)
			return DATA_S_SAMEFORMATETC;
		if (0 == (b->tymed & pformatectIn->tymed))
			return DV_E_TYMED;
		*pformatetcOut = *b;
		return S_OK;   // found it
	}	
	return DV_E_FORMATETC;
}

/* [local] */ HRESULT STDMETHODCALLTYPE DragSourceDataObject::SetData( 
    /* [unique][in] */ FORMATETC __RPC_FAR * /*pformatetc*/,
    /* [unique][in] */ STGMEDIUM __RPC_FAR * /*pmedium*/,
    /* [in] */ BOOL /*fRelease*/)
{
	return E_NOTIMPL;
}


HRESULT STDMETHODCALLTYPE DragSourceDataObject::EnumFormatEtc( 
    /* [in] */ DWORD dwDirection,
    /* [out] */ IEnumFORMATETC __RPC_FAR *__RPC_FAR *ppenum)
{
#if DEBUG
	UnitTest::Log("incoming call to DragSourceDataObject::EnumFormatEtc");
#endif
	if( ppenum == NULL )
		return E_INVALIDARG;

/*  // stupid metrowerks lib does not provide this func...
	std::vector<FORMATETC>& fmts = dragSource->availableFormats;
	uint32 z = fmts.size();
    FORMATETC* fmt_v = & (*fmts.begin());
//	STDAPI ::SHCreateStdEnumFmtEtc(
//			UINT n, 
//			FORMATETC __RPC_FAR *afmt, 
//			IEnumFORMATETC __RPC_FAR *__RPC_FAR *ppenum);
	return ::SHCreateStdEnumFmtEtc(z,fmt_v, ppenum);
*/

	*ppenum = new DragSourceFormatEnum(dragSource, dwDirection==DATADIR_SET);
	if (*ppenum == NULL)
		return E_OUTOFMEMORY;
	(*ppenum)->AddRef();
	return S_OK;
}

HRESULT STDMETHODCALLTYPE DragSourceDataObject::DAdvise( 
    /* [in] */ FORMATETC __RPC_FAR * /*pformatetc*/,
    /* [in] */ DWORD /*advf*/,
    /* [unique][in] */ IAdviseSink __RPC_FAR * /*pAdvSink*/,
    /* [out] */ DWORD __RPC_FAR * /*pdwConnection*/)
{
	// for Async
	return OLE_E_ADVISENOTSUPPORTED; //E_NOTIMPL;
}

HRESULT STDMETHODCALLTYPE DragSourceDataObject::DUnadvise( 
    /* [in] */ DWORD /*dwConnection*/)
{
	// for Async
	return OLE_E_ADVISENOTSUPPORTED; //E_NOTIMPL;
}

HRESULT STDMETHODCALLTYPE DragSourceDataObject::EnumDAdvise( 
    /* [out] */ IEnumSTATDATA __RPC_FAR *__RPC_FAR * /*ppenumAdvise*/)
{
	// for Async
	return OLE_E_ADVISENOTSUPPORTED; //E_NOTIMPL;
}

DragSourceFormatEnum::DragSourceFormatEnum(DragSourceWin32* ds, bool fwr)
: refcount(0)
, dragSource(ds)
, forWriting(fwr)
, currentPos(0)
{
	dragSource->AddRef(); // keep dragsource alive too, we need it
#if DEBUG
UnitTest::Log("Created %08X DragSourceFormatEnum",this);
#endif
}
DragSourceFormatEnum::~DragSourceFormatEnum()
{
	dragSource->Release(); // drag source no longer needed
	dragSource = 0;
#if DEBUG
UnitTest::Log("Delete %08X DragSourceFormatEnum",this);
#endif
}

HRESULT STDMETHODCALLTYPE DragSourceFormatEnum::QueryInterface( 
        REFIID riid,
        void __RPC_FAR *__RPC_FAR *ppvObject)
{
	if (riid == IID_IUnknown)
	{
		*ppvObject = (void*)(IUnknown*) this;
		AddRef(); // we add the refcount for the requestor
		return NOERROR;
	}
	if (riid == IID_IEnumFORMATETC)
	{
		*ppvObject = (void*)(IEnumFORMATETC*) this;
		AddRef(); // we add the refcount for the requestor
		return NOERROR;
	}
#ifdef DEBUG
	UnitTest::Log("!!! DragSourceFormatEnum::QueryInterface unknown interface");
#endif
	// we do not support any other interface
	*ppvObject = NULL;
	return E_NOINTERFACE;
}
ULONG STDMETHODCALLTYPE DragSourceFormatEnum::AddRef()
{
	return ++refcount;
}
ULONG STDMETHODCALLTYPE DragSourceFormatEnum::Release()
{
	uint32 r = --refcount;
	if (r == 0)
		delete this;
	return r;
}
/* [local] */ HRESULT STDMETHODCALLTYPE DragSourceFormatEnum::Next( 
        /* [in] */ ULONG celt,
        /* [length_is][size_is][out] */ FORMATETC __RPC_FAR *rgelt,
        /* [out] */ ULONG __RPC_FAR *pceltFetched)
{
	std::vector<FORMATETC>& fmts = dragSource->availableFormats;
	uint32 z = fmts.size();
	if (currentPos >= z)
		return E_FAIL;
	if (currentPos + celt > z)
	{
		celt = z-currentPos;
		// we need to tell the guy we are returning less than he asked for
		if (pceltFetched == NULL)
			return E_INVALIDARG; 
	}
	if (pceltFetched != NULL)
		*pceltFetched = celt; // how many we actually return
	if (celt > 0)
	{
		if (rgelt == NULL)
			return E_INVALIDARG;
		std::vector<FORMATETC>::iterator b = fmts.begin() + currentPos;
		std::vector<FORMATETC>::iterator e = b+celt;
		for (; b != e; ++b)
		{
			*rgelt++ = *b;
		}
	}
	currentPos += celt;
	return S_OK;
}
HRESULT STDMETHODCALLTYPE DragSourceFormatEnum::Skip( 
        /* [in] */ ULONG celt)
{
	currentPos += celt;
	return S_OK;
}
HRESULT STDMETHODCALLTYPE DragSourceFormatEnum::Reset( void)
{
	currentPos = 0;
	return S_OK;
}
HRESULT STDMETHODCALLTYPE DragSourceFormatEnum::Clone( 
        /* [out] */ IEnumFORMATETC __RPC_FAR *__RPC_FAR *ppenum)
{
	if( ppenum == NULL )
		return E_INVALIDARG;
	if (0 == (*ppenum = new DragSourceFormatEnum(dragSource, forWriting)))
		return E_OUTOFMEMORY;
	(*ppenum)->AddRef();
	return S_OK;
}












} // namespace XSP
#endif
