/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{

class TabBarListener : public 	ButtonView::ActionListener
{
	TabBar* tabBar;
public:
	TabBarListener(TabBar* bar)
	: tabBar(bar)
	{}
	~TabBarListener()
	{
		CancelListener();
	}
	virtual void CancelListener()
	{
		tabBar = 0;
		super::CancelListener();
	}
	virtual void ActionPerform(ButtonView* button)
	{
		if (tabBar != 0)
		    tabBar->NotifyTabSelected(button->GetText());
	}
}; // class TabBarListener

class TabViewListener : public 	TabBar::TabListener
{
	TabView* tabView;
public:
	TabViewListener(TabView* view)
	: tabView(view)
	{}
	~TabViewListener()
	{
		CancelListener();
	}
	virtual void CancelListener()
	{
		tabView = 0;
		super::CancelListener();
	}
	virtual void TabSelected(const String& title)
	{
		if (tabView != 0)
			tabView->GoToPage(title);
	}
}; // class TabViewListener



TabBar::TabBar()
{
	tabListener = new TabBarListener(this);
}

TabBar::~TabBar()
{
	tabListener->CancelListener();
	tabListener=0;
}

void TabBar::SetBounds(const Rect2D& bounds)
{
	View::SetBounds(bounds);
	DoLayout();
}


void TabBar::SetSkin(const ViewSkin::owner& skin)
{
	View::SetSkin(skin);
	ViewSkin::owner sc = GetSkinPartOrDefault(String::From_c_str("tab"));
	Tabs::iterator p = tabs.begin();
	Tabs::iterator e = tabs.end();
	for(; p!=e; ++p)
	{
		(*p)->SetSkin(sc);
	}
	DoLayout();
}

/*void TabBar::SetTabFont(const TextFont& font)
{
	tabFont = font;
	DoLayout();
}*/

void TabBar::SetFont(const TextFont& f)
{
	View::SetFont(f);
	Tabs::iterator p = tabs.begin();
	Tabs::iterator e = tabs.end();
	for(; p!=e; ++p)
	{
		(*p)->SetFont(f);
	}
	DoLayout();
}

void TabBar::AddTab(const String& title)
{
	ButtonView* button = new ButtonView();
	tabs.push_back(button);
	button->SetText(title);

	button->SetFont(tabFont);
	button->SetTextAlign(Rect2D::C, Rect2D::C);
	button->SetSkin(GetSkinPartOrDefault(String::From_c_str("tab"))); 

	button->SetFocusable(true);
	button->Set3StateButton(true);
	button->SetChecked(false);
	button->SetBounds( Rect2D(button->GetPreferredSize()) );

	button->AddActionListener(tabListener);
	AddView(button);
	DoLayout();
}

void TabBar::RemoveTab(const String& title)
{
	Tabs::iterator p = tabs.begin();
	Tabs::iterator e = tabs.end();
	for(; p!=e; ++p)
	{
		if (title == (*p)->GetText())
			break;
	}	
	if (p != e)
	{
		RemoveView((*p).operator->());
		tabs.erase(p);
		DoLayout();
	}
}

void TabBar::GoToTab(const String& title)
{
	Tabs::iterator p = tabs.begin();
	Tabs::iterator e = tabs.end();
	for(; p!=e; ++p)
	{
		bool theOne = (title == (*p)->GetText());
		if ((*p)->IsChecked())
		{
			if (!theOne)
			{
				(*p)->SetChecked(false);
				(*p)->Refresh();
			}
		}
		else
		{
			if (theOne)
			{
				(*p)->SetChecked(true);
				(*p)->Refresh();
			}
		}
	}	
}

const String& TabBar::GetCurrentTab()
{
	Tabs::iterator p = tabs.begin();
	Tabs::iterator e = tabs.end();
	for(; p!=e; ++p)
	{
		if ((*p)->IsChecked())
			return (*p)->GetText();
	}
	return String::kEmpty;
}


Size2D TabBar::GetPreferredSize()
{
	Size2D z(0, tabFont.GetHeight());
	
	Tabs::iterator b = tabs.begin();
	Tabs::iterator e = tabs.end();
	Tabs::iterator p;
	for(p = b; p!=e; ++p)
	{
		Size2D r = (*p)->GetPreferredSize();
		if (r.h > z.h)
			z.h = r.h;
		z.w += r.w;
	}

	{
		ViewSkin::owner skin = GetSkin();
		if (skin != 0)
		{
			StateSkin::owner stateSkin = skin->GetStateSkin(0);
			if (stateSkin != 0)
			{
				Rect2D border = stateSkin->GetContent();
				z.w += border.w;
				z.h += border.h;
			}
		}
	}

	return z;
}
void TabBar::DoLayout()
{
	bool dirty = false;

	Rect2D lb = GetLocalBounds();
	{
		ViewSkin::owner skin = GetSkin();
		if (skin != 0)
		{
			StateSkin::owner stateSkin = skin->GetStateSkin(0);
			if (stateSkin != 0)
				lb = stateSkin->GetContent(lb);
		}
	}

	Tabs::iterator b = tabs.begin();
	Tabs::iterator e = tabs.end();
	Tabs::iterator p;
	for(p = b; p!=e; ++p)
	{
		Rect2D r = (*p)->GetBounds();
		bool adjust = false;
		if (r.y != lb.y)
		{
			r.y = lb.y;
			adjust = true;
		}
		if (r.h != lb.h)
		{
			r.h = lb.h;
			adjust = true;
		}
		if (r.x != lb.x)
		{
			r.x = lb.x;
			adjust = true;
		}
		lb.x += r.w;
		if (adjust)
		{
			(*p)->SetBounds(r);
			dirty = true;
		}
	}

	if (dirty)
		Refresh();
}

void TabBar::NotifyTabSelected(const String& title)
{
	for (TabListener::each e(tabListeners); !e.empty(); e.pop()) 
		if (! e->IsCanceled())
			e->TabSelected(title);
}
void TabBar::AddTabListener(const TabListener::owner& l)
{
	tabListeners.Add(l);
}


#if 0
#pragma mark -
#endif

TabView::TabView()
{
	contentView = new MView();
	tabBar = new TabBar();
	tabBar->SetFocusable(false);
	AddView(tabBar.operator->());
	AddView(contentView.operator->());
	
	tabBar->AddTabListener(new TabViewListener(this));
}

TabView::~TabView()
{
	assocMap.clear();
	tabBar = 0; // this also kills the TabViewListener inside
	contentView = 0;
}

void TabView::SetBounds(const Rect2D& bounds)
{
	View::SetBounds(bounds);

	Rect2D cont = GetSkinContentBounds();
	
	Rect2D bar(cont.GetNW(), tabBar->GetPreferredSize());
	bar.w = cont.w; // make it fill the entire north region
	tabBar->SetBounds(bar); 

	cont.y += bar.h;
	cont.h = std::max(0L, cont.h - bar.h);
	contentView->SetBounds(cont); // adjust the content to the leftover :0
}  
void TabView::SetFont(const TextFont& font)
{
	View::SetFont(font);
	tabBar->SetFont(font);
}

void TabView::SetSkin(const ViewSkin::owner& skin)
{
	View::SetSkin(skin);
	tabBar->SetSkin(GetSkinPartOrDefault(String::From_c_str("bar")));
}

void TabView::AddPage(const String& title, const View::owner& page)
{
	AssocMap::iterator e = assocMap.end();
	AssocMap::iterator p = assocMap.begin();
	for (; p != e; ++p)
	{
		if (p->first == title)
			break;
	}
    if (p!=e) // already have such a page
    {
		contentView->AddView(page); 
		contentView->RemoveView(p->second); 
    	p->second = page;
    }
    else
    {
    	assocMap.push_back(AssocMap::value_type(title,page));
		tabBar->AddTab(title);
		contentView->AddView(page); 
	}
}

void TabView::RemovePage(const String& title)
{
	AssocMap::iterator e = assocMap.end();
	AssocMap::iterator p = assocMap.begin();
	for (; p != e; ++p)
	{
		if (p->first == title)
			break;
	}
	if (p != e)
	{
		contentView->RemoveView(p->second); 
		tabBar->RemoveTab(title);
		assocMap.erase(p);
	}
}

void TabView::GoToPage(const String& title)
{
	AssocMap::iterator e = assocMap.end();
	AssocMap::iterator p = assocMap.begin();
	for (; p != e; ++p)
	{
		if (p->first == title)
			break;
	}
	if (p != e)
	{
		contentView->GoToView(p->second);
		tabBar->GoToTab(title);
	}	
}

void TabView::GoToNext()
{
	const String& title = GetCurrentPage();
	
	AssocMap::iterator e = assocMap.end();
	AssocMap::iterator b = assocMap.begin();
	AssocMap::iterator p = b;
	for (; p != e; ++p)
	{
		if (p->first == title)
			break;
	}
	if (p != e)
		++p;
	if (p == e)
		p = b;
	if (p != e)
	{
		contentView->GoToView(p->second);
		tabBar->GoToTab(p->first);
	}	
}

void TabView::GoToPrev()
{
	const String& title = GetCurrentPage();

	AssocMap::iterator e = assocMap.end();
	AssocMap::iterator b = assocMap.begin();
	AssocMap::iterator p = b;
	for (; p != e; ++p)
	{
		if (p->first == title)
			break;
	}
	if (p == b)
		p = e;
	if (p != b)
		--p;
	if (p != e)
	{
		contentView->GoToView(p->second);
		tabBar->GoToTab(p->first);
	}	
}

const String& TabView::GetCurrentPage()
{
	return tabBar->GetCurrentTab();
}	

const String& TabView::GetPageTitle(const View::owner& page)
{
	AssocMap::iterator e = assocMap.end();
	AssocMap::iterator p = assocMap.begin();
	for(; p!=e; ++p)
	{
		if (page == p->second)
			return p->first;
	}
	return String::kEmpty;
}

bool TabView::HandleKeyEvent( const UIEvent& ev )
{
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_TAB))
	{
		Refresh();
		GoToNext();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		((UIEvent::CtrlState|UIEvent::ShiftState) == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_TAB))
	{
		Refresh();
		GoToPrev();
		return true;
    }
	return View::HandleKeyEvent(ev);
}



} // namespace XSP
