/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{

View::View()
: refcount(0)
, hidden(false)
, parent(0)
, viewState(StateFocusable) 
{
}
void View::addref()
{
	++refcount;
}
void View::release()
{
	if (--refcount == 0)
		delete this;
}

View::~View()
{
	ASSERT(parent == 0);
	// no notification this late during destruction
	viewListeners.CancelAllListeners();
    Close();
}

void View::Close()
{
	NotifyChange_ViewClosed();
	
	if (parent != 0)
		parent->RemoveView(this);

	Children c;
	c.swap(children);
	for (Children::iterator b=c.begin(), e=c.end(); b!=e; ++b)
	{
		const View::owner& v = *b;
		RemoveView(v);
		//?? must call Close recursively otherwise a window close might not go deep enough
		v->Close();
	}
	c.clear();
}

void View::AddListener(const ViewListener::owner& l)
{
	viewListeners.Add(l);
}


void View::_ViewChange::ViewClosed(View* /*view*/)
{}
void View::NotifyChange_ViewClosed()
{
	viewListeners.withinNotify = true;
	for (ViewListener::each e(viewListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ViewClosed(this);
	viewListeners.withinNotify = false;
	viewListeners.CancelAllListeners();
}

void View::_ViewChange::ChildAddRemove(bool /*added*/, const View::owner& /*child*/, View* /*view*/)
{}
void View::NotifyChange_ChildAddRemove(bool added, const View::owner& child)
{
	viewListeners.withinNotify = true;
	for (ViewListener::each e(viewListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ChildAddRemove(added, child, this);
	viewListeners.withinNotify = false;
}

void View::_ViewChange::ViewShowHide(bool /*hide*/, View* /*view*/)
{}
void View::NotifyChange_ViewShowHide()
{
	viewListeners.withinNotify = true;
	for (ViewListener::each e(viewListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ViewShowHide(hidden, this);
	viewListeners.withinNotify = false;
}

void View::_ViewChange::ViewStateChange(uint32 /*oldState*/, uint32 /*newState*/, View* /*view*/)
{}
void View::NotifyChange_ViewStateChange(uint32 oldState, uint32 newState)
{
	viewListeners.withinNotify = true;
	for (ViewListener::each e(viewListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ViewStateChange(oldState, newState, this);
	viewListeners.withinNotify = false;
}

void View::_ViewChange::ViewBoundsChange(const Rect2D& /*oldBounds*/, 
										 const Rect2D& /*newBounds*/, 
										 View* /*view*/)
{}
void View::NotifyChange_ViewBoundsChange(const Rect2D& oldBounds, const Rect2D& newBounds)
{
	viewListeners.withinNotify = true;
	for (ViewListener::each e(viewListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ViewBoundsChange(oldBounds, newBounds, this);
	viewListeners.withinNotify = false;
}

void View::_ViewChange::ViewScroll(const Size2D& /*scrollSize*/, 
								   const Rect2D& /*bounds*/, 
								   View* /*scrollView*/)
{}
void View::NotifyChange_ViewScroll(const Size2D& scrollSize, const Rect2D& bounds)
{
	viewListeners.withinNotify = true;
	for (ViewListener::each e(viewListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ViewScroll(scrollSize, bounds, this);
	viewListeners.withinNotify = false;
}

void View::ListenToScrollNotificationsFrom(const View::owner& /*view*/)
{
	// default implementation does nothing, others should override
}
void View::CancelListenToScrollNotifications()
{
	// default implementation does nothing, others should override
}

void View::AddView(const owner& view)
{
	children.push_back(view);
	view->parent = this;
	Refresh(view->GetBounds());
	
	NotifyChange_ChildAddRemove(true, view);
}

void View::RemoveView(const owner& view)
{
	if (view->parent == this)
	{
		EndMouseCapture(view);
		owner tmp = view; // protect against deletion of view
		view->parent = 0;
		children.remove(view);
		Refresh(tmp->GetBounds());

		NotifyChange_ChildAddRemove(false, view);
	}
}
void View::RemoveAllChildren()
{
	Children c;
	c.swap(children);
	for (Children::iterator b=c.begin(), e=c.end(); b!=e; ++b)
	{
		const View::owner& v = *b;
		RemoveView(v);
	}
	c.clear();
}

void View::SetBounds(const Rect2D& b)
{
	if (bounds != b)
	{
		Rect2D old(bounds);
		bounds = b;

		Rect2D u(old.Union(b));
		Refresh(u);

		NotifyChange_ViewBoundsChange(old, b);
	}
}  

Size2D View::GetPreferredSize()
{
	return bounds.GetSize();
}

void View::Hide(bool on)
{
	if (hidden != on)
	{
		if (!hidden) Refresh();
		hidden = on;
		if (!hidden) Refresh();
		
		NotifyChange_ViewShowHide();
	}
}

uint32 View::SetState(uint32 s) 
{ 
	uint32 old = viewState;
	if (old != s)
	{
		viewState = s; 
		NotifyChange_ViewStateChange(old, s);
	}
	return viewState;
}

void View::Refresh(const Rect2D& r)
{
	if (parent == 0)
		return;
	if (hidden)
		return;
	Rect2D t = r.Intersect(GetLocalBounds());
	if (t.isEmpty())
		return;
	t.OffsetBy(bounds.x,bounds.y);
	parent->Refresh(t);
}

void View::Refresh()
{
	Refresh(GetLocalBounds());
}


Rect2D View::GetLocalBounds() const
{
	return Rect2D(bounds.GetSize());
}

Rect2D View::GetSkinContentBounds()
{
	Rect2D r(bounds.GetSize());

	ViewSkin::owner skin = GetSkin();
	if (skin != 0)
	{
		StateSkin::owner stateSkin = skin->GetStateSkin(viewState);
		if (stateSkin != 0)
			r = stateSkin->GetContent(r);
	}

	return r;
}

Window View::GetParentWindow()
{
    if (parent == 0)	   //?? VERIFY(parent != 0)
    	return Window();
    return parent->GetParentWindow();
}

Window Window::_Data::GetParentWindow()
{
	Window w;
	w.data = this;
	return w;
}

Rect2D View::GetVisibleBoundsOnScreen()
{
	Rect2D b(bounds);
	if (parent != 0)
	{
		Rect2D pb(parent->GetVisibleBoundsOnScreen());
		b.OffsetBy(pb.x, pb.y);
		b.ClipTo(pb);
	}	
	return b;	
}
Rect2D Window::_Data::GetVisibleBoundsOnScreen()
{
	Window w;
	w.data = this;
	return w.GetContentBoundsOnScreen();
}


View::owner View::DetermineCommonAncestor(const View::owner& view1, 
								 		  const View::owner& view2)
{
	View::owner t;		   
	std::list<View::owner> parents1;
	for(t=view1; t!=0; )
		parents1.push_front(t = t->parent);
		
	std::list<View::owner> parents2;
	for(t=view2; t!=0; )
		parents2.push_front(t = t->parent);

	t = 0;
	while(!parents1.empty() && !parents2.empty() && parents1.front() == parents2.front())
	{
		t = parents1.front();
		parents1.pop_front();
		parents2.pop_front();
	}

	return t;
}

View::owner View::GetChild(unsigned vState)
{
	for (Children::iterator b=children.begin(), e=children.end(); b!=e; ++b)
	{
		const View::owner& v = *b;
		unsigned s=v->viewState;
		if (0 != (s & vState))
			return v;
	}
	return View::owner();
}

View::owner View::GetParent(unsigned vState)
{
	for (View::owner p = parent; p!=0; p=p->parent)
	{
		unsigned s=p->viewState;
		if (0 != (s & vState))
			return p;
	}
	return View::owner();
}

View::owner View::GetRootView()
{
	View::owner t = this;
	for (View::owner p = t->parent; p!=0; p=(t=p)->parent)
	{}
	return t;
}

void View::StartMouseCapture(const owner& view, const Point2D& origin)
{
	if (!IsMouseCapturing())
	{
	    VERIFY(parent != 0);
		parent->StartMouseCapture(view, origin);	

		StateSetReset(StateCapturing, 0);
	}
}
void View::EndMouseCapture(const owner& view)
{
	if (IsMouseCapturing())
	{
	    VERIFY(parent != 0);
		parent->EndMouseCapture(view);	

		StateSetReset(0, StateCapturing);
	}
}
void View::MouseCaptureBroken()
{
	if (IsMouseCapturing())
	{
		StateSetReset(0, StateCapturing);
		Refresh(); // we probably have to change our looks
	}
}

bool View::HandleMouseEvent( const UIEvent& )
{
	return false;
}

bool View::ActionFocusNextChild()
{
	Children::iterator v4=children.begin();
	for (; v4!=children.end(); ++v4)
		if ((*v4)->IsFocused())
		{
			++v4;  // found current
			break;
		}
	for (; v4!=children.end(); ++v4)
		if ((*v4)->IsFocusable() && !(*v4)->IsDisabled() && !(*v4)->IsHidden())
		{
			(*v4)->SetFocused();
			(*v4)->ActionFocusFirstChild(); // and his child
			return true;
		}
	return false;
}

bool View::ActionFocusFirstChild()
{
	View::owner focusable; // to hold the focus
	for (Children::iterator v2=children.begin(); v2!=children.end(); ++v2)
		if ((*v2)->IsFocusable() && !(*v2)->IsDisabled() && !(*v2)->IsHidden())
		{
			focusable = *v2;
			break;
		}
	if ((focusable != 0) && !focusable->IsFocused())
	{
		focusable->SetFocused();
		focusable->ActionFocusFirstChild(); // and his child
		return true;
	}
	return false;
}

bool View::ActionFocusPrevChild()
{
	Children::iterator v3=children.end();
	while (v3 != children.begin())
	{
		--v3;
		if ((*v3)->IsFocused())
			break;
	}
	while (v3 != children.begin())
	{
		--v3;
		if ((*v3)->IsFocusable() && !(*v3)->IsDisabled() && !(*v3)->IsHidden())
		{
			(*v3)->SetFocused();
			(*v3)->ActionFocusLastChild(); // and his child
			return true;
		}
	}
	return false;
}

bool View::ActionFocusLastChild()
{
	View::owner focusable; // to hold the focus
	for (Children::iterator v1=children.end(); v1-- != children.begin();)
	{
		if ((*v1)->IsFocusable() && !(*v1)->IsDisabled() && !(*v1)->IsHidden())
		{
			focusable = *v1;
			break;
		}
	}
	if ((focusable != 0) && !focusable->IsFocused())
	{
		focusable->SetFocused();
		focusable->ActionFocusLastChild(); // and his child
		return true;
	}
	return false;
}

bool View::HandleKeyEvent( const UIEvent& ev )
{
	if ((ev.origin == UIEvent::KeyDown) && 
		(ev.code == VK_TAB) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)))
		return ActionFocusNextChild();

	if ((ev.origin == UIEvent::UnusedKey) && 
		(ev.code == VK_TAB) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)))
		return ActionFocusFirstChild();

	if ((ev.origin == UIEvent::KeyDown) && 
		(ev.code == VK_TAB) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)))
		return ActionFocusPrevChild();

	if ((ev.origin == UIEvent::UnusedKey) && 
		(ev.code == VK_TAB) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)))
		return ActionFocusLastChild();

	// none of the key combinations I care about was triggered
	return false;	
}

bool View::_DispatchKeyEventToFocus( const UIEvent& ev )
{
	View::owner foc = GetChild(StateFocused);
	if (foc != 0)
		if (foc->DispatchKeyEvent(ev))
			return true;
	return false;
}

bool View::_DispatchKeyEventAsHotKey( const UIEvent& ev )
{
	UIEvent hot(ev);
	hot.origin = UIEvent::HotKey;
	for (Children::iterator v=children.begin(); v!=children.end(); ++v)
	{
		if ((*v)->DispatchKeyEvent(hot))
			return true;
	}
	return false;
}

bool View::DispatchKeyEvent( const UIEvent& ev )
{
	if (IsDisabled()||IsHidden())
		return false;
	// an unused TAB key will be used to focus first/last
	switch (ev.origin)
	{
	case UIEvent::PriorityKey : // if node can use it leaf does not see it
	case UIEvent::UnusedKey :   // if node can use it leaf does not see it
		if (HandleKeyEvent( ev ))
			return true;
		if (_DispatchKeyEventToFocus(ev))
			return true;
		return false;
	case UIEvent::AsciiCode : 
	case UIEvent::KeyDown:		// first seen by focus then by this then as hotkey
		if (_DispatchKeyEventToFocus(ev))
			return true;
		if (HandleKeyEvent( ev ))
			return true;
		if (_DispatchKeyEventAsHotKey( ev ))
			return true;
		return false;
	case UIEvent::HotKey : 		// this level is as far as hotkeys are allowed to go
		if (HandleKeyEvent( ev ))
			return true;
		return false;
	} // switch
	return false;
}

void View::SetCaret(const Rect2D& r)
{
	if (parent == 0)
		return;
	Rect2D cr(r);
	if (IsHidden())
	{
		ASSERT(cr.isEmpty());
	}
	else
	{
		cr.OffsetBy(bounds.x, bounds.y);
		cr.ClipTo(bounds);
	}
	parent->SetCaret(cr);
}

void View::SetActivated(bool on)
{
	if (on) StateSetReset(StateActivated,0);
	else    StateSetReset(0,StateActivated);
	for (Children::iterator v=children.begin(); v!=children.end(); ++v)
		(*v)->SetActivated(on);
	HandleActivated(on);
}

void View::SetFocusable(bool on)
{
	if (on) StateSetReset(StateFocusable,0);
	else    StateSetReset(0,StateFocusable);
}

void View::SetFocused()
{
	if (!IsFocusable()) // not focusable
		return;
	if (IsFocused()) // already focused
		return;

	// find the parent that is focused or focusable
	bool needActivate = false;
	View::owner t = parent;
	for ( ; t!=0; t=t->parent)
	{
		unsigned s=t->GetState();
		if (0 == (s & StateFocusable))
			return; // some parent is not focusable, drop everything
		if (0 != (s & StateOldFocus)) 
		{
			needActivate = true;
			break;
		}
		if (0 != (s & StateFocused)) 
			break; // this is already focused, no need to go any deeper
	}

	// if we have reached the root and nothing found
	// it means the previous focus is inside some other window
	if (t == 0)
		needActivate = true;

	if (!needActivate)
	{
		// we have found the parent that was focused and will remain focused
		// we have to unfocus the previous focused chain of children
		Children cunf;
		for (;;)
		{
			t = t->GetChild(StateFocused|StateOldFocus);
			if (t == 0)
				break;
			cunf.push_back(t);
		}
		for (Children::iterator v=cunf.end(); v!=cunf.begin(); )
		{
			--v;
			(*v)->StateSetReset(0, StateFocused|StateOldFocus);
			(*v)->HandleFocused(false);
		}
	}

	// now focus our own new chain of focusables
	if (needActivate)
	{
		for ( t=this; t!=0; t=t->parent)
		{
			if (0 != (t->GetState() & StateOldFocus))
			 	break;
			t->StateSetReset(StateOldFocus, 0);
		}
		// rely on the OS to call us back for the callbacks and all
		Window w = GetParentWindow();
		if (w != 0)
			w.BringToFront();
	}
	else
	{
		Children cfoc;
		for ( t=this; t!=0; t=t->parent)
		{
			if (t->IsFocused())
			 	break;
			cfoc.push_back(t);
		}
		for (Children::iterator v=cfoc.begin(); v!=cfoc.end(); ++v)
		{
			(*v)->StateSetReset(StateFocused, 0);
			(*v)->HandleFocused(true);
		}
	}
}

void View::HandleFocused(bool on)
{
/*
UnitTest::Log("%s %08X, %s, [%d,%d]-(%dx%d)",
	on ? "Focused" : "Unfocused",
	this, 
	HasChildren()?"has children":"nochildren",
	bounds.x, bounds.y, bounds.w, bounds.h);
*/

//	if (!HasChildren()) // containers don't care
	{
		if (on)
			SetCaret(Rect2D()); // remove it, text view will override
		Refresh();
	}
}
void View::HandleActivated(bool )
{
}

void View::MouseCursorSetup()
{
	Window wnd = GetParentWindow();
	if (wnd == 0)
		return;
	WinMgr& mgr = wnd.GetWinMgr();
	if (mgr == 0)
		return;
	wnd.SetMouseCursor(mgr.data->arrowCursor);
}

void View::MouseEnter(bool enter)
{
	if (enter)
		MouseCursorSetup();
		
	if (enter != IsHovered())
	{
	   if (enter) StateSetReset(StateHovered, 0);
	   else		  StateSetReset(0, StateHovered);
		// terminal views are refreshed by hover
		if (children.empty())
			Refresh();  
	}
}

bool View::ContainsPoint(const Point2D& where) const
{
	if (hidden)
		return false;
	return bounds.Contains(where);
}

bool View::FindChildAt(Point2D& where, View::owner& view)
{
	Children::iterator b = children.end();
	Children::iterator e = children.begin();
	while(b!=e)
	{
		owner& child = * --b;
		if (child->IsHidden())
			continue;
		Rect2D cbounds(child->GetBounds());
		if (cbounds.Contains(where))
		{
			where.x -= cbounds.x;
			where.y -= cbounds.y;
			view = child;
			return true;
		}
	}

	if (!hidden && GetLocalBounds().Contains(where))
		view = this;
	return false;
}

bool View::FindDeepChildAt(Point2D& where, View::owner& view)
{
	bool found = false;
	View::owner v(this);
	while (v->FindChildAt(where, view))
	{
		found = true;
		if (v == view)
			break;
		v = view;
	}
	return found;
}

void View::DrawDeep(Graphics& g, const Rect2D& area)
{
	if (hidden)
		return;
	Rect2D clip(area.Intersect(GetLocalBounds()));
	if (clip.isEmpty())
		return;
	Rect2D oldclip = g.GetClipRect(); //??
	g.SetClipRect(clip);			  //??
	// draw this views personal look first
	Draw(g, clip);
	// draw all the children	
	Point2D origin(g.GetOrigin());
	Children::iterator b = children.begin();
	Children::iterator e = children.end();
	for(; b!=e; ++b)
	{
		owner& child = *b;
		Rect2D childArea = clip.Intersect(child->bounds);
		if (childArea.isEmpty())
			continue;		
		Point2D childOrigin = child->bounds.GetNW();
		childArea.OffsetBy(-childOrigin.x, -childOrigin.y);
		g.SetOrigin(origin + childOrigin);
		child->DrawDeep(g, childArea);
		g.SetOrigin(origin);
	}
	g.SetClipRect(oldclip);			  //??
}

void View::Draw(Graphics& g, const Rect2D& area)
{
	// the plain view is a container of other views 
	// and has no looks of its own, just maybe a skin
	ViewSkin::owner skin = GetSkin();
	if (skin != 0)
	{
		StateSkin::owner stateSkin = skin->GetStateSkin(GetState());
		if (stateSkin != 0)
		{
			Rect2D bounds = GetLocalBounds();
			stateSkin->Draw(g, bounds,area);
		}
	}
}

void View::SetFont(const TextFont& f)
{	
	font = f;
	Refresh();
}

void View::SetSkin(const ViewSkin::owner& s)
{	
	skin = s;
	Refresh();
}

ViewSkin::owner View::GetSkinPartOrDefault(const String& s) 
{ 
	if (skin != 0)
	{
		ViewSkin::owner sk = skin->GetPart(s); 
		if (sk != 0)
			return sk;

		sk = skin->GetPart(String::kEmpty); // try empty string part (default)
		if (sk != 0)
			return sk;
	}
	return ViewSkin::owner(0); // failed
}

// default implementation for the drop handler is to do nothing 
void View::HandleDrop(DragDropType::TypeSet& /*availableTypes*/, 
					  DragDropType::Effect&  effect,
					  const Point2D& 		 /*mousePos*/,
					  bool 					 /*onlySimulate*/,
					  DragDropType::FatData& /*theFatData*/ )
{
	effect = 0; // this handler can do nothing 
}
bool View::PasteFromClipboard(DragDropType::TypeSet& availableTypes, 
							  bool 					 onlySimulate,
							  DragDropType::FatData& theFatData )
{
	Window wnd = GetParentWindow();
	if (wnd == 0)
		return false;
//	WinMgr& mgr = wnd.GetWinMgr();
//	if (mgr == 0)
//		return false;
	return wnd.PasteFromClipboard(availableTypes, onlySimulate, theFatData);
}

bool View::CopyToClipboard(DragDropType::TypeSet& availableTypes, 
						   DragDropType::FatData& theFatData )
{
	Window wnd = GetParentWindow();
	if (wnd == 0)
		return false;
//	WinMgr& mgr = wnd.GetWinMgr();
//	if (mgr == 0)
//		return false;
	return wnd.CopyToClipboard(availableTypes, theFatData);
}


} // namespace XSP
