/*
 * Copyright (c) 2001-2005 Servertec. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS NOTICE MUST NOT BE ALTERED NOR REMOVED.
 *
 * CopyrightVersion 1.0
 */

import java.io.File;

import java.util.Hashtable;
import java.util.Enumeration;

import stec.iws.Utils;
import stec.iws.iws;

import stec.pos.ObjectStore;
import stec.pos.ObjectStoreReference;
import stec.pos.Index;
import stec.pos.IndexReference;
import stec.pos.ObjectStoreEnumeration;

public class Indexer implements Runnable
{
	Hashtable paths;
	String[] exclude_list;
	Hashtable parsers;

	private Thread thread;
	private long interval;
	private boolean running;

	private ObjectStoreReference documents_store_reference;
	private ObjectStoreReference docids_store_reference;
	private ObjectStoreReference words_store_reference;
	private ObjectStoreReference wordids_store_reference;
	private IndexReference documents_index_reference;
	private IndexReference words_index_reference;

	public Indexer(long interval, Hashtable paths, String[] parsers, String[] exclude_list, int document_count, int word_count) throws Exception
	{
		this.interval = interval * (60L * 1000L);
		this.paths = paths;
		this.exclude_list = exclude_list;
		this.parsers = new Hashtable();

		String value;
		int offset;
		String extension;
		String class_name;
		Parser parser;
		Class cls;

		int count = parsers.length;

		for(int i = 0; i < count; i++)
		{
			value = parsers[i];
			offset = value.indexOf(':');
			if(offset == -1)
			{
				throw new Exception("parser is not valid: " + value);
			}

			extension = value.substring(0, offset).trim();
			class_name = value.substring(offset + 1).trim();
			cls = Class.forName(class_name);
			parser = (Parser)cls.newInstance();

			this.parsers.put(extension, parser);
		}

		String basedir = iws.getBaseDirectory();

		String filename = Utils.concatPaths(basedir, "/search/data/documents");
		if(!ObjectStore.exists(filename))
		{
			ObjectStore.create(filename, 64, document_count, -1, -1, false);
		}

		filename = Utils.concatPaths(basedir, "/search/data/docids");
		if(!ObjectStore.exists(filename))
		{
			ObjectStore.create(filename, 48, document_count, -1, -1, false);
		}

		filename = Utils.concatPaths(basedir, "/search/data/words");
		if(!ObjectStore.exists(filename))
		{
			ObjectStore.create(filename, 16, word_count, -1, -1, false);
		}

		filename = Utils.concatPaths(basedir, "/search/data/wordids");
		if(!ObjectStore.exists(filename))
		{
			ObjectStore.create(filename, 16, word_count, -1, -1, false);
		}

		filename = Utils.concatPaths(basedir, "/search/data/documents");
		if(!Index.exists(filename))
		{
			Index.create(filename, -1, -1, "java.lang.Integer", -1, -1, "java.lang.Integer", -1, -1, null, false, true, true, 2, -1, null, true);
		}

		filename = Utils.concatPaths(basedir, "/search/data/words");
		if(!Index.exists(filename))
		{
			Index.create(filename, -1, -1, "java.lang.Integer", -1, -1, "java.lang.Integer", -1, -1, null, false, true, true, 2, -1, null, true);
		}

		filename = Utils.concatPaths(basedir, "/search/data/documents");
		documents_store_reference = ObjectStore.open(filename, false);

		filename = Utils.concatPaths(basedir, "/search/data/docids");
		docids_store_reference = ObjectStore.open(filename, false);

		filename = Utils.concatPaths(basedir, "/search/data/words");
		words_store_reference = ObjectStore.open(filename, false);

		filename = Utils.concatPaths(basedir, "/search/data/wordids");
		wordids_store_reference = ObjectStore.open(filename, false);

		filename = Utils.concatPaths(basedir, "/search/data/documents");
		documents_index_reference = Index.open(filename, false);

		filename = Utils.concatPaths(basedir, "/search/data/words");
		words_index_reference = Index.open(filename, false);

		if(interval > 0)
		{
			thread = new Thread(this);
			thread.start();
		}
	}

	public void run()
	{
		try
		{
			running = true;

			while(running)
			{
				try
				{
					reindex();
				}
				catch(Exception ex)
				{
					ex.printStackTrace();
				}

				try
				{
					Thread.sleep(interval);
				}
				catch(InterruptedException ex) {}
			}
		}
		finally
		{
			if(documents_store_reference != null)
			{
				try
				{
					documents_store_reference.close();
				}
				catch(Exception ex) {}
			}

			if(docids_store_reference != null)
			{
				try
				{
					docids_store_reference.close();
				}
				catch(Exception ex) {}
			}

			if(words_store_reference != null)
			{
				try
				{
					words_store_reference.close();
				}
				catch(Exception ex) {}
			}

			if(wordids_store_reference != null)
			{
				try
				{
					wordids_store_reference.close();
				}
				catch(Exception ex) {}
			}

			if(documents_index_reference != null)
			{
				try
				{
					documents_index_reference.close();
				}
				catch(Exception ex) {}
			}

			if(words_index_reference != null)
			{
				try
				{
					words_index_reference.close();
				}
				catch(Exception ex) {}
			}
		}
	}

	public void shutdown()
	{
		running = false;
	}

	private void remove() throws Exception
	{
		String relative_filename;
		File file;
		Document document;
		Integer iid;

		ObjectStoreEnumeration e = documents_store_reference.keys();
		while(e.hasMoreElements())
		{
			if(!running)
			{
				break;
			}

			relative_filename = (String)e.nextElement();

			if(!relative_filename.equals("@COUNTER"))
			{
				document = (Document)documents_store_reference.get(relative_filename);

				file = new File(document.path);
				if(!file.exists())
				{
					iid = new Integer(document.id);

					documents_store_reference.remove(relative_filename);
					docids_store_reference.remove(iid);

					remove(iid);
				}
			}
		}
	}

	private void reindex() throws Exception
	{
		remove();

		String relative_path;
		String absolute_path;

		Enumeration e = paths.keys();
		while(e.hasMoreElements())
		{
			if(!running)
			{
				break;
			}

			relative_path = (String)e.nextElement();
			absolute_path = (String)paths.get(relative_path);

			reindex(relative_path, absolute_path);
		}
	}

	private void reindex(String relative_path, String absolute_path) throws Exception
	{
		Document document;
		String filename;
		File tfile;
		long timestamp;
		long length;
		String relative_filename;
		String absolute_filename;
		String extension;
		Parser parser;
		Object obj;
		int id;
		Integer iid;

		File file = new File(absolute_path);

		String[] files = file.list();

		int count = files.length;

		for(int i = 0; i < count; i++)
		{
			if(!running)
			{
				break;
			}

			filename = files[i];

			tfile = new File(file, filename);

			relative_filename = Utils.concatURIs(relative_path, filename);

			if(tfile.isDirectory())
			{
				reindex(relative_filename, tfile.getCanonicalPath());
			}
			else
			{
				extension = getExtension(filename);

				obj = parsers.get(extension);
				if(obj != null)
				{
					parser = (Parser)obj;

					timestamp = tfile.lastModified();
					length = tfile.length();

					if(documents_store_reference.containsKey(relative_filename))
					{
						document = (Document)documents_store_reference.get(relative_filename);

						if(document.timestamp != timestamp || document.length != length)
						{
							document.timestamp = timestamp;
							document.length = length;

							documents_store_reference.put(relative_filename, document);

							iid = new Integer(document.id);

							remove(iid);

							add(parser, tfile, iid);
						}
					}
					else
					{
						if(documents_store_reference.containsKey("@COUNTER"))
						{
							id = ((Integer)documents_store_reference.get("@COUNTER")).intValue();
						}
						else
						{
							id = -1;
						}

						id++;

						iid = new Integer(id);

						documents_store_reference.put("@COUNTER", iid);

						absolute_filename = Utils.concatURIs(absolute_path, filename);

						document = new Document(absolute_filename, id, timestamp, length);

						documents_store_reference.put(relative_filename, document);

						docids_store_reference.put(iid, relative_filename);

						add(parser, tfile, iid);
					}
				}
			}
		}
	}

	private void add(Parser parser, File file, Integer doc_id) throws Exception
	{
		parser.parse(file.getCanonicalPath());

		String word;
		int id;
		Integer iid;

		while((word = parser.nextWord()) != null)
		{
			if(exclude(word))
			{
				continue;
			}

			if(words_store_reference.containsKey(word))
			{
				iid = (Integer)words_store_reference.get(word);
			}
			else
			{
				if(words_store_reference.containsKey("@COUNTER"))
				{
					id = ((Integer)words_store_reference.get("@COUNTER")).intValue();
				}
				else
				{
					id = -1;
				}

				id++;

				iid = new Integer(id);

				words_store_reference.put("@COUNTER", iid);

				words_store_reference.put(word, iid);

				wordids_store_reference.put(iid, word);
			}

			if(!documents_index_reference.contains(doc_id, iid))
			{
				documents_index_reference.add(doc_id, iid);
			}

			if(!words_index_reference.contains(iid, doc_id))
			{
				words_index_reference.add(iid, doc_id);
			}
		}
	}

	private void remove(Integer doc_id) throws Exception
	{
		if(documents_index_reference.setCurrentKey(doc_id))
		{
			Integer iid;

			Object obj = documents_index_reference.getCurrentValue();

			do
			{
				iid = (Integer)obj;

				words_index_reference.remove(iid, doc_id);
			}
			while((obj = documents_index_reference.nextValue(false)) != null);

			documents_index_reference.remove(doc_id);
		}
	}

	private String getExtension(String filename)
	{
		int offset = filename.lastIndexOf('.');
		if(offset == -1)
		{
			return "";
		}

		return filename.substring(offset);
	}

	private boolean exclude(String word)
	{
		if(exclude_list == null)
		{
			return false;
		}

		int count = exclude_list.length;

		for(int i = 0; i < count; i++)
		{
			if(exclude_list[i].equals(word))
			{
				return true;
			}
		}

		return false;
	}
}