/* -*- c -*-
 *
 * Copyright 1989 Massachusetts Institute of Technology
 * Copyright 1991 Leland Stanford Junior University
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * List.c - List widget
 *
 * This is the List widget, it is useful to display a list, without the
 * overhead of having a widget for each item in the list.  It allows 
 * the user to select an item in a list and notifies the application through
 * a callback function.
 *
 *	Original Created: 	8/13/88
 *	       By:		Chris D. Peterson
 *                              MIT X Consortium
 *
 *      Rewritten  by:  Kevin Brock
 *                      Symbolic Systems Resources Group
 *		        Knowledge Systems Laboratory
 *		        Departments of Computer Science and Medicine
 *		        Stanford University
 *		        Stanford, CA 94305
 *
 *        THIS WIDGET IS *NOT* COMPATIBLE WITH THE CURRENT Xaw List Widget.
 *        PROGRAMS DESIGNED TO USE THE ORIGINAL WIDGET MUST BE MODIFIED IN
 *        A NUMBER OF WAYS BEFORE THEY CAN USE THIS WIDGET.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include <X11/Xmu/Drawing.h>

#include <X11/Xaw/XawInit.h>
#include "ListP.h"

/* 
 * Default Translation table.
 */

/*  You may have *either* a double click translation 
 *  *or* an unmodified motion event ( e.g. <Btn1Motion> ),
 *  but NOT BOTH in this translation table.
 */
static char defaultTranslations[] =  
  "Shift<Btn1Down>:       ListToggle()                     \n\
   <Btn1Down>:            ListSet()                        \n\
   Shift <Btn1Motion>:    ListExtendSelect()               \n\
   <Btn1Motion>:          ListMoveSelect()                 \n\
   <Btn1Up>:              ListNotify() ListCheckExtend()   \n\
   <Leave>:               ListCheckExtend()";

/****************************************************************
 *
 * Full class record constant
 *
 ****************************************************************/

/* Private Data */

#define offset(field) XtOffset(XbListWidget, field)

static XtResource resources[] = {
    {XtNrowHeight, XtCRowHeight, XtRDimension, sizeof(Dimension),
	offset(list.row_height), XtRImmediate, (void*) 0},
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	offset(list.foreground), XtRString, "XtDefaultForeground"},
    {XtNcursor, XtCCursor, XtRCursor, sizeof(Cursor),
       offset(simple.cursor), XtRString, "left_ptr"},
    {XtNemphaticFont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	offset(list.emphaticFont),XtRString, "XtDefaultFont"},
    {XtNnormalFont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	offset(list.normalFont),XtRString, "XtDefaultFont"},
    {XtNlist, XtCList, XtRPointer, sizeof(XbListSelectionPtr),
	 offset(list.list), XtRString, NULL},
    {XtNdefaultColumns, XtCColumns, XtRInt,  sizeof(int),
	offset(list.default_cols), XtRImmediate, (caddr_t)2},
    {XtNlongest, XtCLongest, XtRInt,  sizeof(int),
	offset(list.longest), XtRImmediate, (caddr_t)0},
    {XtNnumberStrings, XtCNumberStrings, XtRInt,  sizeof(int),
	offset(list.nitems), XtRImmediate, (caddr_t)0},
    {XtNpasteBuffer, XtCBoolean, XtRBoolean,  sizeof(Boolean),
	offset(list.paste), XtRString, (caddr_t) "False"},
    {XtNforceColumns, XtCColumns, XtRBoolean,  sizeof(Boolean),
	offset(list.force_cols), XtRString, (caddr_t) "False"},
    {XtNverticalList, XtCBoolean, XtRBoolean,  sizeof(Boolean),
	offset(list.vertical_cols), XtRString, (caddr_t) "False"},
    {XtNinternalWidth, XtCWidth, XtRDimension,  sizeof(Dimension),
	offset(list.internal_width), XtRImmediate, (caddr_t)4},
    {XtNinternalHeight, XtCHeight, XtRDimension, sizeof(Dimension),
	offset(list.internal_height), XtRImmediate, (caddr_t)2},
    {XtNcolumnSpacing, XtCSpacing, XtRDimension,  sizeof(Dimension),
	offset(list.column_space), XtRImmediate, (caddr_t)6},
    {XtNrowSpacing, XtCSpacing, XtRDimension,  sizeof(Dimension),
	offset(list.row_space), XtRImmediate, (caddr_t)2},
    {XtNcallback, XtCCallback, XtRCallback, sizeof(caddr_t),
        offset(list.callback), XtRCallback, NULL},
    {XtNdoubleClick, XtCDoubleClick, XtRCallback, sizeof(caddr_t),
       offset(list.double_click), XtRCallback, NULL},
    {XtNnumSelected, XtCNumSelected, XtRInt,  sizeof(int),
	offset(list.num_selected), XtRImmediate, (caddr_t)0},
    {XtNclickDelay, XtCClickDelay, XtRInt, sizeof(long int),
       offset(list.double_click_delay), XtRImmediate, (caddr_t) 500},
};

static void Initialize();
static void ChangeSize();
static void Resize();
static void Redisplay();
static void Destroy();
static Boolean Layout();
static XtGeometryResult PreferredGeom();
static Boolean SetValues();
static void Notify(), Set(), Unset();

static void Toggle();
static void ExtendSelect();
static void CheckExtend();
static void MoveSelect();

static void SelectRange(/* Widget, int, int */);
static void UnSelectRange(/* Widget, int, int */);

static void XbAddToSelectList(/*  Widget, int  */);
static void XbRemoveFromSelectList(/*  Widget, int  */);

static void XbClearSelectList(/*  Widget  */);

static int CvtToPosition(/*w, xloc, yloc, item */);

static XtActionsRec actions[] = {
      {"ListNotify",         Notify},
      {"ListSet",            Set},
      {"ListUnset",          Unset},
      {"ListToggle",         Toggle},
      {"ListExtendSelect",   ExtendSelect},
      {"ListCheckExtend",    CheckExtend},
      {"ListMoveSelect",     MoveSelect},
      {NULL,NULL}
};

XbListClassRec XblistClassRec = {
  {
/* core_class fields */	
#define superclass		(&simpleClassRec)
    /* superclass	  	*/	(WidgetClass) superclass,
    /* class_name	  	*/	"XbList",
    /* widget_size	  	*/	sizeof(XbListRec),
    /* class_initialize   	*/	XawInitializeWidgetSet,
    /* class_part_initialize	*/	NULL,
    /* class_inited       	*/	FALSE,
    /* initialize	  	*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize		  	*/	XtInheritRealize,
    /* actions		  	*/	actions,
    /* num_actions	  	*/	XtNumber(actions),
    /* resources	  	*/	resources,
    /* num_resources	  	*/	XtNumber(resources),
    /* xrm_class	  	*/	NULLQUARK,
    /* compress_motion	  	*/	TRUE,
    /* compress_exposure  	*/	FALSE,
    /* compress_enterleave	*/	FALSE, /* Brock 10-30-90 */
    /* visible_interest	  	*/	FALSE,
    /* destroy		  	*/	Destroy,
    /* resize		  	*/	Resize,
    /* expose		  	*/	Redisplay,
    /* set_values	  	*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus	 	*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private   	*/	NULL,
    /* tm_table		   	*/	defaultTranslations,
    /* query_geometry		*/      PreferredGeom,
  },
/* Simple class fields initialization */
  {
    /* change_sensitive		*/	XtInheritChangeSensitive
  }
};

WidgetClass XblistWidgetClass = (WidgetClass)&XblistClassRec;

/****************************************************************
 *
 * Private Procedures
 *
 ****************************************************************/

static void Destroy(w)
     Widget w;
{
    XbListWidget lw = (XbListWidget) w;    

    XbListFreeSelection(lw->list.list);
}

static void GetGCs(w)
Widget w;
{
    XGCValues	values;
    XbListWidget lw = (XbListWidget) w;    

    values.foreground	= lw->list.foreground;
    values.font		= lw->list.normalFont->fid;
    lw->list.normgc = XtGetGC(w, (unsigned) GCForeground | GCFont, &values);

    values.foreground	= lw->core.background_pixel;
    lw->list.revgc = XtGetGC(w, (unsigned) GCForeground | GCFont, &values);

    values.foreground	= lw->list.foreground;
    values.font		= lw->list.emphaticFont->fid;
    lw->list.boldgc = XtGetGC(w, (unsigned) GCForeground | GCFont, &values);

    values.foreground	= lw->core.background_pixel;
    lw->list.revboldgc = XtGetGC(w, (unsigned) GCForeground | GCFont, &values);

    values.tile       = XmuCreateStippledPixmap(XtScreen(w), 
						lw->list.foreground,
						lw->core.background_pixel,
						lw->core.depth);
    values.fill_style = FillTiled;
}

/*	Function Name: ResetList
 *	Description: Resets the new list when important things change.
 *	Arguments: w - the widget.
 *                 changex, changey - allow the height or width to change?
 *	Returns: none.
 */

static void ResetList(w, changex, changey)
     Widget w;
     Boolean changex, changey;
{
    XbListWidget lw = (XbListWidget) w;
    Dimension width = w->core.width;
    Dimension height = w->core.height;
    register int i, len;
    
    /*
     * If list is NULL then the list will just be the name of the widget.
     *
     */

    if(lw->list.list == NULL)
    {
	lw->list.list = (XbListSelectionPtr) XtMalloc (sizeof(XbListSelection));
	lw->list.list->size = 0;
	lw->list.list->entries = NULL;
    }
	    
    lw->list.nitems = lw->list.list->size;
    if (lw->list.list->entries == NULL) 
    {
        char tmp[512];

	lw->list.list->entries = (XbListEntryPtr)XtMalloc(sizeof(XbListEntry));
	lw->list.list->entries[0].entry = XtMalloc(strlen(lw->core.name)+1);
	strcpy(lw->list.list->entries[0].entry,lw->core.name);
	lw->list.list->entries[0].index = 0;
	lw->list.list->entries[0].bold = 0;
	lw->list.list->entries[0].invert = 0;
	lw->list.list->size = 1; 

	sprintf(tmp, "\nResetList @ 0x%x size 1", lw->list.list->entries);
	ximap_log(tmp);

    }
    else
    {
	int j = 0;
	
	for ( ; j < lw->list.list->size; j++ )
	{
	    if( lw->list.list->entries[j].invert )
		lw->list.num_selected++;
	}
    }
    
    if (lw->list.longest == 0) /* Get column width. */
	for ( i = 0 ; i < lw->list.list->size; i++) 
	{
	    if(!lw->list.list->entries[i].bold)
	    {
		len = XTextWidth(lw->list.normalFont, 
				 lw->list.list->entries[i].entry,
				 strlen(lw->list.list->entries[i].entry));
	    }
	    else
	    {
		len = XTextWidth(lw->list.emphaticFont, 
				 lw->list.list->entries[i].entry,
				 strlen(lw->list.list->entries[i].entry));
	    }
	    if (len > lw->list.longest)
		lw->list.longest = len;
	}
    
    lw->list.col_width = lw->list.longest + lw->list.column_space;
    if( lw->list.col_width <= 0 )
	lw->list.col_width = 1;
    
    if (Layout(w, changex, changey, &width, &height))
	ChangeSize(w, width, height);
}

/*	Function Name: ChangeSize.
 *	Description: Laysout the widget.
 *	Arguments: w - the widget to try change the size of.
 *	Returns: none.
 */

static void ChangeSize(w, width, height)
     Widget w;
     Dimension width;
     Dimension height;
{
    XtWidgetGeometry request, reply;
    
    int widthEqual = 0, heightEqual = 0;

    request.request_mode = CWWidth | CWHeight;
    request.width = width;
    request.height = height;
    
    switch ( XtMakeGeometryRequest(w, &request, &reply) ) 
    {
      case XtGeometryYes:
      case XtGeometryNo:
        break;

      case XtGeometryAlmost:
	if(reply.request_mode & CWWidth)
	    widthEqual = (request.width != reply.width);
	else
	    widthEqual = 1;

	if(reply.request_mode & CWHeight)
	    heightEqual = (request.height != reply.height);
	else 
	    heightEqual = 1;

	Layout(w, (request.height != reply.height), (request.width != reply.width),
	       (unsigned short*) &(reply.width), 
	       (unsigned short*) &(reply.height));

	request = reply;
	switch (XtMakeGeometryRequest(w, &request, &reply) ) 
	{
	  case XtGeometryYes:
	  case XtGeometryNo:
	    break;
	  case XtGeometryAlmost:
	    request = reply;
	    if (Layout(w, FALSE, FALSE,
		       &(request.width), &(request.height))) 
	    {
		char buf[BUFSIZ];
		sprintf(buf, "XbList Widget: %s %s",
			"Size Changed when it shouldn't have",
			"when computing layout");
		XtAppWarning(XtWidgetToApplicationContext(w), buf);
	    }
	    request.request_mode = CWWidth | CWHeight;
	    XtMakeGeometryRequest(w, &request, &reply);
	    break;

	default:
	  XtAppWarning(XtWidgetToApplicationContext(w),
		       "List Widget: Unknown geometry return.");
	  break;
	}
	break;

    default:
	XtAppWarning(XtWidgetToApplicationContext(w),
		     "XbList Widget: Unknown geometry return.");
	break;
    }
}

/*	Function Name: Initialize
 *	Description: Function that initilizes the widget instance.
 *	Arguments: junk - NOT USED.
 *                 new  - the new widget.
 *	Returns: none
 */

/* ARGSUSED */
static void Initialize(junk, new)
     Widget junk;
     Widget new;
{
    XbListWidget lw = (XbListWidget) new;

/* 
 * Initialize all private resources.
 *
 */

    GetGCs(new);

    /* Set row height. */
    lw->list.row_height = lw->list.normalFont->max_bounds.ascent
			+ lw->list.normalFont->max_bounds.descent
			+ lw->list.row_space;

    lw->list.num_selected = 0;

    ResetList(new, (new->core.width == 0), (new->core.height == 0));

    lw->list.inExtend = FALSE;
    lw->list.inMoveSelect = FALSE;
    lw->list.origin = XB_OUT_OF_RANGE;
    lw->list.last = XB_OUT_OF_RANGE;
    lw->list.direction = XB_NONE;
    lw->list.last_click = 0;
    
} /* Initialize */

/*	Function Name: CvtToItem
 *	Description: Converts Xcoord to item number of item containing that
 *                   point.
 *	Arguments: w - the list widget.
 *                 xloc, yloc - x location, and y location.
 *	Returns: the item number.
 */

static int CvtToItem(w, x, y, item)
     Widget w;
     Position x, y;
     int *item;
{
    int one, another;
    XbListWidget lw = (XbListWidget) w;
    int ret_val = XB_OKAY;
    
    if(lw->list.col_width > 0 && lw->list.row_height > 0)
    {
	if (lw->list.vertical_cols)
	{
	    one = 
		(x?(lw->list.nrows * ((x - (int) lw->list.internal_width) / (int) lw->list.col_width)):x);
	    another = 
		(y?((y - (int) lw->list.internal_height) / (int) lw->list.row_height):y);

	    /* If out of range, return minimum possible value. */
	    if (another >= lw->list.nrows) 
	    {
		another = lw->list.nrows - 1;
		ret_val = XB_OUT_OF_RANGE;
	    }
	}
	else 
	{
	    one = 
		(y?(lw->list.ncols * ((y-(int) lw->list.internal_height)/(int) lw->list.row_height)):y) ;

	    /* If in right margin handle things right. */
	    another = (x?((x - (int) lw->list.internal_width) / (int) lw->list.col_width):x);

	    if (another >= lw->list.ncols) 
	    {
		another = lw->list.ncols - 1; 
		ret_val = XB_OUT_OF_RANGE;
	    }
	}  

	if ((x < 0) || (y < 0))
	    ret_val = XB_OUT_OF_RANGE;
	
	if (one < 0) 
	    one = 0;
	if (another < 0) 
	    another = 0;
	
	*item = one + another;
	
	if (*item >= lw->list.nitems) 
	    return(XB_OUT_OF_RANGE);
    }
    else
    {
	ret_val = XB_OUT_OF_RANGE;
    }

    return(ret_val);
}

/*	Function Name: FindCornerItems.
 *	Description: Find the corners of the rectangle in item space.
 *	Arguments: w - the list widget.
 *                 event - the event structure that has the rectangle it it.
 *                 ul_ret, lr_ret - the corners ** RETURNED **.
 *	Returns: none.
 */

FindCornerItems(w, event, ul_ret, lr_ret)
Widget w;
XEvent * event;
int *ul_ret, *lr_ret;
{
    Position xloc, yloc;

    xloc = event->xexpose.x;
    yloc = event->xexpose.y;
    CvtToItem(w, xloc, yloc, ul_ret);
    xloc += event->xexpose.width;
    yloc += event->xexpose.height;
    CvtToItem(w, xloc, yloc, lr_ret);
}

/*	Function Name: ItemInRectangle
 *	Description: returns TRUE if the item passed is in the given rectangle.
 *	Arguments: w - the list widget.
 *                 ul, lr - corners of the rectangle in item space.
 *                 item - item to check.
 *	Returns: TRUE if the item passed is in the given rectangle.
 */
    
ItemInRectangle(w, ul, lr, item)
     Widget w;
     int ul;
     int lr;
     int item;
{
    XbListWidget lw = (XbListWidget) w;
    register int mod_item;
    int things;
    
    if (item < ul || item > lr) 
        return(FALSE);

    if (lw->list.vertical_cols)
        things = lw->list.nrows;
    else
        things = lw->list.ncols;

    mod_item = item % things;

    if ( (mod_item >= ul % things) && (mod_item <= lr % things ) )
        return(TRUE);

    return(FALSE);
}

/*	Function Name: HighlightBackground
 *	Description: paints the color of the background for the given item.
 *	Arguments: w - the widget.
 *                 x, y - ul corner of the area item occupies.
 *                 item - the item we are dealing with.
 *                 gc - the gc that is used to paint this rectangle
 *	Returns: 
 */

HighlightBackground(w, x, y, item, gc)
     Widget w;
     Position x, y;
     int item;
     GC gc;
{
    XbListWidget lw = (XbListWidget) w;

    Position hl_x, hl_y;
    unsigned short width, height;

    hl_x = x - lw->list.column_space/2;
    hl_y = y - lw->list.row_space/2;

    width = lw->list.longest;
    height = lw->list.row_height + lw->list.row_space;

    XFillRectangle(XtDisplay(w), XtWindow(w), gc, hl_x, hl_y, width, height);
}

/*      Function Name: IsHighlighted
 *      Description:   returns TRUE if item is highlighted, else FALSE
 */
Boolean IsHighlighted(w,item)
     Widget w;
     int item;
{
  XbListWidget lw = (XbListWidget) w;

  return( lw->list.list->entries[item].invert );
}
/*
 *      Function Name: XbListCountHighlighted
 *      Returns number of highlighted items
 *      Arguments: w - the list widget
 */
int
XbListCountHighlighted(w)
     Widget w;
{
     XbListWidget lw = (XbListWidget)w;
     int count = 0, i;
     XbListSelectionPtr lst = lw->list.list;
     int size;
     if (!lst) return 0;

     size  = lst->size;
     for (i = 0; i < size; ++i)
       if (lst->entries[i].invert)
	 ++count;
     return count;
}
       
  
/*	Function Name: PaintItemName
 *	Description: paints the name of the item in the appropriate location.
 *	Arguments: w - the list widget.
 *                 item - the item to draw.
 *	Returns: none.
 *
 *      NOTE: no action taken on an unrealized widget.
 */

PaintItemName(w, item)
     Widget w;
     int item;
{
    char * str;
    GC gc;
    Position x, y, str_y;
    XbListWidget lw = (XbListWidget) w;

    if (!XtIsRealized(w)) return; /* Just in case... */

    if ( item != XB_OUT_OF_RANGE )
    {
	if (lw->list.vertical_cols) 
	{
	    x = lw->list.col_width * (item / lw->list.nrows) + lw->list.internal_width;
	    y = lw->list.row_height * (item % lw->list.nrows) + lw->list.internal_height;
	}
	else 
	{
	    x = (int) lw->list.col_width * (item % lw->list.ncols) + lw->list.internal_width;
	    y = lw->list.row_height * (item / lw->list.ncols) + lw->list.internal_height;
	}

	if(lw->list.list->entries[item].bold)
	    str_y = y + lw->list.emphaticFont->max_bounds.ascent;
	else
	    str_y = y + lw->list.normalFont->max_bounds.ascent;
	
	if( !lw->list.list->entries[item].bold )
	{
	    if (IsHighlighted(w,item)) 
	    {
		gc = lw->list.revgc;
		HighlightBackground(w, x, y, item, lw->list.normgc);
	    }
	    else
	    {
		gc = lw->list.normgc;
		HighlightBackground(w, x, y, item, lw->list.revgc);
	    }
	}
	else
	{
	    if (IsHighlighted(w,item)) 
	    {
		gc = lw->list.revboldgc;
		HighlightBackground(w, x, y, item, lw->list.boldgc);
	    }
	    else
	    {
		gc = lw->list.boldgc;
		HighlightBackground(w, x, y, item, lw->list.revboldgc);
	    }
	}

	str =  lw->list.list->entries[item].entry;	/* draw it */
	XDrawString(XtDisplay(w), XtWindow(w), gc, x, str_y, str, strlen(str));
    }
}
    
/*	Function Name: Redisplay
 *	Description: Repaints the widget window on expose events.
 *	Arguments: w - the list widget.
 *                 event - the expose event for this repaint.
 *                 junk - NOT USED.
 *	Returns: 
 */

/* ARGSUSED */
static void Redisplay(w, event, junk)
     Widget w;
     XEvent *event;
     Region junk;
{
    int item;			/* an item to work with. */
    int ul_item, lr_item;       /* corners of items we need to paint. */
    XbListWidget lw = (XbListWidget) w;
    
    if (event == NULL) 	/* repaint all. */
    {
        ul_item = 0;
	lr_item = lw->list.nrows * lw->list.ncols - 1;
	XClearWindow(XtDisplay(w), XtWindow(w));
	for (item = 0; item <= lr_item && item < lw->list.nitems ; item++)
	    PaintItemName(w, item);
    }
    else
    {
        FindCornerItems(w, event, &ul_item, &lr_item);
	for (item = ul_item; (item <= lr_item && item < lw->list.nitems) ; item++)
	{
	    if (ItemInRectangle(w, ul_item, lr_item, item))
	    {
		PaintItemName(w, item);
	    }
	}
    }
}

/*	Function Name: PreferredGeom
 *	Description: This tells the parent what size we would like to be
 *                   given certain constraints.
 *	Arguments: w - the widget.
 *                 intended - what the parent intends to do with us.
 *                 requested - what we want to happen.
 *	Returns: none.
 */

static XtGeometryResult PreferredGeom(w, intended, requested)
     Widget w;
     XtWidgetGeometry *intended;
     XtWidgetGeometry *requested;
{
    unsigned short new_width, new_height;
    Boolean change, width_req, height_req;
    
    width_req = intended->request_mode & CWWidth;
    height_req = intended->request_mode & CWHeight;
    
    if (width_req)
	new_width = intended->width;
    else
	new_width = w->core.width;
    
    if (height_req)
	new_height = intended->height;
    else
	new_height = w->core.height;
    
    requested->request_mode = 0;
    
    /*
     * We only care about our height and width.
     */
    
    if ( !width_req && !height_req)
	return(XtGeometryYes);
    
    change = Layout(w, !width_req, !height_req, &new_width, &new_height);
    
    requested->request_mode |= CWWidth;
    requested->width = new_width;
    requested->request_mode |= CWHeight;
    requested->height = new_height;
    
    if (change)
        return(XtGeometryAlmost);
    return(XtGeometryYes);
}

/*	Function Name: Resize
 *	Description: resizes the widget, by changing the number of rows and
 *                   columns.
 *	Arguments: w - the widget.
 *	Returns: none.
 */

static void Resize(w)
     Widget w;
{
    unsigned short width, height;
    
    width = w->core.width;
    height = w->core.height;
    
    if (Layout(w, FALSE, FALSE, &width, &height))
    {
	XtAppWarning(XtWidgetToApplicationContext(w),
		     "List Widget: Size changed when it shouldn't have when resising.");
    }
}

/*	Function Name: Layout
 *	Description: lays out the item in the list.
 *	Arguments: w - the widget.
 *                 xfree, yfree - TRUE if we are free to resize the widget in
 *                                this direction.
 *                 width, height - the is the current width and height that 
 *                                 we are going to layout the list widget to,
 *                                 depending on xfree and yfree of course.
 *                               
 *	Returns: TRUE if width or height have been changed.
 */
static Boolean Layout(w, xfree, yfree, width, height)
     Widget w;
     Boolean xfree, yfree;
     unsigned short *width, *height;
{
    XbListWidget lw = (XbListWidget) w;
    Boolean change = FALSE;
    
    /* 
     * If force columns is set then always use number of columns specified
     * by default_cols.
     */
    
    if (lw->list.force_cols) 
    {
	lw->list.ncols = lw->list.default_cols;
	if (lw->list.ncols <= 0) 
	    lw->list.ncols = 1;
	
	/* 12/3 = 4 and 10/3 = 4, but 9/3 = 3 */
	lw->list.nrows = ( ( lw->list.nitems - 1) / lw->list.ncols) + 1 ;
	
	if (xfree) 		/* If allowed resize width. */
	{
	    *width = lw->list.ncols *  lw->list.col_width 
		+ 2 * lw->list.internal_width;
	    change = TRUE;
	}
	if (yfree) 		/* If allowed resize height. */
	{
	    *height = (lw->list.nrows * lw->list.row_height)
		+ 2 * lw->list.internal_height;
	    change = TRUE;
	}
	return(change);
    }
    
    /*
     * If both width and height are free to change the use default_cols
     * to determine the number columns and set new width and height to
     * just fit the window.
     */
    
    if (xfree && yfree) 
    {
	lw->list.ncols = lw->list.default_cols;
	if (lw->list.ncols <= 0) 
	    lw->list.ncols = 1;

	lw->list.nrows = ( ( lw->list.nitems - 1) / lw->list.ncols) + 1 ;
	*width = (lw->list.ncols *  lw->list.col_width) + (2 * lw->list.internal_width);
	*height = (lw->list.nrows * lw->list.row_height) + (2 * lw->list.internal_height);
	change = TRUE;
    }
    /* 
     * If the width is fixed then use it to determine the number of columns.
     * If the height is free to move (width still fixed) then resize the height
     * of the widget to fit the current list exactly.
     */
    else if (!xfree) 
    {
	lw->list.ncols = ( (*width - 2 * lw->list.internal_width)
			  / lw->list.col_width);
	if (lw->list.ncols <= 0) 
	    lw->list.ncols = 1;
	lw->list.nrows = ( ( lw->list.nitems - 1) / lw->list.ncols) + 1 ;
	if ( yfree ) 
	{
	    *height = (lw->list.nrows * lw->list.row_height)
		+ 2 * lw->list.internal_height;
	    change = TRUE;
	}
    }
    /* 
     * The last case is xfree and !yfree we use the height to determine
     * the number of rows and then set the width to just fit the resulting
     * number of columns.
     */
    else if (!yfree) 		/* xfree must be TRUE. */
    {
	lw->list.nrows = (*height - 2 * lw->list.internal_height) 
	    / lw->list.row_height;
	if (lw->list.nrows <= 0) 
	    lw->list.nrows = 1;
	lw->list.ncols = (( lw->list.nitems - 1 ) / lw->list.nrows) + 1;
	*width = (lw->list.ncols * lw->list.col_width) + (2 * lw->list.internal_width);
	change = TRUE;
    }      

    return(change);
}

/*	Function Name: Notify
 *	Description: Notifies the user that a button has been pressed, and
 *                   calles the callback, if the XtNpasteBuffer resource
 *                   is true then the name of the item is also put in the
 *                   X cut buffer ( buf (0) ).
 *	Arguments: w - the widget that the notify occured in.
 *                 event - event that caused this notification.
 *                 params, num_params - not used.
 *	Returns: none.
 */

/* ARGSUSED */
static void Notify(w, event, params, num_params)
     Widget w;
     XEvent * event;
     String * params;
     Cardinal *num_params;
{
    XbListWidget lw = ( XbListWidget ) w;
    int item, item_len;
    int count = 0;
    int sel = 0;
    XbListSelectionPtr clientData = (XbListSelectionPtr)XtMalloc(sizeof(XbListSelection));
    char tmp[512];

    /* 
     * Find item and if out of range then unhighlight and return. 
     * 
     * If the current item is unhighlighted then the user has aborted the
     * notify, so unhighlight and return.
     */
    if ((CvtToItem(w, event->xbutton.x, event->xbutton.y, &item)
	 == XB_OUT_OF_RANGE)) 
    {
	XbListUnhighlight(w);
	return;
    }
    
    item_len = strlen(lw->list.list->entries[item].entry);

    /* if XtNpasteBuffer set then paste it. */    
    if ( lw->list.paste )	
	XStoreBytes(XtDisplay(w), lw->list.list->entries[item].entry, item_len);
    

    /* Call Callback function.  */
    clientData->size = lw->list.num_selected;
    clientData->entries = (XbListEntryPtr)XtMalloc(clientData->size*sizeof(XbListEntry));
    
    sprintf(tmp, "\nNotify @ 0x%x size %d", clientData->entries, clientData->size);
    ximap_log(tmp);

    while( count < lw->list.list->size && sel < clientData->size )
    {
	if( lw->list.list->entries[count].invert )
	{
	    clientData->entries[sel].index = count;
	    clientData->entries[sel].entry = 
		XtMalloc(strlen(lw->list.list->entries[count].entry)+1);
	    strcpy(clientData->entries[sel].entry,lw->list.list->entries[count].entry);
	    clientData->entries[sel].invert = TRUE;
	    clientData->entries[sel].bold = lw->list.list->entries[count].bold;
	    sel++;
	}
	count++;
    }

    clientData->size = sel;

    if (event->xbutton.time - lw->list.last_click > lw->list.double_click_delay
	|| lw->list.last_item_clicked != item)
    {
	lw->list.last_click = event->xbutton.time;
	lw->list.last_item_clicked = item;
	XtCallCallbacks( w, XtNcallback, clientData);
    }
    else 
    {
	lw->list.last_item_clicked = XB_OUT_OF_RANGE;
	lw->list.last_click = 0;
	XtCallCallbacks( w, XtNdoubleClick, clientData);
    }
    
/*    XbListFreeSelection(clientData); */
}

/*	Function Name: Unset
 *	Description: unhighlights the current element.
 *	Arguments: w - the widget that the event occured in.
 *                 event - not used.
 *                 params, num_params - not used.
 *	Returns: none.
 */

/* ARGSUSED */
static void Unset(w, event, params, num_params)
    Widget w;
    XEvent * event;
    String * params;
    Cardinal *num_params;
{
    int item;
    
    if (CvtToItem(w, event->xbutton.x, event->xbutton.y, &item)
	!= XB_OUT_OF_RANGE) 
    {
	XbRemoveFromSelectList( w, item );
	return;
    }
}

/*	Function Name: Set
 *	Description: Highlights the current element.
 *	Arguments: w - the widget that the event occured in.
 *                 event - event that caused this notification.
 *                 params, num_params - not used.
 *	Returns: none.
 */

/* ARGSUSED */
static void Set(w, event, params, num_params)
     Widget w;
     XEvent * event;
     String * params;
     Cardinal *num_params;
{
    int item;
    XbListWidget lw = (XbListWidget) w;
    
    if ((CvtToItem(w, event->xbutton.x, event->xbutton.y, &item))
	== XB_OUT_OF_RANGE)
    {
	XbRemoveFromSelectList( w, item );
	return;
    }
    else
    {
	lw->list.inExtend = FALSE;
	lw->list.inMoveSelect = TRUE;
	lw->list.last = item;
	lw->list.origin = XB_OUT_OF_RANGE;
	XbClearSelectList( w );
	XbAddToSelectList( w, item );
	return;
    }
}

/*
 * Set specified arguments into widget
 */

static Boolean SetValues(current, request, new)
     Widget current, request, new;
{
    XbListWidget cl = (XbListWidget) current;
    XbListWidget rl = (XbListWidget) request;
    XbListWidget nl = (XbListWidget) new;
    Boolean redraw = FALSE;
    
    if ((cl->list.foreground != rl->list.foreground) ||
	(cl->core.background_pixel != rl->core.background_pixel) ||
	(cl->list.normalFont != rl->list.normalFont) ||
	(cl->list.emphaticFont != rl->list.emphaticFont)) 
    {
	XtDestroyGC(cl->list.normgc);
	XtDestroyGC(cl->list.revgc);
	XtDestroyGC(cl->list.boldgc);
	XtDestroyGC(cl->list.revboldgc);
	GetGCs(new);
	redraw = TRUE;
    }
    
    /* Reset row height. */
    if (cl->list.row_height != rl->list.row_height)
	nl->list.row_height = cl->list.row_height;
    
    if ((cl->list.row_space != rl->list.row_space) 
	||(cl->list.normalFont != rl->list.normalFont)
	||(cl->list.emphaticFont != rl->list.emphaticFont))
    {
	if(( nl->list.normalFont->max_bounds.descent +  nl->list.normalFont->max_bounds.ascent)
	   != ( cl->list.normalFont->max_bounds.descent +  cl->list.normalFont->max_bounds.ascent))
	{
	    nl->list.normalFont = cl->list.normalFont;
	    nl->list.emphaticFont = cl->list.emphaticFont;
	    XtWarning("XbList: normalFont and emphaticFont must have the same height\n");
	}

	nl->list.row_height = nl->list.normalFont->max_bounds.ascent
	    + nl->list.normalFont->max_bounds.descent
		+ nl->list.row_space;
    }
    
    if ((cl->core.width != rl->core.width)                     ||
	(cl->core.height != rl->core.height)                   ||
	(cl->list.internal_width != rl->list.internal_width)   ||
	(cl->list.internal_height != rl->list.internal_height) ||
	(cl->list.column_space != rl->list.column_space)       ||
	(cl->list.row_space != rl->list.row_space)             ||
	(cl->list.default_cols != rl->list.default_cols)       ||
	((cl->list.force_cols != rl->list.force_cols) 
	 &&(rl->list.force_cols != rl->list.ncols) )           ||
	(cl->list.vertical_cols != rl->list.vertical_cols)     ||
	(cl->list.longest != rl->list.longest)                 ||
	(cl->list.nitems != rl->list.nitems)                   ||
	(cl->list.normalFont != rl->list.normalFont)           ||
	(cl->list.emphaticFont != rl->list.emphaticFont)       ||
	(cl->list.list != rl->list.list)                        ) 
    {
	ResetList(new, TRUE, TRUE);
	redraw = TRUE;
    }
    
    if ((cl->core.sensitive != rl->core.sensitive) ||
	(cl->core.ancestor_sensitive != rl->core.ancestor_sensitive)) 
	redraw = TRUE;
    
    if (!XtIsRealized(current))
	return(FALSE);
    
    return(redraw);
}

/* Exported Functions */

/*	Function Name: XbListChange.
 *	Description: Changes the list being used and shown.
 *	Arguments: w - the list widget.
 *                 list - the new list.
 *                 nitems - the number of items in the list.
 *                 longest - the length (in Pixels) of the longest element
 *                           in the list.
 *                 resize - if TRUE the the list widget will
 *                          try to resize itself.
 *	Returns: none.
 *      NOTE:      If nitems of longest are <= 0 then they will be calculated.
 *                 If nitems is <= 0 then the list needs to be NULL terminated.
 */

void XbListChange(w, list, longest, resize_it)
    Widget w;
    XbListSelectionPtr list;
    int longest;
    Boolean resize_it;
{
    int i = 0;

    XbListWidget lw = (XbListWidget) w;
    XbListSelection *oldList = lw->list.list;

    lw->list.list = list;
    
    if (longest <= 0) 
	longest = 0;
    
    lw->list.longest = longest;
    
    ResetList(w, resize_it, resize_it);
    XbClearSelectList(w);
    if ( XtIsRealized(w) )
	Redisplay(w, NULL, NULL);

    XbListFreeSelection(oldList);
}


/* Brock 11-1-90 */
/*      Function Name: XawReplaceString
 *      Description: replaces the indicated string, freeing the one currently 
 *            in place.
 *	Arguments: w - the list widget.
 *                 no - the index of the string to be replaced ( starting
 *                      0 ).
 *                 string - the string to insert.
 *      If string is NULL, it deletes the indicated string from the list.
 *	Returns: none.
 */

char* XbListReplaceString(w, item, resize_it)
     Widget w;
     XbListEntryPtr item;
     Boolean resize_it;
{
    XbListWidget lw = (XbListWidget) w;
    char *old = lw->list.list->entries[item->index].entry;
    int i = 0;
    int size = 0;
    
    lw->list.list->entries[item->index].entry = item->entry;
    lw->list.list->entries[item->index].bold = item->bold;
    if ( lw->list.list->entries[item->index].bold )
	size = XTextWidth(lw->list.emphaticFont, item->entry, strlen(item->entry));
    else
	size = XTextWidth(lw->list.normalFont, item->entry, strlen(item->entry));
    
    if ( lw->list.longest < size )
    {
	lw->list.longest = size;
	ResetList(w, resize_it, resize_it);
	if ( XtIsRealized(w) )
	    Redisplay(w, NULL, NULL); 
/*	PaintItemName( w, item->index ); */
    }
    else
    { 
	PaintItemName( w, item->index );
    } 
    
    XtFree(old);
    return(NULL);
}


XbListEntryPtr XbListGetItem( w, no )
     Widget w;
     int    no;
{
  XbListWidget lw = (XbListWidget) w;
  return(lw->list.list->entries + no);
}

/*	Function Name: XbListUnhighlight
 *	Description: unlights the current highlighted element.
 *	Arguments: w - the widget.
 *	Returns: none.
 */
void XbListUnhighlight(w)
     Widget w;
{
    XbClearSelectList( w );
}

/*	Function Name: XbListHighlight
 *	Description: Highlights the given item.
 *	Arguments: w - the list widget.
 *                 item - the item to hightlight.
 *	Returns: none.
 */


void XbListHighlight(w, sel)
     Widget w;
     XbListSelectionPtr sel;
{
    register int num_selected = sel->size;
    if (XtIsSensitive(w)) 
    {
	if ( sel != NULL )
	    while ( num_selected-- )
		XbAddToSelectList(w, sel->entries[num_selected].index);
	else
	    XbClearSelectList(w);
    }
}

/*	Function Name: XbListShowCurrent
 *	Description: returns the currently highlighted object.
 *	Arguments: w - the list widget.
 *	Returns: the info about the currently highlighted object.
 */

XbListSelectionPtr XbListShowCurrent(w)
     Widget w;
{
    XbListWidget lw = ( XbListWidget ) w;
    XbListSelectionPtr result;
    int count = 0;
    int total = 0;
    char tmp[512];

    result = (XbListSelectionPtr) XtMalloc (sizeof (XbListSelection));
    
    result->size = lw->list.num_selected;
    result->entries = (XbListEntryPtr) XtMalloc (result->size*sizeof(XbListEntry));

    sprintf(tmp, "\nXbListShowCurrent @ 0x%x size %d", result->entries, 
	    result->size);
    ximap_log(tmp);

    while( total < lw->list.list->size && count < result->size)
    {
	result->entries[count].entry = NULL;
	result->entries[count].index = -1;
	result->entries[count].bold = 0;
	result->entries[count].invert = 0;

	if( lw->list.list->entries[total].invert )
	{
	    result->entries[count].entry = 
		XtMalloc(1+strlen(lw->list.list->entries[total].entry));
	    strcpy(result->entries[count].entry,lw->list.list->entries[total].entry);
	    result->entries[count].index = lw->list.list->entries[total].index;
	    result->entries[count].bold = lw->list.list->entries[total].bold;
	    result->entries[count].invert = TRUE;
	    count++;
	}
	total++;
    }
    result->size = count;
    return(result);
}

static  void XbClearSelectList(w)
     Widget w;
{
    XbListWidget lw = (XbListWidget) w;
    int i = lw->list.list->size;
    while(i--)
    {
	if( lw->list.list->entries[i].invert )
	{
	    lw->list.list->entries[i].invert = FALSE;
	    PaintItemName(w, lw->list.list->entries[i].index);
	}
    }
    lw->list.num_selected = 0;
}

static void XbAddToSelectList(w, item)
     Widget w;
     int item;
{
    XbListWidget lw = (XbListWidget) w;
    XbListSelectionPtr lst = lw->list.list;

    if( item < lw->list.list->size )
    {
	if( lst->entries[item].invert )
	    return;
	
	++lw->list.num_selected;
	lst->entries[item].invert = TRUE;
	PaintItemName( w, item );
    }
}


static  void XbRemoveFromSelectList(w, item)
     Widget w;
     int item;
{
    XbListWidget lw = (XbListWidget) w;
    XbListSelectionPtr lst = lw->list.list;

    if( item < lw->list.list->size )
    {
	if( !lst->entries[item].invert )
	    return;
	
	--lw->list.num_selected;
	lst->entries[item].invert = FALSE;
	PaintItemName( w, item );
    }
}

/* ARGSUSED */
static void Toggle(w, e, par, num)
     Widget w;
     XEvent * e;
     String * par;
     Cardinal *num;
{
    XbListWidget lw = (XbListWidget) w;
    int item;
    
    if (CvtToItem(w, e->xbutton.x, e->xbutton.y, &item)
	!= XB_OUT_OF_RANGE) 
    {
	lw->list.last_item_clicked = XB_OUT_OF_RANGE;
	lw->list.last_click = 0;
	
	if(IsHighlighted(w, item))
	{
	    XbRemoveFromSelectList(w, item);
	}
	else
	{
	    lw->list.origin = item;
	    lw->list.last = item;
	    lw->list.inExtend = TRUE;
	    lw->list.inMoveSelect = FALSE;
	    lw->list.direction = XB_NONE;
	    XbAddToSelectList(w, item);
	}
    }
    else
    {
	lw->list.last = XB_OUT_OF_RANGE;
    }
    return;
}

/* ARGSUSED */
static void CheckExtend(w, e, par, num)
     Widget w;
     XEvent * e;
     String * par;
     Cardinal *num;
{
    XbListWidget lw = (XbListWidget) w;
    
    if (lw->list.inExtend == TRUE)
    {
	lw->list.last_click = 0;
	lw->list.last_item_clicked = XB_OUT_OF_RANGE;
    }
    lw->list.inExtend = FALSE;
    lw->list.last = lw->list.origin = XB_OUT_OF_RANGE;
    lw->list.direction = XB_NONE;
    lw->list.inMoveSelect = FALSE;
}

/* ARGSUSED */
static void ExtendSelect(w, e, par, num)
     Widget w;
     XEvent * e;
     String * par;
     Cardinal *num;
{
    XbListWidget lw = (XbListWidget) w;
    int item;
    
    if (lw->list.inMoveSelect == TRUE
	|| lw->list.inExtend == FALSE )
    {
	lw->list.inMoveSelect = FALSE;
	lw->list.last = XB_OUT_OF_RANGE;
    }
    
    if (CvtToItem(w, e->xmotion.x, e->xmotion.y, &item) != XB_OUT_OF_RANGE 
	&& item != lw->list.last) 
    {
	if ( lw->list.inExtend == FALSE )
	{
	    lw->list.origin = item;
	    lw->list.last = item;
	    XbAddToSelectList(w, item);
	    lw->list.inExtend = TRUE;
	    lw->list.direction = XB_NONE;
	}
	else
	{
	    switch ( lw->list.direction )
	    {
		/*  if no direction set items between origin and current
		 ** position, set direction 
		 */
	      case XB_NONE:
		if ( item < lw->list.origin )
		    lw->list.direction = XB_NEGATIVE;
		else
		    lw->list.direction = XB_POSITIVE;
		
		SelectRange(w, lw->list.origin, item);
		break;
		
		/*  if moving in positive direction check for a 
		 ** possible change in direction
		 */
	      case XB_POSITIVE:
		/* if dist from origin still increasing, set items */
		if ( item > lw->list.last )
		{
		    SelectRange(w, lw->list.last, item);
		}
		/* else, check to see if direction has changed */
		else
		{
		    /* if no, just unselect correct items */
		    if (item > lw->list.origin)
		    {
			UnSelectRange(w, lw->list.last, item);
		    }
		    /* if at origin, direction is none. origin is
		     * always selected.
		     */
		    else if ( item == lw->list.origin )
		    {
			lw->list.direction = XB_NONE;
			UnSelectRange(w, lw->list.last, item + 1);
		    }
		    /* else clear list, change direction and set items. */
		    else
		    {
			lw->list.direction = XB_NEGATIVE;
			XbClearSelectList(w);
			SelectRange(w, lw->list.origin, item);
		    }
		}
		break;
		
		/*  if moving in negative direction check for a 
		 ** possible change in direction
		 */
	      case XB_NEGATIVE:
		/* if dist from origin still decreasing, set items */	    
		if ( item < lw->list.last )
		{
		    SelectRange(w, lw->list.last, item);
		}
		/* else, check to see if direction has changed */
		else
		{
		    /* if no, just unselect correct items */
		    if (item < lw->list.origin)
		    {
			UnSelectRange(w, lw->list.last, item);
		    }
		    /* if at origin, direction is none. origin is
		     * always selected.
		     */
		    else if ( item == lw->list.origin )
		    {
			lw->list.direction = XB_NONE;
			UnSelectRange(w, lw->list.last, item - 1);
		    }
		    /* else clear list, change direction and set items. */
		    else
		    {
			lw->list.direction = XB_POSITIVE;
			XbClearSelectList(w);
			SelectRange(w, lw->list.origin, item);
		    }
		}
		break;
		default:
		lw->list.direction = XB_NONE;
		break;
	    }
	    lw->list.last = item;
	}
    }
}

/* ARGSUSED */
static void MoveSelect(w, e, par, num)
     Widget w;
     XEvent * e;
     String * par;
     Cardinal *num;
{
    int item;
    
    XbListWidget lw = (XbListWidget) w;
    
    if ( lw->list.inExtend == TRUE )
    {
	lw->list.inExtend = FALSE;
	lw->list.last = XB_OUT_OF_RANGE;
	lw->list.inMoveSelect = TRUE;
    }
    
    if (CvtToItem(w, e->xmotion.x, e->xmotion.y, &item) != XB_OUT_OF_RANGE 
	&& item != lw->list.last) 
    {
	lw->list.last = item;
	XbClearSelectList( w );
	XbAddToSelectList( w, item );
	return;
    }
}

/* ARGSUSED */
static void
SelectRange(w, start, end)
     Widget w;
     int    start;
     int    end;
{
    register int i, j;
    
    i = ((start < end) ? start:end);
    j = ((i == start) ? end:start);
    
    while ( i <= j )
    {
	XbAddToSelectList(w, i);
	++i;
    }
}

/* ARGSUSED */
static void UnSelectRange(w, start, end)
     Widget w;
     int    start;
     int    end;
{
    register int i;
    int j;
    
    i = ((start < end) ? start:end);
    j = ((i == start) ? end:start);
    
    while ( i <= j )
    {
	XbRemoveFromSelectList(w, i);
	++i;
    }
}

void XbListFreeSelection(sel)
     XbListSelectionPtr sel;
{
    int i;
    if(sel)
    {
	if(sel->entries)
	{
	    char tmp[512];

	    sprintf(tmp, "\n XbListFreeSelection @ 0x%x size %d", sel,
		    sel->size);
	    ximap_log(tmp);

	    for(i = 0; i < sel->size; i++)
		if(sel->entries[i].entry) {
		    XtFree(sel->entries[i].entry);
		  }
	    
	    XtFree(sel->entries);
	}

	XtFree(sel);
    }
}

/*	Function Name: CvtToPosition
 *	Description: Converts item number to Xcoord of item
 *	Arguments: w - the list widget.
 *                 xloc, yloc - x location, and y location.
 *	Returns: the item number.
 */

static int CvtToPosition(w, xloc, yloc, item)
     Widget w;
     int *xloc;
     int *yloc;
     int item;
{
    XbListWidget lw = (XbListWidget) w;
    int n = lw->list.nitems;

    int ret_val = XB_OKAY;

    *xloc = (int) lw->list.col_width / 2;
    *yloc = (int) lw->list.row_height * (item - .5);

    return(ret_val);
}

XbListSelectionPtr XbListCopySelection(sel)
     XbListSelectionPtr sel;
{
    XbListSelectionPtr ret = NULL;
    ret = (XbListSelectionPtr) XtMalloc (sizeof(XbListSelection));
    ret->size = sel->size;
    ret->entries = NULL;

    if( ret->size )
    {
	int i = 0;
	char tmp[512];

	ret->entries = (XbListEntryPtr)XtMalloc(ret->size*sizeof(XbListEntry));

	sprintf(tmp, "\nXbListCopySelection @ 0x%x size %d", ret->entries, 
		ret->size);
	ximap_log(tmp);

	for( ; i < ret->size; i++)
	    XbListCopyEntry(ret->entries+i, sel->entries+i);
    }

    return(ret);
}

void XbListCopyEntry(ent1, ent2)
	XbListEntry *ent1, *ent2;
{
    if(ent1 && ent2)
    {
	if(ent2->entry)
	{
	    ent1->entry = XtMalloc(strlen(ent2->entry)+1);
	    strcpy(ent1->entry,ent2->entry);
	}
	
	ent1->index = ent2->index;
	ent1->bold  = ent2->bold;
	ent1->invert = ent2->invert;
    }
}

void XbListFreeEntry(ent)
	XbListEntry *ent;
{

}
