/* -*- C -*-
 *
 * Program:	ximap
 * File:        addrutil.c -- Miscellaneous things dealing with ADDRESSES
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
#include <Client/smtp.h>
#include <Client/rfc822.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "structures.h"
 
#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "util.h"
#include "ximap.h"
#include "addrutil.h"

char * get_one_address();
extern AppResources res;

/*
  creates a new address structure and copies the 
  given address list into the new address
*/
ADDRESS* copy_address( add )
     ADDRESS *add;
{
    ADDRESS *newadd = (ADDRESS*) NULL;
    
    newadd = mail_newaddr();
    
    if ( add != NULL )
    {
	newadd->personal   = ((add->personal)?
			      (cpystr(add->personal)):(char*)NULL);
	newadd->adl        = ((add->adl)?
			      (cpystr(add->adl)):(char*)NULL);
	newadd->mailbox   = ((add->mailbox)?
			      (cpystr(add->mailbox)):(char*)NULL);
	newadd->host       = ((add->host)?
			      (cpystr(add->host)):(char*)NULL);
	newadd->error      = ((add->error)?
			      (cpystr(add->error)):(char*)NULL);
	
	if (add->next)
	    newadd->next = copy_address( add->next );
    }
    
    return(newadd);
}


Boolean onAddressList( name, address )
     char *   name;
     ADDRESS *address;
{
    ADDRESS *tempList = address;

    if(name)
	while(tempList)
	{
	    /* if mailbox is not different.. */
	    if ( !strcasecmp(tempList->mailbox, name))
		return( TRUE );
	    
	    tempList = tempList->next;
	}
    
    return( FALSE );
}


/*
 * Function: addressIsInList
 * Checks to see if address is in list
 *
 * Returns:   NULL if address is not in list
 *            A pointer to the address just before the address being searched 
 *                 for if the address is not the only address in the list
 *            A pointer to the address being searched for if it is the only
 *                 address in the list.
 *
 */
ADDRESS* addressIsInList( address, list )
     ADDRESS *address;
     ADDRESS *list;
{
    ADDRESS *tempList = list;
    ADDRESS *lastAddress = tempList;
    
    if(tempList && tempList->host && tempList->mailbox
       && address && address->mailbox && address->host)
    {
	while(tempList)
	{
	    /* if at least one of host and mailbox is not different.. */
	    if (addrEq( tempList, address ))
		return(lastAddress);

	    lastAddress = tempList;
	    tempList = tempList->next;
	}
    }
    
    return((ADDRESS*) NULL);
}


Boolean addrEq( a1, a2 )
     ADDRESS *a1;
     ADDRESS *a2;
{
    return( a1 && a2 && a1->host && a2->host
	   && !(areDifferentStrings(a1->mailbox, a2->mailbox)
		|| areDifferentStrings(a1->host, a2->host)));
}

/*
  adds an address (the first argument) to a list of
  addresses (the second argument) iff the
  address is not already part of the list.
*/
void add_address_to_list(address, list)
     ADDRESS *address;
     ADDRESS **list;
{
    ADDRESS *temp = address;
    ADDRESS *lastAddress = temp;

    if( !(*list) || !(*list)->host )
    {
	*list = copy_address(address);
	return;
    }

    while(temp)
    {
	if(temp->host && !addressIsInList(temp, *list))
	{
	    lastAddress = copy_address(temp);
	    temp = temp->next;
	    
	    lastAddress->next = *list;
	    *list = lastAddress;
	}
	else
	{
	    temp = temp->next;
	}
    }
}

/* 
  removes an address from a list of addresses
  if it's there, and frees the address;
  does nothing otherwise.

  Arguments:
      addr - address to remove
      lst  - list of addresses from which to remove addr

  Returns: 
      nothing
*/
void remove_addr_from_list(address, list)
     ADDRESS *address;
     ADDRESS **list;
{
    ADDRESS *where = NULL;
    ADDRESS *old = NULL;

    while( where = addressIsInList(address, *list) )
    {
	if( addrEq( address, where ))
	{
	    old = where;
	    *list = old->next;
	}
	else
	{
	    where->next = where->next->next;
	}
    }
}

/* 
  unwinds an address list into a string for display in a text
  widget.
*/
char* address_to_text( ad, length )
     ADDRESS *ad;
     int     length;
{
    char *tmp = NULL, *buffer = NULL;
    char *str;
    int totalLen= 0;
    int len = 0;
    int max = ((length > TMPLEN) ? TMPLEN:length);
    if (ad && ad->mailbox && ad->host)
    {
	tmp = buffer = XtMalloc( max + 1);
	while(ad && totalLen < length)
	{
	    str = get_one_address( ad, max );
	    len = strlen(str);
	    if(totalLen + len + 2 < max 
	       && totalLen + len + 2 < length)
	    {
		totalLen += len;
		if(ad->next)
		{
		    sprintf( tmp, "%s, ",str );
		    tmp += 2;
		    totalLen += 2;
		}
		else
		{
		    sprintf( tmp, "%s", str );
		}
		tmp += len;
	    }
	    else if ( totalLen + len + 2 < length )
	    {
		max = ( length > 2*max ? 2*max:length );
		buffer = XtRealloc( buffer, max );
		tmp = buffer+totalLen;
	    }
	    else
	    {
		if( totalLen )
		    tmp[totalLen-2] = '\0';
		totalLen = length;
	    }
	    ad = ad->next;
	    XtFree(str);
	}

	if( totalLen == 0 )
	{
	    XtFree(buffer);
	    buffer = NULL;
	}
    }
    return( buffer );
}

char*  get_one_address( address, maxsize )
     ADDRESS *address;
     int      maxsize;
{
    char *personal = NULL;
    char *route = NULL;
    char *name = NULL;
    char *host = NULL;
    char *tmp = NULL;
    int length = 0;

    if (address->personal)
    {
	personal = cpystr( address->personal );
	length += (strlen(personal) + 1);
    }
    
    if (address->adl)
    {
	route = cpystr( address->adl );
	length += (strlen(route) + 2);
    }

    name = cpystr(address->mailbox);
    host = cpystr(address->host);
    length += (strlen(name) + strlen(host) + 3);

    tmp = XtMalloc(length+1);

    if( personal )
    {
	if( route )
	    sprintf( tmp, "%s <@%s@%s@%s>", personal, route, name, host );
	else
	    sprintf( tmp, "%s <%s@%s>", personal, name, host );
    }
    else
    {
	if( route )
	    sprintf( tmp, "<@%s@%s@%s>", route, name, host );
	else
	    sprintf( tmp, "<%s@%s>", name, host );
    }
    XtFree(personal);
    XtFree(route);
    XtFree(name);
    XtFree(host);
    
    return(tmp);
}

/*
  pretty self explanatory...
  
*/
ENVELOPE* copy_envelope( env )
     ENVELOPE *env;
{
    ENVELOPE *newenv = (ENVELOPE*) NULL;
    
    newenv = mail_newenvelope();
    if ( env != NULL )
    {
	newenv->return_path = copy_address(env->return_path);
	newenv->from        = copy_address(env->from);
	newenv->sender      = copy_address(env->sender);
	newenv->reply_to    = copy_address(env->reply_to);
	newenv->to          = copy_address(env->to);
	newenv->cc          = copy_address(env->cc);
	newenv->bcc         = copy_address(env->bcc);
	
	newenv->remail      = ((env->remail)?
			       (cpystr(env->remail)):(char*)NULL);

	newenv->date        = ((env->date)?
			       (cpystr(env->date)):(char*)NULL);

	newenv->subject     = ((env->subject)?
			       (cpystr(env->subject)):(char*)NULL);

	newenv->in_reply_to = ((env->in_reply_to)?
			       (cpystr(env->in_reply_to)):(char*)NULL);

	newenv->message_id  = ((env->message_id)?
			       (cpystr(env->message_id)):(char*)NULL);

	newenv->newsgroups  = ((env->newsgroups)?
			       (cpystr(env->newsgroups)):(char*)NULL);

    }
    
    return(newenv);
}


/*
  This function sets all of the envelope information which
  is independent of what type of composition is occuring.

*/

ENVELOPE* set_up_envelope(node, host)
     MessageNode*  node;
     char        *host;
{
    char temp[TMPLEN];
    char *s = NULL;
    char *username;
    
    ENVELOPE *env = NULL;

    username = getName(host);

    env = mail_newenvelope();

    env->cc = NULL;   
    env->to = NULL;
    env->remail = NULL;
    env->in_reply_to = NULL;
    env->subject = cpystr("");
    env->newsgroups = NULL;
    env->in_reply_to = NULL;
    
    /* The date and from fields will *always* be 
       set...
       */
    rfc822_date(temp);
    env->date = cpystr(temp);
    
    /* user personal name and address */
    env->from = mail_newaddr();
    if (res.personal_name)
    {
	env->from->personal = cpystr(res.personal_name);
    }
    
    if( res.mail_name )
	env->from->mailbox = cpystr(res.mail_name);
    else if ( username )
	env->from->mailbox = cpystr(username);
    else
	env->from->mailbox = NULL;

    if( res.mail_host )
	env->from->host = cpystr(res.mail_host);
    else if (host)
	env->from->host = cpystr(host);
    else 
	env->from->host = NULL;
    
    env->from->adl = NULL;
    env->from->next = NULL;
    
    /* set to default_bcc or by user */
    env->bcc = NULL;
    env->cc = NULL;

    if(res.default_bcc)
    {
	char *temp;
	s = cpystr(res.default_bcc);
	temp = s;
	env->bcc = rfc822_parse_address(&temp, host);
	free(s);
    }


    if(res.default_cc)
    {
	char *temp;
	s = cpystr(res.default_cc);
	temp =s;
	env->cc = rfc822_parse_address(&temp, host);
	free(s);
    }

    return env;
}
    
    
