/* -*- C -*-
 *
 * Program:	ximap
 * File:        browserutil.c -- Functions for maintaining and redisplaying 
 *              the Browsers.  These are the functions which are specific to 
 *              Browsers-as-views-of-messages.  Functions which are 
 *              Browsers-as-browsers woule be in Xb/Browser.c
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 * Last Edit:
 *    15 Oct 92 - Fixed horrible bug in redisplay browser. See the code.
 *                Bill Yeager - 15 Oct 92.
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <stdio.h>
#include <string.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Paned.h>

#include <Xb/Browser.h>

#include "structures.h"

#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "util.h"

#include "mailbox.h"
#include "browserutil.h"

extern AppResources res;

/* 
   Remake the display for a browser
   after it's changed. Note that the current
   mapping is freed - ms->defaultMapping or ms->zoomMapping,
   and therefor:

   Returns: new value for ms->*Mapping
               Bill Yeager - 24 Nov 92
*/
XbListSelectionPtr redisplayBrowser( be )
     BrowserEntry *be;
{
    XbListSelectionPtr new = NULL;
    XbListSelectionPtr dummy;

    int i = 0;
    
    MessageNode* msg;

    dummy = ximap_newlistmap(0);
    be->redisplay = 0;
    if(be->size)
    {
	char text[MAILTMPLEN];

	new = ximap_newlistmap(be->size);
	for( i = 0; i < be->size; ++i)
	{
	    msg = be->ms->messages[be->order[i]];
	    
	    if(msg->changed)
	    {
		headerline(text, be->ms->mailbox, msg);
		/* Changed msg->header_line to &msg->header_line
		 *     Bill Yeager - 15 Oct 92 */ 
		fs_give(&msg->header_line);
		msg->header_line = cpystr(text);
		msg->changed = FALSE;
	    }
	    
	    new->entries[i].entry = cpystr(msg->header_line);
	    new->entries[i].index = msg->msgno-1;
	    new->entries[i].bold = msg->toMe;
	}
	
	if(be->ms->status | EXPUNGING)
	    changeItemsWithHighlight(be->browser, 
				     new, dummy, 
				     XMS_HLTREMOVE );
	else 
	  changeItemsWithHighlight(be->browser, 
				     new, dummy, 
				     XMS_HLTRETAIN );

    }
    else if( be->main )
    {
	new  = ximap_newlistmap(0);
	changeItemsWithHighlight(be->browser, 
				 new, dummy, 
				 XMS_HLTRETAIN);
    }
    
    XbListFreeSelection(dummy);
    return new;
}

Boolean BackgroundFetch(ms)
     MailBox* ms;
{
    char header[TMPLEN];
    
    MessageNode*  node;
    MessageNode **temp = NULL;
    
    char *old = NULL;
    int number = BAD_MESSAGE;
    
    if (ms->nmsgs 
	&& ms->background_proc
	&& !(ms->status & DONTBACKGROUND))
    {
	XbListEntryPtr entry = (XbListEntryPtr) XtMalloc (sizeof(XbListEntry));

	ms->status |= BACKGROUND;
	temp = ms->last_message;

	while (temp >= ms->messages
	       && (*temp)->elt != NULL)
	    temp--;

	/* check to see if it was really found ..  */
	if (temp >= ms->messages)
	{
	    number = (*temp)->msgno;
	}
	else
	{
	    ms->background_proc = 0;
	    ms->status &= ~BACKGROUND;
	    XtFree(entry);
	    return(TRUE); /* there's nothing more we can do here... */
	}

	node = *temp;
	node->env = mail_fetchstructure(  ms->mailbox, number, &node->body);  
	node->elt = mail_elt( ms->mailbox, number);      

	if(node->env && node->env->to)
	{
	    char to, cc;
	    to = onAddressList( res.mail_name, node->env->to );
	    cc = onAddressList( res.mail_name, node->env->cc );
	    if( to ||  cc )
		node->toMe = TRUE;
	}

	memset( header, 0, TMPLEN );
	headerline( header, ms->mailbox, node);
	    
	old = node->header_line;

	node->header_line = cpystr(header);
	entry->entry = cpystr(header);
	entry->index = number-1;
	entry->bold = node->toMe;
	entry->invert = 0;

	if( ms->zoom == FALSE )
	    XbBrowserReplaceString(ms->browserList->browser, entry);

	XtFree((char*)old);
	XtFree(entry);

	ms->actual_size++;
	ms->status &= ~BACKGROUND;
	
	if ( temp != ms->messages )
	    return( FALSE );

	ms->background_proc = 0;
    }
    else 
    {
	if(ms->background_proc != NULL
	   && ms->nmsgs != 0)
	    return FALSE;
    }
    return( TRUE );
}

/*
 * 
 * Adds new items to the main browser window.
 *
 * It leaves highlighted items highlighted and also
 * highlights the new items.
 *
 *
 */
void updateBrowser(ms, change)
     MailBox* ms;
     int     change;
{
    BrowserEntry *be = ms->browserList;
    Widget        bw = be->browser;
    int count = 0;
    int view = 0;
    int first = 0;
    XbListSelectionPtr newHghlt = NULL;
    XbListSelectionPtr new = NULL;
    int *oldM = be->order;

    if(ms->zoom)
    {
	makeUpdatedZoomMapping(ms, be, change, FALSE);
	be->size = ms->nzoomed;		/* value was updated in above call */
	new = ms->zoomMapping;
/*	new = XbListCopySelection(ms->zoomMapping); */
    }
    else
    {
	be->size = ms->nmsgs;
	makeDefaultMapping(ms, FALSE);
	new = ms->defaultMapping;
/*	new = XbListCopySelection(ms->defaultMapping); */
    }

    newHghlt = ximap_newlistmap(change);

    for( count = 0; count < change; count++ )
    {
	newHghlt->entries[count].index = ms->nmsgs - change + count;
	newHghlt->entries[count].entry = NULL;
    }

    be->order = copyMapping( new );
    /*
     * NOTE: changeItemsWithHighlight results in disposing of the old
     * copy of either ms->zoomMapping or ms->defaultMapping. */
    if( ms->status & UNSOLICITEDCHECK )
	changeItemsWithHighlight( bw, new, NULL, XMS_HLTRETAIN );
    else if( res.highlight_new == TRUE )
	changeItemsWithHighlight( bw, new, newHghlt, XMS_HLTADD );
    else
	changeItemsWithHighlight( bw, new, newHghlt, XMS_HLTREMOVE );

    if( ms->zoom != TRUE )
    {
	view = XbBrowserGetView( bw );
	if ( ms->new > view )
	    first = ms->nmsgs - ms->new;
	else
	    first = ms->nmsgs - view;
	XbBrowserSetPosition( bw, first );
    }

    if(oldM)
	XtFree(oldM);

    if(newHghlt)
	XtFree(newHghlt);
}

void changeHeaders(ms, start, end)
     MailBox* ms;
     int start;
     int end;
{
    MessageNode* node;

    int i = 0;
    int *ord = ms->browserList->order;

    for (i=start; (i <= end)&&(ms->actual_size != ms->virtual_size) ;i++)
    {
	if (ms->messages[ord[i]]->elt == NULL)
	{
	    node = ms->messages[ord[i]];
	    fetchNodeInfo( ms, node );
	}
    }
}

void fetchMapping(ms, map)
     MailBox*      ms;
     XbListSelectionPtr      map;
{
    MessageNode* node = NULL;
    int i;

    for(i = 0; i < map->size; i++)
    {
	node = ms->messages[map->entries[i].index];
	if (node->elt == NULL)
	{
	    fetchNodeInfo( ms, node );
	    map->entries[i].entry = cpystr(node->header_line);
	    map->entries[i].bold  = node->toMe;
	}
    }
}

void fetch_unfetched(ms, nodelist)
     MailBox*  ms;
     NodeList* nodelist;
{
    int i;

    for(i = 0; i < nodelist->length; i++)
	if (nodelist->nodes[i]->elt == NULL)
	    fetchNodeInfo( ms, nodelist->nodes[i] );
}

void redoList(ms)
     MailBox* ms;
{
    BrowserList btemp = ms->browserList;
    BrowserEntry *be = NULL;

    ximap_log("\n  redoList:");

    while( btemp )
    {
	be = btemp->next;
	if( btemp->redisplay ) {
	    char tmp[512];

	    XbListSelectionPtr new = redisplayBrowser( btemp );
	    if (new) {			/* then replace active mapping */
	      if (ms->zoom)		/* zooming! */
		ms->zoomMapping = new;
	      else
		ms->defaultMapping = new;

	      sprintf(tmp, "\n  redoList: new mapping @ 0x%x", new);
	      ximap_log(tmp);
	    }
	  }
	
	if( !btemp->size )
	{
	    if( btemp->main && !ms->zoom )
	    {
		btemp = btemp->next;
		be = btemp;
		while (btemp = be)
		{
		    XtDestroyWidget(be->shell);
		    be = be->next;
		    ximap_freeBrowserEntry( be );
		}
		return;
	    }
	    else if( !btemp->main )
	    {
		XtDestroyWidget(btemp->shell);
		ximap_freeBrowserEntry( btemp );
	    }
	}
	else
	{
	    check_menus(NULL, btemp, NULL);
	}
	btemp = be;
    }
    ximap_log("\n redoList: Done\n");
}

void updateHeaders(ms, nodelist)
     MailBox*   ms;
     NodeList*  nodelist;
{
  BrowserEntry * be = ms->browserList;
  char temp[TMPLEN];
  char *old;
    
  int count;

  XbListEntryPtr item = NULL;

  for (count = 0; count < nodelist->length; ++count) {
    
    memset( temp, 0, TMPLEN );

    /* mtm - only update if it's a real message.... 
     * it may have been expunged from underneath us... 
     */
    if(! nodelist->nodes[count]->elt) 
      continue;
    headerline( temp, ms->mailbox, nodelist->nodes[count]);
    item = (XbListEntryPtr)XtMalloc(sizeof(XbListEntry));
    old = nodelist->nodes[count]->header_line;
    nodelist->nodes[count]->header_line = cpystr(temp);
    
    while( be ) {
      item->entry = (char*) XtMalloc (1 + strlen(temp));
      strcpy(item->entry, temp);
      
      item->bold  = nodelist->nodes[count]->toMe;
      item->index = nodelist->nodes[count]->msgno-1;
      item->invert = 0;
      
      XbBrowserReplaceString(be->browser, item);
      be = be->next;
    }
    
    XtFree((char*)old);
    XtFree(item);
    be = ms->browserList;

  }

}

void changeItemsWithHighlight( bw, map, hlt, mode )
     Widget  bw;
     XbListSelectionPtr map;
     XbListSelectionPtr hlt;
     XmsHighlightMode mode;
{
    int count = 0;
    int i = 0;
    XbListSelectionPtr hghlt = NULL;
    XbListSelectionPtr newHghlt = NULL;

    ximap_log("\nchangeItemsWithHighlight");
    if(mode != XMS_HLTREMOVE)
    {
	hghlt = XbBrowserCurrentHighlight(bw);
	if ( hghlt && ( mode == XMS_HLTADD || mode == XMS_HLTRETAIN ) )
	{
	    if( mode == XMS_HLTADD )
		newHghlt = ximap_newlistmap(hghlt->size + hlt->size);
	    else
		newHghlt = ximap_newlistmap(hghlt->size);

	    for (i = 0; i < hghlt->size; ++i)
	    {
		newHghlt->entries[i].index = hghlt->entries[i].index;
		newHghlt->entries[i].entry = NULL;
	    }
	}
	else 
	{
	    newHghlt = ximap_newlistmap(hlt->size);
	}

	if( hlt && ( mode != XMS_HLTRETAIN ) )
	    for( count = 0; count < hlt->size; count++ )
	    {
		newHghlt->entries[count+i].index = hlt->entries[count].index;
		newHghlt->entries[count+i].entry = NULL;
	    }
    }
    /* IMPORTANT NOTE: The following call stashes map into the
     * XbBrowserWidget bw, and DISPOSES the old map in this same
     * widget. So, callers who save local copy of map should NEVER
     * dipose it if changeItemsWithHighlight is latter called.
     *   This indiscretion caused a very bad bug in ximap.
     *        Bill Yeager -- 24 November 92 */
    XbBrowserChangeItems( bw, map );
    if( newHghlt )
	XbBrowserHighlight( bw, newHghlt );

    XbListFreeSelection(hghlt);
    XbListFreeSelection(newHghlt);

    ximap_log("\nSORTANT: changeItemsWithHighlight");    
}

void setBrowserSize( be )
     BrowserEntry *be;
{
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    Dimension box_height = 0;
    Dimension list_height = 0;
    Dimension int_border_width = 0;

    Dimension height = 0;
    Dimension width = 0;
    Dimension fixed_height = 0;

    XtWidgetGeometry intended, result;

    XtSetArg(warg[n], XtNheight, &list_height); n++;
    XtSetArg(warg[n], XtNwidth, &width); n++;
    XtGetValues( be->browser, warg, n ); n = 0;

    intended.request_mode = CWWidth | XtCWQueryOnly;
    intended.width = width;

    XtQueryGeometry( be->buttons, &intended, &result);

    box_height = result.height;

    XtSetArg(warg[n], XtNinternalBorderWidth, &int_border_width); n++;
    XtGetValues(be->panes,warg,n); n = 0;
    
    /* the user can't change the fixed_height... */
    fixed_height = box_height + (int_border_width);

    height = list_height + fixed_height;

    XtSetArg(warg[n], XtNheight, list_height); n++;
    XtSetValues( be->browser, warg, n ); n = 0;

    XtSetArg(warg[n], XtNwidth, width); n++;
    XtSetArg(warg[n], XtNheight, height); n++;
    XtSetValues(be->panes,warg,n); n = 0;
}

/* 
   The integer array this returns is used
   to find out if a given message is in a given browser.
*/
int* copyMapping( map )
     XbListSelectionPtr map;
{
    int i = 0;
    int *ret = NULL;
    
    ret = (int*)XtMalloc((map->size + 1)*sizeof(int));
    
    for( i = 0; i < map->size; i++ )
	ret[i] = map->entries[i].index;
    
    return( ret );
}

BrowserEntry* ximap_newBrowserEntry()
{
    BrowserEntry *ret = (BrowserEntry*) XtMalloc(sizeof(BrowserEntry));
    
    ret->next = NULL;
    ret->prev = NULL;
    ret->order = NULL;

    ret->browser = NULL;
    ret->buttons = NULL;
    ret->info    = NULL;
    ret->panes   = NULL;

    ret->size = 0;
    ret->main = FALSE;
    ret->redisplay = FALSE;
    ret->active = FALSE;
    
    ret->searchWin = NULL;
    ret->found = NULL;

    ret->ms = NULL;

    ret->currentMap = NULL;
    ret->currentFilter = NULL;

    return( ret );
}

void ximap_freeBrowserEntry( be )
     BrowserEntry *be;
{
    if( be )
    {
	if( be->next)
	    be->next->prev = be->prev;
	if( be->prev )
	    be->prev->next = be->next;

	if( be->order )
	    XtFree(be->order);

	if(be->states)
	    free_statelist(be->states);

	if(be->searchWin)
	{
	    free_statelist(be->searchWin->states);
	    XtDestroyWidget(be->searchWin->shell);
	}
	XtFree(be);
    }
}

XbListSelectionPtr  ximap_newlistmap( size )
     int size;
{
    XbListSelectionPtr ret = 
      (XbListSelectionPtr)XtMalloc(sizeof(XbListSelection));
    char tmp[512];
    
    ret->size = (size > 0 ? size : 0);
    
    if (size > 100)			
      tmp[0] = 'F';			/* Just a place to halt */

    sprintf(tmp, "\nximap_newlistmap: @ 0x%x size %d", ret, size);
    ximap_log(tmp);

    if(size > 0)
    {
	ret->entries = (XbListEntryPtr)XtMalloc(size*sizeof(XbListEntry));
	while(size--)
	{
	    ret->entries[size].index = -1;
	    ret->entries[size].entry = NULL;
	    ret->entries[size].bold = FALSE;
	    ret->entries[size].invert = FALSE;
	}
    }
    else 
    {
	ret->entries = NULL;
    }

    return( ret );
}

void setActiveBrowser(ms,bw)
     MailBox* ms;
     Widget  bw;
{
    BrowserList bl = NULL;
    bl = ms->browserList;
    
    while( bl )
    {
	if(bl->browser != bw)
	    bl->active = FALSE;
	else
	    bl->active = TRUE;

	bl = bl->next;
    }
}

BrowserEntry *getActiveBrowser( ms )
     MailBox* ms;
{
    BrowserEntry *ret = ms->browserList;
    while( ret && (ret->active != TRUE))
	ret = ret->next;

    return( ret );
}

NodeList* listMapToNodeList( mapping, ms )
     XbListSelectionPtr    mapping;
     MailBox*    ms;
{
    NodeList* nodes = (NodeList*)XtMalloc(sizeof(NodeList));
    int      length = mapping->size;
    
    nodes->length = length;
    if( length && ms && ms->messages )
    {
	nodes->nodes = (MessageNode**)XtMalloc(length*sizeof(MessageNode*));
	while( length-- )
	{
	    if(mapping->entries[length].index < ms->nmsgs)
	    nodes->nodes[length] = ms->messages[mapping->entries[length].index];
	}
    }

    return( nodes );
}

XbListSelectionPtr nodeListToListMap( nodes, ms )
     NodeList*      nodes;
     MailBox*       ms;
{
    XbListSelectionPtr mapping = NULL;
    int size = nodes->length;
    
    mapping = ximap_newlistmap(nodes->length);
    
    if(mapping)
    {
	while(size--)
	{
	    mapping->entries[size].index = nodes->nodes[size]->msgno-1;
	    mapping->entries[size].entry = NULL;
	}
    }
    
    return( mapping );
}

/* 
 *  This routine finds the browser for a particular search window.
 *
 */
BrowserEntry *searchWinToBrowser( ms, swin )
     MailBox*       ms;
     SEARCHWINDOW *swin;
{
    BrowserEntry *ret = ms->browserList;

    while( ret && (ret->searchWin != swin))
	ret = ret->next;

    return( ret );
}

/*
 *  Given a widget, this returns the browser of which it is the child.
 *  
 */
Widget widgetToBrowser( w )
     Widget w;
{
    Widget temp = w;
    
    while( !XtIsSubclass(panedWidgetClass) )
	temp = XtParent( temp );
    
    if( temp )
	return( temp );
    
    return( NULL );
}

/*
 *  This routine removes one item from a 
 *  browser.  If expungingItem is TRUE, it adjusts the 
 *  numbers of the other items, since the item is gone from the mailbox.
 *  If expungingItem is FALSE, the change is local to that browser.
 *
 */
void removeItemFromBrowser( be, num, expungingItem)
     BrowserEntry *be;
     int          num;
     Boolean      expungingItem;
{
    int i = 0;
    int found = 0;
    
    found = 0;
    for( i = 0; i < be->size; i++ )
    {
	if( expungingItem && be->order[i] > num )
	{
	    be->order[i]--;
	    be->redisplay = TRUE;
	}
	else if( !found && be->order[i] == num )
	{
	    be->size--;
	    for( found = i; found < be->size; found++ )
		be->order[found] = be->order[found+1];
	    be->redisplay = TRUE;
	    i--;
	}
    }
}

/*
 *  This routine removes an item from every browser
 *  in which it is displayed.
 *
 */
void removeFromBrowserEntries( ms, num )
     MailBox*    ms;
     int        num;
{
    BrowserList bl = ms->browserList;
    while( bl )
    {
	removeItemFromBrowser(bl, num, TRUE);
	bl = bl->next;
    }
}

