/* -*- C -*-
 *
 * Program:	ximap
 * File:        compose.c -- Functions for creating a window and
 *                           sending a message.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <string.h>
#include <netdb.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>
#include <memory.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
#include <Client/smtp.h>
#include <Client/rfc822.h>
 
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Paned.h>

#include <X11/Xaw/Text.h>
#include <X11/Xaw/TextSink.h>
#include <X11/Xaw/AsciiText.h>

#include <X11/Xaw/Label.h>

#include <Xb/Browser.h>

#include "structures.h"

#include "modal.h"
#include "message.h"
#include "globals.h" 
#include "buttons.h"
#include "resources.h"
#include "textutil.h"
#include "util.h"
#include "ximap.h"
#include "mailbox.h"
#include "readwin.h"
#include "compose.h"
#include "addrutil.h"
#include "browserutil.h"
#include "filebrowser.h"
#include "mime.h"

static void compBeep(/* Widget, XEvent*  */);
static void gotoCompFrom(/* Widget, XEvent*  */);
static void gotoCompTo(/* Widget, XEvent*  */);
static void gotoCompSubject(/* Widget, XEvent*  */);
static void gotoCompText(/* Widget, XEvent*  */);
static void gotoCompCc(/* Widget, XEvent*  */);
static void gotoCompBcc(/* Widget, XEvent*  */);

static void compDone(/* Widget, XEvent*  */);
static void compSendIt(/* Widget, XEvent*  */);
static void compSaveIt();
static void compInsertMessage();
static void compRestoreIt();

void saveDraftToFile();
void restoreDraftFromFile();
void insertMessageText();

void attachFile();
void removePart();
static void composeResizeBrowser();

#define NOLIMIT 999999
  
ButtonStruct composeButtons[] =
{
  { "SendQuit", sendQuit, NULL, NEVER},
  { "SaveDraft", saveDraftToFile, NULL,  NEVER},
  { "RestoreDraft", restoreDraftFromFile, NULL,  NEVER},
  { "SendSend", sendMessage, NULL, NEVER},
  { "SendInsert", insertMessageText, NULL, NOREPLYTEXT },
  NULL,
};

ButtonStruct partButtons[] =
{
  { "AttachFile", attachFile, NULL, NEVER},
  { "RemovePart", removePart, NULL,  NOATTACHEDPART},
  NULL,
};

static XtActionsRec compose_actions[] = {
  {"compBeep", compBeep},
  {"gotoCompFrom", gotoCompFrom},
  {"gotoCompTo", gotoCompTo},
  {"compInsertMessage", compInsertMessage},
  {"gotoCompSubject", gotoCompSubject},
  {"gotoCompText", gotoCompText },
  {"gotoCompCc", gotoCompCc },
  {"gotoCompBcc", gotoCompBcc },
  {"compDone", compDone},
  {"compSendIt", compSendIt},
  {"compSaveIt", compSaveIt},
  {"compRestoreIt", compRestoreIt},
};

static char fromTranslations[] = 
 "<Btn1Down>: select-start() gotoCompFrom()           \n\
  Ctrl<Key>I:    compInsertMessage()               \n\
  Ctrl<Key>Q: compDone()                              \n\
  Meta<Key>W: compSaveIt()                            \n\
  Meta<Key>R: compRestoreIt()                         \n\
  <Key>Tab:   gotoCompTo()                            \n\
  Ctrl<Key>/:    compSendIt()";

static char toTranslations[] = 
 "<Btn1Down>: select-start() gotoCompTo()             \n\
  Ctrl<Key>I:    compInsertMessage()               \n\
  Ctrl<Key>Q: compDone()                              \n\
  Meta<Key>W: compSaveIt()                            \n\
  Meta<Key>R: compRestoreIt()                         \n\
  <Key>Tab:   gotoCompSubject()                       \n\
  Ctrl<Key>/:    compSendIt()";

static char subjectTranslations[] = 
 "<Btn1Down>: select-start() gotoCompSubject()        \n\
  Ctrl<Key>I:    compInsertMessage()               \n\
  Ctrl<Key>Q: compDone()                              \n\
  Meta<Key>W: compSaveIt()                            \n\
  Meta<Key>R: compRestoreIt()                         \n\
  <Key>Tab:   gotoCompCc()                            \n\
  Ctrl<Key>/:    compSendIt()";

static char ccTranslations[] = 
 "<Btn1Down>: select-start() gotoCompCc()          \n\
  Ctrl<Key>I:    compInsertMessage()               \n\
  Ctrl<Key>Q: compDone()                              \n\
  Meta<Key>W: compSaveIt()                            \n\
  Meta<Key>R: compRestoreIt()                         \n\
  <Key>Tab:   gotoCompBcc()\n\
  Ctrl<Key>/:    compSendIt()";

static char bccTranslations[] = 
 "<Btn1Down>: select-start() gotoCompBcc()          \n\
  Ctrl<Key>Q: compDone()                              \n\
  Ctrl<Key>I:    compInsertMessage()               \n\
  Meta<Key>W: compSaveIt()                            \n\
  Meta<Key>R: compRestoreIt()                         \n\
  <Key>Tab:   gotoCompText()\n\
  Ctrl<Key>/:    compSendIt()";

static char textTranslations[] = 
 "<Btn1Down>:    select-start() gotoCompText()     \n\
  Ctrl<Key>Q:    compDone()                        \n\
  Ctrl<Key>I:    compInsertMessage()               \n\
  Meta<Key>W:    compSaveIt()                      \n\
  Meta<Key>R:    compRestoreIt()                   \n\
  Ctrl<Key>/:    compSendIt()";


static char *hostlist[] = 
{
    "mailhost",
    "localhost",
    NIL,
};

extern char *typeAcronyms[];

extern Widget toplevel;
extern XtAppContext app_context;
extern AppResources res;
extern MAILSESSION id;

/* These are used by anyone creating widgets or getting widget values */
Arg cargs[ARGLISTSIZE];


static Widget createComposeTextField(name, parent, value, height, trans,
				     status)
     char* name;
     Widget parent;
     char* value;
     int height;
     XtTranslations trans;
     unsigned long status;
{
    Widget retval = NULL;
    int cn= 0;

    XtSetArg(cargs[cn], XtNstring, value);cn++;
    if(height)
	XtSetArg(cargs[cn], XtNheight, height);cn++ ;
    /* See if we want a read-only text widget, eg, the text part of a
     * remailed message */
    if (status & TEXTREADONLY) {
      	XtSetArg(cargs[cn], XtNeditType, XawtextRead);cn++ ;
      }
    retval =  XtCreateManagedWidget(name,asciiTextWidgetClass,parent,
				    cargs,cn);
   cn = 0;

    XtOverrideTranslations(retval,trans);

    return(retval);
}

void addComposeActions( ac )
     XtAppContext ac;
{
    XtAppAddActions( ac, compose_actions, XtNumber(compose_actions));
}

ComposeWindow* ComposeMessage( rwin, ms, env, body, status)
     READWINDOW * rwin;
     MailBox   *ms;
     ENVELOPE  *env;
     BODY      *body;
     unsigned long status;
{
    XbListSelection *dummyList;
    ComposeWindow *cwin;
    Widget CompButtonBox, CompHeaderBox;
    
    Widget from_label;
    
    char temp[TMPLEN];
    
    Dimension width = 0;
    Dimension height = 0;
    
    XtWidgetGeometry intended, ret;
    
    char *window_name;
    
    XtTranslations from_trans, to_trans, subj_trans, text_trans;
    XtTranslations cc_trans, bcc_trans/* , cont_trans, enc_trans */;
    
    char *subject = NULL;
    char *to = NULL;
    char *from = NULL;
    char *cc = NULL;
    char *bcc = NULL;
    char *text = NULL;
    unsigned long loc_status= status;	/* for local status changes */

    XFontStruct *font;
    int cn= 0;

    
    cwin = (ComposeWindow*) XtMalloc (sizeof(ComposeWindow));
    
    from_trans = XtParseTranslationTable(fromTranslations);
    to_trans =   XtParseTranslationTable(toTranslations);
    text_trans = XtParseTranslationTable(textTranslations);
    subj_trans = XtParseTranslationTable(subjectTranslations);
    bcc_trans = XtParseTranslationTable(bccTranslations);
    cc_trans = XtParseTranslationTable(ccTranslations);

    cwin->ms = ms;
    cwin->handle = mail_makehandle(ms->mailbox);
    cwin->message = ms->replyto;
    cwin->status = status;
    cwin->env = env;
    cwin->body = body;
    cwin->host = cpystr(ms->host);
    cwin->msgno = (ms->replyto?ms->replyto->msgno:0);
    cwin->state = NOATTACHEDPART;
    cwin->attachments = NULL;
    cwin->rwin = rwin;

    subject = XtMalloc(mstrlen(cwin->env->subject) + 1);
    strcpy(subject, cwin->env->subject);
    
    from = address_to_text(cwin->env->from, NOLIMIT);
    to = address_to_text(cwin->env->to, NOLIMIT);
    cc = address_to_text(cwin->env->cc, NOLIMIT);
    bcc = address_to_text(cwin->env->bcc, NOLIMIT);

    if(cwin->body)
    {
	text = XtMalloc(mstrlen(cwin->body->contents.text) + 1);
	strcpy(text, cwin->body->contents.text);
    }
    else
    {
	text = XtMalloc(1);
	text[0] = '\0';
    }
    
    if (ms->replyto && (ms->replyto->status & ANSWERING))
	sprintf(temp, "Reply to message #%d", ms->replyto->msgno);
    else if (ms->replyto && (ms->replyto->status & FORWARDMSG))
	sprintf(temp, "Forward messages...");
    else if (ms->replyto && (ms->replyto->status & REMAILMSG))
	sprintf(temp, "Remail message #%d", ms->replyto->msgno);
    else 
	sprintf(temp, "New Message");

    window_name = cpystr(temp);
    
    XtSetArg(cargs[cn], XtNtitle, window_name); cn++;
    cwin->shell =  XtCreatePopupShell("compose_shell", 
					topLevelShellWidgetClass,
					ms->browserList->shell, 
					cargs, cn); cn = 0;
    
    cwin->panes = XtCreateManagedWidget("compose_panes", 
				     panedWidgetClass,
				     cwin->shell, 
				     cargs, cn); cn = 0;
    
    XtSetArg(cargs[cn], XtNshowGrip, FALSE); cn++;
    CompButtonBox = XtCreateManagedWidget("compose_buttons", 
					  boxWidgetClass,
					  cwin->panes,  
					  cargs, cn); cn = 0;

    createButtons(CompButtonBox, 
		  cwin,
		  composeButtons);
    
    XtSetArg(cargs[cn], XtNshowGrip, FALSE); cn++;
    CompHeaderBox = XtCreateManagedWidget("compose_headers", 
					  formWidgetClass,
					  cwin->panes,  
					  cargs, cn); cn = 0;
    
    from_label = XtCreateManagedWidget("from_label", 
				       labelWidgetClass, 
				       CompHeaderBox,
				       cargs, cn); cn = 0;
    
    intended.request_mode = XtCWQueryOnly;
    XtQueryGeometry(from_label, &intended, &ret);
    
    height = ret.height;
    
    XtSetArg(cargs[cn], XtNheight, height);cn++ ;
    XtSetValues(from_label, cargs, cn); cn = 0;
    
    XtSetArg(cargs[cn], XtNheight, 2*height);cn++ ;
    XtCreateManagedWidget("to_label", 
			  labelWidgetClass, 
			  CompHeaderBox,
			  cargs, cn); cn = 0;
    
    XtSetArg(cargs[cn], XtNheight, 2*height);cn++ ;
    XtCreateManagedWidget("subject_label",
			  labelWidgetClass, 
			  CompHeaderBox,
			  cargs, cn); cn = 0;
    
    XtSetArg(cargs[cn], XtNheight, 2*height);cn++ ;
    XtCreateManagedWidget("cc_label", 
			  labelWidgetClass, 
			  CompHeaderBox,
			  cargs, cn); cn = 0;
    
    XtSetArg(cargs[cn], XtNheight, height);cn++ ;
    XtCreateManagedWidget("bcc_label", 
			  labelWidgetClass, 
			  CompHeaderBox,
			  cargs, cn); cn = 0;

    cwin->from = createComposeTextField("from_text", CompHeaderBox,
					  from, height, from_trans, status);

    cwin->to   = createComposeTextField("to_text", CompHeaderBox,
					  to, 2*height, to_trans, status);

    cwin->subject = createComposeTextField("subject_text", CompHeaderBox,
					   subject, 2*height, subj_trans, 
					   status);

    cwin->cc = createComposeTextField("cc_text", CompHeaderBox,
					cc, 2*height, cc_trans, status);

    
    cwin->bcc = createComposeTextField("bcc_text", CompHeaderBox,
					 bcc, height, bcc_trans, status);

    XtSetArg(cargs[cn], XtNshowGrip, FALSE); cn++;
    cwin->part_buttons = XtCreateManagedWidget("part_buttons", 
						 boxWidgetClass,
						 cwin->panes,  
						 cargs, cn); cn = 0;

    cwin->states = createButtons(cwin->part_buttons, 
				   cwin,
				   partButtons);

    XtSetArg(cargs[cn], XtNallowResize, TRUE); cn++;
    XtSetArg(cargs[cn], XtNskipAdjust, TRUE); cn++;
    XtSetArg(cargs[cn], XtNresizeToPreferred, TRUE); cn++;
    XtSetArg(cargs[cn], XtNborderWidth, 0); cn++;
    XtSetArg(cargs[cn], XtNshowGrip, FALSE); cn++;
    cwin->part_browser_form = XtCreateManagedWidget("part_browser_form", formWidgetClass,
						      cwin->panes, cargs, cn); cn = 0;

    dummyList = ximap_newlistmap(1);
    dummyList->entries[0].index = 0;
    dummyList->entries[0].entry = cpystr("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX");
    XtSetArg(cargs[cn], XtNlist, dummyList); cn++;
    XtSetArg(cargs[cn], XtNinitialViewHt, 5); cn++;
    XtSetArg(cargs[cn], XtNresizable, TRUE); cn++;
    cwin->part_browser = XtCreateWidget("part_browser", XbbrowserWidgetClass,
					  cwin->part_browser_form, cargs, cn); cn = 0;

    XtSetArg(cargs[cn], XtNborderWidth, 0); cn++;
    XtSetArg(cargs[cn], XtNlabel, "                                          "); cn++;
    XtCreateManagedWidget("dummy", labelWidgetClass, cwin->part_browser_form, cargs, cn); cn = 0;

    if (status & REMAILMSG)
      loc_status |= TEXTREADONLY;
    cwin->text = createComposeTextField("message_text", cwin->panes, 
					text, 0, text_trans, 
					loc_status);
    
    XtSetArg(cargs[cn], XtNfont, &font); cn++ ;
    XtGetValues(cwin->text, cargs, cn); cn = 0;
    
    height = (Dimension)(font->max_bounds.ascent + font->max_bounds.descent) * 20;
    width = (Dimension)(font->max_bounds.width) * (res.line_length);
    
    XtSetArg(cargs[cn], XtNheight, height); cn++ ;
    XtSetArg(cargs[cn], XtNwidth,  width); cn++;
    XtSetValues(cwin->text, cargs, cn); cn = 0;

    checkButtonStates(cwin->states, cwin->state);
    XtPopup(cwin->shell, XtGrabNone);
    
    XtSetKeyboardFocus( cwin->shell, cwin->from);
    cwin->lastFocus = cwin->from;
    
    XtFree(from);
    XtFree(to);
    XtFree(cc);
    XtFree(bcc);
    XtFree(text);
    return( cwin );
}

char**  makeHostList( resources, hosts )
     AppResources *resources;
     char      **hosts;
{
    char **ret = NULL;
    char **temp = NULL;

    int i = 0;

    for(i = 0; hosts[i]; i++)
	;
    
    temp = ret = (char**) XtMalloc ((i+2)*sizeof(char*));
    
    ret[0] = resources->smtpHost;
    while( *hosts )
	*++ret = *hosts++;

    *++ret = NULL;

    return(temp);
}

void sendMessage(w, cwin, ca)
     Widget w;
     ComposeWindow *cwin;
     XEvent *ca;
{
    char *gettext_for_send(/* Widget */);
    char temp[TMPLEN];
    static ADDRESS* GetAddress(/* Widget, char* */);
    SMTPSTREAM *stream;
    static char **hosts = NULL;
    
    BODY     *newbody;

    ENVELOPE *mtpMsg = mail_newenvelope();
    ENVELOPE *oldenv = cwin->env;

    newbody = mail_newbody();

    /* can't send without being logged in... */
    if(mail_stream(cwin->handle))
    {
	if(res.smtpHost)
	    hosts = makeHostList( &res, hostlist );
	else 
	    hosts = hostlist;
	
	/* if we have a mailer... */ 
	if (stream = smtp_open((char**)hosts, 
			      (cwin->status & DEBUGGING))) 
	{
	    /* free any old text space */ 
	    if (oldenv->remail)
		mtpMsg->remail = cpystr(oldenv->remail);
	    if (oldenv->in_reply_to)
		mtpMsg->in_reply_to = cpystr(oldenv->in_reply_to);
	    
	    mtpMsg->date = cpystr(oldenv->date);
	    
	    if(cwin->status & (REMAILMSG))
	    {
		newbody = cwin->body;
	    }
	    else
	    {
		newbody->contents.text = (unsigned char*)gettext_for_send(cwin->text);
		newbody->type = TYPETEXT;
		newbody->encoding = ENC7BIT;
	    }
	    
	    /* Here we create a multi-part if necessary */
	    if(cwin->attachments)
	    {
		BODY *temp = newbody;
		PART *part = mail_newbody_part();
		newbody = mail_newbody();

		part->body.type = temp->type;
		part->body.encoding = temp->encoding;
		if(temp->description)
		    part->body.description = cpystr(temp->description);
		switch(temp->type)
		{
		  case TYPETEXT:
		  case TYPEMESSAGE:
		    part->body.contents.text = temp->contents.text;
		    break;
		  case TYPEMULTIPART:
		    part->body.contents.part = temp->contents.part;
		    break;
		  default:
		    part->body.contents.binary = temp->contents.binary;
		    part->body.size.bytes = temp->size.bytes;
		    break;
		}
		
		part->next = cwin->attachments;
		cwin->attachments = part;

		newbody->type = TYPEMULTIPART;
		newbody->encoding = ENC7BIT;
		newbody->contents.part = cwin->attachments;
	    }

	    mtpMsg->to = GetAddress(cwin->to, cwin->host);
	    mtpMsg->cc = GetAddress(cwin->cc, cwin->host);
	    mtpMsg->bcc = GetAddress(cwin->bcc, cwin->host);
	    
	    /* 
	      "sender", "return_path" & "reply_to" will, at least
	      for the moment, be identical with "from", unless we're
	      remailing the message...
	      */
	    mtpMsg->from = GetAddress(cwin->from, cwin->host);

	    if( oldenv->reply_to )
		mtpMsg->reply_to = copy_address(oldenv->reply_to);
	    else
		mtpMsg->reply_to = copy_address(mtpMsg->from);

	    mtpMsg->return_path = mail_newaddr();
	    mtpMsg->return_path->mailbox = cpystr(mtpMsg->from->mailbox);
	    mtpMsg->return_path->host = cpystr(mtpMsg->from->host);

	    if (   !cwin->message  
		|| cwin->message->status & ~REMAILMSG )
	    {
		mtpMsg->sender = mail_newaddr();
		mtpMsg->sender->mailbox = cpystr(mtpMsg->from->mailbox);
		mtpMsg->sender->host = cpystr(mtpMsg->from->host);
	    }
	    
	    mtpMsg->subject = gettext_for_send(cwin->subject);
	    
	    sprintf (temp,"<%s.%d.%d.%s@%s>", 
		     "Ximap",time (0),rand () % 10000,id.username,id.localhost);

	    mtpMsg->message_id = cpystr(temp);

	    /* send mail */ 
	    if (smtp_mail (stream,"MAIL",mtpMsg,newbody)) 
	    {
		if( cwin->message )
		    doSetFlag(cwin->ms, 
			      ximap_newnodelist(&cwin->message, 1), 
			      ANSWERED_FLAG );

		sprintf(temp,"Message sent successfully");
		mm_log(temp, NIL);
		sendQuit(NULL,cwin,NULL);
	    }
	    else 		/* otherwise report recipient errors */ 
	    {
		sprintf(temp,"Failed send");
		mm_log(temp, ERROR);
	    }
	    smtp_close (stream);		/* nuke the stream in any case */ 
	}
	else
	{
		sprintf(temp,"Failed send");
		mm_log(temp, ERROR);
	}
    }
    /* Note: We do NOT free oldenv == cwin->env here because the send MAY
     * have failed, and therefor the cwin->env can be again used above.
     * This is liberated in sendQuit(..).
     *                   Bill Yeager -- 3 Dec 92 -- Another seg fault enleve */
    mail_free_envelope(&mtpMsg);
    mail_free_body(&newbody);
}

static ADDRESS* GetAddress(w, host)
     Widget w;
     char *host;
{
    ADDRESS *ret = NULL;
    
    char *t1;
    char *t2 = XbTextCopyBuffer(w);
    
    if( t2 )
    {
	t1 = gettext_for_display(t2);
	remove_eol(t1);
	rfc822_parse_adrlist(&ret, t1, host);
	XtFree(t1);
	XtFree(t2);
    }

    return(ret);
}

void sendQuit( w, cwin, ca)
     Widget w;
     ComposeWindow *cwin;
     XEvent *ca;
{
    ComposeWindow *ctemp;

    if ( !cwin ) 
      return;

    if ( cwin->shell)
      XtDestroyWidget( cwin->shell);

    free_statelist(cwin->states);
    ctemp = cwin->ms->compList;
    if( ctemp )
    {   
	if (ctemp == cwin)
	{
	    cwin->ms->compList = cwin->next;
	}
	else
	{
	    while( ctemp->next && ctemp->next !=  cwin )
		ctemp = ctemp->next;
	    
	    if ( ctemp->next )
		ctemp->next = cwin->next;
	}
    }

    if(cwin->message && cwin->message->insert)
    {
	XtFree(cwin->message->insert);
	cwin->message->insert = NULL;
    }
    /*
     * We free the envelope here because multiple sendMessage calls
     * can occur if a send fails. The envelope is used there
     *                     Bill Yeager -- 3 Dec 92 */
    if (cwin->env)
      mail_free_envelope(&cwin->env);
    XtFree(cwin);
}


static void compBeep( w, e )
     Widget w;
     XEvent* e;
{
    
    
}

static void gotoCompFrom( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);

    XawTextDisplayCaret( cwin->lastFocus, FALSE );
    XtSetKeyboardFocus( cwin->shell, cwin->from);
    XawTextDisplayCaret( cwin->from, TRUE );
    cwin->lastFocus = cwin->from;
}

static void gotoCompTo( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);

    XawTextDisplayCaret( cwin->lastFocus, FALSE );
    XtSetKeyboardFocus( cwin->shell, cwin->to);
    XawTextDisplayCaret( cwin->to, TRUE );
    cwin->lastFocus = cwin->to;
}

static void gotoCompCc( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);

    XawTextDisplayCaret( cwin->lastFocus, FALSE );
    XtSetKeyboardFocus( cwin->shell, cwin->cc);
    XawTextDisplayCaret( cwin->cc, TRUE );
    cwin->lastFocus = cwin->cc;
}

static void gotoCompBcc( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);

    XawTextDisplayCaret( cwin->lastFocus, FALSE );
    XtSetKeyboardFocus( cwin->shell, cwin->bcc);
    XawTextDisplayCaret( cwin->bcc, TRUE );
    cwin->lastFocus = cwin->bcc;
}

static void gotoCompSubject( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);

    XawTextDisplayCaret( cwin->lastFocus, FALSE );
    XtSetKeyboardFocus( cwin->shell, cwin->subject);
    XawTextDisplayCaret( cwin->subject, TRUE );
    cwin->lastFocus = cwin->subject;
}

static void gotoCompText( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);

    XawTextDisplayCaret( cwin->lastFocus, FALSE );
    XtSetKeyboardFocus( cwin->shell, cwin->text);
    XawTextDisplayCaret( cwin->text, TRUE );
    cwin->lastFocus = cwin->text;
}

static void compDone( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);
    sendQuit(cwin->shell, cwin, NULL);
}

static void compSendIt( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);
    sendMessage( w, cwin, e );
}

static void compSaveIt( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);
    saveDraftToFile( w, cwin, e );
}

static void compRestoreIt( w, e )
     Widget w;
     XEvent* e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);
    restoreDraftFromFile( w, cwin, e );
}

static void compInsertMessage( w, e )
     Widget  w;
     XEvent *e;
{
    ComposeWindow *cwin = getCompWinFromWidget(w);
    insertMessageText(w, cwin, e);
}

void insertMessageText(w, cwin, ca)
     Widget w;
     ComposeWindow *cwin;
     XEvent *ca;
{
    Widget textwidget = cwin->text;
    if(cwin->rwin) {
      if(cwin->message && cwin->message->insert)
	XtFree(cwin->message->insert);
      cwin->message->insert = gettext_for_insert(cwin->rwin,
		cwin->message->text, res.indentPrefix);
      XbInsertAtCurrentPoint(textwidget, cwin->message->insert);
    }
      
    else {
      if(cwin->message && cwin->message->insert)
	XbInsertAtCurrentPoint( textwidget, cwin->message->insert );
    }
}

void saveDraftToFile(w, cwin, ca)
     Widget w;
     ComposeWindow *cwin;
     XEvent *ca;
{
    char *saveFile = NULL;
    char *homeDir = NULL;
    char *cptr = NULL;
    char temp[TMPLEN];
    FILE *ofp = NULL;
    ENVELOPE *oldenv = cwin->env;
    struct stat sb;
    
    saveFile = getFilename(cwin->shell, getHomeDir(), 0, 0);

    if (!saveFile)
      return;

    /* Only overwrite existing files with permission */
    if (stat(saveFile, &sb) == 0) {
      if (XbConfirm(cwin->shell,
		    "Overwrite an existing file?",
		    TRUE) == FALSE) {
	free(saveFile);
	return;
      }
    }


    if((ofp = fopen( saveFile, "w" )) == NULL)
    {
      sprintf(temp, "Cannot open file for output");
      XbWarning(cwin->shell, temp);
      return;
    }
    free(saveFile);
    if (oldenv->in_reply_to)
	fprintf(ofp,"InReplyTo: %s\n",oldenv->in_reply_to);

    cptr = XbTextCopyBuffer(cwin->to);
    if ( cptr )
	fprintf(ofp,"To: %s\n", cptr);

    cptr = XbTextCopyBuffer(cwin->cc);
    if ( cptr )
	fprintf(ofp,"Cc: %s\n", cptr);

    cptr = XbTextCopyBuffer(cwin->from);
    if ( cptr )
	fprintf(ofp,"From: %s\n", cptr);

    cptr = XbTextCopyBuffer(cwin->subject);
    if ( cptr )
	fprintf(ofp,"Subject: %s\n", cptr);

    cptr = XbTextCopyBuffer(cwin->text);
    if ( cptr )
	fprintf(ofp,"Body: %s\n", cptr);

    fclose(ofp);
}

void restoreDraftFromFile(w, cwin, ca)
     Widget w;
     ComposeWindow *cwin;
     XEvent *ca;
{
    char *restoreFile = NULL;
    char *homeDir = NULL;
    char temp[TMPLEN];
    char *tptr = temp;
    char *cptr = NULL;
    FILE *ifp = NULL;
    int length = 0;
    Boolean inBody = FALSE;
    Boolean endOfFile = FALSE;

    ENVELOPE *newenv = mail_newenvelope();

    restoreFile = getFilename(cwin->shell, getHomeDir(), 0, 0);

    if (!restoreFile)
      return;

    if((ifp = fopen( restoreFile, "r" )) == NULL)
    {
      sprintf(temp, "Cannot open file %s for input", restoreFile);
      XbWarning(cwin->shell, temp);
      return;
    }
    free(restoreFile);
    while(!endOfFile)
    {
	while(++length < TMPLEN
	      && (*tptr = getc(ifp)) != EOF
	      && ((inBody == TRUE) || *tptr != '\n'))
	{
	    tptr++;
	}
	    
	if(*tptr == EOF)
	    endOfFile = TRUE;
	
	if( *temp == 'B' || inBody == TRUE )
	    if( *tptr == '\n' )
		tptr++;

	*tptr = '\0';
	
	cptr = cpystr(temp);
	if( !inBody )
	{
	    switch(*cptr)
	    {
	      case 'I':
		cptr = strcut(cptr, "InReplyTo: ", TRUE);
		newenv->in_reply_to = cptr;
		break;
	      case 'T':
		cptr = strcut(cptr, "To: ", TRUE);
		XtSetArg(cargs[0], XtNstring, cptr);
		XtSetValues(cwin->to, cargs, 1);
		break;
	      case 'C':
		cptr = strcut(cptr, "Cc: ", TRUE);
		XtSetArg(cargs[0], XtNstring, cptr);
		XtSetValues(cwin->cc, cargs, 1);
		break;
	      case 'F':
		cptr = strcut(cptr, "From: ", TRUE);
		XtSetArg(cargs[0], XtNstring, cptr);
		XtSetValues(cwin->from, cargs, 1);
		break;
	      case 'S':
		cptr = strcut(cptr, "Subject: ", TRUE);
		XtSetArg(cargs[0], XtNstring, cptr);
		XtSetValues(cwin->subject, cargs, 1);
		break;
	      case 'B':
		XbTextClearBuffer(cwin->text);
		cptr = strcut(cptr, "Body: ", TRUE);
		XbTextAppend(cwin->text, cptr);
		inBody = TRUE;
		break;
	      default:
		break;
	    }
	}
	else
	{
	    XbTextAppend( cwin->text, cptr );
	}
	tptr = temp;
	length = 0;
    }
    fclose(ifp);
    cwin->env = newenv;
}


static void attachFile(w, cwin, e)
     Widget w;
     ComposeWindow *cwin;
     XEvent *e;
{
    char temp[TMPLEN];

    char *filename = getFilename(cwin->shell, getHomeDir(), 0, 0);
    if(filename)
    {
	FILE *ifp = fopen(filename, "r");
	if(ifp)
	{
	    BinaryBuffer *bb = readFileToMem(ifp);
	    if(bb)
	    {
		XbListSelection *new;

		/* Get new PART and insert info */
		PART *part = mail_newbody_part();
		
		if(!getTypeInfo(cwin->shell, &part->body, bb, filename))
		{
		    part->body.contents.text = rfc822_binary (bb->bufferContents,
							      bb->bufferSize,
							      &part->body.size.bytes);
		    
		    part->body.type = TYPEAPPLICATION;
		    part->body.subtype = cpystr("octet-stream");
		    part->body.encoding = ENCBASE64;
		    
		    part->body.description = cpystr(filename);
		}

		/* Add PART to list of attachments */
		if(cwin->attachments == NULL)
		{
		    cwin->attachments = part;
		    XtManageChild(cwin->part_browser);
		}
		else
		{
		    PART *pt = cwin->attachments;
		    while(pt->next)
			pt = pt->next;
		    part->next = pt->next;
		    pt->next = part;
		}

		new = makeMultiPartHeaderList(cwin->attachments);
		XbBrowserChangeItems(cwin->part_browser, new);
		composeResizeBrowser(cwin);

		cwin->state &= ~NOATTACHEDPART;
		checkButtonStates(cwin->states, cwin->state);
	    }
	    else
	    {
		sprintf(temp, "Failed readFileToMem()");
		XbWarning(cwin->shell, temp);
	    }

	    fclose(ifp);
	}
	else
	{
	    sprintf(temp, "Cannot open file for read");
	    XbWarning(cwin->shell, temp);
	}
	free(filename);			/* Bill Yeager -- 27 Oct 92 */
    }
}

static void removePart(w, cwin, e)
     Widget        w;
     ComposeWindow *cwin;
     XEvent        *e;
{
    if(cwin->attachments)
    {
	XbListSelection *sel = XbBrowserCurrentHighlight(cwin->part_browser);
	if(sel->size)
	{
	    int i = sel->size;
	    int j = 0;
	    PART *temp, *last = NULL;
	    XbListSelection *new = NULL;

	    while(i--)
	    {
		if(j = sel->entries[i].index)
		{
		    temp = cwin->attachments;
		    while(--j)
		    {
			last = temp;
			temp = temp->next;
		    }
		    
		    if(last)
			last->next = temp->next;
		    else
			cwin->attachments = temp->next;

		    mail_free_body_part(&temp);
		}
	    }

	    new = makeMultiPartHeaderList(cwin->attachments);
	    XbBrowserChangeItems(cwin->part_browser, new);

	    /* Now, if we removed the last attachment... */
	    if(!cwin->attachments)
	    {
		cwin->state |= NOATTACHEDPART;
		checkButtonStates(cwin->states, cwin->state);
		XtUnmanageChild(cwin->part_browser);
	    }
	    else
	    {
		composeResizeBrowser(cwin);
	    }
	}
    }
}

static void composeResizeBrowser(cwin)
     ComposeWindow *cwin;
{
    XtWidgetGeometry ask, reply;
    Dimension width = 0;
    int cn= 0;


    XtSetArg(cargs[cn], XtNwidth, &width); cn++;
    XtGetValues(cwin->panes, cargs, cn); cn = 0;
	
    ask.request_mode = CWHeight | XtCWQueryOnly;
    XtQueryGeometry(cwin->part_browser, &ask, &reply);

    XtSetArg(cargs[cn], XtNheight, reply.height); cn++;
    XtSetValues(cwin->part_browser, cargs, cn); cn = 0;
}

