/* -*- C -*-
 *
 * Program:	ximap
 * File:        textutil.c -- manipulating text.  Text widget convenience functions
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 * Changelog:
 *    - changed gettext_for_insert() to reformat the text to a user definable 
 *      line length before insertion, i.e.  "copywrap()" (in copywrap.c).
 *      Macgirvin (Mike_Macgirvin@CAMIS.Stanford.EDU) 11-16-92
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#ifdef MOTIF
#include <X11/Xm/Text.h>
#else
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Text.h>
#endif

#include "structures.h"
#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "readwin.h"
#include "textutil.h"

extern AppResources res;

void XbTextAppend( w, text )
     Widget w;
     char* text;
{
    XawTextPosition start, end;
    XawTextBlock temp;
    
    temp.firstPos = 0;
    temp.length = strlen(text);
    temp.ptr = (char*) XtMalloc ((1+temp.length) * sizeof(char));
    temp.format = FMT8BIT;
    
    strncpy( temp.ptr, text, temp.length+1 );
    start = XawTextGetInsertionPoint( w );
    end = start + temp.length;
    
    XawTextReplace( w, start, end, &temp );
    XawTextSetInsertionPoint( w, end );
    
    XtFree(temp.ptr);
}

void XbInsertAtCurrentPoint( w, text )
     Widget   w;
     char    *text;
{
    XawTextPosition start, end;
    XawTextBlock temp;
 
    start = XawTextGetInsertionPoint(w);

    temp.firstPos = 0;
    temp.length = strlen(text);
    temp.ptr = (char*) XtMalloc ((1+temp.length) * sizeof(char));
    temp.format = FMT8BIT;
    
    strncpy( temp.ptr, text, temp.length+1 );
    end = start = XawTextGetInsertionPoint( w );
    XawTextReplace( w, start, end, &temp );
    XtFree(temp.ptr);
}

void XbTextClearBuffer( w )
     Widget w;
{
    Arg warg[4];
    int n = 0;
    
    char* empty = "";
    
    XtSetArg(warg[n], XtNstring, empty); n++;
    XtSetValues( w, warg, n ); n = 0;
}

char* XbTextCopyBuffer( w )
     Widget w;
{
#ifdef MOTIF
    char *ret = NULL;
    char* temp = NULL;
    XtVaGetValues(w, XmNvalue, &temp, NULL);
    ret = (char*) malloc (1 + strlen(temp));
    strcpy(ret,temp);
    XtFree(temp);
    return(ret);
#else
    Arg warg[2];
    int n = 0;
    
    char *str;
    char *ret = NULL;
    
    XtSetArg(warg[n], XtNstring, &str); n++;
    XtGetValues(w, warg, n);
    
    if ( str != NULL && strlen(str) > 0)
	ret = cpystr(str);
    
    return(ret);
#endif
}


/* 

  changes CRLF to LF

*/

char* fixnl (text)
     char *text;
{
    register int i,j;
    if (text) 
    {			
	/* we don't want CRLF at the end of lines, just LF,
	   so... */
	for (i = j = 0; text[i] != '\0'; i++)
	    if(text[i] != CRCHAR || text[i+1] != LFCHAR ) 
		text [j++] = text[i];
      
	text[j] = '\0';		/* tie off end of string */
    }
    return (text);
}

/*

  un_fixnl -- Changes bare LF or CR to CRLF

*/

void un_fixnl (dest, text)
     char *dest;
     char *text;
{
    register int i,j;

    if (text && dest) 
    {			
	/* 
	  for send we want crlf at the end of lines,
	  so...
	*/
	for (i = j = 0; text[i] != '\0'; i++)
	{
	    if (text[i] == LFCHAR)
		dest[j++] = CRCHAR;
	    else if (text[i] == CRCHAR
		     && text[i+1] == LFCHAR)
		dest[j++] = text[i++];

	    dest[j++] = text[i];

	    if (text[i] == CRCHAR)
		dest[j++] = LFCHAR;
	}
      
	dest[j] = '\0';		/* tie off end of string */
    }
}

char* gettext_for_display( src )
     char *src;
{
  int length;
  char* text ;
  if(src == NULL)
    return(NULL);
  length = strlen(src);
  text = (char*)XtMalloc((length + 1) * sizeof(char));
  strncpy( text, src, length + 1 );
  fixnl(text);
  return(text);
}

char* gettext_for_send(text_widget)
     Widget text_widget;
{
  int length;
  char *text = NULL;
  char *retval = NULL;
  char *src = XbTextCopyBuffer(text_widget);  

  if( src )
  {
      length = strlen(src);
      if( length > 0 )
      {
	  text = (char*)XtMalloc((length + 1) * sizeof(char));
	  retval = (char*)XtMalloc(2 * (length + 1) * sizeof(char));
	  memset( text, 0, length );
	  strncpy( text, src, length + 1 );
	  fix_line_width(retval, text, res.line_length);
	  XtFree(text);
      }
      XtFree(src);
  }

  return(retval);
}


char* gettext_for_insert( rwin, src, prefix )
     READWINDOW * rwin;
     char *src;
     char *prefix;
{
    int   carets = 0;
    int   length = 0;
    int   prefixLength;
    char *insert = NULL;
    char *source = NULL;
    char * hilited = NULL;
    unsigned insert_size = 0;

    XawTextPosition begin_return = (XawTextPosition) 0;
    XawTextPosition end_return = (XawTextPosition) 0;

    register char *temp = NULL;
    register char *srctemp = NULL;

    if(src == NULL || *src == '\0')
      return("");

    prefixLength = strlen(prefix);

    if(rwin != NULL && rwin->text != NULL) {
      XawTextGetSelectionPos(rwin->text,&begin_return,&end_return);
      if((begin_return != end_return) && begin_return >= rwin->headerlen 
	 && end_return >= rwin->headerlen) {
	insert_size = (unsigned) (end_return - begin_return);
	hilited = (char *) XtMalloc(insert_size + 1);
      }
    }
 
    if(hilited != NULL) {
      strncpy(hilited,src + begin_return - rwin->headerlen,insert_size);
      *(hilited + insert_size) = '\0';
      source = srctemp = copywrap(hilited);
      XtFree(hilited);
    }
    else 
      source = srctemp =  copywrap(src);

    carets = countnl(srctemp);
    length = (carets*prefixLength) + strlen(srctemp) + 1;
    insert = XtMalloc(length+1);

    strcpy( insert, prefix );
    temp = insert + prefixLength;
    while( *srctemp )
    {
	*temp++ = *srctemp;
	if( *srctemp++ == LFCHAR )
	{
	    strncpy( temp, prefix, prefixLength );
	    temp += prefixLength;
	}
    }
    *temp = '\0';
    XtFree(source);
    return( insert );
}

int countnl(text)
     char* text;
{
    register char *temp = text;
    register int  count = 1;
    
    while(*temp)
	if (*temp++ == LFCHAR)
	    ++count;
    
    return count;
}





/* Function Name: strncut
 * Purpose: removes occurrences of fragment from universe and returns 
 *          a new string as the result
 * Arguments: universe -       char*, string to be modified.
 *            fragment -       char*, string to remove from s1
 *            more_matches -   int, number of occurrences to remove, negative
 *                           means remove *all* occurrences.
 *            case_sensitive-  int, non-zero = yes
 *
 * Returns: modified copy of universe
 */

char* strncut( universe, fragment, case_sensitive, more_matches )
     char *universe;
     char *fragment;
     int case_sensitive;
     int more_matches;
{
    register char *unv = universe;
    register char *frg = fragment;
    register char *ret = NULL;
    
    char *start_of_match = NULL;  
    char *returnval = NULL;
    
    int max_length = strlen(universe);
    
    if ((ret = returnval = (char*) XtMalloc(max_length + 1)) == NULL)
	return(NULL);

    if (strlen(fragment) <= max_length)
    {
	if (case_sensitive)
	{
	    while (*unv && more_matches)
	    {
		if (*unv == *frg)
		{	
		    /* true only if starting new match */
		    if(frg == fragment)
			start_of_match = ret;
		    
		    ++frg;
		    
		    /* true only if match is complete (at end of fragment) */
		    if(*frg == '\0')
		    {
			--more_matches;
			ret = start_of_match;
			start_of_match = NULL;
			frg = fragment;
			++unv;
		    }
		}
		/* true only if we've just reached the end of a partial match */
		else if (start_of_match)
		{
		    start_of_match = NULL;
		    frg = fragment;
		}
		
		*ret++ = *unv++;
	    }
	}
	else
	{
	    while (*unv && more_matches)
	    {
		if (tolower(*unv) == tolower(*frg))
		{	
		    /* true only if starting new match */
		    if(frg == fragment)
			start_of_match = ret;
		    
		    ++frg;
		    
		    /* true only if match is complete (at end of fragment) */
		    if(*frg == '\0')
		    {
			--more_matches;
			ret = start_of_match;
			start_of_match = NULL;
			frg = fragment;
			++unv;
		    }
		}
		/* 
		  true only if we've just reached the 
		  end of a partial match                 
		  */
		else if (start_of_match)
		{
		    start_of_match = NULL;
		    frg = fragment;
		}
		
		*ret++ = *unv++;
	    }
	} 
	
	if (*unv || *(ret-1))
	    while( *ret++ = *unv++ )
		; /* EMPTY */
    }
    else
    {
	while(*ret++ = *unv++)
	    ;
    }
    
    return(returnval);
}


/* Function Name: strcut
 * Purpose: removes first occurrence of fragment from universe and returns 
 *          a new string as the result
 * Arguments: universe -       char*, string to be modified.
 *            fragment -       char*, string to remove from s1
 *            case_sensitive-  int, non-zero = yes
 *
 * Returns: modified copy of universe
 */

char* strcut( universe, fragment, case_sensitive )
     char *universe;
     char *fragment;
     int case_sensitive;
{
    register char *unv = universe;
    register char *frg = fragment;
    register char *ret = NULL;
    
    char *start_of_match = NULL;  
    char *returnval = NULL;
    
    int max_length = strlen(universe);
    
    if ((ret = returnval = (char*) XtMalloc(max_length + 1)) == NULL)
	return(NULL);
    
    if (strlen(fragment) <= max_length)
    {
	if (case_sensitive)
	{
	    while (*unv && *frg)
	    {
		if (*unv == *frg)
		{	
		    /* true only if starting new match */
		    if(frg == fragment)
			start_of_match = ret;
		    
		    ++frg;
		    
		    /* true only if match is complete (at end of fragment) */
		    if(*frg == '\0')
		    {
			ret = start_of_match;
			++unv;
		    }
		}
		/* true only if we've just reached the 
		   end of a partial match 
		   */
		else if (start_of_match)
		{
		    start_of_match = NULL;
		    frg = fragment;
		}
		
		/* always write *something*...*/
		*ret++ = *unv;
		if( *unv )
		    ++unv;
	    }
	}
	else
	{
	    while (*unv && *frg)
	    {
		if ((isupper(*unv)?tolower(*unv):*unv) 
		    == (isupper(*frg)?tolower(*frg):*frg))
		{	
		    /* true only if starting new match */
		    if(frg == fragment)
			start_of_match = ret;
		    
		    ++frg;
		    
		    /* true only if match is complete (at end of fragment) */
		    if(*frg == '\0')
		    {
			ret = start_of_match;
			++unv;
		    }
		}
		/* true only if we've just reached the end 
		   of a partial match 
		   */
		else if (start_of_match)
		{
		    start_of_match = NULL;
		    frg = fragment;
		}
		
		/* always write *something*...*/
		*ret++ = *unv++;
	    }
	}
	
	if (*unv || *(ret - 1))
	    while( *ret++ = *unv++ )
		; /* EMPTY */
    }
    else
    {
	while(*ret++ = *unv++)
	    ;
    }
    
    return(returnval);
}

/* 
  returns 0 if there are no differences (excluding case)
  between the two strings
*/
int areDifferentStrings(s1, s2)
     char *s1;
     char *s2;
{
    if(s1 && s2)
    {
	while((*s1 && *s2)
	      && ((isupper(*s1)?tolower(*s1):*s1) 
		  == (isupper(*s2)?tolower(*s2):*s2)))
	{
	    s1++;
	    s2++;
	}
	
	return( !(*s1 == '\0' && *s2 == '\0'));
    }
    else 
    {
	return(s1 || s2);
    }
}

void remove_eol(str)
     char *str;
{
    char *ret = str;
    

    while(*str)
    {
	while ( *str == LFCHAR 
	       || *str == CRCHAR )
	{
	    str++;
	}

	if (*str)
	    *ret++ = *str++;
    }
    
    *ret = *str;
}

void write_line(dest, endofline, startofline)
     char  **dest;
     char  **endofline;
     char  **startofline;
{
    int i = *endofline - *startofline;
    
    memcpy (*dest, *startofline, i);

    *dest += i;			/* update destination */
}

void fix_line_width (dest, text, maxlen)
     char *dest;
     char *text;
     int maxlen;
{
    char *src = text;
    char *dst = dest;

    char *eoword = NULL;
    char *start_of_line = text;

    int i = 0;			/* init character position */

    while (*src) 
    {
	switch (*src) 
	{	/* until source runs out */
	  case CRCHAR:
	  case LFCHAR:
	    write_line (&dst,&src,&start_of_line);
	    if (*src == LFCHAR) 
	    {
		*dst++ = CRCHAR;	        /* write CR before bare LF */
		*dst++ = *src++;
	    }
	    else if(*(src+1) != LFCHAR)
	    {
		*dst++ = *src++;	        /* write LF after bare CR */
		*dst++ = LFCHAR;
	    }
	    i = 0;
	    start_of_line = src;
	    eoword = NULL;
	    break;
	  case TAB:		    
	    i += 7;			/* move to in front of stop, fall into space */
	  case SPACECHAR:
	    if (++i <= maxlen) 		/* count line, increment pointer */
	    {
		src++;
		eoword = src;
	    }
	    else 
	    {
		write_line (&dst,&src,&start_of_line);
		*dst++ = CRCHAR; 
		*dst++ = LFCHAR; 
		i = 0; 
		
		/* skip to end of whitespace */
		while (*src == SPACECHAR || *src == TAB) 
		    src++;

		start_of_line = src;
		eoword = NULL;
	    }
	    break;
	  default:		    
	    if (++i > maxlen) 
	    {	
		if(eoword != NULL)
		{
		    write_line (&dst,&eoword,&start_of_line);
		    start_of_line = eoword;
		}
		else
		{
		    write_line (&dst,&src,&start_of_line);
		    start_of_line = src;
		}

		*dst++ = CRCHAR; 
		*dst++ = LFCHAR; 
		
		eoword = NULL;
		src++;
		i = src - start_of_line;
	    }
	    else
	    {
		src++;
	    }
	    break;
	}
    }
    
    write_line (&dst,&src,&start_of_line); 
    *dst = '\0';			        /* tie off destination */
}



char *richtoplain( str )
     char *str;
{
    int c, i;
    char token[50];
    char *ret;
    char *temp = (char*) malloc (strlen(str));
    ret = temp;

    while(c = *str++) 
    {
	if (c == '<') 
	{
	    for (i=0
		 ;(c = *str++) != '>' && c
		 ;++i) 
	    {
		token[i] = isupper(c) ? tolower(c) : c;
	    }
	    if (c == '\0') 
		break;
	    token[i] = 0;
	    if (!strcmp(token, "lt")) 
	    {
		*temp++ = '<';
	    } 
	    else if (!strcmp(token, "nl")) 
	    {
		*temp++ = '\n';
	    } 
	    else if (!strcmp(token, "comment")) 
	    {
		while (strcmp(token, "/comment")) 
		{
		    while ((c = *str++) != '<' && c ) ;
		    if (c == '\0') 
			break;
		    for (i=0; (c = *str++) != '>'
			 && c; ++i) 
		    {
			token[i] = isupper(c) ?
			    tolower(c) : c;
		    }
		    if (c== '\0') 
			break;
		    token[i] = 0;
		}
	    } 
	} 
	else if (c != '\n') 
	{
	    *temp++ = c;
	}
    }
    *temp++ = '\n';
    *temp++ = '\0';

    return(ret);
}



