/*-------------------------------------------------------------*\
| anaclock --- analog clockface display, with digital readout	|
|								|
| This version randomly repositions and recolors the clockface	|
| every 15 minutes, so that it can run for long periods without	|
| burning in the image.						|
\*-------------------------------------------------------------*/

#define SOUNDS

#ifdef FANSI
/* turn FANSI screen-blanker off and back on */
#define FANSI_BLANKOFF()	puts("\x1b[>4;0z\n")
#define FANSI_BLANK		"\x1b[>4;16000z"
#else
#define FANSI_BLANKOFF()
#define FANSI_BLANK
#endif

const char errmsg[] =
	"syntax: anaclock <-p path-to-driver> <-c> <-d>\n"
	"  -p  <optional> path to find graphics driver in\n"
	"  -c  <optional> force CGA-level graphics\n"
	"  -d  <optional> debug mode\n"
	"options must appear in this order.\n";
const char signature[] = "analog clock by bobmon, v2.5 " __DATE__ FANSI_BLANK ;

#include <stdio.h>
#include <time.h>
#include <math.h>

#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <graphics.h>

#ifdef	SOUNDS
#include <dos.h>
#define FREQ_1 400		/* 15-minute frequency			*/
#define FREQ_2 800		/* hourly frequency			*/
int freq = FREQ_1;		/* The frequency to be sounded.		*/
#endif	/* SOUNDS */

#define RADIUS 60		/* "Raw" halfsize of the clockface.	*/
#define UP      0
#define LEFT   20

#define BKGND BLACK
#define VALS  WHITE
#define SHADE   2
#define HANDS	1

#define BAD_VAL -999		/* flag value for uninitialized vars.	*/

float aspect;			/* Aspect ratio of the display screen.	*/
int x_radius, y_radius;		/* Radius of clock in each direction;	*/
				/*   x is adjusted for aspect ratio.	*/
int xlft = 0, xmid, xwide;	/* Clock size and location on x-axis.	*/
int ytop = 0, ymid, yhigh;	/* Size and location on y-axis.		*/
int ticks[12][2];		/* Relative coordinates of 5-minute	*/
				/*   tickmarks on the clockface.	*/
int h0 = 0, h1 = 1, h2 = 2, ht;	/* Current- & previous-pointers		*/
int m0 = 0, m1 = 1, m2 = 2, mt;	/*   into hand-position arrays.		*/
int hhands[3][2] =		/* Last 3 positions of the hour hand,	*/
{	BAD_VAL, BAD_VAL,	/*   used to "fade out" the hands.	*/
	BAD_VAL, BAD_VAL,	/*   Actually the hour hand is just	*/
	BAD_VAL, BAD_VAL	/*   redrawn in the SHADE color.	*/
};
int mhands[3][2] =		/* Last 3 positions of the minute	*/
{	BAD_VAL, BAD_VAL,	/*   hand, used to slowly "fade out"	*/
	BAD_VAL, BAD_VAL,	/*   the hand.				*/
	BAD_VAL, BAD_VAL
};

int this_h = -1, this_m = -1, this_s = -1;
int hr_angle, min_angle, sec_angle;

int ytpos, ydpos;		/* Y-positions of time, date displays.	*/
int ch_h, ch_w;			/* Pixel-sizes of a displayed char.	*/
char hr_str[4] = "  :";		/* Strings to hold digital times for	*/
char mn_str[4] = "  :";		/*   displaying --- passed to, and	*/
char sc_str[4] = "   ";		/*   updated by, disp_val().		*/
int sec_color = 1;		/* Displayed-secs color (starts at 1).	*/


/*---------------------------------------------------------------------*\
| Calculate the relative positions of the little circles that are drawn	|
| to indicate the 5-minute periods on the clockface (see new_face()).	|
\*---------------------------------------------------------------------*/
void maketicks(void)
{
    float k, r, x ,y;
    int a, d;

    k = M_PI / 180.0;
    for (d = a = 0; a < 12; ++a, d +=30) {
	r = k * (float)d;
	x = cos(r) * (float)(x_radius-2);
	y = sin(r) * (float)(y_radius-2);
	ticks[a][0] = (int)(x + (x >= 0 ? 0.5 : -0.5));
	ticks[a][1] = (int)(y + (y >= 0 ? 0.5 : -0.5));
    }
}

/*---------------------------------------------------------------------*\
| Draw a textstring in a given color, at a given position.		|
\*---------------------------------------------------------------------*/
void colorout(int color1, int color2, int x, int y, char *s)
{
    setcolor(color1);
    outtextxy(x+1, y+1, s);
    setcolor(color2);
    outtextxy(x, y, s);
}

/*---------------------------------------------------------------------*\
| Erase and replace a (numeric) value on the clockface.			|
\*---------------------------------------------------------------------*/
void disp_val(char *fmt, char *strng, int x, int newval)
{
    colorout(BKGND, BKGND, x, ytpos, strng);
    sprintf(strng, fmt, newval);
    colorout(SHADE, VALS, x, ytpos, strng);
}

/*---------------------------------------------------------------------*\
| Draw a hand on8the clockface, in the gyven color.  This is called	|
| to erase old hands, as well as to draw new ones.			|
\*---------------------------------------------------------------------*/
void draw_hand(int *tip, int width, int color)
{
    setcolor(color);
    setlinestyle(SOLID_LINE, 0, width);
    line(xmid, ymid, tip[0], tip[1]);
}

/*---------------------------------------------------------------------*\
| Draw an arc of a given color, about the face midpoint.		|
\*---------------------------------------------------------------------*/
void colorarc(int a0, int a1, int r, int color)
{
    setcolor(color);
    setlinestyle(SOLID_LINE, 0, NORM_WIDTH);
    arc(xmid, ymid, a0, a1, r);
}

/*---------------------------------------------------------------------*\
| Add cute little circles, w/ correct aspect ratio, to the hands.	|
\*---------------------------------------------------------------------*/
void ring(int x, int y, int color)
{
    setcolor(color);
    setlinestyle(SOLID_LINE, 0, NORM_WIDTH);
    ellipse(x, y, 0, 360, 2, 2);
}

/*---------------------------------------------------------------------*\
| Calculate a random new position for the clockface, and a random new	|
| color palette.  Then clear the screen and draw the new face.		|
\*---------------------------------------------------------------------*/
void new_face(void)
{
    int deltax, deltay, i;

  /*
  | Calculate new coordinates
  */
    deltax = random(getmaxx() - xwide) - xlft;
    deltay = random(getmaxy() - yhigh - 2*ch_h-1) - ytop;
    for (i = 0; i < 3; ++i) {
	hhands[i][0] += deltax;
	hhands[i][1] += deltay;
	mhands[i][0] += deltax;
	mhands[i][1] += deltay;
    }
    xlft += deltax;
    xmid = xlft + x_radius;
    ytop += deltay;
    ymid = ytop + y_radius;
    ytpos = ymid + 4*ch_h + 4;
    ydpos = ytop + yhigh + ch_h - 2;

    hr_angle = min_angle = sec_angle = 90;	/* Re-init. hand angles */

  /*
  | Draw new face
  */
    setgraphmode( random(4) );
    setlinestyle(SOLID_LINE, 0, NORM_WIDTH);
    setcolor(SHADE);

#ifdef SOUNDS
    sound(freq);
#endif

    cleardevice();
    circle(xmid, ymid, x_radius);
    for (i = 0; i < 12; ++i) {
	circle(xmid+ticks[i][0], ymid+ticks[i][1], 2);
	circle(xmid+ticks[i][0], ymid+ticks[i][1], 3);

#ifdef SOUNDS
    nosound();
    freq = FREQ_1;
#endif
    }

  /* Main routine will reset seconds-color to NOT(background). */
    sec_color = BKGND;
}

/*---------------------------------------------------------------------*\
| Copy a path-like string to a destination, and mark the last		|
| occurrence of a delimiting '\', '/', or ':'.				|
\*---------------------------------------------------------------------*/
char *copypath(char *dest, char *src, unsigned len)
{
    char *mark = NULL;
    while (*src && len) {
	if (*src == '\\' || *src == ':' || *src == '/')
	    mark = dest;
	*dest++ = *src++;
	--len;
    }
    if (len) *dest = 0;
    return mark;
}


/*---------------------------------------------------------------------*\
| The Main Routine.  So show some Respect.				|
\*---------------------------------------------------------------------*/

int _Cdecl main(int argc, char **argv)
{
    int g_mode, g_driver = DETECT;
    int x_asp, y_asp;
    int old_min_angle, old_sec_angle;
    char ampm[3] = "am";
    char daystr[17], *datestr;	/* String & ptr to hold day/month/year.	*/
    time_t secs;		/* Number of seconds, from time().	*/

    struct tm *tptr;
    int hour, hr_len, min_len;
    struct arccoordstype h_info, m_info;
    char driver_path[129] = "", *slash;
    int DEBUG = 0;

    if (argc >= 2 && argv[argc-1][0] == '-' && argv[argc-1][1] == 'd') {
	++DEBUG;
	--argc;
    }
    if (argc >= 2 && argv[argc-1][0] == '-' && argv[argc-1][1] == 'c') {
	g_driver = CGA;
	--argc;
    }
    if (argc >= 3 && argv[1][0] == '-' && argv[1][1] == 'p') {
	(void)copypath(driver_path, argv[2], 129);
    } else
    if (argc == 1) {
	slash = copypath(driver_path, argv[0], 129);
	if (slash)
	    *(slash+1) = 0;
    } else {
	puts(errmsg); puts(signature);
	return 0;
    }


    FANSI_BLANKOFF();
    randomize();
  /*
  | Set up graph mode and dimensions.
  */
    detectgraph(&g_driver, &g_mode);
    if (g_driver == CGA) {
	g_mode = CGAC3;			/* ...instead of CGA mono.	*/
    }
    initgraph(&g_driver, &g_mode, driver_path);

    getaspectratio(&x_asp, &y_asp);
    aspect = (float)y_asp / (float)x_asp;
    ch_w = textwidth("M");		/* standard character width	*/
    ch_h = textheight("M");		/* standard character height	*/
  /*
  | Set up the invariant values.
  */
    y_radius = RADIUS;
    yhigh = y_radius * 2;
    x_radius = (int)( (float)y_radius * aspect );
    xwide = (float)yhigh * aspect;
    hr_len = 0.7 * y_radius;
    min_len = y_radius;
    maketicks();	/* ...uses x_radius & y_radius. */

    new_face();		/* Start out with a new face. */
    colorarc(90, 89, min_len, VALS);
    getarccoords(&m_info);
    mhands[m0][0] = m_info.xstart;
    mhands[m0][1] = m_info.ystart;
    sec_color = HANDS;

  /*---------------------------------------------------------------*\
  | Update clock face continually until a keyboard hit is detected. |
  \*---------------------------------------------------------------*/

    if (DEBUG) {
	sec_color = BKGND; secs = 641613600L;	/* 10:00pm, 90/05/01	*/
    }
    do {
	if (DEBUG)
	    ++secs;		/* Loop fast, see what's happening. */
	else
	    time( &secs );
	tptr = localtime(&secs);

	if (this_s == tptr->tm_sec)	/* If time hasn't changed, */
	    continue;			/* skip out and try again  */

	/* else... */

      /*
      | Update the "Seconds circle".
      */
	old_sec_angle = sec_angle;
	this_s = tptr->tm_sec;
	sec_angle = 90 - 6 * tptr->tm_sec;
	colorarc(sec_angle, old_sec_angle, x_radius-5, sec_color);

      /*
      | If new minute, update the hands.  Angles of the hands are
      | 360 degrees times a fraction of a circle; offset by a right
      | angle and negated to convert from Clockwise orientation to
      | std. polar angles.
      */
	if (this_m != tptr->tm_min) {

	    putchar(8);		/* Fool screenblankers maybe? */
	    this_m = tptr->tm_min;
	    if (this_h != tptr->tm_hour) {	/* Update hour. */
		hour = tptr->tm_hour;
		if (hour < 12) {
		    ampm[0] = 'a';
		} else {
		    ampm[0] = 'p';
		    hour -= 12;
		}
		if (hour == 0) hour = 12;
	    }
	    if (this_h != hour) {
		this_h = hour;
#ifdef SOUNDS
		freq = FREQ_2;
#else
		putchar(7);		/* If new hour, beep the hour. */
#endif
	    }

	    if (0 == (this_m % 15)) {	/* Move the face every quarter hour. */
		new_face();
	    }

	  /* Write the date. */
	    datestr = ctime(&secs);
	    strncpy(daystr, datestr, 10);
	    daystr[10] = ',';
	    strncpy(daystr+11, datestr+19, 5);
	    daystr[16] = '\0';
	    colorout(SHADE, VALS, xmid-8*ch_w, ydpos, daystr);

	  /*
	  | Calculate new hand angles & coordinates,
	  | drawing arcs in the process.
	  */
	    hr_angle = 90 - ( (30 * hour) + (this_m >> 1) ) ;
	    if (hr_angle == 90) {
		h_info.xstart = xmid;
		h_info.ystart = hhands[h0][1];
	    } else {
		colorarc(hr_angle, 90, hr_len, VALS);
		getarccoords(&h_info);
	    }

	    old_min_angle = min_angle;
	    min_angle = 90 - (6 * this_m);
	    if (min_angle == 90) {
		m_info.xstart = xmid;
		m_info.ystart = mhands[m0][1];
	    } else {
		colorarc(min_angle, old_min_angle, min_len, VALS);
		getarccoords(&m_info);
	    }

	  /*
	  |"Fade out" the old hands...
	  */
	    if (hhands[h0][0] >= 0) {
		if (hhands[h1][0] >= 0) {
#if 0
		/** The following, isn't worth doing. **/
		    if (hhands[h2][0] >= 0)
			draw_hand(hhands[h2], THICK_WIDTH, BKGND);
#endif
		    draw_hand(hhands[h1], THICK_WIDTH, SHADE);
		}
		ring(hhands[h0][0], hhands[h0][1], BKGND);
	    }
	    if (mhands[m0][0] >= 0) {
		if (mhands[m1][0] >= 0) {
		    if (mhands[m2][0] >= 0)
			draw_hand(mhands[m2], NORM_WIDTH, BKGND);
		    draw_hand(mhands[m1], THICK_WIDTH, BKGND);
		    draw_hand(mhands[m1], NORM_WIDTH, SHADE);
		}
		draw_hand(mhands[m0], THICK_WIDTH, SHADE);
	    }

	  /* Store new hands... */
	    ht = h2; h2 = h1; h1 = h0; h0 = ht;
	    hhands[h0][0] = h_info.xstart;
	    hhands[h0][1] = h_info.ystart;
	    mt = m2; m2 = m1; m1 = m0; m0 = mt;
	    mhands[m0][0] = m_info.xstart;
	    mhands[m0][1] = m_info.ystart;

	  /* ...and draw the new hands. */
	    draw_hand(mhands[m0], THICK_WIDTH, HANDS);
	    draw_hand(hhands[h0], THICK_WIDTH, HANDS);
	    ring(hhands[h0][0], hhands[h0][1], SHADE);
	    ring(xmid, ymid, BKGND);

	  /*-----------------------------*\
	  | Now display the digital time. |
	  \*-----------------------------*/
	    disp_val("%2d:", hr_str, xmid-5*ch_w, this_h);
	    disp_val("%02d:", mn_str, xmid-2*ch_w, this_m);
	    colorout(SHADE, VALS, xmid+4*ch_w, ytpos, ampm);

	  /* Toggle the seconds-arc on and off, each minute. */
	    if (this_s == 0) {
		sec_color = ( sec_color == BKGND ?  HANDS  :  BKGND );
	    }
	}

	disp_val("%02d", sc_str, xmid+ch_w, this_s);

    } while ( !kbhit() );

    getch();
    closegraph();
    puts(signature);
    return 0;
}

/*--------------------------------------------------------------*/
