#include <stdio.h>
#include <unistd.h>
#include <glib.h>

#include "syntax_api.h"

/* some well-formatted objects */
#define NUM_GOOD_OBJECTS (sizeof(good_objects)/sizeof(good_objects[0]))
char *good_objects[] = {
  /* person object */
    "person:  Random Number\n"
    "address: Gaussian 1\n"
    "phone:   +1 (234) 567890\n"
    "nic-hdl: RNG1-RIPE\n"
    "changed: rng@ripe.net 20020624\n"
    "source:  RIPE\n",
  /* inetnum object */
    "inetnum:    10.0.0.0 - 10.1.255.255\n"
    "netname:    PRIVATE-NET\n"
    "descr:      We're just testing inetnum\n"
    "            so anything will do here, even a...\n"
    "+\n"
    "            blank line. :)\n"
    "country:    NL\n"
    "admin-c:    RNG1-RIPE\n"
    "admin-c:    FOO1-RIPE  # of course, we don't check refs here!\n"
    "tech-c:     BLAH-TEST\n"
    "rev-srv:    example.net\n"
    "rev-srv:    another-example.net as-a-ripe-list.com 192.168.1.2\n"
    "status:     ALLOCATED PA\n"
    "remarks:\n"
    "     More random remarkz here\n"
    "notify:     bitbucket@ripe.net\n"
    "mnt-by:     THIS-MNT\n"
    "mnt-lower:  THIS-MNT\n"
    "mnt-lower:  THAT-MNT\n"
    "mnt-routes: THOSE-MNTS\n"
    "mnt-routes: MORE-MNTS\n"
    "mnt-routes: STILL-MORE-MNTS\n"
    "mnt-irt:    IRT-XYZZY\n"
    "changed:    rng@ripe.net 20020624\n"
    "source:     WOW\n",
  /* as-set object */
    "as-set:   as-if\n"
    "descr:    just making sure lists work\n"
    "members:  as1, as2, as3,\n"
    "\t        as4, as5, as6\n"
    "mbrs-by-ref: if-not, why-not\n"
    "tech-c:   foo-bar\n"
    "admin-c:  foo-bar\n"
    "mnt-by:   not-mnt\n"
    "changed:  rng@ripe.net\n"
    "source: BoBaSRC\n"
};

/* some broken objects */
#define NUM_BAD_OBJECTS (sizeof(bad_objects)/sizeof(bad_objects[0]))
char *bad_objects[] = {
  /* unknown class */
    "frombulator:  WERDNA-FROM\n"
    "descr:        Just between you, me, and the fence\n"
    "wizard-c:     WRDA-WIZ\n"
    "source:       KROZ\n",
  /* missing required */
    "person:       Bhagwan Guru\n"
    "nic-hdl:      BHAG-GURU\n"
    "source:       GURU\n",
  /* unknown attribute */
    "route:        255.255.255.255/32\n"
    "descr:        Route at the End of the Internet\n"
    "table:        by.the.window\n"
    "origin:       AS65534\n"
    "mnt-by:       FORD-MNT\n"
    "changed:      ford@prefect.com\n"
    "source:       ADAMS\n",
  /* bogus attribute */
    "filter-set:   FLTR-OUT\n"
    "descr:        filter\n"
    "filter:       { 0.0.0.0/33 }\n"
    "tech-c:       FOUT1234\n"
    "admin-c:      FOUT1234\n"
    "mnt-by:       FOUT-MNT\n"
    "changed:      fout@fout.mnt 20020101\n"
    "source:       FOUT\n"
};

int
main (int argc, char *argv[])
{
    int c;
    int verbose;

    int i;
    rpsl_object_t *o;
    const GList *errs;
    const rpsl_error_t *e;
    const rpsl_attr_t *attr;
    const GList *attr_errs;

    /* get arguments (we only support "verbose" now) */
    verbose = 0;
    for (;;) {
        c = getopt(argc, argv, "v");
        if (c == EOF) break;

        switch (c) {
            case 'v': 
                verbose++;
                break;
            case '?':
                fprintf(stderr, "syntax: %s [-v]\n", argv[0]);
                exit(1);
            default:
                fprintf(stderr, "getopt() returned character '%c'\n", c);
                exit(1);
        }
    }
    if (optind != argc) {
        fprintf(stderr, "syntax: %s [-v]\n", argv[0]);
        exit(1);
    }

    /* check correct objects parse correctly */
    for (i=0; i<NUM_GOOD_OBJECTS; i++) {
        if (verbose) {
            printf("--[ Parsing good object ]------------------------\n");
            printf("%s", good_objects[i]);
        }

        rpsl_load_dictionary(RPSL_DICT_CORE);
        o = rpsl_object_init(good_objects[i]);
        if (rpsl_object_has_error(o, RPSL_ERRLVL_DEBUG)) {
            fprintf(stderr, "Unexpected error with object (core)!\n\n");
            exit(1);
        }
        rpsl_object_delete(o);

        rpsl_load_dictionary(RPSL_DICT_FRONT_END);
        o = rpsl_object_init(good_objects[i]);
        if (rpsl_object_has_error(o, RPSL_ERRLVL_DEBUG)) {
            fprintf(stderr, "Unexpected error with object (front-end)!\n\n");
            exit(1);
        }
        rpsl_object_delete(o);
        if (verbose) {
            printf("--[ Okay ]---------------------------------------\n\n");
        }
    }

    /* check bad objects break */
    rpsl_load_dictionary(RPSL_DICT_FRONT_END);
    for (i=0; i<NUM_BAD_OBJECTS; i++) {
        if (verbose) {
            printf("--[ Parsing bad object ]-------------------------\n");
            printf("%s", bad_objects[i]);
        }
        o = rpsl_object_init(bad_objects[i]);
        if (!rpsl_object_has_error(o, RPSL_ERRLVL_DEBUG)) {
            fprintf(stderr, "Unexpected success with object!\n\n");
            exit(1);
        }
        if (verbose) {
            printf("\n");

            errs = rpsl_object_errors(o);
            while (errs != NULL) {
                e = errs->data;
                printf("attr_num:%d level:%d code:%d descr:'%s'\n",
                    e->attr_num, e->level, e->code, e->descr);
                if (e->attr_num >= 0) {
                    attr = rpsl_object_get_attr_by_ofs(o, e->attr_num);
                    attr_errs = rpsl_attr_errors(attr);
                    while (attr_errs != NULL) {
                        e = attr_errs->data;
                        printf("      level:%d code:%d descr:'%s'\n",
                            e->level, e->code, e->descr);
                        attr_errs = g_list_next(attr_errs);
                    }
                }
                errs = g_list_next(errs);
            }

            printf("--[ Expected result ]----------------------------\n\n");
        }
        rpsl_object_delete(o);
    }

    return 0;
}

