#!PERL

#	$RCSfile: backupupd.pl,v $
#	$Revision: 1.4 $
#	$Author: ripe-dbm $
#	$Date: 1996/02/15 12:58:35 $

# This is a client that will update objects read from a file directly
# in the database, without interference of updated.
# It will update the object in the database that is linked to the
# source field of the object, so better make sure you have a source field
# and that it has a database associated to it ...

USE_DBM

@INC = ("LIBDIR", @INC);

require "adderror.pl";
require "cldb.pl";
require "dbadd.pl";
require "dbclose.pl";
require "dbopen.pl";
require "defines.pl";
require "enread.pl";
require "enwrite.pl";
require "getopts.pl";
require "misc.pl";
require "rconf.pl";
require "syslog.pl";

# Parse options:
#
#
# -h host            - hostname for update queries
# -s source          - which source to query for
#
# -V                 - verbose output, undocumented option

&Getopts('h:s:V');

if ((!$opt_h) || (!$opt_s)) {
 
print " 
   
Usage: $PROGRAMNAME -h hostname -s source
   
Where:
       hostname     - hostname for update queries
       source       - which source to query for\n\n\n";

   exit 1;
}


# Need this below for running perl in tainted mode.

$ENV{"PATH"} = "";
$ENV{"SHELL"} = "/bin/sh";
$ENV{"IFS"} = "";

# Read config file from RIPEDBCNF, or set to default.

$conffile=$ENV{"RIPEDBCNF"};
$conffile= "DEFCONFIG" unless $conffile;
print STDERR "backupupd - reading config\n" if $opt_V;
&rconf($conffile);

%entry=();

$locklink=$LOCKDIR.$PROGRAMNAME.".".$opt_s.".RUNNING";

$tmpdata=$TMPDIR."/".$PROGRAMNAME.".".$$;

#
# exit backupupd in a clean way
#

sub exitbackupupd {
   local($returncode)=@_;
   
   if (!$returncode) {
      unlink($tmpdata) || warn "couldn\'t remove temporary datafile: $tmpdata\n";
   }
   
   unlink($locklink) || warn "couldn\'t remove lockfile: $locklink\n";
   
   exit($returncode);
   
}

sub skiperrorserial {
   local(*entry,$action,$source,$msg);
   
   $entry{"pn"}="problem found" if (!&entype(*entry));
   $entry{"so"}=$source if (!$entry{"so"});
   
   &adderror(*entry,$msg);
   
   &writeseriallog($action,*entry);
     
}

#
# doaction ( )
#
# does all the work.
#

sub doaction {

    local(*entry,$action,$source,$serial) = @_;
    
    print STDERR "backupupd - doaction($action,$serial) called\n" if $opt_V;
    
    # open database file associated with "so" for writing

    local(*db) = 'currentdb';
    print STDERR "backupupd - doaction opening database\n" if $opt_V;

    if (&dbopen(db, *entry, 1)) {
       # Open classless database

       print STDERR "backupupd - doaction opening classless db\n" if $opt_V;
       &dbclopen(*entry,1);

       # Looks like we have some locking problem, so let's
       # lock the thing before we do anything else ...

       print STDERR "backupupd - doaction locking databases\n" if $opt_V;
       &dblock(*db);
		
       # do we delete or not ?

       if ($action=~ /^$DELETEACTION$/i) {

          print STDERR "backupupd - doaction deleting entry\n" if $opt_V;
          $dbstat = &dbdel(*db, *entry);

       }
       elsif ($action=~ /^$ADDACTION$/i) {
          print STDERR "backupupd - doaction calling dbadd\n" if $opt_V;
          $dbstat = &dbadd(*db, *entry);
       } 

       # noop, just print stat if verbose but always print errors!!!

       print STDERR "backupupd - doaction checking stat\n" if $opt_V;

       if ($dbstat == $E_NOOP) {
          print STDERR "ERROR, incomingserial: ", $LOGFILE{"INCOMINGSERIALDIR"}, $serial, "[NOOP]\n\n";
          &skiperrorserial(*entry,$action,$source,$MESSAGE[$dbstat]);
       }
       elsif ($dbstat!=$OK) {
          print STDERR "ERROR, incomingserial: ", $LOGFILE{"INCOMINGSERIALDIR"}, $serial, "\n", $MESSAGE[$dbstat], "\n\n";
          &skiperrorserial(*entry,$action,$source,$MESSAGE[$dbstat]);
       }
       elsif ($opt_V) {
          print STDERR "OK, serial: $serial\n";
       }

       &dbunlock(*db);
       &dbclose(*db);
       &dbclclose();
    }
}

sub UpdateVersionOne {
   local($input,$line,$opt_s,$from,$to)=@_;

   local(%entry);
   local($first,$last,$source);
   local($oldhandle,$oldflush);
   
   $line=~ /^\%START\s+Version:\s*\d+\s+(\S+)\s+(\d+)\-(\d+)\s*$/;
   
   $source=$1;
   $first=$2;
   $last=$3;
   
   $basename=$LOGFILE{"SERIALINCOMINGDIR"}.$source;
   
   print STDERR "basename: $basename source: $source first: $first last: $last\n" if $opt_V;
      
   $i=$first;
   $line=<$input>;
   
   die "got other start serial ($first) then expected ($from)\n" if ($first!=$from);
   die "got other end serial ($last) then expected ($to)\n" if (($to!~ /^LAST$/i) && ($last!=$to));
   die "got source ($source) then expected ($opt_s)\n" if ($source ne $opt_s); 
      
   while (($i<=$last) &&
          ($line=<$input>) &&
          ($line!~ /^\s*$/) &&
          ($line!~ /^\%END/)) {
   
      print STDERR "update: $basename.$i\n" if $opt_V;
      
      if ($line=~ /^$ADDACTION$/i) {
         $action=$ADDACTION;
      }
      elsif ($line=~ /^$DELETEACTION$/i) {
         $action=$DELETEACTION;
      }
      else {
         return $line;
      }
      
      open(OUTP,">$basename.$i");
      print OUTP $line;
      
      $line=<$input>;
      print OUTP $line;
      
      &enread(*entry,$input);
      
      $oldhandle=select(OUTP); $oldflush=$|; $|=1;
      &enwrite(*entry,0,1,0);
      $|=$oldflush; select($oldhandle); 
      
      close(OUTP);
      
      print STDERR "action: $action\n" if $opt_V;  
      
      &doaction(*entry,$action,$source,$i);
   
      $i++;
      
   }
   return $line;
}

#
# Main program
#

#
# check if this source *cannot* be updated with other methods...
#

if ($CANUPD{$opt_s}) {
   print STDERR "You cannot allow updates to database: $opt_s\n";
   print STDERR "*and* mirror this database at the same time.\n\n";
   print STDERR "Please remove your CANUPD variable for this database in the \'config\' file.\n";

   exit 1;
}

#
# create lock
#

# print STDERR "files: $locklink and $tmpdata -$$-\n";

if (!symlink($tmpdata,$locklink)) {
   
   local($linkdata)=readlink($locklink);
   
   local($pid)=$linkdata=~ /\.(\d+)$/;
   
   print STDERR "***ERROR***: found other $PROGRAMNAME (PID=$pid) running\n";
   
   print STDERR "please check for presence of process: $pid\n";
   print STDERR "link:  \'$locklink\'\nand file: \'$linkdata\'\n\n";
   print STDERR "The query interval in your cronjob might be too small if this\n";
   print STDERR "problem happens more often and you cannot find these files\n\n";
   print STDERR "If you find these files:\n";
   print STDERR "Remove them to restart automatic mirroring\n\n";
   print STDERR "Please send a bug report to <ripe-dbm@ripe.net> with\n";
   print STDERR "all data (\'ls -al $LOCKDIR\' and contents of \'$linkdata\')\n";
   print STDERR "when you cannot find an explanation for this problem\n";
   
   exit 1;
}

#
# and now really start running...
#

local($from,$to,$version,$line); 

$from=&getcurrentserial($opt_s)+1;

$to="LAST";

system("BINDIR/whois -h $opt_h -g $opt_s:$UPDATEVERSION:$from-$to 1\> $tmpdata 2\>\&1");

open(TMPFILE,"<".$tmpdata) || die "cannot open whois output\n";

local($outputfound)=0;

while (($line=<TMPFILE>) && ($line!~ /^\%START/)) { 
   print STDERR "skip:", $line if $opt_V;
   if (($line=~ /^(ERROR:\s*)?Warning\s*\(\s*1\s*\)\s*\:/i) ||
       ($line=~ /^Connect\s+failed/i) ||
       ($line=~ /^Connection\s+refused/i)) {
      close(TMPFILE);
      &exitbackupupd(0);
   }
   $outputfound=1 if (($line!~ /^\s*\%/) && ($line!~ /^\s*$/));
}

if ($line!~ /^\%START/) {
   if ($outputfound) {
      print STDERR "No valid data found in whois data: $tmpdata\n";
      &exitbackupupd(1);
   }
   else {
      &exitbackupupd(0);
   }
}

print STDERR "last:", $line if $opt_V;

if ($line) {
   
   if ($line=~ /^\%START\s+Version:\s*(\d+)\s+/) {
      print STDERR "found: ", $line if $opt_V;
      $version=$1;
   }
   
   if ($version==1) {
      ($line)=&UpdateVersionOne(TMPFILE,$line,$opt_s,$from,$to);
   }
   else {
      die "got unsupported update version format ($version), we support versions: ".join(" ",@UPDATEVERSIONS);   
      close(TMPFILE);
   }

}

# print STDERR "last line: $line";

# handy for debugging!

$offset=tell(TMPFILE);

if ($line=~ /^\s*$/) {
   $line=<TMPFILE>;
   print STDERR "read extra line: ", $line if $opt_V;
}

close(TMPFILE);

if ($line!~ /^\%END/) {
   print STDERR "Error in output from whois ($basename.$i)\nnear offset: $offset line: ", $line, "\n";
   &exitbackupupd(1);
}
else {
   &exitbackupupd(0);
}

# end of program