/*
 Copyright (c) 2003 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

 $Id: ipmapping.c,v 1.15 2003/04/22 10:21:56 can Exp $
*/

#include "ipmapping.h"

/*
 * Returns previously given ipv4 address.
 * Returns 0 if no address is given before.
 */
guint32 findv62v4(gchar * v6str)
{
    int retval;
    gchar *query, *qryResult;
    guint32 addr[4];
    guint32 result;

    retval = inet_pton(AF_INET6, v6str, &addr);
    if (retval == 0) {
        local_error("invalid ipv6 string: %s", v6str);
    }
    query =
        g_strdup_printf
        ("select v4_addr from %s where v6_addr=\"%s\"",
         (gchar *) getConf("MAPTABLE"), v6str);
    qryResult = g_new0(char, 50);
    db_execQuery1x1(query, qryResult);
    g_free(query);
    if (strcmp(qryResult, "") == 0) {
        g_free(qryResult);
        return 0;
    }
    result = atol(qryResult);
    g_free(qryResult);
    return (result);
}

/*
 * Delete rows
 */
void deleteipv4(MYSQL_ROW row, guint fldCount, gpointer user_data)
{
    gchar *qry, *v4str;
    gchar cFirst[100], cLast[100];
    time_t cF, cL;
    unsigned long v4int, resultn;
    MYSQL_RES *qryResult;
    struct tm cft, clt;

    cF = atol(row[2]);
    cL = atol(row[3]);
    localtime_r(&cF, &cft);
    localtime_r(&cL, &clt);
    strftime(cFirst, 80, "%Y-%m-%d %H:%M:%S", &cft);
    strftime(cLast, 80, "%Y-%m-%d %H:%M:%S", &clt);
    g_strchomp(cFirst);
    g_strchomp(cLast);
    v4int = atol(row[1]);
    resultn = htonl(v4int);
    v4str = g_new(gchar, INET_ADDRSTRLEN);
    inet_ntop(AF_INET, &resultn, v4str, INET_ADDRSTRLEN);
    qry =
        g_strdup_printf("delete from %s where v4_addr=%s",
                        (gchar *) getConf("MAPTABLE"), row[1]);
    qryResult = db_doQuery(qry);
    g_message
        ("unmapping [%s] %s first mapped %s last used %s usage count %s",
         row[0], v4str, cFirst, cLast, row[4]);
    g_free(v4str);
    mysql_free_result(qryResult);
    g_free(qry);
}

/*
 * Unmap given number of IPs
 */
void doUnmap(gint count)
{
    gchar *qry;

    qry =
        g_strdup_printf
        ("select v6_addr,v4_addr,first_mapped_time,last_used_time,usage_count from %s order by last_used_time limit %d",
         (gchar *) getConf("MAPTABLE"), count);
    db_foreach_query(qry, deleteipv4, NULL);
    g_free(qry);
}

/*
 * Finds a new free v4 address
 * Returns 0 if all addresses are full
 */
guint32 newv62v4(gchar * v6str)
{
    static guint32 rangestart = 0, rangeend = 0;
    static gchar *rangestartStr = NULL, *rangeendStr = NULL;
    gchar *query;
    gchar *qryResult, *insQuery, *v4str;
    guint32 result, resultn;
    static GStaticMutex myMutex = G_STATIC_MUTEX_INIT;

    g_static_mutex_lock(&myMutex);
    if (rangestart == 0) {
        rangestart = atol((char *) (gchar *) getConf("MAPRANGESTART"));
        rangeend = atol((char *) (gchar *) getConf("MAPRANGEEND"));
        rangestartStr = g_new(gchar, INET_ADDRSTRLEN);
        inet_ntop(AF_INET, &rangestart, rangestartStr, INET_ADDRSTRLEN);
        rangeendStr = g_new(gchar, INET_ADDRSTRLEN);
        inet_ntop(AF_INET, &rangeend, rangeendStr, INET_ADDRSTRLEN);
    }
    g_static_mutex_unlock(&myMutex);
    qryResult = g_new(char, 50);
    query =
        g_strdup_printf("select max(v4_addr)+1 from %s",
                        (gchar *) getConf("MAPTABLE"));
    db_execQuery1x1(query, qryResult);
    g_free(query);
    if (strcmp(qryResult, "") == 0) {
        result = rangestart;
    } else if (atol(qryResult) > rangeend) {
        result = 0;
    } else {
        result = atol(qryResult);
    }
    if (result > 0) {
        insQuery =
            g_strdup_printf
            ("insert into %s set first_mapped_time=%ld , usage_count=0 , v6_addr=\"%s\", v4_addr=%u",
             (gchar *) getConf("MAPTABLE"), time(NULL), v6str, result);
        db_execQuery1x1(insQuery, NULL);
        g_free(insQuery);
    } else {
        g_message("exhausted %s - %s", rangestartStr, rangeendStr);
        doUnmap(100);           /* 100 is arbitrary */
        return (newv62v4(v6str));
    }
    g_free(qryResult);
    v4str = g_new0(gchar, INET_ADDRSTRLEN);
    resultn = htonl(result);
    inet_ntop(AF_INET, &resultn, v4str, INET_ADDRSTRLEN);
    g_message("mapped [%s] %s", v6str, v4str);
    g_free(v4str);
    return (result);
}

gchar *v62v4(gchar * v6str)
{
    guint32 result, usageCount;
    gchar *query, *qryResult, *v6strPadded;
    static GStaticMutex cntMutex = G_STATIC_MUTEX_INIT;
    gchar *v4str;

    v6strPadded = sockInfo_zeropad(v6str);
    g_static_mutex_lock(&cntMutex);
    result = findv62v4(v6strPadded);
    if (result == 0) {
        result = newv62v4(v6strPadded);
    }
    if (result == 0) {
        local_error("Can't assign an IP for %s", v6strPadded);
    }
    query =
        g_strdup_printf
        ("select usage_count from %s where v6_addr=\"%s\"",
         (gchar *) getConf("MAPTABLE"), v6strPadded);
    qryResult = g_new(char, 50);
    db_execQuery1x1(query, qryResult);
    usageCount = atol(qryResult) + 1;
    g_free(query);
    query =
        g_strdup_printf
        ("update %s set last_used_time=%ld , usage_count=%u where v6_addr=\"%s\"",
         (gchar *) getConf("MAPTABLE"), time(NULL), usageCount,
         v6strPadded);
    db_execQuery1x1(query, qryResult);
    g_static_mutex_unlock(&cntMutex);
    g_free(query);
    g_free(qryResult);
    g_free(v6strPadded);
    v4str = g_new(char, INET_ADDRSTRLEN);
    result = htonl(result);
    inet_ntop(AF_INET, &result, v4str, INET_ADDRSTRLEN);
    return v4str;
}
