/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Structures used when building a parse tree of the input specification and
 * the operations on the structures.
 */

/* The complete specification */

struct spec {
    int		 sp_number;
    double	 sp_timelimit;
    struct bench *sp_bench;
    struct spec  *sp_next;
};
#define NULLSP (struct spec *)NULL

/* Routines */
struct spec *spec_create();
			/* (rpt:int; be: struct bench *; tl: double) */
struct spec *spec_append();	/* (list,single: struct spec *) */
struct spec *spec_free();	/* (sp: struct spec *) */
void specs_free();		/* (sp: struct spec *) */
void spec_print();			/* (sp: struct spec *) */
void specs_print();			/* (sp: struct spec *) */

#define spec_number(sp) 	((sp)->sp_number)
#define spec_bench(sp)		((sp)->sp_bench)
#define spec_timelimit(sp) 	((sp)->sp_timelimit)
#define spec_next(sp)		((sp)->sp_next)

#define TL_UNLIM		(double)-1
#define timelimit_unlim(tl)	((tl) == TL_UNLIM)
    
/*  */
    
/* A single benchmark */
struct bench {
    int		be_tag;
    union {
	struct {
	    struct symbol *beb_symbol;
	    struct datasp *beb_data;
	    struct timing *beb_timing;
	    int beb_background, beb_priority;
	    struct hostspec *beb_hosts;
	} beb;
	struct {
	    struct symbol *bec_symbol;
	    int		  bec_number;
	    struct part   *bec_parts;
	} bec;
    } be_u;
#define be_symbol 	be_u.beb.beb_symbol /* works for both! */
#define be_data 	be_u.beb.beb_data
#define be_timing 	be_u.beb.beb_timing
#define be_hosts	be_u.beb.beb_hosts
#define be_background	be_u.beb.beb_background
#define be_priority	be_u.beb.beb_priority
#define be_number 	be_u.bec.bec_number
#define be_parts 	be_u.bec.bec_parts
};
#define NULLBE (struct bench *)NULL

/* Tags */
#define BET_BASIC	1
#define BET_COMP	2

#define NP_UNSPEC	-1	/* Unspecified number of parts */

    
/* Routines */
struct bench *basic_create();
	/* (se: struct symbol *; dt: struct datasp *; ti: struct timing *;
	   hs: struct hostspec *) */
struct bench *comp_create();
	/* (se: struct symbol *; pt: struct part *; default: struct hostspec *) */
void bench_free();
	/* (be: struct bench *) */
void bench_print();
	/* (be: struct bench *) */

#define bench_isbasic(be) ((be)->be_tag == BET_BASIC)
#define bench_iscomp(be) ((be)->be_tag == BET_COMP)
#define bench_symbol(be)	((be)->be_symbol)
#define bench_name(be)		symbol_name(bench_symbol(be))
#define bench_proc(be)		symbol_client_proc(bench_symbol(be))
#define bench_client_proc(be)	symbol_client_proc(bench_symbol(be))
#define bench_server_proc(be)	symbol_server_proc(bench_symbol(be))
#define bench_numprocesses(be) \
	    ((bench_isbasic(be)) ? 1 : part_numprocesses(comp_parts(be)))

#define basic_data(be)		((be)->be_data)
#define basic_timing(be)	((be)->be_timing)
#define basic_hosts(be)		((be)->be_hosts)
#define basic_client(be)	(hostspec_client(basic_hosts(be)))
#define basic_server(be)	(hostspec_server(basic_hosts(be)))
#define basic_background(be)	((be)->be_background)
#define basic_priority(be)	((be)->be_priority)

#define basic_setbackground(be)	((be)->be_background = 1)
#define basic_setpriority(be,p)	((be)->be_priority = (p))

#define comp_number(be)		((be)->be_number)
#define comp_parts(be)		((be)->be_parts)

/*  */

/* Host specification - isdefault fields means that the default hosts have been
   used so that a composite benchmark specification can override the default
   hosts used in its parts. */

struct hostspec {
    char *server, *client;
    int server_isdefault, client_isdefault;
};
#define NULLHS (struct hostspec *)NULL

/* Routines */
struct hostspec *hostspec_create(); 	/* (client, server: char *) */
void hostspec_free();			/* (hs: struct hostspec *) */
void hostspec_print();			/* (hs: struct hostspec *) */
void hostspec_setdefault();		/* (hs, default: struct hostspec *) */

#define hostspec_client(hs)	((hs)->client)
#define hostspec_server(hs)	((hs)->server)
    
/*  */

/* Parts of composite benchmarks */

struct part {
    int 		pt_number;
    struct bench 	*pt_bench;
    struct part		*pt_next;
};
#define NULLPT (struct part *)NULL

#define NR_DEFAULT	1		/* default number of replicas */
    
/* Routines */
struct part *part_create(); 	/* (num: int; be: struct bench *) */
struct part *part_append();	/* (list,single: struct part *) */
struct part *part_free();	/* (pt: struct part *) */
void parts_free();		/* (pt: struct part *) */
void part_print();			/* (pt: struct part *) */
void parts_print();			/* (pt: struct part *) */

#define part_number(pt)	((pt)->pt_number)
#define part_bench(pt)	((pt)->pt_bench)
#define part_next(pt)	((pt)->pt_next)

/*  */

/* Data specification */
struct datasp {
    int 		dt_type;
    struct sizespec 	*dt_size;
    struct access 	*dt_src, *dt_dst;
};
#define NULLDT (struct datasp *)NULL

/* Routines */
struct datasp *datasp_create();
	/* (type: int; ss: struct sizespec *; src,dst: struct access *) */
void datasp_free();
	/* (dt: struct datasp *) */
void datasp_print();
	/* (dt: struct datasp *) */

#define datasp_type(dt)	((dt)->dt_type)
#define datasp_size(dt)	((dt)->dt_size)
#define datasp_src(dt)	((dt)->dt_src)
#define datasp_dst(dt)	((dt)->dt_dst)

#define datasp_maxsendsize(dt)	sizespec_maxsendsize(datasp_size(dt))
#define datasp_maxrecvsize(dt)	sizespec_maxrecvsize(datasp_size(dt))

#define datasp_iterations(dt) 	sizespec_iterations(datasp_size(dt))

/*  */

/* Access */
struct access {
    int 	ac_type;
    char	*ac_filename;
};
#define NULLAC (struct access *)NULL
    
/* Types */
#define 	ACT_FILE	1
#define 	ACT_MEM		2
#define 	ACT_VM		3

/* Routines */
struct access *access_create();
		/* (type: int; fname: char *) */
void access_free();
		/* (ac: struct access *) */
void access_print();
		/* (ac: struct access *) */

#define access_type(ac)		((ac)->ac_type)
#define access_filename(ac)	((ac)->ac_filename)

/*  */

/* Size specification */
/* Extend with distribution on the message size! */
struct sizespec {
    struct msize *ss_sendmsg, *ss_recvmsg;
    int ss_iterations;
};
#define NULLSS (struct sizespec *)NULL

/* signals that iterations is depending on the (source) file size */
#define SS_DEFAULT	-1	
    
/* Routines */
struct sizespec *sizespec_create();
		/* (sendmsg, recvmsg: struct msize *; iterations: int) */
void sizespec_free();
		/* (ss: struct sizespec *) */
void sizespec_print();
		/* (ss: struct sizespec *) */

#define sizespec_sendmsg(ss)	((ss)->ss_sendmsg)
#define sizespec_recvmsg(ss)	((ss)->ss_recvmsg)
#define sizespec_iterations(ss)	((ss)->ss_iterations)

#define sizespec_maxsendsize(ss)	msize_maxval(sizespec_sendmsg(ss))
#define sizespec_maxrecvsize(ss)	msize_maxval(sizespec_recvmsg(ss))

/*  */

/* Arrival specification */
struct timing {
    struct dist *ti_dist;
    struct timing *ti_next;
    int ti_nvalue;
    double *ti_values;
};
#define NULLTI (struct timing *)NULL

/* Routines */
struct timing *timing_create();
		/* (dist: struct dist *; nvalue: int; values: double[]) */
void timing_generate();
		/* (ti: struct timing *) */
struct timing *timing_append();
		/* (ti: struct timing *) */
struct timing *timing_free();
		/* (ti: struct timing *) */
void timings_free();
		/* (ti: struct timing *) */
void timing_print();
		/* (ti: struct timing *) */
void timings_print();
		/* (ti: struct timing *) */

#define timing_dist(ti)	((ti)->ti_dist)
#define timing_next(ti)	((ti)->ti_next)

#define timing_nvalue(ti)	((ti)->ti_nvalue)
#define timing_values(ti)	((ti)->ti_values)

#define timing_param(ti)	dist_param(timing_dist(ti))
#define timing_symbol(ti)	dist_symbol(timing_dist(ti))
#define timing_proc(ti)		dist_proc(timing_dist(ti))
#define timing_name(ti)		dist_name(timing_dist(ti))

#define timing_value(ti, n) 	(timing_values(ti)[n%timing_nvalue(ti)])
				
/*  */

/* Message sizes */

struct msize {
    struct dist *ms_dist;
    int ms_nvalue;
    int *ms_values;
    int ms_maxval;
};
#define NULLMS (struct msize *)NULL

#define MS_CONSTANT	1
#define MS_DIST		2
    
/* Routines */
struct msize *msize_create();
		/* (tag: int; constant: int; di: struct dist *;
		    nvalue: int; values: int[]) */
void msize_generate();
		/* (ms: struct msize *) */
void msize_free();
		/* (ms: struct msize *) */
void msize_print();
		/* (ms: struct msize *) */

#define msize_dist(ms)		((ms)->ms_dist)
#define msize_nvalue(ms)	((ms)->ms_nvalue)
#define msize_values(ms)	((ms)->ms_values)
#define msize_maxval(ms)	((ms)->ms_maxval)


#define msize_value(ms, n) 	(msize_values(ms)[n%msize_nvalue(ms)])
				
#define msize_param(ms)		dist_param(msize_dist(ms))
#define msize_symbol(ms)	dist_symbol(msize_dist(ms))
#define msize_proc(ms)		dist_proc(msize_dist(ms))
#define msize_name(ms)		dist_name(msize_dist(ms))

/*  */

/* Distributions */

struct dist {
    struct symbol *di_symbol;
    struct param *di_param;
};
#define NULLDI (struct dist *)NULL

/* Routines */
struct dist *dist_create();
		/* (se: struct symbol *; pm: struct param *) */
void dist_free();
		/* (di: struct dist *) */
void dist_print();
		/* (di: struct dist *) */

#define dist_symbol(di)	((di)->di_symbol)
#define dist_param(di)	((di)->di_param)

#define dist_name(di)	symbol_name(dist_symbol(di))    
#define dist_proc(di)	((double (*)())symbol_client_proc(dist_symbol(di)))
				
/*  */

/* Parameters */
struct param {
    double pm_value;
    struct param *pm_next;
};
#define NULLPM (struct param *)NULL


/* Routines */
struct param *param_create(); 	/* (val: double) */
struct param *param_append();	/* (list,single: struct param *) */
struct param *param_free();	/* (pm: struct param *) */
void params_free();		/* (pm: struct param *) */
void param_print();		/* (pm: struct param *) */
void params_print();		/* (pm: struct param *) */

#define param_value(pm)	((pm)->pm_value)
#define param_next(pm)	((pm)->pm_next)

int param_nvalues();			/* (pm: struct param *) */
double param_getvalue();		/* (pm: struct param *; index: int) */

