/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * SunLink TP4 protocol specific code
 */

#include <general.h>
#include <protospec.h>

/*  */
OSI_ADDR *osi_address();

struct address_t client_address;

/*
 * InitClient - compute a client tp address, store in global variable
 * client_address. Hence, only one client entity at a time. InitClient should
 * really return something, probably a struct address_t.
 */

tp4_init_client() {
    if (NULL == osi_address(&client_address.osi_addr)) {
	eprintf(EF_IN3, INTERNAL, "osi_address", "InitClient");
    }
}    

/*  */

tp4_conn_request(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    register int s;
    tp_msg tpmsg;
    register TP_MSG_CONNECT *tp = &tpmsg.tp_connect;
    int resp_len;
    
    pprintf("tp4_conn_request(0x%x, 0x%x, 0x%x)\n",	addr, a_ch, errind);
    
    if ((s = socket(AF_OSI, SOCK_EVENT, OSIPROTO_TP_EVENT)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "socket call", "tp4_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    if (bind(s, (struct sockaddr *) &client_address.osi_addr, sizeof(client_address.osi_addr)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "bind call", "tp4_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    
    tp->tp_event = TP_CONNECT_REQ;
    tp->expedited_selected = FALSE;
    tp->src_address = client_address.osi_addr; /* struct copy */
    tp->dst_address = addr->osi_addr; /* struct copy */
    if (sendto(s, 0, 0, 0, (struct sockaddr *) tp, sizeof(TP_MSG_CONNECT)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "sendto CONN_REQ", "tp4_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    resp_len = sizeof(tp_msg);
    if (recvfrom(s, (char *) 0, 0, 0, (struct sockaddr *) tp, &resp_len) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "recvfrom CONN_RESP", "tp4_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    if (tp->tp_event != TP_CONNECT_CONF) {
	char buf[128];
	sprintf(buf, "Expected TP_CONNECT_CONF, received tp_event %d", tp->tp_event);
	eprintf(EF_IN3, COMMUNICATION, buf, "tp4_conn_request", getsyserr());
	if (tp->tp_event == TP_DISCONNECT_IND) {
	    sprintf(buf, "Disconnected, reason code %d\n",
		    ((TP_MSG_DISCONNECT *)tp)->reason);
	    eprintf(EF_IN3, COMMUNICATION, buf, "tp4_conn_request", getsyserr());
	}
	*errind = NOTOK;
	return;
    }
	
    a_ch->chan = s;
    a_ch->seen_disc_ind = FALSE;
    *errind = OK;
    pprintf("tp4_connreq -> channel %d\n", a_ch->chan);
} /* tp4_conn_request */

/*  */


tp4_create_channel(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    
    pprintf("tp4_create_channel(0x%x, 0x%x, 0x%x)\n",	addr, a_ch, errind);
    
    if ((s = socket (AF_OSI, SOCK_EVENT, OSIPROTO_TP_EVENT)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "socket call", "tp4_create_channel",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    a_ch->chan = s;
    a_ch->seen_disc_ind = FALSE;
    *errind = OK;
} /* tp4_create_channel */

/*  */

tp4_await_conn_ind(server, a_ch, errind)
    struct server_t *server;	
    channel_t *a_ch;
    error_t *errind;
{
    tp_msg tpmsg;
    register TP_MSG_CONNECT *tp = &tpmsg.tp_connect;
    int tplen = sizeof(TP_MSG_CONNECT);
    OSI_ADDR otheraddr;
    int len = sizeof(otheraddr), so;

    
    pprintf("tp4_await_conn_ind(server %d)\n", *server);

    if ((so = accept(server->sock, (struct sockaddr *) &otheraddr, &len)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "accept call", "tp4_await_conn_ind",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    pprintf("tp4_await_conn_ind - got connection %d\n", so);
    if (NOTOK == recvfrom(so, (char *) 0, 0, 0, (struct sockaddr *) tp, &tplen)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "recvmsg CONN_IND", "tp4_await_conn_ind",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    if (tp->tp_event != TP_CONNECT_IND) {
	char buf[128];
	sprintf(buf, "expected TP_CONNECT_IND, got tp_event %d", tp->tp_event);
	eprintf(EF_IN3, COMMUNICATION, buf, "tp4_await_conn_ind");
	*errind = NOTOK;
	close(so);
	return;
    }
    tp->tp_event = TP_CONNECT_RESP;
    tp->expedited_selected = FALSE;
    if (sendto(so, (char *) 0, 0, 0, (struct sockaddr *) tp, sizeof(TP_MSG_CONNECT)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "sendto CONN_RESP", "tp4_await_conn_ind",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    a_ch->chan = so;
    a_ch->seen_disc_ind = FALSE;
    *errind = OK;
} /* tp4_await_conn_ind */

/*  */

/*
 * disc_request - is not guaranteed to work with the way SPIMS currently works
 * Problem: close() means abort in TP4, not terminate connection as in TCP.
 * TP4 sends DISC_REQ out of band, so an immediate close() after DISC_REQ
 * potentially causes loss of data
 */

tp4_disc_request(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    tp_msg tpmsg;
    register TP_MSG_DISCONNECT *tp = &tpmsg.tp_disconnect;

    pprintf("tp4_disc_request(%d, 0x%x)\n", ch.chan, errind);

    tp->tp_event = TP_DISCONNECT_REQ;
    tp->reason = DR_NORM_BY_SESSION;
    if (sendto(ch.chan, 0, 0, 0, (struct sockaddr *) tp, sizeof(TP_MSG_DISCONNECT)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "sendto DISC_REQ", "tp4_disc_request",
	       getsyserr());
  	*errind = NOTOK;
	return;
    }
    if (close(ch.chan)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close call", "tp4_disc_request",
	       getsyserr());
  	*errind = NOTOK;
	return;
    }
    *errind = OK;
} /* tp4_disc_request */

/*  */

tp4_await_disc_ind(ch, errind)
    channel_t ch;
    error_t *errind;
{
    tp_msg tpmsg;
    register TP_MSG_DISCONNECT *tp = &tpmsg.tp_disconnect;
    int tp_len = sizeof(TP_MSG_DISCONNECT);
    
    pprintf("tp4_await_disc_ind(ch %d, 0x%x)\n", ch.chan, errind);

    if (recvfrom(ch.chan, (char *) 0,0, 0, (struct sockaddr *) tp, &tp_len) == NOTOK) {
	if (errno == ENOTCONN && ch.seen_disc_ind) {
	    (void) close(ch.chan);
	    *errind = OK;
	    return;
	}
	else {
	    int tmp = errno;
	    eprintf(EF_SYSCALL, COMMUNICATION, "recvfrom expecting DISC_IND",
		    "tp4_await_disc_ind", getsyserr());
	    (void) close(ch.chan);
	    errno = tmp;
	    *errind = NOTOK;
	    return;
	}
    }
    if (tp->tp_event != TP_DISCONNECT_IND) {
	char buf[128];
	sprintf(buf, "expected TP_DISCONNECT_IND, got event %d", tp->tp_event);
	eprintf(EF_IN3, COMMUNICATION, buf, "tp4_await_disc_ind");
	(void) close(ch.chan);
	*errind = NOTOK;
	return;
    }
    if (tp->reason != DR_NORM_BY_SESSION) {
	char buf[128];
	sprintf(buf, "unexpected reason for disconnect, code %d", tp->reason);
	eprintf(EF_IN3, COMMUNICATION, buf, "tp4_await_disc_ind");
	(void) close(ch.chan);
	*errind = NOTOK;
	return;
    }
    (void) close(ch.chan);
    *errind = OK;
} /* tp4_await_disc_ind */

/*  */

tp4_create_server(aa_server, aa_addr, errind)
struct server_t **aa_server;
struct address_t **aa_addr;
error_t   *errind;
{

    int s;
    OSI_ADDR myaddr;    
    int len = sizeof(myaddr);
    char buf[128];
    
    pprintf("tp4_create_server(0x%x, 0x%x, 0x%x)\n",
		aa_server, aa_addr, errind);

    if ((s = socket (AF_OSI, SOCK_EVENT, OSIPROTO_TP_EVENT)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call", "tp4_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    if (NULL == osi_address(&myaddr)) {
	eprintf(EF_IN3, COMMUNICATION, "create address", "tp4_create_server");
	*errind = NOTOK;
	return;
    }

    if (bind (s, (struct sockaddr *) &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "tp4_create_server",
	       getsyserr());
	return NOTOK;
    }
    
    if (listen(s, 5) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "listen call", "tp4_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    *aa_server = (struct server_t *)malloc(sizeof(struct server_t));
    if (*aa_server == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "tp4_create_server");
	*errind = NOTOK;
	return;
    }
    *aa_addr = (struct address_t *)malloc(sizeof(struct address_t));
    if (*aa_addr == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "tp4_create_server");
	*errind = NOTOK;
	free((char *)*aa_server);
	return;
    }
    (*aa_server)->sock = s;
    (*aa_addr)->osi_addr = myaddr;
    *errind = OK;
} /* tp4_create_server */

/*  */

tp4_destroy_server(a_server, errind)
    struct server_t *a_server;
    error_t   *errind;
{
    
    pprintf("tp4_destroy_server(0x%x, 0x%x)\n", a_server->sock, errind);

    if (close(a_server->sock) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close", "tp4_destroy_server",
	       getsyserr());
	free((char *)a_server);
	*errind = NOTOK;
	return;
    }

    free((char *)a_server);
	
    *errind = OK;
} /* tp4_destroy_server */

/*  */

tp4_data_request(ch, buf, amount, errind)
channel_t ch;
char *buf;
int amount;
error_t *errind;
{
    tp_msg tpmsg;
    register TP_MSG_DATA *tp = &tpmsg.tp_data;
    int n;
    
    pprintf("DataRequest(%d, 0x%x, %d, 0x%X\n", ch.chan, buf, amount, errind);
    tp->tp_event = TP_DATA_REQ;
    tp->eot = 0;

    if ((n = sendto(ch.chan, buf, amount, 0, (struct sockaddr *) tp, sizeof(TP_DATA_REQ))) != amount) {
	if (n == NOTOK) {
	    eprintf(EF_SYSCALL, COMMUNICATION, "sendto TP_DATA_REQ",
		    "tp4_data_request", getsyserr());
	    *errind = NOTOK;
	}
	else {
	    char buf[128];
	    sprintf(buf, "all data not sent - %d vs %d", n, amount);
	    eprintf(EF_IN3, COMMUNICATION, buf, "tp4_data_request");
	    *errind = NOTOK;
	}
    }
    else
	*errind = OK;
}

/*  */

tp4_await_data_ind(ch, buf, amount, errind)
channel_t ch;
char *buf;
int amount;
error_t *errind;

{
    register int left = amount;
    register int got;
    tp_msg tpmsg;
    register TP_MSG_DISCONNECT *tp = &tpmsg.tp_disconnect;
    int tp_len = sizeof(TP_MSG);
      
    pprintf("AwaitDataIndication(%d, 0x%x, %d, 0x%X\n",
	    ch.chan, buf, amount, errind);
    *errind = OK;
    do {
	if ((got = recvfrom(ch.chan, buf, left, 0, (struct sockaddr *) tp, &tp_len)) == NOTOK) {
	    eprintf(EF_SYSCALL, COMMUNICATION, "recvfrom", "tp4_await_data_ind", getsyserr());
	    *errind = NOTOK;
	    break;
	}
	if (tp->tp_event == TP_DATA_IND) {
	    left -= got;
	}
	else if (tp->tp_event == TP_DISCONNECT_IND) {
	    ch.seen_disc_ind = TRUE;
	}
	else {
	    char buf[128];
	    sprintf(buf, "Expected DATA_IND, got event code %d", tp->tp_event);
	    eprintf(EF_IN3, COMMUNICATION, buf, "tp4_await_data_ind");
	    *errind = NOTOK;
	    break;
	}
    } while (left > 0);
    return;
} /* tp4_await_data_ind */

tp4_report_error(err, str)
    error_t *err;
    char *str;
{
    
    pprintf("tp4_report_error(0x%x)\n", err);
    
    if (err == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "Error", str,
	       getsyserr());
    } else {
	eprintf(EF_SYSCALL, COMMUNICATION, "TP4 protocol", str,
	       getsyserr());
    }
} /* tp4_report_error */
