/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * SUN RPC level specific code
 *
 * SUNRPC_UDP if on top of udp
 * SUNRPC_TCP if on top of tcp
 *	(These should be defined in the Makefile)
 */

#include <general.h>
#include <protospec.h>

/* Maximum time allowed in one RPC call - 20 seconds */
struct timeval RPC_total_time = { 20L, 0L };

#define UDP_RETRANSMIT_TIME  3L
#define RPC_UDP_BUFSIZE EightK
#define RPC_TCP_BUFSIZE (EightK * 8)

unsigned long gettransient();


#ifdef SUNRPC_UDP

static struct timeval retransmit_time = { UDP_RETRANSMIT_TIME, 0L };

#define clnt_create(addrp, prognum, sockp) \
	clntudp_create(addrp, prognum, 1, retransmit_time, sockp)

#define TP_PROTO		IPPROTO_UDP

#define svc_create(sock)	svcudp_create(sock)

#endif SUNRPC_UDP

/*  */

#ifdef SUNRPC_TCP	

#define clnt_create(addrp, prognum, sockp) \
	clnttcp_create(addrp, prognum, 1, sockp, \
		       RPC_TCP_BUFSIZE, RPC_TCP_BUFSIZE)

#define TP_PROTO		IPPROTO_TCP

#define svc_create(sock)	\
    		svctcp_create(sock, RPC_TCP_BUFSIZE, RPC_TCP_BUFSIZE)

#endif SUNRPC_TCP

/*  */

sunrpc_conn_request(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
  CLIENT *client;
  int sock = RPC_ANYSOCK;
  enum clnt_stat status;

  pprintf("sunrpc_conn_request(0x%x, 0x%x, 0x%x)\n",  addr, a_ch, errind);

  *errind = OK;

  client = clnt_create(&addr->addr, addr->prognum, &sock);

  if (client == NULL) {
    perror("clnttcp_create/clntudp_create");
    clnt_pcreateerror("sunrpc_conn_request (client creation)");
    *errind = NOTOK;
    return;
  }

/*
  *a_ch = (channel_t) malloc(sizeof(channel_structure));
  if (*a_ch == NULL) {
    perror("malloc failed");
    *errind = NOTOK;
    return;
  }
*/

  a_ch->socket = sock;
  a_ch->clientp = client;
  a_ch->xprtp = NULL;		/* Used to test what side we are on */

  if ((int) clnt_call(client, ConnIndProc, xdr_void, NULL,
		      xdr_void, NULL, RPC_total_time) != 0) {
    clnt_perror(client, "sunrpc_conn_request");
    *errind = NOTOK;
  }

  pprintf("<sunrpc_conn_request\n");
} /* sunrpc_conn_request */

/*  */

sunrpc_disc_request(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    pprintf("sunrpc_disc_request(%#x, %#x)\n", ch, errind);

    clnt_destroy(ch.clientp);

    if (close(ch.socket) == NOTOK) {
      eprintf(EF_SYSCALL, COMMUNICATION, "close call", "sunrpc_disc_request",
	      getsyserr());
    }

/*
    free(ch);
*/

    *errind = OK;

    pprintf("<sunrpc_disc_request\n");
} /* sunrpc_disc_request */






int xdr_byte_arr (xdrsp, byte_arr_p)
     XDR *xdrsp;
     byte_arr_type *byte_arr_p;
{
  register int temp;

  pprintf("sunrpc_xdr_byte_arr(%#x, %#x, %d) => ",
	  xdrsp, byte_arr_p->chars, byte_arr_p->length);
  temp = xdr_bytes(xdrsp, &byte_arr_p->chars, &byte_arr_p->length,
		   MAX(RPC_TCP_BUFSIZE, RPC_UDP_BUFSIZE));
  pprintf("%d\n", temp);
  return temp;
}

/*  */

int conn_ind_flag = 0;
int call_ind_flag = 0;
int start_bulkget_flag = 0;
int stop_bulkget_flag = 0;
int start_bulkput_flag = 0;
int stop_bulkput_flag = 0;
int receive_data_flag = 0;
int send_data_flag = 0;

byte_arr_type call_ind_buf;
SVCXPRT *call_ind_xprt;
byte_arr_type receive_data_buf;
byte_arr_type send_data_buf;




server_handler(request, transp)
     struct svc_req *request;
     SVCXPRT *transp;
{
  pprintf("server_handler(%d, %#x)\n", request->rq_proc, transp);

  switch (request->rq_proc) {
  case NullProc:
    Send_Reply(transp, xdr_void, 0);
    break;
  case RPCCallProc:
    if (call_ind_flag) {
      call_ind_flag = 0;
      /* This xprt is not the same as the one */
      /* registered by svc_register in sunrpc_create_server() */
      call_ind_xprt = transp;
      pprintf("sunrpc_RPCCallProc called\n");
      Get_Args(transp, xdr_byte_arr, &call_ind_buf);
    } else
      pprintf("sunrpc_ call_ind_flag NOT SET YET!!! %#x\n", &call_ind_flag);
    break;
  case BulkGetStartProc:
    if (start_bulkget_flag) {
      start_bulkget_flag = 0;
      pprintf("sunrpc_BulkGetStartProc called\n");
      Get_Args(transp, xdr_void, 0);
      Send_Reply(transp, xdr_void, 0);
    } else
      pprintf("sunrpc_ start_bulkget_flag NOT SET YET!!! %#x\n",
	      &start_bulkget_flag);
    break;
  case BulkGetStopProc:
    if (stop_bulkget_flag) {
      stop_bulkget_flag = 0;
      pprintf("sunrpc_BulkGetStopProc called\n");
      Get_Args(transp, xdr_void, 0);
      Send_Reply(transp, xdr_void, 0);
    } else
      pprintf("sunrpc_ stop_bulkget_flag NOT SET YET!!! %#x\n",
	      &stop_bulkget_flag);
    break;
  case ConnIndProc:
    if (conn_ind_flag) {
      conn_ind_flag = 0;
      pprintf("sunrpc_ConnIndProc called\n");
      Get_Args(transp, xdr_void, 0);
      Send_Reply(transp, xdr_void, 0);
    } else
      pprintf("sunrpc_ conn_ind_flag NOT SET YET!!! %#x\n", &conn_ind_flag);
    break;
  case SendDataProc:
    if (send_data_flag) {
      send_data_flag = 0;
      pprintf("sunrpc_SendDataProc called\n");
      Get_Args(transp, xdr_void, 0);
      Send_Reply(transp, xdr_byte_arr, &send_data_buf);
    } else
      pprintf("sunrpc_ send_data_flag NOT SET YET!!! %#x\n", &send_data_flag);
    break;
  case ReceiveDataProc:
    if (receive_data_flag) {
      receive_data_flag = 0;
      pprintf("sunrpc_ReceiveDataProc called\n");
      Get_Args(transp, xdr_byte_arr, &receive_data_buf);
      Send_Reply(transp, xdr_void, 0);
    } else
      pprintf("sunrpc_ receive_data_flag NOT SET YET!!! %#x\n",
	      &receive_data_flag);
    break;
  case BulkPutStartProc:
    if (start_bulkput_flag) {
      start_bulkput_flag = 0;
      pprintf("sunrpc_BulkPutStartProc called\n");
      Get_Args(transp, xdr_void, 0);
      Send_Reply(transp, xdr_void, 0);
    } else
      pprintf("sunrpc_ start_bulkput_flag NOT SET YET!!! %#x\n",
	      &start_bulkput_flag);
    break;
  case BulkPutStopProc:
    if (stop_bulkput_flag) {
      stop_bulkput_flag = 0;
      pprintf("sunrpc_BulkPutStopProc called\n");
      Get_Args(transp, xdr_void, 0);
      Send_Reply(transp, xdr_void, 0);
    } else
      pprintf("sunrpc_ stop_bulkput_flag NOT SET YET!!! %#x\n",
	      &stop_bulkput_flag);
    break;
  default:
    pprintf("sunrpc_ILLEGAL_PROCEDURE_NUMBER called\n");
    svcerr_noproc(transp);
    exit(2);
  }
  pprintf("<server_handler\n");
} /* server_handler */

/*  */



unsigned long sunrpc_program_number;




unsigned long gettransient (proto, vers, sockp)
     unsigned long proto, vers;
     int *sockp;
{
  static unsigned long prognum = 0x3FFFFFFF;
  int s, length, socktype;
  struct sockaddr_in addr;

  pprintf("gettransient(%s)\n", proto == IPPROTO_TCP ? "tcp" : "udp");

  switch(proto) {
  case IPPROTO_UDP:
    socktype = SOCK_DGRAM;
    break;
  case IPPROTO_TCP:
    socktype = SOCK_STREAM;
    break;
  default:
    eprintf(EF_IN3, COMMUNICATION, "unknown protocol type",
	    "gettransient routine");
    return 0L;
  }

  if (*sockp == RPC_ANYSOCK) {
    if ((s = socket(AF_INET, socktype, 0)) < 0) {
      perror("socket");
      return 0L;
    }
    *sockp = s;
  } else
    s = *sockp;

  addr.sin_addr.s_addr = 0L;
  addr.sin_family = AF_INET;
  addr.sin_port = 0;
  bzero(addr.sin_zero, sizeof(addr.sin_zero));

  length = sizeof(struct sockaddr_in);

  bind(s, &addr, length);
  if (getsockname(s, &addr, &length) < 0) {
    perror("getsockname");
    return 0L;
  }
  while (!pmap_set(++prognum, vers, proto, addr.sin_port))
    continue;

  pprintf("<gettransient: %#x\n", prognum);

  return sunrpc_program_number = prognum;
} /* gettransient */

/*  */


sunrpc_create_server(aa_server, aa_addr, errind)
    struct server_t **aa_server;
    struct address_t **aa_addr;
    error_t   *errind;
{

    int sock = RPC_ANYSOCK;
    unsigned long prognum;
    static struct sockaddr_in myaddr;
    SVCXPRT *xprt;

    pprintf("sunrpc_create_server(0x%x, 0x%x, 0x%x)\n",
		aa_server, aa_addr, errind);

    if (get_myipaddr(&myaddr.sin_addr) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "init_myipaddr", "CreateServer");
	*errind = NOTOK;
	return;
    }
    myaddr.sin_family = AF_INET;
    myaddr.sin_port = 0;

    prognum = gettransient(TP_PROTO, 1L, &sock, &myaddr);
    if (prognum == 0L) { *errind = NOTOK; return; }
    if ((xprt = svc_create(sock)) == NULL)
      {
	printf("server creation failed.\n");
	svcerr_systemerr(NULL);
	*errind = NOTOK;
	return;
      }

    /* Protocol must be zero, gettransient above has done the registering */
    (void) svc_register(xprt, prognum, 1L, server_handler, 0L);

    pprintf("Created socket %d with address: %#x:%d\n",
	    sock, myaddr.sin_addr.s_addr, myaddr.sin_port);

    *aa_server = (struct server_t *)malloc(sizeof(struct server_t));
    if (*aa_server == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	return;
    }
    *aa_addr = (struct address_t *)malloc(sizeof(struct address_t));
    if (*aa_addr == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	free((char *)*aa_server);
	return;
    }

    (*aa_server)->socket = sock;
    (*aa_server)->xprtp = xprt;
    (*aa_server)->clientp = NULL;

    (*aa_addr)->addr = myaddr;
    (*aa_addr)->prognum = prognum;

    *errind = OK;

    pprintf("<sunrpc_create_server: @ %d, %#x\n",
	    (*aa_server)->socket, (*aa_server)->xprtp);
} /* sunrpc_create_server */

/*  */



sunrpc_destroy_server(a_server, errind)
    struct server_t *a_server;
    error_t   *errind;
{
    pprintf("sunrpc_destroy_server(??, %#x)\n", errind);

    if (sunrpc_program_number != 0) { /* Try to reuse program numbers */
      pmap_unset(sunrpc_program_number, 1L);
      sunrpc_program_number = 0L;
    }

    svc_destroy(a_server->xprtp);

    if (close(a_server->socket) == NOTOK) {
      eprintf(EF_SYSCALL, COMMUNICATION, "close", "sunrpc_destroy_server",
	      getsyserr());
      *errind = NOTOK;
      free((char *)a_server);
      return;
    }

    free((char *)a_server);

    *errind = OK;
} /* sunrpc_destroy_server */






sunrpc_report_error(err, str)
    error_t *err;
    char *str;
{
    pprintf("sunrpc_report_error(%#x,%d)\n", err, *err);
    
    if (sunrpc_program_number != 0) { /* Try to reuse program numbers */
      pmap_unset(sunrpc_program_number, 1L);
      sunrpc_program_number = 0L;
    }

    if (err == NULL) {
      eprintf(EF_SYSCALL, COMMUNICATION, "Error", str,
	      getsyserr());
    } else {
	eprintf(EF_SYSCALL, COMMUNICATION, "SUNRPC protocol", str,
	       getsyserr());
    }
} /* sunrpc_report_error */
