/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Header file for benchmarks for the VMTP protocol
 */

#ifndef _VMTP_
#define _VMTP_
#define oldvmtp

#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <vmtp/vmtp_sys.h>
#include <vmtp/vmtp_so.h>
#include <vmtp/vmtp_manager.h>
#include <vmtp/vmtplib.h>

#ifndef VMTP_MAXSEGSIZE
#define VMTP_MAXSEGSIZE 16384
#endif

/* A channel is a vmtp handle plus the entity identifier of the "other end" */
struct sock_addr { 		
  struct vmtphdl 	*handle;
  struct vmtpeid 	addr;
};

#define server_t 	sock_addr
typedef struct sock_addr channel_t;

#include <addrspec.h>

/*
 * Error handling
 */

typedef struct {
    int er_ret;
    int er_code;
} error_t;

#define failed(errind)	((errind)->er_ret == NOTOK || \
			 !vmtp_ok((errind)->er_code))	/* sufficient?? */

#define report_error(errind,str)	vmtp_report_error(errind,str)


#define InitClient()		vmtp_create_client()
						/* save entity id in static? */

#define ConnRequest(addr,a_ch,errind) \
    				vmtp_conn_request(addr, a_ch, errind)
					/* retieve entity id from static? */
					/* create a channel */

#define DiscRequest(ch, errind) \
				vmtp_disc_request(ch, errind)
					/* close VMTP socket */

#define AwaitConnInd(server, a_ch, errind) \
{ \
    (a_ch)->handle = (server)->handle; \
    (errind)->er_ret = OK; (errind)->er_code = VMTP_OK;  \
}

#define AwaitDiscInd(ch, errind) \
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }


#define CreateServer(aa_server, aa_addr, errind) \
    				vmtp_create_server(aa_server, aa_addr, errind)
#define DestroyServer(a_server, errind) \
    				vmtp_destroy_server(a_server, errind)

/*
 * Simulated bulk tranfer capabilities - ONLY FOR PUT - different
 * for client and server
 */
#define BulkPutDataReq(channel,buffer,amount, errind) \
{  \
    int left = amount; \
    int thistime; \
    char *buf = buffer; \
 \
    while (left >0) { \
	thistime = MIN(left, VMTP_MAXSEGSIZE); \
	RPCCall(channel, buf, thistime, NULL, 0, errind); \
	if (failed(errind)) \
	    break; \
	left -= thistime; \
	buf += thistime; \
    } \
}

#define BulkPutAwaitDataInd(channel, buffer, amount, errind)	\
{  \
    int left = amount; \
    int thistime; \
    char *buf = buffer; \
			 \
    while (left >0) { \
	thistime = MIN(left, VMTP_MAXSEGSIZE); \
	RPCAwaitCallInd(channel, buf, thistime, errind); \
	if (failed(errind)) \
	    break; \
	RPCReturn(channel, NULL, 0, errind); \
	if (failed(errind)) \
	    break; \
	left -= thistime; \
	buf += thistime; \
    } \
}

#define BulkGetAwaitDataInd(channel,buffer,amount, errind) \
{  \
    int left = amount; \
    int thistime; \
    char *buf = buffer; \
 \
    while (left >0) { \
	thistime = MIN(left, VMTP_MAXSEGSIZE); \
	RPCCall(channel, NULL, 0, buf, thistime, errind); \
	if (failed(errind)) \
	    break; \
	left -= thistime; \
	buf += thistime; \
    } \
}

#define BulkGetDataReq(channel, buffer, amount, errind)	\
{  \
    int left = amount; \
    int thistime; \
    char *buf = buffer; \
			 \
    while (left >0) { \
	thistime = MIN(left, VMTP_MAXSEGSIZE); \
	RPCAwaitCallInd(channel, NULL, 0, errind); \
	if (failed(errind)) \
	    break; \
	RPCReturn(channel, buf, thistime, errind); \
	if (failed(errind)) \
	    break; \
	left -= thistime; \
	buf += thistime; \
    } \
}

#define RRSendRequest 		BulkPutDataReq
#define RRAwaitResponseInd	BulkGetAwaitDataInd
#define RRAwaitRequestInd	BulkPutAwaitDataInd
#define RRSendResponse		BulkGetDataReq


/*
 * No need to Synchronize with the sender to avoid buffering anomalies!
 */

#define BulkGetStart(ch, errind) \
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }

#define AwaitBulkGetStart(ch, errind)	\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }

#define BulkGetStop(ch, errind) 	\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }

#define AwaitBulkGetStop(ch, errind)	\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }



#define BulkPutStart(ch, errind)	\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }

#define AwaitBulkPutStart(ch, errind)\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }

#define BulkPutStop(ch, errind)	\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }

#define AwaitBulkPutStop(ch, errind)	\
    			{(errind)->er_ret = OK; (errind)->er_code = VMTP_OK; }


#define RPCCall(channel, srcbuf, srclen, dstbuf, dstlen, errind) \
    		vmtp_call(channel, srcbuf, srclen, dstbuf, dstlen, errind)
				

#define RPCAwaitCallInd(ch, dstbuf, dstlen, errind) \
    		vmtp_await_call(&(ch), dstbuf, dstlen, errind)

#define RPCReturn(ch, srcbuf, srclen, errind) \
		vmtp_return_call(ch, srcbuf, srclen, errind)
    
#define QueryCall(addrp, srcbuf, srclen, dstbuf, dstlen, errind) \
{ \
  channel_t ch; \
 \
  ConnRequest(addrp, &ch, errind); \
  if (!failed(errind)) { \
    RPCCall(ch, srcbuf, srclen, dstbuf, dstlen, errind); \
    if (!failed(errind)) { \
      DiscRequest(ch, errind); \
    } \
  } \
}

#define QueryAwaitCallInd(serverp, dstbuf, dstlen, chp, errind) \
{ \
    AwaitConnInd(serverp, chp, errind); \
	if (!failed(errind)) { \
	RPCAwaitCallInd(*(chp), dstbuf, dstlen, errind); \
    } \
}

#define QueryReturn(ch, srcbuf, srclen, errind) \
{ \
    RPCReturn(ch, srcbuf, srclen, errind); \
    if (!failed(errind)) { \
	AwaitDiscInd(ch, errind); \
    } \
}

		
#endif _VMTP_
