/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The benchmark server: process the first specification in the linked
 * list of specification.
 *
 * bench-server.c
 */

/* TODO update the measurement composers to record responder results too. */


#include <general.h>
#include <ipc.h>
#include <ipc_pres.h>

#include <math.h>	/* need HUGE */


extern int ClockDiff;
extern int PrintResponder;

/*  */

static int iterations, nfailed, run_failed;
static double timelimit;

extern char protostr[];

bench_server(thissp)
    struct spec *thissp;
{
    bcpaddr_t myaddr;
    msg_t request, response;
    msg_t *rq = &request, *rp = &response;
    state_t state = IDLE;
    int secs;
    long start_time, stop_time;
    int ret = OK;	/* return value */
    
    tprintf("bench_server(0x%x)\n", thissp);
    
    if (thissp == NULLSP)
	return OK;
    
    start_time = time(NULL);
    
    rp->msg_datalen = 0;
    
    if (state == IDLE) {
	/* initialize things ... */
	myaddr.ba_addr = BA_BENCH;
	myaddr.ba_index = 0;
	if (bcp_init(myaddr) == NOTOK) {
	    eprintf("Can't initialize the Benchmark Control Protocol\n");
	    return DONE;
	}

	if (bench_isbasic(spec_bench(thissp))) {
	    result_init();
	    /* leave at foreground and with priority 0 */
	} else {
	    (*bench_proc(spec_bench(thissp)))(0, spec_bench(thissp));
	}
	if (!NoEcho) {
		printf("Executing benchmark:\n");
		spec_print(thissp);
	    }

	nfailed = 0;
	dprintf("nfailed set to %d\n", nfailed);
	run_failed = 0;
	iterations = spec_number(thissp);
	timelimit = spec_timelimit(thissp);
	if (send_create(thissp) == NOTOK) {
	    eprintf("Failed creating the benchmark processes\n");
	    state = IDLE;
	    ret = NOTOK;
	} else
	    state = CREATING;
    }
    while (state != IDLE) {

	secs = parentstate2timeout(state);
    	if (await_recv_msg(rq, secs) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "await_recv_msg", "bench_server");
	    continue;
	}

	rp->msg_datalen = 0;
	rp->msg_to = rq->msg_from;

	switch (rq->msg_code) {

	case MSG_TIMEOUT:
	    switch (handle_timeout(&state, rq, rp)) { 	
	    case OK:
		break;
	    case DONE:
		(void)send_msg(rp);
		eprintf("Timeout => NOTOK return\n");
		ret = NOTOK;
		break;
	    case NOTOK:
		eprintf(EF_IN3, INTERNAL, "handle_timeout", "bench_server");
		state = IDLE;
		break;
	    }
	    break;
	
	case MSG_RESET:
	case MSG_CREATE:
	case MSG_DESTROY:
	case MSG_START:
	case MSG_SPEC:
	    /* should't get this - report the message */
	    eprintf(EF_IN3, PROTOCOL, "Received a bad message",
		    "bench_server");
	    dprintf("\tCode %s, from 0x%x:%d\n", code2str(rq->msg_code),
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    dprintf("\tin state %s\n", state2str(state));
	    rp->msg_code = MSG_PROERROR;
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_msg(rp);
	    break;
	
	case MSG_PROERROR:
	    eprintf("Received a MSG_PROERROR message\n");
	    dprintf("\tCode %s, from 0x%x:%d\n", code2str(rq->msg_code),
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    dprintf("\tin state %s\n", state2str(state));
	    eprintf("Protocol error: resetting benchmark\n");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;

	case MSG_FATAL:
	    eprintf("Benchmark killed by an external signal\n");
	    state = IDLE;
	    ret = DONE;
	    break;

	case MSG_ABORT:
	    eprintf("Benchmark aborted by %s\n",
		    bcpaddr2str(rq->msg_from));
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	    
	case MSG_TERMINATE:
	    eprintf("Benchmark terminated by 0x%x:%d (presumably demon)\n",
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    state = WAIT_TERM;
	    rp->msg_to.ba_addr = BA_CLIENTS;
	    rp->msg_code = MSG_TERMINATE;
	    (void)send_msg(rp);
	    break;

	default:
	    if (state != TAREDOWN && rq->msg_code == MSG_CLOSED) {
		eprintf("Unexpected MSG_CLOSED from %s\n",
			bcpaddr2str(rq->msg_from));
		if (rq->msg_from.ba_addr & BA_SYSTEM)
		    (void)route_remove_dest(rq->msg_from);
		state = IDLE;
		ret = NOTOK;
		break;
	    }

	    if (ret == OK) {
		ret = caseonstate(&state, rq, rp, thissp);
	    } else {
		(void)caseonstate(&state, rq, rp, thissp);
	    }
	    break;
	}
	stprintf("State %s\n", state2str(state));
	
	if (rq->msg_datalen) {
	    dprintf("bench-server: freeing rq data\n");
	    dprintf("\t(0x%x)\n", rq->msg_data);
	    free(rq->msg_data);
	    rq->msg_datalen = 0;
	}
	if (rp->msg_datalen) {
	    dprintf("bench-server: freeing rp data\n");
	    dprintf("\t(0x%x)\n", rp->msg_data);
	    free(rp->msg_data);
	    rp->msg_datalen = 0;
	}
    } /* while (state != IDLE) */

    tprintf("Done - shutting things up (thissp = 0x%x)\n", thissp);
    
    if (ret == NOTOK || ret == DONE) {
	dprintf("Failure - making sure all processes are gone\n");
	/* make sure things are reset */
	rp->msg_from.ba_addr = BA_CLIENTS | BA_SERVERS;
	rp->msg_to.ba_addr = BA_SYSTEMS;
	rp->msg_code = MSG_DESTROY;
	(void)bcp_send_msg(rp);

	(void)route_remove_all();

	rp->msg_from.ba_addr = BA_CLIENTS | BA_SERVERS;
	(void)set_state(rp->msg_from, DEAD);
	dprintf("...printing..\n");
	stop_time = time(NULL);

	printf("Benchmark using the %s protocol\n",
	       protostr);
	printf("Started on %s", ctime(&start_time));
	printf("Failed on %s",  ctime(&stop_time));

	return ret;
    }

    dprintf("Reporting results (bench = 0x%x)\n", spec_bench(thissp));

    stop_time = time(NULL);

    printf("Benchmark using the %s protocol\n",
	   protostr);
    printf("Started on %s", ctime(&start_time));
    printf("Ended on %s", ctime(&stop_time));
    
    /* present results */
    if (bench_isbasic(spec_bench(thissp))) {
	result_print(thissp, nfailed);
    } else {
	(*bench_proc(spec_bench(thissp)))(2, thissp, nfailed);
    }

    dprintf("bench-server - done\n");
    return ret;
} /* bench_server */

/*  */

static bcpaddr_t ALL_PROCS = { BA_CLIENTS | BA_SERVERS, 0 };
static bcpaddr_t ALL_CLIENTS = { BA_CLIENTS, 0 };
static bcpaddr_t ALL_SERVERS = { BA_SERVERS, 0 };

static int caseonstate(a_state, rq, rp, sp)
    state_t *a_state;
    msg_t *rq, *rp;
    struct spec *sp;
{
    int ret = OK;
    state_t state = *a_state;

    switch (state) {
/*  SETUP */
	
    case CREATING:
	switch (rq->msg_code) {
	case MSG_OK:
	    if (set_state(rq->msg_from, SETUP) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "creating processes");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING; 
	    ret = NOTOK;
	    break;
	}
	break;

/*  */
	
    case SETUP:
	switch (rq->msg_code) {
	    struct address_t *addr;
	    int tmp;

	case MSG_ADDR:
	    if (rq->msg_from.ba_addr != BA_SERVER) {
		eprintf(EF_IN3, PROTOCOL, "Received a MSG_ADDR",
			"bench_server - Not from a server procedure");
		(void)handle_proerror(rq, rp, state);
		(void)send_abort(rp);
		state = ABORTING;
	    	ret = NOTOK;
		break;
	    }
	    tmp = rq->msg_datalen;
	    addr = str2address(rq->msg_data, &tmp);
	    if (addr == NULL) {
		eprintf(EF_IN4, INTERNAL, PROTOCOL, "str2address",
			"bench_server");
		(void)send_abort(rp);
		state = ABORTING;
	    	ret = NOTOK;
		break;
	    }
	    if (set_state(rq->msg_from, INIT) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }
	    set_addr(rq->msg_from, addr);
	    break;
	    
	case MSG_OK:
	    if (set_state(rq->msg_from, WAIT_INIT) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }
	    break;
	    
	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "seting up processes with specification");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;

/*  */

/* wait for all OK's from clients */
	
    case WAIT_INIT:
	switch (rq->msg_code) {
	case MSG_OK:
	    if (set_state(rq->msg_from, INIT) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "giving server addresses to local procedures");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;

/*  EXECUTION */
	
    case WAIT_STARTED:
	switch (rq->msg_code) {
	case MSG_STARTED:
	    if (set_state(rq->msg_from, EXECUTING) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "starting remote processes");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;
	
/*  */
	
    case EXECUTING:
	switch (rq->msg_code) {
	case MSG_ALARM:
	    rp->msg_to.ba_addr = BA_CLIENTS;
	    rp->msg_code = MSG_TERMINATE;
	    state = WAIT_TERM;
	    (void)send_msg(rp);
	    break;
	    
	case MSG_STOPPED:
	    if (set_state(rq->msg_from, INIT) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = IDLE;
		break;
	    }
	    {
		statistics_t *stat;
		int tmp = rq->msg_datalen;
		
		stat = str2statistics(rq->msg_data, &tmp);
		if (stat == NULL) {
		    eprintf(EF_IN4, INTERNAL, PROTOCOL, "str2statistics",
			    "bench_server");
		    (void)send_abort(rp);
		    state = ABORTING;
		    ret = NOTOK;
		    break;
		}
		stat->st_starttime = host_clockdiff(rq->msg_from,
						    stat->st_starttime);
		stat->st_stoptime = host_clockdiff(rq->msg_from,
						    stat->st_stoptime);
		(void)set_result(rq->msg_from, stat);
	    }
	    break;
	    
	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "benchmark execution");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;

/*  */

    case WAIT_TERM:
    case WAIT_STOPPED:
	switch (rq->msg_code) {
	case MSG_STOPPED:
	case MSG_TERMINATED:
	    if (set_state(rq->msg_from, INIT) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = IDLE;
		break;
	    }
	    {
		statistics_t *stat;
		int tmp = rq->msg_datalen;
		
		stat = str2statistics(rq->msg_data, &tmp);
		if (stat == NULL) {
		    eprintf(EF_IN4, INTERNAL, PROTOCOL, "str2statistics",
			    "bench_server");
		    (void)send_abort(rp);
		    state = ABORTING;
		    ret = NOTOK;
		    break;
		}
		stat->st_starttime = host_clockdiff(rq->msg_from,
						    stat->st_starttime);
		stat->st_stoptime = host_clockdiff(rq->msg_from,
						    stat->st_stoptime);
		(void)set_result(rq->msg_from, stat);
	    }
	    break;
	    
	case MSG_NOTOK:
	case MSG_FAILURE:
	    if (rq->msg_from.ba_addr == BA_SERVER) {
		stprintf("Ignoring an MSG_FAILURE from %s in state %s\n",
			 bcpaddr2str(rq->msg_from), state2str(state));
		break;
	    }
	    report_failure(rq, "benchmark execution - terminating");
	    state = FAILING;
	    rp->msg_to.ba_addr = BA_SERVERS | BA_CLIENTS;
	    rp->msg_code = MSG_TERMINATE;
	    (void)send_msg(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;

/*  */
	
    case TAREDOWN:
	switch (rq->msg_code) {
	case MSG_CLOSED:
	    if (rq->msg_from.ba_addr & BA_SYSTEM) {
		state = IDLE;
		break;
	    }

	    if (set_state(rq->msg_from, UNKNOWN) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
 	   	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }	
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "destroying processes");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;

/*  */

    case FAILING:
	switch (rq->msg_code) {
	case MSG_TERMINATED:
	case MSG_STOPPED:
	    /*
	     * Ignore the data portion - no results when failed
	     * Increase the failure count (reset when new spec)
	     */
	    if  (set_state(rq->msg_from, INIT) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    if (rq->msg_from.ba_addr == BA_SERVER
		|| rq->msg_from.ba_addr == BA_CLIENT) {
		stprintf("Ignoring an MSG_FAILURE from %s in state %s\n",
			 bcpaddr2str(rq->msg_from), state2str(state));
		break;
	    }
	    report_failure(rq, "failing the benchmark execution");
	    state = ABORTING;
	    ret = NOTOK;
	    (void)send_abort(rp);
	    break;
	default:
	    (void)handle_proerror(rq, rp, state);
	    (void)send_abort(rp);
	    state = ABORTING;
	    ret = NOTOK;
	    break;
	}
	break;

/*  */
	
    case ABORTING:
	switch (rq->msg_code) {
	case MSG_STOPPED:
	case MSG_TERMINATED:
	    /* ignore - an ABORTED will follow */
	case MSG_ABORTED:
	    if (set_state(rq->msg_from, IDLE) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_state", "bench_server");
		state = ABORTING;
	    	ret = NOTOK;
		(void)send_abort(rp);
		break;
	    }	
	    break;

	case MSG_NOTOK:
	case MSG_FAILURE:
	    report_failure(rq, "aborting benchmark execution - ignored");
	    ret = NOTOK;
	    break;
	default:
	    eprintf("%s: %s %s in state %s - ignored\n",
		    INTERNAL, "Received a",
		code2str(rq->msg_code),
		state2str(state));
	    break;
	}
	break;

/*  BAD states */
	
    default:
	eprintf(EF_IN3, INTERNAL, "In bad state", "bench_server");
	eprintf("%s: %s %s in state %s\n", INTERNAL, "Received a",
		code2str(rq->msg_code),
		state2str(state));
	break;

    }

/*  SECOND PART */

    stprintf("Entering second part in state %s\n", state2str(state));

 redo:
    
    switch (state) {
    case CREATING:
	if (check_state(ALL_PROCS, SETUP) == OK) {
	    /* Estimate the clock differences ... */
	    if (ClockDiff)
		host_getclockdiff(); 
	    
	    if (send_spec(sp) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "send_spec", "bench_server");
		state = ABORTING;
		ret = NOTOK;
		(void)send_abort(rp);
	    } else
		state = SETUP;
	}
	break;
	
    case SETUP:
	if (check_state(ALL_SERVERS, INIT) == OK
	    &&  check_state(ALL_CLIENTS, WAIT_INIT) == OK) {
	    state = WAIT_INIT;
	    (void)send_addrs(sp);
	}
	break;
	
    case INIT:
    case WAIT_INIT:
	if (check_state(ALL_PROCS, INIT) == OK) {
	    if (iterations-- > 0) {
		run_failed = 0;
		rp->msg_code = MSG_START;
		rp->msg_to.ba_addr = BA_SERVERS;
		state = WAIT_STARTED;
		(void)send_msg(rp);
		(void)send_bg_clients(rp, sp);
	    } else {
		dprintf("Destroying processes\n");
		rp->msg_code = MSG_DESTROY;
		rp->msg_from.ba_addr = BA_CLIENTS | BA_SERVERS;
		rp->msg_to.ba_addr = BA_SYSTEMS;
		state = TAREDOWN;
		(void)bcp_send_msg(rp);
		
		(void)route_remove_all();
	    }
	}
	break;

    case WAIT_STARTED:
	if (check_state(ALL_SERVERS, EXECUTING) == OK
	    && check_bg_client_state(EXECUTING) == OK) {
	    if (set_fg_client_state(EXECUTING) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_fg_client_state",
			"bench_server");
		break;
	    }
	    state = EXECUTING;
	    rp->msg_code = MSG_START;
	    if (send_fg_clients(rp, sp) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "send_fg_clients",
			"bench_server");
		break;
	    }
	    if (!timelimit_unlim(timelimit))
		bench_alarm(timelimit);
	}	
	break;
	
    case ABORTING:
	if (check_state(ALL_PROCS, IDLE) == OK) {
	    state = IDLE;
	}
	break;

    case TAREDOWN:
	if (check_state(ALL_PROCS, UNKNOWN) == OK) {
	    state = IDLE;
	}
	break;
	
    case FAILING:
	if (check_state(ALL_PROCS, INIT) == OK) {
	    nfailed++;
	    run_failed = 1;
	    eprintf("One benchmark run failed: number %d out of %d (%d failed)\n",
		    spec_number(sp)-iterations, spec_number(sp),
		    nfailed);
	}

	/* fall thru ... */
	
    case EXECUTING:
    case WAIT_TERM:
    case WAIT_STOPPED:
	if (check_state(ALL_PROCS, INIT) == OK) {
	    if (!timelimit_unlim(timelimit))
		bench_unalarm();
	    stprintf("Processing the results of one run\n");
	    /* compress the results if any - need flag for this run failed */
	    if (!run_failed) {
		statistics_t *stats, *rstats;
		int nstats;
		
		nstats = get_results(&stats);
		(void)get_respresults(&rstats);

		if (bench_isbasic(spec_bench(sp))) {
		    /* use default filter on results */
		    if (result_handle(nstats, stats, rstats, sp) == NOTOK) {
			nfailed++;
			run_failed = 1;
			eprintf("One benchmark run ignored by filter: number %d out of %d (%d failed)\n",
				spec_number(sp)-iterations, spec_number(sp),
				nfailed);
		    }
		} else {
		    /* use supplied filter */
		    if ((*bench_proc(spec_bench(sp)))(1, nstats, stats, rstats, sp) == NOTOK) {
			nfailed++;
			run_failed = 1;
			eprintf("One benchmark run ignored by filter: number %d out of %d (%d failed)\n",
				spec_number(sp)-iterations, spec_number(sp),
				nfailed);
		    }
		}
		free((char *)stats);
	    }
	    state = INIT;
	    goto redo;
	    /* NOTREACHED */
	} else if (check_state(ALL_CLIENTS, INIT) == OK
#ifdef notdef
		   && rq->msg_from.ba_addr == BA_CLIENT
#else
		   && state != WAIT_STOPPED
#endif notdef
		   ) {
	    rp->msg_to.ba_addr = BA_SERVERS;
	    rp->msg_code = (state == EXECUTING) ? MSG_STOP : MSG_TERMINATE;
	    (void)send_msg(rp);
	    state = WAIT_STOPPED;
	} else if (check_fg_client_state(INIT) == OK
#ifdef notdef
		   && rq->msg_from.ba_addr == BA_CLIENT
#else
		   && state != WAIT_STOPPED
#endif notdef
		   ) {
	    state = WAIT_STOPPED;
	    rp->msg_code = (state == EXECUTING) ? MSG_STOP : MSG_TERMINATE;
	    (void)send_bg_clients(rp, sp);
	    rp->msg_to.ba_addr = BA_SERVERS;
	    rp->msg_code = (state == EXECUTING) ? MSG_STOP : MSG_TERMINATE;
	    (void)send_msg(rp);
	}
	break;

    }
    *a_state = state;
    return ret;
} /* caseonstate */

/*  */

static int handle_proerror(rq, rp, state)
    msg_t *rq, *rp;
    int state;
{
    eprintf("%s: %s %s in state %s\n", PROTOCOL, "Received a",
	    code2str(rq->msg_code),
	    state2str(state));
    rp->msg_code = MSG_PROERROR;
    return send_msg(rp);
} /* handle_proerror */

/*  */

static report_failure(msg, str)
    msg_t *msg;
    char *str;
{
    eprintf("Error occured when %s:\n\t", str);
    if (msg->msg_datalen)
	eprintf("%s\n", msg->msg_data);
    else
	eprintf("Unspecified error - message code %s\n",
		code2str(msg->msg_code));
} /* report_failure */

/*  */

static int send_abort(msg)
    msg_t *msg;
{
    msg->msg_code = MSG_ABORT;
    msg->msg_to.ba_addr = BA_CLIENTS | BA_SERVERS;
    msg->msg_datalen = 0;
    return send_msg(msg);
} /* send_abort */

/*  */

static int send_create(sp)
    struct spec *sp;
{
    int nproc;
    int i, j;
    struct part *pt;

    tprintf("send_create(0x%x)\n", sp);
    
    nproc = bench_numprocesses(spec_bench(sp));
    dprintf("send_create: %d processes\n", nproc);

    if (alloc_childstate(nproc) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "alloc_childstate", "send_create");
	return NOTOK;
    }

    if (bench_isbasic(spec_bench(sp))) {
	if (send_create_msgs(spec_bench(sp), 0) == NOTOK)
	    return NOTOK;
	if (basic_background(spec_bench(sp))) {
	    bcpaddr_t addr;

	    addr.ba_index = 0;
	    addr.ba_addr = BA_CLIENT | BA_SERVER;
	    set_statebackground(addr);
	}
    } else {

	j = 0;
	for (pt = comp_parts(spec_bench(sp)); pt != NULLPT; pt = part_next(pt)) {
	    for (i = 0; i < part_number(pt); i++, j++) {
		if (send_create_msgs(part_bench(pt), j) == NOTOK)
		    return NOTOK;
		if (basic_background(part_bench(pt))) {
		    bcpaddr_t addr;
		    
		    addr.ba_index = j;
		    addr.ba_addr = BA_CLIENT | BA_SERVER;
		    set_statebackground(addr);
		}
	    }
	}
    }

    return OK;
} /* send_create */

/*  */

int send_create_msgs(be, index)
    struct bench *be;
    int index;
{
    msg_t msgs, *msg = &msgs;

    tprintf("send_create_msgs(0x%X, %d)\n", be, index);
    
    if (!bench_isbasic(be)) {
	eprintf(EF_IN3, PARAMETER, "be", "send_create_msgs");
	return NOTOK;
    }
    msg->msg_code = MSG_CREATE;
    msg->msg_datalen = 0;
    msg->msg_data = NULL;
    
    if (host_add_connect(basic_client(be), &msg->msg_to) == NOTOK){
	eprintf(EF_IN3, PARAMETER, "host_add_connect",
		"send_create_msgs - client host");
	return NOTOK;
    }
    msg->msg_from.ba_addr = BA_CLIENT;
    msg->msg_from.ba_index = index;
    if (bcp_send_msg(msg) == NOTOK) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "bcp_send_msg",
		"send_create");
	return NOTOK;
    }
    (void)route_enter(msg->msg_from, msg->msg_to);

    if (host_add_connect(basic_server(be), &msg->msg_to) == NOTOK){
	eprintf(EF_IN3, PARAMETER, "host_add_connect",
		"send_create_msgs - server host");
	return NOTOK;
    }
    msg->msg_from.ba_addr = BA_SERVER;
    msg->msg_from.ba_index = index;
    if (bcp_send_msg(msg) == NOTOK) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "bcp_send_msg",
		"send_create");
	return NOTOK;
    }
    (void)route_enter(msg->msg_from, msg->msg_to);
    return OK;
} /* send_create_msgs */

/*  */

/* #define SENDDEMONSPEC */

static int send_spec(sp)
    struct spec *sp;
{
    int nproc = 0;
    char *str;
    int strlen;
    
    tprintf("send_spec(0x%x)\n", sp);

#ifdef SENDDEMONSPEC
    str = spec2str(sp, &strlen);
    if (str == NULL) {
	eprintf(EF_IN3, INTERNAL, "spec2str", "send_spec");
	return NOTOK;
    }
    if (send_spec_msg(BA_SYSTEMS, 0, str, strlen) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "send_spec_msg", "send_spec");
	free(str);
	return NOTOK;
    }
    free(str);
#endif SENDDEMONSPEC
    
    if (bench_isbasic(spec_bench(sp))) {
	nproc = 1;
	dprintf("send_spec: %d processes\n", nproc);

	dprintf("send_spec: calling basic2str\n");
	str = basic2str(spec_bench(sp), &strlen);
	dprintf("send_spec: %d processes\n", nproc);
	if (str == NULL) {
	    eprintf(EF_IN3, INTERNAL, "basic2str", "send_spec");
	    return NOTOK;
	}
	dprintf("send_spec: str 0x%x\n", str);
	if (send_spec_msg(BA_CLIENT, 0, str, strlen) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "send_spec_msg", "send_spec");
	    free(str);
	    return NOTOK;
	}
	if (send_spec_msg(BA_SERVER, 0, str, strlen) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, "send_spec_msg", "send_spec");
	    free(str);
	    return NOTOK;
	}
	free(str);
	return OK;
    } else {
	int i, ptnum;
	struct part *pt;


	nproc = bench_numprocesses(spec_bench(sp));
	
	pt = comp_parts(spec_bench(sp));
	ptnum = part_number(pt);
	
	dprintf("send_spec: %d processes\n", nproc);
	
	for (i = 0; i < nproc; i++) {
	    str = basic2str(part_bench(pt), &strlen);
	    if (str == NULL) {
		eprintf(EF_IN3, INTERNAL, "basic2str", "send_spec");
		return NOTOK;
	    }
	    if (send_spec_msg(BA_CLIENT,
			      i, str, strlen) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "send_spec_msg", "send_spec");
		free(str);
		return NOTOK;
	    }
	    if (send_spec_msg(BA_SERVER,
			      i, str, strlen) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "send_spec_msg", "send_spec");
		free(str);
		return NOTOK;
	    }
	    free(str);
	    if (--ptnum == 0) {
		pt = part_next(pt);
		if (pt != NULLPT)
		    ptnum = part_number(pt);
	    }
	}
	return OK;
    }
    /* NOTREACHED */
} /* send_spec */
    
/*  */

send_spec_msg(ba, index, str, strlen)
    int ba, index;
    char *str;
    int strlen;
{
    msg_t msgs, *msg = &msgs;

    tprintf("send_spec_msg(0x%x:%d, 0x%x, %d)\n", ba, index, str, strlen);
    
    msg->msg_to.ba_addr = ba;
    msg->msg_to.ba_index = index;
    msg->msg_code = MSG_SPEC;
    msg->msg_data = str;
    msg->msg_datalen = strlen;
    if (send_msg(msg) == NOTOK) {
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "send_msg",
		"send_spec_msg");
	return NOTOK;
    }
    return OK;
} /* send_spec_msg */

/*  */


static int send_addrs(sp)
    struct spec *sp;
{
    int nproc;
    int i;
    msg_t msgs, *msg = &msgs;
    struct address_t *addr;
    
    tprintf("send_addrs()\n");

    nproc = bench_numprocesses(spec_bench(sp));
    
    dprintf("send_addrs: %d processes\n", nproc);

    for (i = 0; i < nproc; i++) {
	msg->msg_to.ba_addr = BA_CLIENT;
	msg->msg_to.ba_index = i;
	msg->msg_code = MSG_ADDR;
	addr = get_addr(i);
	if (addr == NULL) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "get_addr",
		    "send_addrs");
	    return NOTOK;
	}
	msg->msg_data = address2str(addr, &msg->msg_datalen);
	if (msg->msg_data == NULL) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "address2str",
		    "send_addrs");
	    return NOTOK;
	}
	if (send_msg(msg) == NOTOK) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "bcp_send_msg",
		    "send_addrs");
	    return NOTOK;
	}
	free(msg->msg_data);
    }

    return OK;
} /* send_addrs */
    

/*  RESULT MODULE */

static int nresult = 0;
static statistics_t *sums = NULL, *sqsums = NULL, *mins = NULL, *maxs = NULL;
static statistics_t *rsums = NULL, *rsqsums = NULL, *rmins = NULL, 
	*rmaxs = NULL;	/* responder results */

result_init()
{
    tprintf("result_init()\n");
    
    nresult = 0;
    if (sums != NULL)
	free((char *)sums);
    if (sqsums != NULL)
	free((char *)sqsums);
    if (mins != NULL)
	free((char *)mins);
    if (maxs != NULL)
	free((char *)maxs);
    if (rsums != NULL)
	free((char *)rsums);
    if (rsqsums != NULL)
	free((char *)rsqsums);
    if (rmins != NULL)
	free((char *)rmins);
    if (rmaxs != NULL)
	free((char *)rmaxs);

    sums = sqsums = mins = maxs = NULL;
    rsums = rsqsums = rmins = rmaxs = NULL;

} /* result_init */

/*  */

int result_handle(nstats, stats, rstats, sp)
    int nstats;
    statistics_t stats[];
    statistics_t rstats[];
    struct spec *sp;
{
    statistics_t ress, *res = &ress;
    statistics_t respress, *respres = &respress;
    int i;
    
    tprintf("result_handle(%d, 0x%x, 0x%x)\n", nstats, stats, sp);

    if (nresult == 0) {
	if (bench_isbasic(spec_bench(sp))) {
	    nresult = 1;
	} else {
	    struct part *pt;
	    
	    if ((nresult = comp_number(spec_bench(sp))) == NP_UNSPEC) {
		nresult = 0;
		pt = comp_parts(spec_bench(sp));
		while (pt != NULLPT) {
		    nresult++;
		    pt = part_next(pt);
		}
		comp_number(spec_bench(sp)) = nresult;
	    }
	}
	dprintf("result_handle: nresult set to %d\n", nresult);
	sums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (sums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	sqsums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (sqsums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	mins = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (mins == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	maxs = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (maxs == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	rsums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rsums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	rsqsums = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rsqsums == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	rmins = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rmins == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	rmaxs = (statistics_t *)malloc(sizeof(statistics_t) * nresult);
	if (rmaxs == NULL) {
	    eprintf(EF_IN3, INTERNAL, RESOURCE, "malloc - result_handle");
	    return NOTOK;
	}
	for (i = 0; i < nresult; i++) {
	    InitStats(&sums[i], (double)0);
	    InitStats(&sqsums[i], (double)0);
	    InitStats(&mins[i], (double)HUGE);
	    InitStats(&maxs[i], (double)-HUGE);
	    InitStats(&rsums[i], (double)0);
	    InitStats(&rsqsums[i], (double)0);
	    InitStats(&rmins[i], (double)HUGE);
	    InitStats(&rmaxs[i], (double)-HUGE);
	}
    }
    if (nstats == 0) {
	eprintf(EF_IN3, INTERNAL, "No result to handle", "result_handle");
	return NOTOK;
    }
    if (nstats == 1) {
	*res = stats[0];
	*respres = rstats[0];
	/* update statistics */
	result_update(res, &sums[0], &mins[0], &maxs[0], &sqsums[0]);
	result_update(respres, &rsums[0], &rmins[0], &rmaxs[0], &rsqsums[0]);
    } else {
	int i, part;
	struct part *pt;
	double minstart, maxstop;
	statistics_t *st = stats;
	statistics_t *rst = rstats;

	if (bench_isbasic(spec_bench(sp))) {
	    eprintf(EF_IN3, INTERNAL, PARAMETER,
		    "result_update - basic benchmark and many stats");
	    return NOTOK;
	}
	pt = comp_parts(spec_bench(sp));
	for (part = 0; part < nresult; part++, pt = part_next(pt)) {
	    dprintf("result_update: part %d, %d replicas\n", part,
		    part_number(pt));
	    TotalStatistics(part_number(pt), st, res);
	    minstart =  st[0].st_starttime;
	    maxstop = st[0].st_stoptime;
	    for (i = 1; i < part_number(pt); i++) {
		if (st[i].st_starttime < minstart)
		    minstart =  st[i].st_starttime;
		if (st[i].st_stoptime > maxstop)
		    maxstop = st[i].st_stoptime;
	    }
	    res->st_starttime /= part_number(pt);
	    res->st_stoptime /= part_number(pt);
	    res->st_elapsed = 1000*(maxstop - minstart);

	    /* And the responders */
	    TotalStatistics(part_number(pt), rst, respres);
	    minstart =  rst[0].st_starttime;
	    maxstop = rst[0].st_stoptime;
	    for (i = 1; i < part_number(pt); i++) {
		if (rst[i].st_starttime < minstart)
		    minstart =  rst[i].st_starttime;
		if (rst[i].st_stoptime > maxstop)
		    maxstop = rst[i].st_stoptime;
	    }
	    respres->st_starttime /= part_number(pt);
	    respres->st_stoptime /= part_number(pt);
	    respres->st_elapsed = 1000*(maxstop - minstart);
	    
	    
	    /* update statistics */
	    result_update(res, &sums[part], &mins[part],
			  &maxs[part], &sqsums[part]);
	    result_update(respres, &rsums[part], &rmins[part],
			  &rmaxs[part], &rsqsums[part]);

	    st = st + part_number(pt);
	    rst = rst + part_number(pt);
	}
    }

    return OK;
} /* result_handle */

/*  */

static int result_update(result, sum, min, max, sqsum)
    statistics_t *result, *sum, *min, *max, *sqsum;
{
    tprintf("result_update(0x%x, 0x%x, 0x%x, 0x%x, 0x%x)\n",
	    result, sum, min, max, sqsum);
    
    SumtoStats(result, sum);
    MintoStats(result, min);
    MaxtoStats(result, max);
    SqsumtoStats(result, sqsum);
} /* result_update */

/*  */

int result_print(sp, nfailure)
    struct spec *sp;
    int nfailure;
{
    int i;
    int nrun;

    tprintf("result_print(0x%x, %d)\n", sp, nfailure);

    nrun = spec_number(sp);
    printf("Results for %d runs:\n", nrun);
    if (nfailure != 0)
	printf("Failure percentage: %.2f%%\n",
	       100*(double)nfailure/nrun);

    for (i = 0; i < nresult; i++) {
	printf("Initiator %d:\n", i);
	stats_print("", &sums[i], &mins[i], &maxs[i], &sqsums[i],
		    nrun - nfailure);
	if (PrintResponder) {
		printf("Responder %d:\n", i);
		stats_print("r", &rsums[i], &rmins[i], &rmaxs[i], &rsqsums[i],
			    nrun - nfailure);
	    }
    }
} /* result_print */

/*  */

int send_fg_clients(msg, sp)
    msg_t *msg;
    struct spec *sp;
{
    int i;
    int ret = OK;
    int nproc;

    tprintf("send_fg_clients(0x%x)\n", msg);

    nproc = bench_numprocesses(spec_bench(sp));

    msg->msg_to.ba_addr = BA_CLIENT;
    for (i = 0; i < nproc; i++) {
	msg->msg_to.ba_index = i;
	if (!get_statebackground(msg->msg_to)) {
	    msg->msg_to.ba_index = i;
	    if (send_msg(msg) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "send_msg", "send_fg_clients");
		ret = NOTOK;
	    }
	} else
	    dprintf("\tClient %d is in background\n", i);
    }
    return ret;
} /* send_fg_clients */

/*  */

int send_bg_clients(msg, sp)
    msg_t *msg;
    struct spec *sp;
{
    int i;
    int ret = OK;
    int nproc;

    tprintf("send_bg_clients(0x%x)\n", msg);

    nproc = bench_numprocesses(spec_bench(sp));

    msg->msg_to.ba_addr = BA_CLIENT;
    for (i = 0; i < nproc; i++) {
	msg->msg_to.ba_index = i;
	if (get_statebackground(msg->msg_to)) {
	    msg->msg_to.ba_index = i;
	    if (send_msg(msg) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "send_msg", "send_bg_clients");
		ret = NOTOK;
	    }
	} else
	    dprintf("\tClient %d is in foreground\n", i);
    }
    return ret;
} /* send_bg_clients */

