/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The lexical analyzer interfacing the yacc-generated parser
 */

/*
 * Exports:
 * 	yylex()
 *
 * Imports:
 *	symbol_lookup() - for reserved words
 */

#include <general.h>

#include <y.tab.h>

#define Trace 0
#define Debug 0

int atoi();
double atof();

extern YYSTYPE yylval;

#define BUFFSIZE 128

yylex()
{
    int c;


    c = getchar();
    while (c == ' ' || c == '\n' || c == '\t') /* skip white space */
    	c = getchar();

    if (c == EOF) {
	if (Debug)
	    printf("yylex: caught an end of file\n");
	return 0;
    }
    
    if (isalpha(c))
	return getstring(c);

    if(isdigit(c) || c == '+' || c == '-' || c == '.')
	return getnumber(c);

    if (Debug)
	printf("yylex: passing character %c (%d)\n", c, c);
    
    return (c);
}


getnumber(c)
    int c;
{
    char buff[BUFFSIZE], *cp = buff;
    enum {MANTISSA, FRAC, SIGNEXP, EXP} state = MANTISSA;

    if (c == '.')
	state = FRAC;
    
    *cp++ = c;
    while (cp - buff < BUFFSIZE-1) {
	c = getchar();
	if (isdigit(c)) {
	    *cp++ = c;
	    continue;
	}
	switch (state) {
	case MANTISSA:
	    if (c == '.') {
		state = FRAC;
		*cp++ = c;
		continue;
	    }
	    if (c == 'e' || c == 'E') {
		state = SIGNEXP;
		*cp++ = c;
		continue;
	    }
	    break;

	case FRAC:
	    if (c == 'e' || c == 'E') {
		state = SIGNEXP;
		*cp++ = c;
		continue;
	    }
	    break;

	case SIGNEXP:
	    if (isdigit(c) || c == '+' || c == '-') {
		state = EXP;
		*cp++ = c;
		continue;
	    }
	    break;

	case EXP:
	    break;
	} /* If we have fallen through the number has ended */
    	break;
    }
    if (cp-buff >= BUFFSIZE-1) {
	buff[BUFFSIZE-1] = '\0';
	fprintf(stderr, "Too long constant: %s\n", buff);
	exit(1);
    }
    ungetc(c, stdin);

    *cp = '\0';

    if (Debug)
	dprintf("scanned number: %s interpreted as %s\n", buff,
		(state == MANTISSA) ? "int" : "float");
    
    if (state == MANTISSA) {
	/* Interpret as an integer */
	yylval.uint = atoi(buff);
	return INTEGER;
    }
    yylval.ufloat = atof(buff);
    return FLOAT;
}


/*
 * Read chars until a space - look it up in the symbol table for "tokenizing"
 */
getstring(c)
    int c;
{
    char buff[BUFFSIZE], *cp = buff;
    struct symbol *se;
    
    *cp++ = c;
    while (cp - buff < BUFFSIZE -1) {
	c = getchar();
	if (!isalnum(c) && c != '.' && c != '_' && c != '-')
	    break;
	*cp++ = c;
    }
    if (cp-buff >= BUFFSIZE-1) {
	buff[BUFFSIZE-1] = '\0';
	fprintf(stderr, "Too long string: %s\n", buff);
	exit(1);
    }

    ungetc(c, stdin);
    *cp++ = '\0';

    se = symbol_lookup(buff);
    if (se == NULL || !symbol_istoken(se)) {
	char *res;

	res = (char *)malloc(cp-buff);
	if (res == NULL) {
	    fprintf(stderr, "Internal error: malloc in the scanner failed!\n");
	    exit(1);
	}
	strncpy(res, buff, cp-buff);
	yylval.ustring = res;
	if (Debug)
	    dprintf("getstring found string: %s\n", res);
	return STRING;
    }
    if (Debug)
	dprintf("getstring found token: %s\n", buff);

    return se->se_token;
}

