#include <sys/types.h>
#include <sys/time.h>

#include "asn1.h"
#include "snmp.h"
#include "snmp_impl.h"
#include "snmp_vars.h"

#include "view.h"

#define OIDCMP(l1, l2, o1, o2) (((l1) == (l2)) \
				&& !bcmp((char *)(o1), (char *)(o2), \
					 (l1)*sizeof(oid)))

#define VIEWPARTY_MASK       1
#define VIEWSUBTREE_MASK     2
#define VIEWSTATUS_MASK      4
#define VIEWMASK_MASK        8

#define VIEWCOMPLETE_MASK	15 /* all columns */


struct viewEntry *
view_rowCreate(viewParty, viewPartyLen, viewSubtree, viewSubtreeLen)
    oid *viewParty, *viewSubtree;
    int viewPartyLen, viewSubtreeLen;
{
    struct viewEntry *vp;

    if (viewPartyLen > 32)
	return NULL;
    if (viewSubtreeLen > 32)
	return NULL;
    vp = view_createEntry(viewParty, viewPartyLen,
			  viewSubtree, viewSubtreeLen);
    vp->viewBitMask = 0;
    vp->reserved->viewStatus = VIEWINVALID;

    vp->viewBitMask = vp->reserved->viewBitMask =
	VIEWPARTY_MASK | VIEWSUBTREE_MASK;
    /* Watch out for this becoming permanent by accident:
     * If during FREE stage below we discover row didn't exist before,
     * free row.
     */
    return vp;
}

view_rowDelete(viewParty, viewPartyLen, viewSubtree, viewSubtreeLen)
    oid *viewParty, *viewSubtree;
    int viewPartyLen, viewSubtreeLen;
{
    view_destroyEntry(viewParty, viewPartyLen, viewSubtree, viewSubtreeLen);
}

/*
 * If statP is non-NULL, the referenced object is at that location.
 * If statP is NULL and vp is non-NULL, the instance exists, but not this
 * variable.
 * If statP is NULL and vp is NULL, then neither this instance nor the
 * variable exists.
 */
int
write_view(action, var_val, var_val_type, var_val_len, statP, name, length)
   int      action;
   u_char   *var_val;
   u_char   var_val_type;
   int      var_val_len;
   u_char   *statP;
   oid      *name;
   int      length;
{
    struct viewEntry *vp, *rp;
    int var, size, viewPartyLen, viewSubtreeLen;
    oid *viewParty, *viewSubtree;
    long val;
    int bigsize = 1000;

/*
 * This routine handles requests for variables of the form:
 * ...snmpSecrets.partyViews.viewTable.viewEntry.X.oidlen.oid.oidlen.oid
 * or .1.3.6.1.2.1.21.3.1.1.X.oidlen.oid.oidlen.oid, where the oid suffixes are
 * variable length.
 * Therefore, the length of the first index is name[11] and the index starts
 * at name[12].  The length of the second index starts at name[12 + name[11]],
 * and the second index starts at name[13 + name[11]].
 */
    if (length < 16)
	return SMP_ERR_NOCREATION;
    var = name[10];
    viewPartyLen = name[11];
    viewParty = name + 12;
    if (length <= 12 + viewPartyLen)
	return SMP_ERR_NOCREATION;
    viewSubtreeLen = name[12 + viewPartyLen];
    viewSubtree = name + 13 + viewPartyLen;
    if (length != 13 + viewPartyLen + viewSubtreeLen)
	return SMP_ERR_NOCREATION;
    /* XXX are these length checks necessary?  If not, take them out of
       here and party_vars.c */

    vp = view_getEntry(viewParty, viewPartyLen, viewSubtree, viewSubtreeLen);
    if (vp)
	rp = vp->reserved;
    if (action == RESERVE1 && !vp){
	if ((vp = view_rowCreate(viewParty, viewPartyLen,
				viewSubtree, viewSubtreeLen)) == NULL)
	    return SMP_ERR_RESOURCEUNAVAILABLE;
	rp = vp->reserved;
	/* create default vals here in reserve area */
	rp->viewStatus = VIEWINCLUDED;
	rp->viewMaskLen = 0;
	rp->viewBitMask = VIEWCOMPLETE_MASK;
    } else if (action == COMMIT){
	if (vp->viewStatus == VIEWNONEXISTENT){
	    /* commit the default vals */
	    /* This havpens at most once per entry because the status is set to
	       valid after the first pass.  After that, this commit code
	       does not get executed.  It is also important to note that this
	       gets executed before any of the commits below (and never after
	       them), so they overlay their data on top of these defaults.
	       This commit code should allow for the object specific code
	       to have overlayed data after the code above has executed.
	      */
	    vp->viewMaskLen = rp->viewMaskLen;
	    bcopy(rp->viewMask, vp->viewMask, rp->viewMaskLen);
	    vp->viewStatus = rp->viewStatus;
	    vp->viewBitMask = rp->viewBitMask;
	}
    } else if (action == FREE){
	if (vp && vp->viewStatus == VIEWNONEXISTENT){
	    view_rowDelete(viewParty, viewPartyLen,
			   viewSubtree, viewSubtreeLen);
	    vp = rp = NULL;
	}
	if (vp)	/* satisfy postcondition for bitMask */
	    rp->viewBitMask = vp->viewBitMask;
    }

/* XXX !!! check return values from the asn_parse_* routines */
    switch(var){
      case VIEWMASK:
        if (action == RESERVE1){
            if (var_val_type != ASN_OCTET_STR)
                return SMP_ERR_WRONGTYPE;
            size = sizeof(rp->viewMask);
            asn_parse_string(var_val, &bigsize, &var_val_type,
                             rp->viewMask, &size);
            rp->viewMaskLen = size;
            if (size > 16)
                return SMP_ERR_WRONGVALUE;
            rp->viewBitMask |= VIEWMASK_MASK;
        } else if (action == COMMIT){
            vp->viewMaskLen = rp->viewMaskLen;
            bcopy(rp->viewMask, vp->viewMask, vp->viewMaskLen);
        }
	break;
      case VIEWSTATUS:
	if (action == RESERVE1){
	    if (var_val_type != ASN_INTEGER)
		return SMP_ERR_WRONGTYPE;
	    asn_parse_int(var_val, &bigsize, &var_val_type, &val, sizeof(val));
	    if (val < 1 || val > 3)
		return SMP_ERR_WRONGVALUE;
	    rp->viewStatus = val;
	    rp->viewBitMask |= VIEWSTATUS_MASK;
	} else if (action == RESERVE2){
	    if ((rp->viewStatus == VIEWINCLUDED
		 || rp->viewStatus == VIEWEXCLUDED)
		&& (rp->viewBitMask != VIEWCOMPLETE_MASK))
		return SMP_ERR_INCONSISTENTVALUE;
	    /* tried to set incomplete row valid */
	} else if (action == COMMIT){
	    vp->viewStatus = rp->viewStatus;
	} else if (action == ACTION && vp->viewStatus == VIEWINVALID){
		view_rowDelete(vp->viewParty, vp->viewPartyLen,
			      vp->viewSubtree, vp->viewSubtreeLen);
	}
	break;
      case VIEWPARTY:
      case VIEWSUBTREE:
      default:
	    return SMP_ERR_NOCREATION;
    }
    if (action == COMMIT)	/* make any new columns avpear */
	vp->viewBitMask = rp->viewBitMask;

    return TRUE;
}

u_char *
var_view(vp, name, length, exact, var_len, write_method)
    register struct variable *vp;   /* IN - pointer to variable entry that points here */
    register oid *name;      /* IN/OUT - input name requested, output name found */
    register int *length;    /* IN/OUT - length of input and output oid's */
    int          exact;      /* IN - TRUE if an exact match was requested. */
    int          *var_len;   /* OUT - length of variable or 0 if function returned. */
    int          (**write_method)(); /* OUT - pointer to function to set variable, otherwise 0 */
{
    oid newname[MAX_NAME_LEN], lowname[MAX_NAME_LEN], *np;
    int newnamelen, lownamelen;
    struct viewEntry *vwp, *lowvwp = NULL;
    u_long mask;
    oid *viewParty, *viewSubtree;
    int viewPartyLen, viewSubtreeLen;
/*
 * This routine handles requests for variables of the form:
 * ...snmpSecrets.partyViews.viewTable.viewEntry.X.oidlen.oid.oidlen.oid
 * or .1.3.6.1.2.1.21.3.1.1.X.oidlen.oid.oidlen.oid, where the oid suffixes are
 * variable length.
 * Therefore, the length of the first index is name[11] and the index starts
 * at name[12].  The length of the second index starts at name[12 + name[11]],
 * and the second index starts at name[13 + name[11]].
 */
    mask = 1 << (vp->magic - 1);
    bcopy((char *)vp->name, (char *)newname, (int)vp->namelen * sizeof(oid));
    if (exact){
        if (*length < 16 ||
	    bcmp((char *)name, (char *)vp->name, 11 * sizeof(oid)))
	    return NULL;
	viewPartyLen = name[11];
	viewParty = name + 12;
	if (*length <= 12 + viewPartyLen)
	    return NULL;
	viewSubtreeLen = name[12 + viewPartyLen];
	viewSubtree = name + 13 + viewPartyLen;
	if (*length != 13 + viewPartyLen + viewSubtreeLen)
	    return NULL;
    	*write_method = write_view;
        vwp = view_getEntry(viewParty, viewPartyLen,
			   viewSubtree, viewSubtreeLen);
	if (vwp == NULL)
	    return NULL;
	if (!(vwp->viewBitMask & mask))
	    return NULL;
    } else {
      /* find "next" control entry */
      view_scanInit();
      for(vwp = view_scanNext(); vwp; vwp = view_scanNext()){
	if (!(vwp->viewBitMask & mask))
	    continue;
	np = newname + 11;
	*np++ = (oid)vwp->viewPartyLen;
	bcopy((char *)vwp->viewParty, (char *)np,
	      vwp->viewPartyLen * sizeof(oid));
	np += vwp->viewPartyLen;
	*np++ = (oid)vwp->viewSubtreeLen;
	bcopy((char *)vwp->viewSubtree, (char *)np,
	      vwp->viewSubtreeLen * sizeof(oid));
	newnamelen = 13 + vwp->viewPartyLen + vwp->viewSubtreeLen;
	if ((compare(newname, newnamelen, name, *length) > 0) &&
	    (!lowvwp || compare(newname, newnamelen,
			       lowname, lownamelen) < 0)){
	    /*
	     * if new one is greater than input and closer to input than
	     * previous lowest, save this one as the "next" one.
	     */
	    bcopy((char *)newname, (char *)lowname, newnamelen * sizeof(oid));
	    lownamelen = newnamelen;
	    lowvwp = vwp;
	}
      }
      if (lowvwp == NULL)
	  return NULL;
      vwp = lowvwp;
      bcopy((char *)lowname, (char *)name, lownamelen * sizeof(oid));
      *length = lownamelen;
    }

    *var_len = sizeof(long);
    long_return = 0;

    switch (vp->magic){
      case VIEWPARTY:
	*var_len = vwp->viewPartyLen * sizeof(oid);
	return (u_char *)vwp->viewParty;
      case VIEWSUBTREE:
	*var_len = vwp->viewSubtreeLen * sizeof(oid);
	return (u_char *)vwp->viewSubtree;
      case VIEWMASK:
	*var_len = vwp->viewMaskLen;
	return (u_char *)vwp->viewMask;
      case VIEWSTATUS:
	return (u_char *)&vwp->viewStatus;
      default:
            ERROR("");
    }
    return NULL;
}

in_view(name, namelen, dstParty, dstPartyLength)
    oid *name, *dstParty;
    int namelen, dstPartyLength;
{
    struct viewEntry *vwp, *savedvwp = NULL;

    view_scanInit();
    for(vwp = view_scanNext(); vwp; vwp = view_scanNext()){
	if (vwp->viewStatus == VIEWINVALID)
	    continue;
	if (vwp->viewPartyLen != dstPartyLength
	    || bcmp(vwp->viewParty, dstParty, dstPartyLength * sizeof(oid)))
	    continue;
	if (vwp->viewSubtreeLen > namelen
	    || bcmp(vwp->viewSubtree, name, vwp->viewSubtreeLen * sizeof(oid)))
	    continue;
	/* no wildcards here yet */
	if (!savedvwp){
	    savedvwp = vwp;
	} else {
	    if (vwp->viewSubtreeLen > savedvwp->viewSubtreeLen)
		savedvwp = vwp;
/*
	    else if (vwp->viewSubtreeLen == savedvwp->viewSubtreeLen
		     && greater(vwp->viewSubtree, savedvwp->viewSubtree))
		savedvwp = vwp;
 */
		
	}
    }
    if (!savedvwp)
	return FALSE;
    if (savedvwp->viewStatus == VIEWINCLUDED)
	return TRUE;
    return FALSE;
}
