/**** X interface for network map program.  ****/

/*	Copyright (C) 1989 Massachusetts Institute of Technology
 *		Right to copy granted under the terms described
 *		in the file Copyright accompanying this distribution.
 */


/* This is the X interface for the program that displays network connectivity.
 * The data presently is more or less fixed by values in certain data files.
 * Eventually I hope to be able to do this more dynamically.
 */

#include "map.h"

#include <X/Xlib.h>
#include <X/Xkeyboard.h>

#include <X/cursors/target.cursor>



#ifdef EDIT_HISTORY

Nbr Date	Author		Description
--- ----	------		-----------
 3  20-Feb-89	M. A. Patton	First release version.
 3A  4-Mar-89	M. A. Patton	No longer precompute Vertex list, computed on
				the fly while displaying.  Finally got all X
				stuff in here, moved X includes here from map.h
 4  20-Mar-89	M. A. Patton	Added pxl2rgb to remove X from TEK module.
				(From report by P. V. O'Neill)
 4A 23-Mar-89	M. A. Patton	Added waiting flag so redisplay in middle of
				things gets correct bottom line.
 5  12-Jun-90	M. A. Patton	Added getOption (for SNMP).

#endif
/****	Internal Documentation and External calling sequences	****/


#ifdef DOCUMENTATION

General conventions:
    subroutine and function names:
	Place...	Does recalculations of X internal (i.e. display
				position) parts of the data structure.
	Display...	Actually draws something on the screen, does nothing
				else.
	Erase...	Draws something in the background color, may erase
				parts of other things.
	ReDraw...	Calls appropriate Erase... and Display... routines for
				the given thing and then calls ReDraw... of
				other things that might be affected.

    !!!! Needs More !!!!

#endif
/****	Parameters used by this module.	****/


int border_width = 3;		/* The width of the border */
char *display_name = NULL;	/* The name of the display */
char *border_color = "Black";	/* The color for the border */
char *back_color = "White";	/* The background color */
char *fore_color = "Black";	/* The text color */
char *highlight_color = "LightBlue"; /* Indicate the selected host */
char *font_name = "6x10";	/* The font to use */
char *geometry = NULL;		/* Pointer to geometry string for window */


/* A list of colors to use for defaulting.  */
static char *default_color[] = {
    "Blue",
    "Yellow",
    "Red",
    "Orange",
    "LightGray",
    "DimGray",
    "Khaki",
    "Goldenrod",
    NULL			/* Terminator */
};
static int default_color_index = 0;	/* Which one we're on */




/* Global information, used during operation. */
int display_height;
int display_width;
int foreground;
int background;
int highlight;
int IP_pixel, CHAOS_pixel;
OpaqueFrame frame;		/* Frame for original creation (and info) */
#define window (frame.self)
FontInfo *font_info;
#define font (font_info->id)


/* Random state variables */
static int need_redisplay = FALSE;
static waiting = TRUE;

static char *prompt = "Command:";
static char *waitmsg = "Wait ...";
static char input[INPUT_LEN];
static int input_index = 0;
/**** Initialization ****/




caddr_t
InitDisplay()
{   Display *dpy;

    dpy = XOpenDisplay(display_name);
    if (dpy!=NULL)
    {	if (!(font_info = XOpenFont(font_name)))
	    error("XOpenFont failed");
	OpenWindow();
	DisplayWait();
    }
    return((caddr_t)dpy);
}




GetDefaults()
{   char *option;

    if ((option = XGetDefault(Argv[0],"BorderWidth")) !=NULL)
	border_width = atoi(option);
    if ((option = XGetDefault(Argv[0],"Border")) !=NULL)
	border_color = option;
    if ((option = XGetDefault(Argv[0],"BorderColor")) !=NULL)
	border_color = option;
    if ((option = XGetDefault(Argv[0],"Background")) !=NULL)
	back_color = option;
    if ((option = XGetDefault(Argv[0],"Foreground")) !=NULL)
	fore_color = option;
    if ((option = XGetDefault(Argv[0],"Highlight")) !=NULL)
	highlight_color = option;
    if ((option = XGetDefault(Argv[0],"Font")) !=NULL)
	font_name = option;
}

/* Get one option (for SNMP configuration, eventually better?) */
char *
getOption(optname)
char *optname;
{   return (XGetDefault (Argv[0], optname));
}
/**** Set up the window ****/



OpenWindow()
{   frame.bdrwidth = border_width;
    /* Deal with colors */
    /* Background */
    background = InterpretColor(back_color,WhitePixel);
    frame.background = XMakeTile(background);
    /* Foreground */
    foreground = InterpretColor(fore_color,BlackPixel);
    /* Border */
    frame.border = XMakeTile(InterpretColor(border_color,foreground));
    /* Highlight (selected host) */
    highlight = InterpretColor(highlight_color,background);
    /* IP and CHAOS protocol colors (need to save for later use) */
    IP_pixel = InterpretColor("IP",foreground);
    CHAOS_pixel = InterpretColor("CHAOS",foreground);
    if (!XCreate("Net Display",Argv[0],geometry,"=668x529",&frame,20,20))
	error("XCreate failed");
    XSelectInput(window,ButtonPressed|ExposeWindow|KeyPressed);
    /* Make the cursor */
    XDefineCursor(window,XCreateCursor(target_width,target_height,
				       target_bits,target_bits,
				       target_x_hot,target_y_hot,
				       foreground,background,GXcopy));
    XMapWindow(window);
}
/**** Display management ****/


ForceRedisplay()
{   need_redisplay = TRUE;
    XClear(window);
}


ForceDisplay()
{   /* Should be more integrated and handle some kinds of events that	*/
    /* might be in the queue.						*/
    if (need_redisplay) Redisplay();
    XFlush();
}




Redisplay()
{   register Network *np;
    register Machine *hp;
    register Title *tp;
    WindowInfo window_info;

    need_redisplay = FALSE;
    if (!XQueryWindow(window,&window_info))
	error("Can't get window info");
    display_height = window_info.height;
    display_width = window_info.width;
    MAPCAR (np,net_base)
	DisplayNetwork(np);
    MAPCAR (hp,machine_base)
	DisplayHost(hp->host);
    MAPCAR (tp,title_base)
	DisplayTitle(tp);
    if (waiting)
    {	DisplayWait();
    }
    else
    {	DisplayPrompt();
	DisplayInput();
    }
}



Display_Message(cp)
char *cp;
{   fputs(cp, stdout);
}
/****	Compute placement for a Host entry	****/


PlaceHost(p)
register Host *p;
{   register int w, h;
    register Machine *gp;

    if ((gp=p->machine) == NULL)
    {	gp = (Machine *)malloc(sizeof(Machine));
	if (gp==NULL)
	{   XFeep(7);
	    fprintf(stderr,"Can't allocate!!\n");
	    return;
	}
	p->machine = gp;
	gp->next = machine_base;
	machine_base = gp;
	gp->name = p->preferred_name;
	if (gp->name == NULL)
	    gp->name = p->names->name;
	gp->host = p;
    }
    /* Default to natural size if width or height not given */
    if (WidthValue&p->geo_flag)
	w = p->width;
    else
	w = XStringWidth(gp->name,font_info,0,0) + 4;
    if (HeightValue&p->geo_flag)
	h = p->height;
    else
	h = font_info->height + 4;
    p->x1 = p->x_pos - (w/2);
    p->y1 = p->y_pos - (h/2);
    p->x2 = p->x1 + w;
    p->y2 = p->y1 + h;
    PlaceTaps(p);
    p->geo_flag |= AlreadyPlaced;
}
/****	Calculate placement of Network	****/

PlaceNetwork(p)
register Network *p;
{   p->vertical = (p->height > p->width);
    p->x1 = p->x_pos - (p->width/2);
    p->y1 = p->y_pos - (p->height/2);
    p->x2 = p->x1 + p->width;
    p->y2 = p->y1 + p->height;
    /* Set up the bounding box for the net itself. */
    if (p->vertical)
    {	register int phase = 1;		/* 1=Draw on left, 0=Draw on right */
	register AddressList *ap;

	p->side1 = p->x1;
	p->side2 = p->x2;
	MAPCAR (ap,p->addresses)
	{   if (phase)
	    {	phase = 0;
		p->x1 -= 3;
	    }
	    else
	    {	phase = 1;
		p->x2 += 3;
	    }
	}
	p->label_x = p->x2+2;
	p->label_y = (p->y1+p->y2-font_info->height)/2;
    }
    else
    {	register int phase = 1;		/* 1=Draw on top, 0=Draw on bottom */
	register AddressList *ap;

	p->side1 = p->y1;
	p->side2 = p->y2;
	MAPCAR (ap,p->addresses)
	{   if (phase)
	    {	phase = 0;
		p->y1 -= 3;
	    }
	    else
	    {	phase = 1;
		p->y2 += 3;
	    }
	}
	p->label_x = (p->x1+p->x2-XStringWidth(p->name,font_info,0,0))/2;
	p->label_y = p->y1-font_info->height;
    }
}
/****	Compute placement for a Title entry	****/


PlaceTitle(p)
register Title *p;
{   register int w, h;

    /* Default to natural size if width or height not given */
    if (WidthValue&p->geo_flag)
	w = p->width;
    else
	w = XStringWidth(p->text,font_info,0,0) + 4;
    if (HeightValue&p->geo_flag)
	h = p->height;
    else
	h = font_info->height + 4;
    p->x1 = p->x_pos - (w/2);
    p->y1 = p->y_pos - (h/2);
    p->x2 = p->x1 + w;
    p->y2 = p->y1 + h;
    p->geo_flag |= AlreadyPlaced;
}
/****	Compute placement of the taps from a Host to the Networks	****/



PlaceTaps(p)
register Host *p;
{   register AddressList *ap;
    Network *FindNetwork();

    MAPCAR (ap,p->addresses)
    {   if (ap->net_p == NULL)
	    ap->net_p = FindNetwork(ap->address);
    }
}
/****	Draw a host	****/


DisplayHost(hp)
Host *hp;
{   if (! (hp->geo_flag&AlreadyPlaced))
	PlaceHost(hp);
    UpdateHost(hp, foreground);
}


EraseHost(hp)
Host *hp;
{   UpdateHost(hp, background);
}



static
UpdateHost(hp, pix)
Host *hp;
int pix;
{   register int drawn = FALSE;
    register AddressList *ap;
    Vertex v[5];

    MAPCAR (ap,hp->addresses)
    {	if ( (show_chaos && ap->address.class==AF_CHAOS) ||
	     (show_IP && ap->address.class==AF_INET) )
	{   if (!drawn)
	    {	v[0].flags = v[1].flags = v[2].flags =
				v[3].flags = v[4].flags = 0;
		v[0].x = v[3].x = v[4].x = hp->x1;
		v[1].x = v[2].x = hp->x2;
		v[0].y = v[1].y = v[4].y = hp->y1;
		v[2].y = v[3].y = hp->y2;
		XDrawFilled(window,v,5,
				((hp == selected_host))?highlight:background,
				GXcopy,AllPlanes);
		XDraw(window,v,5,1,1,pix,GXcopy,AllPlanes);
		XTextMask(window,hp->x1+2,hp->y1+2,
			  hp->machine->name,strlen(hp->machine->name),
			  font,pix);
		drawn = TRUE;
	    }
	    if (ap->net_p != NULL)
	    {   Network *np = ap->net_p;
		v[0].flags = v[3].flags = 0;
		v[1].flags = v[4].flags = VertexDrawLastPoint;
		v[2].flags = VertexDontDraw;
		if (np->vertical)
		{   v[0].y = v[1].y = v[3].y = hp->y_pos;
		    v[2].y = hp->y_pos - 4;
		    v[4].y = hp->y_pos + 4;
		    if (np->x_pos < hp->x1)
		    {	v[0].x = hp->x1;
			v[3].x = np->side2 + 8;
			v[1].x = v[2].x = v[4].x = np->side2;
		    }
		    else
		    {	v[0].x = hp->x2;
			v[3].x = np->side1 - 8;
			v[1].x = v[2].x = v[4].x = np->side1;
		    }
		}
		else
		{   v[0].x = v[1].x = v[3].x = hp->x_pos;
		    v[2].x = hp->x_pos - 4;
		    v[4].x = hp->x_pos + 4;
		    if (np->y_pos < hp->y1)
		    {   v[0].y = hp->y1;
			v[3].y = np->side2 + 8;
			v[1].y = v[2].y = v[4].y = np->side2;
		    }
		    else
		    {   v[0].y = hp->y2;
			v[3].y = np->side1 - 8;
			v[1].y = v[2].y = v[4].y = np->side1;
		    }
		}
		XDraw(window,v,5,1,1,pix,GXcopy,AllPlanes);
	    }
	}
    }
}


ReDrawHost(hp)
Host *hp;
{   if (hp->geo_flag&AlreadyPlaced)
	EraseHost(hp);
    DisplayHost(hp);
}
/****	Draw a Title	****/


DisplayTitle(tp)
Title *tp;
{   if (! (tp->geo_flag&AlreadyPlaced))
	PlaceTitle(tp);
    UpdateTitle(tp, foreground);
}


EraseTitle(tp)
Title *tp;
{   UpdateTitle(tp, background);
}



static
UpdateTitle(tp, pix)
Title *tp;
int pix;
{   XTextMask(window,tp->x1+2,tp->y1+2,tp->text,strlen(tp->text),font,pix);
}


ReDrawTitle(tp)
Title *tp;
{   if (tp->geo_flag&AlreadyPlaced)
	EraseTitle(tp);
    DisplayTitle(tp);
}
/****	Draw a network	****/


DisplayNetwork(np)
register Network *np;
{   if (! (np->geo_flag&AlreadyPlaced))
	PlaceNetwork(np);
    UpdateNetwork(np, foreground);
}
    

EraseNetwork(np)
register Network *np;
{   UpdateNetwork(np, background);
}
    
static
UpdateNetwork(np, pix)
register Network *np;
int pix;
{   register int draw = FALSE;
    register AddressList *ap;
    int a, b, t;

    a = np->side1;
    b = np->side2;
    t = 3;
    MAPCAR (ap,np->addresses)
    {   if ( (show_chaos && ap->address.class==AF_CHAOS) ||
	     (show_IP && ap->address.class==AF_INET) )
	{   if (np->vertical)
		X_Rect(a-t,np->y1,a,np->y2,
			(ap->address.class==AF_INET)?IP_pixel:CHAOS_pixel);
	    else
		X_Rect(np->x1,a-t,np->x2,a,
			(ap->address.class==AF_INET)?IP_pixel:CHAOS_pixel);
	    {int tmp=a-t;a=b;b=tmp;}
	    t = -t;
	    draw = TRUE;
	}
    }
    if (draw||np->addresses==NULL)
    {   if (np->vertical)
	    X_Rect(np->side1,np->y1,np->side2,np->y2,np->media->pixel);
	else
	    X_Rect(np->x1,np->side1,np->x2,np->side2,np->media->pixel);
	XTextMask(window,np->label_x,np->label_y,
		  np->name,strlen(np->name),font,pix);
    }
}


static
X_Rect(x1,y1,x2,y2,pix)
int x1,y1,x2,y2;
int pix;
{   Vertex v[5];

    v[0].flags = v[1].flags = v[2].flags = v[3].flags = v[4].flags = 0;
    v[0].x = v[3].x = v[4].x = x1;
    v[1].x = v[2].x = x2;
    v[0].y = v[1].y = v[4].y = y1;
    v[2].y = v[3].y = y2;
    XDrawFilled(window,v,5,pix,GXcopy,AllPlanes);
}



ReDrawNetwork(np)
Network *np;
{   if (np->geo_flag&AlreadyPlaced)
	EraseNetwork(np);
    DisplayNetwork(np);
    /* !!!!>>>>	Should also find hosts that might need redrawing <<<<!!!! */
}
/****	Maintenance of interaction area	****/


static
DisplayWait()
{   waiting = TRUE;
    XPixSet(window,0,display_height-(font_info->height+2),
		   display_width,font_info->height+2,background);
    XTextMask(window,5,display_height-(font_info->height+2),
		     waitmsg,strlen(waitmsg),font,foreground);
    XFlush();			/* Anticipation!! */
}



static
DisplayPrompt()
{   waiting = FALSE;
    XPixSet(window,0,display_height-(font_info->height+2),
		   display_width,font_info->height+2,background);
    XTextMask(window,5,display_height-(font_info->height+2),
		     prompt,strlen(prompt),font,foreground);
}




static
DisplayInput()
{   XPixSet(window,XStringWidth(prompt,font_info,0,0)+9,
		   display_height-(font_info->height+2),
		   display_width,font_info->height+2,background);
    if (input_index != 0)
	XTextMask(window,XStringWidth(prompt,font_info,0,0)+10,
			 display_height-(font_info->height+2),
			 input,input_index,font,foreground);
}
/****	External representations.	****/


int
InterpretColor(cp,dflt)
char *cp;			/* Name for the color */
int dflt;			/* Default value if not available */
{   Color color;

    if (!XParseColor(cp,&color)) /* That name isn't a color */
    {   /* Try seeing if it's named in Xdefaults */
	if ((cp = XGetDefault(Argv[0],cp)) == NULL)
	{   /* Oh, well just default it. */
	    if ((cp = default_color[default_color_index++]) == NULL)
	    {	default_color_index--;
		return(dflt);
	    }
	}
	if (!XParseColor(cp,&color)) /* Not a color we can figure out */
	    return(dflt);
    }
    if (!XGetHardwareColor(&color))
	return(dflt);		/* This display can't do that */
    return(color.pixel);
}



fprintgeo(fd,flag,width,height,x_pos,y_pos)
FILE *fd;
int flag,width,height,x_pos,y_pos;
{   if (WidthValue&flag)
	fprintf(fd,"%d",width);
    if (HeightValue&flag)
	fprintf(fd,"x%d",height);
    fprintf(fd,"%c%d%c%d",(XNegative&flag)?'-':'+',abs(x_pos),
			  (YNegative&flag)?'-':'+',abs(y_pos));
}



int
pxl2rgb(p,r,g,b)
int p, *r, *g, *b;
{   Color color;

    color.pixel = p;
    if (!XQueryColor(&color))
	return (FALSE);
    *r = color.red;
    *g = color.green;
    *b = color.blue;
    return(TRUE);
}
/****	Background Loop for display interaction	****/


Display_BackgroundLoop()
{   XEvent event;

    DisplayPrompt();
    FOREVER
    {	if (need_redisplay && !XPending())
	    Redisplay();
	XNextEvent(&event);
	switch(event.type)
	{
	case ButtonPressed:
	    HandleMouseButton(&event);
	    break;
	case ExposeWindow:
	    need_redisplay = TRUE;
	    break;
	case KeyPressed:
	    HandleKey(&event);
	    break;
	default:
	    error("Unknown event type");
	}
    }
}
/**** input handling ****/



static
HandleMouseButton(event_p)
XButtonEvent *event_p;
{   DisplayWait();
    switch (event_p->detail&0xFF)
    {
    case RightButton:
	XFeep(0);
	break;

    case LeftButton:
	Describe(event_p->x,event_p->y,event_p->detail&0xFF00);
	break;

    case MiddleButton:
	Adjust(event_p->x,event_p->y,event_p->detail&0xFF00);
	break;

    default:
	fprintf(stderr,"Don't get it: MouseButton (detail=0x%X).\n",
		event_p->detail);
	break;
    }
    DisplayPrompt();
    DisplayInput();
}
/* Describe(x,y,flags)
 *
 * Locate an object in the data base given display coordinates and describe it.
 */
static
Describe(x,y,flags)
int x, y;
int flags;
{   Machine *gp;
    Network *np;
    Machine *LocateMachine();
    Network *LocateNetwork();

    flags = ( (flags&ShiftMask)?MOD_VERBOSE:0 )
	  | ( (flags&ControlMask)?MOD_GEOMETRY:0 )  ;
    DeselectHost();
    printf("----------------------------------\n");
    if ( (gp=LocateMachine(x,y)) != NULL)
    {   SelectHost(gp->host);
	DescribeMachine(gp,flags);
    }
    else if ( (np=LocateNetwork(x,y)) != NULL)
    {	DescribeNetwork(np,flags);
    }
    else
    {	fprintf(stderr,"Can't figure out what you want me to describe.\n");
	XFeep(0);
    }
}

static Machine *
LocateMachine(x,y)
int x, y;
{   Machine *gp;

    MAPCAR (gp,machine_base)
    {	Host *hp = gp->host;

	if ((x >= hp->x1) && (x <= hp->x2) && (y >= hp->y1) && (y <= hp->y2))
	    return (gp);
    }
    return (NULL);
}

static Network *
LocateNetwork(x,y)
int x, y;
{   Network *np;

    MAPCAR (np,net_base)
    {	if ((x >= np->x1) && (x <= np->x2) && (y >= np->y1) && (y <= np->y2))
	    return (np);
    }
    return(NULL);
}
/**** Screen modification ****/


static
Adjust(x,y,flags)
int x, y;
int flags;
{   char *cp;
    Machine *gp, *LocateMachine();
    Network *np, *LocateNetwork();

    switch (flags)
    {
    default:
	XFeep(0);
	return;
    case 0:
	if (selected_host==NULL)
	{   XFeep(0);
	    fprintf(stderr,"No host to place.\n");
	    return;
	}
	AddHost(x,y,selected_host);
	return;
    case ShiftMask:
	DeselectHost();
	if ( (gp=LocateMachine(x,y)) != NULL)
	{   DeleteMachine(gp);
	    return;
	}
	if ( (np=LocateNetwork(x,y)) != NULL)
	{   DeleteNetwork(np);
	    return;
	}
	printf("Can't figure out what you want me to delete.\n");
    }
}


static
AddHost(x,y,hp)
int x, y;
Host *hp;
{   char *cp;
    Machine *gp;

    if ((gp=hp->machine) == NULL)
    {	gp = (Machine *)malloc(sizeof(Machine));
	if (gp==NULL)
	{   XFeep(7);
	    fprintf(stderr,"Can't allocate!!\n");
	    return;
	}
	hp->machine = gp;
	gp->next = machine_base;
	machine_base = gp;
	gp->name = hp->preferred_name;
	if (gp->name == NULL)
	    gp->name = hp->names->name;
	gp->host = hp;
    }
    else
    {	XClear(window);
	need_redisplay = TRUE;
    }
    hp->geo_flag = XValue | YValue;
    hp->x_pos = x;
    hp->y_pos = y;
    PlaceHost(hp);
    if (!need_redisplay)
	DisplayHost(hp);
}


int
DeleteMachine(gp)
Machine *gp;
{   Machine *p, *op;

    /* First entry is a special case */
    if ( (p=machine_base) == NULL) return (FALSE);
    if (gp==p)
    {	machine_base = gp->next;
	if (gp->host != NULL)
	    gp->host->machine = NULL;
	/* FreeMachine(gp); */
	XClear(window);
	need_redisplay = TRUE;
	return (TRUE);
    }
    for (op=p,p=op->next;p!=NULL;op=p,p=op->next)
    {	if (gp==p)
	{   op->next = gp->next;
	if (gp->host != NULL)
	    gp->host->machine = NULL;
	    /* FreeMachine(gp); */
	    XClear(window);
	    need_redisplay = TRUE;
	    return (TRUE);
	}
    }
    return (FALSE);
}


DeleteNetwork(np)
Network *np;
{   Network *p, *op;

    /* First entry is a special case */
    if ( (p=net_base) == NULL) return;
    if (np==p)
    {	net_base = np->next;
	/* FreeNetwork(np); */
	XClear(window);
	need_redisplay = TRUE;
	return;
    }
    for (op=p,p=op->next;p!=NULL;op=p,p=op->next)
    {	if (np==p)
	{   op->next = np->next;
	    /* FreeNetwork(np); */
	    XClear(window);
	    need_redisplay = TRUE;
	    return;
	}
    }
}
/**** Keyboard input, buffer it up. ****/



static
HandleKey(event_p)
XKeyEvent *event_p;
{   char *cp;
    int n = 0;

    if (IsShiftKey((event_p->detail)&0xFF))
	return;
    if (IsTypewriterKey((event_p->detail)&0xFF))
    {	cp = XLookupMapping(event_p,&n);
	if (n != 1)
	{   fprintf(stderr,
     "TypeWriter key not 1 char!\nKeycode %X mapped to \"%s\" (%d chars).\n",
		    event_p->detail,cp,n);
	    return;
	}
	HandleChar(*cp);
	return;
    }
    switch ((event_p->detail)&0xFF)
    {
    case KC_F15:		/* Help key on LK201 keyboard */
	DisplayWait();
	if (input_index==0)
	    HelpOn("");
	else
	{   input[input_index] = '\0';
	    HelpOn(input);
	}
	DisplayPrompt();
	DisplayInput();
	return;
    default:
	XFeep(0);
	return;
    }
}
/****	Handle normal characters	****/



static
HandleChar(c)
char c;
{   if (c < ' ') switch (c)
    {
    case '\n':
    case '\r':
	if (input_index != 0)
	    input[input_index] = '\0';
	DisplayWait();
	HandleInput(input);
	input_index = 0;
	DisplayPrompt();
	DisplayInput();
	return;
    case 'l'&037:
    case 'w'&037:
	ForceRedisplay();
	return;
    default:
	XFeep(0);
	return;
    }
    if (c == 0177)
    {   if (--input_index >= 0)
	{   DisplayInput();
	    return;
	}
	input_index = 0;
	DisplayInput();
	XFeep(0);
	return;
    }
    if (input_index >= INPUT_LEN)
    {	XFeep(0);
	return;
    }
    input[input_index++] = c;
    DisplayInput();
    return;
}
