/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMSCAN.C - scanner (lexical analyser) for MIB Compiler
 *
 * $Revision:   1.3  $ $Date:   08 Jul 1992 17:35:54  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMSCAN.C_V  $
 * 
 *    Rev 1.3   08 Jul 1992 17:35:54   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.2   08 Jul 1992 16:43:54   gfoster
 * Added keyword "IMPLIED".
 * 
 *    Rev 1.1   19 Jun 1992 16:19:54   gfoster
 * Copyright text was reformated.
 * 
 * Added support for environment variable for include file
 * directory.
 * 
 * Made token types for all characters needed in parser.
 * 
 * Added checks for badly formed "::=" and "..".
 * 
 *    Rev 1.0   27 May 1992 16:05:36   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <string.h>
#include <ctype.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"

#include "smtoks.h"


USHORT NEAR usInclLevel;            /* current level of include files */
FILE * NEAR afhIn[MXLV];            /* file handle stack */
PSZ NEAR apszInFile[MXLV];          /* file name stack */
USHORT NEAR ausLineNo[MXLV];        /* line number stack */
USHORT NEAR ausColNo[MXLV];         /* column position stack */
INT NEAR aLexChar[MXLV];            /* input char stack */


typedef struct _KeyWordName { /* keyword definition */
    PSZ pszName;                /* keyword name */
    SHORT sType;                /* value for parser */
    USHORT usStatus;            /* status (see KWSTxxxx) */
    struct _KeyWordName *pHashNext; /* next in hash chain */
    USHORT cUse;                /* use count */
} KEYWORDNAME;

/* status of keyword */
#define KWSTdef  0x8000         /* flag that name is defined */
#define KWSTbase 0x0001         /* part of ASN.1 */
#define KWSTsmi  0x0002         /* defined in SMI */
#define KWSTot   0x0004         /* OBJECT-TYPE macro */
#define KWSTotm  0x0008         /* defined in OBJECT-TYPE macro */
#define KWSTtt   0x0010         /* TRAP-TYPE macro */
#define KWSTttm  0x0020         /* defined with TRAP-TYPE macro */

/* note: this table is updated by the program */
KEYWORDNAME NEAR keyWordName[] = {
    {"ACCESS",          kwACCESS,       KWSTbase|KWSTdef,   NULL},
    {"BEGIN",           kwBEGIN,        KWSTbase|KWSTdef,   NULL},
    {"Counter",         kwCOUNTER,      KWSTsmi,            NULL},
    {"DEFINITIONS",     kwDEFINITIONS,  KWSTbase|KWSTdef,   NULL},
    {"DEFVAL",          kwDEFVAL,       KWSTotm,            NULL},
    {"DESCRIPTION",     kwDESCRIPTION,  KWSTotm|KWSTttm,    NULL},
    {"END",             kwEND,          KWSTbase|KWSTdef,   NULL},
    {"ENTERPRISE",      kwENTERPRISE,   KWSTttm,            NULL},
    {"FROM",            kwFROM,         KWSTbase|KWSTdef,   NULL},
    {"Gauge",           kwGAUGE,        KWSTsmi,            NULL},
    {"IDENTIFIER",      kwIDENTIFIER,   KWSTbase|KWSTdef,   NULL},
    {"IMPLIED",         kwIMPLIED,      KWSTotm,            NULL},
    {"IMPORTS",         kwIMPORTS,      KWSTbase|KWSTdef,   NULL},
    {"INDEX",           kwINDEX,        KWSTotm,            NULL},
    {"INTEGER",         kwINTEGER,      KWSTbase|KWSTdef,   NULL},
    {"IpAddress",       kwIPADDRESS,    KWSTsmi,            NULL},
    {"NOLENGTH",        kwNOLENGTH,     KWSTotm,            NULL},
    {"NULL",            kwNULL,         KWSTbase|KWSTdef,   NULL},
    {"NetworkAddress",  kwNETWORKADDRESS,KWSTsmi,           NULL},
    {"OBJECT",          kwOBJECT,       KWSTbase|KWSTdef,   NULL},
    {"OBJECT-TYPE",     kwOBJECT_TYPE,  KWSTot,             NULL},
    {"OCTET",           kwOCTET,        KWSTbase|KWSTdef,   NULL},
    {"OF",              kwOF,           KWSTbase|KWSTdef,   NULL},    
    {"Opaque",          kwOPAQUE,       KWSTsmi,            NULL},
    {"REFERENCE",       kwREFERENCE,    KWSTotm|KWSTttm,    NULL},
    {"SEQUENCE",        kwSEQUENCE,     KWSTbase|KWSTdef,   NULL},
    {"SIZE",            kwSIZE,         KWSTbase|KWSTdef,   NULL},
    {"SMI",             kwSMI,          KWSTbase|KWSTdef,   NULL},
    {"STATUS",          kwSTATUS,       KWSTotm,            NULL},
    {"STRING",          kwSTRING,       KWSTbase|KWSTdef,   NULL},
    {"SYNTAX",          kwSYNTAX,       KWSTotm,            NULL},
    {"TRAP-TYPE",       kwTRAP_TYPE,    KWSTtt,             NULL},
    {"TimeTicks",       kwTIMETICKS,    KWSTsmi,            NULL},
    {"VARIABLES",       kwVARIABLES,    KWSTttm,            NULL},
    {"deprecated",      kwDEPRECATED,   KWSTotm,            NULL},
    {"mandatory",       kwMANDATORY,    KWSTotm,            NULL},
    {"not-accessible",  kwNOT_ACCESSIBLE,KWSTotm,           NULL},
    {"obsolete",        kwOBSOLETE,     KWSTotm,            NULL},
    {"optional",        kwOPTIONAL,     KWSTotm,            NULL},
    {"read-only",       kwREAD_ONLY,    KWSTotm,            NULL},
    {"read-write",      kwREAD_WRITE,   KWSTotm,            NULL},
    {"write-only",      kwWRITE_ONLY,   KWSTotm,            NULL}
};

#define SZHASH 143 /* hash table size */

KEYWORDNAME *pHash[SZHASH];

/** isKWlistSorted - check to see that keyword list is sorted
*
* returns:
*   TRUE - if sorted
*   FALSE - if not sorted
*/
    BOOL
#ifdef __STDC__
isKWlistSorted(VOID)
#else
isKWlistSorted()
#endif /* __STDC__ */
{
    SHORT i;
    SHORT iLast;


    iLast = sizeof(keyWordName)/sizeof(KEYWORDNAME) - 1;
    for (i = 0; i < iLast; i++) {
        if (strcmp(keyWordName[i].pszName, keyWordName[i+1].pszName) >= 0) {
            fprintf(fhMsg, "Keywords \"%s\" and \"%s\" are out of order\n",
                    keyWordName[i].pszName, keyWordName[i+1].pszName);
            return(FALSE);
        }
    }
    return(TRUE);

} /* isKWlistSorted */


/** hashKW - hash keyword value
*
* call with:
*   pszName - name
*
* returns:
*   hash value
*/
    USHORT
#ifdef __STDC__
hashKW(PSZ pszName)
#else
hashKW(pszName)
    PSZ pszName;
#endif /* __STDC__ */
{
    register USHORT usVal;

    for (usVal = 0; *pszName != 0; pszName++) {
        usVal += (pszName[0] - ' ')<<((pszName[1]&0x0a)>>1);
    }

    return(usVal%SZHASH);

} /* hashKW */


/** bldKWhash - build hash table for keywords
*
*/
    VOID
#ifdef __STDC__
bldKWhash(VOID)
#else
bldKWhash()
#endif /* __STDC__ */
{
    INT i;
    USHORT iHash;

    for (i = sizeof(keyWordName)/sizeof(KEYWORDNAME) - 1; i >= 0; i--) {
        iHash = hashKW(keyWordName[i].pszName);
        keyWordName[i].pHashNext = pHash[iHash];
        pHash[iHash] = &(keyWordName[i]);
    }

} /* bldKWhash */


#define MXFREQ 10
/** freqKWhash - print frequences of keyword hash collisions
*
*/
    VOID
#ifdef __STDC__
freqKWhash(VOID)
#else
freqKWhash()
#endif /* __STDC__ */
{
    INT i;
    USHORT iHash;
    USHORT acFreq[MXFREQ];
    KEYWORDNAME *pKWN;


    for (i = 0; i < MXFREQ; i++) {
        acFreq[i] = 0;
    }

    for (iHash = 0; iHash < SZHASH; iHash++) {
        for (i = 0, pKWN = pHash[iHash]; (pKWN != NULL) && (i < (MXFREQ-1)); i++) {
            pKWN = pKWN->pHashNext;
        }
        acFreq[i]++;
        if (i > 1) {
            fprintf(fhMsg, "Hash: %d, %d hits\n", iHash, i);
            for (pKWN = pHash[iHash]; pKWN != NULL; pKWN = pKWN->pHashNext) {
                fprintf(fhMsg, "  \"%s\"\n", pKWN->pszName);
            }
        }

    }

    for (i = 0; i < MXFREQ; i++) {
        if (acFreq[i] != 0)
            fprintf(fhMsg, "%3d: %d\n", i, acFreq[i]);
    }

} /* freqKWhash */


/** makeKWdef - make all keywords defined
*
*/
    VOID
#ifdef __STDC__
makeKWdef(VOID)
#else
makeKWdef()
#endif /* __STDC__ */
{
    register USHORT i;

    for (i = 0; i < (sizeof(keyWordName)/sizeof(KEYWORDNAME)); i++) {
        keyWordName[i].usStatus |= KWSTdef;
    }

} /* makeKWdef */


/** makeKWundef - make all (other than "BASE") keywords undefined
*                 and set use counts to zero
*/
    VOID
#ifdef __STDC__
makeKWundef(VOID)
#else
makeKWundef()
#endif /* __STDC__ */
{
    register USHORT i;

    for (i = 0; i < (sizeof(keyWordName)/sizeof(KEYWORDNAME)); i++) {
        keyWordName[i].cUse = 0;
        if (!(keyWordName[i].usStatus & KWSTbase)) 
            keyWordName[i].usStatus &= ~KWSTdef;
    }

} /* makeKWundef */


/** getKWindex - get index of keyword
*
* call with:
*   pszName - name to lookup
*   fDef - if true, return only defined or base names
*
* returns:
*   index of keyword or
*   -1 if name not a keyword
*/
    SHORT
#ifdef __STDC__
getKWindex(PSZ pszName, BOOL fDef)
#else
getKWindex(pszName, fDef)
    PSZ pszName;
    BOOL fDef;
#endif /* __STDC__ */
{
    register USHORT iHash;
    register SHORT sCmp;
    KEYWORDNAME *pKWN;
    static KEYWORDNAME *pKWNbase = &keyWordName[0];


    iHash = hashKW(pszName);
    pKWN = pHash[iHash];
    while (pKWN != NULL) { /* while hash chain not empty */
        if ((sCmp = strcmp(pszName, pKWN->pszName)) == 0) {
            /* name match */
            if (fDef && !(pKWN->usStatus & KWSTdef))
                return(-1);     /* item not defined */
            else {
                pKWN->cUse++;
                return(pKWN-pKWNbase);
            }
        }
        if (sCmp < 0)           /* item not in sorted chain */
            break;
        pKWN = pKWN->pHashNext;
    }

    return(-1);

} /* getKWindex */


/** addSMIitem - add SMI name to set of defined names
*
* call with:
*   pszNa - ptr to name
*
* returns:
*   TRUE - name added
*   FALSE - name not SMI name
*
*/
    BOOL
#ifdef __STDC__
addSMIitem(PSZ pszNa)
#else
addSMIitem(pszNa)
    PSZ pszNa;
#endif /* __STDC__ */
{
    register SHORT i;
    register USHORT usStat;


    /* lookup name to see if known */
    if ((i = getKWindex(pszNa, FALSE)) == -1) {
        /* name not known */
        return(FALSE);
    }

    /* get status */
    usStat = keyWordName[i].usStatus;

    /* check if already defined */
    if (usStat & KWSTdef) {
        yyerror("\"%s\" is already defined", pszNa);
        return(FALSE);
    }

    if (usStat & KWSTsmi) {
        /* item defined in SMI */
        keyWordName[i].usStatus |= KWSTdef; /* mark defined */
        keyWordName[i].cUse = 0;
        return(TRUE);
    }

    if (usStat & KWSTot) {
        /* OBJECT-TYPE macro - define it and all contained items */
        keyWordName[i].usStatus |= KWSTdef;
        keyWordName[i].cUse = 0;
        for (i = 0; i < sizeof(keyWordName)/sizeof(KEYWORDNAME); i++) {
            if (keyWordName[i].usStatus & KWSTotm)
                keyWordName[i].usStatus |= KWSTdef;
        }
        return(TRUE);
    }

    if (usStat & KWSTtt) {
        /* TRAP-TYPE macro - define it and all contained items */
        keyWordName[i].usStatus |= KWSTdef;
        keyWordName[i].cUse = 0;
        for (i = 0; i < sizeof(keyWordName)/sizeof(KEYWORDNAME); i++) {
            if (keyWordName[i].usStatus & KWSTttm)
                keyWordName[i].usStatus |= KWSTdef;
        }
        return(TRUE);
    }

    if (usStat & KWSTotm) {
        yyerror("\"%s\" is defined in OBJECT-TYPE macro");
        return(FALSE);
    }

    if (usStat & KWSTttm) {
        yyerror("\"%s\" is defined in TRAP-TYPE macro");
        return(FALSE);
    }

    yyerror("INTERNAL ERROR in addSMIitem");
    return(FALSE);

} /* addSMIitem */


/** checkSMIname - check to see if name is SMI item name
*                  (or TRAP-TYPE or OBJECT-TYPE)
*
* call with:
*   pszNa - ptr to name
*   pusIndx - addr to store index 
*
* returns:
*   name status (see SMISTxxx)
*/
    USHORT
#ifdef __STDC__
checkSMIname(PSZ pszNa, USHORT *pusIndx)
#else
checkSMIname(pszNa, pusIndx)
    PSZ pszNa;
    USHORT *pusIndx;
#endif /* __STDC__ */
{
    register SHORT i;
    register USHORT usStat;


    /* lookup name to see if known */
    if ((i = getKWindex(pszNa, FALSE)) == -1) {
        /* name not known */
        return(SMISTnsmi);
    }

    /* get status */
    usStat = keyWordName[i].usStatus;


    /* check if SMI item or OBJECT-TYPE or TYPE-TYPE */
    if ((usStat & (KWSTsmi | KWSTot | KWSTtt)) == 0) {
        /* item not defined in SMI */
        return(SMISTnsmi);
    }

    /* check if already defined */
    if (usStat & KWSTdef) {
        *pusIndx = i;
        return(SMISTad);
    }

    *pusIndx = i;
    return(SMISTsmi);

} /* checkSMIname */


/** getKWuse - get use count for keyword
*
* call with:
*   iKW - index of keyword
*
* returns:
*   use count
*/
    USHORT
#ifdef __STDC__
getKWuse(USHORT iKW)
#else
getKWuse(iKW)
    USHORT iKW;
#endif /* __STDC__ */
{
    return(keyWordName[iKW].cUse);

} /* getKWuse */


/** yylexinit - initialize lexical analyser
*
* returns:
*   TRUE - success
*   FALSE - failure
*
*/
    BOOL
#ifdef __STDC__
yylexinit(VOID)
#else
yylexinit()
#endif /* __STDC__ */
{
    bldKWhash();                /* init hash table */
    usInclLevel = 0;            /* not in INCLUDE file */
    usLineNo = 1;               /* on first line */
    usColNo = 0;                /* before first character */
    cStrSpace = 0;              /* haven't allocated string space yet */

    if (!allocStrSp())          /* allocate string space */
        return(FALSE);

    return(TRUE);

} /* yylexinit */


/** openUsingIncl - open a file using the value of
*                   the include file environment variable
*
* call with:
*   pszNa - name of file
*
* returns:
*   NULL if error
*   or file handle
*/
    FILE *
#ifdef __STDC__
openUsingIncl(PSZ pszNa)
#else
openUsingIncl(pszNa)
    PSZ pszNa;
#endif /* __STDC__ */
{
    FILE *fhIn;
    PSZ pszDirS;
    PSZ pszDirE;
    USHORT cDir;
    USHORT cNa;
    PSZ pszNewNa;


    fhIn = fopen(pszNa, "rb");
    if ((pszSmicIncl == NULL) || (fhIn != NULL) ||
            (*pszNa == chDirChar))
        /* no Include path or file opened or name is absolute */
        return(fhIn);

    cNa = strlen(pszNa);
    /* for each directory in include path, try openning file */
    for (pszDirS = pszSmicIncl; *pszDirS != 0;) {
        /* find start of a directory name */
        /* skip spaces */
        while (*pszDirS == ' ') {
            pszDirS++;
        }
        if (*pszDirS == 0)
            return(NULL);

        /* find end of directory name */
        pszDirE = pszDirS;
        while ((*pszDirE != 0) && (*pszDirE != chSepChar) &&
                (*pszDirE != ' ')) {
            pszDirE++;
        }

        /* compute length of dir name */
        cDir = pszDirE - pszDirS;
        if (cDir != 0) {
            /* allocate space for filename */
            pszNewNa = (CHAR *)malloc(cDir+1+cNa+1);
            if (pszNewNa == NULL)
                return(NULL);

            /* build file name */
            memcpy(pszNewNa, pszDirS, cDir);
            pszNewNa[cDir] = chDirChar;
            memcpy(&(pszNewNa[cDir+1]), pszNa, cNa);
            pszNewNa[cDir+1+cNa] = 0;

            /* open the file */
            fhIn = fopen(pszNewNa, "rb");
            free(pszNewNa);
            if (fhIn != NULL)
                return(fhIn);
        }
        if (*pszDirE == 0)
            return(NULL);
        pszDirS = pszDirE+1;
    }
    return(NULL);

} /* openUsingIncl */


/** openInclude - open include file
*
* call with:
*   pNa - include file name
*
* returns:
*   TRUE - file opened OK
*   FALSE - error opening file
*/
    BOOL
#ifdef __STDC__
openInclude(STRTAB *pNa)
#else
openInclude(pNa)
    STRTAB *pNa;
#endif /* __STDC__ */
{

    /* An include file */
    /* check if already in an include file */
    if (usInclLevel >= MXLV) {
        yyerror("Include file nested too deeply");
        return(FALSE);
    }

    apszInFile[usInclLevel] = pszInFile;    /* save current file name */
    afhIn[usInclLevel] = fhIn;              /* save current file handle */

    /* try to open include file */
    fhIn = openUsingIncl(pNa->pszVal);
    if (fhIn == NULL) {
        yyerror("Error opening Include file \"%s\"", pNa->pszVal);
        fhIn = afhIn[usInclLevel]; /* restore file handle */

        return(FALSE);
    }
    pszInFile = pNa->pszVal;
    yylval.strval.pszFn = pNa->pszVal;

    /* save state of current file */
    ausLineNo[usInclLevel] = usLineNo;
    ausColNo[usInclLevel] = usColNo;
    aLexChar[usInclLevel] = LexChar;

    /* set new state */
    usLineNo = 1;
    usColNo = 0;
    usTokLineNo = 0;
    usTokColNo = 0;
    LexChar = ' ';
    usInclLevel++;

    if (fPrintIname)
        fprintf(fhMsg, "In file: %s\n", pNa->pszVal);

    return(TRUE);

} /* openInclude */


/* Char types */
#define CTeof   0               /* end of file */
#define CTspace 1               /* horizontal space */
#define CTquote 2               /* the " char */
#define CTback  3               /* the \ char */
#define CTline  4               /* the newline char */
#define CTat    5               /* the @ char */
#define CTalnum 6               /* alpha numeric chars */
#define CTminus 7               /* the - char */
#define CTaps   8               /* the ' char */
#define CTubar  9               /* the _ char */
#define CTpound 10              /* the # char */
#define CTlcb   11              /* the { char */
#define CTrcb   12              /* the } char */
#define CTlpr   13              /* the ( char */
#define CTrpr   14              /* the ) char */
#define CTsemi  15              /* the ; char */
#define CTcomma 16              /* the , char */
#define CTdot   17              /* the . char */
#define CTcolon 18              /* the : char */
#define CTeq    19              /* the = char */
#define CTother 20              /* everything else */

/* char type table */
UCHAR NEAR ucCT[257] = {
/*  -1 */    CTeof,
/*           NUL      SOH      STX      ETX      EOT */
/*   0 */    CTother, CTother, CTother, CTother, CTother,
/*           ENQ      ACK      BEL      BS       HT */
/*   5 */    CTother, CTother, CTother, CTother, CTspace,
/*           LF       VT       FF       CR       SO */
/*  10 */    CTline,  CTspace, CTspace, CTspace, CTother,
/*           SI       DLE      DC1      DC2      DC3 */ 
/*  15 */    CTother, CTother, CTother, CTother, CTother,
/*           DC4      NAK      SYN      ETB      CAN */
/*  20 */    CTother, CTother, CTother, CTother, CTother,
/*           EM       SUB      ESC      FS       GS */
/*  25 */    CTother, CTother, CTother, CTother, CTother,
/*           RS       US       blank    !        "  */
/*  30 */    CTother, CTother, CTspace, CTother, CTquote,
/*           #        $        %        &        '  */
/*  35 */    CTpound, CTother, CTother, CTother, CTaps,
/*           (        )        *        +        ,  */
/*  40 */    CTlpr,   CTrpr,   CTother, CTother, CTcomma,
/*           -        .        /        0        1  */
/*  45 */    CTminus, CTdot,   CTother, CTalnum, CTalnum,
/*           2        3        4        5        6  */
/*  50 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           7        8        9        :        ;  */
/*  55 */    CTalnum, CTalnum, CTalnum, CTcolon, CTsemi,
/*           <        =        >        ?        @  */
/*  60 */    CTother, CTeq,    CTother, CTother, CTat,
/*           A        B        C        D        E  */
/*  65 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           F        G        H        I        J  */
/*  70 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           K        L        M        N        O  */
/*  75 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           P        Q        R        S        T  */
/*  80 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           U        V        W        X        Y  */
/*  85 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           Z        [        \        ]        ^  */
/*  90 */    CTalnum, CTother, CTback,  CTother, CTother,
/*           _        `        a        b        c  */ 
/*  95 */    CTubar,  CTother, CTalnum, CTalnum, CTalnum,
/*           d        e        f        g        h  */
/* 100 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           i        j        k        l        m  */
/* 105 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           n        o        p        q        r  */
/* 110 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           s        t        u        v        w  */
/* 115 */    CTalnum, CTalnum, CTalnum, CTalnum, CTalnum,
/*           x        y        z        {        |  */
/* 120 */    CTalnum, CTalnum, CTalnum, CTlcb,   CTother,
/*           }        ~        DEL */     
/* 125 */    CTrcb,   CTother, CTother, CTother, CTother,
/* 130 */    CTother, CTother, CTother, CTother, CTother,
/* 135 */    CTother, CTother, CTother, CTother, CTother,
/* 140 */    CTother, CTother, CTother, CTother, CTother,
/* 145 */    CTother, CTother, CTother, CTother, CTother,
/* 150 */    CTother, CTother, CTother, CTother, CTother,
/* 155 */    CTother, CTother, CTother, CTother, CTother,
/* 160 */    CTother, CTother, CTother, CTother, CTother,
/* 165 */    CTother, CTother, CTother, CTother, CTother,
/* 170 */    CTother, CTother, CTother, CTother, CTother,
/* 175 */    CTother, CTother, CTother, CTother, CTother,
/* 180 */    CTother, CTother, CTother, CTother, CTother,
/* 185 */    CTother, CTother, CTother, CTother, CTother,
/* 190 */    CTother, CTother, CTother, CTother, CTother,
/* 195 */    CTother, CTother, CTother, CTother, CTother,
/* 200 */    CTother, CTother, CTother, CTother, CTother,
/* 205 */    CTother, CTother, CTother, CTother, CTother,
/* 210 */    CTother, CTother, CTother, CTother, CTother,
/* 215 */    CTother, CTother, CTother, CTother, CTother,
/* 220 */    CTother, CTother, CTother, CTother, CTother,
/* 225 */    CTother, CTother, CTother, CTother, CTother,
/* 230 */    CTother, CTother, CTother, CTother, CTother,
/* 235 */    CTother, CTother, CTother, CTother, CTother,
/* 240 */    CTother, CTother, CTother, CTother, CTother,
/* 245 */    CTother, CTother, CTother, CTother, CTother,
/* 250 */    CTother, CTother, CTother, CTother, CTother,
/* 255 */    CTother};

/** yylex - lexical analyser
*
* returns:
*   token type (also sets global yylval)
*
* NOTE: This is a quick and dirty implementation,
*       don't use this an an example of efficiency
*
*/
    INT
#ifdef __STDC__
yylex(void)
#else
yylex()
#endif /* __STDC__ */
{
    register INT rLexChar;
    register INT rClass;
    SHORT i;
    STRTAB *pStrTab;


    rLexChar = LexChar;
    yylval.strval.pszFn = pszInFile;

    for (;;) {

    /* skip white space */
    while(((rClass = ucCT[rLexChar+1]) == CTspace) || (rClass == CTline)) {
        rLexChar = yylexin();
    }

    /* save info on current token */
    yylval.strval.usLineNo = (usTokLineNo = usLineNo);
    yylval.strval.usColNo = (usTokColNo = usColNo);

    switch(rClass) {

    case CTeof:
        /* END OF FILE */
        if (usInclLevel == 0) {
            /* not in include file */
            ausStrSpaceSz[cStrSpace-1] = usStrSpaceIndx;
            LexChar = rLexChar;
            return(sTokType = tokEOF);
        }

        /* in Include file, restore original position */
        fclose(fhIn);
        usInclLevel--;
        fhIn = afhIn[usInclLevel];
        pszInFile = apszInFile[usInclLevel];
        yylval.strval.pszFn = pszInFile;
        if (fPrintIname)
            fprintf(fhMsg, "In file: %s\n", pszInFile);
        usLineNo = ausLineNo[usInclLevel];
        usColNo = ausColNo[usInclLevel];
        rLexChar = aLexChar[usInclLevel];

        /* start skipping white space */
        continue;

    case CTquote:
        /* A string */
        /* Trim leading spaces after a newline upto start column of string */
        /* Note: two quotes are used to embed a quote, or \" can be used */
        {
            register BOOL fEOLtrim;


            pszTokVal = (PSZ)&(pbStrSpace[usStrSpaceIndx]);
            usStrSpaceTokIndx = usStrSpaceIndx;
            fEOLtrim = FALSE;

            for(;;) {
                rLexChar = yylexin();

                switch (ucCT[rLexChar+1]) {

                case CTeof:
                    /* EOF in string */
                    yyerror("EOF inside a string");
                    goto finish_str;

                case CTquote:
                    /* check for embedded quotes or end of quote */
                    rLexChar = yylexin();
                    if (rLexChar != '\"')
                        goto finish_str;    /* all done */

                    if (!fNoStr)
                        yylexout('\"');
                    fEOLtrim = FALSE;
                    break;

                case CTback:
                    /* a '\' */
                    fEOLtrim = FALSE;
                    rLexChar = yylexin();
                    if (rLexChar == '\"') {
                        if (!fNoStr)
                            yylexout('\"'); /* embedded quote */
                    } else {
                        /* unknown char after '\' - output both */
                        if (!fNoStr) {
                            yylexout('\\');
                            yylexout(rLexChar);
                        }
                    }
                    break;

                case CTline:
                    /* a new line - add it to string */
                    if (!fNoStr)
                        yylexout('\n');
                    fEOLtrim = TRUE;    /* eat leading spaces */
                    break;

                case CTspace:
                    /* check for trimming after EOL */
                    if ((!fEOLtrim) || (usColNo >= usTokColNo)) {
                        fEOLtrim = FALSE;
                        /* output space */
                        if (!fNoStr)
                            yylexout(' ');
                    }
                    break;

                default:
                    fEOLtrim = FALSE;
                    if (!fNoStr)
                        yylexout(rLexChar);
                    break;
                }
            } /* for(;;) */

            /* finished with string */
finish_str:
            yylexout(0);
            sTokType = tokSTRING;
            pStrTab = StrTabInsert(pszTokVal);
            if (pStrTab->pszVal != pszTokVal) {
                /* string already in string table */
                /* so don't need another copy */
                usStrSpaceIndx = usStrSpaceTokIndx;
                pszTokVal = pStrTab->pszVal;
            }
            yylval.strval.pStr = pStrTab;
        }
        break;

    case CTaps:
        /* A hex or binary string */

        pszTokVal = (PSZ)&(pbStrSpace[usStrSpaceIndx]);
        usStrSpaceTokIndx = usStrSpaceIndx;

        /* scan until trailing apostrophe */
        for(;;) {
            rLexChar = yylexin();
            switch (ucCT[rLexChar+1]) {
            case CTeof:
                /* EOF in string */
                yyerror("EOF inside a string");
                goto bad_str;

            case CTaps:
                /* end of string */
                goto good_str;

            case CTline:
                /* a newline - don't add it to string */
                yyerror("End of line inside string");
                goto bad_str;

            default:
                /* anything else - add to string */
                yylexout(rLexChar);
                break;
            }
        }

        /* get next char and check for B or H */
good_str:
        rLexChar = yylexin();
        yylexout(0);
        if ((rLexChar == 'B') || (rLexChar == 'b')) {
#ifdef OLD
            if (rLexChar == 'b') {
                yywarning("Binary string should use uppercase \"B\"");
            }
#endif
            /* check that all chars are zero or one */
            for (i = 0; pszTokVal[i] != 0; i++) {
                if ((pszTokVal[i] != '1') && (pszTokVal[i] != '0')) {
                    yyerror("bad binary string");
                    goto bad_str;
                }
            }
            sTokType = tokBSTR;
        } else if ((rLexChar == 'H') || (rLexChar == 'h')) {
#ifdef OLD
            if (rLexChar == 'h') {
                yywarning("Hex string should use uppercase \"H\"");
            }
#endif
            /* check that all chars are hex digits */
            for (i = 0; pszTokVal[i] != 0; i++) {
                if (!isxdigit(pszTokVal[i])) {
                    yyerror("bad hex string");
                    goto bad_str;
                }
            }
            sTokType = tokHSTR;
        } else {
            yyerror("string needs to be followed by 'B' or 'H'");
            goto bad_str;
        }
#ifdef OLD
        if (pszTokVal[0] == 0) {
            yyerror("Zero length string");
            goto bad_str;
        }
#endif
        rLexChar = yylexin();

        /* put in string table */
        pStrTab = StrTabInsert(pszTokVal);
        if (pStrTab->pszVal != pszTokVal) {
            /* string already in string table */
            /* so don't need another copy */
            usStrSpaceIndx = usStrSpaceTokIndx;
            pszTokVal = pStrTab->pszVal;
        }
        yylval.strval.pStr = pStrTab;
        break;

bad_str:
        sTokType = tokBADSTR;
        usStrSpaceIndx = usStrSpaceTokIndx; /* remove from string table */
        szTok[0] = 0;
        pszTokVal = szTok;
        break;

    case CTat:
        /* An include file */
        /* check if already in an include file */
        if (usInclLevel >= MXLV) {
            yyerror("Include file nested too deeply");

            /* skip over file name */
            rLexChar = yylexin();
            while((rLexChar != EOF) && !(isspace(rLexChar))) {
                rLexChar = yylexin();
            }
            continue;           /* continue scanning */   
        }

        /* get filename */

        pszTokVal = (PSZ)&(pbStrSpace[usStrSpaceIndx]);
        usStrSpaceTokIndx = usStrSpaceIndx;

        for (;;) {
            rLexChar = yylexin();
            if ((rLexChar == EOF) || (isspace(rLexChar)))
                break;
            yylexout(rLexChar);
        }
        yylexout(0);

        apszInFile[usInclLevel] = pszInFile; /* save current file name */
        afhIn[usInclLevel] = fhIn;           /* save current file handle */

        /* try to open include file */
        fhIn = fopen(pszTokVal, "rb");
        if (fhIn == NULL) {
            yyerror("Error opening Include file \"%s\"", pszTokVal);
            fhIn = afhIn[usInclLevel]; /* restore file handle */
            usStrSpaceIndx = usStrSpaceTokIndx; /* free up string space */

            continue;           /* continue scanning */
        }
        pszInFile = pszTokVal;
        yylval.strval.pszFn = pszInFile;

        /* save state of base file */
        ausLineNo[usInclLevel] = usLineNo;
        ausColNo[usInclLevel] = usColNo;
        aLexChar[usInclLevel] = rLexChar;

        /* set new state */
        usLineNo = 1;
        usColNo = 0;
        usTokLineNo = 0;
        usTokColNo = 0;
        rLexChar = ' ';
        usInclLevel++;

        if (fPrintIname)
            fprintf(fhMsg, "In file: %s\n", pszInFile);

        /* start at top */
        continue;

    case CTpound:
        /* Check for scanner/compiler directives */

        /* get directive name */
        pszTokVal = (PSZ)&(pbStrSpace[usStrSpaceIndx]);
        usStrSpaceTokIndx = usStrSpaceIndx;
        for (;;) {
            /* get directive name */
            rLexChar = yylexin();
            if ((rLexChar == EOF) || (isspace(rLexChar)))
                break;
            yylexout(rLexChar);
        }
        yylexout(0);

        /* free up string space */
        usStrSpaceIndx = usStrSpaceTokIndx;

        /* check if "include" directive */
        if (strcmp(pszTokVal, "include") == 0) {
            /* include a file */
            sTokType = dirINCLUDE;
            pszTokVal = "#include";
            break;

        } else if (strcmp(pszTokVal, "aliasModule") == 0) {
            /* alias a module */
            sTokType = dirALIASMODULE;
            pszTokVal = "#aliasModule";
            break;

        } else if (strcmp(pszTokVal, "aliasSymbol") == 0) {
            /* alias a symbol */
            sTokType = dirALIASSYMBOL;
            pszTokVal = "#aliasSymbol";
            break;

        } else if (strcmp(pszTokVal, "popOpt") == 0) {
            /* alias a symbol */
            sTokType = dirPOPOPT;
            pszTokVal = "#popOpt";
            break;

        } else if (strcmp(pszTokVal, "pushOpt") == 0) {
            /* alias a symbol */
            sTokType = dirPUSHOPT;
            pszTokVal = "#pushOpt";
            break;

        } else if (strcmp(pszTokVal, "addOpt") == 0) {
            /* alias a symbol */
            sTokType = dirADDOPT;
            pszTokVal = "#addOpt";
            break;

        } else if (strcmp(pszTokVal, "removeOpt") == 0) {
            /* alias a symbol */
            sTokType = dirREMOVEOPT;
            pszTokVal = "#removeOpt";
            break;

        } else if (strcmp(pszTokVal, "printOpt") == 0) {
            /* alias a symbol */
            sTokType = dirPRINTOPT;
            pszTokVal = "#printOpt";
            break;

        } else if (strcmp(pszTokVal, "help") == 0) {
            /* print help for directives */
            sTokType = dirHELP;
            pszTokVal = "#help";
            break;

        } else {
            /* unknown directive */
            yyerror("unknown directive \"%s\"", pszTokVal);

            /* scan to end of line */
            rLexChar = yylexin();
            while((rLexChar != EOF) && (rLexChar != '\n')) {
                rLexChar = yylexin();
            }
        }

        /* start at top */
        continue;

    case CTminus:
        /* check for start of a comment or a minus */
        rLexChar = yylexin();

        if (rLexChar == '-') {
            /* ASN.1 style comment */
            /* scan until end of line */
            for (;;) {
                rLexChar = yylexin();
                if (rLexChar == EOF) {
                    yyerror("EOF inside a comment");
                    break;
                } else if (rLexChar == '\n') {
                    /* end of comment */
                    rLexChar = yylexin();
                    break;
                }
#ifdef OLD
                else if (rLexChar == '-') {
                    /* possible end of comment */
                    rLexChar = yylexin();
                    if (rLexChar == '-') {
                        /* end of comment */
                        rLexChar == yylexin();
                        break;
                    }
                }
#endif
            }
            continue;           /* start over again */
        }
        /* just a minus */
        szTok[0] = '-';
        sTokType = chMINUS;
        pszTokVal = szTok;
        break;

    case CTlcb:     /* the { char */
        sTokType = chLCB;
common_char:
        szTok[0] = (CHAR)rLexChar;
        pszTokVal = szTok;
        rLexChar = yylexin();
        break;

    case CTrcb:     /* the } char */
        sTokType = chRCB;
        goto common_char;

    case CTlpr:     /* the ( char */
        sTokType = chLPR;
        goto common_char;

    case CTrpr:     /* the ) char */
        sTokType = chRPR;
        goto common_char;
    
    case CTsemi:    /* the ; char */
        sTokType = chSEMI;
        goto common_char;

    case CTcomma:   /* the , char */
        sTokType = chCOMMA;
        goto common_char;

    default:        /* anything else */
        sTokType = chOTHER;
        goto common_char;

    case CTdot:     /* check for ".." */
        rLexChar = yylexin();   /* get next input char */
        if (rLexChar == '.') {
            /* double dot token */
            pszTokVal = "..";
            sTokType = tokDOTDOT;
            rLexChar = yylexin();
            while (rLexChar == '.') {
                /* bad ".." */
                sTokType = tokBADDOTDOT;
                rLexChar = yylexin();
            }
        } else {
            /* just a dot which is its own type */
            sTokType = chDOT;
            szTok[0] = '.';
            pszTokVal = szTok;
        }
        break;

    case CTeq:      /* the = char, assume bad tokIS */
        sTokType = tokBADIS;
        pszTokVal = "::=";      /* set to bad assignment */
        rLexChar = yylexin();
        while (rLexChar == ':') {
            /* a colon - try next char */
            rLexChar = yylexin();
        }
        break;

    case CTcolon:   /* check for "::=" */
        rLexChar = yylexin();   /* get next input char */
        sTokType = tokIS;       /* assume type OK */
        pszTokVal = "::=";
        if (rLexChar == ' ') {
            /* a space - try next char */
            rLexChar = yylexin();
            sTokType = tokBADIS; /* set to bad assignment */
        }
        if (rLexChar == ':') {
            /* looking for "::=" */
            rLexChar = yylexin();
            if (rLexChar == ' ') {
                /* a space - try next char */
                rLexChar = yylexin();
                sTokType = tokBADIS; /* set to bad assignment */
            }
            if (rLexChar != '=') {
                /* found bad version */
                sTokType = tokBADIS;
            } else {
                /* found complete version */
                rLexChar = yylexin();
            }
        } else if (rLexChar == '=') {
            /* found bad version */
            rLexChar = yylexin();
            sTokType = tokBADIS;
        }
        break;

    case CTalnum:
        /* token is either a number, name, or keyword */

        /* scan to end of token */
        pszTokVal = (PSZ)&(pbStrSpace[usStrSpaceIndx]);
        usStrSpaceTokIndx = usStrSpaceIndx;
        while(((rClass = ucCT[rLexChar+1]) == CTalnum) ||
             (rClass == CTminus) ||
             (rClass == CTubar)) {
            yylexout(rLexChar);
            rLexChar = yylexin();
        }
        yylexout(0);

        /* check if token is a number */
        sTokType = tokNAME;     /* assume a name */
        if (isdigit(*pszTokVal)) {
            {
                INT c1;
                ULONG ulVal;

                /* check for all digits */
                for (ulVal = 0L, i = 0, c1 = *pszTokVal;
                        (c1 != 0) && isdigit(c1);) {
                    ulVal = ulVal * 10L + (c1 - '0');
                    /* make sure length OK */
                    if (i < (MXDFINT-1))
                        szNum[i] = (CHAR)c1;
                    c1 = pszTokVal[++i];
                }
                if (c1 == 0) {
                    /* end of string, so a number, maybe */
                    /* check for too big values */
                    if ((i >= MXDFINT) || ((i == (MXDFINT-1)) &&
                            (strcmp(pszTokVal, "4294967295") > 0))) {
                        /* value too big */
                        yyerror("number too big, using value of one instead");
                        szNum[0] = '1';
                        szNum[1] = 0;
                        yylval.numval.ul = 1L;
                    } else {
                        /* it is a number */
                        szNum[i] = 0;
                        yylval.numval.ul = ulVal;
                    }
                    sTokType = tokNUMBER;
                    usStrSpaceIndx = usStrSpaceTokIndx;
                    pszTokVal = szNum;
                }
            }
        }

        /* if token is not a number, check if a keyword */
        if (sTokType != tokNUMBER) {
            /* check if token a keyword */
            if ((i = getKWindex(pszTokVal, TRUE)) == -1) {
                /* token is a name */
                /* lookup name in string table and add if not present */
                pStrTab = StrTabInsert(pszTokVal);
                if (pStrTab->pszVal != pszTokVal) {
                    /* name already in string table */
                    /* so don't need another copy */
                    usStrSpaceIndx = usStrSpaceTokIndx;
                    pszTokVal = pStrTab->pszVal;
                }
                yylval.strval.pStr = pStrTab;
            } else {
                /* token is a KeyWord */
                sTokType = keyWordName[i].sType;
                usStrSpaceIndx = usStrSpaceTokIndx;
                pszTokVal = keyWordName[i].pszName;
            }
        }
        break;

#ifdef OLD
    default:
        /* BUG in scanner implementation */
        yyerror("Internal scanner error: bad case");
        exit(1);
#endif
    }

    LexChar = rLexChar;
    return(sTokType);

    } /* for(;;) */

} /* yylex */

/* end of file: SMSCAN.C */
