package Webupdates;

BEGIN {

    # Adding current path and per-lib to search libraries
    my $progPath = $0;
    $progPath =~ /^(.+)\/[^\/]+$/;
    $progPath = $1;
    unshift ( @INC, $progPath );
    unshift ( @INC, "$progPath/perl-lib/" );

    # Needed for tainted mode - see 'man perlsec'
    delete @ENV{ 'IFS', 'CDPATH', 'ENV', 'BASH_ENV' };
    $ENV{'PATH'} = '/bin:/usr/local/bin';
}

use strict;

use Cache::FileCache;
use Crypt::Simple;
use HTTP::Request::Common qw(POST GET);
use LWP::UserAgent;
use Net::Whois::RIPE::RPSL;
use IO::Socket::INET;
use Getopt::Std;

use base 'CGI::Application';

# Per source variables
my $UPDSOURCE     = "";
my $WHOISHOST     = "";
my $WHOISPORT     = "";
my $SYNCUPDATEURL = "";
my $UPDCAPTION    = "";
my $UPDINDEX      = "";
my $QUERYURL      = "";

# Configuration
my $BARE           = 0;        # use fancy headers
my $MAXEXPIREHOURS = 5;        # for passwords
my $MAXRESULTS     = 50;       # maximum number of results accepted from whois
my $FORMMETHOD     = 'POST';
my $FORMACTION     = '';
my $IMGURL         = '/img';
my $HTMLURL      = 'http://127.0.0.1/';                       #full url required
my $COPYRIGHTURL = 'http://www.ripe.net/db/copyright.html';
my @BANNEDOBJECTS= ('mntner','irt','as-block');
my @SYNCUPDATES = (    # caption -must- be unique
    {
        url       => "http://127.0.0.1/cgi-bin/sup_frontend",
        whoishost => "127.0.0.1",
        whoisport => "43001",
        source    => "SAMPLE",
        caption   => "SAMPLE Database",
        queryurl  => "http://127.0.0.1/whois",
    },
    {
        url       => "https://127.0.0.1/cgi-bin/sup_frontend",
        whoishost => "127.0.0.1",
        whoisport => "43001",
        source    => "SAMPLE",
        caption   => "SAMPLE SSL Database",
        queryurl  => "https://127.0.0.1/whois",
    }
);

my $COLORCYCLE = "#FFFFFF #CCCCCC";    # for cycling colors of table rows

# End of Configuration

# static - no need to compute at every run
my @minutes = (
    0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
    32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59
);

# This caches output from whois
my $cacheWhois;

# This caches help text
my $cacheHelp;

# This keeps various html parts, cached
my %htmlCache;

# Recommended DOS preventions
$CGI::POST_MAX        = 409600;    # max 400K posts
$CGI::DISABLE_UPLOADS = 1;         # we don't need it right now

# For convinence, not to show the header twice
my ( $topShown, $bottomShown ) = ();

#
# Init actions - before all
#
sub cgiapp_init {
    my $self = shift;
    my $q    = $self->query;

    $cacheWhois = new Cache::FileCache(
        {
            'namespace'           => 'webupdates',
            'default_expires_in'  => "10 minutes",
            'auto_purge_interval' => '30 minutes',
            'auto_purge_on_set'   => 1
        }
      )
      or croak("Couldn't instantiate cacheWhois");
    $cacheHelp = new Cache::FileCache(
        {
            'namespace'           => 'webupdateshelp',
            'default_expires_in'  => "1 day",
            'auto_purge_interval' => '1 day',
            'auto_purge_on_set'   => 1
        }
      )
      or croak("Couldn't instantiate cacheHelp");

    # Putting various html parts into a hash
    $htmlCache{FORMSTART} =
      $q->start_form( { -method => "$FORMMETHOD", -action => "$FORMACTION" } );

}

#
# required for CGI::Application - sets up form handling
#
sub setup {
    my $self = shift;
    my $q    = $self->query;

    $self->start_mode('help');
    $self->mode_param('formid');
    $self->run_modes(
        'mainform'     => 'mainForm',
        'help'         => 'helpForm',
        'add'          => 'addForm',
        'auth'         => 'authForm',
        'selectsource' => 'selectSourceForm',
    );
}

#
# This runs just before form functions run
#
sub cgiapp_prerun {
    my $self       = shift;
    my $q          = $self->query;
    my @attributes = ();
    my @values     = ();

    $topShown    = 0;
    $bottomShown = 0;

    # register the password via cookies
    if ( defined $q->param('registerpassword') ) {
        registerPassword( $self, $q );
    }

    # setup sources
    if (
        ( !defined $q->param('updsource') )
        || (   ( defined $q->param('updsource') )
            && ( $q->param('updsource') !~ /^[0-9]*$/ ) )
      )
    {
        $UPDINDEX = 0;
    }
    else {
        $UPDINDEX = $q->param('updsource');
    }
    if ( !defined $SYNCUPDATES[$UPDINDEX]{caption} ) {
        $UPDINDEX = 0;
    }
    $WHOISHOST     = $SYNCUPDATES[$UPDINDEX]{whoishost};
    $WHOISPORT     = $SYNCUPDATES[$UPDINDEX]{whoisport};
    $UPDSOURCE     = $SYNCUPDATES[$UPDINDEX]{source};
    $SYNCUPDATEURL = $SYNCUPDATES[$UPDINDEX]{url};
    $UPDCAPTION    = $SYNCUPDATES[$UPDINDEX]{caption};
    $QUERYURL      = $SYNCUPDATES[$UPDINDEX]{queryurl};

    # manipulate helplist - which field will be shown the helplist
    if ( $self->get_current_runmode eq "add" ) {
        if ( !( defined $q->param('helplist') ) ) {
            $q->param( -name => 'helplist', -value => '' );
        }
        my $helplist = $q->param('helplist');
        foreach my $param ( $q->param ) {
            if ( $param =~ /^help:(\d+):[^:]+/ ) {
                if ( $helplist !~ / $1 / ) {
                    $helplist .= " $1 ";
                }
            }
            elsif ( $param =~ /^closehelp:(\d+):[^:]+/ ) {
                $helplist =~ s/ $1 //;
            }
        }
        $q->param( -name => 'helplist', -value => $helplist );
    }

    # switch from plain mode to fancy mode, if 'Switch View' clicked
    if ( not defined $q->param('viewformat') ) {
        $q->param( 'viewformat', 0 );
    }
    else {
        if ( defined $q->param('Switch') ) {
            if ( $q->param('viewformat') eq '0' ) {
                $q->param( 'viewformat', 1 );
            }
            else {
                $q->param( 'viewformat', 0 );
            }
        }
    }
}

#
# read a file and return the output
#
sub readFile($) {
    my $fileName = shift;
    my $result   = "";
    my $req;
    my $ua = LWP::UserAgent->new;

    $req = GET $HTMLURL. $fileName;
    my @result = split ( /\n\n/, ( $ua->request($req)->as_string ) );
    shift (@result);
    return ( join ( "", @result ) );
}

#
# Query whois server and return the result
#
sub whoisQuery($) {
    my $qry    = shift;
    my $result = "";
    my $cnt    = 0;
    my $par    = 0;
    my $whoisF;
    my %opts;

    if (
        !(
            $whoisF = new IO::Socket::INET(
                PeerAddr => $WHOISHOST,
                PeerPort => $WHOISPORT,
                Proto    => 'tcp',
                Type     => SOCK_STREAM
            )
        )
      )
    {
        return "";
    }
    else {

        # make sure the client doesn't pass -k
        my @BARGV = @ARGV;
        @ARGV = split ( / 	/, $qry );
        getopt( 'k', \%opts );
        @ARGV = @BARGV;
        if ( not exists $opts{k} ) {
            print $whoisF "$qry\n";
            while ( my $raw_object = <$whoisF> ) {
                $result .= $raw_object;
                if ( $result =~ /\n\n$/ ) {
                    $cnt++;
                }
                if ( $cnt > $MAXRESULTS ) {
                    if ( $qry !~ /^-v / ) {
                        last;
                    }
                }
            }
        }
    }
    close($whoisF);
    return $result;
}

#
# Query whois server and keep in cache
#
sub cachedWhoisQuery($;$) {
    my $qry    = shift;
    my $expire = shift;
    my $qryResult;

    $qryResult = $cacheWhois->get($qry);
    if (   ( not defined $qryResult )
        || ( ( defined $qryResult ) && ( $qryResult eq "" ) ) )
    {
        $qryResult = whoisQuery($qry);
        $cacheWhois->set( $qry, $qryResult, $expire );
    }

    return $qryResult;
}

#
# Returns a sorted array of available objects
#
sub getObjectList() {
    my @classList = Net::Whois::RIPE::RPSL::get_classes();
    my @classList = ();
    foreach my $item (@classListOne) {
        my $banned=0;
        foreach my $item2 (@BANNEDOBJECTS) {
            if ($item eq $item2) {
                $banned=1;
            }
        }
        if ($banned==0) {
            push @classList, $item;
        }
    }
    return ( sort @classList );
}

#
# Returns true if the object is valid
#
sub isValidObjectType ($) {
    my $objectType = shift;
    my $result     = 0;

    if ( !$objectType ) {
        return 0;
    }
    my @objects = getObjectList();
    foreach my $obj (@objects) {
        if ( $objectType eq $obj ) {
            return 1;
        }
    }
    return $result;
}

#
# Returns true if the attribute is required
#
sub is_required($;$;$) {
    my $template   = shift;
    my $objectType = shift;
    my $attr       = shift;

    foreach my $item ( keys %$template ) {
        my $attrt = $template->{$item};
        if ( $attr eq $attrt->name ) {
            if ( $attrt->is_required ) {
                return 1;
            }
            else {
                return 0;
            }
        }
    }
    return 0;
}

#
# Returns an empty template with mandatory fields
#
sub emptyTemplate($) {
    my $objectType   = shift;
    my $templateData = "";
    my $c            = 0;

    if ( !$objectType ) {
        return "";
    }
    if ( !isValidObjectType($objectType) ) {
        return "";
    }
    my $template = Net::Whois::RIPE::RPSL::get_template($objectType);
    foreach my $item ( keys %$template ) {
        my $attrt = $template->{$item};
        if ( $attrt->is_required ) {
            $templateData .= "$c:" . $attrt->name . " ";
        }
        $c++;
    }
    return $templateData;
}

#
# Returns the help text in html for a given attribute
#
sub helpText($;$;$) {
    my $q          = shift;
    my $objectType = shift;
    my $attribute  = shift;
    my $key        = "$objectType:$attribute";
    my $result     = "";

    my $helpTextData = $cacheHelp->get($key);
    if ( not defined $helpTextData ) {
        my $info = cachedWhoisQuery( "-v $objectType", "1 day" );

        # strip header
        $info =~
s/^.*The content of the attributes of the \S+ class are defined below://s;
        foreach my $attr ( split ( /\n(?=[a-z])/, $info ) ) {

            # clean whitespace
            $attr =~ s/^\s+//;
            $attr =~ s/\s+$//;

            # skip empty information
            next unless ( $attr =~ /\S/ );

            # get name of attr
            $attr =~ /^\S+/;
            my $name = $&;

            # remove the name
            $attr =~ s/^\S+\s*//;

            # escape HTML
            $attr = $q->escapeHTML($attr);

            # save the mapped information
            $cacheHelp->set( "$objectType:$name", $attr, "1 day" );
        }
        return ( $cacheHelp->get($key) );
    }
    else {
        $result = $helpTextData;
    }
    return $result;
}

#
# Parse input to form the template
#
sub getTemplateFromForm($) {
    my $q            = shift;
    my $templateData = "";
    my @result       = ();

    foreach my $item ( $q->param ) {
        if ( $item =~ /^\d+:.+$/ ) {
            if ( ( $item !~ /^\d+:delete/ ) ) {
                $item =~ s/:+ *$//;
                push @result, $item;
            }
        }
    }
    foreach my $item (@result) {
        $templateData .= $item . " ";
    }
    return $templateData;
}

#
# Returns the list of addable attributes for a given object
#
sub addableFields($;$) {
    my $attrList    = shift;
    my $objectType  = shift;
    my $addableData = "";

    if ( !$objectType ) {
        return "";
    }
    my $template = Net::Whois::RIPE::RPSL::get_template($objectType);
    foreach my $item ( keys %$template ) {
        my $attr = $template->{$item};
        if ( !( $attr->is_generated ) ) {
            if ( $attr->is_multivalued ) {
                $addableData .= $attr->name . " ";
            }
        }
        if ( $attr->is_required ) {
            my $found = 0;
            foreach my $attrl (@$attrList) {
                my $attrx = ( split ( /:/, $attrl ) )[1];
                if ( $attrx eq $item ) {
                    $found = 1;
                }
            }
            if ( $found == 0 ) {
                $addableData .= $attr->name . " ";
            }
        }
    }
    $addableData .= "password ";
    return $addableData;
}

#
# Top of html output
#
sub showTop ($) {
    my $output = '';
    my $q      = shift;

    if ($topShown) {
        return "";
    }
    else {
        $topShown = 1;
    }
    $output .= '<?xml version="1.0" encoding="iso-8859-1"?>
    <!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
      "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
    <html xmlns="http://www.w3.org/1999/xhtml" lang="en-US">';
    if ( $BARE eq 0 ) {
        $output .= readFile("htmlTop.html");
    }
    else {
        $output .= $q->start_html;
    }
    $output .= $q->start_table(
        {
            -width       => "645",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0",
        }
    );
    $output .= $q->Tr(
        { -bgcolor => "#DDDDDD" },
        [
            $q->th(
                [
                    $q->font(
                        { -size => '-1' },
                        [
                            $q->a(
                                { href => $q->script_name() . "?formid=help" },
                                "[Help]"
                              )
                              . " "
                              . $q->a(
                                {
                                    href => $q->script_name()
                                      . "?formid=add&updsource=$UPDINDEX"
                                },
                                "[Add]"
                              )
                              . " "
                              . $q->a(
                                {
                                    href => $q->script_name()
                                      . "?formid=add&updateObject=yes&updsource=$UPDINDEX"
                                },
                                "[Edit]"
                              )
                              . " "
                              . $q->a(
                                {
                                    href => $q->script_name()
                                      . "?formid=auth&updsource=$UPDINDEX"
                                },
                                "[Authorization]"
                              )
                              . " "
                              . $q->a(
                                {
                                    href => $q->script_name()
                                      . "?formid=selectsource&updsource=$UPDINDEX"
                                },
                                "[Select Source]"
                              )
                              . " "
                              . $q->a(
                                { href => $QUERYURL }, "[Query Database]"
                              )
                              . " "
                              . $q->a( { href => $COPYRIGHTURL },
                                "[Copyright]" )
                              . $q->br
                              . "Updating $UPDCAPTION"
                        ]
                    )
                ]
            )
        ]
      )
      . $q->Tr( [ $q->td( [ $q->hr ] ) ] );
    $output .= $q->start_Tr . $q->start_td;

    return $output;
}

#
# Bottom of html output
#
sub showBottom ($) {
    my $output = '';
    my $q      = shift;

    if ($bottomShown) {
        return "";
    }
    else {
        $bottomShown = 1;
    }
    $output .= $q->end_td . $q->end_Tr;
    $output .= $q->end_table;
    if ( $BARE eq 0 ) {
        $output .= readFile("htmlBottom.html");
    }
    else {
        $output .= $q->end_html;
    }
    return $output;
}

#
# Shows a form of valid object types
#
sub askObjectType($;$) {
    my $q      = shift;
    my $formid = shift;
    my $output = '';

    $output = $q->Tr(
        [
            $q->td(
                [
                    $q->div(
                        { -align => "center" },
                        [
                            $htmlCache{FORMSTART}
                              . $q->popup_menu(
                                -name   => 'objectType',
                                -values => [
                                    "Please Select Object Type", getObjectList
                                ],
                                -defaults => [ 'Please Select Object Type', '' ]
                              )
                              . $q->hidden( 'formid', $formid )
                              . $q->submit( { -name => 'Add Object' } )
                              . $q->hidden( 'updsource', $UPDSOURCE )
                              . $q->end_form
                              . $q->br
                              . "You can't create objects of type mntner, irt or as-block using this interface. Please use the mail interface for the creation of these objects."
                              . $q->br
                        ]
                    )
                ]
            )
        ]
    );

    return $output;
}

#
# Main Form
#
sub mainForm {
    my $self   = shift;
    my $output = '';
    my $q      = $self->query();

    $output .= showTop($q);
    $output .= $q->table(
        {
            -width       => "100%",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0"
        },
        [
            $q->Tr(
                [
                    $q->td(
                        [
                            $q->div(
                                { -align => 'center' },
                                ['Please Select an Action']
                            )
                        ]
                    )
                ]
            )
        ]
    );
    $output .= showBottom($q);
}

#
# Arrange helplist upon addition or deletion
#
sub arrangeHelpList($;$;$) {
    my $q           = shift;
    my $index       = shift;
    my $step        = shift;
    my $helplistNew = "";

    foreach my $item ( split ( / +/, $q->param('helplist') ) ) {
        if ( ( $item =~ /^\d+$/ ) && ( $item >= $index ) ) {
            $item = $item + $step;
        }
        $helplistNew .= " $item ";
        $q->param( 'helplist', $helplistNew );
    }
}

#
# If the user wanted to add a new field,
# find the offset and insert
#
sub doAdd($;$) {
    my $q        = shift;
    my $attrList = shift;

    my $toAdd = "";
    foreach my $fieldName ( $q->param ) {
        if ( $fieldName =~ /^addline:/ ) {
            $fieldName =~ s/^addline://;
            $toAdd = $fieldName;
            $toAdd =~ s/\s+$//;
            $toAdd =~ s/^\d+://;
            $toAdd =~ s/\..*$//;
            $toAdd =~ s/:+$//;
            $q->param( 'AddNewField', 'yes' );
            $q->param( 'beforeAfter', 'after' );
            $q->param( 'whereToAdd',  $toAdd );
            $q->param( 'fieldToAdd',  $toAdd );
            last;
        }
    }
    if ( $q->param('AddNewField') ) {
        my $beforeAfter = $q->param('beforeAfter');
        my $whereToAdd  = $q->param('whereToAdd');
        my $fieldToAdd  = $q->param('fieldToAdd');
        my $found       = -1;
        my $offset      = -1;
        foreach my $item (@$attrList) {
            $offset++;
            my $tempItem = $item;
            if ( $beforeAfter eq "before" ) {
                if ( $found == -1 ) {
                    if ( $item =~ /$whereToAdd$/ ) {
                        $found = $offset;
                    }
                }
            }
            else {
                if ( $item =~ /$whereToAdd$/ ) {
                    $found = $offset + 1;
                }
            }
        }
        if ( $found == -1 ) {
            $found = $offset;
        }
        my $attrCount = $#{$attrList};
        for ( my $i = $attrCount ; $i >= $found ; $i-- ) {
            my $cnt = $$attrList[$i];
            $cnt++;
            my $attrListOld = $$attrList[$i];
            $$attrList[$i] =~ s/^(\d):/$cnt:/;
            $q->param( $$attrList[$i], $q->param($attrListOld) );
            $q->param( $attrListOld, "" );
        }
        splice( @$attrList, $found, 0, "$found:$fieldToAdd" );
        $q->param( "$found:$fieldToAdd", "" );
        arrangeHelpList( $q, $found, 1 );
    }
}

#
# If the user wanted to delete a field,
# delete it
#
sub doDelete($;$) {
    my $q        = shift;
    my $attrList = shift;
    my $toDelete = "";

    foreach my $fieldName ( $q->param ) {
        if ( $fieldName =~ /^delete:/ ) {
            ( $toDelete = $fieldName ) =~ s/^delete://;
        }
    }
    if ( $toDelete ne "" ) {
        $toDelete =~ s/^(\d+):.*$/$1/;
        splice( @$attrList, $toDelete, 1 );
        arrangeHelpList( $q, $toDelete, -1 );
    }

}

#
# If Up or Down selected for an attribute,
# move it
#
sub doMoving($;$) {
    my $q        = shift;
    my $attrList = shift;
    my ( $moving, $item1, $item2 ) = ( 0, 0, 0 );
    my ( $val1, $val2, $fldName1, $fldName2 ) = ();

    foreach my $fieldName ( $q->param ) {
        if ( $fieldName =~ /^move(Up|Down):/ ) {
            $moving = 1;
        }
    }
    if ( $moving == 0 ) {
        return;
    }

    # If moving attributes,
    if ($moving) {

        # determine which ones will be swapped
        foreach my $item ( $q->param ) {
            if ( $item =~ /^moveUp/ ) {
                ( $item1    = $item ) =~ s/^.+:(\d+):.*$/$1/;
                ( $fldName1 = $item ) =~ s/^.+:\d+:([^\.]+).*$/$1/;
                $item2 = $item1 - 1;
                $item =~ s/^[^:]+:*//;
                $item =~ s/\..*$//;
                $val1 = $q->param($item);
            }
            elsif ( $item =~ /^moveDown/ ) {
                ( $item1    = $item ) =~ s/^.+:(\d+):.*$/$1/;
                ( $fldName1 = $item ) =~ s/^.+:\d+:([^\.]+).*$/$1/;
                $item2 = $item1 + 1;
                $item =~ s/^[^:]+:*//;
                $item =~ s/\..*$//;
                $val1 = $q->param($item);
            }
        }
        foreach my $item ( $q->param ) {
            if ( ( $item2 ne 0 ) && ( $item =~ /^$item2:.*$/ ) ) {
                ( $fldName2 = $item ) =~ s/^\d+:([^:]+):*$/$1/;
                $val2 = $q->param($item);
            }
        }

        # do the swapping
        $q->delete("$item1:$fldName1");
        $q->delete("$item2:$fldName2");
        $q->param( -name => "$item2:$fldName1", -value => "$val1" );
        $q->param( -name => "$item1:$fldName2", -value => "$val2" );
        my $temp = $$attrList[$item1];
        $$attrList[$item1] = $$attrList[$item2];
        $$attrList[$item2] = $temp;
        $$attrList[$item1] =~ s/^[^:]+:/$item1:/;
        $$attrList[$item2] =~ s/^[^:]+:/$item2:/;

        # move the help text also
        my $helplistnew = '';
        my $helptoadd   = '';
        if ( $q->param('helplist') =~ / $item2 / ) {
            ( $helplistnew = $q->param('helplist') ) =~ s/ $item2 //;
            $helptoadd .= " $item1 ";
            $q->param( -name => 'helplist', -value => $helplistnew );
        }
        if ( $q->param('helplist') =~ / $item1 / ) {
            ( $helplistnew = $q->param('helplist') ) =~ s/ $item1 //;
            $helptoadd .= " $item2 ";
            $q->param( -name => 'helplist', -value => $helplistnew );
        }
        $helplistnew = $q->param('helplist');
        $helplistnew .= $helptoadd;
        $q->param( -name => 'helplist', -value => $helplistnew );
    }
}

#
# sort fields
#
sub doSort($;$) {
    my $q        = shift;
    my $attrList = shift;
    my $c        = 0;
    my %vals     = ();

    foreach my $item (@$attrList) {
        my $oldItem = $item;
        $item =~ s/^\d+:/$c:/;
        $vals{$item} = $q->param($oldItem);
        $c++;
    }
    foreach my $item ( keys %vals ) {
        $q->param( -name => $item, -value => $vals{$item} );
    }

}

#
# Mini table for help texts
#
sub helpFrame($;$;$) {
    my $q             = shift;
    my $attr          = shift;
    my $tableRowColor = shift;
    my $output        = '';

    my $attrNo = ( split ( /:/, $attr ) )[0];
    if (   ( defined $q->param('helplist') )
        && ( $q->param('helplist') =~ / $attrNo / ) )
    {
        my $helpAttr = $attr;
        $helpAttr =~ s/^\d:([^:]+):{0,1}$/$1/;
        $output .=
          $q->start_Tr( { -bgcolor => "$tableRowColor", -colspan => '3' } )
          . $q->start_td(
            { -align => 'center', -colspan => '2', -width => '100%' } )
          . $q->start_table( { -border => '1', -width => '80%' } )
          . $q->start_Tr
          . $q->start_td
          . $q->image_button(
            {
                -width    => '16',
                -alt      => "Close",
                -src      => "$IMGURL/x.gif",
                -name     => "closehelp:$attr",
                -value    => "Close",
                -tabindex => '1000',
                -align    => 'right'
            }
          )
          . $q->br
          . $q->start_pre
          . $q->start_font( { -size => '-1' } );
        $output .= helpText( $q, $q->param('objectType'), $helpAttr );
        $output .= $q->end_font
          . $q->end_pre
          . $q->end_td
          . $q->end_Tr
          . $q->end_table
          . $q->end_td
          . $q->end_Tr;
    }
    return $output;
}

#
# button for help for each attribute
#
sub helpButton($;$) {
    my $q      = shift;
    my $attr   = shift;
    my $output = '';

    my $attrNo = ( split ( /:/, $attr ) )[0];
    if (   ( !( defined $q->param('helplist') ) )
        || ( !( $q->param('helplist') =~ / $attrNo / ) ) )
    {
        $output .= $q->image_button(
            {
                -alt      => "Help",
                -src      => "$IMGURL/question.gif",
                -width    => '16',
                -name     => "help:$attr",
                -value    => 'Help',
                -tabindex => '100',
                -align    => 'top'
            }
        );
    }
    else {
        $output .= $q->img(
            {
                -width => '16',
                -alt   => ' ',
                -align => 'top',
                -src   => "$IMGURL/blank.gif"
            }
        );
    }

    return $output;
}

#
# Button to switch multi/single line input
#
sub multiButton($;$) {
    my $q      = shift;
    my $attr   = shift;
    my $output = '';

    if ( $q->param("$attr") =~ /\n/ ) {
        $output .= $q->img(
            {
                -width => '16',
                -alt   => ' ',
                -src   => "$IMGURL/blank.gif",
                -align => 'top'
            }
        );
    }
    else {
        $output .= $q->image_button(
            {
                -alt      => "Multiline",
                -src      => "$IMGURL/multiline.gif",
                -width    => '16',
                -name     => "multiline:$attr",
                -value    => "multiline",
                -tabindex => '105',
                -align    => 'top'
            }
        );
    }
    return $output;
}

#
# button for moving up attributes
#
sub moveUpButton($;$) {
    my $q      = shift;
    my $attr   = shift;
    my $output = '';

    if ( $attr !~ /^0:/ ) {
        if ( $attr !~ /^1:/ ) {
            $output .= $q->image_button(
                {
                    -alt      => "Move This Field Up",
                    -src      => "$IMGURL/up.gif",
                    -width    => '16',
                    -name     => "moveUp:$attr",
                    -value    => "Up",
                    -tabindex => '110',
                    -align    => 'top'
                }
            );
        }
        else {
            $output .= $q->img(
                {
                    -width => '16',
                    -alt   => ' ',
                    -src   => "$IMGURL/blank.gif",
                    -align => 'top'
                }
            );
        }
    }
    return $output;
}

#
# button for moving down attributes
#
sub moveDownButton($;$;$) {
    my $q        = shift;
    my $attr     = shift;
    my $endAttrN = shift;
    my $output   = '';

    if ( $attr !~ /^0:/ ) {
        if ( $attr !~ /^$endAttrN:/ ) {
            $output .= $q->image_button(
                {
                    -alt      => "Move This Field Down",
                    -width    => '16',
                    -src      => "$IMGURL/dn.gif",
                    -name     => "moveDown:$attr",
                    -value    => "Down",
                    -tabindex => '120',
                    -align    => 'top'
                }
            );
        }
        else {
            $output .= $q->img(
                {
                    -width => '16',
                    -alt   => ' ',
                    -align => 'top',
                    -src   => "$IMGURL/blank.gif"
                }
            );
        }
    }
    return $output;

}

#
# attribute deletion button
#
sub deleteButton($;$;$) {
    my $q        = shift;
    my $attr     = shift;
    my $attrList = shift;
    my $output   = '';

    if ( $attr !~ /^0:/ ) {

        my $templateClass =
          Net::Whois::RIPE::RPSL::get_template( $q->param('objectType') );
        my $tempAttr = $attr;
        $tempAttr =~ s/^\d+://;
        $tempAttr =~ s/:\s*$//;
        my @matches = grep { /^\d+:${tempAttr}:*$/ } @$attrList;
        if (
            ( ( scalar @matches ) > 1 )
            || (
                !(
                    is_required(
                        $templateClass, $q->param('objectType'), $tempAttr
                    )
                )
            )
          )
        {
            $output .= $q->image_button(
                {
                    -width    => '16',
                    -alt      => "Delete",
                    -src      => "$IMGURL/x.gif",
                    -name     => "delete:$attr",
                    -value    => "Delete",
                    -tabindex => '130',
                    -align    => 'top'
                }
            );
        }
        else {
            $output .= $q->img(
                {
                    -width => '16',
                    -alt   => ' ',
                    -align => 'top',
                    -src   => "$IMGURL/blank.gif"
                }
            );
        }
    }
    return $output;
}

#
# quick attribute addition button
#
sub addButton($;$;$) {
    my $q        = shift;
    my $attr     = shift;
    my $attrList = shift;
    my $output   = '';

    if ( $attr !~ /^0:/ ) {

        my $addableFieldsStr =
          addableFields( $attrList, $q->param('objectType') );
        ( my $attrName = $attr ) =~ s/^\d+://;
        $attrName =~ s/:*$//;
        if (   ( $addableFieldsStr =~ /^$attrName / )
            || ( $addableFieldsStr =~ / $attrName / )
            || ( $addableFieldsStr =~ / $attrName$/ ) )
        {
            $output .= $q->image_button(
                {
                    -width    => '16',
                    -alt      => "Add",
                    -src      => "$IMGURL/plus.gif",
                    -name     => "addline:$attr",
                    -value    => "Add",
                    -tabindex => '130',
                    -align    => 'top'
                }
            );
        }
        else {
            $output .= $q->img(
                {
                    -width => '16',
                    -alt   => ' ',
                    -align => 'top',
                    -src   => "$IMGURL/blank.gif"
                }
            );
        }
    }
    return $output;
}

#
# input box for each attribute
#
sub inputBox($;$) {
    my $q           = shift;
    my $attr        = shift;
    my $rhs         = '';
    my $useTextArea = 0;
    my $output      = '';

    if ( defined( $q->param("$attr") ) ) {
        $rhs = $q->param("$attr");
    }
    else {
        $rhs = '';
    }

    # If the user asked for a multiline field, add a newline to the input
    # I admit this may be a hack, but a maintainable one 8)
    if ( defined( $q->param("multiline:$attr.x") ) ) {
        $rhs = $q->param("$attr") . "\n";
    }

    # If the attribute name is source and empty, add the default source
    if ( ( $attr =~ /[0-9]+:source/ ) && ( $rhs eq "" ) ) {
        $rhs = "$UPDSOURCE";
    }

    # If there are newlines in the input, textarea will be used
    # Normal input otherwise
    if ( $rhs =~ /\n/ ) {
        $useTextArea = 1;
    }

    # If it's multiline input, mercilessly slaughter leading white space
    # and extra newlines
    if ($useTextArea) {
        $rhs =~ s/^[+ \t][ \t]*//gm;
    }

    $q->param( "$attr", $rhs );

    # Now comes the input box
    if ($useTextArea) {

        $output .= $q->textarea(
            {
                -style    => 'width: 50%',
                -rows     => "4",
                -tabindex => "10",
                -name     => "$attr",
                -value    => "$rhs"
            }
        );
    }
    else {
        $output .= $q->input(
            {
                -style    => 'width: 50%',
                -tabindex => "10",
                -name     => "$attr",
                -value    => "$rhs"
            }
        );
    }

    return $output;
}

#
# generate input line for adding a new attribute
#
sub addNewButtons($;$) {
    my $q         = shift;
    my $attrList  = shift;
    my $output    = "";
    my @uniqAttrs = ();
    my %seen      = ();

    foreach my $item (@$attrList) {
        my $tempItem = $item;
        if ( $tempItem !~ /^0:/ ) {
            $tempItem =~ s/^\d+://;
            $tempItem =~ s/:+ *$//;
            push ( @uniqAttrs, $tempItem ) unless $seen{$tempItem}++;
        }
    }
    $output .= $q->Tr(
        [
            $q->td(
                { -colspan => '2', -align => 'center' },
                [
                    "Add New Field: "
                      . $q->popup_menu(
                        -name   => 'fieldToAdd',
                        -values => [
                            (
                                split (
                                    /\s/,
                                    addableFields(
                                        $attrList, $q->param('objectType')
                                    )
                                )
                            )
                        ],
                        -tabindex => '500'
                      )
                      . $q->popup_menu(
                        -name     => 'beforeAfter',
                        -values   => [ 'before', 'after' ],
                        -default  => 'after',
                        -tabindex => '500'
                      )
                      . " The "
                      . $q->popup_menu(
                        -name     => 'whereToAdd',
                        -values   => [@uniqAttrs],
                        -tabindex => '500'
                      )
                      . $q->hidden( 'formid', 'add' )
                      . " Field. " .= $q->submit(
                        -tabindex => '500',
                        -name     => "AddNewField",
                        -value    => 'Add a field'
                      )
                ]
            )
        ]
    );

    return $output;
}

#
# Deletion form
#
sub deleteForm ($) {
    my $q      = shift;
    my $output = "";

    if (   ( !( defined $q->param('updateObject') ) )
        && ( !( defined $q->param('prevupdate') ) ) )
    {
        return "";
    }
    $output .= $q->hidden( 'searchterm', $q->param('searchterm') )
      . $q->hidden( 'objectoffset', $q->param('objectoffset') )
      . $q->hidden( 'prevupdate',   'yes' )
      . $q->Tr(
        [
            $q->td(
                { -colspan => '2', -align => 'center' },
                [
                    "Reason:"
                      . $q->input( { -name => 'deletereason', -value => '' } )
                      . $q->submit(
                        -name  => 'deleteobject',
                        -value => 'Delete Object'
                      )
                ]
            )
        ]
      );

    return $output;
}

#
# checkbox for 'force new' which prohibits updates
#
sub forceNewCheckBox($) {
    my $q      = shift;
    my $output = '';

    if (   ( defined $q->param('updateObject') )
        || ( defined $q->param('prevupdate') ) )
    {
        return "";
    }
    $output .= $q->Tr(
        [
            $q->td(
                { -colspan => '2', -align => 'center' },
                [
                    $q->checkbox(
                        -name     => 'forceNew',
                        -label    => 'Force New',
                        -tabindex => '600'
                    )
                ]
            )
        ]
    );

    return $output;
}

#
# the submit button
#
sub templateSubmitButton($) {
    my $q      = shift;
    my $output = '';

    $output .= $q->Tr(
        [
            $q->td(
                { -colspan => '2', -align => 'center' },
                [
                    $q->submit(
                        {
                            -tabindex => '20',
                            -name     => 'Submit',
                            -value    => 'Submit Update'
                        }
                      )
                      . $q->hidden( 'viewformat', $q->param('viewformat') )
                      . $q->submit(
                        {
                            -tabindex => '21',
                            -name     => 'Switch',
                            -value    => 'Switch View'
                        }
                      )
                ]
            )
        ]
    );

    return $output;
}

#
# Give the next color to use;
#
sub cycleColors($;$) {
    my $current    = shift;
    my @colorArray = split ( /\s/, shift );

    if ( ( !$$current ) || ( $$current eq '' ) ) {
        $$current = $colorArray[0];
        return;
    }
    while ( $colorArray[0] ne $$current ) {
        push ( @colorArray, shift (@colorArray) );
    }
    push ( @colorArray, shift (@colorArray) );
    $$current = $colorArray[0];
}

#
# Horizontal Ruler in a table
#
sub horizontalLine($) {
    my $q = shift;
    return ( $q->Tr( [ $q->td( { -colspan => '2' }, $q->hr ) ] ) );
}

#
# Form the message to submit to whois server
#
sub formSubmitMessage($;$) {
    my $q        = shift;
    my $attrList = shift;
    my $result   = "";
    my $lhs      = '';
    my $rhs      = '';

    foreach my $attr (@$attrList) {
        $lhs = $attr;
        $lhs =~ s/^[^:]+://;
        if ( $lhs !~ /:$/ ) {
            $lhs .= ":";
        }

        # indent multilines and crop superio
        $rhs = $q->param("$attr");
        my $indentation = " " x ( length($lhs) + 1 );
        $rhs =~ s/\n/\n$indentation/gs;
        $rhs =~ s/[+ \t][ \t]*\n//gs;
        $rhs =~ s/[+ \t][ \t]*$//m;
        $rhs =~ s/\n*$//s;

        $result .= "$lhs $rhs\n";
    }
    my $pass = getPassword($q);
    if ( $pass ne "" ) {
        $result .= "password: $pass\n";
    }

    return "$result\n";
}

#
# Send the update to syncupdate, communicate the result
# 
sub processUpdate($;$;$) {
    my $q            = shift;
    my $update       = shift;
    my $updateErrors = shift;
    my $userAgent    = LWP::UserAgent->new;
    my $request      = "";
    my @result       = ();
    my $lineStripped = '';

    if ( defined $q->param('forceNew') ) {
        $request = POST $SYNCUPDATEURL,
          [ NEW => 'yes', DATA => $update, ORIGIN => $q->remote_host ];
    }
    else {
        $request = POST $SYNCUPDATEURL,
          [ DATA => $update, ORIGIN => $q->remote_host ];
    }
    @result = split ( /\n\n/, $userAgent->request($request)->as_string );
    splice @result, 0, 1;
    my $onTop    = 1;
    my $onBottom = 0;
    foreach my $piece (@result) {
        foreach my $line ( split /\n/, $piece ) {
            ( $lineStripped = $line ) =~ s/^\s+//;
            $lineStripped =~ s/\s+$//;
            if ( ( $lineStripped ne "" ) && ( $line !~ /^[+ \t]/ ) ) {
                push @$updateErrors, $line;
            }
        }
    }
}

#
#
#
sub showUpdateForm($;$;$) {
    my $q            = shift;
    my $attrList     = shift;
    my $updateErrors = shift;

    if ( $q->param('viewformat') eq '0' ) {
        return ( templateForm( $q, $attrList, $updateErrors ) );
    }
    else {
        return ( plainForm( $q, $attrList, $updateErrors ) );
    }
}

#
# Shows the object in a plain form
#
sub plainForm($;$;$) {
    my $q            = shift;
    my $attrList     = shift;
    my $updateErrors = shift;

    my $output          = '';
    my $errorsFormatted = '';
    my $hiddenAttr      = '';
    my $objstr          = '';
    while ( ( scalar @$updateErrors ) > 0 ) {
        $errorsFormatted .= splice( @$updateErrors, 0, 1 ) . $q->br;
    }
    foreach my $attr (@$attrList) {
        my $val .= $q->param($attr);
        $val =~ s/\n([\s]*)\n/\n+$1\n/gm;
        $val =~ s/\n([\s]*)$/\n+$1/gm;
        $val =~ s/\n(?=[^ 	+])/\n /gm;
        $objstr .= ( split ( /:/, $attr ) )[1] . ": " . $val . "\n";
        $hiddenAttr .= $q->hidden( $attr, $q->param($attr) );
    }
    $output .= $htmlCache{FORMSTART}
      . $q->hidden( 'formid',     'add' )
      . $q->hidden( 'helplist',   $q->param('helplist') )
      . $q->hidden( 'objectType', $q->param('objectType') )
      . $q->div( { -align => 'left' },
        [ $q->span( { -style => 'color: red;' }, [$errorsFormatted] ) ] )
      . $q->textarea(
        {
            -style    => 'width: 100%',
            -rows     => "15",
            -tabindex => "10",
            -name     => "plainobject",
            -value    => $objstr
        }
      )
      . $hiddenAttr
      . horizontalLine($q)
      . templateSubmitButton($q)
      . $q->end_form;
}

#
# Shows the object in a template form
#
sub templateForm($;$;$) {
    my $q            = shift;
    my $attrList     = shift;
    my $updateErrors = shift;

    my $output        = '';
    my $tableRowColor = '';

    # if we're switching from plain object, check if the syntax is correct
    # return the plain form if there are errors
    if ( defined $q->param('plainobject') ) {
        @$attrList = ();
        my $p = $q->param('plainobject');
        chomp($p);
        formParamList( $q, $attrList, $p );
        my @errList   = ();
        my $paragraph = $q->param('plainobject');
        chomp($paragraph);
        my $obj = Net::Whois::RIPE::RPSL->new($paragraph);

        if ( $obj->has_error ) {
            my $errcnt = 0;
            foreach my $e ( $obj->errors ) {
                push @errList, $e->descr;
                if ( $e->descr != /Error with the attribute "password"/ ) {
                    $errcnt++;
                }
            }
            if ( $errcnt gt 0 ) {
                $q->param( 'viewformat', '1' );
                return ( showUpdateForm( $q, $attrList, \@errList ) );
            }
        }
    }
    doMoving( $q, $attrList );
    doAdd( $q,    $attrList );
    doDelete( $q, $attrList );
    doSort( $q,   $attrList );

    # endAttrN will be used to determine the last attribute
    # which can't be moved down
    my $endAttrN = scalar @$attrList;
    $endAttrN--;

    # generate input lines for each attribute in the template
    my $inputLines = "";

    # if there is an error from previous update,
    # make a row containing the first line.
    while ( ( scalar @$updateErrors ) > 0 ) {
        if (   ( $$updateErrors[0] !~ /^\S+:/ )
            || ( $$updateErrors[0] =~ /^\*\*\*/ ) )
        {
            $inputLines .= $q->Tr(
                [
                    $q->td(
                        { -align => 'center', -colspan => '2' },
                        [
                            $q->span(
                                { -style => 'color: red;' },
                                [ splice @$updateErrors, 0, 1 ]
                            )
                        ]
                    )
                ]
            );
        }
        else {
            last;
        }
    }
    foreach my $attr (@$attrList) {
        if ( $attr =~ /delete/ ) {
            splice @$updateErrors, 0, 1;
            next;
        }
        cycleColors( \$tableRowColor, $COLORCYCLE );
        my @attrL = split ( /:/, $attr );
        $inputLines .= "\n";
        $inputLines .= $q->Tr(
            { -bgcolor => "$tableRowColor" },
            [
                $q->td( { -align => 'right', -width => '20%' },
                    [ $attrL[1] . ":" ] )
                  . $q->td(
                    { -width => '80%' },
                    [
                        inputBox( $q, $attr )
                          . helpButton( $q,     $attr )
                          . multiButton( $q,    $attr )
                          . addButton( $q,      $attr, $attrList )
                          . moveUpButton( $q,   $attr )
                          . moveDownButton( $q, $attr, $endAttrN )
                          . deleteButton( $q, $attr, $attrList )
                    ]
                  )
            ]
        );
        if ( ( scalar @$updateErrors ) > 0 ) {
            my $errorForLine = "";
            splice @$updateErrors, 0, 1;
            while (( defined $$updateErrors[0] )
                && ( $$updateErrors[0] =~ /^[+ \t]/ ) )
            {
                splice @$updateErrors, 0, 1;
            }
            while (( ( scalar @$updateErrors ) > 0 )
                && ( $$updateErrors[0] =~ /^\*\*\*/ ) )
            {
                $errorForLine .= $$updateErrors[0] . $q->br;
                splice @$updateErrors, 0, 1;
            }
            if ( $errorForLine ne "" ) {
                $inputLines .= $q->Tr(
                    [
                        $q->td(
                            { -align => 'left', colspan => '2' },
                            [
                                $q->span(
                                    { -style => 'color: red;' },
                                    [$errorForLine]
                                )
                            ]
                        )
                    ]
                );
            }
        }
        $inputLines .= helpFrame( $q, $attr, $tableRowColor );
    }
    if ( ( scalar @$updateErrors ) > 0 ) {
        $inputLines .= $q->Tr(
            [
                $q->td(
                    { -align => 'left', -colspan => '2' },
                    [
                        $q->span(
                            { -style => 'color: red;' },
                            [ join ( $q->br, @$updateErrors ) ]
                        )
                    ]
                )
            ]
        );
    }

    # start the main form
    $output .= $htmlCache{FORMSTART}
      . $q->hidden( 'helplist',   $q->param('helplist') )
      . $q->hidden( 'objectType', $q->param('objectType') )
      . $q->table(
        {
            -width       => "100%",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0"
        },
        [
            $inputLines
              . horizontalLine($q)
              . addNewButtons( $q, $attrList )
              . horizontalLine($q)
              . forceNewCheckBox($q)
              . deleteForm($q)
              . horizontalLine($q)
              . templateSubmitButton($q)
        ]
      )
      . $q->hidden( 'updsource', $UPDSOURCE )
      . $q->end_form;

    return $output;
}

#
# Form for entering search term
#
sub showSearchForm($) {
    my $q      = shift;
    my $output = "";

    $output = $htmlCache{FORMSTART}
      . $q->Tr(
        [
            $q->td(
                [
                    $q->div(
                        { -align => 'center' },
                        [
                            $q->input( { -name => 'searchterm' } )
                              . $q->hidden( 'formid',       'add' )
                              . $q->hidden( 'updateObject', 'yes' )
                              . $q->submit(
                                { -name => 'Edit', -value => 'Edit' }
                              )
                        ]
                    )
                ]
            )
        ]
      )
      . $q->hidden( 'updsource', $UPDSOURCE )
      . $q->end_form;

    return $output;
}

#
# Show a list of search results, if more than one result found
#
sub showSearchResults($;$;$) {
    my $q           = shift;
    my $whoisResult = shift;
    my $searchTerm  = shift;
    my $output      = '';
    my @popupValues = (-1);
    my %popupLabels = ();
    my $resultCnt   = 0;

    $popupLabels{-1} = 'Please Select Object';
    foreach my $paragraph ( split ( /\n\n/, $whoisResult ) ) {
        if ( ( $paragraph !~ /^%/ ) && ( $paragraph =~ /^\S+:\s*/ ) ) {
            my $val = ( split ( /\n/, $paragraph ) )[0];
            push @popupValues, $resultCnt;
            $popupLabels{$resultCnt} = $val;
            $resultCnt++;
        }
    }
    $output = $htmlCache{FORMSTART}
      . $q->Tr(
        [
            $q->td(
                [
                    $q->div(
                        { -align => 'center' },
                        [
                            $q->hidden( 'formid', 'add' )
                              . $q->hidden( 'updateObject', 'yes' )
                              . $q->hidden( 'searchterm',   $searchTerm )
                              . "(Only the first $MAXRESULTS results will be shown)"
                              . $q->br
                              . $q->popup_menu(
                                -name    => 'objectoffset',
                                -values  => [@popupValues],
                                -labels  => \%popupLabels,
                                -default => -1
                              )
                              . $q->submit( -value => 'Edit', -name => 'Edit' )
                        ]
                    )
                ]
            )
        ]
      );

    return $output;
}

#
#
#
sub submitUpdate($;$) {
    my $q            = shift;
    my $attrList     = shift;
    my @updateErrors = ();
    my $output       = '';

    my $update = formSubmitMessage( $q, $attrList );
    processUpdate( $q, $update, \@updateErrors );
    if ( ( scalar @updateErrors ) == 1 ) {
        $output .=
          $q->div( { -align => 'center' }, [ $updateErrors[0], $q->br ] );
    }
    else {
        $output .= showUpdateForm( $q, $attrList, \@updateErrors );
    }
    return $output;
}

#
# Form Parameters from object string
#
sub formParamList($;$;$) {
    my $q         = shift;
    my $attrList  = shift;
    my $paragraph = shift;

    my $cnt = 0;
    foreach my $attr ( split ( /\n(?=[a-zA-Z0-9])/, $paragraph ) ) {
        my $attrName = $attr;
        my $attrVal  = $attr;
        $attr    =~ s/^([^:]+):.*$/$1/s;
        $attrVal =~ s/^[^:]+:\s*(.*$)/$1/s;
        if ( !( defined $q->param('objectType') ) ) {
            $q->param( 'objectType', $attr );
        }
        push @$attrList, "$cnt:$attr";
        $q->param( "$cnt:$attr", $attrVal );
        $cnt++;
    }
}

#
# Form for adding objects
#
sub addForm {
    my $self         = shift;
    my $output       = '';
    my $q            = $self->query();
    my @updateErrors = ();
    my $template     = "";
    my $paragraph    = '';

    # attrList is the array that holds attributes
    my @attrList = ();

    # Receive the template.
    # First try to generate it by the input from the previous form. If
    # no information available, provide an empty template
    if ( !( defined $q->param('updateObject') ) ) {
        $template = getTemplateFromForm($q);
        if ( $template eq "" ) {
            $template = emptyTemplate( $q->param('objectType') );
        }
        @attrList = ( split ( /\s/, $template ) );
    }

    # Start html output
    $output .= showTop($q);

    $output .= $q->start_table(
        {
            -width       => "100%",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0"
        }
    );

    # If the user has clicked submit, we send the update
    if (   ( !defined( $q->param('updateObject') ) )
        && ( defined( $q->param('Submit') ) ) )
    {
        $output .= submitUpdate( $q, \@attrList );
    }

    # If the user wants to delete an existing object
    elsif ( defined $q->param('deleteobject') ) {
        push @attrList, "1:delete:";
        $q->param( "1:delete:", $q->param('deletereason') );
        $output .= submitUpdate( $q, \@attrList );
    }

    # If the user wants to update an existing object
    elsif ( defined $q->param('updateObject') ) {

        # Search
        my $searchCount = 0;
        if ( !( defined $q->param('searchterm') ) ) {
            $q->param( 'searchterm', '' );
        }

        # The user wants to update object and submitted search
        my $whoisResult;
        if (   ( defined $q->param('searchterm') )
            && ( $q->param('searchterm') ne "" ) )
        {
            $whoisResult = whoisQuery( $q->param('searchterm') );
        }
        else {
            $whoisResult = "";
        }
        $whoisResult =~ s/\n\n+/\n\n/;
        foreach $paragraph ( split ( /\n\n/, $whoisResult ) ) {
            if ( ( $paragraph !~ /^%/ ) && ( $paragraph =~ /^\S+:\s*/ ) ) {
                $searchCount++;
            }
        }

        # If no results exist
        if ( $searchCount == 0 ) {
            if (
                (
                    defined $q->param('Edit') && ( $q->param('Edit') eq 'Edit' )
                )
              )
            {
                $output .= $q->div( { -align => 'center' },
                    ["No objects found, please try again"] )
                  . $q->br;
            }
            $output .= showSearchForm($q);
        }

        # If only one object found
        elsif (
            ( $searchCount == 1 )
            || (
                (
                    defined $q->param('objectoffset')
                    && ( $q->param('objectoffset') >= 0 )
                    && ( $q->param('objectoffset') < $searchCount )
                )
            )
          )
        {
            my $objectoffset = 0;
            if ( $searchCount > 1 ) {
                $objectoffset = $q->param('objectoffset');
            }
            my $formExists = 0;
            foreach my $item ( $q->param ) {
                if ( $item =~ /^\d:[^:]+$/ ) {
                    $formExists = 1;
                    last;
                }
            }
            if ($formExists) {
                $template = getTemplateFromForm($q);
                @attrList = ( split ( /\s/, $template ) );
            }
            else {
                foreach my $paragraphs ( split ( /\n\n/, $whoisResult ) ) {
                    if (   ( $paragraphs !~ /^%/ )
                        && ( $paragraphs =~ /^\S+:\s*/ ) )
                    {
                        $objectoffset--;
                        if ( $objectoffset < 0 ) {
                            $paragraph = $paragraphs;
                            last;
                        }
                    }
                }
                formParamList( $q, \@attrList, $paragraph );
            }
            if ( defined( $q->param('Submit') ) ) {
                $output .= submitUpdate( $q, \@attrList );
            }
            else {
                $output .= showUpdateForm( $q, \@attrList, \@updateErrors );
            }
        }

        # If more than one objects found
        elsif ( $searchCount > 1 ) {
            $output .=
              showSearchResults( $q, $whoisResult, $q->param('searchterm') );
        }
    }

    # see if we know which object type was requested
    elsif ( !isValidObjectType( $q->param('objectType') ) ) {
        $output .= askObjectType( $q, 'add' );
    }

    # If we got the object type, let's do the template
    else {
        $output .= showUpdateForm( $q, \@attrList, \@updateErrors );
    }

    $output .= $q->end_table;

    # end of html output
    $output .= showBottom($q);
    return $output;
}

#
# Return the password from the cookie if exists, "" otherwise
#
sub getPassword($) {
    my $q = shift;

    my $cook = $q->cookie( -name => 'ripepwd' );
    if ( defined $cook ) {
        return decrypt($cook);
    }
    else {
        return "";
    }
}

#
# Form for adding new auth 
#
sub addAuthForm($) {
    my $q      = shift;
    my $output = '';
    my @expireHours;
    my $currPass = getPassword($q);

    for ( my $i = 0 ; $i < $MAXEXPIREHOURS ; $i++ ) {
        push @expireHours, $i;
    }
    $output .= $q->Tr(
        [
            $q->td(
                [
                    $htmlCache{FORMSTART}
                      . $q->hidden( 'formid', 'auth' )
                      . "Password: "
                      . $q->password_field(
                        -value     => $currPass,
                        -name      => 'passwordtoadd',
                        -maxlength => 80
                      )
                      . " expires in "
                      . $q->popup_menu(
                        -name    => 'expirehours',
                        -values  => [@expireHours],
                        -default => '0'
                      )
                      . " hours "
                      . $q->popup_menu(
                        -name    => 'expireminutes',
                        -values  => [@minutes],
                        -default => '30'
                      )
                      . " minutes "
                      . $q->submit(
                        -name  => 'registerpassword',
                        -value => 'Register'
                      )
                      . $q->hidden( 'updsource', $UPDSOURCE )
                      . $q->end_form
                ]
            )
        ]
    );
    return $output;
}

#
# Register a new password
#
sub registerPassword($;$) {
    my $self   = shift;
    my $q      = shift;
    my $output = '';

    my $pass = $q->param('passwordtoadd');
    if ( ( defined $pass ) && ( $pass ne "" ) ) {
        $pass = encrypt($pass);
        my $hours         = $q->param('expirehours');
        my $minutes       = $q->param('expireminutes');
        my $expireMinutes = $hours * 60 + $minutes;
        my $cookie_out    = $q->cookie(
            {
                -value   => "$pass",
                -name    => 'ripepwd',
                -expires => "+${expireMinutes}m"
            }
        );
        $self->header_props(
            -cookie => "$cookie_out",
            -url    => $ENV{'REQUEST_URI'}
        );
        $self->header_type("redirect");
    }
    return $output;
}

#
# Form for authorization
# 
sub authForm {
    my $self   = shift;
    my $output = '';
    my $q      = $self->query();

    $output .= showTop($q);
    $output .= $q->table(
        {
            -width       => "100%",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0"
        },
        [ addAuthForm($q) ]
    );
    $output .= showBottom($q);
    return $output;
}

#
# Form for selecting the source
#
sub selectSourceForm {
    my $self          = shift;
    my $q             = $self->query();
    my @options       = ();
    my %labels        = ();
    my $output        = '';
    my $tableRowColor = '';

    $output .= showTop($q);
    $output .= $q->start_table(
        {
            -width       => "100%",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0"
        }
    );
    $output .= $q->start_Tr;
    $output .= $q->start_td;
    $output .= $q->div( { -align => 'center' } );

    for my $i ( 0 .. $#SYNCUPDATES ) {
        push @options, $i;
        $labels{$i} = $SYNCUPDATES[$i]{caption};
    }
    $output .= $q->start_table(
        {
            -width       => "100%",
            -cellpadding => "0",
            -cellspacing => "0",
            -border      => "0"
        }
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr(
        { -bgcolor => "$tableRowColor" },
        [
            $q->th(
                { -align => 'center', -colspan => '2' },
                [ "Current Configuration:" . $q->br . $q->br ]
            )
        ]
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr(
        { -bgcolor => "$tableRowColor" },
        [
            $q->th( { -width => '50%', -align => 'right' }, ["Source: "] )
              . $q->td( { -width => '50%', -align => 'left' }, ["$UPDSOURCE"] )
        ]
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr(
        { -bgcolor => "$tableRowColor" },
        [
            $q->th( { -width => '50%', -align => 'right' }, ["Update Url: "] )
              . $q->td(
                { -width => '50%', -align => 'left' },
                ["$SYNCUPDATEURL"]
              )
        ]
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr(
        { -bgcolor => "$tableRowColor" },
        [
            $q->th( { -width => '50%', -align => 'right' }, ["Query Host: "] )
              . $q->td( { -width => '50%', -align => 'left' }, ["$WHOISHOST"] )
        ]
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr(
        { -bgcolor => "$tableRowColor" },
        [
            $q->th( { -width => '50%', -align => 'right' }, ["Query Port: "] )
              . $q->td( { -width => '50%', -align => 'left' }, ["$WHOISPORT"] )
        ]
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr(
        { -bgcolor => "$tableRowColor" },
        [
            $q->th( { -width => '50%', -align => 'right' }, ["Query Url: "] )
              . $q->td( { -width => '50%', -align => 'left' }, ["$QUERYURL"] )
        ]
    );
    cycleColors( \$tableRowColor, $COLORCYCLE );
    $output .= $q->Tr( { -bgcolor => "$tableRowColor" },
        [ $q->td( { -colspan => '2' }, [ $q->br . $q->br ] ) ] );
    $output .= $q->end_table;
    $output .= $htmlCache{FORMSTART};
    $output .= $q->hidden( 'formid', 'selectsource' );
    $output .= $q->popup_menu(
        -name   => 'updsource',
        -values => [@options],
        -labels => \%labels
    );
    $output .= $q->submit( -name => 'Select Update Source' );
    $output .= $q->end_div;
    $output .= $q->hidden( 'updsource', $UPDSOURCE );
    $output .= $q->end_form;
    $output .= $q->end_td;
    $output .= $q->end_Tr;
    $output .= $q->end_table;
    $output .= showBottom($q);
    return $output;
}

sub helpForm {
    my $self   = shift;
    my $q      = $self->query();
    my $output = '';

    $output .= showTop($q);
    $output .= readFile("helptext.html");
    $output .= showBottom($q);
    return $output;
}

1;
