/*
     Filter - Packet filtering software for the Drawbridge package
     Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

     MESSAGE.C - Version 1.0 - 4/21/93
*/

// MESSAGE.C
//
// The routines in this file implement the protocol that fm uses
//   to communicate with the filter.
//
// NOTE:
//
// When we allocate network tables, we always allocate the size
//   of the table + 16 and then adjust the pointer to the first
//   paragraph in the table. The reason we do this is that Class
//   B tables are 64K large. Using malloc causes an extra 4 bytes
//   to be added on to the front of the block. Thus we are trying
//   to work with a 64K + 4B structure which you can't do with
//   far pointers (the last four bytes in the host table would
//   wrap around to the beginning and try to use malloc's information.)
//   We don't want the overhead of huge pointers so we overallocate
//   by a paragraph and then shift to the first paragraph boundary
//   in the block. We then end up with a pure pointer to a 64K block
//   which will not wrap. (Don't you just love DOS and Intel?)
//
#include <dos.h>
#include <io.h>
#include <alloc.h>
#include <process.h>
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <mem.h>
#include <string.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <dir.h>
#include "filter.h"
#include "wd.h"

unsigned long fmSeq = 0;
unsigned long filterSeq = 0;
unsigned long newFmSeq = 0;
unsigned long newFilterSeq = 0;
int syncing = NO;
QueryPacket queryPacket;

// Build a response packet to be sent back to the machine running fm.
void buildPacket(
	int side,
	EtherAddress *toAddr,
	unsigned char *data,
	int length,
	int type,
	unsigned long *theFmSeq,
	unsigned long *theFilterSeq)
{
    FiltHeader *filtHead;
    EtherHeader *etherHead;
    WDInterface *theInterface;
    DESauth tempAuth;

    //fprintf(stdout,"building packet for side %d, size = %d, data = %08lX\n",side,length,data);

    // Pick the correct interface.
    theInterface = (side == INSIDE_MASK) ? campus : internet;

    // Put my ethernet address and the destination address in the packet.
    etherHead = (EtherHeader *) theInterface->packet;
    etherHead->etherDestHost = *toAddr;
    etherHead->etherSrcHost  = theInterface->etherAddress;
    etherHead->etherType     = FILTERPROT;

    filtHead = (FiltHeader *) (((unsigned char *) etherHead) + sizeof(EtherHeader));

    // Set up the authentication packet.
    tempAuth.fmSeq     = ++*theFmSeq;
    tempAuth.filterSeq = ++*theFilterSeq;

    // fprintf(stdout,"sequence going out: fm = %08lX filter = %08lX\n",
    //          tempAuth.fmSeq,tempAuth.filterSeq);

    // Encrypt if we are in that mode.
    if (desKeyLoaded == NO) {
	filtHead->flags = NO;
	filtHead->auth  = tempAuth;
    }
    else {
	filtHead->flags = YES;
	encrypt((unsigned char *) &filtHead->auth,
		(unsigned long *) desKey,
		(unsigned char *) &tempAuth);
    }

    filtHead->type = type;

    length = (length + 1) >> 1;

    // Copy the data portion of the packet to the buffer.
    moveWords(length,((unsigned char *) filtHead) + sizeof(FiltHeader),data);

    // Set all of the flags.
    theInterface->packetGood    = YES;
    theInterface->packetWaiting = YES;
    theInterface->packetLength  = length + sizeof(EtherHeader) + sizeof(FiltHeader);

    // Floor the packet length so that it goes out as a legal packet.
    if (theInterface->packetLength < 64)
	theInterface->packetLength = 64;
}

// Save the DES key off to disk.
int installNewKey(unsigned char *newKey)
{
    int fd;

    // fprintf(stdout,"installing newkey\n");

    // Open the DES key file.
    fd = open(DES_KEY_FILE,
	      O_WRONLY | O_BINARY | O_CREAT,
	      S_IREAD | S_IWRITE);

    if (fd != -1) {

	if (write(fd,(char *)newKey,sizeof(desKey)) == -1) {
	    // fprintf(stdout,"Error writing out DES key\n");
	    return -2;
	}

	// fprintf(stdout,"Saved DES key\n");

	close(fd);
    }
    else {
	// fprintf(stdout,"Could not open file errno = %d\n",errno);
	return -1;
    }

    return 0;
}

void handleReboot(EtherHeader *etherHead,FiltHeader *filtHead,int from)
{
    //fprintf(stdout,"reboot requested\n");

    // Send back a REBOOTACK packet.
    buildPacket(from,&etherHead->etherSrcHost,
		(char *) NULL,0,
		FM_M_REBOOTACK,&fmSeq,&filterSeq);

    // Set the reboot flag. This will be acted upon when the ack has
    //   been delivered.
    rebootRequested = YES;
}

void handleNewkey(EtherHeader *etherHead,FiltHeader *filtHead,int from)
{
    ErrorPacket error;

#ifdef DES
    unsigned char *newKey;
    unsigned char *p;
    unsigned char tempBuf[8];
    int result;

    // Get the key and decrypt it with the old key if there was one.
    newKey = ((unsigned char *) filtHead) + sizeof(FiltHeader);

    if (desKeyLoaded == YES) {
	for (p = newKey;p < newKey + sizeof(desKey);p += 8) {
	    decrypt(
		(unsigned char *) tempBuf,
		(unsigned long *) desKey,
		(unsigned char *) p);
	    memcpy(p,tempBuf,8);
	}
    }

    // Save the key to disk.
    result = installNewKey(newKey);

    switch (result) {
	case -1:
	    // Couldn't open the file. Set the appropriate error code.
	    error.errorCode = FM_ERROR_DESFILE;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    break;

	case -2:
	    // Couldn't write to the file. Set the appropriate error code.
	    error.errorCode = FM_ERROR_DESWRITE;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    break;

	case 0:
	    // The install succeeded. Send back an ack and enable
	    //   the key.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) NULL,0,
			FM_M_NEWKEYACK,&fmSeq,&filterSeq);

	    memcpy(desKey,newKey,sizeof(desKey));
	    desKeyLoaded = YES;

	    break;
	default:
	    //fprintf(stdout,"oops, bad return code from installDesKey\n");
	    break;
    }
#else
    // DES is not compiled in. Return error code.
    error.errorCode = FM_ERROR_NODES;

    // Send back an error packet.
    buildPacket(from,&etherHead->etherSrcHost,
		(unsigned char *) &error,sizeof(ErrorPacket),
		FM_M_ERROR,&fmSeq,&filterSeq);
#endif
}

void handleQuery(EtherHeader *etherHead,FiltHeader *filtHead,int from)
{
    int i;
    int j;
    int index;
    unsigned int host;
    in_addr network;
    QueryPacket *query;
    ErrorPacket error;

    //fprintf(stdout,"received QUERY\n");

    query = (QueryPacket *) (((unsigned char *) filtHead) + sizeof(FiltHeader));

    //fprintf(stdout,"type = %d\n",query->type);

    // Zero out the query reply packet.
    memset((unsigned char *) &queryPacket,0,sizeof(QueryPacket));

    switch (query->type) {

	case FM_QUERY_NETWORK:
	    // Send back a list of all the currently loaded network tables.

	    //fprintf(stdout,"network query\n");

	    // Build the table of networks for the manager.
	    queryPacket.type = FM_QUERY_NETWORK;

	    for (i = 0,j = 0;i < MAX_NUM_NETWORKS;++i) {
		if (addrTable[i].network.S_addr != 0L) {
		    queryPacket.queryResult.networks[j] =
			    addrTable[i].network;
		    ++j;
		}
	    }

	    // Send back the answer.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &queryPacket,sizeof(QueryPacket),
			FM_M_QUERYACK,&fmSeq,&filterSeq);
	    break;

	case FM_QUERY_HOST:
	    // Get the network and host from the address.
	    if (IN_CLASSB(query->queryValue.addr.S_addr)) {
		// Class B address.
		network.S_addr = query->queryValue.addr.S_addr & CLASSB_NETWORK;
		host = (unsigned int) (query->queryValue.addr.S_addr & CLASSB_HOST);
	    }
	    else if (IN_CLASSC(query->queryValue.addr.S_addr))
	    {
		// Class C address.
		network.S_addr = query->queryValue.addr.S_addr & CLASSC_NETWORK;
		host = (unsigned int) (query->queryValue.addr.S_addr & CLASSC_HOST);
	    }
	    else {
		// We don't handle class A or D addresses. Send
		//   back an error packet.
		error.errorCode = FM_ERROR_NONETWORK;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
			   (unsigned char *) &error,sizeof(ErrorPacket),
			   FM_M_ERROR,&fmSeq,&filterSeq);
		break;
	    }

	    for (i = 0;i < MAX_NUM_NETWORKS;++i)
		if (addrTable[i].network.S_addr == network.S_addr)
		    break;

	    if (i == MAX_NUM_NETWORKS) {
		// Couldn't find the proper network. Send
		//   back an error packet.
		error.errorCode = FM_ERROR_NONETWORK;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
			   (unsigned char *) &error,sizeof(ErrorPacket),
			   FM_M_ERROR,&fmSeq,&filterSeq);
	    }
	    else {
		// Build the answer to the query.
		queryPacket.type = FM_QUERY_HOST;
		queryPacket.queryResult.index = addrTable[i].hostTable[host];

		// Send back the answer.
		buildPacket(from,&etherHead->etherSrcHost,
			    (unsigned char *) &queryPacket,sizeof(QueryPacket),
			    FM_M_QUERYACK,&fmSeq,&filterSeq);
	    }

	    break;

	case FM_QUERY_REJECT:
	    //fprintf(stdout,"reject query\n");

	    // Build the class tables for the manager.
	    queryPacket.type = FM_QUERY_REJECT;

	    memcpy((unsigned char *) queryPacket.queryResult.reject,
		   (unsigned char *) rejectTable,
		   sizeof(RejectTableEntry) * MAX_NUM_REJECT_ENTRIES);

	    // Send back the answer.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &queryPacket,sizeof(QueryPacket),
			FM_M_QUERYACK,&fmSeq,&filterSeq);
	    break;

	case FM_QUERY_ALLOW:
	    //fprintf(stdout,"allow query\n");

	    // Build the class tables for the manager.
	    queryPacket.type = FM_QUERY_ALLOW;

	    memcpy((unsigned char *) queryPacket.queryResult.allow,
		   (unsigned char *) allowTable,
		   sizeof(AllowTableEntry) * MAX_NUM_ALLOW_ENTRIES);

	    // Send back the answer.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &queryPacket,sizeof(QueryPacket),
			FM_M_QUERYACK,&fmSeq,&filterSeq);
	    break;

	case FM_QUERY_CLASS:
	    //fprintf(stdout,"class query\n");

	    // Get the index of the class being asked for.
	    index = query->queryValue.index;

	    //fprintf(stdout,"index = %d\n",index);

	    // Build the class tables for the manager.
	    queryPacket.type = FM_QUERY_CLASS;

	    // Copy the class to the packet.
	    memcpy((unsigned char *)queryPacket.queryResult.accessList.in,
		   (unsigned char *) (in + index * MAX_NUM_ACCESS_RANGES),
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
	    memcpy((unsigned char *)queryPacket.queryResult.accessList.out,
		   (unsigned char *) (out + index * MAX_NUM_ACCESS_RANGES),
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
	    memcpy((unsigned char *)queryPacket.queryResult.accessList.src,
		   (unsigned char *) (source + index * MAX_NUM_ACCESS_RANGES),
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);
	    memcpy((unsigned char *)queryPacket.queryResult.accessList.udp,
		   (unsigned char *) (udp + index * MAX_NUM_ACCESS_RANGES),
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

	    // Send back the answer.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &queryPacket,sizeof(QueryPacket),
			FM_M_QUERYACK,&fmSeq,&filterSeq);

	    break;

	default:
	    // Bad query. Send back a gripe.
	    error.errorCode = FM_ERROR_COMMAND;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    break;
    }
}

void handleLoad(EtherHeader *etherHead,FiltHeader *filtHead,int from)
{
    int i;
    int index;
    unsigned int curr;
    unsigned long size;
    unsigned long offset;
    unsigned long hash;
    in_addr network;
    ErrorPacket error;
    LoadPacket *load;

    //fprintf(stdout,"received load command\n");

    load = (LoadPacket *) (((unsigned char *) filtHead) + sizeof(FiltHeader));

    switch (load->type) {

	case FM_LOAD_NETWORK:
	    // Get the network and offset.
	    network = load->loadValue.networkBlock.network;
	    offset  = load->loadValue.networkBlock.offset;

	    // Determine the size of the host table.
	    if (IN_CLASSB(network.S_addr))
		size = 0x10000UL;
	    else if (IN_CLASSC(network.S_addr))
		size = 0x100UL;
	    else {
		// We do not support A or D networks. Gripe.
		error.errorCode = FM_ERROR_NONETWORK;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
			   (unsigned char *) &error,sizeof(ErrorPacket),
			   FM_M_ERROR,&fmSeq,&filterSeq);

		break;
	    }

	    //fprintf(stdout,"network = %08lX offset = %ld\n",
	    //          network,offset);

	    // Find the network in the new networks array.
	    for (i = 0;i < MAX_NUM_NEW_NETWORKS;++i)
		if (newAddrTable[i].network.S_addr ==
		    network.S_addr)
		    break;

	    if (load->flags & FM_LOAD_FLAGS_BEGIN) {

		//fprintf(stdout,"BEGIN\n");

		if (i == MAX_NUM_NEW_NETWORKS) {
		    // We didn't find the network in the new network
		    //   list so find an empty slot to insert it in.
		    for (i = 0;i < MAX_NUM_NEW_NETWORKS;++i)
			if (newAddrTable[i].network.S_addr ==
				0L)
			    break;

		    // If there is no room in the table for a
		    //   new entry then send back an error message.
		    if (i == MAX_NUM_NEW_NETWORKS) {
			error.errorCode = FM_ERROR_NOMEMORY;

			// Send back an error packet.
			buildPacket(from,&etherHead->etherSrcHost,
				    (unsigned char *) &error,sizeof(ErrorPacket),
				    FM_M_ERROR,&fmSeq,&filterSeq);
			break;
		    }

		    // Install the network into the new network array.
		    newAddrTable[i].network = network;
		}

		// If a buffer has not been allocated for the entry
		//   then allocate one and clear it.
		if (newAddrTable[i].hostTable != (unsigned char *) NULL) {

		    //fprintf(stdout,"deleting old table\n");
		    free(newAddrTable[i].realHostTable);
		}

		newAddrTable[i].realHostTable = (unsigned char *) farmalloc(size + 16);

		if (newAddrTable[i].realHostTable == (unsigned char *) NULL) {
		    // Return an out of memory error.
		    buildPacket(from,&etherHead->etherSrcHost,
			    (unsigned char *) &error,sizeof(ErrorPacket),
			    FM_M_ERROR,&fmSeq,&filterSeq);

		    // Uninstall the network in the new network array.
		    newAddrTable[i].network.S_addr = 0L;
		    break;
		}

		// Adjust the table address we use to a paragraph boundary.
		//fprintf(stdout,"new table address is %08lX\n",newAddrTable[i].realHostTable);
		newAddrTable[i].hostTable = (unsigned char *)
				 MK_FP(FP_SEG(newAddrTable[i].realHostTable) + 1,0);
	    }

	    // Transfer the data to the new network buffer.
	    if (i == MAX_NUM_NEW_NETWORKS) {
		// The network is not defined!
		error.errorCode = FM_ERROR_NONETWORK;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
			    (unsigned char *) &error,sizeof(ErrorPacket),
			    FM_M_ERROR,&fmSeq,&filterSeq);
		break;
	    }

	    memcpy(newAddrTable[i].hostTable + (unsigned int) offset,
		   load->loadData.networkBlock,
		   (unsigned int) (size == 0x100UL ? 0x100UL : 1024UL));

	    if (load->flags & FM_LOAD_FLAGS_END) {
		// Get the network and host from the address.
		//   Install the network into the address hash
		//   table and delete the entry in the new networks
		//   array.

		// Hash into the table and see if the network has been defined.
		hash = (network.S_addr & NETWORK_HASH_MASK) >> 19;
		curr = (unsigned int) hash;

		while (addrTable[curr].network.S_addr != network.S_addr &&
		       addrTable[curr].network.S_addr != 0) {

		    curr = (curr + 1) & (MAX_NUM_NETWORKS - 1);

		    if (curr == hash) {
			// Network hash table is full.
			error.errorCode = FM_ERROR_NONETWORK;

			// Send back an error packet.
			buildPacket(from,&etherHead->etherSrcHost,
				    (unsigned char *) &error,sizeof(ErrorPacket),
				    FM_M_ERROR,&fmSeq,&filterSeq);

			// Release the memory that was allocated during the
			//   load. Probably should check for this case before
			//   we allow the load to begin.
			free(newAddrTable[i].realHostTable);
			newAddrTable[i].realHostTable = (unsigned char *) NULL;
			newAddrTable[i].hostTable = (unsigned char *) NULL;
			newAddrTable[i].network.S_addr = 0UL;

			break;
		    }
		}

		//fprintf(stdout,"inserted network at %d\n",curr);

		// Set the address for the table.
		addrTable[curr].network.S_addr = network.S_addr;

		// Install the network and mark it as dirty (not saved).
		addrTable[curr].dirty = YES;

		// Free up the previous table if it existed. (This should not
		//   happen since if we are inserting here the entry should
		//   already be free()'d.)
		if (addrTable[curr].hostTable)
		    free(addrTable[curr].realHostTable);

		addrTable[curr].hostTable = newAddrTable[i].hostTable;
		addrTable[curr].realHostTable = newAddrTable[i].realHostTable;
		newAddrTable[i].realHostTable = (unsigned char *) NULL;
		newAddrTable[i].hostTable = (unsigned char *) NULL;
		newAddrTable[i].network.S_addr = 0UL;
	    }

	    // Send back a LOADACK packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_LOADACK,&fmSeq,&filterSeq);

	    break;
	case FM_LOAD_REJECT:
	    // Load in the new reject table.
	    memcpy(rejectTable,
		   load->loadData.reject,
		   sizeof(rejectTable));

	    // Mark the table as dirty.
	    rejectTableDirty = YES;

	    // Send back a LOADACK packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_LOADACK,&fmSeq,&filterSeq);

	    break;

	case FM_LOAD_ALLOW:
	    // Load in the new allow table.
	    memcpy(allowTable,
		   load->loadData.allow,
		   sizeof(allowTable));

	    // Mark the table as dirty.
	    allowTableDirty = YES;

	    // Send back a LOADACK packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_LOADACK,&fmSeq,&filterSeq);
	    break;

	case FM_LOAD_CLASS:
	    //fprintf(stdout,"loading class\n");

	    // Get the network and offset.
	    index = load->loadValue.index;

	    //fprintf(stdout,"index = %d\n",index);
	    //fprintf(stdout,"load = %08X\n",load);
	    //fprintf(stdout,"index = %08X\n",&load->loadValue.index);

	    if (load->flags & FM_LOAD_FLAGS_BEGIN) {
		//fprintf(stdout,"begin\n");

		if (newIn == (AccessListTableEntry *) NULL)
		    newIn = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
			     MAX_NUM_ACCESS_RANGES *
			     sizeof(AccessListTableEntry));

		if (newOut == (AccessListTableEntry *) NULL)
		    newOut = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
			      MAX_NUM_ACCESS_RANGES *
			      sizeof(AccessListTableEntry));

		if (newSource == (AccessListTableEntry *) NULL)
		    newSource = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
				 MAX_NUM_ACCESS_RANGES *
				 sizeof(AccessListTableEntry));

		if (newUdp == (AccessListTableEntry *) NULL)
		    newUdp = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
				 MAX_NUM_ACCESS_RANGES *
				 sizeof(AccessListTableEntry));

		memset((void *) newIn,0,MAX_NUM_ACCESS_LISTS *
			MAX_NUM_ACCESS_RANGES *
			sizeof(AccessListTableEntry));

		memset((void *) newOut,0,MAX_NUM_ACCESS_LISTS *
			MAX_NUM_ACCESS_RANGES *
			sizeof(AccessListTableEntry));

		memset((void *) newSource,0,MAX_NUM_ACCESS_LISTS *
			MAX_NUM_ACCESS_RANGES *
			sizeof(AccessListTableEntry));

		memset((void *) newUdp,0,MAX_NUM_ACCESS_LISTS *
			MAX_NUM_ACCESS_RANGES *
			sizeof(AccessListTableEntry));

	    }

	    // Copy the access list.
	    memcpy(newIn + index * MAX_NUM_ACCESS_RANGES,
		   load->loadData.accessList.in,
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

	    memcpy(newOut + index * MAX_NUM_ACCESS_RANGES,
		   load->loadData.accessList.out,
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

	    memcpy(newSource + index * MAX_NUM_ACCESS_RANGES,
		   load->loadData.accessList.src,
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

	    memcpy(newUdp + index * MAX_NUM_ACCESS_RANGES,
		   load->loadData.accessList.udp,
		   sizeof(AccessListTableEntry) * MAX_NUM_ACCESS_RANGES);

	    // Install the loaded table.
	    if (load->flags & FM_LOAD_FLAGS_END) {
		//fprintf(stdout,"end\n");

		// The tables will have always been allocated so we don't
		//   need to check the pointers.
		free(in);
		in = newIn;
		newIn = (AccessListTableEntry *) NULL;

		free(out);
		out = newOut;
		newOut = (AccessListTableEntry *) NULL;

		free(source);
		source = newSource;
		newSource = (AccessListTableEntry *) NULL;

		free(udp);
		udp = newUdp;
		newUdp = (AccessListTableEntry *) NULL;

		accessTableDirty = YES;
	    }

	    // Send back a LOADACK packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_LOADACK,&fmSeq,&filterSeq);
	    break;

	default:
	    //fprintf(stdout,"unknown load type %d\n",load->type);

	    // Bad load. Send back a gripe.
	    error.errorCode = FM_ERROR_COMMAND;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    break;
    }
}

void handleWrite(EtherHeader *etherHead,FiltHeader *filtHead,int from)
{
    int i;
    int fd;
    unsigned int writeAmount;
    unsigned long size;
    unsigned long currSize;
    char filename[15];
    ErrorPacket error;

    //fprintf(stdout,"write requested\n");

    if (rejectTableDirty == YES) {
	fd = open(REJECT_LIST_FILE,
		O_WRONLY | O_BINARY | O_CREAT,
		S_IREAD | S_IWRITE);

	if (fd != -1) {

	    writeAmount = sizeof(RejectTableEntry) *
		MAX_NUM_REJECT_ENTRIES;

	    if (write(fd,(char *) rejectTable,writeAmount) != writeAmount) {
		// Send back an error message.
		error.errorCode = FM_ERROR_DATAWRITE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		close(fd);

		return;
	    }

	    rejectTableDirty = NO;
	    close(fd);
	}
	else {
	    // Could not open the data file. Gripe.
	    error.errorCode = FM_ERROR_DATAFILE;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    return;
	}
    }

    if (allowTableDirty == YES) {

	fd = open(ALLOW_LIST_FILE,
		O_WRONLY | O_BINARY | O_CREAT,
		S_IREAD | S_IWRITE);

	if (fd != -1) {

	    writeAmount = sizeof(AllowTableEntry) *
			MAX_NUM_ALLOW_ENTRIES;

	    if (write(fd,(char *) allowTable,writeAmount) != writeAmount) {
		// Send back an error message.
		error.errorCode = FM_ERROR_DATAWRITE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		close(fd);

		return;
	    }

	    allowTableDirty = NO;
	    close(fd);
	}
	else {
	    // Could not open the data file. Gripe.
	    error.errorCode = FM_ERROR_DATAFILE;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    return;
	}
    }

    //fprintf(stdout,"working on access table\n");

    if (accessTableDirty == YES) {
	// Write out the access lists.
	fd = open(ACCESS_LIST_FILE,
		O_WRONLY | O_BINARY | O_CREAT,
		S_IREAD | S_IWRITE);

	if (fd != -1) {

	    //fprintf(stdout,"wrote in part\n");

	    writeAmount = sizeof(AccessListTableEntry) *
		MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES;

	    if (write(fd,(char *)in,writeAmount) != writeAmount) {

		// Send back an error message.
		error.errorCode = FM_ERROR_DATAWRITE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		close(fd);

		//fprintf(stdout,"wrote in\n");

		return;
	    }

	    if (write(fd,(char *)out,writeAmount) != writeAmount) {
		// Send back an error message.
		error.errorCode = FM_ERROR_DATAWRITE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		close(fd);

		return;
	    }

	    if (write(fd,(char *)source,writeAmount) != writeAmount) {
		// Send back an error message.
		error.errorCode = FM_ERROR_DATAWRITE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		close(fd);

		//fprintf(stdout,"write failed\n");

		return;
	    }

	    if (write(fd,(char *)udp,writeAmount) != writeAmount) {
		// Send back an error message.
		error.errorCode = FM_ERROR_DATAWRITE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		close(fd);

		//fprintf(stdout,"write failed\n");

		return;
	    }

	    accessTableDirty = NO;
	    close(fd);
	}
	else {
	    // Could not open the data file. Gripe.
	    error.errorCode = FM_ERROR_DATAFILE;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    return;
	}
    }

    //fprintf(stdout,"wrote access table\n");

    // Write out all of the dirty network tables.
    for (i = 0;i < MAX_NUM_NETWORKS;++i) {

	if (addrTable[i].dirty == YES) {

	    // Create the file name for the network.
	    sprintf(filename,"%08lx.%s",addrTable[i].network.S_addr,
			NETWORK_EXTENSION);

	    //fprintf(stdout,"creating new network file %s\n",filename);

	    fd = open(filename,
			O_WRONLY | O_BINARY | O_CREAT,
			S_IREAD | S_IWRITE);

	    if (fd != -1) {

		// Get the right size.
		if (IN_CLASSB(addrTable[i].network.S_addr))
		    size = 0x10000UL;
		else if (IN_CLASSC(addrTable[i].network.S_addr))
		    size = 0x100UL;

		// Write the network first.
		if (write(fd,(void *)&addrTable[i].network,
			   sizeof(in_addr)) != sizeof(in_addr)) {
		    // Send back an error message.
		    error.errorCode = FM_ERROR_DATAWRITE;

		    // Send back an error packet.
		    buildPacket(from,&etherHead->etherSrcHost,
				(unsigned char *) &error,sizeof(ErrorPacket),
				FM_M_ERROR,&fmSeq,&filterSeq);

		    close(fd);

		    return;
		}

		// Since write() won't take longs.....
		currSize = size;
		while (currSize) {

		    writeAmount = (unsigned short) (currSize < 32768UL ? currSize : 32768UL);

		    //fprintf(stdout,"writeAmount = %d\n",writeAmount);

		    // Write the host table out. Be careful with the pointer
		    //   math in calculating the read address.
		    if (write(fd,addrTable[i].hostTable + (unsigned short) (size - currSize),
			      writeAmount) != writeAmount) {

			error.errorCode = FM_ERROR_DATAWRITE;

			// Send back an error packet.
			buildPacket(from,&etherHead->etherSrcHost,
				    (unsigned char *) &error,sizeof(ErrorPacket),
				    FM_M_ERROR,&fmSeq,&filterSeq);

			close(fd);

			return;
		    }

		    currSize -= writeAmount;
		}

		addrTable[i].dirty = NO;
		close(fd);

		//fprintf(stdout,"closed file\n");
	    }
	    else {
		// Could not open the data file. Gripe.
		error.errorCode = FM_ERROR_DATAFILE;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
			    (unsigned char *) &error,sizeof(ErrorPacket),
			    FM_M_ERROR,&fmSeq,&filterSeq);
		return;
	    }
	}
    }

    // Send back a WRITEACK packet.
    buildPacket(from,&etherHead->etherSrcHost,
		(char *) NULL,0,
		FM_M_WRITEACK,&fmSeq,&filterSeq);

}

void handleRelease(EtherHeader *etherHead,FiltHeader *filtHead,int from)
{
    char filename[15];
    int i;
    unsigned int curr;
    unsigned long hash;
    in_addr network;
    ErrorPacket error;
    ReleasePacket *release;

    //fprintf(stdout,"release requested\n");
    release = (ReleasePacket *) (((unsigned char *) filtHead) + sizeof(FiltHeader));

    switch (release->type) {
	case FM_RELEASE_NETWORK:
	    network = release->network;

	    for (i = 0;i < MAX_NUM_NETWORKS;++i)
		if (addrTable[i].network.S_addr ==
		    network.S_addr)
		    break;

	    if (i != MAX_NUM_NETWORKS) {

		// First create the file name.
		sprintf(filename,"%08lx.%s",addrTable[i].network.S_addr,
			NETWORK_EXTENSION);

		// Delete the file. Don't worry if it succeeded since the
		//   the network may still be dirty and the file not
		//   exist yet.
		unlink(filename);

		// Free the table.
		free(addrTable[i].realHostTable);

		// Delete the entry out of the hash table.
		hash = (addrTable[i].network.S_addr & NETWORK_HASH_MASK) >> 19;
		curr = (unsigned int) ((hash + 1) & (MAX_NUM_NETWORKS - 1));

		// Shuffle up the hash table entries.
		while (addrTable[curr].network.S_addr != 0) {

		    if (((addrTable[curr].network.S_addr & NETWORK_HASH_MASK) >> 19) == hash) {

			// Shift it up.
			addrTable[i] = addrTable[curr];
			i = curr;
		    }

		    curr = (curr + 1) & (MAX_NUM_NETWORKS - 1);
		}

		// Clean out this last entry.
		addrTable[i].network.S_addr = 0UL;
		addrTable[i].realHostTable = (unsigned char *) NULL;
		addrTable[i].hostTable = (unsigned char *) NULL;
		addrTable[i].dirty = NO;
	    }
	    else {
		// Network could not be found.
		error.errorCode = FM_ERROR_NONETWORK;

		// Send back an error packet.
		buildPacket(from,&etherHead->etherSrcHost,
			    (unsigned char *) &error,sizeof(ErrorPacket),
			    FM_M_ERROR,&fmSeq,&filterSeq);
		break;
	    }

	    // Send back an ack.
	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_RELEASEACK,&fmSeq,&filterSeq);

	    break;
	case FM_RELEASE_ALLOW:
	    // Clear out the allow table.
	    memset((void *) allowTable,0,sizeof(allowTable));

	    // Delete the file.
	    unlink(ALLOW_LIST_FILE);

	    allowTableDirty = NO;

	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_RELEASEACK,&fmSeq,&filterSeq);
	    break;
	case FM_RELEASE_REJECT:
	    // Clear out the reject table.
	    memset((void *) rejectTable,0,sizeof(rejectTable));

	    // Delete the file.
	    unlink(REJECT_LIST_FILE);

	    rejectTableDirty = NO;

	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_RELEASEACK,&fmSeq,&filterSeq);
	    break;
	case FM_RELEASE_CLASSES:

	    // Clean out the access lists.
	    memset((void *) in,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

	    memset((void *) out,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

	    memset((void *) source,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

	    memset((void *) udp,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

	    // Set up the default tables. These are all allow lists.
	    in[0].begin = 25;            // Mail
	    in[0].end = 25;
	    in[1].begin = 53;            // Name service
	    in[1].end = 53;
	    out[0].begin = 1;            // Everything
	    out[0].end = 0xFFFF;
	    source[0].begin = 20;        // FTP data connections
	    source[0].end = 20;
	    udp[0].begin    = 0;             // Disallow TFTP (69) and Portmapper (111).
	    udp[0].end      = 68;
	    udp[1].begin    = 70;
	    udp[1].end      = 110;
	    udp[2].begin    = 112;
	    udp[2].end      = 0xFFFF;

	    // Delete the file. (May fail if there was not anything
	    //   loaded in the first place.)
	    unlink(ACCESS_LIST_FILE);

	    accessTableDirty = NO;

	    buildPacket(from,&etherHead->etherSrcHost,
			(char *) NULL,0,
			FM_M_RELEASEACK,&fmSeq,&filterSeq);
	    break;
	default:
	    //fprintf(stdout,"unknown message type\n");
	    // Bad release. Send back a gripe.
	    error.errorCode = FM_ERROR_COMMAND;

	    // Send back an error packet.
	    buildPacket(from,&etherHead->etherSrcHost,
			(unsigned char *) &error,sizeof(ErrorPacket),
			FM_M_ERROR,&fmSeq,&filterSeq);
	    break;
    }

}

// Note that all of the code in this routine ignores byte ordering problems.
//   It assumes that the manager side will handle all of that for it.
void filtMessage(EtherHeader *etherHead,int from)
{
    WDInterface *theInterface;
    FiltHeader *filtHead;
    DESauth tempAuth;
    ErrorPacket error;

    //fprintf(stdout,".\n");

    // Check the ethernet address.
    theInterface = (from == INSIDE_MASK) ? campus : internet;

    // Make sure that the packet is addressed to this card's address.
    if (theInterface->etherAddress.words[0] !=
	etherHead->etherDestHost.words[0] ||
	theInterface->etherAddress.words[1] !=
	etherHead->etherDestHost.words[1] ||
	theInterface->etherAddress.words[2] !=
	etherHead->etherDestHost.words[2]) {

	// Right protocol but not my ethernet address.
	//fprintf(stdout,"not my address\n");
	return;
    }

    // Get a pointer to the filter protocol packet header.
    filtHead = (FiltHeader *) (((unsigned char *) etherHead) + sizeof(EtherHeader));

    // Check the encryption mode of the manager. If it is not the
    //   same as ours, then return an error.
    if (filtHead->flags != desKeyLoaded) {
	//fprintf(stdout,"wrong mode\n");

	// Set the appropriate error code.
	error.errorCode = desKeyLoaded ? FM_ERROR_SECURE : FM_ERROR_INSECURE;

	// Send back an error packet.
	buildPacket(from,&etherHead->etherSrcHost,
		    (unsigned char *) &error,sizeof(ErrorPacket),
		    FM_M_ERROR,&fmSeq,&filterSeq);
	return;
    }

    // Else decrypt the sequence numbers and validate the message.
    if (desKeyLoaded == YES) {
	//fprintf(stdout,"decrypting sequence numbers\n");
	//fprintf(stdout,"encrypted sequence coming in: fm = %08lX filter = %08lX\n",
	//      filtHead->auth.fmSeq,filtHead->auth.filterSeq);
	//fprintf(stdout,"des key is\n");
	//for (i = 0;i < 128;++i) {
	//    fprintf(stdout,"%02X ",desKey[i]);
	//    if ((i % 16) == 0)
	//      fprintf(stdout,"\n");
	//}
	//fprintf(stdout,"\n");

	memset((void *) &tempAuth,0,sizeof(tempAuth));

	decrypt((unsigned char *) &tempAuth,
		(unsigned long *) desKey,
		(unsigned char *) &filtHead->auth);
    }
    else {
	// fprintf(stdout,"using sequence numbers unencrypted\n");
	tempAuth = filtHead->auth;
    }

    // fprintf(stdout,"sequence coming in: fm = %08lX filter = %08lX\n",
    //          tempAuth.fmSeq,tempAuth.filterSeq);
    // fprintf(stdout,"sequence expecting: fm = %08lX filter = %08lX\n",
    //          fmSeq,filterSeq);

    // Check if this was a repeated request.
    if (tempAuth.fmSeq     == fmSeq &&
	tempAuth.filterSeq == filterSeq &&
	theInterface->packetGood == YES) {

	// Resend the last packet.
	//fprintf(stdout,"repeated request - resending the ack\n");
	theInterface->packetWaiting = YES;
    }
    else {

	// If we just synced then set the new sequence counters.
	if (syncing            == YES &&
	    tempAuth.fmSeq     == newFmSeq + 1 &&
	    tempAuth.filterSeq == newFilterSeq + 1) {

	    // fprintf(stdout,"finishing a synchronization\n");

	    fmSeq     = newFmSeq;
	    filterSeq = newFilterSeq;
	    syncing   = NO;
	}

	// If the sequence numbers match or if this is a SYNC message. Otherwise
	//   we just ignore the message.
	if ((tempAuth.fmSeq == fmSeq + 1 && tempAuth.filterSeq == filterSeq + 1) ||
	    filtHead->type == FM_M_SYNC) {

	    if (filtHead->type != FM_M_SYNC) {
		// fprintf(stdout,"updating the sequence numbers\n");

		// Increment the sequence numbers.
		++fmSeq;
		++filterSeq;
	    }

	    switch (filtHead->type) {
		case FM_M_SYNC:
		    //fprintf(stdout,"received SYNC request\n");

		    // Save the new sequence number that is being established.
		    newFmSeq  = tempAuth.fmSeq;
		    newFilterSeq = rand();
		    syncing = YES;

		    // Send back a SYNCACK packet.
		    buildPacket(from,&etherHead->etherSrcHost,
				(char *) NULL,0,
				FM_M_SYNCACK,&newFmSeq,&newFilterSeq);
		    break;
		case FM_M_REBOOT:
		    handleReboot(etherHead,filtHead,from);
		    break;
		case FM_M_NEWKEY:
		    handleNewkey(etherHead,filtHead,from);
		    break;
		case FM_M_QUERY:
		    handleQuery(etherHead,filtHead,from);
		    break;
		case FM_M_LOAD:
		    handleLoad(etherHead,filtHead,from);
		    break;
		case FM_M_WRITE:
		    handleWrite(etherHead,filtHead,from);
		    break;
		case FM_M_RELEASE:
		    handleRelease(etherHead,filtHead,from);
		    break;
		default:
		    //fprintf(stdout,"unknown message type\n");
		    break;
	    }
	}
    }
}
