/* Copyright (c) 1995,1996 NEC Corporation.  All rights reserved.            */
/*                                                                           */
/* The redistribution, use and modification in source or binary forms of     */
/* this software is subject to the conditions set forth in the copyright     */
/* document ("COPYRIGHT") included with this distribution.                   */

/* This file has the main function in it for socks5, as well as variables    */
/* which most or several modules will use...                                 */
#include "socks5p.h"
#include "threads.h"
#include "daemon.h"
#include "socket.h"
#include "protocol.h"
#include "msgids.h"
#include "log.h"


#ifndef NUMCLIENTS
#define NUMCLIENTS 64
#endif

#ifndef HAVE_SETSID
#ifdef  HAVE_SETPGID
#define setsid() setpgid(0, getpid())
#elif defined(SETPGRP_TWOARG)
#define setsid() setpgrp(0, getpid())
#else
#define setsid() setpgrp()
#endif
#endif

MUTEX_T env_mutex = MUTEX_INITIALIZER; /* *env mutex         */
MUTEX_T gpw_mutex = MUTEX_INITIALIZER; /* getpwd* mutex      */
MUTEX_T gh_mutex  = MUTEX_INITIALIZER; /* gethost* mutex     */
MUTEX_T gs_mutex  = MUTEX_INITIALIZER; /* getserv* mutex     */
MUTEX_T lt_mutex  = MUTEX_INITIALIZER; /* localtime mutex    */

int nthreads   = 0;
int nservers   = 0;
int isthreaded = 0;
int servermode = 0;

static void version(void) {
    fprintf(stdout, "Socks5 version: %s", SOCKS5_VERSION_NAME);
    exit(0);
}

static void usage(void) {
    fprintf(stderr, "Usage incorrect...\n");
    fprintf(stderr, "usage: socks5 ");
    fprintf(stderr, "[-d|--debug] ");
    fprintf(stderr, "[-w|--warnings] ");
    fprintf(stderr, "[-s|--stderr] ");
    fprintf(stderr, "[-v|--version] ");
    fprintf(stderr, "\n");
    fprintf(stderr, "[-f|--nofork] ");
    fprintf(stderr, "[-i|--inetd] ");
    fprintf(stderr, "[-p|--prefork] ");
    fprintf(stderr, "[-o|--oneshot] ");
    fprintf(stderr, "[-t|--threaded] ");
    fprintf(stderr, "[-n X|--nchildren X]");
    fprintf(stderr, "\n");
    exit(-1);
}

int main(int argc, char **argv, char **envp) {
    char *maxc = getenv("SOCKS5_MAXCHILD"), tbuf[1024];
    int how = -1, level = -1;
    time_t now = time(NULL);
    int separate = 1;

    if ((nservers = maxc?atoi(maxc):MAXCLIENTS) > NUMCLIENTS) nservers = NUMCLIENTS;

    for (argc--, argv++; argc > 0; argc--, argv++) {
	if (!strncmp(*argv, "-d", strlen("-d")) || !strncmp(*argv, "--debug", strlen("--debug")))  {
	    if      ((*argv)[1] == '-' && (*argv)[strlen("--debug")] != '\0') level = S5_LOG_DEBUG(atoi(*argv + strlen("--debug")));
	    else if ((*argv)[1] != '-' && (*argv)[strlen("-d"     )] != '\0') level = S5_LOG_DEBUG(atoi(*argv + strlen("-d")));
	    else                                                              level = S5_LOG_DEBUG_MAX;
	} else if (!strcmp(*argv, "-w") || !strcmp(*argv, "--warnings"))   {
	    level = S5_LOG_WARNING;
	} else if (!strcmp(*argv, "-s") || !strcmp(*argv, "--stderr"))   {
	    how = S5_LOG_LOCAL;
	} else if (!strcmp(*argv, "-i") || !strcmp(*argv, "--inetd"))  {
	    servermode = INETD;
	} else if (!strcmp(*argv, "-f") || !strcmp(*argv, "--nofork")) {
	    separate = 0;
	} else if (!strcmp(*argv, "-p") || !strcmp(*argv, "--prefork")) {
	    servermode = PREFORKING;
	} else if (!strcmp(*argv, "-n") || !strcmp(*argv, "--nchildren")) {
	    nservers = (*++argv)?atoi(*argv):nservers; argc--;
	} else if (!strcmp(*argv, "-v") || !strcmp(*argv, "--version"))   {
	    version();
	} else if (!strcmp(*argv, "-t") || !strcmp(*argv, "--threaded"))  {
	    S5LogShowThreadIDS = 1;
	    servermode = THREADED;
	} else if (!strcmp(*argv, "-o") || !strcmp(*argv, "--oneshot"))   {
	    servermode = SINGLESHOT;
	    level      = S5_LOG_DEBUG_MAX;
	    how        = S5_LOG_LOCAL;
	    separate   = 0;
	} else {
	    usage();
	}
    }

    if (separate && fork() > 0) exit(0);
    chdir("/");
    umask(0);
    setsid();

    nthreads = MAXOPENS/4 - 1;
#ifdef RLIMIT_NOFILE
    if (servermode == THREADED) {
	struct rlimit rl;
	
        /* try allow for maxc open file descriptors                          */
	if (getrlimit(RLIMIT_NOFILE, &rl) == 0) {
	    if (rl.rlim_cur < rl.rlim_max) {
	    	rl.rlim_cur = rl.rlim_max;
	    	setrlimit(RLIMIT_NOFILE, &rl);
	    }

	    nthreads = rl.rlim_cur/4 - 1;
	}
    }
#endif

#ifdef RLIMIT_NPROC
    if (servermode != INETD) {
	struct rlimit rl;
	
	/* try allow for maxc children (or MAXCLIENTS)                       */
	if (getrlimit(RLIMIT_NPROC, &rl) == 0 && nservers > rl.rlim_cur && nservers <= rl.rlim_max) {
	    rl.rlim_cur = nservers;
	    setrlimit(RLIMIT_NPROC, &rl);
	}
    }
#endif

    S5LogStart(&S5LogDefaultHandle, how, level, "Socks5");
    LIBPREFIX2(init)("Socks5");


    MUTEX_LOCK(lt_mutex);
    strftime(tbuf, sizeof(tbuf), "%c", localtime(&now));
    S5LogUpdate(S5LogDefaultHandle, S5_LOG_INFO, MSGID_SERVER_START, "Socks5 starting at %s", tbuf);
    MUTEX_UNLOCK(lt_mutex);

    GetConnection();
    return(-1);
}

