/* cmd-gw.c */

/* Copyright 1997 by Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
   Donated to the public domain.  No warranty.

   Version 0.1	1997-01-04 Initial version
   Version 0.2	1997-01-07 Kill child on exit
   Version 0.3	1997-01-17 Reset child_pid
   Version 0.4	1997-03-20 Print error message for execv() failure
   Version 0.5	1997-04-05 Reorganize libraries
   Version 0.6  1997-07-20 Use const */

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/file.h>
#include "firewall.h"
#include "../libem/libemfw.h"
#include "../libem/libemtn.h"

static int cmd_help ();
static int cmd_quit ();

/* Important: The compiler should put this array into read-only memory
   (.text).  Therefore we make it `const'. */

static Cmd const cmds[] =
{
  {"dig",	 1, "   dig <host>",			0},
  {"exit",	 0, "   exit",				cmd_quit},
  {"finger",	 2, "   finger [<user>@]<host>",	0},
  {"help",	 0, "   help/?",			cmd_help},
  {"ping",	 3, "   ping <host>",			0},
  {"quit",	 0, "   quit",				cmd_quit},
  {"traceroute", 4, "   traceroute <host>",		0},
  {"?",		 0, 0,					cmd_help},
  {0,		 0, 0,					0}
};

/* Important: The compiler should put this array into read-only memory
   (.text).  Therefore we make it `const'. */

static const char * const pgms[] =
{
  "/usr/bin/dig",
  "/usr/bin/finger",
  "/bin/ping",
  "/usr/bin/traceroute"
};

static int child_pid;
static char *tokv[20];
static char riaddr[512];
static char rladdr[512];
static char input[512];
static char tokbuf[512];
static tnconn *t;

static void cleanup (void);
static void run_prog (int argc, char *argv[], const char *pgm);


int main (int argc, char *argv[])
{
  Cfg *confp;
  Cmd *cp;
  int tokc;

  openlog ("cmd-gw", LOG_PID|LOG_NDELAY, LFAC);

  if (argc >= 3 && strcmp (argv[1], "-daemon") == 0)
    {
      int port = str_to_port (argv[2]);
      argc -= 2; argv += 2;
      syslog (LLEV, "Starting daemon mode on port %d", port);
      do_daemon (port);
    }

  if ((confp = cfg_read ("cmd-gw")) == (Cfg *)-1)
    exit (1);
  config_groupid (confp, 1);
  config_userid (confp, 1);

  if (peername (0, rladdr, riaddr, sizeof (riaddr)) != 0)
    {
      syslog (LLEV, "fwtksyserr: cannot get peer name: %m");
      exit (1);
    }

  if (config_hosts (confp, rladdr, riaddr) == (Cfg *)0)
    exit (1);

  t = tn_init (0, TN_INIT_TELNET);
  tn_puts (t, "Welcome to cmd-gw.\r\n");

  atexit (cleanup);
  for (;;)
    {
      tn_puts (t, "Command: ");
      tn_gets (t, input, sizeof (input), PROXY_TIMEOUT, TN_GETS_ECHO);
      tokc = enargv (input, tokv, sizeof (tokv) / sizeof (tokv[0]),
		     tokbuf, sizeof (tokbuf));
      if (tokc <= 0)
	continue;
      cp = find_command ((Cmd *)cmds, tokv[0]);
      if (cp == (Cmd *)0)
	tn_puts (t, "Ambiguous command.\r\n");
      else if (cp->cnam == (char *)0)
	tn_puts (t, "Unrecognized command.\r\n");
      else if (cp->flg == 0)
	cp->cfun (tokc, tokv, input);
      else
	run_prog (tokc, tokv, pgms[cp->flg-1]);
    }
}


static int cmd_help (int argc, char *argv[], char *raw)
{
  tn_puts (t, "Command list:\r\n");
  tn_help (t, cmds);
  return 0;
}


static int cmd_quit (int argc, char *argv[], char *raw)
{
  tn_puts (t, "Good bye!\r\n");
  exit (0);
}


/* This function is called by exit(). */

static void cleanup (void)
{
  if (child_pid > 0)
    {
      kill (child_pid, SIGINT);
      child_pid = -1;
    }
}


static sig_atomic_t child_died;

static void sigchld (int signo)
{
  int status;

  /* As we have only one child process, handling SIGCHLD is simple. */

  wait (&status);
  child_died = 1; child_pid = -1;
}


static void run_prog (int argc, char *argv[], const char *pgm)
{
  int fd, pgm_output[2];

  /* Feed the program's output through a pipe to replace \n with
     \r\n. */

  if (pipe (pgm_output) != 0)
    {
      tn_printf (t, "pipe(): %s\r\n", strerror (errno));
      return;
    }

  if (pgm_output[0] >= FD_SETSIZE)
    {
      tn_printf (t, "file descriptor out of range for select()\r\n");
      close (pgm_output[0]); close (pgm_output[1]);
      return;
    }

  /* Ensure that pgm_output[1] is > 2.  This simplifies the code in
     the child process. */

  if (pgm_output[1] <= 2)
    {
      errno = 0;
      fd = fcntl (pgm_output[1], F_DUPFD, 3);
      if (fd < 3)
	{
	  tn_printf (t, "fcntl(F_DUPFD): %s\r\n", strerror (errno));
	  return;
	}
      close (pgm_output[1]);
      pgm_output[1] = fd;
    }

  /* Set up SIGCHLD. */

  child_died = 0;
  signal (SIGCHLD, sigchld);

  /* Create a child process. */

  child_pid = fork ();
  if (child_pid == -1)
    {
      tn_printf (t, "fork(): %s\r\n", strerror (errno));
      close (pgm_output[0]); close (pgm_output[1]);
    }
  else if (child_pid == 0)
    {
      /* Child process. */

      int e;
      char *s;

      signal (SIGCHLD, SIG_DFL);
      close (0); close (1); close (2);
      if (pgm_output[0] > 2)
	close (pgm_output[0]);
      errno = 0;
      if (open ("/dev/null", O_RDWR) != 0) /* This may fail in jail */
	syslog (LLEV, "/dev/null: %m");
      errno = 0;
      if (dup2 (pgm_output[1], 1) != 1)
	{
	  syslog (LLEV, "dup2(): %m");
	  exit (2);
	}
      errno = 0;
      if (dup2 (pgm_output[1], 2) != 2)
	{
	  syslog (LLEV, "dup2(): %m");
	  exit (2);
	}
      close (pgm_output[1]);
      /* TODO: Close all other file descriptors (authdb!) */
      argv[0] = (char *)pgm;
      argv[argc] = 0;		/* Don't trust enargv() */
      execv (pgm, argv);
      e = errno;
      write (1, pgm, strlen (pgm)); write (1, ": ", 2);
      s = strerror (e);
      write (1, s, strlen (s)); write (1, "\n", 1);
      exit (2);
    }
  else
    {
      /* Parent process. */

      fd_set rfds;
      struct timeval timeout;
      int n;

      tn_reset_dfa (t);
      close (pgm_output[1]);
      for (;;)
	{
	  FD_ZERO (&rfds);
	  FD_SET (pgm_output[0], &rfds);
	  FD_SET (0, &rfds);
	  timeout.tv_sec = child_died ? 0 : PROXY_TIMEOUT;
	  timeout.tv_usec = 0;
	  n = select (FD_SETSIZE, &rfds, (fd_set *)0, (fd_set *)0, &timeout);
	  if (n == -1)
	    {
	      if (errno == EINTR)
		continue;
	      syslog (LLEV, "select(): %m");
	      exit (1);
	    }
	  if (n == 0)
	    {
	      syslog (LLEV, "Timeout");
	      tn_puts (t, "Timeout\r\n");
	      exit (0);
	    }
	  if (FD_ISSET (0, &rfds))
	    {
	      int c = tn_getc (t);
	      tndfa dfa = tn_dfa (t, c);
	      if (dfa == TN_IP || (dfa == TN_CHAR && c == 0x03))
		kill (child_pid, SIGINT);
	    }
	  if (FD_ISSET (pgm_output[0], &rfds))
	    {
	      char c;		/* TODO: use bigger buffer, non-blocking */
	      n = read (pgm_output[0], &c, 1);
              if (n == -1 && errno == EINTR)
                continue;
              if (n <= 0)
                break;
	      if (c == '\n')
		tn_putnl (t);
	      else
		tn_putc (t, c);
	    }
	}
      close (pgm_output[0]);
      signal (SIGCHLD, SIG_DFL);
      child_pid = -1;
    }
}
