#       enparse - read RIPE database and check syntax errors
#
#	$RCSfile: enparse.pl,v $
#	$Revision: 2.0 $
#	$Author: ripe-dbm $
#	$Date: 1996/08/08 10:47:30 $
#
#	ARGUMENTS:	filehandle to read from
#	RETURNS:	INTEGER object_status
#			ASSOC   object
#
#	Object status = $O_OK, $O_WARNING, $O_ERROR, $EOF, or $NOK
#		$O_OK = object read and no errors/warnings generated
#		$O_WARNING = object read, but generated warnings
#		$O_ERROR = object read, but generated errors
#		$EOF = EndOfFile reached
#		$NOK = no object found, just garbage
#
#	Object has warnings and errors included.
#
#	This routine is a modified version of enread. It will read any
#	garbage, until it finds something of the form:
#	"xxxxxxx: " (no fixed length, spaces MUST be there)
#	or
#	"*xx: "
#	and then continues to read until it finds a line that does not
#	match these patterns. It then assumes it read an object, and will
#	start doing syntax checks.

require "entype.pl";
require "syntax.pl";
require "defines.pl";
require "adderror.pl";

sub readsomething {
    local($file)=@_;

    local($inentry)=$EOF;
    local(%entry)=();
    local($tag)="&&"; # this one needs to be initialized
    local(@lines)=();
    local(@continuedlines)=();
    
    local($newtag);

    while (<$file>) {
        
        #
        # read the password in global var
                
        if (/^\s*password\s*\:\s*(\S.*)$/i) {
           
           $PASSWORD=$1;
           $PASSWORD=~ s/\s*$//;
           
           print "password found: ", $PASSWORD, "\n" if ($opt_V);
           
           next;
           
        }
        
        s/\s/ /g;
        s/ +$//;
        
        #
        # skip error & warning attributes
                
        next if (/^ *(\*?ue|\*?uw|WARNING|\*ERROR\*) *\:/i);
        
        
	if (s/^ *(\*?[$VALIDATTRCHAR]+) *\: *//o) {
	   
	   $newtag=$1;
	   $newtag=~ s/^\*//;
	   $newtag=~ tr/A-Z/a-z/;
	   
	   s/ *$//;
	   
	   if (($tag eq $newtag) || ($tag eq $ATTL{$newtag}) || ($tag eq $ATTR{$newtag})) {
	      
	      if (s/ *\\$//) {
	         @continuedlines=($_);
	      }
	      elsif ($_) {
	      
	         push(@lines, $_);
	      
	      }
	      else {
	         push(@lines,"\n");
	      }
	       
	   }
	   else {
	      
              if ($entry{$tag}) {
                 $entry{$tag}=join("\n",$entry{$tag},@lines) if (scalar(@lines));
              }
              else {
                 $entry{$tag}=join("\n",@lines) if (scalar(@lines));
              }
               
              $tag=$ATTR{$newtag};
                
              # we have an unknown attribute 
              
              $tag=$newtag if (!$tag);
              
              if (s/ *\\$//) {
	         @continuedlines=($_);
	      }
              elsif ($_) {
                 
	         @lines=($_);

	      }
	      else {
	         @lines=();
	      }
                
	   }
	       
           $inentry=$O_OK;
           
           if (@continuedlines) {
	   
	      while (($_=<$file>) && (s/\s*\\\s*$//)) {
	       
	          s/\s/ /g;
                  s/^ +//;
               
	          push(@continuedlines, $_) if ($_);
	       
	      }
	   
	      s/\s/ /g;
              s/^ +//;
              s/ +$//;
	   
	      if (@continuedlines) {
	         if ($_) {
	            push(@lines, join(" ", @continuedlines, $_));
	         } 
	         else {
	            push(@lines, join(" ", @continuedlines));
	         }
	      }
	      else {
	         if ($_) {
	           push(@lines, $_);
	         }
	         else {
	           push(@lines, "\n");
	         }
	      }
	   
	      @continuedlines=();
	   
           }
	

	   next;
	
	}
        
        # skip comments

        next if (/^ *#/);
	
	# read further if we didn't find any object yet
	
        next if ($inentry==$EOF);
	
	# nothing useful found, end of object
	
	last;

    }
    
    if ($inentry==$O_OK) {
       
       if ($entry{$tag}) {
          $entry{$tag}=join("\n",$entry{$tag},@lines) if (scalar(@lines));
       }
       else {
          $entry{$tag}=join("\n",@lines) if (scalar(@lines));
       }
       
       #
       # delete warning/error attributes
       #
       # this is convenient for people who find errors
       # and resubmit the object
       
       delete($entry{"uw"});
       delete($entry{"ue"});
       
       # try to find a new object if we only found 
       # error/warning attributes
       
       return &enparse($file) if (!%entry);
       
       return ($inentry, %entry) if (%entry);
       
    }
    
    return $EOF;
    
}


sub checkobject	{
    local(*object, $type) = @_;

    local($rtcode)=$O_OK;
    local(%knownfield) = ();
    local(%mandfield) = ();
    local(%multfield) = ();
    local(%knownfield) = ();
    local(%usefield) = ();
    local($i);

    print STDERR "checkobject - called\n" if $opt_V;
    
    foreach $i ((split(/\s+/, $OBJATSQ{$type}),"ud","uo","uw","ue")) {
	$knownfield{$i}=1;
    }

    foreach $i (split(/\s+/, $OBJMAND{$type})) {
	$mandfield{$i} = 1;
    }

    foreach $i (split(/\s+/, $OBJMULT{$type})) {
	$multfield{$i} = 1;
    }

    foreach $i (keys %object) {
	$usefield{$i} = 1;
    }

    foreach (split(/\s+/, $OBS{$type})) {
	if ($object{$_}) {
	    &addwarning(*object,
			"attribute \"$ATTL{$_}\" has been obsoleted,".
			" value removed from object");
	    delete $object{$_};
	    delete $usefield{$_};
	    $rtcode=$O_WARNING if ($rtcode!=$O_ERROR);
	}
    }

    foreach $i (keys %usefield) {
	
	if (!$knownfield{$i}) {
	    if ($ATTL{$i}) {
		&adderror(*object,
			  "attribute \"$ATTL{$i}\" unknown ".
			  "in $ATTL{$type} object");
	    } else {
		&adderror(*object,
			  "attribute \"$i\" unknown in $ATTL{$type} object");
	    }
	    $rtcode = $O_ERROR;
	}

	undef $mandfield{$i} unless $object{$i} eq "";
	
	if ($object{$i}=~ /^\s*$/) {
	
	   delete $object{$i};
	   
	   &addwarning(*object,
	    	       "removed empty attribute: \"$ATTL{$i}\"");
	    	       
           $rtcode=$O_WARNING if ($rtcode!=$O_ERROR);
	
	}
	
	if ($object{$i}=~ /\n/) {
	    
	    if ($multfield{$i}) {
	       
	       if (($object{$i}=~ s/^\n+//) || ($object{$i}=~ s/\n+$//)) {
	          
	          &addwarning(*object,
	    	              "removed empty line(s) in attribute: \"$ATTL{$i}\"");
	    	  
	    	  $rtcode=$O_WARNING if ($rtcode!=$O_ERROR);
	       
	       }
	       
	    }
	    else {

		&adderror(*object,
			  "multiple lines not allowed for: \"$ATTL{$i}\"");
		$rtcode = $O_ERROR;

	    }

	}

    }

    foreach $i (keys %mandfield) {
	if ($mandfield{$i}) {
	    if (defined($object{$i}) && ($object{$i} =~ /^\n*$/)) {
		&adderror(*object,
			  "mandatory field \"$ATTL{$i}\" must have a value");
	    } else {
		&adderror(*object,
			  "mandatory field \"$ATTL{$i}\" missing");
	    }
	    $rtcode = $O_ERROR;
	}
    }
    
    print STDERR "checkobject - returned\n" if $opt_V;
    
    return $rtcode;
    
}

sub enparse {
    local($file) = @_;

    local(%entry);
    local($rtcode);
    local($stat,$type);
	
    print STDERR "enparse - reading something\n" if $opt_V;

    ($stat, %entry)=&readsomething($file);

    # print STDERR %entry, "\n";
    
    return $EOF if ($stat==$EOF);

    $type=&entype(*entry);
    
    # Now start the syntax checking
    #
    # we first do some checks that are always needed
    # even when we delete objects
    
    $rtcode=&requiredsyntaxcheck(*entry, $type);
    
    # print STDERR %entry, "\n";
    print STDERR "enparse - left requiredsyntaxcheck($rtcode)\n" if ($opt_V);
    
    #
    # do all other checks 
    
    if ((!defined($entry{"ud"})) && ($rtcode!=$O_ERROR)) {
       
       print STDERR "enparse - checking object format\n" if $opt_V;
       
       local($code)=&checkobject(*entry, $type);
       
       # print STDERR %entry, "\n";
       
       print STDERR "enparse - left checkobject($code)\n" if ($opt_V);
       
       $rtcode=$code if (($code==$O_ERROR) || (($code==$O_WARNING) && ($rtcode!=$O_ERROR)));
       
       if ($rtcode!=$O_ERROR) {
       
          print STDERR "enparse - checking object syntax\n" if $opt_V;
	  
	  $code=&checksyntax(*entry);
	  
	  # print STDERR %entry, "\n";
	  
	  print STDERR "enparse - left checksyntax($code)\n" if ($opt_V);
	  
	  $rtcode=$code if (($code==$O_ERROR) || (($code==$O_WARNING) && ($rtcode!=$O_ERROR)));
       
       }
       
    }
    
    # print STDERR %entry, "\n";
    
    return ($rtcode, $type, %entry);
    
    
}

1;
