/*
 Copyright (c) 2002 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

------------------------------------------------------------------------------
 Module Header
 Filename          : queue.c
 Purpose           : Determine the queue to send the submission to
 Author            : Can Bican <can@ripe.net>
 Description       : determines the queue # depending on the configuration
                     file and the submission itself
 Language Version  : C
 $Id: queue.c,v 1.12 2003/01/06 15:29:15 can Exp $
------------------------------------------------------------------------------
*/

#include "queue.h"

/*
 * Hash eliminator
*/
void freeHash(gpointer key, gpointer value, gpointer user_data)
{
  g_free(key);
  g_free(value);
}

/*
 * Hash dumper
 */
void logHash(gpointer key, gpointer value, gpointer user_data)
{
  gulong v;
  gchar *appLogDomain = (gchar *) user_data;

  v = (*(gulong *) value);
  g_log(appLogDomain, G_LOG_LEVEL_DEBUG, "%s=%lu", (gchar *) key, v);
}

/*
 * Return the numerical value of remote_addr
 */
gulong *whoConnects(void)
{
  gchar *remoteAddrStr;
  gulong remoteAddrl;

  remoteAddrStr = getenv("REMOTE_ADDR");
  if (remoteAddrStr == NULL)
  {
	  remoteAddrl=INADDR_LOOPBACK;
  }
  else
  {
	  remoteAddrl=inet_addr(remoteAddrStr);
  }
	  return(g_memdup(&remoteAddrl,sizeof(remoteAddrl)));
}

/*
 * Resets scores for rule entries
 */
void resetScores(gpointer data, gpointer user_data)
{
  queueDefinition *q;

  q = (queueDefinition *) data;
  q->matchCount = 0;
  q->matches = FALSE;
}

/*
 * Checks if the given operation returns true
 */
gboolean matchRule(glong lhs, char op, glong rhs)
{
  return (((op == '=')&&(lhs==rhs))||((op == '<')&&(lhs<rhs))||((op == '>')&&(lhs>rhs))||((op == '!')&&(lhs!=rhs)));
}

/*
   Determine the queue which the update will go to
*/
gint findQueue(gchar * data, gchar * appLogDomain)
{
  gchar *ruleNow, *tmpStr, *dataStr, *objectStr,*tmpStrToFree;
  gint queue = 0;
  gboolean finished = FALSE, ruleFound;
  gulong *objectCount , *remoteAddr;
  GHashTable *props;
  GSList *currentNode;
  queueDefinition *currentRule;
  gint regexStatus;
  gulong queueNow,*p;

  props = g_hash_table_new(g_str_hash, g_str_equal);
  objectCount=g_new(gulong,1);
  objectCount[0]=0;
  /* total and source are builtin properties */
  g_hash_table_insert(props, g_strdup("total"), objectCount);
  remoteAddr = whoConnects();
  g_hash_table_insert(props, g_strdup("source"), remoteAddr);
  /* Count objects - start */
  tmpStr = g_strdup(data);
  tmpStrToFree=tmpStr;
  dataStr = tmpStr;
  /* remove \r's */
  while (tmpStr[0] != 0) {
    while (tmpStr[0] == '\r') {
      g_snprintf(tmpStr, MAXDATALEN, "%s", tmpStr + 1);
    }
    tmpStr++;
  }
  /* remove extra \n's */
  tmpStr = dataStr;
  while (tmpStr[0] != 0) {
    while (strncmp(tmpStr, "\n\n\n", 3) == 0) {
      g_snprintf(tmpStr, MAXDATALEN, "%s", tmpStr + 1);
    }
    tmpStr++;
  }
  /* count objects and properties */
  g_slist_foreach(queueDefinitions, resetScores, NULL);
  while ((!finished) && (dataStr[0] != 0)) {
    objectStr = strstr(dataStr, "\n\n");
    if (objectStr != NULL) {
      objectStr[0] = 0;
    } else {
      finished = TRUE;
    }
    currentNode = queueDefinitions;
    while (currentNode != NULL) {
      currentRule = ((queueDefinition *) currentNode->data);
      currentNode = currentNode->next;
      if (currentRule->isRegex) {
	regexStatus =
	    regexec(&currentRule->opLhsRegex, dataStr, (size_t) 0, NULL,
		    0);
	if (regexStatus == 0) {
	  currentRule->matchCount++;
	}
      } else {
	currentRule->matchCount=0;
      }
    }
    dataStr = objectStr;
    objectCount[0]++;
    dataStr = objectStr;
    dataStr += 2;
  }
  /* Tag matching rule items */
  currentNode = queueDefinitions;
  while (currentNode != NULL) {
    currentRule = (queueDefinition *) currentNode->data;
    currentNode = currentNode->next;
    if (!(currentRule->isRegex)) {
      p = g_hash_table_lookup(props, currentRule->opLhs);
      if (p != NULL) {
	currentRule->matchCount = (*p);
      }
    }
    if ((matchRule(currentRule->matchCount, currentRule->op, currentRule->opRhs)==TRUE))
      currentRule->matches = TRUE;
  }
  /* Determine the queue */
  queue = (-1);
  currentNode = queueDefinitions;
  while (currentNode != NULL) {
    currentRule = (queueDefinition *) currentNode->data;
    ruleNow = currentRule->ruleName;
    queueNow = currentRule->queueNumber;
    ruleFound = TRUE;
    while ((currentNode != NULL)
	   &&
	   (strcmp
	    (((queueDefinition *) currentNode->data)->ruleName,
	     ruleNow) == 0)&&(((queueDefinition*)currentNode->data)->queueNumber)==queueNow) {
      if (!(((queueDefinition *) currentNode->data)->matches))
	ruleFound = FALSE;
      currentNode = currentNode->next;
    }
    if (ruleFound) {
      queue = currentRule->queueNumber;
  g_log(appLogDomain, G_LOG_LEVEL_DEBUG,"Queue #%ld rule %s matches", currentRule->queueNumber,
	     currentRule->ruleName);
      break;
    }
  }
  g_free(tmpStrToFree);
  /* log findings and return the queue */
  g_log(appLogDomain, G_LOG_LEVEL_DEBUG,
	"Object properties in update follows");
  g_hash_table_foreach(props, logHash, appLogDomain);
  g_log(appLogDomain, G_LOG_LEVEL_INFO, "sending to queue #%d", queue);
  g_hash_table_foreach(props, freeHash, NULL);
  g_hash_table_destroy(props);
  return --queue;
}
