
/*
     Program PM-Info: a program for viewing GNU-style hypertext info
     documentation files.
     Copyright (C) 1992,1993  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#define INCL_GPIPRIMITIVES
#include <os2.h>
#include <ctype.h>

#include "bugme.h"
#include "attr.h"
#include "view.h"

TextRegion::TextRegion(TextPoint astart, TextPoint aend)
    : start(astart),
      end(aend)
{
}

int TextRegion::contains(TextPoint pt)
{
    if (start.line > pt.line) return 0;
    if (end.line < pt.line) return 0;
    if (start.line == pt.line && start.position > pt.position) return 0;
    if (end.line == pt.line && end.position < pt.position) return 0;
    return 1;
}

Attribute::Attribute(TextRegion aregion)
    : region(aregion)
{
}

Attribute::~Attribute()
{
}

TextAttributes::TextAttributes()
    : allocated(0),
      current(0),
      attr(NULL)
{
}

void TextAttributes::add_attribute(Attribute *new_attribute)
{
    if (new_attribute == NULL) return;  // Why are we here? Because we're here

    if (allocated == 0) {
	allocated = 16;
	attr = (Attribute **) malloc(sizeof(Attribute *) * allocated);
    }
    while (current+1 >= allocated) {
	allocated += 16;
	attr = (Attribute **) realloc(attr, sizeof(Attribute *) * allocated);
    }
    attr[current++] = new_attribute;
}

TextAttributes::~TextAttributes()
{
    int i;
    for (i = 0; i < current; i++) {
	delete attr[i];
    }
    free(attr);
}

TextPoint TextAttributes::next_transition(TextPoint start_point)
{
    start_point.position++;

    int i;
    TextPoint next_point(INT_MAX, INT_MAX);
    TextRegion current_region(start_point, next_point);

    for (i = 0; i < current; i++) {
	TextRegion attr_region = attr[i]->region;
	TextPoint transition1 = attr_region.start;
	TextPoint transition2 = attr_region.end;
	transition2.position++;
	if (current_region.contains(transition1)) {
	    next_point = transition1;
	    current_region = TextRegion(start_point, next_point);
	} else if (current_region.contains(transition2)) {
	    next_point = transition2;
	    current_region = TextRegion(start_point, next_point);
	}
    }
    return next_point;
}

void TextAttributes::set_attributes(TextPoint pt)
{
    int i;

    for (i = 0; i < current; i++) {
	if (attr[i]->region.contains(pt)) {
	    attr[i]->set();
	}
    }
}

void TextAttributes::set_button(TextPoint pt)
{
    int i;

    for (i=0; i < current; i++) {
	if (attr[i]->region.contains(pt)) {
	    attr[i]->button();
	}
    }
}



HPS VisibleAttribute::hps;

void VisibleAttribute::set_default_attributes(HPS ahps)
{
    hps = ahps;
    GpiSetAttrs(hps, PRIM_CHAR, (unsigned long) -1, (unsigned long) -1, NULL);
}

void Bold::set()
{
    CHARBUNDLE attr;
    attr.lColor = CLR_BLUE;
    GpiSetAttrs(hps, PRIM_CHAR, CBB_COLOR, 0, (PBUNDLE) &attr);
}

void Color::set()
{
    CHARBUNDLE attr;
    attr.lColor = color;
    GpiSetAttrs(hps, PRIM_CHAR, CBB_COLOR, 0, (PBUNDLE) &attr);
}

RedAttr::RedAttr(TextRegion r)
    : Color(CLR_DARKCYAN, r)
{
}

NodeReference::NodeReference(TextRegion r, const char *reference)
    : Bold(r),
      nodename(NULL),
      refname(NULL)
{
    BUGME(("NodeReference::NodeReference([(%d,%d),(%d,%d)],\"%s\")",
	   r.start.line, r.start.position,
	   r.end.line, r.end.position,
	   reference));
    if (reference == NULL) {
	BUGME(("NodeReference::NodeReference: Ref is NULL"));
	return;
    }
    while(isspace(*reference)) reference++;
    char *colon = strchr(reference, ':');
    if (colon == NULL) {
	BUGME(("NodeReference::NodeReference: No reference found"));
	return;
    }

    int len = colon-reference;
    BUGME(("NodeReference::NodeReference: Ref length is %d", len));
    refname = (char *) malloc(len+1);
    strncpy(refname, reference, len);
    refname[len] = '\0';
    BUGME(("refname: \"%s\"", refname));

    colon++;
    if (*colon == ':') {
	nodename = strdup(refname);
    } else {
	while(isspace(*colon)) colon++;
	char *p = colon;
	while (*p != '0' && strchr("\t\n,.", *p) == NULL) {
	    if (*p == '(') {
		// Skip the filename component
		while (*p != ')' && *p != '\0') p++;
	    } else {
		p++;
	    }
	}
	len = p - colon;
	nodename = (char *) malloc(len+1);
	strncpy(nodename, colon, len);
	nodename[len] = '\0';
    }
    BUGME(("nodename: \"%s\"", nodename));
}

NodeReference::~NodeReference()
{
    if (nodename != NULL) free(nodename);
    if (refname != NULL) free(refname);
}

char *NodeReference::button_nodename = NULL;

void NodeReference::button()
{
    button_nodename = nodename;
}

SimpleView *MenuItem::all_items = NULL;
SimpleView *MenuItem::all_nodenames = NULL;

MenuItem::MenuItem(TextRegion r, const char *reference)
    : NodeReference(r, reference)
{
    if (all_items == NULL) {
	all_items = new SimpleView;
	all_nodenames = new SimpleView;
    }
    all_items->append(reference);
    all_nodenames->append(nodename);
    BUGME(("Inserting menu ``%s'' -> ``%s''", reference, nodename));
}

MenuItem::~MenuItem()
{
    if (all_items != NULL) delete all_items;
    all_items = NULL;
    if (all_nodenames != NULL) delete all_nodenames;
    all_nodenames = NULL;
}

SimpleView *ReferenceItem::all_items = NULL;
SimpleView *ReferenceItem::all_nodenames = NULL;

ReferenceItem::ReferenceItem(TextRegion r, const char *reference)
    : NodeReference(r, reference)
{
    BUGME(("ReferenceItem::ReferenceItem(...)"));
    if (nodename == NULL) {
	BUGME(("ReferenceItem::ReferenceItem: node name bogosity"));
	return;
    }
    if (all_items == NULL) {
	all_items = new SimpleView;
	all_nodenames = new SimpleView;
    }
    all_items->append(reference);
    all_nodenames->append(nodename);
    BUGME(("Inserting reference ``%s'' -> ``%s''", reference, nodename));
}

ReferenceItem::~ReferenceItem()
{
    delete all_items;
    all_items = NULL;
    delete all_nodenames;
    all_nodenames = NULL;
}
