#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#define INCL_DOSPROCESS
#include <os2.h>

typedef struct name_list {
    struct name_list *next;
    char *name;
} NameList;

static int verbose_flag = 0;
static int debug_flag = 0;
static int nodefaultlibs_flag = 0;

static void append_str(NameList **list, char *str)
{
    while (*list != NULL) list = &(*list)->next;
    (*list) = malloc(sizeof(NameList));
    (*list)->next = NULL;
    (*list)->name = str;
}

static char *convert_filename(char *filename)
{
    char *ret = filename = strdup(filename);
    while (*filename != '\0') {
	if (*filename == '/') *filename = '\\';
	filename++;
    }
    return ret;
}

static int suffix(const char *filename, const char *ext)
{
    int l_filename = strlen(filename);
    int l_ext = strlen(ext);
    if (l_filename <= l_ext) return 0;
    return !strcmp(ext, filename + l_filename - l_ext);
}

static int file_exists(const char *filename)
{
    FILE *f = fopen(filename, "rb");
    if (f == NULL) return 0;
    fclose(f);
    return 1;
}

static char *convert_unix_filename(const char *dot_o)
{
    char *dot_obj;
    if (!suffix(dot_o, ".o")) return strdup(dot_o);
    dot_obj = malloc(strlen(dot_o) + 3);
    sprintf(dot_obj, "%sbj", dot_o);
    return dot_obj;
}

static char *make_lib_filename(const char *libname)
{
    char *lib = malloc(strlen(libname)+5);
    sprintf(lib, "%s.lib", libname);
    return lib;
}

NameList *lib_search_path = 0;
int max_lib_search_path_name_length = 0;

static void init_lib_search_path(NameList *explicit_search_paths)
{
  NameList *list;
  const char *libpath;
  char *libparts, *p;

  /* Always search the current path first (?) */
  append_str (&lib_search_path, ".");
  
  /* Copy the explicitly provided search paths into our search path */
  for (; explicit_search_paths != 0; 
       explicit_search_paths = explicit_search_paths->next)
    {
      append_str (&lib_search_path, 
		  convert_filename (explicit_search_paths->name));
    }

  /* Copy paths provided via environment variable into our search path */
  libpath = getenv ("LIB");
  if (libpath != 0)
    {
      libparts = strdup (libpath);
      for (p = strtok (libparts, ";"); p != NULL; p = strtok (NULL, ";"))
	append_str (&lib_search_path, convert_filename (p));
    }

  for (list = lib_search_path; list != NULL; list = list->next)
    {
      int len = strlen (list->name);
      if (len > max_lib_search_path_name_length)
	max_lib_search_path_name_length = len;
    }
}

static char *find_lib_path(const char *lib)
{
  NameList *search = lib_search_path;
  char pathbuf [strlen(lib) + max_lib_search_path_name_length + 2];

  /* Leave absolute paths alone */
  if (strchr (lib, '\\') != 0)
    return strdup (lib);
  /* Search for floating libs */
  for (search = lib_search_path; search; search = search->next)
    {
      sprintf(pathbuf, "%s\\%s", search->name, lib);
      if (access (pathbuf, F_OK) == 0)
	return strdup (pathbuf);
    }
  return 0;
}

static NameList *find_lib_paths(NameList *list)
{
  NameList *newlist;
  char *path;
  if (list == 0)
    return 0;
  path = find_lib_path(list->name);
  if (path == 0)
    {
      fprintf (stderr, "ld: Error: Unable to find library file `%s'\n",
	       list->name);
      exit (0);
    }
  newlist = NULL;
  append_str (&newlist, path);
  newlist->next = find_lib_paths (list->next);
  return newlist;
}

static NameList *permute_lib_paths_2(NameList *paths, const char *permution)
{
  NameList *newpath = 0;
  char buf[((paths == 0) ? 0 : strlen (paths->name)) + strlen (permution) + 1];
  if (paths == NULL)
    return NULL;
  strcpy (buf, paths->name);
  strcat (buf, permution);
  append_str (&newpath, paths->name);
  append_str (&newpath, strdup (buf));
  newpath->next->next = permute_lib_paths_2 (paths->next, permution);
  return newpath;
}

static void permute_lib_paths(const char *permution)
{
  lib_search_path = permute_lib_paths_2 (lib_search_path, permution);
  max_lib_search_path_name_length += strlen(permution);
}

static char *make_flat(NameList *list)
{
    int len = 0;
    NameList *l2;
    char *flat;
    char *p;

    if (list == NULL) return "";
    for (l2 = list; l2 != NULL; l2 = l2->next)
	len += 1 + strlen(l2->name);
    p = flat = malloc(len);
    for (l2 = list; l2 != NULL; l2 = l2->next) {
	sprintf(p, "%s ", l2->name);
	p += strlen(p);
    }
    *--p = '\0';
    return flat;
}

static int link386(NameList *objs, const char *exe, NameList *libs, 
		   const char *def, const char *map)
{
    RESULTCODES result;
    int code;
    char *objarg = make_flat(objs);
    char *libarg = make_flat(libs);
    char *sys_format = 
	"link386.exe /BATCH /NOLOGO /NOI %s %s %s %s,%s,%s,%s,%s,";
    char *windowcompat = (*def=='\0') ? "/PM:VIO /ST:2000000" : "";
    char *debugflag = 
      (debug_flag) 
	? (map ? "/DEBUG /MAP:full" : "/DEBUG")
	  : (map ? "/MAP:full" : "");
    char *nodeflibsflag = 
	(nodefaultlibs_flag) ? "/NODEFAULTLIBRARYSEARCH" : "";
    char sys[strlen(sys_format)+strlen(objarg)+strlen(exe)+strlen(libarg)
	     +strlen(def)+strlen(windowcompat)+strlen(debugflag)
	     +strlen(nodeflibsflag)
	     + ((map == 0) ? 3 : strlen(map))];
    if (sizeof(sys) > 200) {
	/* Too many args - make a response file */
	FILE *f;
	char filename[512];
	NameList *name;
	char *t;
	char *tmp_dir = getenv("TMPDIR");
	if (tmp_dir == NULL) tmp_dir = "";
	sprintf(filename, "@%s/ldXXXXXX", tmp_dir);
	mktemp(filename+1);
	f = fopen(filename+1, "wt");
	if (f == NULL) return 1;
	for (name = objs; name != NULL; name = name->next)
	    fprintf(f, "%s +\n", name->name);
	fprintf(f, "\n%s\n%s\n", exe, (map == NULL) ? "nul" : map);
	if (libs != NULL) {
	    for (name = libs; name != NULL; name = name->next)
		fprintf(f, "%s +\n", name->name);
	}
	fprintf(f, "\n%s\n", def);
	fclose(f);
	t = sys;
	strcpy(t, "link386.exe");
	t += strlen(t) + 1;
	sprintf(t, "/BATCH /NOLOGO /NOI %s %s %s", windowcompat, debugflag,
		nodeflibsflag);
	t += strlen(t);
	strcpy(t, convert_filename(filename));
	t += strlen(t) + 1;
	*t = '\0';
	if (verbose_flag) {
	    fprintf(stderr, "\t%s %s\n", sys, sys+strlen(sys)+1);
	}
	code = DosExecPgm(NULL, 0, EXEC_SYNC, (PSZ) sys, NULL, &result,
			  (PSZ) sys);
	unlink(filename+1);
	if (code != 0) {
	    fprintf(stderr, "ld: Unable to execute link386: Error code %d\n",
		    code);
	    return 1;
	}
	if (result.codeTerminate != TC_EXIT) {
	    fprintf(stderr, "ld: link386 terminated abnormally\n");
	    return 1;
	} 
	return result.codeResult;
    } else {
	sprintf(sys, sys_format, windowcompat, debugflag, nodeflibsflag, 
		objarg, exe, (map == NULL) ? "nul" : map, libarg, def);
	if (verbose_flag)
	    fprintf(stderr, "\t%s\n", sys);
	sys[strlen(sys)+1] = '\0';
	sys[11] = '\0';
	code = DosExecPgm(NULL, 0, EXEC_SYNC, (PSZ) sys, NULL, &result, 
			  (PSZ) sys);
	if (code != 0) {
	    fprintf(stderr, "ld: Unable to execute link386: Error code %d\n",
		    code);
	    return 1;
	}
	if (result.codeTerminate != TC_EXIT) {
	    fprintf(stderr, "ld: link386 terminated abnormally\n");
	    return 1;
	} 
	return result.codeResult;
    }
}

int main(int argc, char **argv)
{
    int i;
    char *arg;
    NameList *libs = NULL;
    NameList *objs = NULL;
    NameList *lib_paths = NULL;
    char *output_file = NULL;
    char *link_def_file = NULL;
    char *map = NULL;
    char *p;
    NameList *n;
    int static_flag = 0;
    
    void getarg(void)
    {
	if (argv[i][2] == '\0') {
	    i++;
	    if (i >= argc) {
		fprintf(stderr, "ld: Fatal error: -%c requires an argument\n",
			argv[i][1]);
		exit(1);
	    }
	    arg = strdup(argv[i]);
	} else {
	    arg = strdup(argv[i]+2);
	}
    }
    
    for (i=1; i < argc; i++) {
	argv[i] = convert_filename(argv[i]);
	if (!strcmp(argv[i], "-no-default-library-search")) {
	    nodefaultlibs_flag = 1;
	  } else if (!strcmp(argv[i], "-static")) {
	    static_flag = 1;
	    
	} else if (argv[i][0] == '-') {
	    switch(argv[i][1]) {
	    case 'o':
		getarg();
		if (output_file != NULL) {
		    fprintf(stderr, 
			    "ld: Fatal error: Too many output filenames\n");
		    exit(1);
		}
		output_file = arg;
		break;
	    case 'L':
		getarg();
		append_str(&lib_paths, convert_filename(arg));
		break;
	    case 'l':
		getarg();
		append_str(&libs, make_lib_filename(arg));
		break;
	    case 'v':
		if (argv[i][2] != '\0') {
		    fprintf(stderr, "ld: Fatal error: Unrecognized option "
			    "`%s'\n", argv[i]);
		    exit(1);
		}
		verbose_flag = 1;
		break;
	    case 'g':
		debug_flag = 1;
		break;
	    default:
		fprintf(stderr, "ld: Fatal error: Unrecognized option `-%c'\n",
			argv[i][1]);
		exit(1);
	    }
	    
	} else if (suffix(argv[i], ".lib")) {
	    /* Library files */
	    append_str(&libs, convert_filename (argv[i]));
	    
	} else if (suffix(argv[i], ".obj")) {
	    /* Object files */
	    append_str(&objs, argv[i]);
	    
	} else if (suffix(argv[i], ".o")) {
	    /* UNIX object filename - Try to be reasonable */
	    if (file_exists(argv[i])) {
		fprintf(stderr, "ld: Warning: "
			"Linking object file with UNIX suffix: `%s'\n", 
			argv[i]);
		append_str(&objs, argv[i]);
	    } else {
		p = convert_unix_filename(argv[i]);
		if (file_exists(p)) {
		    fprintf(stderr, 
			    "ld: Warning: Linking `%s' instead of `%s'\n",
			    p, argv[i]);
		    append_str(&objs, p);
		} else {
		    fprintf(stderr, "ld: Warning: "
			    "Linking object file with UNIX suffix: `%s'\n", 
			    argv[i]);
		    append_str(&objs, argv[i]);
		}
	    }

	} else if (suffix(argv[i], ".def")) {
	    /* Linker definition file */
	    if (link_def_file != NULL) {
		fprintf(stderr, 
			"ld: Fatal error: Two linker definition files passed "
			"to ld\n");
		exit(1);
	    }
	    link_def_file = argv[i];

	} else if (suffix(argv[i], ".map")) {
	    /* Map file */
	    if (map != NULL) {
		fprintf(stderr,
			"ld: Fatal error: Two map files passed to ld\n");
		exit(1);
	    }
	    map = argv[i];
	    
	} else {
	    /* Unknown filename - Warn and add to object list */
	    fprintf(stderr, "ld: Warning: Assuming `%s' is an object file\n", 
		    argv[i]);
	    append_str(&objs, argv[i]);
	}
    }

    if (output_file == NULL) output_file = "aout.exe";

    /* Init the library search path */
    init_lib_search_path(lib_paths);

    /* Permute the libpath for dynamic vs. static linking */
    permute_lib_paths(static_flag ? "\\static" : "\\dynamic");

    /* Verify the existence of the obj files */
    for (n = objs; n != NULL; n = n->next) {
	if (!file_exists(n->name)) {
	    fprintf(stderr, "ld: Warning: `%s' does not exist\n", n->name);
	}
    }

    /* Verify the existence of the def file */
    if (link_def_file == NULL) {
	link_def_file = "";
    } else if (!file_exists(link_def_file)) {
	fprintf(stderr, "ld: Warning: `%s' does not exist\n", link_def_file);
    }

    /* Find the absolute path to the library files */
    libs = find_lib_paths(libs);

    if (verbose_flag)
	fprintf(stderr, "os2ld: Version 1.2\n");

    return link386(objs, output_file, libs, link_def_file, map);
}

