/* Plug-n-Play Sound Thread by Semir Patel (spatel@cs.utexas.edu) */

#define INCL_OS2MM
#define INCL_MACHDR
#define INCL_WIN

#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include <mem.h>
#include <os2me.h>
#include <process.h>

#include "skeleton.h"
#include "sound.h"

#define NUMBER_OF_COMMANDS      2
#define NUMBER_OF_WAVES         3

#define MCIERROR if(ulError){ulError=mciGetErrorString(ulError,(PSZ)&acErrorStringBuffer[ 0 ],(USHORT) 255 ); ShowMessageBox (acErrorStringBuffer);}


/*************************** Function Prototypes ******************************/

MRESULT EXPENTRY SoundWndProc (HWND,ULONG,MPARAM,MPARAM);
void    SoundThread ();
void    InitStrucs ();
void    ReadWavesIntoMem ();
void    OpenSoundDevice ();
void    AdjustSoundDevice (USHORT);
void    PlayWave (USHORT);
void    CloseSoundDevice ();
void    ShowMessageBox (char *);

/*************************** Globals & Strucs *********************************/

typedef struct pls      /* playlist structure */
{
   ULONG ulCommand;
   ULONG ulOperandOne;
   ULONG ulOperandTwo;
   ULONG ulOperandThree;
} PLAY_LIST_STRUCTURE;

/* example declaration for a memory playlist with 2 commands */
PLAY_LIST_STRUCTURE PlayList [NUMBER_OF_COMMANDS] =
{
        DATA_OPERATION,    0, 0, 0,       /* play command       */
        EXIT_OPERATION,    0, 0, 0        /* terminate playlist */
};

struct SOUNDINFO {
        char    szFileName[255];
        LONG    *Address;
        ULONG   ulSize;
        ULONG   ulSamplesPerSec;
        USHORT  usBitsPerSample;
        } SoundFiles[NUMBER_OF_WAVES];

struct SOUNDDEVICE {
        USHORT  usSoundDeviceID;
        ULONG   ulSamplesPerSec;
        USHORT  usBitsPerSample;
        } SoundDevice;;

MCI_OPEN_PARMS  mciOpenParameters;                   /* Open structure.       */
MMAUDIOHEADER   mmAudioHeader;                       /* Contains info for SET */
ULONG           ulError;
char            acErrorStringBuffer[255];
HAB             habSound;
HWND            hwndSound;
/*******************************************************/
/*                    SoundThread                      */
/*******************************************************/
void SoundThread ()
{
    HMQ     hmq;
    HWND    hwndObject,
            hwndParent;
    QMSG    qmsg;

    habSound = WinInitialize (0);

    hmq = WinCreateMsgQueue(habSound, 0);

    WinRegisterClass(habSound,
                     "Sound Server",
                     SoundWndProc,
                     CS_SAVEBITS | CS_MOVENOTIFY | CS_CLIPCHILDREN | CS_CLIPSIBLINGS,
                      0);

    hwndSound = WinCreateWindow(HWND_OBJECT,
                                 "Sound Server",
                                 "",0, 0, 0, 0, 0,
                                 HWND_OBJECT,
                                 HWND_BOTTOM,
                                 0, NULL, NULL );

    while( WinGetMsg ( habSound, &qmsg, 0, 0, 0 ))
        WinDispatchMsg ( habSound,&qmsg );

    WinPostMsg( hWndClient, WM_QUIT, 0, 0 );

    WinDestroyWindow( hwndSound );
    WinDestroyMsgQueue( hmq );
    WinTerminate (habSound);
    _endthread ();
}

/*******************************************************/
/*                SoundWndProc                         */
/*******************************************************/
MRESULT EXPENTRY SoundWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    ULONG   mciRetval;


    switch (msg)
    {
        case WM_CREATE:
                InitStrucs();
                ReadWavesIntoMem();
                OpenSoundDevice();
                WinPostMsg (hWndClient, WM_SOUNDS_READY, NULL,NULL);
                break;

        case WM_PLAY:
                PlayWave((USHORT)mp1);
                break;

        case WM_TERMINATE:
                CloseSoundDevice();
                WinPostMsg (hwndSound,WM_QUIT,(MPARAM)NULL,(MPARAM)NULL);
                break;
    }
    return WinDefWindowProc (hwnd, msg, mp1, mp2);
}

/*******************************************************/
/*                    SoundThread                      */
/*******************************************************/
void InitStrucs()
{
    strcpy (SoundFiles[0].szFileName,"doh.wav");
    strcpy (SoundFiles[1].szFileName,"woohoo.wav");
    strcpy (SoundFiles[2].szFileName,"terminat.wav");

    SoundDevice.ulSamplesPerSec = 0;
    SoundDevice.usBitsPerSample = 0;
}

/****************************************************************************
 * Name        : ReadWavesIntoMem                                           *
 *               - Create memory buffers for the waveform files.            *
 *               - Load the sound files into the buffers( mmioRead ).       *
 ***************************************************************************/
void ReadWavesIntoMem()
{
   HMMIO    hmmioFileHandle;              /* Handle to the audio file handle.    */
   ULONG    ulRC;                         /* Return code from mmioGetHeader.     */
   ULONG    ulBytesRead;                  /* Returned from mmioGetHeader.        */
   LONG     lMMIOResults;                 /* Results from the MMIO calls.        */
   USHORT   usSoundFileID;

   /* loop thru sound files and mciOpen,malloc,mciRead,mciClose 'em */
   for(usSoundFileID=0; usSoundFileID<NUMBER_OF_WAVES; usSoundFileID++)
   {
        /* Open the wave file.*/
        hmmioFileHandle = mmioOpen(SoundFiles[usSoundFileID].szFileName,(PMMIOINFO) NULL,MMIO_READ );

        /* make sure file was opened properly */
        if ( hmmioFileHandle == (HMMIO) NULL )
            ShowMessageBox ("Error opening file");

        /* Get the header information from the waveform file. */
        ulRC = mmioGetHeader (hmmioFileHandle,(PVOID)&mmAudioHeader,sizeof (MMAUDIOHEADER),(PLONG) &ulBytesRead,(ULONG) NULL,(ULONG) NULL);

        /* make sure header information successful */
        if (ulRC != MMIO_SUCCESS) ShowMessageBox ("Error obtaining header info");

        /* assign size of audio file */
        SoundFiles[usSoundFileID].ulSize          = mmAudioHeader.mmXWAVHeader.XWAVHeaderInfo.ulAudioLengthInBytes;
        SoundFiles[usSoundFileID].ulSamplesPerSec = mmAudioHeader.mmXWAVHeader.WAVEHeader.ulSamplesPerSec;
        SoundFiles[usSoundFileID].usBitsPerSample = mmAudioHeader.mmXWAVHeader.WAVEHeader.usBitsPerSample;

        /* malloc size of file bytes */
        if ((SoundFiles[usSoundFileID].Address = (LONG *) malloc(SoundFiles[usSoundFileID].ulSize)) == (LONG *) NULL )
                ShowMessageBox ("The memory cannot be allocated");

        /* Move the data from the wave file into the buffer. */
        lMMIOResults = mmioRead(hmmioFileHandle,(PSZ)SoundFiles[usSoundFileID].Address, SoundFiles[usSoundFileID].ulSize);

        /* check for error */
        if ( lMMIOResults == MMIO_ERROR )
            ShowMessageBox ("Error transferring wavefile into memory");

        /* Close the file */
        lMMIOResults = mmioClose(hmmioFileHandle,0 );

        /* check for error */
        if ( lMMIOResults != MMIO_SUCCESS )
            ShowMessageBox ("Error closing file");
   }
}


/******************************************************************************
 *                     OpenSoundDevice                                        *
 *****************************************************************************/
void OpenSoundDevice()
{

   ULONG    ulOpenFlags = MCI_WAIT         | MCI_OPEN_PLAYLIST |
                          MCI_OPEN_TYPE_ID | MCI_OPEN_SHAREABLE;

   /* Open the correct waveform device for the waves with MCI_OPEN */
   mciOpenParameters.pszDeviceType = (PSZ) MAKEULONG ( MCI_DEVTYPE_WAVEFORM_AUDIO, 1 );

   /* The address of the buffer containing the waveform file. */
   mciOpenParameters.pszElementName = (PSZ)&PlayList[0];

   mciOpenParameters.hwndCallback  = (HWND) NULL;
   mciOpenParameters.pszAlias      = (CHAR) NULL;

   /* Open the waveform file in the playlist mode. */
   ulError = mciSendCommand(0,                         /* We don't know the device yet.        */
                          MCI_OPEN,                    /* MCI message.                         */
                          ulOpenFlags,                 /* Flags for the MCI message.           */
                          (PVOID) &mciOpenParameters,  /* Parameters for the message.          */
                          0 );                         /* Parameter for notify message.        */
   MCIERROR;

   /* save device ID */
   SoundDevice.usSoundDeviceID = mciOpenParameters.usDeviceID;
}


/*****************************************************************************
 *                     AdjustSoundDevice                                     *
 ****************************************************************************/
void AdjustSoundDevice (USHORT usSoundFileID)
{
    MCI_WAVE_SET_PARMS mwspWaveFormParameters;   /* Waveform parameters.       */

    /* set samples per sec */
    if ((SoundFiles[usSoundFileID].ulSamplesPerSec != SoundDevice.ulSamplesPerSec) ||
        (SoundFiles[usSoundFileID].usBitsPerSample != SoundDevice.usBitsPerSample))
    {

         /* Fill the structure with zeros. */
         memset( &mwspWaveFormParameters, 0, sizeof(mwspWaveFormParameters));

         SoundDevice.ulSamplesPerSec = SoundFiles[usSoundFileID].ulSamplesPerSec;
         SoundDevice.usBitsPerSample = SoundFiles[usSoundFileID].usBitsPerSample;

         /* copy samps/sec */
         mwspWaveFormParameters.ulSamplesPerSec = SoundDevice.ulSamplesPerSec;
         mwspWaveFormParameters.usBitsPerSample = SoundDevice.usBitsPerSample;
         mwspWaveFormParameters.ulAudio         = MCI_SET_AUDIO_ALL;

         ulError = mciSendCommand(SoundDevice.usSoundDeviceID,      /* Device to play the waves.     */
                                  MCI_SET,                          /* MCI message.                  */
                                  MCI_WAIT |
                                  MCI_WAVE_SET_SAMPLESPERSEC |      /* Flags for the MCI message.    */
                                  MCI_WAVE_SET_BITSPERSAMPLE ,
                                  (PVOID) &mwspWaveFormParameters,  /* Parameters for the message.   */
                                  0);                               /* Parameter for notify message. */

         MCIERROR;
    }
}


/*****************************************************************************
 *                     PlayWave                                              *
 ****************************************************************************/
VOID PlayWave (USHORT usSoundFileID)
{

   /* setup wave file in playlist */
   PlayList[0].ulOperandOne = (ULONG)SoundFiles[usSoundFileID].Address;
   PlayList[0].ulOperandTwo = SoundFiles[usSoundFileID].ulSize;

   /* make sure bit/samp and samp/sec match for wave and sound device */
   AdjustSoundDevice (usSoundFileID);

   ulError = mciSendCommand(SoundDevice.usSoundDeviceID,    /* Device to play the waves.     */
                            MCI_STOP,                       /* MCI message.                  */
                            MCI_WAIT,                       /* Flags for the MCI message.    */
                            (PVOID) &mciOpenParameters,     /* Parameters for the message.   */
                            usSoundFileID);                 /* Parameter for notify message. */

   /* rewind sound */
   ulError = mciSendCommand(SoundDevice.usSoundDeviceID,    /* Device to play the waves.     */
                            MCI_SEEK,                       /* MCI message.                  */
                            MCI_TO_START,                   /* Flags for the MCI message.    */
                            (PVOID) &mciOpenParameters,     /* Parameters for the message.   */
                            usSoundFileID);                 /* Parameter for notify message. */
   MCIERROR;

   /* play sound */
   ulError = mciSendCommand(SoundDevice.usSoundDeviceID,  /* Device to play the waves.     */
                            MCI_PLAY,                     /* MCI message.                  */
                            MCI_WAIT,                     /* Flags for the MCI message.    */
                            (PVOID) &mciOpenParameters,   /* Parameters for the message.   */
                            usSoundFileID);               /* Parameter for notify message. */
    MCIERROR;
}

/******************************************************************************/
/*                          CloseSoundDevice                                  */
/******************************************************************************/
void CloseSoundDevice()
{
   int      usCounter;


   /* Get rid of the space that was malloc to hold the clock waveform */
   for( usCounter=0; usCounter<NUMBER_OF_WAVES; usCounter++ )
        free( (VOID *) SoundFiles[usCounter].Address);

   /* close sound device */
   ulError = mciSendCommand(SoundDevice.usSoundDeviceID,  /* Device to play the chimes.    */
                            MCI_CLOSE,                    /* MCI message.                  */
                            MCI_WAIT,                     /* Flags for the MCI message.    */
                            (ULONG) NULL,                 /* Parameters for the message.   */
                            (ULONG) NULL );               /* Parameter for notify message. */
   MCIERROR
}


/*******************************************************/
/*                 ShowMessageBox                      */
/*******************************************************/
void ShowMessageBox (char *message)
{
WinMessageBox (HWND_DESKTOP, HWND_DESKTOP, message,"Sound Server",0,MB_OK);
}

