/***************************************************************** SUPPORT.CC
 *                                                                          *
 *                Presentation Manager Support Functions                    *
 *                                                                          *
 ****************************************************************************/

#define INCL_BASE
#define INCL_PM
#include <os2.h>

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "support.h"
#include "debug.h"


/****************************************************************************
 *                                                                          *
 *      Definitions & Declarations                                          *
 *                                                                          *
 ****************************************************************************/

  // Local Function Prototypes

static ULONG BuildExtendedAttributeItem ( PFEA2 pFEA, PEADATA Item ) ;


/****************************************************************************
 *                                                                          *
 *                        Message Dispatcher                                *
 *                                                                          *
 ****************************************************************************/

extern MRESULT DispatchMessage
(
  HWND    hwnd,
  ULONG   msg,
  MPARAM  mp1,
  MPARAM  mp2,
  PMETHOD MethodTable,
  USHORT  MethodCount,
  PFNWP   DefaultProcessor
)
{
 /***************************************************************************
  * Process messages according to object's class method table.              *
  ***************************************************************************/

  PMETHOD pMethod = MethodTable ;
  USHORT cNumberLeft = MethodCount ;

  while ( ( cNumberLeft ) AND ( pMethod->Action != msg ) ) {
    pMethod ++ ;
    cNumberLeft -- ;
  }

  MRESULT mr ;

  if ( cNumberLeft ) {
    mr = pMethod->pFunction ( hwnd, msg, mp1, mp2 ) ;
  } else {
    if ( DefaultProcessor )
      mr = DefaultProcessor ( hwnd, msg, mp1, mp2 ) ;
    else
      mr = 0 ;
  }

 /***************************************************************************
  * Return result from message processor.                                   *
  ***************************************************************************/

  return ( mr ) ;
}

/****************************************************************************
 *                                                                          *
 *                         Add Item to System Menu                          *
 *                                                                          *
 ****************************************************************************/

extern VOID AddSysMenuItem ( HWND hwndFrame, MENUITEM *Item, PSZ Text ) {

 /***************************************************************************
  * Obtain the system menu window handle.                                   *
  ***************************************************************************/

  HWND hwndSysMenu = WinWindowFromID ( hwndFrame, FID_SYSMENU ) ;

 /***************************************************************************
  * Get the system menu's base item and its window handle.                  *
  ***************************************************************************/

  USHORT idSysMenu = SHORT1FROMMR ( WinSendMsg ( hwndSysMenu, MM_ITEMIDFROMPOSITION, NULL, NULL ) ) ;

  MENUITEM miSysMenu ;
  WinSendMsg ( hwndSysMenu, MM_QUERYITEM,
    MPFROM2SHORT(idSysMenu,FALSE), MPFROMP(&miSysMenu) ) ;

  HWND hwndSysSubMenu = miSysMenu.hwndSubMenu ;

 /***************************************************************************
  * Add the new item to the system menu's submenu, which is what we see.    *
  ***************************************************************************/

  WinSendMsg ( hwndSysSubMenu, MM_INSERTITEM, MPFROMP(Item), MPFROMP(Text) ) ;
}

/****************************************************************************
 *                                                                          *
 *                   Add Item to Submenu on System Menu                     *
 *                                                                          *
 ****************************************************************************/

extern VOID AddSysSubMenuItem
(
  HWND hwndFrame,
  USHORT SubMenuID,
  MENUITEM *Item,
  PSZ Text
)
{
 /***************************************************************************
  * Obtain the system menu window handle.                                   *
  ***************************************************************************/

  HWND hwndSysMenu = WinWindowFromID ( hwndFrame, FID_SYSMENU ) ;

 /***************************************************************************
  * Get the system menu's base item and its window handle.                  *
  ***************************************************************************/

  USHORT idSysMenu = SHORT1FROMMR ( WinSendMsg ( hwndSysMenu, MM_ITEMIDFROMPOSITION, NULL, NULL ) ) ;

  MENUITEM MenuItem ;
  WinSendMsg ( hwndSysMenu, MM_QUERYITEM,
    MPFROM2SHORT(idSysMenu,FALSE), MPFROMP(&MenuItem) ) ;

  HWND hwndSysSubMenu = MenuItem.hwndSubMenu ;

 /***************************************************************************
  * Get the submenu's base item and its window handle.                      *
  ***************************************************************************/

  WinSendMsg ( hwndSysSubMenu, MM_QUERYITEM,
    MPFROM2SHORT ( SubMenuID, TRUE ),
    (MPARAM) &MenuItem ) ;

  HWND hwndSubMenu = MenuItem.hwndSubMenu ;

 /***************************************************************************
  * Add the new item to the system menu's submenu, which is what we see.    *
  ***************************************************************************/

  WinSendMsg ( hwndSubMenu, MM_INSERTITEM, MPFROMP(Item), MPFROMP(Text) ) ;
}

/****************************************************************************
 *                                                                          *
 *                           Add Item to Menu                               *
 *                                                                          *
 ****************************************************************************/

extern VOID AddMenuItem
(
  HWND hwndFrame,
  USHORT MenuID,
  MENUITEM *Item,
  PSZ Text
)
{
 /***************************************************************************
  * Obtain the menu window handle.                                          *
  ***************************************************************************/

  HWND hwndMenu = WinWindowFromID ( hwndFrame, MenuID ) ;

 /***************************************************************************
  * Add the new item to the menu.                                           *
  ***************************************************************************/

  WinSendMsg ( hwndMenu, MM_INSERTITEM, MPFROMP(Item), MPFROMP(Text) ) ;
}

/****************************************************************************
 *                                                                          *
 *                        Add Item to SubMenu                               *
 *                                                                          *
 ****************************************************************************/

extern VOID AddSubMenuItem
(
  HWND hwndFrame,
  USHORT MenuID,
  USHORT SubMenuID,
  MENUITEM *Item,
  PSZ Text
)
{
 /***************************************************************************
  * Obtain the menu window handle.                                          *
  ***************************************************************************/

  HWND hwndMenu = WinWindowFromID ( hwndFrame, MenuID ) ;

 /***************************************************************************
  * Obtain the submenu window handle.                                       *
  ***************************************************************************/

  MENUITEM MenuItem ;
  WinSendMsg ( hwndMenu, MM_QUERYITEM,
    MPFROM2SHORT ( SubMenuID, TRUE ),
    (MPARAM) &MenuItem ) ;

  HWND hwndSubMenu = MenuItem.hwndSubMenu ;

 /***************************************************************************
  * Add the new item to the menu.                                           *
  ***************************************************************************/

  WinSendMsg ( hwndSubMenu, MM_INSERTITEM, MPFROMP(Item), MPFROMP(Text) ) ;
}

/****************************************************************************
 *                                                                          *
 *                       Remove Item from SubMenu                           *
 *                                                                          *
 ****************************************************************************/

extern VOID RemoveSubMenuItem
(
  HWND hwndFrame,
  USHORT MenuID,
  USHORT SubMenuID,
  USHORT ItemID
)
{
 /***************************************************************************
  * Obtain the menu window handle.                                          *
  ***************************************************************************/

  HWND hwndMenu = WinWindowFromID ( hwndFrame, MenuID ) ;

 /***************************************************************************
  * Obtain the submenu window handle.                                       *
  ***************************************************************************/

  MENUITEM MenuItem ;
  WinSendMsg ( hwndMenu, MM_QUERYITEM,
    MPFROM2SHORT ( SubMenuID, TRUE ),
    (MPARAM) &MenuItem ) ;

  HWND hwndSubMenu = MenuItem.hwndSubMenu ;

 /***************************************************************************
  * Remove the item from the menu.                                          *
  ***************************************************************************/

  WinSendMsg ( hwndSubMenu, MM_REMOVEITEM, MPFROM2SHORT(ItemID,TRUE), 0 ) ;
}

/****************************************************************************
 *                                                                          *
 *      Enable/Disable menu item.                                           *
 *                                                                          *
 ****************************************************************************/

extern VOID EnableMenuItem ( HWND hwndFrame, USHORT MenuID, USHORT ItemID, BOOL Enable ) {

 /***************************************************************************
  * Get the menu's window handle.                                           *
  ***************************************************************************/

  HWND hwndMenu = WinWindowFromID ( hwndFrame, MenuID ) ;

 /***************************************************************************
  * Set the menu item's enable/disable status.                              *
  ***************************************************************************/

  WinSendMsg ( hwndMenu, MM_SETITEMATTR, MPFROM2SHORT ( ItemID, TRUE ),
    MPFROM2SHORT ( MIA_DISABLED, Enable ? 0 : MIA_DISABLED ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Check/Uncheck menu item.                                            *
 *                                                                          *
 ****************************************************************************/

extern VOID CheckMenuItem ( HWND hwndFrame, USHORT MenuID, USHORT ItemID, BOOL Enable ) {

 /***************************************************************************
  * Get the menu's window handle.                                           *
  ***************************************************************************/

  HWND hwndMenu = WinWindowFromID ( hwndFrame, MenuID ) ;

 /***************************************************************************
  * Set the menu item's enable/disable status.                              *
  ***************************************************************************/

  WinSendMsg ( hwndMenu, MM_SETITEMATTR, MPFROM2SHORT ( ItemID, TRUE ),
    MPFROM2SHORT ( MIA_CHECKED, Enable ? MIA_CHECKED : 0 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *                        Add Program to Task List                          *
 *                                                                          *
 ****************************************************************************/

extern VOID Add2TaskList ( HWND hwnd, PSZ Name ) {

 /***************************************************************************
  * Get the window's process ID.                                            *
  ***************************************************************************/

  PID pid ;
  WinQueryWindowProcess ( hwnd, &pid, PTID(NULL) ) ;

 /***************************************************************************
  * Add an entry to the system task list.                                   *
  ***************************************************************************/

  SWCNTRL swctl ;
  swctl.hwnd = hwnd ;
  swctl.hwndIcon = 0 ;
  swctl.hprog = 0 ;
  swctl.idProcess = pid ;
  swctl.idSession = 0 ;
  swctl.uchVisibility = SWL_VISIBLE ;
  swctl.fbJump = SWL_JUMPABLE ;
  strcpy ( swctl.szSwtitle, (PCHAR)Name ) ;

  WinAddSwitchEntry ( &swctl ) ;
}

/****************************************************************************
 *                                                                          *
 *  Build Presentation Parameters                                           *
 *                                                                          *
 ****************************************************************************/

extern PPRESPARAMS BuildPresParams
(
  USHORT ParmCount,
  PULONG Ids,
  PULONG ByteCounts,
  PBYTE *Parms
)
{
 /***************************************************************************
  * Determine final size of presentation parameter block.                   *
  ***************************************************************************/

  ULONG Size = sizeof(ULONG) ;

  for ( int i=0; i<ParmCount; i++ ) {
    Size += sizeof(ULONG) ;
    Size += sizeof(ULONG) ;
    Size += ByteCounts[i] ;
  }

 /***************************************************************************
  * Allocate memory for block.  Return if unable to do so.                  *
  ***************************************************************************/

  PPRESPARAMS PresParams = (PPRESPARAMS) AllocateMemory ( (USHORT) Size ) ;

  if ( PresParams == NULL )
    return ( PresParams ) ;

 /***************************************************************************
  * Initialize the block header.                                            *
  ***************************************************************************/

  PresParams->cb = Size - sizeof(PresParams->cb) ;

 /***************************************************************************
  * Load the presentation parameters into the block.                        *
  ***************************************************************************/

  PPARAM Param = PresParams->aparam ;

  for ( i=0; i<ParmCount; i++ ) {
    Param->id = Ids[i] ;
    Param->cb = ByteCounts[i] ;
    memcpy ( Param->ab, Parms[i], (USHORT)ByteCounts[i] ) ;
    PBYTE p = PBYTE ( Param ) ;
    p += sizeof(Param->id) ;
    p += sizeof(Param->cb) ;
    p += ByteCounts[i] ;
    Param = PPARAM ( p ) ;
  }

 /***************************************************************************
  * Return the pointer to the block.  It will need freeing by the caller.   *
  ***************************************************************************/

  return ( PresParams ) ;
}

/****************************************************************************
 *                                                                          *
 *      Build Extended Attributes                                           *
 *                                                                          *
 ****************************************************************************/

extern PEAOP2 BuildExtendedAttributes ( ULONG Count, EADATA Table[] ) {

 /***************************************************************************
  * Find out how much memory will be needed for the block.                  *
  ***************************************************************************/

  ULONG cbEA = sizeof(FEA2LIST) - sizeof(FEA2) ;

  for ( int i=0; i<Count; i++ ) {
    cbEA += BuildExtendedAttributeItem ( PFEA2(NULL), &Table[i] ) ;
    while ( cbEA % 4 ) {
      cbEA ++ ;
    }
  }

 /***************************************************************************
  * Allocate memory for the FEA2 list.                                      *
  ***************************************************************************/

  PFEA2LIST pFEAList = PFEA2LIST ( AllocateMemory ( cbEA ) ) ;

  if ( pFEAList == NULL ) {
    return ( PEAOP2(NULL) ) ;
  }

 /***************************************************************************
  * Construct the extended attributes.                                      *
  ***************************************************************************/

  PFEA2 pFEA = pFEAList->list ;

  for ( i=0; i<Count; i++ ) {
    PFEA2 Start = PFEA2 ( pFEA ) ;
    PBYTE p = PBYTE(pFEA) + BuildExtendedAttributeItem ( pFEA, &Table[i] ) ;
    pFEA = PFEA2 ( p ) ;
    while ( (PBYTE(pFEA)-PBYTE(pFEAList->list)) % 4 ) {
      PBYTE p = PBYTE ( pFEA ) + 1 ;
      pFEA = PFEA2 ( p ) ;
    }
    if ( i < Count-1 )
      Start->oNextEntryOffset = PBYTE(pFEA) - PBYTE(Start) ;
    else
      Start->oNextEntryOffset = 0 ;
  }

  pFEAList->cbList = PBYTE(pFEA) - PBYTE(pFEAList) ;

 /***************************************************************************
  * Allocate memory for the EA header block.                                *
  ***************************************************************************/

  PEAOP2 pExtendedAttributes = PEAOP2 ( AllocateMemory ( sizeof(EAOP2) ) ) ;

  if ( pExtendedAttributes == NULL ) {
    FreeMemory ( pFEAList ) ;
    return ( PEAOP2(NULL) ) ;
  }

 /***************************************************************************
  * Fill in the extended attribute header block and return its address.     *
  ***************************************************************************/

  pExtendedAttributes->fpGEA2List = PGEA2LIST(NULL) ;
  pExtendedAttributes->fpFEA2List = pFEAList ;
  pExtendedAttributes->oError = 0 ;

  return ( pExtendedAttributes ) ;
}

/****************************************************************************
 *                                                                          *
 *      Build Extended Attribute Item                                       *
 *                                                                          *
 ****************************************************************************/

static ULONG BuildExtendedAttributeItem ( PFEA2 pFEA, PEADATA Item ) {

 /***************************************************************************
  * Store next entry's offset (set to zero at this point).                  *
  ***************************************************************************/

  PBYTE p = PBYTE ( pFEA ) ;

  if ( pFEA ) {
    *(PULONG(p)) = 0 ;
  }

  p += sizeof(ULONG) ;

 /***************************************************************************
  * Store flag byte.                                                        *
  ***************************************************************************/

  if ( pFEA ) {
    *p = 0 ;
  }

  p ++ ;

 /***************************************************************************
  * Store count of bytes in name.                                           *
  ***************************************************************************/

  if ( pFEA ) {
    *p = BYTE ( strlen ( PCHAR(Item->Name) ) ) ;
  }

  p ++ ;

 /***************************************************************************
  * Store count of bytes in value.                                          *
  ***************************************************************************/

  if ( pFEA ) {
    *(PUSHORT(p)) = Item->Length + 2 * sizeof(USHORT) ;
  }

  p += sizeof(USHORT) ;

 /***************************************************************************
  * Store name.                                                             *
  ***************************************************************************/

  if ( pFEA ) {
    strcpy ( PCHAR(p), PCHAR(Item->Name) ) ;
  }

  p += strlen ( PCHAR(Item->Name) ) + 1 ;

 /***************************************************************************
  * Store value's type.                                                     *
  ***************************************************************************/

  if ( pFEA ) {
    *(PUSHORT(p)) = Item->Type ;
  }

  p += sizeof(USHORT) ;

 /***************************************************************************
  * Store value's length.                                                   *
  ***************************************************************************/

  if ( pFEA ) {
    *(PUSHORT(p)) = Item->Length ;
  }

  p += sizeof(USHORT) ;

 /***************************************************************************
  * Store value.                                                            *
  ***************************************************************************/

  if ( pFEA ) {
    memcpy ( p, Item->Value, Item->Length ) ;
  }

  p += Item->Length ;

 /***************************************************************************
  * Return count of bytes needed for item.                                  *
  ***************************************************************************/

  return ( p - PBYTE(pFEA) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Build Multi-Value Multi-Type EA Item's Value                        *
 *                                                                          *
 ****************************************************************************/

extern ULONG BuildMVMTValue ( PVOID Value, ULONG Count, MVMT_VALUE Table[] ) {

 /***************************************************************************
  * Store the number of values.                                             *
  ***************************************************************************/

  PBYTE p = PBYTE ( Value ) ;

  if ( Value )
    *(PUSHORT(p)) = Count ;

  p += sizeof(USHORT) ;

 /***************************************************************************
  * Store the multiple values.                                              *
  ***************************************************************************/

  for ( int i=0; i<Count; i++ ) {

    if ( Value )
      *(PUSHORT(p)) = Table[i].Type ;

    p += sizeof(USHORT) ;

    if ( Value )
      *(PUSHORT(p)) = Table[i].Length ;

    p += sizeof(USHORT) ;

    if ( Value )
      memcpy ( p, Table[i].Value, Table[i].Length ) ;

    p += Table[i].Length ;
  }

 /***************************************************************************
  * Return the total byte count.                                            *
  ***************************************************************************/

  return ( p - PBYTE(Value) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Process Exit menu command.                                          *
 *                                                                          *
 ****************************************************************************/

extern MRESULT APIENTRY Exit ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

 /***************************************************************************
  * Send a WM_CLOSE message to the window.                                  *
  ***************************************************************************/

  WinSendMsg ( hwnd, WM_CLOSE, 0L, 0L ) ;

 /***************************************************************************
  * Done.                                                                   *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Process Help For Help menu command.                                 *
 *                                                                          *
 ****************************************************************************/

extern MRESULT APIENTRY HelpForHelp ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

 /***************************************************************************
  * Get the help instance window handle, if any.                            *
  ***************************************************************************/

  HWND hwndHelp = WinQueryHelpInstance ( hwnd ) ;

 /***************************************************************************
  * If help is available, pass the request on to the help window.           *
  ***************************************************************************/

  if ( hwndHelp ) {
    WinSendMsg ( hwndHelp, HM_DISPLAY_HELP, 0L, 0L ) ;
  }

 /***************************************************************************
  * Done.                                                                   *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Process Extended Help menu command.                                 *
 *                                                                          *
 ****************************************************************************/

extern MRESULT APIENTRY ExtendedHelp ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

 /***************************************************************************
  * Get the help instance window handle, if any.                            *
  ***************************************************************************/

  HWND hwndHelp = WinQueryHelpInstance ( hwnd ) ;

 /***************************************************************************
  * If help is available, pass the request on to the help window.           *
  ***************************************************************************/

  if ( hwndHelp ) {
    WinSendMsg ( hwndHelp, HM_EXT_HELP, 0L, 0L ) ;
  }

 /***************************************************************************
  * Done.                                                                   *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Process Keys Help menu command.                                     *
 *                                                                          *
 ****************************************************************************/

extern MRESULT APIENTRY KeysHelp ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

 /***************************************************************************
  * Get the help instance window handle, if any.                            *
  ***************************************************************************/

  HWND hwndHelp = WinQueryHelpInstance ( hwnd ) ;

 /***************************************************************************
  * If help is available, pass the request on to the help window.           *
  ***************************************************************************/

  if ( hwndHelp ) {
    WinSendMsg ( hwndHelp, HM_KEYS_HELP, 0L, 0L ) ;
  }

 /***************************************************************************
  * Done.                                                                   *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Process Help Index menu command.                                    *
 *                                                                          *
 ****************************************************************************/

extern MRESULT APIENTRY HelpIndex ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 ) {

 /***************************************************************************
  * Get the help instance window handle, if any.                            *
  ***************************************************************************/

  HWND hwndHelp = WinQueryHelpInstance ( hwnd ) ;

 /***************************************************************************
  * If help is available, pass the request on to the help window.           *
  ***************************************************************************/

  if ( hwndHelp ) {
    WinSendMsg ( hwndHelp, HM_HELP_INDEX, 0L, 0L ) ;
  }

 /***************************************************************************
  * Done.                                                                   *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *                                                                          *
 *      Copy Bitmap                                                         *
 *                                                                          *
 ****************************************************************************/

extern HBITMAP CopyBitmap ( HAB Anchor, HBITMAP Original ) {

   HBITMAP Copy = 0 ;
   BITMAPINFOHEADER2 Header ;
   Header.cbFix = 16 ;
   GpiQueryBitmapInfoHeader ( Original, &Header ) ;
   PSZ DeviceData [] = { PSZ(NULL), PSZ("Display"), PSZ(NULL), PSZ(NULL) } ;
   HDC hdcMemory1 = DevOpenDC  ( Anchor, OD_MEMORY, PSZ("*"), 4, DeviceData, 0 ) ;
   if ( hdcMemory1 ) {
      HDC hdcMemory2 = DevOpenDC  ( Anchor, OD_MEMORY, PSZ("*"), 4, DeviceData, 0 ) ;
      if ( hdcMemory2 ) {
         SIZEL PageSize ;
         PageSize.cx = Header.cx ;
         PageSize.cy = Header.cy ;
         HPS hpsMemory1 = GpiCreatePS ( Anchor, hdcMemory1, &PageSize, GPIA_ASSOC | PU_PELS ) ;
         if ( hpsMemory1 ) {
            HPS hpsMemory2 = GpiCreatePS ( Anchor, hdcMemory2, &PageSize, GPIA_ASSOC | PU_PELS ) ;
            if ( hpsMemory2 ) {
               Copy = GpiCreateBitmap ( hpsMemory2, &Header, FALSE, 0, 0 ) ;
               if ( Copy ) {
                  GpiSetBitmap ( hpsMemory1, Original ) ;
                  GpiSetBitmap ( hpsMemory2, Copy ) ;
                  POINTL Points [4] ;
                  Points[0].x = Points[0].y = Points[2].x = Points[2].y = 0 ;
                  Points[1].x = Points[3].x = Header.cx ;
                  Points[1].y = Points[3].y = Header.cy ;
                  GpiBitBlt ( hpsMemory2, hpsMemory1, 4L, Points, ROP_SRCCOPY, BBO_IGNORE ) ;
                  GpiSetBitmap ( hpsMemory1, 0 ) ;
                  GpiSetBitmap ( hpsMemory2, 0 ) ;
               } else {
                  ERRORID Error = WinGetLastError ( Anchor ) ;
                  Log ( "CopyBitmap: ERROR: Unable to create bitmap.  Status %i.", Error ) ;
               } /* endif */
               GpiDestroyPS ( hpsMemory2 ) ;
            } else {
               ERRORID Error = WinGetLastError ( Anchor ) ;
               Log ( "CopyBitmap: ERROR: Unable to open second memory presentation space.  Status %i.", Error ) ;
            } /* endif */
            GpiDestroyPS ( hpsMemory1 ) ;
         } else {
            ERRORID Error = WinGetLastError ( Anchor ) ;
            Log ( "CopyBitmap: ERROR: Unable to open first memory presentation space.  Status %i.", Error ) ;
         } /* endif */
         DevCloseDC ( hdcMemory2 ) ;
      } else {
         ERRORID Error = WinGetLastError ( Anchor ) ;
         Log ( "CopyBitmap: ERROR: Unable to open second memory device context.  Status %i.", Error ) ;
      } /* endif */
      DevCloseDC ( hdcMemory1 ) ;
   } else {
      ERRORID Error = WinGetLastError ( Anchor ) ;
      Log ( "CopyBitmap: ERROR: Unable to open first memory device context.  Status %i.", Error ) ;
   } /* endif */

   return ( Copy ) ;
}
