;**************************************************************************
;** para-util.cl                                                         **
;**                                                                      **
;**  Dans ce programme sont traites les fonctions aidant a faire tourner **
;** le systeme dans une forme "pseudo-parallele" (multitraitement). Il   **
;** donne les elements necessaires pour sauvegarder un systeme expert et **
;** ses donnees dans un segments de memoire ou dans un string pour l'e-  **
;** xecuter parallellement apres avec differentes options                **
;**                                                                      **
;** Claudia Coiteux-Rosu                                    Octobre 1990 **
;**************************************************************************
;** Fonctions:                                                           **
;** Segments de memoire:                                                 **
;**  calc-taille-seg           save-exp-seg         fact-seg             **
;**  loadfacts-seg             load-reg-seg         load-f-seg           **
;**                                                                      **
;** Strings:                                                             **
;**  save-exp-str              load-reg-str                              **
;**                                                                      **
;** Options d'execution:                                                 **
;**  take-modes                take-str             lire-str             **
;**  reponse_o-n               take-prior-reg       take-priors          **
;**                                                                      **
;** Execution:                                                           **
;**  make-p-exp                exec-parall          processus-inactives  **
;**                                                                      ** 
;**************************************************************************
 
(eval-when (compile) (load "varenv"))

;**************************************************************************
;** Fonctions utiles a sauvegarder et appeler l'expert dans un segment   **
;** de memoire.                                                          **
;**************************************************************************
;** Cette fonctions calcule la taille de segment a creer                 **
(defun calc-taille-seg (str data)
   (* 1024 (1+ (round (/ (1+ (round (/ (+ (length str) 
                                       (length data) 
                                       (length (write-to-string data)))
                                       8.0)))
                          1024.0)))))


;** Sauvegarde l'expert dans un segment, si `avec-nom' est t garde la de-**
;** finition du nom de la base, sinon non.                               **
(defun save-exp-seg (nom-exp addr &optional(avec-nom t))
   (let* ((ad addr)
          (base (get '*psy-kb* nom-exp))
          straux
          (error nil))
     (and avec-nom 
          (setq straux (write-to-string `(define-kb ',nom-exp)))
          (ecrireseg ad straux)
          (setq ad (+ ad (length straux) 1))
          (setq straux (write-to-string `(use-kb ',nom-exp)))
          (ecrireseg ad straux)
          (setq ad (+ ad (length straux) 1)))
     (for i in (reverse (symeval-in-instance base 'rules))
          bind nom-regle p-droite
          until error
          do (setq nom-regle (symeval-in-instance i 'name)) 
             (setq p-droite (symbol-function  (concat 'rhs- nom-regle)))
             (cond ((typep p-droite 'compiled-function)
                       (setq error t))
                   (t (setq straux 
                          (write-to-string 
                               `(p ,(take-rule-name nom-regle)
                                   ,@(send i 'p-gauche) 
                                     --> 
                                   ,@(cddar (cdddr p-droite)))))
                      (ecrireseg ad  straux)
                      (setq ad (+ ad (length straux) 1)))))
     (cond (error (msg "Systeme compile, je ne peut pas le sauver" #\N))
           (t ad))))


;** Sauvegarde un fait dans un segment de memoire                        **
(defmacro fact-seg (addr &rest f)
   (eval `(ecrireseg ,addr (write-to-string '(fact ,@f))))
  `(addrsuivseg ,addr))


;** Sauvegarde une liste de faits dans un segment                        **
(defun loadfacts-seg (addr liste-faits)
   (for f in liste-faits
        bind (ad addr)
        do (setq ad (eval `(fact-seg ',ad ,@f)))
        finally ad))


;** Charge les regles d'un experts stockees dans un segment              **
(defun load-reg-seg (adr adf)
 (for bind (ad1 adr)
            str
      until (eq ad1 adf)
      do (setq str (lireseg ad1))
         (setq ad1 (+ ad1 (length str) 1))
         (eval (read-from-string str))))


;** Charge les donnees d'un expert stockees dans un segment              **
(defun load-f-seg (addr)
 (for bind (ad1 addr)
           (str "xxx")
      until (equal str "")
      do (setq str (lireseg ad1))
         (setq ad1 (+ 1 (length str) ad1))
         (and (not (equal str ""))
              (eval (read-from-string str)))))


;**************************************************************************
;** Fonctions utiles a sauvegarder et appeler l'expert dans un string    **
;**************************************************************************
;** Sauvegarde les regles dans expert dans un string, avec ou sans la de-**
;** finition du nom de la base selon `avec-nom'                          **
(defun save-exp-str (nom-exp &optional(avec-nom t))
   (let* ((o-str (make-string-output-stream))
          (base (get '*psy-kb* nom-exp))
          straux
          (error nil))
     (and avec-nom 
          (prog ()
           (format o-str "~A~%" (write-to-string `(define-kb ',nom-exp)))
           (format o-str "~A~%" (write-to-string `(use-kb ',nom-exp))))
            t)
     (for i in (reverse (symeval-in-instance base 'rules))
          bind nom-regle fonct
          until error
          do (setq nom-regle (symeval-in-instance i 'name))
             (setq fonct  (symbol-function (concat 'rhs- nom-regle)))
             (cond ((typep fonct 'compiled-function) (setq error t))
                   (t (format o-str "~A~%"
                           (write-to-string 
                                   `(p ,(take-rule-name nom-regle)
                                       ,@(send i 'p-gauche) 
                                       --> 
                                       ,@(cddar (cdddr fonct))))))))
     (cond (error (msg "Systeme compile, je ne peut pas le sauver" #\N))
           (t (setq straux (get-output-stream-string o-str))))))


;** Charge les regles sauvegardes dans un string                         **
(defun load-reg-str (str)
 (let ((lim (length str)))
   (for bind (ad1 0)
              ad2
              str1
        until (eq ad1 lim)
        do (setq ad2 (for bind (ad ad1 (1+ ad))
                          until (eq (char str ad) '#\newline)
                          finally ad))
           (setq str1 (subseq str ad1 ad2))
           (eval (read-from-string str1))
           (setq ad1 (+ ad2 1)))))


;**************************************************************************
;** Options d'un execution en multitraitement                            **
;**************************************************************************
;** Saissi des modes d'execution                                         **
(defun take-modes ()
  (let (aut pr dc dcv tr)
    (setq aut (reponse_o-n "  Voulez-vous executer en mode automatique"))
    (if (setq dc (reponse_o-n 
                   "  Voulez-vous detecter les cycles d'execution?")) 
        (setq dcv  (reponse_o-n 
          "  Pour les detecter voulez-vous comparer les faits par valeur?"
              )) 
        (setq pr (reponse_o-n 
                  "  Voulez-vous executer en mode reexecutable?")))
    (setq tr (reponse_o-n "  Voulez-vous garder la trace d'execution?"))
    (list (cond (aut '(autorun))
                (t   '(no-autorun)))
          (cond (tr '(psy-trace))
                (t  '(psy-untrace)))  
          (cond (dcv '(detecte-cycles t))
                (dc  '(detecte-cycles))
                (pr  '(permet-reexec))
                (t   '(no-reexec))))))


;** Saissi de la strategie d'execution                                   **
(defun take-str (base)
   (msg #\N "  Strategies existentes :" #\N)
   (msg #\T "1 Le fait le plus recent" #\N)
   (msg #\T "2 Le fait le plus age" #\N)
   (msg #\T "3 La regle avec le plus de faits" #\N)
   (msg #\T "4 La regle avec le moins de faits" #\N)
   (msg #\T "5 La regle moins executee" #\N)
   (msg #\T "6 Random " #\N)
   (msg #\T #\T "Donnez votre choix pour la base " base ":")
   (finish-output t)
   (lire-str))


;** Lecture du choix de strategie d'execution                            **
(defun lire-str()
   (case (cond (*psy-var-fen* 
                    (read-from-string (multi-line-read-para *psy-var-fen*)))
               (t (read *standard-input*)))
     (1 '(str-jeune-fait))
     (2 '(str-vieux-fait))
     (3 '(str-max-faits))
     (4 '(str-min-faits))
     (5 '(str-moins-exec))
     (6 '(str-random))
     (otherwise    (msg #\T #\T "Entrez un nombre entre 1 et 6 !!! :")
                   (finish-output t)
                   (lire-str))))


;** Fonction auxiliere pour saissir une reponse affirmative ou negative  **
(defun reponse_o-n(str)
   (msg #\N str "(o/n)")
   (finish-output t)
   (let (lu)
     (cond (*psy-var-fen* (setq lu (multi-line-read-para *psy-var-fen*))
                          (setq lu (read-from-string lu)))
           (t   (setq lu (read *standard-input*))))
     (cond ((eq lu 'o) t)
           ((eq lu 'n) nil)
           (t (reponse_o-n "  Repondez 'o pour oui ou 'n pour non !!!")
              (finish-output t)))))
       

;** Saissi de la priorite d'une regle                                    **
(defun take-prior-reg (r)
  (let (lu)
   (msg #\N #\T "Regle : " r "      Priorite :")
   (finish-output)
   (cond (*psy-var-fen* (setq lu (multi-line-read-para *psy-var-fen*))
                        (setq lu (read-from-string lu)))
           (t   (setq lu (read *standard-input*))))
   (cond ((numberp lu) lu)
         (t (msg #\T "La priorite est numerique !!")
            (take-prior-reg r)))))


;** Saissi des changements dans les priorites de regles                  **
(defun take-priors (base addr addf)
  (msg #\N #\N "**** Base " base  "****" #\N)
  (if (reponse_o-n "  Changez-vous les priorites des regles?")
      (progn() 
         (msg #\N  "  Donnez la priorite de chaque regle" #\N)
         (cond ((numberp addr) ; regles dans un segment
                  (for bind (ad addr)
                            str liste prior
                       until (equal ad addf)
                       collect (progn()
                                 (setq str (lireseg ad))
                                 (setq ad (+ 1 (length str) ad))
                                 (setq liste (read-from-string str))
                                 (setq prior 
                                      (take-prior-reg (cadr liste)))
                                `(set-prior ',(cadr liste) ',prior))))
               (t  ; regles dans un string
                   (for bind (lim (length addr)) 
                             (ad1 0) ad2 str1
                             liste prior
                        until (eq ad1 lim)
                        collect (progn()
                                  (setq ad2 (for bind (ad ad1 (1+ ad))
                                        until (eq (char addr ad)
                                                  '#\newline)
                                        finally ad))
                                  (setq str1 (subseq addr ad1 ad2))
                                  (setq liste (read-from-string str1))
                                  (setq ad1 (+ ad2 1))
                                  (setq prior (take-prior-reg 
                                                  (cadr liste)))
                                 `(set-prior ',(cadr liste) ',prior))))))))


;**************************************************************************
;** Execution en mutitraitement                                          **
;**************************************************************************
(in-package :user)
(require :process)
(proclaim '(special *psy-binding*))
(proclaim '(special *psy-db*))
(proclaim '(special *psy-name-db*))
(proclaim '(special *psy-execreg*))
(proclaim '(special *psy-autorun*))
(proclaim '(special *psy-halt*))
(proclaim '(special *psy-just-point*))
(proclaim '(special *psy-runbase*))
(proclaim '(special *psy-autorun*))     
(proclaim '(special *psy-paths*))
(proclaim '(special *psy-root*))
(proclaim '(special *psy-trace*))
(proclaim '(special *standard-output*))


;** Execute un systeme expert en multitraitement. `nom-base' est le nom  **
;** qui aura la base de connaissances, `adregs' et `adfaits' contiennent **
;** les regles et les faits dans un segment ou dans un string et `autres-**
;** donnees' est une liste contenant les options d'execution.            **
(defun make-p-exp (nom-base adregs adfaits autres-donnees)
   (let* (*psy-binding*
          *psy-db*
          *psy-name-db*
          *psy-execreg*
          *psy-halt*
          *psy-just-point*
          *psy-runbase*
          *psy-autorun*     
          *psy-paths*
          *psy-root*
          *psy-trace*
         (st-out *standard-output*)
         (*standard-output* (make-string-output-stream)))
      (makech)
      (define-kb nom-base)
      (use-kb nom-base)
      (cond ((stringp adregs) (load-reg-str adregs))
            (t (load-reg-seg adregs adfaits)))
      (for i in autres-donnees
           do (eval i))
      (cond ((numberp adfaits) (load-f-seg adfaits))
            (t (loadfacts adfaits)))
      (if (null *psy-autorun*) (run))
      (msg "********* FIN EXECUTION BASE " nom-base " **********" #\N)
      (putprop '*psy-exec-para* (cons *standard-output* autres-donnees)
                nom-base)
      (setq *standard-output* st-out)))


;** Execute parallellement la base de regles `adr' et faits `adf' sous   **
;** les noms donnes dans `noms-bases'. `adr' et `adf' peuvent etre des   **
;** strings ou des adresses des segments. Cette fonction effectue egale- **
;** ment la saissi d'options d'execution.                                **
(defun exec-parall (adr adf &rest noms)
  (let ((noms-bases (cond ((listp (car noms)) (car noms))
                          (t                   noms))))
  (let ((donnees (for b in noms-bases
                      collect (let ((lp (take-priors b adr adf)))
                                  (cond (lp `(,b ,(take-str b) 
                                              ,@lp ,@(take-modes)))
                                        (t `(,b ,(take-str b)
                                             ,@(take-modes))))))))
   (for d in donnees
          collect  (mp:process-run-function 
                    `(:name ,(write-to-string (car d))
                      :initial-bindings ,*psy-var-env*)
                     #'make-p-exp (car d) adr adf (cdr d))))))


;** Execute en multitraitement la base de connaissances (regles et faits)**
;** en partant du nom de la base, de la liste de faits et de la liste de **
;** noms de bases a executer                                             **
(defun multiexec (nom-base faits &rest noms-bases)
  (exec-parall (save-exp-str nom-base nil) faits noms-bases))


;** Verifie la fin de l'execution des processus en multitraitement       **
(defun processus-inactives (lp)
  (cond ((eq 0 (for i in lp
                    bind (count-act 0)
                    do (and (mp:process-active-p i)
                    (setq count-act (1+ count-act)))
                    finally count-act)) T)
        (t (processus-inactives lp))))



