/* -- AIX/6000 System monitor 
**
**     monitor.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-1998 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
**
** Author: Jussi Maki
**         Center for Scientific Computing
**         Tietotie 6
**         P.O. Box 405
**         FIN-02101 Espoo
**         FINLAND
**         Internet-mail: jmaki@csc.fi
**         Phone:  +358-0-457 2747
**         Telefax: +358-0-457 2302
**
**         Marcel J.E. Mol
**         Netherlands Europe
**         E-mail: marcel@mesa.nl
**         Phone (home): (+31) 015-3101310
**
**
*/

#include "monitor.h"
#include "print.h"
#include <string.h>
#include <strings.h>


#define MAXTOPCPU NPROCS
#ifndef min
#define min(a,b) ((a)<(b)?(a):(b))
#endif
#ifndef max
#define max(a,b) ((a)>(b)?(a):(b))
#endif


#define DEFAULT_SAMPLE		10      /* seconds */
#define DEFAULT_INTERVAL	900     /* seconds */
#define DEFAULT_INTERVAL_SAMPLE	30      /* seconds */
#define closelog()              fclose(logfp)

int sleep_sec    = 0;                   /* sample time */
int interval_sec = DEFAULT_INTERVAL;    /* interval time */
int synctime     = 0;

char * progname;
char   hostnm[128];
struct utsname unames;

/*
 * Main indications for interactive monitoring
 */
int show_sys	     = 0;    /* show all system info */
int show_sys_summary = 0;    /* show only system summary */
/*
 * Details on what is show by default or not
 */
int show_top         = 0;    /* show process top list in normal screen */
int show_top_a       = 1;    /* show process top list in alternative screen */
int show_top_running = 0;    /* show process top list */
int show_disk_full   = 0;    /* show all disk info */
int show_inet_full   = 0;    /* show more network info */
int show_nfs         = 0;    /* show nfs traffic info */
#if defined(HAVE_AFS)
int show_afs         = 0;    /* show afs traffic info */
#endif
int show_smp         = 0;    /* show smp cpu stats */
int highlight        = HIGHLIGHT; /* highlight headers */
#if defined(_HAS_COLOR)
#if 0
int namecol          = COLOR_PAIR(7); /* metric name color */
int valuecol         = COLOR_PAIR(7); /* metric value color */
#endif
int namecol          = 0; /* metric name color */
int valuecol         = 0; /* metric value color */
#endif
int sort_disks       = 0;    /* sort disk prints on read+write rate */
int include_root     = 0;    /* include root sessions in user calculations */
int logmode	     = 0;    /* log interval based info to file */
int logtop           = 0;    /* log top records when in logmode */
int logallfs         = 0;    /* log all instead of only local filesys info */ 
int dump_sample      = 0;    /* also log sample data to file */
int do_curses        = 0;    /* will curses be used */
int do_color         = 0;    /* will curses be used */
int compress         = 0;    /* compress old log files */
int g_count          = 0;    /* if positive, only do g_count intervals */
int current          = 0;    /* indicates current set of variables */
int disk_cnt;                /* number of disk measured */
int net_cnt;                 /* number of network interfaces measured */
int cpu_cnt;                  /* number of cpus measured */

#ifdef COMPPROG
char *compprog = COMPPROG;
#else
char *compprog = "/usr/local/bin/gzip";
#endif

char *logfilefmt               = NULL;
char logfile[FILENAME_MAX]     = "";
char logfile_old[FILENAME_MAX] = "";
FILE *logfp = NULL;

#if defined(DEBUG)
FILE *debugfp;
#endif

char ts[128]; /* timestring */
int numcolumns, numlines;
time_t tmp_time;
struct tm tmp_tm;
int topflag_usersystemtime = 0;
int topflag_showusername = 1;
int topflag_sort = TOPSORT_CPU;
topcpu_t  topcpu[MAXTOPCPU];
int ntop = MAXTOPCPU; /* top procs to show by default */
int nprocs; /* number of procs found */

double time1, time2, timea[3];    /* for sample, previous sample and */
struct sysinfo si[3];             /*   previous interval */
struct cpuinfo *cpus[3];
struct vminfo vm[3];
struct vmker vmk;
struct dkstat *dk[3][MAX_DISKS];
struct ifnet *ifnets[3][MAX_IFNETS];
t_nfs_stats nfs[3];
#if defined(HAVE_AFS)
afsstat_t afs[3];
#endif

extern time_t active_threshold;


/*
 * Function declarations 
 */
void   usage();
void   copy_log_variables();
int    openlogfile(struct tm *tim);
void   sighandler(int signum);
void   wait_input_or_sleep(long secs, long usecs);
double realtime();
void   start_screen();
void   quit_monitor();
void   checkinput();
void   display_help();

/*
 * External function declarations 
 */
char *termdef(int FileDescriptor, char Characteristic);




/**************************************************************************/

void
main(int argc, char **argv)
{
    time_t nextinterval = 0, nextsample = 0;
    double refresh_time;
    struct timeval tp;
    int i, t_row, t_col, c_row, c_col;

    progname = *argv++;        /* go point to arguments */

    gethostname(hostnm, sizeof(hostnm));
    uname(&unames);
    setlocale(LC_NUMERIC, "C"); /* be sure that decimal-point is "." */

#if defined(DEBUG)
    OPENDB;
    tmp_time = time(0);
    fprintf(debugfp, "Starting new log at %s\n", ctime(&tmp_time));
    CLOSEDB;
#endif

    while (*argv) {
        if ((*argv)[0] == '-') {
	    switch ((*argv)[1]) {
	        case 'a': /* -a[lternative] */
	              show_sys = 0;
                      show_sys_summary = 1;
	              break;
                case 'A': /* -A[active_threshold]  for active user calculation */
                      active_threshold = atoi(*++argv);
	              break;
#if defined(HAVE_AFS)
	        case 'B': /* AFS */
	              show_afs = 1;
	              show_nfs = 0;
	              break;
#endif
	        case 'c': /* -c[ompress] [compress-program] */
                      compress = 1;
	              if (*(argv+1) && **(argv+1) != '-')
                          compprog = *++argv;
	              break;
	        case 'C': /* -C[ount] number */
	              g_count = atoi(*++argv);
	              break;
	        case 'd': /* -d[isk] */
	              show_disk_full = 1;
	              break;
	        case 'D': /* -D[isk] */
                      sort_disks = 1;
	              break;
	        case 'F': /* -F[ilesys] */
                      logallfs = 1;
	        case 'H': /* -H[ighlight] bold|reverse|none|c|m|v */
                      if (*(argv+1) == NULL || **(argv+1) == '-') 
                          highlight = highlight == 0 ? A_STANDOUT : 0;
                      else
                          switch (**++argv) {
                              case 'r': highlight = A_STANDOUT; /* reverse */
	                                break;
                              case 'b': highlight = A_BOLD;     /* bold */
	                                break;
#if defined(_HAS_COLOR)
                              case 'c': highlight = COLOR_PAIR(atoi((*argv)+1));
	                                break;
                              case 'm': namecol = COLOR_PAIR(atoi((*argv)+1));
	                                break;
                              case 'v': valuecol = COLOR_PAIR(atoi((*argv)+1));
	                                break;
#endif
                              case 'n':
                              default : highlight = 0;          /* none */
                          }
	              break;
	        case 'i': /* -i[nterval] length, in seconds */
	              interval_sec = atoi(*++argv);
	              if (interval_sec == 1)
                          interval_sec = DEFAULT_INTERVAL;
	              break;
	        case 'I': /* -I[nteractive] mode */
                      if (!show_sys && !show_sys_summary)
                          show_sys = 1;
	              break;
                case 'l': /* -l[ogmode] logfilename */
	              logmode = 1;
	              synctime = 1; /* force sync mode */
                      if (**(++argv) == '-') {
                          fprintf(stderr, "Logmode needs file name\n");
                          usage();
                      }
		      logfilefmt = *argv;
                      if (sleep_sec == 0) /* if user did not set it yet */
                          sleep_sec = DEFAULT_INTERVAL_SAMPLE;
#if defined(DEBUG)
                      OPENDB;
                      fprintf(debugfp, "Logfile format is %s\n", logfilefmt);
                      CLOSEDB;
#endif
	              break;
	        case 'L': /* -L[ogsample] when in logmode */
                      dump_sample = 1;
	              break;
	        case 'n': /* -n[et]  */
	              show_inet_full = 1;
	              break;
	        case 'N': /* -n[fs] */
	              show_nfs = 1;
#if defined(HAVE_AFS)
	              show_afs = 0;
#endif
	              break;
		case 'r':
                      show_top_running = 1;
	              break;
		case 'R': /* -R[ootinclude] */
                      include_root = 1;
	              break;
	        case 's': /* -s[ample] length in seconds  || -sm[p]*/
                      if ((*argv)[2] == 'm') {
                          show_sys = 0;
                          show_sys_summary = 1;
                          show_smp = 1;
                      }
                      else {
	                  sleep_sec = atoi(*++argv);
	                  if (sleep_sec < 1)
                              sleep_sec = 1;
                      }
	              break;
                case 'S': /* -S[ynctime] */
	              synctime = 1;
	              break;
	        case 't': /* -t[op] [nproc] */
	              show_top = 1; /* show top in main screen also */
	              if (*(argv+1) && **(argv+1) != '-')
	                  ntop = atoi(*++argv);
	              break;
                case 'T': /* -T[oplog] */
                      logtop = 1;
                      break;
	        case 'u': /* -u   ... dont show user names in top */
	              topflag_showusername = 0;
	              break;
	        default:
                      fprintf(stderr, "Something wrong with %s\n", *argv);
                      usage();
	              break;
	    }
        }
        argv++;
    }

    /*
     * we want to do something ...
     */
    if (!logmode && !show_sys_summary) {
        show_sys = 1;
#if defined(HAVE_AFS)
        if (!show_afs)
#endif
            show_nfs = 1;
    }

    /*
     * When online display is needed, flag curses
     */
    if (show_sys || show_sys_summary)
        do_curses = 1;

    /*
     * if user did not specify, set a reasonable sleep time
     */
    if (sleep_sec == 0)
        sleep_sec = DEFAULT_SAMPLE;

    /*
     * sample time may not larger that interval time...
     */
    if (logmode && sleep_sec > interval_sec)
        sleep_sec = interval_sec;

    /*
     * Setup synctime if wanted
     */
    if (synctime) {
	time_t sec1970   = time(0);
	struct tm *now = localtime(&sec1970);
	time_t daystart  = sec1970 - now->tm_hour*3600 - 
			 now->tm_min*60 - now->tm_sec;

	nextinterval  = daystart +
			((sec1970-daystart)/interval_sec + 1)*interval_sec;
	nextsample    = daystart +
			((sec1970-daystart)/sleep_sec + 1)*sleep_sec;
#if defined(DEBUG)
        OPENDB;
        fprintf(debugfp, "In synctime 1: now=%ld, daystart=%ld, "
                         "nextinterval=%ld, nextsample=%ld\n",
                         sec1970, daystart, nextinterval, nextsample);
        CLOSEDB;
#endif
    }
    else
        nextsample = time(0) + 1; /* to get something to display soon */

    /*
     * init network and disk arrays
     */
    for (i = 0; i < MAX_IFNETS; i++)
        ifnets[0][i] = ifnets[1][i] = ifnets[2][i] = 0;

    for (i = 0; i < MAX_DISKS; i++)
        dk[0][i] = dk[1][i] = dk[2][i] = 0;


    /*
     * get first time data
     * /dev/kmem will be opened as root. After that the euid is reset to the
     * user starting the program. So it is quite save to open files and
     * run programs from now on!
     */
    get_sys_vm_info(&si[current], &vmk, &vm[current]);
    if (show_smp || logmode)
        cpu_cnt   = get_cpuinfo(&cpus[current]);
    if (show_sys || show_disk_full || logmode) 
        disk_cnt = get_dkstat(dk[current]);
    if (show_sys || show_inet_full || logmode)
        net_cnt  = get_ifnet(ifnets[current]);
    if ((show_sys && show_nfs) || logmode)
        get_nfsstat(&nfs[current]);
#if defined(HAVE_AFS)
    if ((show_sys && show_afs) || logmode)
        get_afsstat(&afs[current]);
#endif
    if ((show_sys && show_top) || (show_sys_summary && show_top_a) ||
        (logmode && logtop))
        nprocs = get_topcpu(topcpu, logtop ? MAXTOPCPU : ntop);
    timea[current] = realtime();

    if (logmode) {
        /*
         * setup initial logmode variables, and setup umask for logfiles
         */
	copy_log_variables();
        umask(022);
    }

    /*
     * setup signals and curses
     */
    if (do_curses) {
        signal(SIGTSTP,  sighandler);
        signal(SIGINT,   sighandler);
        signal(SIGQUIT,  sighandler);
        signal(SIGWINCH, sighandler);
        start_screen();
    }
    else {
        /*
         * daemon mode, detach from terminal and continue in background
         */
        if ((i = fork()) > 0) {
            printf("Monitor entered daemon mode: process id %d\n", i);
            exit(0);
        }
        if (i == -1) {
            perror("monitor deamon mode");
            exit(1);
        }
        /* child */
        fclose(stdin);
        fclose(stdout);
        fclose(stderr);
        setpgrp();
    }
      


    /*
     * Go for it...
     *      switch vars
     *      wait till end of sample
     *      get current stats
     *      print stats
     */
    gettimeofday(&tp, 0);
    while (1) {

        /*
         * wait for end of interval or keypress
	 * calculate exact wait time
         */

#if defined(DEBUG)
        OPENDB;
        fprintf(debugfp, "In synctime loop: now=%ld.%ld, nextsample=%ld\n",
                             tp.tv_sec, tp.tv_usec, nextsample);
        CLOSEDB;
#endif
        wait_input_or_sleep(nextsample - tp.tv_sec - 1, 1000000 - tp.tv_usec);


        if (do_curses)
	    checkinput();

        current ^= 1; /* switch to next set of variables */

        /*
         * Get the info
         */
	get_sys_vm_info(&si[current], &vmk, &vm[current]);
        if (show_smp || logmode)
            get_cpuinfo(&cpus[current]);
	if (show_sys || show_disk_full || logmode) 
	    get_dkstat(dk[current]);
	if (show_sys || show_inet_full || logmode)
            get_ifnet(ifnets[current]);
        if ((show_sys && show_nfs) || logmode)
	    get_nfsstat(&nfs[current]);
#if defined(HAVE_AFS)
        if ((show_sys && show_afs) || logmode)
	    get_afsstat(&afs[current]);
#endif
        if ((show_sys && show_top) || (show_sys_summary && show_top_a) ||
            (logmode && logtop))
            nprocs = get_topcpu(topcpu, logtop ? MAXTOPCPU : ntop);

	timea[current] = realtime();
	refresh_time = timea[current] - timea[current^1];
#if defined(DEBUG)
        OPENDB;
        fprintf(debugfp, "Refresh: old=%lf, new=%lf, refresh=%f\n",
                         timea[current^1], timea[current], refresh_time);
        CLOSEDB;
#endif

        calc_sysinfo(refresh_time, 
                          &si[current], &si[current^1],
                          &vmk, &vm[current], &vm[current^1]);
	if (show_sys || show_disk_full || logmode) 
            calc_dkstat(refresh_time, dk[current], dk[current^1]);
	if (show_sys || show_inet_full || logmode)
            calc_ifnet(refresh_time, ifnets[current], ifnets[current^1]);
        if (show_smp || logmode)
            calc_cpuinfo(refresh_time, cpus[current], cpus[current^1]);
        if ((show_sys && show_nfs) || logmode)
            calc_nfsstat(refresh_time, &nfs[current], &nfs[current^1]);
#if defined(HAVE_AFS)
        if ((show_sys && show_afs) || logmode)
            calc_afsstat(refresh_time, &afs[current], &afs[current^1]);
#endif
        if ((show_sys && show_top) || (show_sys_summary && show_top_a) ||
            (logmode && logtop))
            calc_top(refresh_time, topcpu,
                     (logmode && logtop) ? nprocs : min(ntop, nprocs));

        if (do_curses) {
            /*
             * Display the stats
             */
            VALUECOLON;
            c_row = 0;
            c_col = 0;

            if (show_sys_summary) {
	        print_summary(refresh_time, (time_t) timea[current],
                              &si[current], &si[current^1],
                              &vmk, &vm[current], &vm[current^1]);
                if (show_smp)
                    print_cpuinfo(refresh_time, cpus[current], cpus[current^1]);
	        if (show_disk_full)
	            print_dkstat_full(refresh_time, dk[current], dk[current^1]);
	        if (show_inet_full)
	            print_ifnet(refresh_time, ifnets[current],
                                              ifnets[current^1]);
            }
	    if (show_sys) {
                print_sysinfo(refresh_time, (time_t) timea[current],
                              &si[current], &si[current^1],
                              &vmk, &vm[current], &vm[current^1]);
                if (show_nfs)
	            print_nfsstat(refresh_time, &nfs[current], &nfs[current^1]);
#if defined(HAVE_AFS)
                else if (show_afs)
	            print_afsstat(refresh_time, &afs[current], &afs[current^1]);
#endif
                getyx(stdscr, t_row, t_col);
                c_row = max(c_row, t_row);
	        print_dkstat(refresh_time, dk[current], dk[current^1]);
                getyx(stdscr, t_row, t_col);
                c_row = max(c_row, t_row);
	        print_ifnet(refresh_time, ifnets[current], ifnets[current^1]);
	    } 
            if ((show_sys && show_top) || (show_sys_summary && show_top_a)) {
                getyx(stdscr, t_row, t_col);
                c_row = max(c_row, t_row);
	        print_topcpu(refresh_time, topcpu, min(ntop, nprocs),
                             timea[current], c_row+1);
            }
            getyx(stdscr, t_row, t_col);
            c_row = max(c_row, t_row);
	    move(c_row+1, 0);
            clrtobot();
	    move(0, 0);
	    refresh();
        }

        /*
         * print out logging info
         */
        if (logmode) {
            if (dump_sample) {
#if defined(DEBUG)
                OPENDB;
                fprintf(debugfp, "Dumping sample\n");
                CLOSEDB;
#endif
                tmp_time = (int) timea[current^1];
                tmp_tm = *localtime(&tmp_time);
                if (openlogfile(&tmp_tm)) {
                    strftime(ts, 9, "%T", &tmp_tm);
                    tmp_time = (int) timea[current];
                    strftime(ts+strlen(ts), 10, " %T", localtime(&tmp_time));
                    strftime(ts+strlen(ts), 127, " %a %d %m %Y %b %w %U",
                                                   &tmp_tm);
                    fprintf(logfp, "sample %s %d %s\n", hostnm,
                                   sleep_sec, ts);
                    print_log(refresh_time,
                              &si[current], &si[current^1], &vmk,
                              &vm[current], &vm[current^1],
                              cpus[current], cpus[current^1],
                              dk[current], dk[current^1],
                              ifnets[current], ifnets[current^1],
                              &nfs[current], &nfs[current^1],
#if defined(HAVE_AFS)
                              &afs[current], &afs[current^1],
#endif
                              topcpu, nprocs);
                    closelog();
                }
            }


            /*
             * if log the interval ended, dump the data
             */
	    if (timea[current] >= nextinterval - 1) {
#if defined(DEBUG)
                OPENDB;
                fprintf(debugfp, "Dumping interval\n");
                CLOSEDB;
#endif
                /*
                 * Print interval stats
                 */
                tmp_time = (int) timea[2];
                tmp_tm = *localtime(&tmp_time);
                if (openlogfile(&tmp_tm)) {
                    strftime(ts, 9, "%T", &tmp_tm);
                    tmp_time = (int) timea[current];
                    strftime(ts+strlen(ts), 10, " %T", localtime(&tmp_time));
                    strftime(ts+strlen(ts), 127, " %a %d %m %Y %b %w %U",
                                                   &tmp_tm);
                    if (g_count > 0)
                        sprintf(ts+strlen(ts), " s%d", g_count);
                    fprintf(logfp, "interval %s %d %s\n", hostnm,
                                   interval_sec, ts);
                    print_log(timea[current] - timea[2],
                              &si[current], &si[2], &vmk,
                              &vm[current], &vm[2],
                              cpus[current], cpus[2],
                              dk[current], dk[2],
                              ifnets[current], ifnets[2],
                              &nfs[current], &nfs[2],
#if defined(HAVE_AFS)
                              &afs[current], &afs[2],
#endif
                              topcpu, nprocs);
                    dump_fs(1);
                    print_log_max(dk[current], ifnets[current]);
                    closelog();
                }

	        /*
                 * save current interval data
                 */
	        copy_log_variables();

                /*
                 * update when interval past
                 */
                gettimeofday(&tp, 0);
                while (nextinterval <= tp.tv_sec + 1)
	            nextinterval += interval_sec;
	    }
	}

        if (g_count > 0) {        /* do count samples ? */
            if (--g_count == 0)
                quit_monitor();
        }

        gettimeofday(&tp, 0);
        while (nextsample > tp.tv_sec) /* take care of sample_time changes */
                nextsample -= sleep_sec;
        if (nextsample < tp.tv_sec + 1) /* update when sample past */
            while (nextsample <= tp.tv_sec) 
                nextsample += sleep_sec;
     
    }

    /* NOTREACHED */
    if (do_curses)
       endwin();

    exit(0);

} /* main */



/***************************************************************************
 *                      UTILITY FUNCTIONS                                  *
 ***************************************************************************/

void
usage()
{

    fprintf(stderr,"Usage: %s [options]\n\n", progname);
    fprintf(stderr,"%s\n(c) 1991-1998 Jussi Maki (jmaki@csc.fi), "
                   "Marcel Mol (marcel@mesa.nl)\n\n", MONITOR_NAME);
    fprintf(stderr,"Monitor is a curses-based AIX system event monitor\n");
    fprintf(stderr,"The following options apply:\n");
    fprintf(stderr,
   "   -alternative  start in alternative display mode\n"
   "   -sample sec   set sample length (default %d seconds in interactive\n"
   "                 mode and %d seconds in logging mode)\n"
   "   -interval sec set interval length (default %d seconds)\n"
   "   -Sync         synchronise sample and interval time on day boundary\n"
   "   -smp          show SMP (multiprocessor) info in alternative display\n"
   "   -disk         show detailed disks info in alternative display\n"
   "   -D            sort disk info on read+write rate\n"
   "   -net          show detailed network info in alternative display\n"
   "   -Nfs          show NFS information"
#if defined(HAVE_AFS)
   " (instead of AFS)\n"
   "   -B            show AFS information (instead of NFS)"
#endif
   "\n"
   "   -top [nproc]  show top processes sorted by cpu-usage\n"
   "   -running      show only top processes consuming cpu\n"
   "   -user         show userid instead if usernames in top display\n"
   "   -log logfile  log mode, dump interval date to logfile. Logfile is used\n"
   "                 as a strftime format string, based on start of interval.\n"
   "                 The logfile will be opened/closed for every dump.\n"
   "   -L            also dump samples to logfile\n"
   "   -Toplog       also log top records to logfile\n"
   "   -F            also log non-local filesystem information (e.g. NFS)\n"
   "   -Active sec   number of seconds before user is counted as inactive\n"
   "   -Root         include root logins in user count\n"
   "   -Count num    number of samples (not intervals!) to display. Quit afterwards\n"
   "   -compress [prog] compress logfiles with prog (default %s)\n"
   "   -Highlight [none|bold|reverse%s] highlighting headers (default %s)\n"

   "\nPress h or ? in interactive mode to see its interactive commands\n"
   "This version of monitor is compiled for AIX %d.%d\n",

   DEFAULT_SAMPLE, DEFAULT_INTERVAL_SAMPLE, DEFAULT_INTERVAL,
   compprog,
#if defined(_HAS_COLOR)
   "|c|m|v",
#else
   "",
#endif
   highlight == A_STANDOUT ? "reverse"
              : highlight == A_BOLD ? "bold"
              : highlight ? "color" : "off",
    AIX_VERSION, AIX_RELEASE);

    exit(0);

} /* usage */



void
copy_log_variables()
{
    int i;

    timea[2] = timea[current];
    si[2]    = si[current];
    vm[2]    = vm[current];
    nfs[2]   = nfs[current];
#if defined(HAVE_AFS)
    afs[2]   = afs[current];
#endif
    if (!cpus[2])
        cpus[2] = (struct cpuinfo *)calloc(sizeof(struct cpuinfo), cpu_cnt);
    memcpy(cpus[2], cpus[current], sizeof(struct cpuinfo) * cpu_cnt);
    for (i = 0; i < dk_cnt; i++) {
        if (!dk[2][i])
            dk[2][i] = malloc(sizeof(struct dkstat));
        memcpy(dk[2][i], dk[current][i], sizeof(struct dkstat));
    }
    for (i = 0; i < if_cnt; i++) {
        if (!ifnets[2][i])
            ifnets[2][i] = malloc(sizeof(struct ifnet));
        memcpy(ifnets[2][i], ifnets[current][i], sizeof(struct ifnet));
    }
    init_max_values(dk[current], ifnets[current]);

    return;

} /* copy_log_variables */



int
openlogfile(struct tm *tim)
{
    struct stat st_buf;

    /*
     * Expect problems when using %D in format string.
     * It may generate '/' characters...
     */
    strftime(logfile, sizeof(logfile), logfilefmt, tim);
    if ((logfp = fopen(logfile, "a")) == NULL) {
	perror(logfile);
        return 0;
    }

    if (strcmp(logfile_old, logfile)) {
#if defined(DEBUG)
        OPENDB;
        fprintf(debugfp, "\tgonna print headers\n");
        CLOSEDB;
#endif
        stat(logfile, &st_buf);
        if (st_buf.st_size == 0)
            print_log_header();
        if (compress && *logfile_old != '\0') {
            /*
             * Compress file in a seperate process!
             */
            if (!fork()) {
                char buf[256];
                /*
                 * child will compress the old logfile
                 */
                if (do_curses) { /* not daemon mode */
                    fclose(stdin);
                    fclose(stdout);
                    fclose(stderr);
                }
                setpgrp(); /* detach from parent */
                strcpy(buf, compprog);
                strcat(buf, " ");
                strcat(buf, logfile_old);
                /*
                 * Should check if we are sh-ish or csh-ish ???
                 */
#if defined(DEBUG)
                strcat(buf, "> /tmp/moncompress.log 2>&1");
#else
                strcat(buf, "> /dev/null 2>&1");
#endif
                exit(system(buf));
            }
        }
        /*
         * Now remember the new filename
         */
        strcpy(logfile_old, logfile);
    }

#if defined(DEBUG)
    OPENDB;
    fprintf(debugfp, "\tlogfile opened\n");
    CLOSEDB;
#endif

    return 1;

} /* openlogfile */



void
sighandler(int signum)
{

    switch (signum) {
        default:
                fprintf(stderr, "unknown signal %d\n", signum);
                fflush(stderr); /* fall through */
        case SIGINT:
                quit_monitor();
        case SIGTSTP:
                if (do_curses) {
                    nocbreak();
                    endwin();
                }
                kill(getpid(), SIGSTOP);
                /* returned here from sigstop */
                if (do_curses)
                    start_screen();
                break;
	case SIGWINCH:
                if (do_curses) {
                    /* nocbreak(); */
                    endwin();
                    start_screen();
                }
		break;
    }
    signal(signum, sighandler);

    return;

} /* sighandler */



/*
 * wait for terminal input with secs-timeout
 */
void
wait_input_or_sleep(long secs, long usecs)
{
    fd_set input_fd;
    struct timeval timeout;

    timeout.tv_sec = secs;
    timeout.tv_usec = usecs;
    if (do_curses) {
        FD_ZERO(&input_fd);
        FD_SET(fileno(stdin), &input_fd);
        select(fileno(stdin)+1, &input_fd, 0, 0, &timeout);
    }
    else
        select(0, 0, 0, 0, &timeout);

    return;

} /* wait_input_or_sleep */



double
realtime()
{
    struct timeval tp;

    gettimeofday(&tp, 0);

    return ((double)tp.tv_sec + tp.tv_usec*1.0e-6);

} /* realtime */



void
start_screen()
{
    int i,j;

    numcolumns = atoi(termdef(fileno(stdout), 'c')); /* c like numcolumns */
    numlines   = atoi(termdef(fileno(stdout), 'l')); /* l like numlines */
    if (numcolumns == 0)
        numcolumns = 80;
    if (numlines == 0)
        numlines = 24;
    topflag_usersystemtime = numcolumns > 94 ? 1 : 0; 
    initscr();
#if defined(_HAS_COLOR)
    if ((namecol || valuecol || highlight&A_COLOR) && has_colors()) { 
        start_color();
        do_color = 1;
        /*
        fprintf(stderr, "pairs: %d, colors: %d\n", COLOR_PAIRS, COLORS);
        */
        for (i=0; i < COLORS; i++)
            for (j=0; j < COLORS; j++)
                init_pair(i*COLORS + j, j, i);
    }

/*
    else
        fprintf(stderr, " no colors\n");
*/
#endif
    cbreak();
    initscreen();

    return;

} /* start_screen */



void
quit_monitor()
{

    if (do_curses) {
        nocbreak();
        endwin();
    }

    exit(0);

} /* quit_monitor */



/*
 * checkinput is the subroutine to handle user input
 * this should only be called when fiullscreen mode is active
 */
void
checkinput()
{
    char inbuf[1024];
    static int nbytes;
    int inbytes;

    ioctl(fileno(stdin), FIONREAD, &nbytes);
    if (nbytes > 0) {
        inbytes = read(fileno(stdin), inbuf, nbytes);
        if (inbytes == 0 || inbytes == -1)
            quit_monitor();
        switch (inbuf[0]) {
            case 'q' : quit_monitor();
            case 12: /* ctrl-L */
                       initscreen();
                       break;
            case 'a' : show_sys ^= 1;
                       show_sys_summary = !show_sys;
                       initscreen();
#if defined(HAVE_AFS)
            case 'B' : show_afs ^= 1;
                       if (show_afs) {
                           get_afsstat(&afs[current]);
                           show_nfs = 0;
                       }
                       if (show_sys)
                           initscreen();
                       break;
#endif
            case 'c' : topflag_sort = TOPSORT_CPU;
                       break;
            case 'd' : show_disk_full ^= 1;
                       if (show_disk_full && show_sys_summary)
                           disk_cnt  = get_dkstat(dk[current]);
                       if (show_sys_summary)
                           initscreen();
                       break;
            case 'h' : 
            case '?' : display_help();
                       read(fileno(stdin), inbuf, 1);
                       initscreen();
                       break;
            case 'm' : topflag_sort = TOPSORT_RES;
                       break;
            case 'n' : show_inet_full ^= 1;
                       if (show_inet_full && show_sys_summary)
                           net_cnt  = get_ifnet(ifnets[current]);
                       if (show_sys_summary)
                           initscreen();
                       break;
            case 'N' : show_nfs ^= 1;
                       if (show_nfs) {
                           get_nfsstat(&nfs[current]);
#if defined(HAVE_AFS)
                           show_afs = 0;
#endif
                       }
                       if (show_sys)
                           initscreen();
                       break;
            case 'p' : topflag_sort = TOPSORT_PAGE;
                       break;
            case 's' : show_smp ^= 1;
                       if (show_smp) /* need to get get cpu_cnt now! */
                           cpu_cnt = get_cpuinfo(&cpus[current]);
                       if (show_sys_summary)
                           initscreen();
                       break;
            case 'r' : show_top_running ^= 1;
                       break;
            case 'R' : include_root ^= 1;
	               break;
            case 't' : if (show_sys)
                           show_top ^= 1;
                       else
                           show_top_a ^= 1;
                       if ((show_top || show_top_a) && nprocs == 0)
                           nprocs = get_topcpu(topcpu,
                                               logtop ? MAXTOPCPU : ntop);
                       break;
            case 'u' : topflag_showusername ^= 1;
                       break;
            case 'v' : topflag_sort = TOPSORT_VIR;
                       break;
            case '-' : sleep_sec--;
                       if (sleep_sec < 1)
                           sleep_sec = 1;
                       break;
            case '+' : sleep_sec++;
                       if (logmode && sleep_sec > interval_sec)
                           sleep_sec = interval_sec;
                       break;
            case '<' : sleep_sec -= 5;
                       if (sleep_sec < 1)
                           sleep_sec = 1;
                       break;
            case '>' : sleep_sec += 5;
                       if (logmode && sleep_sec > interval_sec)
                           sleep_sec = interval_sec;
                       break;
        }
    }

    return;

} /* checkinput */



void
display_help()
{
  clear();
  BOLDON;
  printw("monitor -- %s\ncompiled on AIX %d.%d running on AIX %s.%s\n\n",
          MONITOR_NAME, AIX_VERSION, AIX_RELEASE,
          unames.version, unames.release);
  BOLDOFF;
  NAMECOLON;
  printw("(C) Copyright 1991-1998 Marcel Mol (marcel@mesa.nl), Jussi Maki (jmaki@csc.fi)\n\n");
  printw("Command toggles available in full screen mode:\n");
  printw("\n");
  printw("    ^L             refresh screen\n");
  printw("     a             display alternative screen\n");
  printw("     d             display detailed disk info\n");
  printw("     n             display detailed network info\n");
  printw("     N             display NFS info (only on global screen)\n");
#if defined(HAVE_AFS)
  printw("     B             display AFS info (only on global screen)\n");
#endif
  printw("     s             display SMP multiprocessor cpuinfo\n");
  printw("     t             display top cpu processes\n");
  printw("     u             display username/userid in top\n");
  printw("     r             top only displays running processes\n");
  printw("     c, p, m, v    sort top on cpu, pagefautls, resident mem, virtual mem\n");
  printw("     R             include root in user count/activity\n");
  printw("     +, -          increase, decrease sample time by 1 second\n");
  printw("     >, <          increase, decrease sample time by 5 seconds\n");
  printw("     q             quit monitor\n");
  printw("     h or ?        this message\n");
  printw("\n");
  printw("The a, d, n and s command can be displayed together\n");
  printw("\n");
  NAMECOLOFF;
  BOLDON;
  printw("Press any key to continue.");
  BOLDOFF;

  refresh();

  return;

} /* display_help */

