/* -- AIX/6000 System monitor 
**
**     print_dkstat.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-1998 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <curses.h>
#include "get_dkstat.h"
#include "print_dkstat.h"
#include "print.h"

extern int show_disk_full;
extern int show_smp;
extern int sort_disks;
extern int cpu_cnt;

extern FILE *logfp;

/*
 * Variables to calculate rates
 */
float readrate[MAX_DISKS], writerate[MAX_DISKS], readsize[MAX_DISKS],
      writesize[MAX_DISKS], iorate[MAX_DISKS], seeksize[MAX_DISKS],
      diskbusy[MAX_DISKS];
int disk_sorted[MAX_DISKS];

/*
 * Variables for maintaining max values
 */
float readrate_max[MAX_DISKS], writerate_max[MAX_DISKS],
      readsize_max[MAX_DISKS], writesize_max[MAX_DISKS],
      iorate_max[MAX_DISKS], seeksize_max[MAX_DISKS],
      diskbusy_max[MAX_DISKS];

/*
 * Variables to determine screen location for disk figures
 */
int disk_x, disk_y;

/*
 * Local function declarations
 */
int cmp_dkstat(int *i, int *j);

/*******************************************************************
 * Functions
 *******************************************************************/

#define DKiDELTA(i,a)  ((dk1[i]->a) - (dk2[i]->a))

int
cmp_dkstat(int *i, int *j)
{
    int speed_i = readrate[*i] + writerate[*i];
    int speed_j = readrate[*j] + writerate[*j];

    if (speed_i > speed_j)
        return -1;
    else if (speed_i < speed_j)
        return 1;

    return 0;

} /* cmp_dkstat */



void
calc_dkstat(double refresh_time, struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i;
    long xfers;

    for (i = 0; i < dk_cnt; i++) {
	xfers        = DKiDELTA(i, dk_xfers);
        readrate[i]  = DKiDELTA(i, dk_rblks) * dk1[i]->dk_bsize /
                       1024.0 / refresh_time;
        writerate[i] = DKiDELTA(i, dk_wblks) * dk1[i]->dk_bsize /
                       1024.0 / refresh_time;
        readsize[i]  = xfers == 0 ? 0 :
                      DKiDELTA(i, dk_rblks) * dk1[i]->dk_bsize / 1024.0 / xfers;
	writesize[i] = xfers == 0 ? 0 :
                      DKiDELTA(i, dk_wblks) * dk1[i]->dk_bsize / 1024.0 / xfers;
        iorate[i]    = xfers / refresh_time;
        seeksize[i]  = DKiDELTA(i, dk_seek) / refresh_time,
	diskbusy[i]  = DKiDELTA(i, dk_time) / refresh_time;
        SETMAX(readrate[i],  readrate_max[i]);
        SETMAX(writerate[i], writerate_max[i]);
        SETMAX(readsize[i],  readsize_max[i]);
        SETMAX(writesize[i], writesize_max[i]);
        SETMAX(iorate[i],    iorate_max[i]);
        SETMAX(seeksize[i],  seeksize_max[i]);
        SETMAX(diskbusy[i],  diskbusy_max[i]);
        disk_sorted[i] = i;
    }

    if (sort_disks)
        qsort((void *)disk_sorted, (size_t)dk_cnt, sizeof(int), cmp_dkstat);

    return;

} /* calc_dkstat */



void
print_dkstat_init()
{

    disk_x = 0; disk_y = 13;
    move(disk_y + 0, disk_x);
    BOLDON;
    printw("IO (kB/s) read  write busy%%");
    BOLDOFF;

    return;

} /* print_dkstat_init */



void
print_dkstat(double refresh_time, struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i, j;
    
    for (i = 0; i < dk_cnt; i++) {
	move(disk_y + 1 + i, disk_x);
        j = sort_disks ? disk_sorted[i] : i;
        NAMECOLON;
	printw("%-7s", dk1[j]->diskname);
        NAMECOLOFF;
        VALUECOLON;
	printw(" %6.1f %6.1f  %3.0f", readrate[j], writerate[j], diskbusy[j]);
        VALUECOLOFF;
    }

    return;

} /* printf_dkstat */



void
print_dkstat_full_init()
{

    disk_x = 0; disk_y = 5;
    if (show_smp)
        disk_y += (cpu_cnt + 2);
    move(disk_y + 0, disk_x);
    BOLDON;
    printw("DiskIO    read  write     rsize  wsize  xfers seeks blksize"
           " xrate busy");
    BOLDOFF;

    return;

} /* print_dkstat_full_init */



void
print_dkstat_full(double refresh_time,
                  struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i, j, active;
    struct dkstat sum;

    active = summarize_dkstat(&sum, dk1, dk2);
    for (i = 0; i < dk_cnt; i++) {
	move(disk_y + 1 + i, disk_x);
        j = sort_disks ? disk_sorted[i] : i;
	printw("%-7s %6.1f %6.1f kB/s %4.1f %4.1f kB %5.1f %5.1f %7d %4d"
               " %3.0f%%",
	       dk1[j]->diskname, readrate[j], writerate[j],
               readsize[j], writesize[j], iorate[j],
               seeksize[j], dk1[j]->dk_bsize, dk1[j]->dk_xrate, diskbusy[j]);
    }
    move(disk_y + 1 + i, disk_x);
    printw("Total   %6.1f %6.1f kB/s %4.1f %4.1f kB %5.1f %5.1f %7d %4d"
           " %3.0f%%",
           sum.dk_rblks * sum.dk_bsize / 1024.0 / refresh_time,
           sum.dk_wblks * sum.dk_bsize / 1024.0 / refresh_time,
           sum.dk_xfers ? sum.dk_rblks * sum.dk_bsize /1024.0/ sum.dk_xfers : 0,
           sum.dk_xfers ? sum.dk_wblks * sum.dk_bsize /1024.0/ sum.dk_xfers : 0,
           sum.dk_xfers / refresh_time, 
           sum.dk_seek / refresh_time,
           sum.dk_bsize,
           sum.dk_xrate,
           sum.dk_time / refresh_time);
    move(disk_y + 2 + i, disk_x);
    printw("Active disks: %d", active);


    return;

} /* print_dkstat_full */



void
log_dkstat_header()
{

    fprintf(logfp, "# device[_max] name readrate writerate readsize writesize"
                   " iorate seeksize diskbusy xrate\n");
    fprintf(logfp, "# device    readrate     kB read / second\n");
    fprintf(logfp, "# device    writerate    kB written / second\n");
    fprintf(logfp, "# device    readsize     kB read / io\n");
    fprintf(logfp, "# device    writesize    kB written / io\n");
    fprintf(logfp, "# device    iorate       io's / second\n");
    fprintf(logfp, "# device    seeksize     cylinders seek / io\n");
    fprintf(logfp, "# device    diskbusy     busy%%\n");
    fprintf(logfp, "# device    xrate        disk max transfer rate\n");

    return;

} /* log_dkstat_header */



void
log_dkstat(double refresh_time, struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i;

    for (i = 0; i < dk_cnt; i++)
	fprintf(logfp, "device %s %.1f %.1f %.1f %.1f %.1f %.1f %.0f %ld\n",
	               dk1[i]->diskname, readrate[i], writerate[i], readsize[i],
                       writesize[i], iorate[i], seeksize[i], diskbusy[i],
                       dk1[i]->dk_xrate);

    return;

} /* log_dkstat */



void
max_dkstat_init()
{
    int i;

    for (i = 0; i < dk_cnt; i++) {
        readrate_max[i] = writerate_max[i] = 0.0;
        readsize_max[i] = writesize_max[i] = 0.0;
        iorate_max[i] = seeksize_max[i] = 0.0;
        diskbusy_max[i] = 0.0;
    }

    return;

} /* max_dkstat_init */



void
log_dkstat_max(struct dkstat *dk[])
{
    int i;

    for (i = 0; dk[i]; i++)
	fprintf(logfp, "device_max %s %.1f %.1f %.1f %.1f %.1f %.1f %.0f\n",
	               dk[i]->diskname, readrate_max[i], writerate_max[i],
                       readsize_max[i], writesize_max[i],
                       iorate_max[i], seeksize_max[i], diskbusy_max[i]);

    return;

} /* log_dkstat_max */
