/* -- AIX/6000 System monitor 
**
**     print_top.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-1998 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <curses.h>
#include <pwd.h>
#include "get_top.h"
#include "print.h"


extern int show_top_running;    /* show process top list */

extern int topflag_usersystemtime;
extern int topflag_showusername;
extern int numlines;
extern FILE *logfp;
extern int valuecol;

void cpu_timestr(double cputime, char* cpustr);

char *statstr[] = {"non", "slp", "2?", "run",
                   "cre", "zom", "stp", "run", "swp"};
/*
 * Process states AIX 3 
 * SNONE           0       slot is available
 * SSLEEP          1       awaiting an event
 * SRUN            3       runable
 * SIDL            4       intermediate state in process creation
 * SZOMB           5       intermediate state in process termination
 * SSTOP           6       process is stopped
 *
 * Process states AIX 4
 * SNONE           0       slot is available
 * SIDL            4       process is created
 * SZOMB           5       process is dying
 * SSTOP           6       process is stopped
 * SACTIVE         7       process is active
 * SSWAP           8       process is swapped
 */


void
print_topcpu(double refresh_time, topcpu_t *top, int ntop, time_t now, int y)
{
    double cputime;
    int i;
    char schar, rchar;
    u_long memsize, ressize;
    struct passwd *passwd;
    char username[16];
    char cpustr[16];
    char cpustr2[16];
    static int maxtop = 0, old_maxtop = 0;

    move(y, 0);
    if (!topflag_usersystemtime) {
        BOLDON;
        printw("   PID USER     PRI NICE SIZE   RES STAT     TIME  "
               "CPU%%     COMMAND\n");
        BOLDOFF;
        VALUECOLON;
        y++;
        for (i = 0; i < ntop && y < numlines; i++) {
            if (show_top_running && !top[i].cputime_prs)
                break;

            move(y, 0);
            y++;
            cputime = top[i].cpu_utime + top[i].cpu_stime;
            cpu_timestr(cputime, cpustr);
            if (topflag_showusername) {
	        passwd = getpwuid(top[i].uid);
	        strcpy(username, passwd->pw_name);
            } else
	        sprintf(username, "%ld", top[i].uid);
            memsize = top[i].memsize_1k;
            if (memsize > 1024*9999) {
                memsize /= (1024*1024); 
                schar = 'G';
            }
            else if (memsize > 9999) {
                memsize /= 1024; 
                schar = 'M';
            }
            else
                schar = 'k';
            ressize = top[i].ressize_1k;
            if (ressize > 1024*9999) {
                ressize /= (1024*1024); 
                rchar = 'G';
            }
            else if (ressize > 9999) {
                ressize /= 1024; 
                rchar = 'M';
            }
            else
                rchar = 'k';
            printw("%6d %-8s %3d %3d %4d%c %4d%c %c%-3s %8s %4.1f/%4.1f %s\n",
	           top[i].pid, username, top[i].pri, top[i].nice - 20,
	           memsize, schar, ressize, rchar,
                   top[i].tty ? 'F' : ' ', statstr[top[i].stat], cpustr,
	           top[i].cputime_prs/10.0,
                   (float) 100 * cputime / (float)(now - top[i].starttime),
                   top[i].progname);
        }
    }
    else {
        BOLDON;
        printw("   PID USER     PRI NICE SIZE   RES PFLTS STAT     "
               "USER/SYSTIME   CPU%%     COMMAND\n");
        BOLDOFF;
        VALUECOLON;
        y++;
        for (i = 0; i < ntop && y < numlines; i++) {
            if (show_top_running && !top[i].cputime_prs)
                break;

            move(y, 0);
            y++;
            if (topflag_showusername) {
	        passwd = getpwuid(top[i].uid);
	        strcpy(username, passwd->pw_name);
            } else
	        sprintf(username,"%ld",top[i].uid);
            cputime = top[i].cpu_utime + top[i].cpu_stime;
            cpu_timestr(top[i].cpu_utime, cpustr);
            cpu_timestr(top[i].cpu_stime, cpustr2);
            memsize = top[i].memsize_1k;
            if (memsize > 1024*9999) {
                memsize /= (1024*1024); 
                schar = 'G';
            }
            else if (memsize > 9999) {
                memsize /= 1024; 
                schar = 'M';
            }
            else
                schar = 'k';
            ressize = top[i].ressize_1k;
            if (ressize > 1024*9999) {
                ressize /= (1024*1024); 
                rchar = 'G';
            }
            else if (ressize > 9999) {
                ressize /= 1024; 
                rchar = 'M';
            }
            else
                rchar = 'k';
            printw("%6d %-8s %3d %3d %4d%c %4d%c %5.1f %c%-3s %8s/%8s"
                   " %4.1f/%4.1f %s\n",
	           top[i].pid, username, top[i].pri, top[i].nice-20,
	           memsize, schar, ressize, rchar,
	           top[i].deltapageflt/refresh_time,
                   top[i].tty ? 'F' : ' ', statstr[top[i].stat],
                   cpustr, cpustr2,
	           top[i].cputime_prs/10.0,
                   (float) 100 * cputime / (float)(now - top[i].starttime),
	           top[i].progname);
        }
    }

    /*
     * clear out unwanted numlines from previous interval
     */
    old_maxtop = maxtop;
    maxtop = y;
    for (; y < old_maxtop && y < numlines;  y++)
        printw("\n");

    return;

} /* print_topcpu */



void
cpu_timestr(double cputime, char* cpustr)
{

    if  (cputime > 100*24*3600L)
        sprintf(cpustr, "%5ld+%02ld", (long)cputime / (24*3600),
                      ((long)cputime % (24*3600)) / 3600);
    else if  (cputime > 100*3600L)    /* more 100 hours */
        sprintf(cpustr, "%2ld+%02ld:%02ld", (long)cputime / (24*3600),
                        ((long)cputime % (24*3600)) / 3600,
                        ((long)cputime % 3600) / 60);
    else if  (cputime >= 3600)        /* more than 1 hour */
        sprintf(cpustr, "%2ld:%02ld:%02ld", (long)cputime / 3600,
                        ((long)cputime % 3600) / 60, (long)cputime % 60);
    else
        sprintf(cpustr, "%5d:%02d",
                        (int) cputime / 60, (int) cputime % 60);

    return;

} /* cpu_timestr */



void
log_top_header()
{

    fprintf(logfp, "# nprocs  number of processes in process list\n");
    fprintf(logfp, "# process PID USER PRI NICE SIZE RES PFLTS STAT USER "
                   "SYSTIME CPU%% START COMMAND\n");

    return;

} /* log_top_header */



void
calc_top(double refresh_time, topcpu_t *top, int nproc)
{

    /* Nothing yet */

    return;

} /* calc_top */



void
log_top(double refresh_time, topcpu_t *top, int nprocs)
{
    int i;

    fprintf(logfp, "nprocs %d\n", nprocs);

    for (i = 0; i < nprocs; i++) {
       if (show_top_running && !top[i].cputime_prs)
           continue;
       fprintf(logfp,
              "process %u %u %lu %u %u %lu %lu %.1f %s %.2f %.2f %.1f %lu %s %s\n",
	           top[i].pid, top[i].ppid, top[i].uid, top[i].pri, top[i].nice,
	           top[i].memsize_1k, top[i].ressize_1k,
	           top[i].deltapageflt/refresh_time, statstr[top[i].stat],
	           top[i].cpu_utime, top[i].cpu_stime,
	           top[i].cputime_prs/10.0,
                   top[i].starttime, top[i].progname, 
                   top[i].tty ? "fg" : "bg");
    }

    return;

} /* log_top */



void
log_top_max()
{

    /* Nothing yet */

    fflush(logfp);

    return;

} /* log_top_max */



void
max_top_init()
{

    /* Nothing yet */

    return;

} /* max_top_init */
