/* monitor -- AIX RS/6000 System monitor 
 *
 * Copyright (c) 1991 - 1996 Jussi Maki. All Rights Reserved.
 * NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO RE-DISTRIBUTE 
 * THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL COPYRIGHTS.
 * Email: jmaki@csc.fi
 */
/* Author: Jussi Maki
 *         Center for Scientific Computing
 *         Tietotie 6
 *         P.O. Box 405
 *         FIN-02101 Espoo
 *         FINLAND
 *
 * Phone:   +358-0-457 2747
 * Telefax: +358-0-457 2302
 *
 * Internet-mail: jmaki@csc.fi
 *
 * created:  15.5.1991 v1.0        first release
 *           14.5.1994 v1.12 -0      moved previous history to HISTORY file
 *           15.6.1994 v1.12 -4
 * latest:   18.7.1995 v1.13-b0     support for plain ascii output
 *           31.8.1995 v1.13-b1     cludged support for FDDI in AIX4
 *           16.11.1995 v1.13-b2    bugfixes for the network part (beta)
 *           06.02.1996 v1.14-b0    support for SMP machines (-smp)
 *           25.03.1996 v1.14       time to release the 1.14
 *            
 */

#define MONITOR_NAME "AIX monitor v1.14:"

#include <curses.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <locale.h>

#include <sys/types.h>
#include <sys/select.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <time.h>
#include <sys/time.h>
#include <pwd.h>
#include <stdarg.h>

#include "get_nfsstat.h"
#include "get_topcpu.h"
#include "get_sysvminfo.h"
#include "get_dkstat.h"
#include "get_ifnet.h"
#include "get_cpuinfo.h"
#include "getloadavg.h"

#define MONITOR_NTOP_DEFAULT 16
#define MAXTOPCPU 1000
#ifndef min
#define min(a,b) ((a)<(b)?(a):(b))
#endif
#ifndef max
#define max(a,b) ((a)>(b)?(a):(b))
#endif

double realtime();
void sighandler(int signum);
void print_sysinfo(double refresh_time, 
	      struct sysvminfo *sv1, struct sysvminfo *sv2);
int mon_print(int y, int x, const char *fmt, ...);
void display_help();


int topflag_showusername;
int topflag_usersystemtime;
int dk_cnt;  /* number of disks in system  */
int sleep_sec=10;
int end_monitor=0;
int monflag_show_nfs=0;
int g_ntop = 17; /* top procs to show by default */
int show_all=0,show_top=0;
int show_disk_full=0;
int show_net_full=0;
int show_smp=0;
int columns,lines;
int show_top_running=0;
int plain_ascii=0;
int g_count=-1;

int main(int argc,char *argv[])
{
    double update_time[2];
    double refresh_time;
    struct sysvminfo *sv[2];
    struct dkstat    *dk[2];
    struct ifnet     *ifnet[2];
    struct cpuinfo   *cpus[2]={NULL,NULL}; int n_cpus;
    nfsstat_t        *nfs[2];
    topcpu_t         topcpu[MAXTOPCPU];
    int ntop_found;
    int s_old=0, s_new=0;

    topflag_showusername=1;
    setlocale(LC_NUMERIC,"C"); /* be sure that decimal-point is "." */

    getscreensize(&columns,&lines,&plain_ascii);
    if (columns > 94) topflag_usersystemtime=1; 
    g_ntop = lines - 7; /* for default ntop leave space for display */
    parse_args(argc, argv); /* this will also set the ntop if '-top NUM' */
    g_ntop = min(g_ntop, MAXTOPCPU);

    /*signal(SIGTSTP,sighandler);*/
    /*    signal(SIGINT,sighandler);*/
    /*    signal(SIGQUIT,sighandler);*/

    if (!plain_ascii) { initscr(); clear(); cbreak(); }

    if (show_top) ntop_found = get_topcpu(topcpu, g_ntop);
    update_time[s_new] = realtime();
    get_sysvminfo(&sv[s_new]);
    get_dkstat(&dk[s_new]);
    get_ifnet(&ifnet[s_new]);
    get_nfsstat(&nfs[s_new]);
    get_cpuinfo(&cpus[s_new], &n_cpus);
    s_old=s_new; s_new = (s_new+1)&1;
    if (g_count==0) {
      g_count=1;
      usleep(100000);
    } else {
      sleep(1);
    }

    while (! end_monitor) {
      get_sysvminfo(&sv[s_new]);
      get_dkstat(&dk[s_new]);
      get_ifnet(&ifnet[s_new]);
      get_nfsstat(&nfs[s_new]);
      get_cpuinfo(&cpus[s_new], &n_cpus);
      if (show_top) ntop_found = get_topcpu(topcpu, g_ntop);
      update_time[s_new] = realtime();
      refresh_time = update_time[s_new] - update_time[s_old];
      if (show_smp) {
	print_summary(refresh_time, sv[s_new],     sv[s_old]);
	print_cpuinfo(refresh_time, cpus, n_cpus, s_old);
      } else if (show_net_full) {
	print_summary(refresh_time, sv[s_new],     sv[s_old]);
	print_ifnet_full(refresh_time, ifnet[s_new], ifnet[s_old]);
      } else if (show_disk_full) {
	print_summary(refresh_time, sv[s_new],     sv[s_old]);
	print_dkstat_full(refresh_time, dk[s_new], dk[s_old]);
      } else if (show_top && !show_all) {
	print_summary(refresh_time, sv[s_new],     sv[s_old]);
      } else {
	print_sysinfo(refresh_time, sv[s_new],     sv[s_old]);
	print_ifnet(refresh_time,   ifnet[s_new],  ifnet[s_old]);
	print_dkstat(refresh_time,  dk[s_new],     dk[s_old]);
	print_nfsstat(refresh_time, nfs[s_new],    nfs[s_old]);
      } 
      if (show_top) {
	print_topcpu(refresh_time, topcpu, ntop_found);
      }
      if (!plain_ascii) { move(0,0); refresh(); }
      s_old=s_new; s_new = (s_new+1)&1;
      if (--g_count == 0) break;
      wait_input_or_sleep(sleep_sec);
      checkinput();
    }
    if (!plain_ascii) {
      endwin();
    }
}

parse_args(int argc, char **argv)
{
    while (argc>1) {
      if (argv[1][0] == '-') {
	switch(argv[1][1]) {
	case 's':
	  if (strcmp(argv[1], "-smp")==0) {
	    show_smp=1;
	  } else {
	    sleep_sec = atoi(argv[2]); argc--;argv++;
	    if (sleep_sec < 1) sleep_sec=1;
	  }
	  break;
	case 'p': /* plain_ascii */
	  plain_ascii=1;
	  lines = 40000; /* no limits on line length */
	  break;
	case 'a': /* -all */
	  show_all = 1;  show_top = 1;
	  g_ntop = lines-27;
	  if (g_ntop < 0) show_top = 0;
	  break;
	case 't': /* -top [nproc] */
	  show_top = 1;
	  if (atoi(argv[2])>0) {
	    g_ntop=atoi(argv[2]);
	    argc--;argv++;
	  } else {
	    g_ntop = MONITOR_NTOP_DEFAULT;
	  }
	  break;
	case 'r':
	  show_top_running=1;
	  break;
	case 'u': /* -u   ... dont show user names in top */
	  topflag_showusername=0;
	  break;
	case 'd': /* -d[isk] */
	  show_disk_full=1;
	  break;
	case 'n': /* -n[et] */
	  show_net_full=1;
	  break;
	case 'c': /* -c[ount] count */
	  g_count=atoi(argv[2]); argc--;argv++;
	  break;
	default:
	  fprintf(stderr,"Usage: monitor [-s sleep] [-top [nproc]] [-u] [-all] [-d]\n");
	  fprintf(stderr,"Monitor is a curses-based AIX system event monitor\n");
	  fprintf(stderr,"     -s sleep      set refresh time\n");
	  fprintf(stderr,"     -top [nproc]  show top processes sorted by cpu-usage\n");
	  fprintf(stderr,"     -u            don't show usernames in top display\n");
	  fprintf(stderr,"     -all          show all variable (needs high window)\n");
	  fprintf(stderr,"     -disk         show more on disks\n");
	  fprintf(stderr,"     -net          show more on network\n");
	  fprintf(stderr,"     -smp          show SMP (multiprocessor) info\n");
	  fprintf(stderr,"     -p            show plain ascii output\n");
	  fprintf(stderr,"     -c count      show count times\n");
	  exit(0);
	  break;
	}
      }
      argv++;argc--;
    }
}

/* wait for terminal input with secs-timeout */
wait_input_or_sleep(int secs)
{
  fd_set input_fd;
  struct timeval timeout;

  FD_ZERO(&input_fd);
  FD_SET(fileno(stdin),&input_fd);
  timeout.tv_sec=secs; timeout.tv_usec=0;
  select(fileno(stdin)+1,&input_fd,0,0,&timeout);
}

getscreensize(int *cols, int *rows, int *not_a_tty)
{
  if (isatty(fileno(stdout))) {
    *cols = atoi((char *)termdef(fileno(stdout),'c')); /* c like columns */
    *rows = atoi((char *)termdef(fileno(stdout),'l')); /* l like lines */
    if (*cols==0) *cols=80;
    if (*rows==0) *rows=24;
    *not_a_tty=0;
  } else {
    *not_a_tty=1;
    *cols=100;
    *rows=40000;
  }
}

double realtime()
{
  struct timeval tp;
  gettimeofday(&tp,0);
  return((double)tp.tv_sec+tp.tv_usec*1.0e-6);
}

quit_monitor()
{
  if (!plain_ascii) {
    nocbreak();
    endwin();
  }
  exit(0);
}

/* checkinput is the subroutine to handle user input */
checkinput()
{
  char inbuf[1024];
  int nbytes;
  int inbytes;

  if ((nbytes=bytesreadable(fileno(stdin))) > 0) {
    inbytes = read(fileno(stdin),inbuf,nbytes);
    if (inbuf[0]=='q' || inbytes == 0 || inbytes == -1) {
      quit_monitor();
    }
    switch (inbuf[0]) {
    case 12:  /* cltr-L ^L */
      clear();     
      break;
    case 's': /* show smp */
      show_smp = (show_smp+1)&1;
      clear();
      break;
    case 'd': /* toggle show disk full */
      show_disk_full = (show_disk_full+1)&1;
      clear();
      break;
    case 'n': /* toggle show net full */
      show_net_full = (show_net_full+1)&1;
      clear();
      break;
    case 't':
      show_top = (show_top+1)&1;
      clear();
      break;
    case 'h':
    case '?':
      display_help();
      while (bytesreadable(fileno(stdin))==0) usleep(200000);
      read(fileno(stdin),inbuf,1);
      clear();
      break;
    }
  }
}

void display_help()
{
  clear();
  printw("monitor -- IBM RS/6000 %s\n",MONITOR_NAME);
  printw("(C) Copyright 1991-1996 Jussi Maki;  Email: jmaki@csc.fi\n");
  printw("\n");
  printw("Monitor displayes various AIX utilization characterics.\n");
  printw("Following command line arguments and commands are available:\n");
  
  printw("     -s sleep      set refresh time\n");
  printw("     -top [nproc]  show top processes sorted by cpu-usage\n");
  printw("     -u            don't show usernames in top display\n");
  printw("     -all          show all variable (needs high window)\n");
  printw("     -disk         show more on disks\n");
  printw("     -net          show more on network\n");
  printw("     -smp          show SMP (multiprocessor) info\n");
  printw("     -p            show plain ascii output\n");
  printw("     -c count      show count times\n");
  printw("\n");
  printw("Following character commands are available in full screen mode:\n");
  printw("    ^L             refresh the screen\n");
  printw("     s             display SMP multiprosessor cpuinfo\n");
  printw("     t             display top cpu processes\n");
  printw("     n             display detailed network info\n");
  printw("     d             display detailed disk info\n");
  printw("\n");
  printw("Press any key to continue.\n");
  refresh();
}

int bytesreadable(int in)
{
  static int bytes;
  ioctl(in,FIONREAD,&bytes);
  return(bytes);
}

/**********************************************************************/

#define BARLEN 72
#define SIDELTA(a) (si->a - si2->a)
#define VMDELTA(a) (vm->a - vm2->a)


void print_sysinfo(double refresh_time, 
	      struct sysvminfo *sv1, struct sysvminfo *sv2)
{
    double cpu_sum;
    double str_cpu_sum;
    double swp_proc;
    double runnable,runque_tmp, runocc_tmp;
    char bar[BARLEN];
    time_t time1;
    double loadv[3];
    char hostnm[128];
    int x,y;
    struct sysinfo *si,*si2;
    struct vmker *vmk;
    struct vminfo *vm,*vm2;
    char *timestrp;

    si = &sv1->sysinfo;
    si2= &sv2->sysinfo;
    vm = &sv1->vminfo;
    vm2= &sv2->vminfo;
    vmk= &sv1->vmker;

    gethostname(hostnm,sizeof(hostnm));
    time1=time(0);
    timestrp = ctime(&time1);
    timestrp[strlen(timestrp)-1]='\0'; /* remove last \n */
    mon_print(0,0, "%-19s %-31s %s",MONITOR_NAME,hostnm,timestrp);
    cpu_sum = SIDELTA(cpu[CPU_IDLE]) + SIDELTA(cpu[CPU_USER]) 
      + SIDELTA(cpu[CPU_KERNEL]) + SIDELTA(cpu[CPU_WAIT]);
    str_cpu_sum = cpu_sum/(double)BARLEN;
    cpu_sum = cpu_sum/100.0;

    mon_print(1,0, "Sys %4.1lf%% Wait %4.1lf%% User %4.1lf%% Idle %5.1lf%%           Refresh: %5.2f s",
	   SIDELTA(cpu[CPU_KERNEL])/cpu_sum,
	   SIDELTA(cpu[CPU_WAIT])/cpu_sum,
	   SIDELTA(cpu[CPU_USER])/cpu_sum,
	   SIDELTA(cpu[CPU_IDLE])/cpu_sum, 
	   refresh_time);
    mon_print(2,0,"0%%             25%%              50%%               75%%              100%%");
    bar[0]=0;
    strchgen(bar,'=',(int)(SIDELTA(cpu[CPU_KERNEL])/str_cpu_sum));
    strchgen(bar,'W',(int)(SIDELTA(cpu[CPU_WAIT])/str_cpu_sum));
    strchgen(bar,'>',(int)(SIDELTA(cpu[CPU_USER])/str_cpu_sum));
    strchgen(bar,'.',(int)(SIDELTA(cpu[CPU_IDLE])/str_cpu_sum));
    if (!plain_ascii) { move(3,0); clrtoeol(); }
    mon_print(3,0,"%s",bar);

    getloadavg(loadv,3);
    runque_tmp = (double) SIDELTA(runque);
    runocc_tmp = (double) SIDELTA(runocc);
    if(runocc_tmp==0.0)
      runnable=0.0;
    else
      runnable=runque_tmp/runocc_tmp-1.0;
    mon_print(4,0,"Runnable processes %5.2lf load average: %5.2lf, %5.2lf, %5.2lf\n",
	  (runnable+1.0)*SIDELTA(runocc)/refresh_time, loadv[0],loadv[1],loadv[2]);
    x=0;y=6;
    mon_print(y+0,x, "Memory    Real     Virtual     Paging (4kB)");
    mon_print(y+1,x, "free   %6.1lf MB %6.1lf MB  %6.1f pgfaults",
	      vmk->freemem*4/1024.0,vmk->freevmem*4/1024.0,
	      VMDELTA(pgexct)/refresh_time);
    mon_print(y+2,x, "procs  %6.1lf MB %6.1lf MB   %5.1f pgin", 
	      (vmk->totalmem-vmk->freemem-vmk->numperm)*4/1024.0,
	      (vmk->totalvmem - vmk->freevmem)*4/1024.0,
	      VMDELTA(pageins)/refresh_time);
    mon_print(y+3,x, "files  %6.1lf MB             %5.1f pgout", 
	      vmk->numperm*4/1024.0,
	      VMDELTA(pageouts)/refresh_time);
    mon_print(y+4,x, "total  %6.1lf MB %6.1lf MB   %5.1f pgsin",
	      (vmk->totalmem)*4/1024.0,vmk->totalvmem*4/1024.0,
	      VMDELTA(pgspgins)/refresh_time);
    mon_print(y+5,x, "                             %5.1f pgsout", 
	      VMDELTA(pgspgouts)/refresh_time);

    x=48;y=6;
    mon_print(y+0,x-2, "Process events     File/TTY-IO");
    mon_print(y+1,x, "%5.0f pswitch %7d iget", 
	      SIDELTA(pswitch)/refresh_time,(int)
	      (SIDELTA(iget)/refresh_time));
    mon_print(y+2,x, "%5.0f syscall %7d namei", 
	      SIDELTA(syscall)/refresh_time,
	      (int)(SIDELTA(namei)/refresh_time));
    mon_print(y+3,x, "%5.0f read    %7d dirblk",  
	      SIDELTA(sysread)/refresh_time,
	      (int)(SIDELTA(dirblk)/refresh_time));
    mon_print(y+4,x, "%5.0f write   %7d readch", 
	      SIDELTA(syswrite)/refresh_time,
	      (int)(SIDELTA(readch)/refresh_time));
    mon_print(y+5,x, "%5.0f fork    %7d writech",  
	      SIDELTA(sysfork)/refresh_time,
	      (int)(SIDELTA(writech)/refresh_time));
    mon_print(y+6,x, "%5.0f exec    %7d ttyrawch", 
	      SIDELTA(sysexec)/refresh_time,
	      (int)(SIDELTA(rawch)/refresh_time));
    mon_print(y+7,x, "%5.0f rcvint  %7d ttycanch", 
	      SIDELTA(rcvint)/refresh_time,
	      (int)(SIDELTA(canch)/refresh_time));
    mon_print(y+8,x, "%5.0f xmtint  %7d ttyoutch",
	      SIDELTA(xmtint)/refresh_time,
	      (int)(SIDELTA(outch)/refresh_time));
    
}

print_summary(double refresh_time,struct sysvminfo *sv1,struct sysvminfo *sv2)
{
  time_t time1;
  double loadv[3];
  char hostnm[128];
  double cpu_sum;
  struct sysinfo *si,*si2;
  struct vmker *vmk;
  struct vminfo *vm,*vm2;
  char  *timestrp;

  si = &sv1->sysinfo;
  si2= &sv2->sysinfo;
  vm = &sv1->vminfo;
  vm2= &sv2->vminfo;
  vmk= &sv1->vmker;

  /* hostname loadavg date-time */
  /* cpu-states */
  /* memory-states */
  gethostname(hostnm,sizeof(hostnm));
  getloadavg(loadv,3);
  time1=time(0);
  timestrp = ctime(&time1);
  timestrp[strlen(timestrp)-1]='\0'; /* remove last \n */
  cpu_sum = SIDELTA(cpu[CPU_IDLE]) + SIDELTA(cpu[CPU_USER]) 
      + SIDELTA(cpu[CPU_KERNEL]) + SIDELTA(cpu[CPU_WAIT]);
  cpu_sum = cpu_sum/100.0;
  mon_print(0,0, "%-16s load averages: %5.2lf, %5.2lf, %5.2lf   %s",
		    hostnm,
		    loadv[0],loadv[1],loadv[2],
		    timestrp);
  mon_print(1,0,"Cpu states:     %5.1f%% user, %5.1f%% system, %5.1f%% wait, %5.1f%% idle",
	   SIDELTA(cpu[CPU_USER])/cpu_sum,
	   SIDELTA(cpu[CPU_KERNEL])/cpu_sum,
	   SIDELTA(cpu[CPU_WAIT])/cpu_sum,
	   SIDELTA(cpu[CPU_IDLE])/cpu_sum);
  mon_print(2,0, "Real memory:    %6.1lfM free %6.1lfM procs %6.1lfM files %6.1lfM total",
	 vmk->freemem*4/1024.0,
	 (vmk->totalmem-vmk->numperm-vmk->freemem)*4/1024.0, 
	 vmk->numperm*4/1024.0,
	 vmk->totalmem*4/1024.0);
  mon_print(3,0, "Virtual memory: %6.1lfM free %6.1lfM used                %6.1lfM total",
	 vmk->freevmem*4/1024.0,
	 (vmk->totalvmem-vmk->freevmem)*4/1024.0, 
	 vmk->totalvmem*4/1024.0);
}

#define dkD(a)  ((dk1->a) - (dk2->a))

typedef struct dksort {
  struct dkstat *dk1,*dk2;
  ulong speed;
} dksort_t;

int cmp_dksort(void *a, void *b)
{
  if (((dksort_t *)a)->speed > ((dksort_t *)b)->speed) return (-1);
  else if (((dksort_t *)a)->speed < ((dksort_t *)b)->speed) return ( 1);
  return (0);
}

print_dkstat(double refresh_time,struct dkstat *dk1,struct dkstat *dk2)
{
    int i;
    int x,y;
    static struct dkstat *dks=NULL;
    static dksort_t *dksorted;
    static int ndisks_sorted=0;
    int ndisks, active;
    char active_str[16];
    
    ndisks = summary_dkstat(&dks, dk1,dk2, &active);
    if (ndisks_sorted = 0) {
      dksorted = (dksort_t *) malloc(sizeof(dksort_t)*ndisks);
      ndisks_sorted = ndisks;
    }
    if (ndisks > ndisks_sorted) { 
      dksorted = (dksort_t *) realloc(dksorted, sizeof(dksort_t)*ndisks);
      ndisks_sorted = ndisks;
    }
      
    x=0;y=12;
    mon_print(y  ,x, "DiskIO     Total Summary");
    mon_print(y+1,x, "read      %7.1f kByte/s",
	dks->dk_rblks*dks->dk_bsize/1024.0/refresh_time);
    mon_print(y+2,x, "write     %7.1f kByte/s",
	dks->dk_wblks*dks->dk_bsize/1024.0/refresh_time);
    mon_print(y+3,x, "transfers %7.1f tps",dks->dk_xfers/refresh_time);
    sprintf(active_str, "%d/%d", active,ndisks);
    mon_print(y+4,x, "active    %7s disks",active_str);

    i=0;
    while (dk1) { /* initialize the sorting list */
      dksorted[i].dk1 = dk1;
      dksorted[i].dk2 = dk2;
      dksorted[i].speed = dkD(dk_rblks)+dkD(dk_wblks);
      if (! dk1->dknextp) break;
      dk1 = dk1->dknextp;
      dk2 = dk2->dknextp;
      i++;
    }
    qsort((void *)dksorted, (size_t)ndisks, (size_t)sizeof(dksort_t), cmp_dksort);
    x=0;y+=6;
    mon_print(y+0,x, "TOPdisk read write     busy");
    i = 0;
    for (i=0; y+i+1<lines && i<min(10,ndisks); i++) {
      dk1 = dksorted[i].dk1;
      dk2 = dksorted[i].dk2;
      mon_print(y+1+i,x, "%-7s %4.0f %4.0f kB/s %3.0f%%",
	     dk1->diskname,
	     dkD(dk_rblks)*dk1->dk_bsize/1024.0/refresh_time,
	     dkD(dk_wblks)*dk1->dk_bsize/1024.0/refresh_time,
	     dkD(dk_time)/refresh_time);
    }
    mon_print(y+((i+2)<11?11:i+2),x,"");
}

print_dkstat_full(double refresh_time,struct dkstat *dk1,struct dkstat *dk2)
{
    int i;
    int x,y;
    double rsize,wsize,xfers;
    static struct dkstat *dks=NULL;
    int ndisks, active;
    
    ndisks = summary_dkstat(&dks, dk1,dk2, &active);
    x=0;y=5;
    mon_print(y+0,x, "DiskIO    read       write rsize  wsize xfers busy");
    i = 0;
    while (1) {
	move(y+1+i,x);
	if (y+1+i>=lines) {
	  printw("...more disks with bigger window...");
	  break;
	}
	xfers = dkD(dk_xfers);
	if (xfers == 0) {
	  rsize = wsize = 0;
	} else {
	  rsize = dkD(dk_rblks)*dk1->dk_bsize/1024.0/xfers;
	  wsize = dkD(dk_wblks)*dk1->dk_bsize/1024.0/xfers;
	}
	mon_print(y+1+i,x, "%-7s %6.1f %6.1f kB/s %4.1f %4.1f kB %5.1f %3.0f%%",
	       dk1->diskname,
	       dkD(dk_rblks)*dk1->dk_bsize/1024.0/refresh_time,
	       dkD(dk_wblks)*dk1->dk_bsize/1024.0/refresh_time,
	       rsize, wsize, xfers/refresh_time,
	       dkD(dk_time)/refresh_time);
	if (!dk1->dknextp) break;
	dk1 = dk1->dknextp;
	dk2 = dk2->dknextp;
	i++;
    }
    x=53;
    mon_print(y, x, "Summary    Total");
    mon_print(y+1,x, "read     %7.1f kbyte/s", 
	dks->dk_rblks*dks->dk_bsize/1024.0/refresh_time);
    mon_print(y+2,x, "write    %7.1f kbyte/s", 
	dks->dk_wblks*dks->dk_bsize/1024.0/refresh_time);
    mon_print(y+3,x, "xfers    %7.1f tps",dks->dk_xfers/refresh_time);
    mon_print(y+4,x, "busy     %7.1f %%",dks->dk_time/refresh_time);
    mon_print(y+6,x, "active   %7d disks", active);
    mon_print(y+7,x, "total    %7d disks",ndisks);
    move(y+i+2,0);
}

#define ifD(a) ((if1->a) - (if2->a))

print_ifnet(double refresh_time,struct ifnet *if1,struct ifnet *if2)
{
    int i;
    int x,y;
    char ifname[IFNAMSIZ];

    i=0;
    x=54;y=16;
    mon_print(y+0,x, "Netw   read  write");
    while (if1) {
      sprintf(ifname, "%s%d",if1->if_name, if1->if_unit);
      mon_print(y+1+i,x, "%-4s %6.1f %6.1f kB/s",
	     ifname,
#ifdef AIXv3r1
	     ifD(ifInOctets)/1024.0/refresh_time,
	     ifD(ifOutOctets)/1024.0/refresh_time);
#else
             ifD(if_ibytes)/1024.0/refresh_time,
             ifD(if_obytes)/1024.0/refresh_time);
#endif
      if (!if1->if_next) break;
      if1 = if1->if_next;
      if2 = if2->if_next;
      i++;
    }
}


print_ifnet_full(double refresh_time,struct ifnet *if1, struct ifnet *if2)
{
    int i;
    int x,y;
    char ifname[IFNAMSIZ];

    i=0;
    x=0;y=6;
    mon_print(y+0,x, "Netw   read   write        rcount wcount rsize wsize\n");
    while (if1) {
	int isize,osize;
	if (ifD(if_ipackets)) isize = ifD(if_ibytes)/ifD(if_ipackets);
	else isize = 0;
	if (ifD(if_opackets)) osize = ifD(if_obytes)/ifD(if_opackets);
	else osize = 0;
	sprintf(ifname, "%s%d",if1->if_name, if1->if_unit);
	mon_print(y+1+i,x, "%-4s %6.1f %6.1f kbyte/s %6d %6d %5d %5d",
    	       ifname,
#ifdef AIXv3r1
	       ifD(ifInOctets)/1024.0/refresh_time,
	       ifD(ifOutOctets)/1024.0/refresh_time,
#else
	       ifD(if_ibytes)/1024.0/refresh_time,
	       ifD(if_obytes)/1024.0/refresh_time,
#endif
	       ifD(if_ipackets), ifD(if_opackets),isize, osize);

	       
	if (!if1->if_next) break;
	if1 = if1->if_next;
	if2 = if2->if_next;
	i++;
    }
/*    printw("\n");*/
}

#define NFSclD(a) (n1->cl.a - n2->cl.a)
#define NFSsvD(a) (n1->sv.a - n2->sv.a)

print_nfsstat(double refresh_time, nfsstat_t *n1, nfsstat_t *n2)
{
  int x,y,client_other, server_other;
  double r;
  if (no_nfs) return;
  r = refresh_time;
  x=30; y=13;
  client_other = NFSclD(null)+NFSclD(setattr)+NFSclD(root)+NFSclD(readlink)+NFSclD(wrcache)
    + NFSclD(create)+NFSclD(remove)+NFSclD(rename)+NFSclD(link)+NFSclD(symlink)
      + NFSclD(mkdir)+NFSclD(rmdir)+NFSclD(readdir)+NFSclD(fsstat);

  server_other = NFSsvD(null)+NFSsvD(setattr)+NFSsvD(root)+NFSsvD(readlink)+NFSsvD(wrcache)
    + NFSsvD(create)+NFSsvD(remove)+NFSsvD(rename)+NFSsvD(link)+NFSsvD(symlink)
      + NFSsvD(mkdir)+NFSsvD(rmdir)+NFSsvD(readdir)+NFSsvD(fsstat);

  mon_print(y+0,x, "Client Server NFS/s");
  mon_print(y+1,x, "%6.1f %6.1f calls", NFSclD(calls)/r, NFSsvD(calls)/r);
  mon_print(y+2,x, "%6.1f %6.1f retry", (n1->rc.retrans-n2->rc.retrans)/r, 0);
  mon_print(y+3,x, "%6.1f %6.1f getattr", NFSclD(getattr)/r, NFSsvD(getattr)/r);
  mon_print(y+4,x, "%6.1f %6.1f lookup",  NFSclD(lookup)/r, NFSsvD(lookup)/r);
  mon_print(y+5,x, "%6.1f %6.1f read",  NFSclD(read)/r, NFSsvD(read)/r);
  mon_print(y+6,x, "%6.1f %6.1f write", NFSclD(write)/r, NFSsvD(write)/r);
  mon_print(y+7,x, "%6.1f %6.1f other", client_other/r, server_other/r);
  mon_print(y+8,0, "");
}

/* generate 'len' characters 'ch' in the end of string 'str' */
strchgen(str,ch,len)
char *str;
char ch;
int len;
{
  while (*str != 0) str++;
  while (len) { len--; *str++ = ch; }
  *str = 0;
}

void sighandler(int signum)
{
  switch (signum) {
  case SIGTSTP:
    if (!plain_ascii) {
      nocbreak();
      endwin();
    }
    kill(getpid(),SIGSTOP);
    /* returned here from sigstop */
    if (!plain_ascii) {
      cbreak();
      initscr();
      clear();
    }
    break;
  case SIGINT:
    if (!plain_ascii) {
      nocbreak();
      endwin();
    }
    exit(0);
    break;
  default: fprintf(stderr,"unknown signal %d\n",signum); fflush(stderr);
    if (!plain_ascii) {
      nocbreak();
    }
    endwin();
    exit(0);
  }
  signal(signum,sighandler);
}

char *statstr[] = {"?","sleep","?","run","T","zombie","sleep"};

print_topcpu(double refresh_time, topcpu_t *top, int ntop)
{
  int cputime;
  int i;
  int x,y;
  struct passwd *passwd;
  char username[16];
  static int maxtop=0,old_maxtop=0;

  if (plain_ascii) y=x=0;
  else getyx(stdscr, y, x);
  y+=2;
  if (!topflag_usersystemtime) {
    mon_print(y,0, "   PID USER     PRI NICE   SIZE     RES STAT      TIME   CPU%% COMMAND");
    y++;
    for (i=0; i<ntop && y+i+2<lines; i++) {
      if (show_top_running && !top[i].cputime_prs) break;

      cputime = top[i].cpu_utime + top[i].cpu_stime;
      if (topflag_showusername) {
	passwd = getpwuid(top[i].uid);
	strcpy(username, passwd->pw_name);
      } else {
	sprintf(username,"%d",top[i].uid);
      }
      mon_print(y+i,0, "%6d %-8s %3d %3d %6dK %6dK %-6s%5d:%02d %5.1f%% %s\n",
	     top[i].pid,	   username,
	     top[i].pri,	   top[i].nice-20,
	     top[i].memsize_1k,    top[i].ressize_1k,
	     statstr[top[i].stat],
	     cputime/60,cputime%60,
	     top[i].cputime_prs/10.0,
	     top[i].progname);
    }
  } else {
    mon_print(y,0,"   PID USER     PRI NICE   SIZE     RES PFLTS STAT USERTIME  SYSTIME CPU%% COMMAND");y++;
    for (i=0; i<ntop && y+i+2<lines; i++) {
      if (show_top_running && !top[i].cputime_prs) break;

      cputime = top[i].cpu_utime + top[i].cpu_stime;
      if (topflag_showusername) {
	passwd = getpwuid(top[i].uid);
	strcpy(username, passwd->pw_name);
      } else {
	sprintf(username,"%d",top[i].uid);
      }
      cputime = top[i].cpu_utime + top[i].cpu_stime;
      mon_print(y+i,0, "%6d %-8s %3d %3d %6dK %6dK %4.1f %-6s%5d:%02d %5d:%02d %5.1f%% %s\n",
	     top[i].pid,	   username,
	     top[i].pri,	   top[i].nice-20,
	     top[i].memsize_1k,  top[i].ressize_1k,
	     top[i].deltapageflt/refresh_time,
	     statstr[top[i].stat],
	     top[i].cpu_utime/60,top[i].cpu_utime%60,
	     top[i].cpu_stime/60,top[i].cpu_stime%60,
	     top[i].cputime_prs/10.0,
	     top[i].progname);
    }
  }
  mon_print(y+i,0,"\n");
  if (!plain_ascii) { /* clear the bottom of screen */
    old_maxtop=maxtop;
    maxtop=i;
    for (i=maxtop; i<old_maxtop; i++) {
      printw("\n");
    }
  }
}

int mon_print(int y, int x, const char *fmt, ...)
{
  char buf[4096];
  int ret;
  va_list ap;
  va_start(ap, fmt);
  ret = vsprintf(buf, fmt, ap);
  va_end(ap);
  if (plain_ascii) {
    fputs(buf,stdout);
    if (buf[strlen(buf)-1]!='\n') fprintf(stdout,"\n");
    fflush(stdout);
  } else {
    move (y,x);
    addstr(buf);
  }
  return(ret);
}
