#############################################################################
#                                                                           #
# AWK SCRIPT           M.J.E. Mol    (C) 1994, 1998                         #
#                      marcel@mesa.nl                                       #
#                                                                           #
# Date:     16apr1998                                                       #
# File:     cpu_week.awk                                                    #
# Version:  1.2                                                             #
#                                                                           #
# Description                                                               #
#    Take output from cpu_extract.awk and generate a gnuplot script that    #
#    creates postcript graphs of cpubusy per interval, averaged over one    #
#    week. Sample records are ignored.                                      #
#                                                                           #
# Version Date      Name             Comments                               #
# ------- --------- --------------   -------------------------------------- #
#  1.0    07jul94   M.J.E. Mol       Initial version                        #
#  1.1    08sep94   M.J.E. Mol       Reset minday/maxday after each plot    #
#                                    Skip weekends                          #
#  1.2    16apr1998 M.J.E. Mol       Rename script. Allow comments in input #
#                                                                           #
#############################################################################
#############################################################################
#
# Expected input:
#
#     <hostname> <weeknumber_of_year> <daynumber_of_month><monthname><year>
#         <daynumber_of_week> <interval_start_time> <interval_length>
#         <cpu_system> <cpu_iowait> <cpu_user> <cpu_idle> 
#
# The <cpu_idle> field is not really used.
#
# Outputs:
#
#    A file specified by the 'gpf' variable containing a gnuplot program.
#    Datafiles per host and per week and used by the gnuplot program 
#    containing records of the following layout: 
#
#	<hostname> <hour>.<min> <cpu_sys> <cpu_sys+user> <cpu_sys+user+wait>
#
#    where <min> indicates a fraction of the hour, e.g. 1.5 means 1:30.
#    <cpu....> are averages of the intervals over all workdays of the week.
#
#############################################################################

BEGIN {
        firsttime = 1
        minday = 100
        maxday = 0
    
    printf "set terminal postscript\n" > gpf
    printf "set time\n" >> gpf
    printf "set grid\n" >> gpf
    printf "set xlabel \"Time\"\n" >> gpf
    printf "set ylabel \"Busy %%\"\n" >> gpf
    printf "set tics out\n" >> gpf
    printf "set xrange[0:24]\n" >> gpf
    printf "set yrange[0:100]\n" >> gpf
    printf "set xtics 0,2,24\n" >> gpf
    printf "set ytics 0,10,100\n" >> gpf
    printf "set data style lines\n" >> gpf

}

#
# Skip comments
#
/^#/ { next }

{
        if (host != $1 || week != $2) {
            if (firsttime)
                firsttime = 0
            else
	        dump_plot()
            datafile = dataf "." $1 "." $2
        }

	host = $1
        week = $2
        date = $3
        daynum = $4
        time = $5
        interval = $6

        # skip saterday and sunday 
        if (daynum == 6 || daynum == 0)
	    next

        if (daynum > maxday) {
            maxday = daynum
            maxdate = date
        }
        if (daynum < minday) {
            minday = daynum
            mindate = date
        }

	cpusys[time, host]  += $7
	cpusys_cnt[time, host]  += 1
	cpuwait[time, host] += $8
	cpuwait_cnt[time, host] += 1
	cpuuser[time, host] += $9
	cpuuser_cnt[time, host] += 1
	# cpuidle[time, host] += $10
	# cpuidle_cnt[time, host] += 1

}

END {
    dump_plot()
}

function dump_plot () {
    system("rm -f " datafile)   
    sortcomm = "sort +1n >>" datafile
    printf "set title \"CPU Busy Figures for %s in week %s - %s-%s\"\n",
           host, week, mindate, maxdate >> gpf

    printf "plot '%s' using 2:5 title \"CPU wait\",\\\n", datafile >> gpf
    printf "     '%s' using 2:4 title \"CPU user\",\\\n", datafile >> gpf
    printf "     '%s' using 2:3 title \"CPU sys\"\n", datafile >> gpf

    printf "#pause -1 \"bye\"\n" >> gpf

    for (combined in cpusys) {
        split(combined, sep, SUBSEP)
        time = sep[1]
	host = sep[2]
        hour = substr(time,1,2)
	min  = substr(time,4,2) * 100/60
        sys  = cpusys[combined]/cpusys_cnt[combined]
        wait = cpuwait[combined]/cpuwait_cnt[combined] 
        user = cpuuser[combined]/cpuuser_cnt[combined] 
	printf "%s %d.%02d %f %f %f\n",
               host, hour, min, 
               sys, sys+user, sys+user+wait | sortcomm
        delete cpusys[combined]
        delete cpusys_cnt[combined]
        delete cpuwait[combined]
        delete cpuwait_cnt[combined]
        delete cpuuser[combined]
        delete cpuuser_cnt[combined] 
        # delete cpuidle[combined]
        # delete cpuidle_cnt[combined]
    }
    close (sortcomm)
    minday = 100
    maxday = 0
}
