#############################################################################
#                                                                           #
# AWK SCRIPT           M.J.E. Mol    (C) 1994, 1998                         #
#                      marcel@mesa.nl                                       #
#                                                                           #
# Date:     16apr1998                                                       #
# File:     disk_week.awk                                                   #
# Version:  1.2                                                             #
#                                                                           #
# Description                                                               #
#    Take output from disk_extract.awk and generate a gnuplot script that   #
#    creates postcript graphs of disk usage per interval, averaged over one #
#    week.                                                                  #
#    Four plots for each disk per system:                                   #
#        - Disk I/O rate  in I/Os per second                                #
#        - Disk read/write rate in kilobytes per second                     #
#        - Disk busy percentage                                             #
#        - Disk average block size per read/write in kilobyte per I/O       #
#    Expected input record format (space seperated items):                  #
#       host name                                                           #
#       week number                                                         #
#       date e.g. 07jul1994                                                 #
#       day of week (numeric)                                               #
#       start time of interval <hours>:<minutes>                            #
#       interval length in seconds                                          #
#       disk name                                                           #
#       disk readrate in kilobytes per second                               #
#       disk writerate in kilobytes per second                              #
#       disk iorate in I/O's per second                                     #
#       disk busy percentage                                                #
#       disk readsize in kilobyte per I/O                                   #
#       disk writesize in kilobyte per I/O                                  #
#    Graphs are auto scaling.                                               #
#    Sample records are ignored.                                            #
#                                                                           #
# Version Date      Name            Comments                                #
# ------- --------- --------------  --------------------------------------  #
#  1.0    07jul94   M.J.E. Mol      Initial version                         #
#  1.1    08sep94   M.J.E. Mol      Reset minday/maxday after each plot     #
#                                   Skip weekends                           #
#  1.2    16apr1998 M.J.E. Mol      Script has meaningful name now.         #
#                                   Allow comments in input file.           #
#                                                                           #
#############################################################################
#############################################################################
#
# Expected input:
#
#     <hostname> <weeknumber_of_year> <daynumber_of_month><monthname><year>
#         <daynumber_of_week> <interval_start_time> <interval_length>
#         <device_name> <readrate> <writerate> <iorate> <diskbusy>
#         <readsize> <writesize>
#
# Where:
#            readrate     kB read / second
#            writerate    kB written / second
#            readsize     kB read / io
#            writesize    kB written / io
#            iorate       io's / second
#            diskbusy     busy%%
#
# Outputs:
#
#    A file specified by the 'gpf' variable containing a gnuplot program.
#    Datafiles per host per device and per week and used by the gnuplot
#    program containing records of the following layout: 
#
#	<hostname> <hour>.<min> 
#           <readrate> <writerate> <iorate> <diskbusy> <readsize> <writesize> 
#
#    where <min> indicates a fraction of the hour, e.g. 1.5 means 1:30.
#    the others are averages of the intervals over all workdays of the week.
#
#############################################################################


BEGIN {
        RWRES     = 10      # resolution of read/write rate graphs
        SIZERES   =  5      # resolution of read/write size graphs
        IORATERES =  5      # resolution of io rate graphs

        firsttime = 1
        minday = 100
        maxday = 0

    printf "set terminal postscript\n" > gpf
    printf "set time\n" >> gpf
    printf "set grid\n" >> gpf
    printf "set xlabel \"Time\"\n" >> gpf
    printf "set tics out\n" >> gpf
    printf "set xtics 0,2,24\n" >> gpf
    printf "set xrange[0:24]\n" >> gpf
#    printf "set yrange[0:]\n" >> gpf
    printf "set data style lines\n" >> gpf

}

/^#/ { next }

{
        if (host != $1 || week != $2 || name != $7) {
            if (firsttime)
                firsttime = 0
            else
	        dump_plot()
            datafile = dataf $1 $2 $7
        }

	host = $1
        week = $2
        date = $3
        daynum = $4
        time = $5
        interval = $6
        name = $7

        # skip saterday and sunday 
        if (daynum == 6 || daynum == 0)
	    next

        if (daynum > maxday) {
            maxday = daynum
            maxdate = date
        }
        if (daynum < minday) {
            minday = daynum
            mindate = date
        }

	readrate[time, host, name]  += $8
	readrate_cnt[time, host, name]  += 1
	writerate[time, host, name]  += $9
	writerate_cnt[time, host, name]  += 1
	iorate[time, host, name]  += $10
	iorate_cnt[time, host, name]  += 1
	busy[time, host, name]  += $11
	busy_cnt[time, host, name]  += 1
	readsize[time, host, name]  += $12
	readsize_cnt[time, host, name]  += 1
	writesize[time, host, name]  += $13
	writesize_cnt[time, host, name]  += 1

}

END {
    dump_plot()
}

function dump_plot () {
    system("rm -f " datafile)   
    sortcomm = "sort +1n >>" datafile

    maxrwrate = 0
    maxsze = 0 
    maxio = 0
    for (combined in iorate) {
        split(combined, sep, SUBSEP)
        time = sep[1]
	host = sep[2]
        name = sep[3]
        hour = substr(time,1,2)
	min  = substr(time,4,2) * 100/60
     
        read     = readrate[combined]/readrate_cnt[combined]
        write    = writerate[combined]/writerate_cnt[combined]
        readsze  = readsize[combined]/readsize_cnt[combined]
        writesze = writesize[combined]/writesize_cnt[combined]
        diskio   = iorate[combined]/iorate_cnt[combined]
        diskbusy = busy[combined]/busy_cnt[combined] 
        if (read > maxrwrate)
            maxrwrate = read
        if (write > maxrwrate)
            maxrwrate = write
        if (readsze > maxsze)
            maxsze = readsze
        if (writesze > maxsze)
            maxsze = writesze
        if (diskio > maxio)
            maxio = diskio
	printf "%s %d.%02d %f %f %f %f %f %f\n",
               host, hour, min, 
               read, write, diskio, diskbusy, readsze, writesze | sortcomm
	delete readrate[combined]
	delete readrate_cnt[combined]
	delete writerate[combined]
	delete writerate_cnt[combined]
	delete readsize[combined]
	delete readsize_cnt[combined]
	delete writesize[combined]
	delete writesize_cnt[combined]
	delete iorate[combined]
	delete iorate_cnt[combined]
	delete busy[combined]
	delete busy_cnt[combined]
        minday = 100
        maxday = 0
    }
    close (sortcomm)

    maxrwrate = int((maxrwrate+RWRES)/RWRES)*RWRES
    maxsze = int((maxsze+SIZERES)/SIZERES)*SIZERES
    maxio = int((maxio+IORATERES)/IORATERES)*IORATERES
    printf "set title \"Disk Busy Figures for %s on %s in week %s - %s-%s\" ,1\n",
           name, host, week, mindate, maxdate >> gpf
    printf "set ylabel \"Busy %%\"\n" >> gpf
    printf "set yrange[0:100]\n" >> gpf
    printf "plot '%s' using 2:6 title \"Busy %%\"\n", datafile >> gpf

    printf "set title \"Disk Data Transfer for %s on %s in week %s - %s-%s\" ,1\n",
           name, host, week, mindate, maxdate >> gpf
    printf "set ylabel \"Data Transfer Rate (kB/sec)\"\n" >> gpf
    printf "set yrange[0:%d]\n", maxrwrate >> gpf
    printf "plot '%s' using 2:3 title \"kB Read\",\\\n", datafile >> gpf
    printf "     '%s' using 2:4 title \"kB Write\"\n", datafile >> gpf

    printf "set title \"Disk Transfer Size for %s on %s in week %s - %s-%s\" ,1\n",
           name, host, week, mindate, maxdate >> gpf
    printf "set ylabel \"Data Transfer Size (kB)\"\n" >> gpf
    printf "set yrange[0:%d]\n", maxsze >> gpf
    printf "plot '%s' using 2:7 title \"Read size\",\\\n", datafile >> gpf
    printf "     '%s' using 2:8 title \"Write size\"\n", datafile >> gpf

    printf "set title \"Disk IO Rate for %s on %s in week %s - %s-%s\" ,1\n",
           name, host, week, mindate, maxdate >> gpf
    printf "set ylabel \"IO / sec\"\n" >> gpf
    printf "set yrange[0:%d]\n", maxio >> gpf
    printf "plot '%s' using 2:5 title \"IO Rate\"\n", datafile >> gpf


    printf "#pause 30 \"bye\"\n" >> gpf

    minday = 100
    maxday = 0
}
