#############################################################################
#                                                                           #
# AWK SCRIPT           M.J.E. Mol    (C) 1994, 1998                         #
#                      marcel@mesa.nl                                       #
#                                                                           #
# Date:     16apr1998                                                       #
# File:     users_week.awk                                                  #
# Version:  1.2                                                             #
#                                                                           #
# Description                                                               #
#    Take output from users_extract.awk and generate a gnuplot script that  #
#    creates postcript graphs of logged on users per interval, averaged     #
#    over one week. Sample records are ignored.                             #
#                                                                           #
# Version Date      Name            Comments                                #
# ------- --------- --------------  --------------------------------------  #
#  1.0    07jul94   M.J.E. Mol      Initial version                         #
#  1.1    08sep94   M.J.E. Mol      Reset minday/maxday after each plot     #
#                                   Skip weekends                           #
#  1.2    16apr1998 M.J.E. Mol      Renamed script. Allow comment in input  #
#                                                                           #
#############################################################################
#############################################################################
#
# Expected input:
#
#     <hostname> <weeknumber_of_year> <daynumber_of_month><monthname><year>
#         <daynumber_of_week> <interval_start_time> <interval_length>
#         <number_of_users> <max_number_of_users>
#
# Outputs:
#
#    A file specified by the 'gpf' variable containing a gnuplot program.
#    Datafiles per host and per week and used by the gnuplot program 
#    containing records of the following layout: 
#
#	<hostname> <hour>.<min> <number_of_users>,<max_number_of_users>
#
#    where <min> indicates a fraction of the hour, e.g. 1.5 means 1:30.
#    <...users> are averages of the intervals over all workdays of the week.
#
#############################################################################


BEGIN {
        firsttime = 1
        minday = 100
        maxday = 0
    
    printf "set terminal postscript\n" > gpf
    printf "set time\n" >> gpf
    printf "set grid\n" >> gpf
    printf "set xlabel \"Time\"\n" >> gpf
    printf "set ylabel \"Number of Users\"\n" >> gpf
    printf "set tics out\n" >> gpf
    printf "set xrange[0:24]\n" >> gpf
    printf "set xtics 0,2,24\n" >> gpf
    printf "#set ytics 0,10,100\n" >> gpf
    printf "set data style lines\n" >> gpf

}

#
# Skip comment
#
/^#/ { next }

{
        if (host != $1 || week != $2) {
            if (firsttime)
                firsttime = 0
            else
	        dump_plot()
            datafile = dataf "." $1 "." $2
        }

	host = $1
        week = $2
        date = $3
        daynum = $4
        time = $5
        interval = $6

        # skip saterday and sunday 
        if (daynum == 6 || daynum == 0)
	    next

        if (daynum > maxday) {
            maxday = daynum
            maxdate = date
        }
        if (daynum < minday) {
            minday = daynum
            mindate = date
        }

	users[time, host] += $7
        if ($7 > users_max[time, host])
        	users_max[time, host] = $8
	users_cnt[time, host]  += 1
}

END {
    dump_plot()
}

function dump_plot () {
    system("rm -f " datafile)   
    sortcomm = "sort +1n >>" datafile

    maxy = 0
    for (combined in users) {
        split(combined, sep, SUBSEP)
        time = sep[1]
	host = sep[2]
        hour = substr(time,1,2)
	min  = substr(time,4,2) * 100/60
        usersave  = users[combined]/users_cnt[combined]
	printf "%s %d.%02d %f %f\n",
               host, hour, min, 
               usersave, users_max[combined] | sortcomm
        if (usersave > maxy)
            maxy = usersave
        if (users_max[combined] > maxy)
            maxy = users_max[combined]
        delete users[combined]
        delete users_cnt[combined]
        delete users_max[combined]
    }
    close (sortcomm)

    maxy = int((maxy+50)/50)*50
    printf "set yrange[0:%d]\n", maxy >> gpf
    printf "set title \"Logged on users for %s in week %s - %s-%s\"\n",
           host, week, mindate, maxdate >> gpf

    printf "plot '%s' using 2:4 title \"Max Users\",\\\n", datafile >> gpf
    printf "     '%s' using 2:3 title \"Users\"\n", datafile >> gpf

    printf "#pause -1 \"bye\"\n" >> gpf

    minday = 100
    maxday = 0
}
