/* -- AIX/6000 System monitor 
**
**     vmker.c
**
** Copyright (c) 1998 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <unistd.h>
#include <stdio.h>
#include <nlist.h>
#include <fcntl.h>
#include <sys/types.h>

/* vmker struct is kernelstruct (undocumented) */
/* vmker seems to hold some kernels virtual memory variables */
struct vmker {
    uint n0, n1, n2, n3, n4, n5, n6, n7;
    uint pageresv; /* reserved paging space blocks */
    uint totalmem; /* total number of pages memory */
    uint badmem; /* this is used in RS/6000 model 220 and C10 */
    uint freemem; /* number of pages on the free list */
    uint maxperm; /* max number of frames no working */
    uint numperm; /* seems to keep other than text and data segment usage */
                  /* the name is taken from /usr/lpp/bos/samples/vmtune.c */
    uint totalvmem, freevmem;
    uint n16;
    uint nonpinnable; /* number of reserved (non-pinable) memory pages */
    uint n18, n19;
    uint numclient; /* number of client frames */
    uint maxclient; /* max number of client frames */
    uint n22, n23;
    uint n24, n25;
    uint n26, n27;
    uint n28, n29;
    uint n30, n31;
    uint n32, n33;
    uint n34, n35;
    uint n36, n37;
    uint n38, n39;
    uint n40, n41;
    uint n42, n43;
    uint n44, n45;
    uint n46, n47;

};

void get_sys_vm_info(struct vmker *vmk);
int getkmemdata(void *buf, int bufsize, caddr_t address);

#define N_VALUE(index) ((caddr_t)kernelnames[index].n_value)
#define KNSTRUCTS_CNT 1
#define NLIST_VMKER 0

struct nlist kernelnames[] = {
    {"vmker",   0, 0, 0, 0, 0},
    {NULL,      0, 0, 0, 0, 0},
};

int kmemfd = -1;
int nlistdone = 0;

/***************************************************************************
 *                      DATA CAPTURE FUNCTIONS                             *
 ***************************************************************************/

int getkmemdata(void *buf, int bufsize, caddr_t address)
{
    int n;

    /*
     * Do stuff we only need to do once per invocation, like opening
     * the kmem file and fetching the parts of the symbol table.
     */
    if (kmemfd < 0) {
	if ((kmemfd = open("/dev/kmem", O_RDONLY)) < 0) {
	    perror("kmem");
	    exit(1);
	}
    }
    /*
     * Get the structure from the running kernel.
     */
    lseek(kmemfd, (off_t) address, SEEK_SET);
    n = read(kmemfd, buf, bufsize);

    return(n);

} /* getkmemdata */



void get_sys_vm_info(struct vmker *vmk)
{

    if (!nlistdone) {
        if (knlist(kernelnames, KNSTRUCTS_CNT, sizeof(struct nlist)) == -1)
	    perror("knlist, entry not found");
        nlistdone = 1;
    }

    /*
     * Get the kernel virtual memory vmker structure
     */
    getkmemdata(vmk, sizeof(struct vmker),   N_VALUE(NLIST_VMKER));

    return;

} /* get_sys_vm_info */



main()
{
    struct vmker vmk;
    unsigned char * v;
    int i;
    uint j;

    get_sys_vm_info(&vmk);

    printf("    Fieldname     Pagecount          MB Info\n");
    printf("n09 totalmem    %11d %11.1f Total memory \n", vmk.totalmem,    vmk.totalmem    * 4 / 1024.0);
    printf("n11 freemem     %11d %11.1f Available free memory\n", vmk.freemem,     vmk.freemem     * 4 / 1024.0);
    printf("n10 badmem      %11d %11.1f\n", vmk.badmem,      vmk.badmem      * 4 / 1024.0);
    printf("n13 numperm     %11d %11.1f Allocated to files\n", vmk.numperm,     vmk.numperm     * 4 / 1024.0);
    printf("n12 maxperm     %11d %11.1f\n", vmk.maxperm,     vmk.maxperm     * 4 / 1024.0);
    printf("n20 numclient   %11d %11.1f\n", vmk.numclient,   vmk.numclient   * 4 / 1024.0);
    printf("n21 maxclient   %11d %11.1f\n", vmk.maxclient,   vmk.maxclient   * 4 / 1024.0);
    printf("n17 nonpinnable %11d %11.1f\n", vmk.nonpinnable, vmk.nonpinnable * 4 / 1024.0);
    printf("n14 totalvmem   %11d %11.1f Total paging space\n", vmk.totalvmem,   vmk.totalvmem   * 4 / 1024.0);
    printf("n15 freevmem    %11d %11.1f Free paging space\n", vmk.freevmem,    vmk.freevmem    * 4 / 1024.0);
    printf("n08 pageresv    %11d %11.1f\n", vmk.pageresv,    vmk.pageresv    * 4 / 1024.0);

    v = (char *) &vmk;
    j = 0;
    for (i = 0; i <sizeof(vmk); i++) {
        if (i % 4) {
            putchar(' ');
        }
        else {
            if (i > 0) 
                printf("  %11d %11.1f\nn%02d: ", j, j * 4 / 1024.0, i / 4);
            else
                printf("n00: ");
            j = 0;
        }
        printf("%02X", v[i]);
        j = (j << 8) | v[i];
    }
    printf("  %11d %11.1f\n", j, j * 4 / 1024.0);
    
} /* main */
