/*
 * Copyright (C) 2004 Daniel Eischen <deischen@freebsd.org>
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice(s), this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified other than the possible
 *    addition of one or more copyright notices.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice(s), this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: stable/5/lib/libpthread/thread/thr_execve.c 136394 2004-10-11 20:04:59Z julian $
 */

#include <errno.h>
#include <pthread.h>
#include <unistd.h>
#include "thr_private.h"

__weak_reference(_execve, execve);

int 
_execve(const char *name, char *const *argv, char *const *envp)
{
	sigset_t omask;
	struct pthread *curthread;
	kse_critical_t crit;
	int saved_errno;
	int ret;

	/*
	 * When exec'ing, set the kernel signal mask to the thread's
	 * signal mask to satisfy POSIX requirements.  We have to enter
	 * a critical region so that the kernel thread doesn't get
	 * changed out from under us after setting the signal mask.
	 */
	curthread = _get_curthread();
	crit = _kse_critical_enter();
	__sys_sigprocmask(SIG_SETMASK, &curthread->sigmask, &omask);
	 ret = __sys_execve(name, argv, envp);

	/*
	 * If something went wrong, set the signal mask back but don't
	 * destroy errno.
	 */
	saved_errno = errno;
	__sys_sigprocmask(SIG_SETMASK, &omask, NULL);
	errno = saved_errno;
	_kse_critical_leave(crit);
	return (ret);
}
