/****************************************************************************** 
 *
 * File:        midifile.c
 * Version:     $Id: midifile.c,v 1.28 1995/08/06 18:56:48 burgaard Exp $
 *              $Version: 0.2$
 *
 * Purpose:     Standard MIDI File I/O.
 *
 * Project:     MIDI/Sequencer library.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994, 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Compile and see ../doc/libmidifile.tex for documentation on this file.
 *
 * The short of the long is that it can parse and load a Standard MIDI File
 * and put the resulting events onto some queues. They are defined in
 * midiqueue.h and should be straight forward to understand.
 *
 * The loading can be controlled in some ways. It can for example be told not
 * to store anything but just run through the file. Thus, it can be used to
 * check a MIDI file for errors or to compute the total playing time. This is
 * exactly what `smfplay' does before it actually plays anything.
 *
 * Save (in format 1) and miscellaneous convertion routines will be added with time...
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "miditypes.h"
#include "midiconst.h"
#include "midiqueue.h"
#include "midiprint.h"
#include "midifile.h"

/*** CONSTANTS ****************************************************************/

/* This array is indexed after the most significant 4 BITS of a CHANNEL MESSAGE
 *                                         0 1 2 3 4 5 6 7 8 9 A B C D E F */
static const char MIDI_CHMSG_ARGS[0x10] = {0,0,0,0,0,0,0,0,2,2,2,2,1,1,2,0};

static const byte MIDI_FILE_HEAD[] = "MThd";
static const byte MIDI_FILE_TRACK[] = "MTrk";

/*** GLOBAL VARIABLES *********************************************************/

static midi_file_data queues;
static midi_file_timing timing;
static midi_file midifile;

static midi_file_data *pqueues;
static midi_file_timing *ptiming;
static midi_file *pmidifile;

static char verbose  = 0;

/*** MEMORY HANDLING **********************************************************/

static char * my_malloc(long size)
{
  char *tmp = NULL;

  if (!size)
    {
      fprintf(stderr, "Warning: malloc called with 0\n");
      return NULL;
    };

  if ( !(tmp = malloc(size)) )
    {
      fprintf(stderr, "Error: Out of memory!\n");
      exit(-3);
    };

  return tmp;
}

inline midi_event * midi_event_alloc(void)
{
  return (midi_event *)my_malloc(sizeof(midi_event));
}

inline void midi_event_free(midi_event *event)
{
  if (!event) return;
  if (event->data) free(event->data);
  free(event);
}

/*** MIDI FILE HANDLING *******************************************************/

inline static int fileerror(void)
{
  return ferror(midifile.file);
}

inline static long filetell(void)
{
  return ftell(midifile.file);
}

inline static int fileseek(long offset, int origin)
{
  return fseek(midifile.file, offset, origin);
}

inline static int readbyte(void)
{
  return fgetc(midifile.file);
}

inline static word readword(void)
{
  return ((readbyte() & 0xff) << 8) | (readbyte() & 0xff);
}

inline static long readlong(void)
{
  return (((readbyte() & 0xff) << 24) | ((readbyte() & 0xff) << 16) |
	  ((readbyte() & 0xff) << 8) | (readbyte() & 0xff));
}

static long readvarlen(void)
{
  long value = 0;
  byte c = 0;

  if ( (value = readbyte()) & 0x80 && !fileerror() )
    {
      value &= 0x7f;
      do
	{
	  value = (value << 7) + ( (c = readbyte()) & 0x7f );
	}
      while ( c & 0x80 && !fileerror() );
    };
  return value;
}

inline static size_t fileread(void *ptr, size_t size)
{
  /* if (!ptr) return 0; */
  return fread(ptr, size, 1, midifile.file);
}

inline static int writebyte(byte data)
{
  return fputc(data, midifile.file);
}

inline static int writeword(word data)
{
  writebyte( (data >> 8) & 0xff );
  return writebyte( data & 0xff );
}

inline static int writelong(long data)
{
  writebyte( ( data >> 24 ) & 0xff );
  writebyte( ( data >> 16 ) & 0xff );
  writebyte( ( data >> 8 ) & 0xff );
  return writebyte( data & 0xff );
}

static int writevarlen(long value)
{
  int i = 0;
  long buffer = 0;

  buffer = value & 0x7f;
  
  while ( (value >>= 7) > 0 )
    {
      buffer <<= 8;
      buffer |= 0x80;
      buffer += (value & 0x7f);
    };

  for (;;)
    {
      i = writebyte(buffer);
      if (buffer & 0x80) buffer >>= 8;
      else break;
    };

  return i;
}

inline static size_t filewrite(const void *ptr, size_t size)
{
  return (!size || !ptr) ? 0 : fwrite(ptr, size, 1, midifile.file);
}

/* This is actually a *lot* faster than doing a `fseek' with small size values */
inline static void fileskip(long size)
{
  int i = 0;
  for (i = 0; i < size; i++) readbyte();
}

static byte voice_run = 0;
static byte voice_old = 0;

static byte sysex_run = 0;
static char got_eot   = 0;

static midi_queue tempo_map;
static long lasttime = 0;

inline static long gettime(void)
{
  return readvarlen();
}

/* speed things up a bit */
static byte buffer[1024];

/* <byte type> <var len size> <0..size-1 byte data> */
inline static int loadmeta(long time)
{
  byte * data = NULL;
  long ot = time;
  long size = 0;
  int tmp = 0;  

  if (timing.newdivision) time *= ( (double)timing.newdivision / (double)timing.division );
  if ( (tmp = readbyte()) == EOF )
    {
      fprintf(stderr, "Unexpected end of file\n");
      return -EINVAL;
    };
  buffer[0] = tmp&0xff;

  size = readvarlen();
  fileread(&buffer[1], size);
  switch (buffer[0]&0xff)
    {
    case MID_META_SET_TEMPO:
      if ( !( ((buffer[1]&0xff)<<16) | ((buffer[2]&0xff)<<8) | ((buffer[3]&0xff)) ) )
	{
	  fprintf(stderr, "\nInfinite BPM request (this is an error ;-)\n");
	  return -EINVAL;
	};

      data = (byte *)my_malloc(size + 1);
      memmove(data, &buffer[0], size + 1);
      midi_queue_put_event(&tempo_map, MIDI_TYP_META, ot, size + 1, data);
      break;

    case MID_META_SEQ_NAME:	/* Sequence Name */
      if (!midifile.title)
	{
	  midifile.title = my_malloc(size + 1);
	  memmove(midifile.title, &buffer[1], size);
	  midifile.title[size] = 0;
	};
      break;

    case MID_META_COPYRIGHT:	/* Copyright Notice */
      if (!midifile.copyright)
	{
	  midifile.copyright = my_malloc(size + 1);
	  memmove(midifile.copyright, &buffer[1], size);
	  midifile.copyright[size] = 0;
	};
      break;

    case MID_META_EOT:		/* End Of Track */
      got_eot = 1;
      break;
    case MID_META_TIME_SIG:	/* Time signature */
    case MID_META_KEY_SIG:	/* Key signature */
    case MID_META_SMPTE_OFFSET:
    case MID_META_SEQ_NUMBER:	/* Sequence number */
    case MID_META_TEXT:		/* Text */
    case MID_META_INST_NAME:	/* Instrument name */
    case MID_META_LYRIC:	/* Lyric */
    case MID_META_MARKER:	/* Marker */
    case MID_META_CUE_POINT:	/* Cue point */
    case MID_META_CH_PREFIX:	/* Channel Prefix */
    case MID_META_SEQ_SPECIFIC: /* ? */
    default:
      /* do nothing -- keep 'em around for future implementations */
      break;
    };

  if (queues.dometa)
    {
      data = (byte *)my_malloc(size + 1);
      memmove(data, &buffer[0], size + 1);
      midi_queue_put_event(&queues.meta, MIDI_TYP_META, time, size + 1, data);
    };

  if (verbose) midi_printrawevent(MIDI_TYP_META, time, size + 1, &buffer[0], verbose);
  return 0;
}

inline static int loadsysex(long time, byte type)
{
  byte * data = NULL;
  long  size = 0;

  if (timing.newdivision) time *= ( (double)timing.newdivision / (double)timing.division );
  size = readvarlen();

  buffer[0] = type;
  fileread(&buffer[1], size);

  if (queues.dosysex)
    {
      data = (byte *)my_malloc(size + 1);
      memmove(&data[0], &buffer[0], size + 1);
      midi_queue_put_event(&queues.sysex, MIDI_TYP_SYSEX, time, size + 1, data);
    };

  if ( (buffer[size]&0xff) != MID_SYS_EOX && !verbose )
    {
      fprintf(stderr, "Warning: Bad SysEx (missing 0xF7).\n");
      return fileerror();
    };

  if (verbose) midi_printrawevent(MIDI_TYP_SYSEX, time, size + 1, &buffer[0], verbose);

  return 0;
}

/* The following table is used to prevent sending multiple NOTE OFF's after multiple
 * NOTE ON's on the same channel *and* key:
 *
 *   ON                             OFF
 *   |------------NOTE 1-------------|
 *
 *                            ON                             OFF
 *                            |------------NOTE 2-------------|
 *
 * The situation depicted above (NOTE 1 and 2 are on the very same channel *and* key)
 * will be converted as depicted below: 
 *
 *   ON                     OFF
 *   |------------NOTE 1-----|
 *                            ON                             OFF
 *                            |------------NOTE 2-------------|
 *
 * i.e. the intersection between the two notes will be removed...
 */
typedef byte voice_channel[0xff];
static voice_channel voice_channels[0x0f];

inline static int loadvoice(long time, byte c)
{
  char dostore = 1;
  byte * data = NULL;
  long size = 0;
  int i = 0;
  byte j, k;

  if (timing.newdivision) time *= ( (double)timing.newdivision / (double)timing.division );
  if ( (c & 0xF0) > MID_EVENT_CHMSG_LAST )
    {
      fprintf(stderr, "\nUnrecognized MIDI event (%02hx).\n", c);
      return -EINVAL;
    };
  
  if ( !(voice_run = (c & 0xF0 ) < MID_EVENT_CHMSG_FIRST) )
    {
      size = MIDI_CHMSG_ARGS[(c >> 4) & 0x0f];
      voice_old = buffer[0] = c & 0xff;
      for (i = 1; i < size + 1; i++) buffer[i] = readbyte();
    }
  else if (voice_old)
    {
      size = MIDI_CHMSG_ARGS[(voice_old >> 4) & 0x0f];
      buffer[0] = voice_old;
      buffer[1] = c & 0xff;
      if (size == 2) buffer[2] = readbyte();
    }
  else
    {
      printf("\nIllegal running status.\n");
      return -EINVAL;
    };
  
  if (queues.dovoice)
    {
      dostore = 1;
      j = buffer[0]&0x0f;
      k = buffer[1]&0xff;

      /* check for multiple Note ON/OFF's on the same channel+key */
      switch (buffer[0]&0xf0) /* check event type */
	{
	case MID_EVENT_NOTE_ON:
	  /* NOTE ON level 0 is equivalent to NOTE OFF */
	  if (buffer[2])
	    {
	      if ((voice_channels[j][k]++) && (time > 0))
		{
		  /* Oops, two NOTE ON's on the same channel/key in a row...
		   * ...retrigger note and remove intervening NOTE OFF */
		  data = (byte *)my_malloc(3);
		  data[0] = MID_EVENT_NOTE_OFF|j;
		  data[1] = buffer[1]; /* note key number */
		  data[2] = 64; /* default value */
		  midi_queue_put_event(&queues.voice, MIDI_TYP_VOICE, time-1, 3, data);
		};
	      break;
	    };

	  /* convert to NOTE OFF */
	  buffer[0] = MID_EVENT_NOTE_OFF|j;
	  buffer[2] = 64; /* std. value */
	  /* FALL TROUGH */

	case MID_EVENT_NOTE_OFF:
	  if (--voice_channels[j][k]) dostore = 0; /* remove obsolete NOTE OFF */
	  break;
	};

      if (dostore)
	{
	  data = (byte *)my_malloc(size + 1);
	  memmove(data, &buffer[0], size + 1);
	  midi_queue_put_event(&queues.voice, MIDI_TYP_VOICE, time, size + 1, data);
	};
    };

  if (verbose) midi_printrawevent(MIDI_TYP_VOICE, time, size + 1, &buffer[0], verbose);
  return 0;
}

static int loadtrack(int track)
{
  long time = 0;
  long size = 0;
  long head = 0;
  int c = 0;

  queues.voice.current = queues.sysex.current = queues.meta.current = NULL;

  if (verbose) printf("Loading track    %d ", track);

  if ( fileread(&buffer[0], 4) != 1 )
    {
      fprintf(stderr, "\nUnexpected end of file.\n");
      return -EINVAL;
    };
  buffer[4] = 0;
  if ( strcmp(&buffer[0], &MIDI_FILE_TRACK[0]) )
    {
      fprintf(stderr, "\nUnrecognized MIDI data (MTrk header expected).\n");
      return -EINVAL;
    };

  size = readlong();
  head = filetell();

  if (verbose) printf("(%ld bytes).\n", size);

  got_eot = 0;

  while (!got_eot && !fileerror())
    {
      time += gettime();
      if ( (c = readbyte()) == EOF)
	{
	  fprintf(stderr, "Unexpected end of file\n");
	  return -EINVAL;
	};

      switch (c&0xff)
	{
	case MID_EVENT_META:
	  if (loadmeta(time)) return -EINVAL;
	  voice_old = 0;
	  sysex_run = 0;
	  if (got_eot) continue;
	  break;

	case MID_SYS_SYSEX:
	  if (loadsysex(time, MID_SYS_SYSEX)) return -EINVAL;
	  voice_old = 0;
	  sysex_run = 1;
	  break;

	case MID_EVENT_OTHER:
	  if (!sysex_run)
	    {
	      fprintf(stderr, "\nInvalid SysEx message (non-run status).\n");
	      return -EINVAL;
	    };
	  if (loadsysex(time, MID_EVENT_OTHER)) return -EINVAL;
	  voice_old = 0;
	  break;

	default:
	  if (loadvoice(time, c)) return -EINVAL;
	  sysex_run = 0;
	  break;
	};
    };

  if (time > lasttime) lasttime = time;

  if (filetell() - head > size) fileread(&buffer[0], filetell() - head);
  return fileerror();
}

static void update(char to_user)
{
  if (to_user)
    {
      memmove(pqueues, &queues, sizeof(midi_file_data));
      memmove(ptiming, &timing, sizeof(midi_file_timing));
      memmove(pmidifile, &midifile, sizeof(midi_file));
    }
  else
    {
      memmove(&queues, pqueues, sizeof(midi_file_data));
      memmove(&timing, ptiming, sizeof(midi_file_timing));
      memmove(&midifile, pmidifile, sizeof(midi_file));
    };
}

int midi_load(midi_file *f, midi_file_timing *t, midi_file_data *d, char verb)
{
  midi_event * event;
  int size = 0;
  int bad = 0;
  int i = 0;
  byte j, k;
  long mysectemp = 6000000;
  long l;
  float ft = 0.0;

  if ( !(pqueues = d) ) return -EINVAL;
  if ( !(ptiming = t) ) return -EINVAL;
  if ( !(pmidifile = f) ) return -EINVAL;
  if ( !(pmidifile->file) ) return -EINVAL;
  update(0);

  midi_queue_reset(&tempo_map);

  verbose   = verb;
  voice_run = voice_old = sysex_run = got_eot = 0;

  /* clear the voice channel tables */
  for (j = 0; j < 0x0f; j++)
    for (k = 0; k < 0xff; k++)
	voice_channels[j][k] = 0;

  buffer[4] = 0;
  if ( fileread(&buffer[0], 4) != 1 || strcmp(&buffer[0], &MIDI_FILE_HEAD[0]) )
    {
      if (midifile.name)
	fprintf(stderr, "`%s' is not a Standard MIDI File\n", midifile.name);
      else
	fprintf(stderr, "This is not a Standard MIDI File\n");
      return -EINVAL;
    };

  size = readlong();

  midifile.format = readword();
  midifile.tracks = readword();
  timing.division = readword();
  timing.mintempo = 100;
  timing.maxtempo = 100;
  lasttime = 0;
 
  if ( fileerror() )
    {
      update(1);
      return -EINVAL;
    };

  size -= 6;
  if (size > 0 && !fileerror()) fileskip(size); /* flush rest of header */

  if (midifile.format > 1)
    {
      fprintf(stderr, "MIDI File format 2 or higher not supported!\n");
      update(1);
      return -EINVAL;
    };

  if (verbose)
    {
      printf("\n");
      if (midifile.name) printf("File:            `%s'\n", midifile.name);
      printf("Format:          %d\n", midifile.format);
      printf("Tracks:          %d\n", midifile.tracks);
    };

  if (timing.division < 0)
    {
      if (verbose) printf("SMPTE Division: %hd frames/sec, %hd frame resolution\n", 
			  (char)((timing.division >> 8) & 0xff), (char)(timing.division & 0xff));

      printf("\nSMPTE time coding not (yet) supported!\n");
      update(1);
      return -EINVAL;
    };

  if (verbose) printf("Division:        %d\n", timing.division);

  for (i = 0; i < midifile.tracks && !fileerror() && !bad; i++) bad = loadtrack(i);

  /* calulate file playing time */
  l = 0;
  i = 100;
  ft = 0.0;
  mysectemp = 600000;
  while ( (event = midi_queue_get(&tempo_map)) )
    {
      ft += ((float)(event->time - l) * ((float)mysectemp/(float)timing.division)) / 1000000.0;
      mysectemp = ((event->data[1]&0xff)<<16)|((event->data[2]&0xff)<<8)|(event->data[3]&0xff);
      l = event->time;
      if (mysectemp > 0)
	{
	  i = (60.0 * 1000000.0) / (float)mysectemp;
	  if (timing.mintempo > i) timing.mintempo = i;
	  if (timing.maxtempo < i) timing.maxtempo = i;
	};
      midi_event_free(event);
    };
  midi_queue_flush(&tempo_map);
  ft += ((float)(lasttime - l) * ((float)mysectemp/(float)timing.division)) / 1000000.0;
  timing.time = ft;

  if (timing.newdivision) timing.division = timing.newdivision;
  if (bad)
    {
      update(1);
      return bad;
    }
  if ( fileerror() )
    {
      perror("File error ");
      update(1);
      return fileerror();
    };

  if (timing.mintempo == -1) timing.mintempo = timing.maxtempo = 100;

  update(1);
  return 0;
}

/******************************************************************************
 *
 * Write a single track with the events given in `events'. Finish it with a
 * META EOT message.
 *
 ******************************************************************************/
static int midi_write_track(midi_queue * events, long endtime)
{
  midi_event * event = NULL;

  long tracksize = 0;
  long sizepos = 0;
  long sizeposend = 0;

  long eventcount = 0;
  long oldtime = 0;
  byte oldvoice = 0;

  /* Track */
  filewrite(&MIDI_FILE_TRACK[0], 4);
 
  fflush(midifile.file);
  sizepos = filetell(); /* We'll need this position in order to update size later! */
  fflush(midifile.file);
  if (fileerror()) return fileerror();

  writelong(0);         /* write a temporary zero... */

  oldvoice = 0;
  oldtime = 0;

  event = midi_queue_get(events);
  while ( event && !fileerror())
    {
      switch (event->type&0xff)
	{
	case MIDI_TYP_VOICE:
	  eventcount++;
	  writevarlen(event->time - oldtime);
	  if (oldvoice == event->data[0]&0xff) filewrite(&event->data[1], event->size - 1);
	  else filewrite(&event->data[0], event->size);
	  oldvoice = event->data[0]&0xff;
	  break;

	case MIDI_TYP_SYSEX:
	  eventcount++;
	  writevarlen(event->time - oldtime);
	  writebyte(event->data[0]&0xff); /* write SYSEX ID byte */
	  writevarlen(event->size - 1);
	  filewrite(&event->data[1], event->size - 1);
	  oldvoice = 0;
	  break;

	case MIDI_TYP_META:	  
	  if (event->data[0] == MID_META_EOT) break; /* there can be only one EOT per track */
	  eventcount++;
	  writevarlen(event->time - oldtime);
	  writebyte(MID_EVENT_META); /* write META ID byte */
	  writebyte(event->data[0]&0xff); /* write META TYPE byte */
	  writevarlen(event->size - 1);
	  filewrite(&event->data[1], event->size - 1);
	  oldvoice = 0;
	  break;

	default:
	  fprintf(stderr, "\nIllegal event type 0x%02hx!\n", (byte)event->type);
	  return -EINVAL;
	  break;
	};
      
      oldtime = event->time;
      midi_event_free(event);
      event = midi_queue_get(events);
    };
  
  if (fileerror()) return fileerror();

  eventcount++;
  writevarlen( (endtime) ? endtime - oldtime : 0 );
  writebyte(MID_EVENT_META); /* write META ID byte */
  writebyte(MID_META_EOT); /* write META TYPE byte */
  writebyte(0); /* META End Of Track don't carry redundant information around :-) */

  /* update file size field */
  fflush(midifile.file);
  sizeposend = filetell();
  tracksize = sizeposend - sizepos - 4;
  fileseek(sizepos, SEEK_SET);
  fflush(midifile.file);

  if (verbose) printf("Wrote %ld events = %ld bytes to track\n", eventcount, tracksize);

  /* update file posistion */
  writelong(tracksize);
  fflush(midifile.file);
  fileseek(sizeposend, SEEK_SET);

  return fileerror();
};

/******************************************************************************
 *
 * MIDI format 0 puts all kind of events together in one single track.
 *
 ******************************************************************************/
static int midi_save_0(void)
{
  static midi_queue events;

  midi_event * event = NULL;
  byte * data = NULL;

  midi_queue_reset(&events);
  midifile.tracks = 1;

  /* VOICE events has priority over META and SYSEX events */
  event = (queues.dovoice) ? queues.voice.head : NULL;
  while (event)
    {
      data = (byte *)my_malloc(event->size);
      memmove(data, event->data, event->size);
      midi_queue_put_event(&events, MIDI_TYP_VOICE, event->time, event->size, event->data);
      event = event->next;
    };

  events.current = NULL;

  event = (queues.dometa) ? queues.meta.head : NULL;
  while (event)
    {
      data = (byte *)my_malloc(event->size);
      memmove(data, event->data, event->size);
      midi_queue_put_event(&events, MIDI_TYP_META, event->time, event->size, data);
      event = event->next;
    };

  events.current = NULL;

  event = (queues.dosysex) ? queues.sysex.head : NULL;
  while (event)
    {
      data = (byte *)my_malloc(event->size);
      memmove(data, event->data, event->size);
      midi_queue_put_event(&events, MIDI_TYP_SYSEX, event->time, event->size, data);
      event = event->next;
    };

  if ( !(events.count) ) fprintf(stderr, "Warning: Nothing to save.\n");

  /* Write header */
  if (verbose) printf("Writing file header\n"); 
  filewrite(&MIDI_FILE_HEAD[0], 4);
  writelong(6);
  writeword(0);
  writeword(midifile.tracks);
  writeword(timing.division);
  return midi_write_track(&events, 0);
}

/******************************************************************************
 *
 * Saving in MIDI format 1 according to MIDI File Spec. 1.0
 * 
 * Strategy:
 *
 * + Need to copy all queued events to temporary queue
 *   since we're going to sort 'em in another way:
 *
 *   1. queue:         META {TEMPO,TIMESIG,SMPTE_OFFSET}
 *   2. queue:         SYSEX
 *   3 to n + 3 queue: Queue for each n channels ( 0<=n<=15)
 *
 * + Write 1 to n+3 queues continuously, separated by header and META EOT.
 *   Note: Queue 1's META EOT must have the same time as the last event!
 *
 *   Track 1 is used to perform tempo map dumps on sequencers (mainly
 *   hardware based ones).
 *
 ******************************************************************************/

static int midi_save_1()
{
  static midi_queue tempotrack;
  static midi_queue sysextrack;
  static midi_queue voicetrack[16]; 

  midi_event *event = NULL;
  byte * data = NULL;
  long i = 0;
 
  lasttime = 0;
  midi_queue_reset(&tempotrack);
  midi_queue_reset(&sysextrack);
  for (i = 0; i < 16; i++) midi_queue_reset(&voicetrack[i]);
    
  /* VOICE events has priority over META and SYSEX events */
  event = (queues.dovoice) ? queues.voice.head : NULL;
  while (event)
    {
      data = (byte *)my_malloc(event->size);
      memmove(data, event->data, event->size);
      midi_queue_put_event(&voicetrack[data[0]&0x0f], MIDI_TYP_VOICE, event->time, event->size, event->data);
      event = event->next;
    };
  
  event = (queues.dometa) ? queues.meta.head : NULL;
  while (event)
    {
      data = (byte *)my_malloc(event->size);
      memmove(data, event->data, event->size);
      switch(data[0]&0xff)
	{
	case MID_META_EOT:
	  /* We'll add this later on */
	  break;
	case MID_META_SET_TEMPO:
	case MID_META_SMPTE_OFFSET:
	case MID_META_TIME_SIG:
	case MID_META_COPYRIGHT:
	case MID_META_SEQ_NAME:
	  midi_queue_put_event(&tempotrack, MIDI_TYP_META, event->time, event->size, data);
	  break;
	case MID_META_SEQ_NUMBER:
	case MID_META_CH_PREFIX:
	case MID_META_KEY_SIG:
	case MID_META_INST_NAME:
	case MID_META_TEXT:
	case MID_META_LYRIC:
	case MID_META_MARKER:
	case MID_META_CUE_POINT:
	case MID_META_SEQ_SPECIFIC:
	default:
	  /* the information on where these should be placed got lost either since the
	     source was a format 0 file or because the queues doesn't store original track
	     positions. Thus, we'll put them in the sysex track. If someone knows better
	     please don't hesitate telling me */
	  midi_queue_put_event(&sysextrack, MIDI_TYP_META, event->time, event->size, data);
	  break;
	};
      event = event->next;
    };
  
  event = (queues.dosysex) ? queues.sysex.head : NULL;
  while (event)
    {
      data = (byte *)my_malloc(event->size);
      memmove(data, event->data, event->size);
      midi_queue_put_event(&sysextrack, MIDI_TYP_SYSEX, event->time, event->size, data);
      event = event->next;
    };
  
  lasttime = 0;
  if (queues.voice.tail && queues.voice.tail->time > lasttime) lasttime = queues.voice.tail->time;
  if (queues.meta.tail && queues.meta.tail->time > lasttime) lasttime = queues.meta.tail->time;
  if (queues.sysex.tail && queues.sysex.tail->time > lasttime) lasttime = queues.sysex.tail->time;
  
  midifile.tracks = 2;
  for (i = 0; i < 16; i++) if (voicetrack[i].count) midifile.tracks++;
  
  /* Write header */
  if (verbose) printf("Writing file header\n"); 
  filewrite(&MIDI_FILE_HEAD[0], 4);
  writelong(6);
  writeword(1);
  writeword(midifile.tracks);
  writeword(timing.division);

  /* write the tracks */
  if (verbose) printf("Writing tempo map track\n"); 
  if (midi_write_track(&tempotrack, lasttime)) return fileerror();

  if (verbose) printf("Writing sysex track\n"); 
  if (midi_write_track(&sysextrack, 0)) return fileerror();

  for (i = 0; i < 16; i++)
    if (voicetrack[i].count) 
      {
	if (verbose) printf("Writing track %ld\n", i); 
	if (midi_write_track(&voicetrack[i], 0)) return fileerror();
      };
  return fileerror();
}

int midi_save(midi_file *f, midi_file_timing *t, midi_file_data *d, char verb)
{
  if ( !(pqueues = d) ) return -EINVAL;
  if ( !(ptiming = t) ) return -EINVAL;
  if ( !(pmidifile = f) ) return -EINVAL;
  if ( !(pmidifile->file) ) return -EINVAL;
  update(0);

  verbose = verb;

  if (midifile.format == 0) return midi_save_0();
  else if (midifile.format == 1) return midi_save_1();

  fprintf(stderr, "Error: MIDI Format %d is not supported!\n", midifile.format);
  return -EINVAL;
}

/*** END OF FILE **************************************************************/
