/*******************************************************************************
  
  Copyright(c) 2003 - 2004 Intel Corporation. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/
#ifndef _IPW2100_H
#define _IPW2100_H

#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/list.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/netdevice.h>
#include <asm/io.h>
#include <linux/socket.h>
#include <linux/if_arp.h>
#include <linux/wireless.h>
#include <linux/version.h>

#include "ieee80211.h"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#define work_struct tq_struct
#define schedule_work schedule_task
#define INIT_WORK INIT_TQUEUE
#define get_seconds(x) CURRENT_TIME
#endif

struct ipw2100_priv;
struct ipw2100_tx_packet;
struct ipw2100_rx_packet;

#ifdef CONFIG_IPW2100_DEBUG
#define IPW2100_DEBUG(level, fmt, args...) \
do { if (ipw2100_debug_level & level) \
  printk(KERN_DEBUG "ipw2100: %c %s " fmt, \
         in_interrupt() ? 'I' : 'U', __FUNCTION__, ## args); } while (0) 
extern unsigned long ipw2100_debug_level;
#else
#define IPW2100_DEBUG(level, fmt, args...) do {} while (0);
#endif /* CONFIG_IPW2100_DEBUG */

#define IPW2100_DL_UNINIT    0x80000000
#define IPW2100_DL_NONE      0x00000000
#define IPW2100_DL_ALL       0x7FFFFFFF

/*
 * To use the debug system;
 *
 * If you are defining a new debug classification, simply add it to the #define
 * list here in the form of:
 *
 * #define IPW2100_DL_xxxx VALUE
 * 
 * shifting value to the left one bit from the previous entry.  xxxx should be
 * the name of the classification (for example, WEP)
 *
 * You then need to either add a IPW2100_xxxx_DEBUG() macro definition for your
 * classification, or use IPW2100_DEBUG(IPW2100_DL_xxxx, ...) whenever you want
 * to send output to that classification.
 *
 * To add your debug level to the list of levels seen when you perform
 *
 * % cat /proc/net/ipw2100/debug_level
 *
 * you simply need to add your entry to the ipw2100_debug_levels array.
 *
 * If you do not see debug_level in /proc/net/ipw2100 then you do not have 
 * CONFIG_IPW2100_DEBUG defined in your kernel configuration
 *
 */

#define IPW2100_DL_ERROR         0x00000001
#define IPW2100_DL_WARNING       0x00000002
#define IPW2100_DL_INFO          0x00000004
#define IPW2100_DL_WIRELESS_EXT  0x00000008
#define IPW2100_DL_ISR           0x00000010
#define IPW2100_DL_SCAN          0x00000020
#define IPW2100_DL_STATUS        0x00000040
#define IPW2100_DL_TRACE         0x00000080
#define IPW2100_DL_RX            0x00000100
#define IPW2100_DL_TX            0x00000200
#define IPW2100_DL_MANAGE        0x00000400
#define IPW2100_DL_WEP           0x00000800
#define IPW2100_DL_FW_COMMAND    0x00001000
#define IPW2100_DL_FRAG          0x00002000
#define IPW2100_DL_FW            0x00004000
#define IPW2100_DL_RF_KILL       0x00008000
#define IPW2100_DL_DROP          0x00010000
#define IPW2100_DL_IO		 0x00020000

#ifdef CONFIG_IPW2100_DEBUG 
#ifdef IPW2100_MAIN
struct ipw2100_dl {
	const char *name;
	unsigned long value;
};
#define IPW2100_DL(v) { .name = #v, .value = v }

const struct ipw2100_dl ipw2100_debug_levels[] = {
	IPW2100_DL(IPW2100_DL_ERROR),
	IPW2100_DL(IPW2100_DL_WARNING),
	IPW2100_DL(IPW2100_DL_INFO),
	IPW2100_DL(IPW2100_DL_WIRELESS_EXT),
	IPW2100_DL(IPW2100_DL_ISR),
	IPW2100_DL(IPW2100_DL_SCAN),
	IPW2100_DL(IPW2100_DL_STATUS),
	IPW2100_DL(IPW2100_DL_TRACE),
	IPW2100_DL(IPW2100_DL_RX),
	IPW2100_DL(IPW2100_DL_TX),
	IPW2100_DL(IPW2100_DL_MANAGE),
	IPW2100_DL(IPW2100_DL_WEP),
	IPW2100_DL(IPW2100_DL_FW_COMMAND),
	IPW2100_DL(IPW2100_DL_FRAG),
	IPW2100_DL(IPW2100_DL_FW),
	IPW2100_DL(IPW2100_DL_RF_KILL),
	IPW2100_DL(IPW2100_DL_DROP),
	IPW2100_DL(IPW2100_DL_IO),
};
#endif
#endif

#define IPW2100_DEBUG_ERROR(f, a...)   IPW2100_DEBUG(IPW2100_DL_ERROR, f, ## a)
#define IPW2100_DEBUG_WARNING(f, a...) IPW2100_DEBUG(IPW2100_DL_WARNING, f, ## a)
#define IPW2100_DEBUG_INFO(f, a...)    IPW2100_DEBUG(IPW2100_DL_INFO, f, ## a)


#define IPW2100_DEBUG_WX(f, a...)     IPW2100_DEBUG(IPW2100_DL_WIRELESS_EXT, f, ## a)
#define IPW2100_DEBUG_SCAN(f, a...)   IPW2100_DEBUG(IPW2100_DL_SCAN, f, ## a)
#define IPW2100_DEBUG_STATUS(f, a...) IPW2100_DEBUG(IPW2100_DL_STATUS, f, ## a)
#define IPW2100_DEBUG_TRACE(f, a...)  IPW2100_DEBUG(IPW2100_DL_TRACE, f, ## a)
#define IPW2100_DEBUG_RX(f, a...)     IPW2100_DEBUG(IPW2100_DL_RX, f, ## a)
#define IPW2100_DEBUG_TX(f, a...)     IPW2100_DEBUG(IPW2100_DL_TX, f, ## a)
#define IPW2100_DEBUG_ISR(f, a...)    IPW2100_DEBUG(IPW2100_DL_ISR, f, ## a)
#define IPW2100_DEBUG_MANAGEMENT(f, a...) IPW2100_DEBUG(IPW2100_DL_MANAGE, f, ## a)
#define IPW2100_DEBUG_WEP(f, a...)    IPW2100_DEBUG(IPW2100_DL_WEP, f, ## a)
#define IPW2100_DEBUG_FW_COMMAND(f, a...) IPW2100_DEBUG(IPW2100_DL_FW_COMMAND, f, ## a)
#define IPW2100_DEBUG_FRAG(f, a...) IPW2100_DEBUG(IPW2100_DL_FRAG, f, ## a)
#define IPW2100_DEBUG_FW(f, a...) IPW2100_DEBUG(IPW2100_DL_FW, f, ## a)
#define IPW2100_DEBUG_RF_KILL(f, a...) IPW2100_DEBUG(IPW2100_DL_RF_KILL, f, ## a)
#define IPW2100_DEBUG_DROP(f, a...) IPW2100_DEBUG(IPW2100_DL_DROP, f, ## a)
#define IPW2100_DEBUG_IO(f, a...) IPW2100_DEBUG(IPW2100_DL_IO, f, ## a)


#define VERIFY(f) \
{ \
  int status = 0; \
  status = f; \
  if(status) \
     return status; \
}

enum {
	STATE_DISABLED = 1,
	STATE_ENABLED = 0
};

struct ssid_context {
	char ssid[IW_ESSID_MAX_SIZE + 1];
	unsigned char bssid[ETH_ALEN];
	int port_type;
	int channel;

};

extern const char *port_type_str[];
extern const char *band_str[];

#define DEFAULT_FRAG_THRESHOLD 2342U
#define MIN_FRAG_THRESHOLD     256U
#define	MAX_FRAG_THRESHOLD     2342U

#define NUMBER_OF_BD_PER_COMMAND_PACKET		1
#define NUMBER_OF_BD_PER_DATA_PACKET		2

#define NUMBER_OF_OVERHEAD_BDS_PER_PACKETR	2
#define NUMBER_OF_BDS_TO_LEAVE_FOR_COMMANDS	1

#define REQUIRED_SPACE_IN_RING_FOR_COMMAND_PACKET \
    (IPW_BD_QUEUE_W_R_MIN_SPARE + NUMBER_OF_BD_PER_COMMAND_PACKET)

struct bd_status {
	union {
		struct { u8 nlf:1, txType:2, intEnabled:1, reserved:4; };
		u8 field;
	};
} __attribute__ ((packed));

#define	IPW_BUFDESC_LAST_FRAG 0

struct ipw2100_bd {
	u32 host_addr;
	u32 buf_length;
	struct bd_status status;
        /* number of fragments for frame (should be set only for 
	 * 1st TBD) */
	u8 num_fragments;	
	u8 reserved[6];
} __attribute__ ((packed));

#define IPW_BD_QUEUE_LENGTH(n) (1<<n)
#define IPW_BD_ALIGNMENT(L)    (L*sizeof(struct ipw2100_bd))

#define IPW_BD_STATUS_TX_FRAME_NOT_LAST_FRAGMENT	0x01
#define IPW_BD_STATUS_TX_FRAME_COMMAND		0x02
#define IPW_BD_STATUS_TX_FRAME_802_11	        0x04
#define IPW_BD_STATUS_TX_INTERRUPT_ENABLE	        0x08
#define IPW_BD_STATUS_TX_FRAME_802_3              0x00

struct ipw2100_bd_queue {
	/* driver (virtual) pointer to queue */
	struct ipw2100_bd *drv;

	/* firmware (physical) pointer to queue */
	dma_addr_t nic;

	/* Length of phy memory allocated for BDs */
	u32 size;

	/* Number of BDs in queue (and in array) */
	u32 entries;

	/* Number of available BDs (invalid for NIC BDs) */
	u32 available;

	/* Offset of oldest used BD in array (next one to 
	 * check for completion) */
	u32 oldest;

	/* Offset of next available (unused) BD */
	u32 next;
};

#define STATUS_TYPE_MASK	0x0000000f
#define COMMAND_STATUS_VAL	0
#define STATUS_CHANGE_VAL	1
#define P80211_DATA_VAL 	2
#define P8023_DATA_VAL		3
#define HOST_NOTIFICATION_VAL	4

#define IPW2100_RSSI_TO_DBM (-98)

struct ipw2100_status {
	u32 frame_size;
	u16 status_fields;
	u8 flags;
#define IPW_STATUS_FLAG_DECRYPTED	(1)
#define IPW_STATUS_FLAG_WEP_ENCRYPTED	(1<<1)
	u8 rssi;
} __attribute__ ((packed));

struct ipw2100_status_queue {
	/* driver (virtual) pointer to queue */
	struct ipw2100_status *drv;

	/* firmware (physical) pointer to queue */
	dma_addr_t nic;

	/* Length of phy memory allocated for BDs */
	u32 size;
};

enum {
	BSS = 1,
	IBSS,
	MONITOR
};

/**
 * @struct _tx_cmd - HWCommand
 * @brief H/W command structure.
 */
#define HOST_COMMAND_PARAMS_REG_LEN	100
#define CMD_STATUS_PARAMS_REG_LEN 	3

struct ipw2100_cmd_header {
	u32 host_command_reg;
	u32 host_command_reg1;
	u32 sequence;
	u32 host_command_len_reg;
	u32 host_command_params_reg[HOST_COMMAND_PARAMS_REG_LEN];
	u32 cmd_status_reg;
	u32 cmd_status_params_reg[CMD_STATUS_PARAMS_REG_LEN];
	u32 rxq_base_ptr;
	u32 rxq_next_ptr;
	u32 rxq_host_ptr;
	u32 txq_base_ptr;
	u32 txq_next_ptr;
	u32 txq_host_ptr;
	u32 tx_status_reg;
	u32 reserved;
	u32 status_change_reg;
	u32 reserved1[3];
	u32 *ordinal1_ptr;
	u32 *ordinal2_ptr;
} __attribute__ ((packed));

struct ipw2100_data_header {
	u32 host_command_reg;	
	u32 host_command_reg1;
	u8 encrypted;	// BOOLEAN in win! TRUE if frame is enc by driver
	u8 needs_encryption;	// BOOLEAN in win! TRUE if frma need to be enc in NIC
	u8 wep_index;		// 0 no key, 1-4 key index, 0xff immediate key
	u8 key_size;	// 0 no imm key, 0x5 64bit encr, 0xd 128bit encr, 0x10 128bit encr and 128bit IV
	u8 key[16];
	u8 reserved[10];	// f/w reserved
	u8 src_addr[ETH_ALEN];
	u8 dst_addr[ETH_ALEN];
	u16 fragment_size;
} __attribute__ ((packed));

// Host command data structure
struct host_command {
	u32 host_command;		// COMMAND ID
	u32 host_command1;		// COMMAND ID
	u32 host_command_sequence;	// UNIQUE COMMAND NUMBER (ID)
	u32 host_command_length;	// LENGTH
	u32 host_command_parameters[HOST_COMMAND_PARAMS_REG_LEN];	// COMMAND PARAMETERS
} __attribute__ ((packed));


typedef enum {
	POWER_ON_RESET,
	EXIT_POWER_DOWN_RESET,
	SW_RESET,
	EEPROM_RW,
	SW_RE_INIT
} ipw2100_reset_event;

enum {
	COMMAND = 0xCAFE,
	DATA,
	RX
};


struct ipw2100_tx_packet {
	int type;
	int index;
	union {
		struct { /* COMMAND */
			struct ipw2100_cmd_header* cmd;
			dma_addr_t cmd_phys;
		};
		struct { /* DATA */
			struct ipw2100_data_header* data;
			dma_addr_t data_phys;
			dma_addr_t skb_phys;
			struct sk_buff_head skb_queue;
			int fragments;
		};
	};
	int jiffy_start;

	struct list_head list;
};


struct ipw2100_rx_packet {
	struct ipw2100_rx *rxp;
	dma_addr_t dma_addr;
	int jiffy_start;
	struct sk_buff *skb;
	struct list_head list;
};

#define DEFAULT_RTS_THRESHOLD     2304U
#define DEFAULT_BEACON_INTERVAL   100U
#define	DEFAULT_SHORT_RETRY_LIMIT 7U
#define	DEFAULT_LONG_RETRY_LIMIT  4U

struct ipw2100_ordinals {
	u32 table1_addr;
	u32 table2_addr;
	u32 table1_size;
	u32 table2_size;
};

/* Host Notification header */
struct ipw2100_notification {
	u32 hnhdr_subtype;	/* type of host notification */
	u32 hnhdr_size;		/* size in bytes of data 
				   or number of entries, if table.
				   Does NOT include header */
} __attribute__ ((packed));


/* SWEEP TABLE ENTRIES NUBMER*/
#define MAX_SWEEP_TAB_ENTRIES		  42
#define MAX_SWEEP_TAB_ENTRIES_PER_PACKET  7
#define MAX_RATES_LENGTH                  ((u8)8)
#define MAX_BEACON_COUNT                  128

#define SSID_LENGTH                       ((u8)32)
#define CRC_LENGTH                 4U

#define SSID_11B_RATE_1MB          0x0001
#define SSID_11B_RATE_2MB          0x0002
#define SSID_11B_RATE_5MB          0x0004
#define SSID_11B_RATE_11MB         0x0008
#define SSID_11B_RATE_22MB         0x0010

struct ipw2100_beacon;

#define MAX_KEY_SIZE	16
#define	MAX_KEYS	8

#define IPW2100_WEP_DROP_CLEAR (1<<1)
#define IPW2100_WEP_ENABLE     (1)

#define IPW_NONE_CIPHER   0x00000001
#define IPW_WEP40_CIPHER  0x00000002
#define IPW_WEP104_CIPHER 0x00000020

#define	IPW_AUTH_OPEN     0
#define	IPW_AUTH_SHARED   1

struct statistic {
	int value;
	int hi;
	int lo;
};

#define INIT_STAT(x) do {  \
  (x)->value = (x)->hi = 0; \
  (x)->lo = 0x7fffffff; \
} while (0)
#define SET_STAT(x,y) do { \
  (x)->value = y; \
  if ((x)->value > (x)->hi) (x)->hi = (x)->value; \
  if ((x)->value < (x)->lo) (x)->lo = (x)->value; \
} while (0)
#define INC_STAT(x) do { if (++(x)->value > (x)->hi) (x)->hi = (x)->value; } \
while (0)
#define DEC_STAT(x) do { if (--(x)->value < (x)->lo) (x)->lo = (x)->value; } \
while (0)

#define IPW2100_ERROR_QUEUE 5

struct ipw2100_priv {
	struct ieee80211_device ieee;
	int reset_pending; /* 1 if a reset has been scheduled or is running */
	int open;
	int stopped;
	int disabled;
	int phy_off;
	int connected;
	unsigned long connect_start;
	int been_powered; /* 1 if HW has been powered */
	u32 fatal_error;
	u32 fatal_errors[IPW2100_ERROR_QUEUE];
	u32 fatal_index;
	int rf_kill;
	int eeprom_version;
	int firmware_version;
	int irq_enabled; /* 1 when IRQ on 0 when off */
	unsigned long hw_features;
	int resets;
	int hangs;
	u32 last_rtc;
	int dump_raw; /* 1 to dump raw bytes in /proc/.../memory */
	u8* snapshot[0x30];
	u32 scans;

	int long_preamble; /* 1 when long preamble is configured */

	struct timer_list hang_check_timer;
	int stop_hang_check; /* Set 1 when shutting down to kill timer */
	struct timer_list rf_switch_timer;
	int stop_rf_check; /* Set 1 when shutting down to kill timer */
 
	/* WEP data */
	struct ieee80211_security sec;
	int security_updated;
	int wep_enabled;

	int message_sending;
	int messages_sent;
	

	int short_retry_limit;
	int long_retry_limit;

	int rts_threshold;
	u32 frag_threshold;
	u8 frag_enabled;
	int in_isr;

	int tx_power;
	u32 beacon_interval;
	int port_type;

	char nick[IW_ESSID_MAX_SIZE + 1];

	struct ssid_context *ctx;
	struct ssid_context active_ctx;
	struct ssid_context backup_ctx;

	struct ipw2100_status_queue status_queue;

	struct statistic txq_stat;
	struct statistic rxq_stat;
	struct ipw2100_bd_queue rx_queue;
	struct ipw2100_bd_queue tx_queue;
	struct ipw2100_rx_packet *rx_buffers;

	struct list_head beacon_free_list;
	struct list_head beacon_list;
	struct ipw2100_beacon *beacons;

	struct statistic fw_pend_stat;
	struct list_head fw_pend_list;

	struct statistic msg_free_stat;
	struct statistic msg_pend_stat;
	struct list_head msg_free_list;
	struct list_head msg_pend_list;
	struct ipw2100_tx_packet *msg_buffers;

	struct statistic tx_free_stat;
	struct statistic tx_pend_stat;
	struct list_head tx_free_list;
	struct list_head tx_pend_list;
	struct ipw2100_tx_packet *tx_buffers;

	struct ipw2100_ordinals ordinals;

	struct pci_dev *pdev;

	struct proc_dir_entry *dir_dev;

	struct net_device *ndev;
	struct iw_statistics wstats;

	struct tasklet_struct irq_tasklet;

	struct work_struct reset_work;
	struct work_struct security_work;
	struct work_struct wx_event_work;
	
	int wx_ap_event_pending;

/*
	struct sk_buff_head last_tx;
	struct sk_buff_head last_rx;
*/

	u32 interrupts;
	int tx_interrupts;
	int rx_interrupts;
	int inta_other;
	
	u32 proc_cardmemaddr;

	spinlock_t low_lock;

	wait_queue_head_t wait_command_queue;
#ifdef CONFIG_PM
	u32 pm_state[16];
#endif

#ifdef CONFIG_IPW2100_PROMISC
	int(*rx_dat_packet)(struct ipw2100_priv*, int);
	int(*rx_ctl_packet)(struct ipw2100_priv*, int);
	int(*rx_mgt_packet)(struct ipw2100_priv*, int);
#endif
};

/*********************************************************
 * Host Command -> From Driver to FW
 *********************************************************/

/**
 * Host command identifiers
 */
#define HOST_COMPLETE           2
#define SYSTEM_CONFIG           6
#define SSID                    8
#define MANDATORY_BSSID         9
#define AUTHENTICATION_TYPE    10
#define ADAPTER_ADDRESS        11
#define PORT_TYPE              12
#define INTERNATIONAL_MODE     13
#define CHANNEL                14
#define RTS_THRESHOLD          15
#define FRAG_THRESHOLD         16
#define POWER_MODE             17
#define TX_RATES               18
#define BASIC_TX_RATES         19
#define WEP_KEY_INFO           20
#define WEP_KEY_INDEX          25
#define WEP_FLAGS              26
#define ADD_MULTICAST          27
#define CLEAR_ALL_MULTICAST    28
#define BEACON_INTERVAL        29
#define ATIM_WINDOW            30
#define CLEAR_STATISTICS       31
#define SEND		       33
#define TX_POWER_INDEX         36
#define BROADCAST_SCAN         43
#define CARD_DISABLE           44
#define PREFERRED_BSSID        45
#define SET_SCAN_OPTIONS       46
#define SCAN_DWELL_TIME        47
#define SWEEP_TABLE            48
#define AP_OR_STATION_TABLE    49
#define GROUP_ORDINALS         50
#define SHORT_RETRY_LIMIT      51
#define LONG_RETRY_LIMIT       52

#define HOST_PRE_POWER_DOWN    58
#define CARD_DISABLE_PHY_OFF   61
#define MSDU_TX_RATES          62


// Rogue AP Detection 
#define SET_STATION_STAT_BITS      64
#define CLEAR_STATIONS_STAT_BITS   65
#define LEAP_ROGUE_MODE            66	//TODO tbw replaced by CFG_LEAP_ROGUE_AP
#define SET_SECURITY_INFORMATION   67
#define DISASSOCIATION_BSSID	   68
#define SET_WPA_ASS_IE             69



// system configuration bit mask:
//#define CFG_ANTENNA_SETTING           0x03
//#define CFG_ANTENNA_A                 0x01
//#define CFG_ANTENNA_B                 0x02
#define CFG_PROMISCUOUS         0x00004
//#define CFG_TX_STATUS_ENABLE    0x00008
#define CFG_PREAMBLE_LEN        0x00010
#define CFG_IBSS_AUTO_START     0x00020
//#define CFG_KERBEROS_ENABLE     0x00040
#define CFG_LOOPBACK            0x00100
//#define CFG_WNMP_PING_PASS      0x00200
//#define CFG_DEBUG_ENABLE        0x00400
#define CFG_ANSWER_BCSSID_PROBE 0x00800
//#define CFG_BT_PRIORITY         0x01000
#define CFG_BT_SIDEBAND_SIGNAL	0x02000
#define CFG_802_1x_ENABLE       0x04000
#define CFG_BSS_MASK		0x08000
#define CFG_IBSS_MASK		0x10000
//#define CFG_DYNAMIC_CW          0x10000 


#define IPW_NIC_FATAL_ERROR 0x2A7F0
#define IPW_ERROR_ADDR(x) (x & 0x3FFFF)
#define IPW_ERROR_CODE(x) ((x & 0xFF000000) >> 24)
#define IPW2100_ERR_C3_CORRUPTION (0x10 << 24)
#define IPW2100_ERR_MSG_TIMEOUT   (0x11 << 24)
#define IPW2100_ERR_FW_LOAD       (0x12 << 24)

#define IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND			0x200
#define IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND  	IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x0D80

#define IPW_MEM_HOST_SHARED_TX_QUEUE_0_BD_BASE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x00)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_0_BD_SIZE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x04)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_1_BD_BASE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x08)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_1_BD_SIZE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x0c)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_2_BD_BASE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x10)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_2_BD_SIZE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x14)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_3_BD_BASE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x18)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_3_BD_SIZE          (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x1c)
#define IPW_MEM_HOST_SHARED_RX_BD_BASE                  (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x40)
#define IPW_MEM_HOST_SHARED_RX_STATUS_BASE              (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x44)
#define IPW_MEM_HOST_SHARED_RX_BD_SIZE                  (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x48)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_0_READ_INDEX       (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x80)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_1_READ_INDEX       (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x84)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_2_READ_INDEX       (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x88)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_3_READ_INDEX       (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x8c)
#define IPW_MEM_HOST_SHARED_RX_READ_INDEX               (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0xa0)

#define IPW_MEM_HOST_SHARED_TX_QUEUE_BD_BASE(QueueNum) \
    (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + (QueueNum<<3))
#define IPW_MEM_HOST_SHARED_TX_QUEUE_BD_SIZE(QueueNum) \
    (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x0004+(QueueNum<<3))
#define IPW_MEM_HOST_SHARED_TX_QUEUE_READ_INDEX(QueueNum) \
    (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x0080+(QueueNum<<2))

#define IPW_MEM_HOST_SHARED_TX_QUEUE_0_WRITE_INDEX \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + 0x00)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_1_WRITE_INDEX \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + 0x04)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_2_WRITE_INDEX \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + 0x08)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_3_WRITE_INDEX \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + 0x0c)
#define IPW_MEM_HOST_SHARED_RX_WRITE_INDEX \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + 0x20)
#define IPW_MEM_HOST_SHARED_SLAVE_MODE_INT_REGISTER \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + 0x78)
#define IPW_MEM_HOST_SHARED_TX_QUEUE_WRITE_INDEX(QueueNum) \
    (IPW_MEM_SRAM_HOST_INTERRUPT_AREA_LOWER_BOUND + (QueueNum<<2))

#define IPW_MEM_HOST_SHARED_ORDINALS_TABLE_1   (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x180)
#define IPW_MEM_HOST_SHARED_ORDINALS_TABLE_2   (IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND + 0x184)

#define IPW2100_INTA_TX_TRANSFER               (0x00000001)	// Bit 0 (LSB)
#define IPW2100_INTA_RX_TRANSFER               (0x00000002)	// Bit 1
#define IPW2100_INTA_TX_COMPLETE	       (0x00000004)	// Bit 2
#define IPW2100_INTA_EVENT_INTERRUPT           (0x00000008)     // Bit 3 
#define IPW2100_INTA_STATUS_CHANGE             (0x00000010)	// Bit 4
#define IPW2100_INTA_BEACON_PERIOD_EXPIRED     (0x00000020)	// Bit 5
#define IPW2100_INTA_SLAVE_MODE_HOST_COMMAND_DONE  (0x00010000)	// Bit 16
#define IPW2100_INTA_FW_INIT_DONE              (0x01000000)	// Bit 24
#define IPW2100_INTA_FW_CALIBRATION_CALC       (0x02000000)	// Bit 25
#define IPW2100_INTA_FATAL_ERROR               (0x40000000)	// Bit 30
#define IPW2100_INTA_PARITY_ERROR              (0x80000000)	// Bit 31 (MSB)

#define IPW_AUX_HOST_RESET_REG_PRINCETON_RESET              (0x00000001)
#define IPW_AUX_HOST_RESET_REG_FORCE_NMI                    (0x00000002)
#define IPW_AUX_HOST_RESET_REG_PCI_HOST_CLUSTER_FATAL_NMI   (0x00000004)
#define IPW_AUX_HOST_RESET_REG_CORE_FATAL_NMI               (0x00000008)
#define IPW_AUX_HOST_RESET_REG_SW_RESET                     (0x00000080)
#define IPW_AUX_HOST_RESET_REG_MASTER_DISABLED              (0x00000100)
#define IPW_AUX_HOST_RESET_REG_STOP_MASTER                  (0x00000200)

#define IPW_AUX_HOST_GP_CNTRL_BIT_CLOCK_READY           (0x00000001)	// Bit 0 (LSB)
#define IPW_AUX_HOST_GP_CNTRL_BIT_HOST_ALLOWS_STANDBY   (0x00000002)	// Bit 1
#define IPW_AUX_HOST_GP_CNTRL_BIT_INIT_DONE             (0x00000004)	// Bit 2
#define IPW_AUX_HOST_GP_CNTRL_BITS_SYS_CONFIG           (0x000007c0)	// Bits 6-10
#define IPW_AUX_HOST_GP_CNTRL_BIT_BUS_TYPE              (0x00000200)	// Bit 9
#define IPW_AUX_HOST_GP_CNTRL_BIT_BAR0_BLOCK_SIZE       (0x00000400)	// Bit 10
#define IPW_AUX_HOST_GP_CNTRL_BIT_USB_MODE              (0x20000000)	// Bit 29
#define IPW_AUX_HOST_GP_CNTRL_BIT_HOST_FORCES_SYS_CLK   (0x40000000)	// Bit 30
#define IPW_AUX_HOST_GP_CNTRL_BIT_FW_FORCES_SYS_CLK     (0x80000000)	// Bit 31 (MSB)

#define IPW_BIT_GPIO_GPIO1_MASK         0x0000000C
#define IPW_BIT_GPIO_GPIO3_MASK         0x000000C0
#define IPW_BIT_GPIO_GPIO1_ENABLE       0x00000008
#define IPW_BIT_GPIO_RF_KILL            0x00010000

#define IPW_BIT_GPIO_LED_OFF            0x00002000	// Bit 13 = 1

#define IPW_REG_DOMAIN_0_OFFSET 	0x0000
#define IPW_REG_DOMAIN_1_OFFSET 	IPW_MEM_SRAM_HOST_SHARED_LOWER_BOUND

#define IPW_REG_INTA			IPW_REG_DOMAIN_0_OFFSET + 0x0008
#define IPW_REG_INTA_MASK		IPW_REG_DOMAIN_0_OFFSET + 0x000C
#define IPW_REG_INDIRECT_ACCESS_ADDRESS	IPW_REG_DOMAIN_0_OFFSET + 0x0010
#define IPW_REG_INDIRECT_ACCESS_DATA	IPW_REG_DOMAIN_0_OFFSET + 0x0014
#define IPW_REG_AUTOINCREMENT_ADDRESS	IPW_REG_DOMAIN_0_OFFSET + 0x0018
#define IPW_REG_AUTOINCREMENT_DATA	IPW_REG_DOMAIN_0_OFFSET + 0x001C
#define IPW_REG_RESET_REG		IPW_REG_DOMAIN_0_OFFSET + 0x0020
#define IPW_REG_GP_CNTRL		IPW_REG_DOMAIN_0_OFFSET + 0x0024
#define IPW_REG_GPIO			IPW_REG_DOMAIN_0_OFFSET + 0x0030
#define IPW_REG_FW_TYPE                 IPW_REG_DOMAIN_1_OFFSET + 0x0188
#define IPW_REG_FW_VERSION 		IPW_REG_DOMAIN_1_OFFSET + 0x018C
#define IPW_REG_FW_COMPATABILITY_VERSION IPW_REG_DOMAIN_1_OFFSET + 0x0190

#define IPW_REG_INDIRECT_ADDR_MASK	0x00FFFFFC

#define IPW_INTERRUPT_MASK		0xC1010013

#define IPW2100_CONTROL_REG             0x220000
#define IPW2100_CONTROL_PHY_OFF         0x8

#define IPW2100_COMMAND			0x00300004
#define IPW2100_COMMAND_PHY_ON		0x0
#define IPW2100_COMMAND_PHY_OFF		0x1

/* in DEBUG_AREA, values of memory always 0xd55555d5 */
#define IPW_REG_DOA_DEBUG_AREA_START    IPW_REG_DOMAIN_0_OFFSET + 0x0090
#define IPW_REG_DOA_DEBUG_AREA_END      IPW_REG_DOMAIN_0_OFFSET + 0x00FF
#define IPW_DATA_DOA_DEBUG_VALUE        0xd55555d5

#define IPW_INTERNAL_REGISTER_HALT_AND_RESET	0x003000e0

#define IPW_WAIT_CLOCK_STABILIZATION_DELAY	    50	// micro seconds
#define IPW_WAIT_RESET_ARC_COMPLETE_DELAY	    10	// micro seconds
#define IPW_WAIT_RESET_MASTER_ASSERT_COMPLETE_DELAY 10	// micro seconds

// BD ring queue read/write difference
#define IPW_BD_QUEUE_W_R_MIN_SPARE 2	

#define IPW_CACHE_LINE_LENGTH_DEFAULT		    0x80

#define IPW_CARD_DISABLE_COMPLETE_WAIT		    100	// 100 milli
#define IPW_CARD_DISABLE_PHY_OFF_COMPLETE_WAIT	    100	// 100 milli
#define IPW_PREPARE_POWER_DOWN_COMPLETE_WAIT	    100	// 100 milli



/*

 802.11 data frame from AP

      ,-------------------------------------------------------------------.
Bytes |  2   |  2   |    6    |    6    |    6    |  2   | 0..2312 |   4  |
      |------|------|---------|---------|---------|------|---------|------|
Desc. | ctrl | dura |  DA/RA  |   TA    |    SA   | Sequ |  frame  |  fcs |
      |      | tion | (BSSID) |         |         | ence |  data   |      |
      `-------------------------------------------------------------------'

Total: 28-2340 bytes

*/

struct ieee80211_header_data {
	u16 frame_control;
	u16 duration_id;
	u8 addr1[6];
	u8 addr2[6];
	u8 addr3[6];
	u16 seq_ctrl;
};

#define IPW_HEADER_802_11_SIZE		 sizeof(struct ieee80211_header_data)
#define IPW_MAX_802_11_PAYLOAD_LENGTH		2312
#define IPW_MAX_ACCEPTABLE_TX_FRAME_LENGTH	1536
#define IPW_MIN_ACCEPTABLE_RX_FRAME_LENGTH	60
#define IPW_MAX_ACCEPTABLE_RX_FRAME_LENGTH \
	(IPW_MAX_ACCEPTABLE_TX_FRAME_LENGTH + IPW_HEADER_802_11_SIZE - \
        sizeof(struct ethhdr))

#define IPW_802_11_FCS_LENGTH 4
#define IPW_RX_NIC_BUFFER_LENGTH \
        (IPW_MAX_802_11_PAYLOAD_LENGTH + IPW_HEADER_802_11_SIZE + \
		IPW_802_11_FCS_LENGTH)

#define IPW_802_11_PAYLOAD_OFFSET \
        (sizeof(struct ieee80211_header_data) + \
         sizeof(struct ieee_802_11_snap_header))

#define BEACON_PROBE_SSID_ID_POSITION 12

/* Management Frame Information Element Types */
#define MFIE_TYPE_SSID       0
#define MFIE_TYPE_RATES      1
#define MFIE_TYPE_FH_SET     2
#define MFIE_TYPE_DS_SET     3
#define MFIE_TYPE_CF_SET     4
#define MFIE_TYPE_TIM        5
#define MFIE_TYPE_IBSS_SET   6
#define MFIE_TYPE_CHALLENGE  16

struct ieee80211_info_element_hdr {
	u8 id;
	u8 len;
} __attribute__ ((packed));

struct ieee80211_info_element {
	u8 id;
	u8 len;
	u8 data[1];
} __attribute__ ((packed));

/*
 * These are the data types that can make up management packets
 *
	u16 auth_algorithm;
	u16 auth_sequence;
	u16 beacon_interval;
	u16 capability;
	u8 current_ap[ETH_ALEN];
	u16 listen_interval;
	struct {
		u16 association_id:14, reserved:2;
	} __attribute__ ((packed));
	u32 time_stamp[2];
	u16 reason;
	u16 status;
*/

struct ieee80211_authentication {
	struct ieee80211_header_data header;
	u16 algorithm;
	u16 transaction;
	u16 status;
	struct ieee80211_info_element info_element;
} __attribute__ ((packed));


struct ieee80211_probe_response {
	struct ieee80211_header_data header;
	u32 time_stamp[2];
	u16 beacon_interval;
	u16 capability;
	struct ieee80211_info_element info_element;
} __attribute__ ((packed));

struct ipw2100_beacon {
	u8 bssid[ETH_ALEN];
	u8 ssid_len;
/*	u8 beacon[IPW_MAX_802_11_PAYLOAD_LENGTH];
	u32 beacon_len;
	u8 response[IPW_is Z MAX_802_11_PAYLOAD_LENGTH];
	u32 response_len;*/
	s8 rssi;
	u16 capability;
	u8 channel;
	u8 rates[MAX_RATES_LENGTH];
	u8 rates_len;
	u32 last_scanned;
	/* Ensure null-terminated for any debug msgs */
	u8 ssid[SSID_LENGTH + 1];
	struct list_head list;
};

struct ipw2100_rx {
	union {
		unsigned char payload[IPW_RX_NIC_BUFFER_LENGTH];
		struct ieee80211_hdr header;
		u32 status;
		struct ipw2100_notification notification;
		struct ipw2100_cmd_header command;
	};
} __attribute__ ((packed));

// Bit 0-7 are for 802.11b tx rates - .  Bit 5-7 are reserved  
#define TX_RATE_1_MBIT              0x0001
#define TX_RATE_2_MBIT              0x0002
#define TX_RATE_5_5_MBIT            0x0004
#define TX_RATE_11_MBIT             0x0008
#define TX_RATE_MASK                0x000F

#define IPW_POWER_MODE_CAM           0	//(always on)
#define IPW_POWER_INDEX_1            1
#define IPW_POWER_INDEX_2            2
#define IPW_POWER_INDEX_3            3
#define IPW_POWER_INDEX_4            4
#define IPW_POWER_INDEX_5            5
#define IPW_POWER_AUTOMATIC          6

#define FW_SCAN_DONOT_ASSOCIATE     0x0001 // Dont Attempt to Associate after Scan
#define FW_SCAN_PASSIVE             0x0008 // Force PASSSIVE Scan

#define REG_MIN_CHANNEL             0
#define REG_MAX_CHANNEL             14

#define REG_CHANNEL_MASK            0x00003FFF

#define DIVERSITY_EITHER            0	// Use both antennas
#define DIVERSITY_ANTENNA_A         1	// Use antenna A
#define DIVERSITY_ANTENNA_B         2	// Use antenna B


#define HOST_COMMAND_WAIT 0
#define HOST_COMMAND_NO_WAIT 1

#define LOCK_NONE 0
#define LOCK_DRIVER 1
#define LOCK_FW 2

#define STATUS_INIT_COMPLETE		0x0001
#define STATUS_COUNTRY_FOUND		0x0002
#define STATUS_ASSOCIATED		0x0004
#define STATUS_ASSN_LOST		0x0008
#define STATUS_ASSN_CHANGED 		0x0010
#define STATUS_SCAN_COMPLETE		0x0020
#define STATUS_ENTERED_PSP		0x0040
#define STATUS_WAKEN_FROM_PSP		0x0080
#define STATUS_RF_KILL_SWITCH		0x0100
#define STATUS_CARD_DISABLE_COMPLETE	0x0200
#define STATUS_PREP_PWR_DOWN_COMPLETE	0x0400
#define STATUS_SCANNING 		0x0800	// STATUS_FW_SCAN_KEEP_ALIVE

#define TYPE_SWEEP_ORD                  0x000D
#define TYPE_IBSS_STTN_ORD              0x000E
#define TYPE_BSS_AP_ORD                 0x000F
#define TYPE_RAW_BEACON_ENTRY           0x0010
#define TYPE_CALIBRATION_DATA           0x0011
#define TYPE_ROGUE_AP_DATA              0x0012
#define TYPE_ASSOCIATION_REQUEST	0x0013
#define TYPE_REASSOCIATION_REQUEST	0x0014


#define HW_FEATURE_RFKILL (0x0001)
#define RF_KILLSWITCH_OFF (1)
#define RF_KILLSWITCH_ON  (0)

#define IPW_COMMAND_POOL_SIZE        40

#define IPW_START_ORD_TAB_1			1
#define IPW_START_ORD_TAB_2			1000

#define IPW_ORD_TAB_1_ENTRY_SIZE		sizeof(u32)

#define IS_ORDINAL_TABLE_ONE(mgr,id) \
    ((id >= IPW_START_ORD_TAB_1) && (id < mgr->table1_size))
#define IS_ORDINAL_TABLE_TWO(mgr,id) \
    ((id >= IPW_START_ORD_TAB_2) && (id < (mgr->table2_size + IPW_START_ORD_TAB_2)))

#define BSS_ID_LENGTH               6

// Fixed size data: Ordinal Table 1
typedef enum _ORDINAL_TABLE_1 {	// NS - means Not Supported by FW
// Transmit statistics
	IPW_ORD_STAT_TX_HOST_REQUESTS = 1,// # of requested Host Tx's (MSDU)
	IPW_ORD_STAT_TX_HOST_COMPLETE,	// # of successful Host Tx's (MSDU)
	IPW_ORD_STAT_TX_DIR_DATA,	// # of successful Directed Tx's (MSDU)

	IPW_ORD_STAT_TX_DIR_DATA1 = 4,	// # of successful Directed Tx's (MSDU) @ 1MB
	IPW_ORD_STAT_TX_DIR_DATA2,	// # of successful Directed Tx's (MSDU) @ 2MB
	IPW_ORD_STAT_TX_DIR_DATA5_5,	// # of successful Directed Tx's (MSDU) @ 5_5MB
	IPW_ORD_STAT_TX_DIR_DATA11,	// # of successful Directed Tx's (MSDU) @ 11MB
	IPW_ORD_STAT_TX_DIR_DATA22,	// # of successful Directed Tx's (MSDU) @ 22MB

	IPW_ORD_STAT_TX_NODIR_DATA1 = 13,// # of successful Non_Directed Tx's (MSDU) @ 1MB
	IPW_ORD_STAT_TX_NODIR_DATA2,	// # of successful Non_Directed Tx's (MSDU) @ 2MB
	IPW_ORD_STAT_TX_NODIR_DATA5_5,	// # of successful Non_Directed Tx's (MSDU) @ 5.5MB
	IPW_ORD_STAT_TX_NODIR_DATA11,	// # of successful Non_Directed Tx's (MSDU) @ 11MB

	IPW_ORD_STAT_NULL_DATA = 21,	// # of successful NULL data Tx's
	IPW_ORD_STAT_TX_RTS,	        // # of successful Tx RTS
	IPW_ORD_STAT_TX_CTS,	        // # of successful Tx CTS            
	IPW_ORD_STAT_TX_ACK,	        // # of successful Tx ACK
	IPW_ORD_STAT_TX_ASSN,	        // # of successful Association Tx's               
	IPW_ORD_STAT_TX_ASSN_RESP,	// # of successful Association response Tx's   
	IPW_ORD_STAT_TX_REASSN,	        // # of successful Reassociation Tx's           
	IPW_ORD_STAT_TX_REASSN_RESP,	// # of successful Reassociation response Tx's                
	IPW_ORD_STAT_TX_PROBE,	        // # of probes successfully transmitted 
	IPW_ORD_STAT_TX_PROBE_RESP,	// # of probe responses successfully transmitted                
	IPW_ORD_STAT_TX_BEACON,	        // # of tx beacon             
	IPW_ORD_STAT_TX_ATIM,	        // # of Tx ATIM            
	IPW_ORD_STAT_TX_DISASSN,	// # of successful Disassociation TX           
	IPW_ORD_STAT_TX_AUTH,	        // # of successful Authentication Tx                 
	IPW_ORD_STAT_TX_DEAUTH,	        // # of successful Deauthentication TX                

	IPW_ORD_STAT_TX_TOTAL_BYTES = 41,// Total successful Tx data bytes
	IPW_ORD_STAT_TX_RETRIES,         // # of Tx retries
	IPW_ORD_STAT_TX_RETRY1,          // # of Tx retries at 1MBPS
	IPW_ORD_STAT_TX_RETRY2,          // # of Tx retries at 2MBPS
	IPW_ORD_STAT_TX_RETRY5_5,	 // # of Tx retries at 5.5MBPS
	IPW_ORD_STAT_TX_RETRY11,	 // # of Tx retries at 11MBPS

	IPW_ORD_STAT_TX_FAILURES = 51,	// # of Tx Failures
	IPW_ORD_STAT_TX_ABORT_AT_HOP,	//NS // # of Tx's aborted at hop time
	IPW_ORD_STAT_TX_MAX_TRIES_IN_HOP,// # of times max tries in a hop failed
	IPW_ORD_STAT_TX_ABORT_LATE_DMA,	//NS // # of times tx aborted due to late dma setup
	IPW_ORD_STAT_TX_ABORT_STX,	//NS // # of times backoff aborted
	IPW_ORD_STAT_TX_DISASSN_FAIL,	// # of times disassociation failed
	IPW_ORD_STAT_TX_ERR_CTS,         // # of missed/bad CTS frames
	IPW_ORD_STAT_TX_BPDU,	        //NS // # of spanning tree BPDUs sent
	IPW_ORD_STAT_TX_ERR_ACK,	// # of tx err due to acks

	// Receive statistics
	IPW_ORD_STAT_RX_HOST = 61,	// # of packets passed to host
	IPW_ORD_STAT_RX_DIR_DATA,	// # of directed packets                
	IPW_ORD_STAT_RX_DIR_DATA1,	// # of directed packets at 1MB            
	IPW_ORD_STAT_RX_DIR_DATA2,	// # of directed packets at 2MB           
	IPW_ORD_STAT_RX_DIR_DATA5_5,	// # of directed packets at 5.5MB            
	IPW_ORD_STAT_RX_DIR_DATA11,	// # of directed packets at 11MB            
	IPW_ORD_STAT_RX_DIR_DATA22,	// # of directed packets at 22MB            

	IPW_ORD_STAT_RX_NODIR_DATA = 71,// # of nondirected packets        
	IPW_ORD_STAT_RX_NODIR_DATA1,	// # of nondirected packets at 1MB            
	IPW_ORD_STAT_RX_NODIR_DATA2,	// # of nondirected packets at 2MB           
	IPW_ORD_STAT_RX_NODIR_DATA5_5,	// # of nondirected packets at 5.5MB            
	IPW_ORD_STAT_RX_NODIR_DATA11,	// # of nondirected packets at 11MB            

	IPW_ORD_STAT_RX_NULL_DATA = 80,	// # of null data rx's            
	IPW_ORD_STAT_RX_POLL,	//NS // # of poll rx                    
	IPW_ORD_STAT_RX_RTS,	// # of Rx RTS                
	IPW_ORD_STAT_RX_CTS,	// # of Rx CTS                
	IPW_ORD_STAT_RX_ACK,	// # of Rx ACK                
	IPW_ORD_STAT_RX_CFEND,	// # of Rx CF End                
	IPW_ORD_STAT_RX_CFEND_ACK,	// # of Rx CF End + CF Ack             
	IPW_ORD_STAT_RX_ASSN,	// # of Association Rx's                    
	IPW_ORD_STAT_RX_ASSN_RESP,	// # of Association response Rx's            
	IPW_ORD_STAT_RX_REASSN,	// # of Reassociation Rx's            
	IPW_ORD_STAT_RX_REASSN_RESP,	// # of Reassociation response Rx's            
	IPW_ORD_STAT_RX_PROBE,	// # of probe Rx's                
	IPW_ORD_STAT_RX_PROBE_RESP,	// # of probe response Rx's            
	IPW_ORD_STAT_RX_BEACON,	// # of Rx beacon            
	IPW_ORD_STAT_RX_ATIM,	// # of Rx ATIM                    
	IPW_ORD_STAT_RX_DISASSN,	// # of disassociation Rx            
	IPW_ORD_STAT_RX_AUTH,	// # of authentication Rx                    
	IPW_ORD_STAT_RX_DEAUTH,	// # of deauthentication Rx            

	IPW_ORD_STAT_RX_TOTAL_BYTES = 101,// Total rx data bytes received
	IPW_ORD_STAT_RX_ERR_CRC,	 // # of packets with Rx CRC error
	IPW_ORD_STAT_RX_ERR_CRC1,	 // # of Rx CRC errors at 1MB
	IPW_ORD_STAT_RX_ERR_CRC2,	 // # of Rx CRC errors at 2MB
	IPW_ORD_STAT_RX_ERR_CRC5_5,	 // # of Rx CRC errors at 5.5MB
	IPW_ORD_STAT_RX_ERR_CRC11,	 // # of Rx CRC errors at 11MB

	IPW_ORD_STAT_RX_DUPLICATE1 = 112, // # of duplicate rx packets at 1MB
	IPW_ORD_STAT_RX_DUPLICATE2,	 // # of duplicate rx packets at 2MB
	IPW_ORD_STAT_RX_DUPLICATE5_5,	 // # of duplicate rx packets at 5.5MB
	IPW_ORD_STAT_RX_DUPLICATE11,	 // # of duplicate rx packets at 11MB
	IPW_ORD_STAT_RX_DUPLICATE = 119, // # of duplicate rx packets            

	IPW_ORD_PERS_DB_LOCK = 120,	// # locking fw permanent  db
	IPW_ORD_PERS_DB_SIZE,	// # size of fw permanent  db           
	IPW_ORD_PERS_DB_ADDR,	// # address of fw permanent  db           
	IPW_ORD_STAT_RX_INVALID_PROTOCOL,	// # of rx frames with invalid protocol        
	IPW_ORD_SYS_BOOT_TIME,	// # Boot time 
	IPW_ORD_STAT_RX_NO_BUFFER,	// # of rx frames rejected due to no buffer 
	IPW_ORD_STAT_RX_ABORT_LATE_DMA,	//NS // # of rx frames rejected due to dma setup too late    
	IPW_ORD_STAT_RX_ABORT_AT_HOP,	//NS // # of rx frames aborted due to hop            
	IPW_ORD_STAT_RX_MISSING_FRAG,	// # of rx frames dropped due to missing fragment            
	IPW_ORD_STAT_RX_ORPHAN_FRAG,	// # of rx frames dropped due to non-sequential fragment        
	IPW_ORD_STAT_RX_ORPHAN_FRAME,	// # of rx frames dropped due to unmatched 1st frame            
	IPW_ORD_STAT_RX_FRAG_AGEOUT,	// # of rx frames dropped due to uncompleted frame        
	IPW_ORD_STAT_RX_BAD_SSID,	//NS // Bad SSID (unused)            
	IPW_ORD_STAT_RX_ICV_ERRORS,	// # of ICV errors during decryption        

// PSP Statistics
	IPW_ORD_STAT_PSP_SUSPENSION = 137,// # of times adapter suspended        
	IPW_ORD_STAT_PSP_BCN_TIMEOUT,	// # of beacon timeout           
	IPW_ORD_STAT_PSP_POLL_TIMEOUT,	// # of poll response timeouts       
	IPW_ORD_STAT_PSP_NONDIR_TIMEOUT,// # of timeouts waiting for last broadcast/muticast pkt
	IPW_ORD_STAT_PSP_RX_DTIMS,	// # of PSP DTIMs received
	IPW_ORD_STAT_PSP_RX_TIMS,	// # of PSP TIMs received
	IPW_ORD_STAT_PSP_STATION_ID,	// PSP Station ID

// Association and roaming
	IPW_ORD_LAST_ASSN_TIME = 147,	// RTC time of last association               
	IPW_ORD_STAT_PERCENT_MISSED_BCNS,// current calculation of % missed beacons       
	IPW_ORD_STAT_PERCENT_RETRIES,	// current calculation of % missed tx retries            
	IPW_ORD_ASSOCIATED_AP_PTR,	// If associated, this is ptr to the associated
	// AP table entry. set to 0 if not associated
	IPW_ORD_AVAILABLE_AP_CNT,	// # of AP's decsribed in the AP table
	IPW_ORD_AP_LIST_PTR,	// Ptr to list of available APs
	IPW_ORD_STAT_AP_ASSNS,	// # of associations       
	IPW_ORD_STAT_ASSN_FAIL,	// # of association failures       
	IPW_ORD_STAT_ASSN_RESP_FAIL,	// # of failuresdue to response fail       
	IPW_ORD_STAT_FULL_SCANS,	// # of full scans

	IPW_ORD_CARD_DISABLED,	// # Card Disabled 
	IPW_ORD_STAT_ROAM_INHIBIT,	// # of times roaming was inhibited due to ongoing activity       
	IPW_FILLER_40,
	IPW_ORD_RSSI_AT_ASSN = 160,	// RSSI of associated AP at time of association           
	IPW_ORD_STAT_ASSN_CAUSE1,	// # of reassociations due to no tx from AP in last N
	// hops or no probe responses in last 3 minutes           
	IPW_ORD_STAT_ASSN_CAUSE2,	// # of reassociations due to poor tx/rx quality            
	IPW_ORD_STAT_ASSN_CAUSE3,	// # of reassociations due to tx/rx quality with excessive 
	// load at the AP        
	IPW_ORD_STAT_ASSN_CAUSE4,	// # of reassociations due to AP RSSI level fell below
	// eligible group             
	IPW_ORD_STAT_ASSN_CAUSE5,	// # of reassociations due to load leveling            
	IPW_ORD_STAT_ASSN_CAUSE6,	//NS // # of reassociations due to dropped by Ap            
	IPW_FILLER_41,
	IPW_FILLER_42,
	IPW_FILLER_43,
	IPW_ORD_STAT_AUTH_FAIL,	// # of times authentication failed        
	IPW_ORD_STAT_AUTH_RESP_FAIL,	// # of times authentication response failed       
	IPW_ORD_STATION_TABLE_CNT,	// # of entries in association table        

// Other statistics
	IPW_ORD_RSSI_AVG_CURR = 173,	// Current avg RSSI
	IPW_ORD_STEST_RESULTS_CURR,	//NS // Current self test results word       
	IPW_ORD_STEST_RESULTS_CUM,	//NS // Cummulative self test results word       
	IPW_ORD_SELF_TEST_STATUS,	//NS //        
	IPW_ORD_POWER_MGMT_MODE,	// Power mode - 0=CAM, 1=PSP       
	IPW_ORD_POWER_MGMT_INDEX,	//NS //            
	IPW_ORD_COUNTRY_CODE,	// IEEE country code as recv'd from beacon
	IPW_ORD_COUNTRY_CHANNELS,	// channels suported by country           
// IPW_ORD_COUNTRY_CHANNELS:
// For 11b the lower 2-byte are used for channels from 1-14 
//   and the higher 2-byte are not used.
	IPW_ORD_RESET_CNT,	// # of adapter resets (warm)       
	IPW_ORD_BEACON_INTERVAL,	// Beacon interval       

	IPW_ORD_PRINCETON_VERSION = 184,	//NS // Princeton Version       
	IPW_ORD_ANTENNA_DIVERSITY,	// TRUE if antenna diversity is disabled       
	IPW_ORD_CCA_RSSI,	//NS // CCA RSSI value (factory programmed)           
	IPW_ORD_STAT_EEPROM_UPDATE,	//NS // # of times config EEPROM updated       
	IPW_ORD_DTIM_PERIOD,	// # of beacon intervals between DTIMs       
	IPW_ORD_OUR_FREQ,	// current radio freq lower digits - channel ID

	IPW_ORD_RTC_TIME = 190,	// current RTC time           
	IPW_ORD_PORT_TYPE,	// operating mode       
	IPW_ORD_CURRENT_TX_RATE,	// current tx rate
	IPW_ORD_SUPPORTED_RATES,	// Bitmap of supported tx rates       
	IPW_ORD_ATIM_WINDOW,	// current ATIM Window
	IPW_ORD_BASIC_RATES,	// bitmap of basic tx rates       
	IPW_ORD_NIC_HIGHEST_RATE,	// bitmap of basic tx rates       
	IPW_ORD_AP_HIGHEST_RATE,	// bitmap of basic tx rates       
	IPW_ORD_CAPABILITIES,	// Management frame capability field           
	IPW_ORD_AUTH_TYPE,	// Type of authentication       
	IPW_ORD_RADIO_TYPE,	// Adapter card platform type       
	IPW_ORD_RTS_THRESHOLD = 201,	// Min length of packet after which RTS handshaking is used       
	IPW_ORD_INT_MODE,	// International mode           
	IPW_ORD_FRAGMENTATION_THRESHOLD,	// protocol frag threshold       
	IPW_ORD_EEPROM_SRAM_DB_BLOCK_START_ADDRESS,	// EEPROM offset in SRAM
	IPW_ORD_EEPROM_SRAM_DB_BLOCK_SIZE,	// EEPROM size in SRAM
	IPW_ORD_EEPROM_SKU_CAPABILITY,	// EEPROM SKU Capability    206 =  
	IPW_ORD_EEPROM_IBSS_11B_CHANNELS,	// EEPROM IBSS 11b channel set       

	IPW_ORD_MAC_VERSION = 209,	// MAC Version       
	IPW_ORD_MAC_REVISION,	// MAC Revision       
	IPW_ORD_RADIO_VERSION,	// Radio Version       
	IPW_ORD_NIC_MANF_DATE_TIME,	// MANF Date/Time STAMP
	IPW_ORD_UCODE_VERSION,	// Ucode Version       
	IPW_ORD_HW_RF_SWITCH_STATE = 214,	// HW RF Kill Switch State       
} ORDINALTABLE1;
//ENDOF TABLE1

// ordinal table 2
// Variable length data:
#define IPW_FIRST_VARIABLE_LENGTH_ORDINAL   1001

typedef enum _ORDINAL_TABLE_2 {	// NS - means Not Supported by FW
	IPW_ORD_STAT_BASE = 1000,	// contains number of variable ORDs
	IPW_ORD_STAT_ADAPTER_MAC = 1001,	// 6 bytes: our adapter MAC address  
	IPW_ORD_STAT_PREFERRED_BSSID = 1002,	// 6 bytes: BSSID of the preferred AP
	IPW_ORD_STAT_MANDATORY_BSSID = 1003,	// 6 bytes: BSSID of the mandatory AP
	IPW_FILL_1,		//NS // 
	IPW_ORD_STAT_COUNTRY_TEXT = 1005,	// 36 bytes: Country name text, First two bytes are Country code
	IPW_ORD_STAT_ASSN_SSID = 1006,	// 32 bytes: ESSID String
	IPW_ORD_STATION_TABLE = 1007,	// ? bytes: Station/AP table (via Direct SSID Scans)
	IPW_ORD_STAT_SWEEP_TABLE = 1008,	// ? bytes: Sweep/Host Table table (via Broadcast Scans)
	IPW_ORD_STAT_ROAM_LOG = 1009,	// ? bytes: Roaming log
	IPW_ORD_STAT_RATE_LOG = 1010,	//NS // 0 bytes: Rate log
	IPW_ORD_STAT_FIFO = 1011,	//NS // 0 bytes: Fifo buffer data structures
	IPW_ORD_STAT_FW_VER_NUM = 1012,	// 14 bytes: fw version ID string as in (a.bb.ccc; "0.08.011")
	IPW_ORD_STAT_FW_DATE = 1013,	// 14 bytes: fw date string (mmm dd yyyy; "Mar 13 2002")
	IPW_ORD_STAT_ASSN_AP_BSSID = 1014,	// 6 bytes: MAC address of associated AP
	IPW_ORD_STAT_DEBUG = 1015,	//NS // ? bytes:
	IPW_ORD_STAT_NIC_BPA_NUM = 1016,	// 11 bytes: NIC BPA number in ASCII
	IPW_ORD_STAT_UCODE_DATE = 1017,	// 5 bytes: uCode date
	IPW_ORD_SECURITY_NGOTIATION_RESULT = 1018,
} ORDINALTABLE2;		// NS - means Not Supported by FW

#define IPW_LAST_VARIABLE_LENGTH_ORDINAL   1018

extern int ipw2100_get_ordinal(struct ipw2100_priv *, u32, void *, u32 *);
extern int ipw2100_set_channel(struct ipw2100_priv *, u32, int);
extern int ipw2100_set_port_type(struct ipw2100_priv *, u32, int);
extern int ipw2100_set_mandatory_bssid(struct ipw2100_priv *, u8 *, int);
extern int ipw2100_set_essid(struct ipw2100_priv *, char *, int);
extern int ipw2100_set_power_mode(struct ipw2100_priv *, int);
extern int ipw2100_set_rts_threshold(struct ipw2100_priv *, u32);
extern int ipw2100_set_short_retry(struct ipw2100_priv *, u32);
extern int ipw2100_set_long_retry(struct ipw2100_priv *, u32);
extern int ipw2100_start_scan(struct ipw2100_priv *priv);
#ifdef CONFIG_IPW2100_PROMISC
extern int ipw2100_set_promisc(struct ipw2100_priv *, int, u32);
extern int ipw2100_reset_adapter(struct ipw2100_priv *priv);
#endif

#include <linux/ctype.h>
extern inline char *snprint_line(char *buf, size_t count, 
				 u8 *data, u32 len, u32 ofs)
{
	int out, i, j, l;
	char c;
	
	out = snprintf(buf, count, "%08X", ofs);

	for (l = 0, i = 0; i < 2; i++) {
		out += snprintf(buf + out, count - out, " ");
		for (j = 0; j < 8 && l < len; j++, l++) 
			out += snprintf(buf + out, count - out, "%02X ", 
					data[(i * 8 + j)]);
		for (; j < 8; j++)
			out += snprintf(buf + out, count - out, "   ");
	}
	
	out += snprintf(buf + out, count - out, " ");
	for (l = 0, i = 0; i < 2; i++) {
		out += snprintf(buf + out, count - out, " ");
		for (j = 0; j < 8 && l < len; j++, l++) {
			c = data[(i * 8 + j)];
			if (!isascii(c) || !isprint(c))
				c = '.';
			
			out += snprintf(buf + out, count - out, "%c", c);
		}

		for (; j < 8; j++)
			out += snprintf(buf + out, count - out, " ");
	}
	
	return buf;
}

extern inline void printk_buf(u8 *data, u32 len)
{
	char line[81];
	u32 ofs = 0;

	while (len) {
		printk(KERN_DEBUG "%s\n",
		       snprint_line(line, sizeof(line), &data[ofs], 
				    min(len, 16U), ofs));
		ofs += 16;
		len -= min(len, 16U);
	}
}

#endif	/* _IPW2100_H */
