#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/grinternal.h>

int
gr_handle_chroot_unix(const pid_t pid)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_UNIX
	struct task_struct *p, **htable;

	if (unlikely(!grsec_enable_chroot_unix))
		return 1;

	if (likely(!proc_is_chrooted(current)))
		return 1;

	read_lock(&tasklist_lock);

	htable = &pidhash[pid_hashfn(pid)];

	for (p = *htable; p && p->pid != pid; p = p->pidhash_next) ;

	if (unlikely(p && !have_same_root(current, p))) {
		read_unlock(&tasklist_lock);
		security_alert(GR_UNIX_CHROOT_MSG, DEFAULTSECARGS);
		return 0;
	}
	read_unlock(&tasklist_lock);
#endif
	return 1;
}

int
gr_handle_chroot_nice(void)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_NICE
	if (grsec_enable_chroot_nice && proc_is_chrooted(current)) {
		security_alert(GR_NICE_CHROOT_MSG, DEFAULTSECARGS);
		return -EPERM;
	}
#endif
	return 0;
}

int
gr_handle_chroot_setpriority(struct task_struct *p, const int niceval)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_NICE
	if (grsec_enable_chroot_nice && (niceval < task_nice(p))
			&& proc_is_chrooted(current)) {
		security_alert(GR_PRIORITY_CHROOT_MSG, p->comm, p->pid,
			       DEFAULTSECARGS);
		return -EACCES;
	}
#endif
	return 0;
}

int
gr_handle_chroot_capset(const struct task_struct *target)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_CAPS
	if (grsec_enable_chroot_caps && proc_is_chrooted(current) &&
	    !have_same_root(current, target)) {
		security_alert(GR_CAPSET_CHROOT_MSG, target->comm,
			       target->pid, DEFAULTSECARGS);
		return 1;
	}
#endif
	return 0;
}

int
gr_pid_is_chrooted(const struct task_struct *p)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_FINDTASK
	if (!grsec_enable_chroot_findtask || (current->pid <= 1))
		return 0;

	if (p && p->fs && p->fs->root && p->fs->root->d_inode &&
	    child_reaper && child_reaper->fs && child_reaper->fs->root &&
	    child_reaper->fs->root->d_inode && current && current->fs &&
	    current->fs->root && current->fs->root->d_inode) {
		if (proc_is_chrooted(current) && !have_same_root(current, p))
			return 1;
	}
#endif
	return 0;
}

int
gr_chroot_fchdir(struct dentry *u_dentry, struct vfsmount *u_mnt)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_FCHDIR
	if (!grsec_enable_chroot_fchdir)
		return 1;

	if (!proc_is_chrooted(current))
		return 1;
	else {
		struct dentry *dentry = u_dentry;
		struct vfsmount *mnt = u_mnt;
		struct dentry *realroot;
		struct vfsmount *realrootmnt;
		struct dentry *currentroot;
		struct vfsmount *currentmnt;

		read_lock(&child_reaper->fs->lock);
		realrootmnt = mntget(child_reaper->fs->rootmnt);
		realroot = dget(child_reaper->fs->root);
		read_unlock(&child_reaper->fs->lock);

		read_lock(&current->fs->lock);
		currentmnt = mntget(current->fs->rootmnt);
		currentroot = dget(current->fs->root);
		read_unlock(&current->fs->lock);

		spin_lock(&dcache_lock);
		for (;;) {
			if (unlikely
			    ((dentry == realroot && mnt == realrootmnt)
			     || (dentry == currentroot && mnt == currentmnt)))
				break;
			if (unlikely
			    (dentry == mnt->mnt_root || IS_ROOT(dentry))) {
				if (mnt->mnt_parent == mnt)
					break;
				dentry = mnt->mnt_mountpoint;
				mnt = mnt->mnt_parent;
				continue;
			}
			dentry = dentry->d_parent;
		}
		spin_unlock(&dcache_lock);

		dput(currentroot);
		mntput(currentmnt);

		if (dentry == realroot && mnt == realrootmnt) {
			/* ok, they're definitely trying to fchdir outside of the
			   chroot. */
			dput(realroot);
			mntput(realrootmnt);
			security_alert(GR_CHROOT_FCHDIR_MSG,
				       gr_to_filename(u_dentry, u_mnt),
				       DEFAULTSECARGS);
			return 0;
		} else {
			dput(realroot);
			mntput(realrootmnt);
			return 1;
		}
	}
#endif
	return 1;
}

int
gr_chroot_shmat(const pid_t shm_cprid, const pid_t shm_lapid,
		const time_t shm_createtime)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_SHMAT
	struct task_struct *p, **htable;

	if (unlikely(!grsec_enable_chroot_shmat))
		return 1;

	if (likely(!proc_is_chrooted(current)))
		return 1;

	read_lock(&tasklist_lock);

	htable = &pidhash[pid_hashfn(shm_cprid)];

	for (p = *htable; p && p->pid != shm_cprid; p = p->pidhash_next) ;

	if (unlikely(p && !have_same_root(current, p) &&
		     (p->start_time < shm_createtime))) {
		read_unlock(&tasklist_lock);
		security_alert(GR_SHMAT_CHROOT_MSG, DEFAULTSECARGS);
		return 0;
	}

	if (unlikely(!p)) {
		htable = &pidhash[pid_hashfn(shm_lapid)];
		for (p = *htable; p && p->pid != shm_lapid;
		     p = p->pidhash_next) ;

		if (unlikely(p && !have_same_root(current, p))) {
			read_unlock(&tasklist_lock);
			security_alert(GR_SHMAT_CHROOT_MSG, DEFAULTSECARGS);
			return 0;
		}
	}

	read_unlock(&tasklist_lock);
#endif
	return 1;
}

void
gr_log_chroot_exec(const struct dentry *dentry, const struct vfsmount *mnt)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_EXECLOG
	if (grsec_enable_chroot_execlog && proc_is_chrooted(current))
		security_audit(GR_EXEC_CHROOT_MSG, gr_to_filename(dentry, mnt),
			       DEFAULTSECARGS);
#endif
	return;
}

int
gr_handle_chroot_mknod(const struct dentry *dentry,
		       const struct vfsmount *mnt, const int mode)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_MKNOD
	if (grsec_enable_chroot_mknod && !S_ISFIFO(mode) &&
	    proc_is_chrooted(current)) {
		security_alert(GR_MKNOD_CHROOT_MSG,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);
		return -EPERM;
	}
#endif
	return 0;
}

int
gr_handle_chroot_mount(const struct dentry *dentry,
		       const struct vfsmount *mnt, const char *dev_name)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_MOUNT
	if (grsec_enable_chroot_mount && proc_is_chrooted(current)) {
		security_alert(GR_MOUNT_CHROOT_MSG, dev_name,
			       gr_to_filename(dentry, mnt),
			       DEFAULTSECARGS);
		return -EPERM;
	}
#endif
	return 0;
}

int
gr_handle_chroot_pivot(void)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_PIVOT
	if (grsec_enable_chroot_pivot && proc_is_chrooted(current)) {
		security_alert(GR_PIVOT_CHROOT_MSG, DEFAULTSECARGS);
		return -EPERM;
	}
#endif
	return 0;
}

int
gr_handle_chroot_chroot(const struct dentry *dentry, const struct vfsmount *mnt)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_DOUBLE
	if (grsec_enable_chroot_double && proc_is_chrooted(current)) {
		security_alert(GR_CHROOT_CHROOT_MSG,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);
		return -EPERM;
	}
#endif
	return 0;
}

void
gr_handle_chroot_caps(struct task_struct *task)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_CAPS
	if (grsec_enable_chroot_caps && proc_is_chrooted(task)) {
		task->cap_permitted =
		    cap_drop(task->cap_permitted, GR_CHROOT_CAPS);
		task->cap_inheritable =
		    cap_drop(task->cap_inheritable, GR_CHROOT_CAPS);
		task->cap_effective =
		    cap_drop(task->cap_effective, GR_CHROOT_CAPS);
	}
#endif
	return;
}

int
gr_handle_chroot_sysctl(const int op)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_SYSCTL
	if (grsec_enable_chroot_sysctl && proc_is_chrooted(current)
	    && (op & 002))
		return -EACCES;
#endif
	return 0;
}

void
gr_handle_chroot_chdir(struct dentry *dentry, struct vfsmount *mnt)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_CHDIR
	if (grsec_enable_chroot_chdir)
		set_fs_pwd(current->fs, mnt, dentry);
#endif
	return;
}

int
gr_handle_chroot_chmod(const struct dentry *dentry,
		       const struct vfsmount *mnt, const int mode)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_CHMOD
	if (grsec_enable_chroot_chmod &&
	    ((mode & S_ISUID) || (mode & S_ISGID)) &&
	    proc_is_chrooted(current)) {
		security_alert(GR_CHMOD_CHROOT_MSG,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);
		return -EPERM;
	}
#endif
	return 0;
}
