#define UwsUtil_c

// **********************************************************************
// * uwsutil.c                                                          *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// **********************************************************************
// * uwsutil.c - This source file provides assorted utility routines.   *
// *                                                                    *
// * Exported Functions                                                 *
// * ------------------                                                 *
// * sysSpCmdTimeout     - registered as a timeout routine when a       *
// *                       command is sent to ASM, in case a timely     *
// *                       response is not received                     *
// * sysSpRunSpCmd       - called by a completed command, to cause the  *
// *                       next queued command to proceed in the        *
// *                       sending process                              *
// *                                                                    *
// * Changes                                                            *
// * ------------------------------------------------------------------ *
// * 01/29/2001 - Cleaned up for open source release.                   *
// * ------------------------------------------------------------------ *
// * 06/27/2002 - PHR_MULTI_4  - Added multi node support               *
// *                                                                    *
// **********************************************************************
                               
// ************
// * INCLUDES *
// ************

#include "uwscmd.h"

// ***********************
// * EXTERNAL PROTOTYPES *
// ***********************
extern PWORK_BLOCK sysSpDequeue(PDRIVER_INFO pDriverNode, void *pHead);
extern void        sysSpRunSpCmd(PDRIVER_INFO pDriverNode);
extern PWORK_BLOCK sysSpPeekQueue(void *pHead);
extern void        sysSpDisplayQueue(void *pHead);
extern void        sysSpProcessIsaRequest(PCMD_BLOCK pCmdBlk, unsigned long ultimeoutValue);
extern void        sysSpProcessPciRequest(PDRIVER_INFO pDriverNode, PCMD_BLOCK pCmdBlk, unsigned long ultimeoutValue);

// ********************
// * LOCAL PROTOTYPES *
// ********************
void sysSpRevHbTimeout(PDRIVER_INFO pDriverNode);
void sysSpCmdTimeout(PDRIVER_INFO pDriverNode);
void sysSpRunSpCmd(PDRIVER_INFO pDriverNode);

// ************************************************************************
// * sysSpCmdTimeout() - This function is registered as a callback for    *
// *                     the SP command timeout timer, and is called when *
// *                     it expires. It sets that command's completion    *
// *                     code and releases the corresponding blocked      *
// *                     thread.                                          *
// *                                                                      *
// * Parameters:                                                          *
// *      pDriverNode - Pointer to the driver info structure, which       *
// *                    contians a pointer to the corresponding timed out *
// *                    command blk.                                      *
// ************************************************************************
void sysSpCmdTimeout(PDRIVER_INFO pDriverNode)
{
   unsigned long flags = 0;

   // LOCK COMMAND QUEUE
   spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );

   // CHECK WHETHER THIS COMMAND HAS COMPLETED YET
   if ( pDriverNode->pActiveCmdBlk->completionCode == DDERR_CMD_PENDING )
   {
       if (DBG_CMDTIMEOUT)
       {
          printk(KERN_CRIT "ibmasm: DBG_CMDTIMEOUT - Command time out detected.\n");
       }

       // SET COMPLETION CODE TO TIMEOUT
       pDriverNode->pActiveCmdBlk->completionCode = DDERR_RCV_TIMEOUT;

       // STOP COMMAND TIMEOUT TIMER
       del_timer(&(pDriverNode->CommandTimer));

       if (DBG_CMDTIMEOUT)
       {
           printk(KERN_CRIT "ibmasm: DBG_CMDTIMEOUT - Deleting expired command time out timer.\n"); 
       }

   }

   // UNLOCK COMMAND QUEUE
   spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

   // IF THERE WAS A TIMED-OUT COMMAND
   if (pDriverNode->pActiveCmdBlk->completionCode == DDERR_RCV_TIMEOUT)
   {
       // HEART BEATS DO NOT WAIT ON A COMMAND RESPONSE SO JUST DEQUEUE THE COMMAND BLOCK AND RETURN
       if ( pDriverNode->pActiveCmdBlk->PID == PID_HEART_BEAT )
       {
          sysSpDequeue(pDriverNode, pDriverNode->CmdQueue);
       }
       else // THERE WAS A COMMAND ISSUED TO THE SP SO RELEASE THE BLOCKED THREAD THAT WAS WAITING ON A COMMAND RESPONSE
       {
          if (DBG_CMDTIMEOUT)
          {
            printk(KERN_CRIT "ibmasm: DBG_CMDTIMEOUT - sysSpCmdTimeout is unblocking the timed out process.\n"); 
          }

          wake_up(&(pDriverNode->pActiveCmdBlk->pWaitQueueRun));	
       }
   }

   return;
}

// *******************************************************************************
// * sysSpRunSpCmd() - This function checks whether there's a command block on   *
// *                   the command queue, if so, it unblocks it for execution.   *
// *                                                                             *
// * Parameters:                                                                 *
// *******************************************************************************
void sysSpRunSpCmd(PDRIVER_INFO pDriverNode)
{
    PCMD_BLOCK pCmdBlk = NULL;
    unsigned long flags = 0;

    spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );

    if ( pDriverNode->CmdQueue == NULL )
    {
        spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

        return;
    }

    pCmdBlk = sysSpPeekQueue(pDriverNode->CmdQueue);

    spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

    if ( pCmdBlk )
    {
       if ( pCmdBlk->PID == PID_HEART_BEAT )
       {
           if (pDriverNode->pHB_Block->completionCode == DDERR_CMD_INITIATED )  // PHR_MULTI_2
           {
               if (DBG_HB)
               {
                  printk(KERN_CRIT "ibmasm: DBG_HB - Sending heart beat to the SP.\n");
               }

               // SEND THE QUEUED HEARTBEAT RESPONSE TO THE SP
               if ( pDriverNode->AsmType == ASMTYPE_WISEMAN || 
                    pDriverNode->AsmType == ASMTYPE_EAGLE   )
               {
                   sysSpProcessPciRequest(pDriverNode, pCmdBlk, CMD_TIMEOUT_45);
               }
               #ifndef IA64      // PHR_180930
               else
               {
                   sysSpProcessIsaRequest(pCmdBlk, CMD_TIMEOUT_45);
               }
               #endif            // PHR_180930 
           } // end if - DDERR_CMD_INITIATED
       }
       else if (pCmdBlk->completionCode == DDERR_CMD_INITIATED )
       {
          // RELEASE THE BLOCKED THREAD WAITING FOR A COMMAND RESPONSE
          wake_up(&(pCmdBlk->pWaitQueueRun));                        
       }
    }
    return;
}

//**************************************************************************
// *  Start of PHR_181239                                                  *  
// *                                                                       *
// * sysSpRevHbTimeout() - This function is registered as a callback for   *
// *                       the SP reverse heart beat timer, and is called  *
// *                       when it expires. It simply releases the blocked *
// *                       Reverse Heart Beat thread which sends a reverse *
// *                       heart beat to the SP and waits for the timeout  *
// *                       to occur again until a degegister is received   *
// *                                                                       *
// * Parameters:                                                           *
// *       pDriverNode - Pointer to the driver structure which contains a  *
// *                     pointer to the corresponding/active command blk.  *
// *************************************************************************
void sysSpRevHbTimeout(PDRIVER_INFO pDriverNode)
{
    // WAKE UP THE BLOCKED REVERSE HEART BEAT THREAD
    wake_up_interruptible(&(pDriverNode->pActiveRHBBlk->pWaitQueueRun));	

    return;
}

#undef UwsUtil_c


