/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#include <linux/version.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,4)
#include <linux/slab.h>
#else
#include <linux/malloc.h>
#endif
#include <linux/string.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/blk.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/unistd.h>
#include <linux/timex.h>
#include <linux/timer.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/irq.h>

#include "fc_os.h"
#include "fc_hw.h"
#include "fc.h"
#include "dfc.h"
#include "fcdiag.h"
#include "fcmsg.h"
#include "fc_crtn.h"      
#include "fc_ertn.h"     


#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
#include <linux/spinlock.h>
#include <linux/rtnetlink.h>
#else
#include <asm/spinlock.h>
#endif
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <asm/byteorder.h>
#include <asm/uaccess.h>

#ifdef powerpc
#include <asm/pci_dma.h>

#ifdef NO_TCE
#define INVALID_PHYS       NO_TCE
#else
#define INVALID_PHYS       0
#endif

#else
#define INVALID_PHYS       0
#endif

#define is_invalid_phys(addr)   ((addr) == (void *)((ulong)INVALID_PHYS))

static long IOcnt = 0;
static long lpfcdiag_cnt = 0;

#define LPFC_DRIVER_VERSION "1.23a"
_static_ char *lpfc_release_version = LPFC_DRIVER_VERSION;

/* Declare memory for global structure that is used to access
 * per adapter specific info.c
 */
_static_ fc_dd_ctl_t DD_CTL;
_static_ spinlock_t lpfc_smp_lock;
_static_ struct watchdog lpfc_clktimer;
_static_ int lpfc_initTimer = 0;
_static_ int lpfc_one_cpu = 1;   /* Just bind DPC to CPU 0 */
_static_ int lpfc_use_hostptr = 0;

_static_ spinlock_t lpfc_q_lock[MAX_FC_BRDS];
_static_ spinlock_t lpfc_mempool_lock[MAX_FC_BRDS];

struct lpfc_dpc {
   struct task_struct  *dpc_handler;    /* kernel thread */
   struct semaphore    *dpc_wait;       /* DPC waits on this semaphore */
   struct semaphore    *dpc_notify;     /* requester waits for DPC on sem */
   int                  dpc_active;     /* DPC routine is active */
   int                  dpc_ticks;      /* DPC routine current tick count */
   struct semaphore     dpc_sem;
} lpfc_dpc[MAX_FC_BRDS];

_static_  int       lpfc_dpc_timer = 0;
 
_forward_ void      lpfc_timer(void *p);
_forward_ int       do_fc_timer(fc_dev_ctl_t *p_dev_ctl);
_forward_ void      lpfc_do_dpc(void *p);
_forward_ int       fc_dpc_lstchk(fc_dev_ctl_t *p_dev_ctl, Scsi_Cmnd *Cmnd);

/* Binding Definitions: Max string size  */
#define FC_MAX_DID_STRING       6
#define FC_MAX_WW_NN_PN_STRING 16

int lpfcMallocCnt = 0;
int lpfcMallocByte = 0;
int lpfcFreeCnt = 0;
int lpfcFreeByte = 0;

/* This defines memory for the common configuration parameters */
#define DEF_ICFG  1  
#include "fcfgparm.h"

#define SHUTDOWN_SIGS   (sigmask(SIGKILL)|sigmask(SIGINT)|sigmask(SIGTERM))

#ifdef MODULE

#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif

#include <linux/module.h>

MODULE_PARM(lpfc_vendor, "i");
MODULE_PARM(lpfc_bind_entries, "i");
MODULE_PARM(lpfc_fcp_bind_WWPN, "1-" __MODULE_STRING(MAX_FC_BINDINGS) "s");
MODULE_PARM(lpfc_fcp_bind_WWNN, "1-" __MODULE_STRING(MAX_FC_BINDINGS) "s");
MODULE_PARM(lpfc_fcp_bind_DID, "1-" __MODULE_STRING(MAX_FC_BINDINGS) "s");

MODULE_PARM(lpfc_lun0_missing, "i");
MODULE_PARM(lpfc_lun_skip, "i");
MODULE_PARM(lpfc_use_removable, "i");
MODULE_PARM(lpfc_max_lun, "i");
MODULE_PARM(lpfc_use_data_direction, "i");


#ifndef FC_NEW_EH
int lpfc_reset(Scsi_Cmnd *, unsigned int);
int fc_proc_info( char *, char **, off_t, int, int, int);
#endif
int fc_abort(Scsi_Cmnd *);
int fc_reset_device(Scsi_Cmnd *);
int fc_reset_bus(Scsi_Cmnd *);
int fc_reset_host(Scsi_Cmnd *);
int fc_queuecommand(Scsi_Cmnd *, void (*done)(Scsi_Cmnd *));
void fc_queue_done_cmd(fc_dev_ctl_t  * , struct buf *);
void fc_flush_done_cmds(fc_dev_ctl_t *, ulong);
void lpfc_nodev(unsigned long);
void local_timeout(unsigned long data);
void do_fc_intr_handler(int , void *, struct pt_regs *);
int do_fc_intr(struct intr *);
void * lpfc_kmalloc( unsigned int, unsigned int, void **, fc_dev_ctl_t *);
void lpfc_kfree( unsigned int, void *, void *, fc_dev_ctl_t *);

EXPORT_SYMBOL(fc_abort);
EXPORT_SYMBOL(fc_reset_device);
EXPORT_SYMBOL(fc_reset_bus);
EXPORT_SYMBOL(fc_reset_host);
EXPORT_SYMBOL(local_timeout);
EXPORT_SYMBOL(do_fc_intr_handler);
EXPORT_SYMBOL(fc_queuecommand);
EXPORT_SYMBOL(fc_queue_done_cmd);
EXPORT_SYMBOL(fc_flush_done_cmds);
EXPORT_SYMBOL(do_fc_intr);
#else  /* MODULE */
#ifndef FC_NEW_EH
int fc_reset_device(Scsi_Cmnd *);
int fc_reset_bus(Scsi_Cmnd *);
int fc_reset_host(Scsi_Cmnd *);
#endif
void local_timeout(unsigned long data);
void do_fc_intr_handler(int , void *, struct pt_regs *);
int do_fc_intr(struct intr *);
void * lpfc_kmalloc( unsigned int, unsigned int, void **, fc_dev_ctl_t *);
void lpfc_kfree( unsigned int, void *, void *, fc_dev_ctl_t *);
extern int lpfn_probe(void);
static int  lpfc_detect_called = 0;
#endif /* MODULE */
int do_fc_abort(fc_dev_ctl_t *);
int do_fc_reset_device(fc_dev_ctl_t *);
int do_fc_reset_bus(fc_dev_ctl_t *);
int do_fc_reset_host(fc_dev_ctl_t *);
int do_fc_queuecommand(fc_dev_ctl_t *, ulong);
void fc_select_queue_depth(struct Scsi_Host *, Scsi_Device *);
int fc_device_queue_depth(fc_dev_ctl_t *, Scsi_Device *);
int fc_DetectInstance(int, struct pci_dev *pdev, uint, Scsi_Host_Template *);

#include "lpfc.conf.defs"

#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
#ifdef MODULE
Scsi_Host_Template driver_template = EMULEXFC;
#include "scsi_module.c"
#endif
#else   /* new kernel scsi initialization scheme */
static  Scsi_Host_Template driver_template = EMULEXFC;
#include "scsi_module.c"
#endif

#ifndef __GENKSYMS__
#include "fcmsgcom.c"
extern char fwrevision[32];

_local_ int         lpfcdfc_init(void);
_local_ int         fc_rtalloc(fc_dev_ctl_t *, struct dev_info *);
_local_ int         fc_bind_wwpn(fc_dev_ctl_t  *, char **, u_int );
_local_ int         fc_bind_wwnn(fc_dev_ctl_t  *, char **, u_int );
_local_ int         fc_bind_did(fc_dev_ctl_t  *, char **, u_int );
_local_ dvi_t      *fc_getDVI(fc_dev_ctl_t  *, int, fc_lun_t);
_local_ ulong       fc_po2(ulong);
_local_ int         linux_attach(int, Scsi_Host_Template *, struct pci_dev *);
_local_ int         lpfc_find_cmd( fc_dev_ctl_t *p_dev_ctl, Scsi_Cmnd *cmnd);
_local_  void       deviFree(fc_dev_ctl_t *, dvi_t *, node_t *);
#ifdef MODULE
_local_ int         linux_detach(int );
#endif /* MODULE */
_local_ void        *fc_kmem_zalloc(unsigned int );

extern int dfc_ioctl( struct dfccmdinfo *infop, struct cmd_input *cip);

int lpfcdiag_ioctl(struct inode * inode, struct file * file,
          unsigned int cmd, unsigned long arg);
int lpfcdiag_open(struct inode * inode, struct file * file);
int lpfcdiag_release(struct inode * inode, struct file * file);
int fc_ioctl(int , void *);

static struct file_operations lpfc_fops = {
    ioctl:      lpfcdiag_ioctl,
        open:       lpfcdiag_open,
        release:    lpfcdiag_release,
};

static int lpfc_major = 0;

/* If we want to define a new entry for Emulex boards*/
/* #define PROC_SCSI_EMULEXFC PROC_SCSI_FILE+1 */
/* For now we use the FC entry */
#define NAMEEMULEX    "lpfc"
#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
static struct proc_dir_entry proc_scsi_emulex  = {
    PROC_SCSI_FCAL , 4, "lpfc",
    S_IFDIR | S_IRUGO | S_IXUGO, 2
};
#endif

struct dfc {
   uint32               dfc_init;
   uint32               filler;
   uchar                bufout[sizeof(FC_BRD_INFO)];
   struct dfc_info      dfc_info[MAX_FC_BRDS];
};
extern struct dfc dfc;

/*Extra configuration parameters as defined in lpfc.conf.c*/
extern int  lpfc_vendor;
extern int  lpfc_bind_entries;
extern char *lpfc_fcp_bind_WWPN[];
extern char *lpfc_fcp_bind_WWNN[];
extern char *lpfc_fcp_bind_DID[];
extern int  lpfc_lun0_missing;
extern int  lpfc_lun_skip;
extern int  lpfc_use_removable;
extern int  lpfc_max_lun;
extern int  lpfc_use_data_direction;

/*Other configuration parameters, not available to user*/
static int  lpfc_pci_latency_clocks  =0;
static int  lpfc_pci_cache_line  =0;

/*Other configuration parameters, not available to user*/
static int  lpfc_mtu = 4032;        /* define IP max MTU size */
static int  lpfc_intr_ack  = 1;
static int  lpfc_first_check  = 1;
static int  lpfc_zone_rscn = 1;
static int  lpfc_qfull_retry = 5;

int  lpfc_nethdr = 1;
int  lpfc_driver_unloading = 0;

/* The size of a physical memory page */
uint32 fcPAGESIZE = 4096; /*PAGE_SIZE;*/

/* Can be used to map driver instance number and hardware adapter number */
int fcinstance[MAX_FC_BRDS];
int fcinstcnt = 0;

/* Current driver state for diagnostic mode, online / offline, see fcdiag.h */
uint32 fc_diag_state;
uint32   fc_dbg_flag = 0;
#define FC_MAX_SEGSZ 4096

#define FC_MAX_POOL  1024
struct fc_mem_pool {
   void *p_virt;
   void *p_phys;
   ushort p_refcnt;
   ushort p_left;
};
struct fc_mem_pool *fc_mem_dmapool[MAX_FC_BRDS];
int fc_idx_dmapool[MAX_FC_BRDS];
int fc_size_dmapool[MAX_FC_BRDS];

#define ScsiResult(host_code, scsi_code) (((host_code) << 16) | scsi_code)

#define ZERO_PAN 0

_static_ unsigned int   lpfc_page_mask;

/* Used in generating timeouts for timers */ 
_static_ uint32 fc_scsi_abort_timeout_ticks;
_static_ uint32 fc_ticks_per_second;

/* Can be used to map driver instance number and hardware adapter number */
extern int fcinstance[];
extern int fcinstcnt;

/* Current driver state for diagnostic mode, online / offline, see fcdiag.h */
extern uint32 fc_diag_state;

extern int      fc_check_for_vpd;
extern int      fc_reset_on_attach;
extern int      fc_max_ns_retry;
extern int      fc_fdmi_on;
extern int      fc_max_els_sent;



void lpfc_scsi_add_timer(Scsi_Cmnd *, int);
int lpfc_scsi_delete_timer(Scsi_Cmnd *);

#ifdef powerpc 
#if LINUX_VERSION_CODE > LinuxVersionCode(2,4,14) 
#define NO_BCOPY 1
#endif
#endif

#ifndef FC_NEW_EH
/******************************************************************************
* Function name : fc_proc_info
*
* Description   : 
* 
******************************************************************************/
int fc_proc_info(char *buffer, 
                 char **start, 
                 off_t offset, 
                 int length,
                 int hostno, 
                 int inout)
{
   return(0);
}
#endif

#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0) 
/******************************************************************************
* Function name : fc_pci_alloc_consistent
*
* Description   : 
* 
******************************************************************************/
void * fc_pci_alloc_consistent(struct pci_dev *hwdev,
                               size_t size,
                               dma_addr_t *dma_handle)
{
  void *virt_ptr;
  u_long  a_size;
  int     order;

  if ((size % PAGE_SIZE) == 0) {
    for (order = 0, a_size = PAGE_SIZE;
         a_size < size; order++, a_size <<= 1);
    virt_ptr = (void *) __get_free_pages(GFP_ATOMIC, order);
  } 
  else{
    a_size = fc_po2(size);
    if(a_size == 256)
      a_size = 512;
    virt_ptr = kmalloc(a_size, GFP_KERNEL);
  }
  *dma_handle = virt_to_bus(virt_ptr);
  return virt_ptr;
}

/******************************************************************************
* Function name : fc_pci_free_consistent
*
* Description   : 
* 
******************************************************************************/
void fc_pci_free_consistent(struct pci_dev *hwdev,
                            size_t          size,
                            void           *virt_ptr,
                            dma_addr_t      dma_handle)
{
  u_long  a_size;
  int     order;

  if(!virt_ptr)
    return;

  /*
   * Check which method was used to allocate the memory
   */
  if ((size % PAGE_SIZE) == 0) {
    for (order = 0, a_size = PAGE_SIZE;
         a_size < size; order++, a_size <<= 1)
      ;
    free_pages((unsigned long)virt_ptr, order);
  } 
  else{
    kfree(virt_ptr);
  }
}
#else
/******************************************************************************
* Function name : fc_pci_dma_sync_single
*
* Description   : 
* 
******************************************************************************/
void fc_pci_dma_sync_single(struct pci_dev *hwdev, 
                            dma_addr_t      h,
                            size_t          size, 
                            int             c)
{
         pci_dma_sync_single(hwdev, h, 4096, c);
}
#endif

#if defined (MODULE) || defined (NO_BCOPY)
/******************************************************************************
* Function name : bcopy
*
* Description   : kernel-space to kernel-space copy
* 
******************************************************************************/
_static_ void bcopy(void   *src, 
                    void   *dest,
                    size_t  n)
{
  memcpy(dest, src, n);
}
#else
/******************************************************************************
* Function name : bcopy
*
* Description   : kernel-space to kernel-space copy
* 
******************************************************************************/
_static_ void bcopy(void   *src, void   *dest, size_t  n);

#endif /* MODULE or NO_BCOPY */

/******************************************************************************
* Function name : lpfc_DELAYMS
*
* Description   : Called to delay cnt ms 
* 
******************************************************************************/
_static_ int lpfc_DELAYMS(fc_dev_ctl_t *new_dev_ctl,
                          int           cnt)
{
  int i;
  fc_dev_ctl_t *p_dev_ctl;
  FC_BRD_INFO  *binfo;
  struct lpfc_dpc *ldp;

  for (i = 0; i < MAX_FC_BRDS; i++) {
    if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
       if(new_dev_ctl == p_dev_ctl)
          continue;
       binfo = &BINFO;
       ldp = &lpfc_dpc[binfo->fc_brd_no];
       if ((ldp->dpc_active == 0) && ldp->dpc_wait)
          up(ldp->dpc_wait);
    }
  }
  if(new_dev_ctl->info.fc_ffstate != FC_INIT_START) {
     barrier();
     schedule();
  }
  mdelay(cnt);
  for (i = 0; i < MAX_FC_BRDS; i++) {
    if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
       if(new_dev_ctl == p_dev_ctl)
          continue;
       binfo = &BINFO;
       ldp = &lpfc_dpc[binfo->fc_brd_no];
       if ((ldp->dpc_active == 0) && ldp->dpc_wait)
          up(ldp->dpc_wait);
    }
  }
  if(new_dev_ctl->info.fc_ffstate != FC_INIT_START) {
     barrier();
     schedule();
  }
  return(0);
}

/******************************************************************************
* Function name : kmem_alloc
*
* Description   : Kernel memory alloc and free
* 
******************************************************************************/
_static_ void *fc_kmem_alloc(unsigned int size)
{
  void *ptr;
  lpfcMallocCnt++;
  lpfcMallocByte += size;
  ptr = lpfc_kmalloc(size, GFP_ATOMIC, 0, 0);
  return ptr;
 
}

/******************************************************************************
* Function name : fc_kmem_free
*
* Description   : 
* 
******************************************************************************/
_static_ void fc_kmem_free(void         *obj,
                           unsigned int  size)
{
  lpfcFreeCnt++;
  lpfcFreeByte += size;
  if(obj)
    lpfc_kfree(size, obj, (void *)((ulong)INVALID_PHYS), 0);
}

/******************************************************************************
* Function name : fc_kmem_zalloc
*
* Description   : allocate memory and initialize to zeros
* 
******************************************************************************/
_static_ void *fc_kmem_zalloc(unsigned int size)
{
  void *ptr = fc_kmem_alloc(size);
  if(ptr)
    fc_bzero(ptr,size);
  return ptr;
}

/******************************************************************************
* Function name : dfc_disable_lock
*
* Description   : 
* 
******************************************************************************/
_static_ ulong dfc_disable_lock(ulong        p1,
                                Simple_lock *p2)

{
   ulong iflg;

   iflg = 0;
   spin_lock_irqsave(&lpfc_smp_lock, iflg);
   return(iflg);
}

/******************************************************************************
* Function name : dfc_unlock_enable
*
* Description   : 
* 
******************************************************************************/
_static_ void dfc_unlock_enable(ulong        p1,
                                Simple_lock *p2)
{
   ulong iflg;

   iflg = p1;
   spin_unlock_irqrestore(&lpfc_smp_lock, iflg);
   return;
}

_static_ ulong lpfc_q_disable_lock(fc_dev_ctl_t *p_dev_ctl)
{
   ulong iflg;

   iflg = 0;
   spin_lock_irqsave(&lpfc_q_lock[p_dev_ctl->info.fc_brd_no], iflg);
   return(iflg);
}


_static_ void lpfc_q_unlock_enable(fc_dev_ctl_t *p_dev_ctl, ulong p1)
{
   ulong iflg;

   iflg = p1;
   spin_unlock_irqrestore(&lpfc_q_lock[p_dev_ctl->info.fc_brd_no], iflg);
   return;
}

_static_ ulong lpfc_mempool_disable_lock(fc_dev_ctl_t *p_dev_ctl)
{
   ulong iflg;

   iflg = 0;
   spin_lock_irqsave(&lpfc_mempool_lock[p_dev_ctl->info.fc_brd_no], iflg);
   return(iflg);
}


_static_ void lpfc_mempool_unlock_enable(fc_dev_ctl_t *p_dev_ctl, ulong p1)
{
   ulong iflg;

   iflg = p1;
   spin_unlock_irqrestore(&lpfc_mempool_lock[p_dev_ctl->info.fc_brd_no], iflg);
   return;
}

/******************************************************************************
* Function name : fc_flush_done_cmds
*
* Description   : flush all done commands at once
* 
******************************************************************************/
void fc_flush_done_cmds(fc_dev_ctl_t *p_dev_ctl, 
                        ulong         siflg)
{
  int count, first_inq;
  Scsi_Cmnd *cmd;
  struct buf * head;
  FC_BRD_INFO  *binfo;
  struct dev_info   *devp;
  struct sc_buf  *sp;
  uint32 *iptr;
  ulong iflg;
  
  iflg = 0;
  LPFC_LOCK_DRIVER(1);

  head = p_dev_ctl->iodone_head;
  binfo = &BINFO;
  count = 0;
  while(head) {
    count++;
    cmd = head->cmnd;
    devp = ((struct sc_buf *)head)->current_devp;
    head=head->av_forw;

    if(devp)
      devp->iodonecnt++;
    else
      panic("NULL devp in flush_done\n");

    if(cmd && (cmd->scsi_done != NULL)) {
      sp = (struct sc_buf *)cmd->host_scribble;
      if (!sp) {
         /* NULL sp in flush_done */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0708,                   /* ptr to msg structure */
                 fc_mes0708,                      /* ptr to msg */
                  fc_msgBlk0708.msgPreambleStr,   /* begin varargs */
                   cmd->cmnd[0],
                    cmd->serial_number,
                     cmd->retries,
                      cmd->result);               /* end varargs */
         continue;
      }

      FCSTATCTR.fcpRsvd1++;

      if(devp->scp) {
         sp->bufstruct.av_forw = devp->scp;
         devp->scp = sp;
      }
      else {
         devp->scp = sp;
         devp->scp->bufstruct.av_forw = 0;
      }
      devp->scpcnt++;
      cmd->host_scribble = 0;

      first_inq = 0;
      if(devp->first_check & FIRST_IO) {
         uchar *buf;
         if(cmd->cmnd[0] == FCP_SCSI_INQUIRY) {
            buf = (uchar *)cmd->request_buffer;
            if((cmd->result) ||
               ((*buf & 0x70) != 0)) {          /* lun not there */
#ifdef FREE_LUN
               deviFree(p_dev_ctl, devp, devp->nodep);
#else
               devp->first_check &= ~FIRST_IO;
#endif
            } else {
               devp->first_check &= ~FIRST_IO;
            }
            first_inq = 1;
          }
       }


      LPFC_UNLOCK_DRIVER;
      iptr = (uint32 *)&cmd->sense_buffer[0];
      if((cmd->result) || *iptr) {
         devp->errorcnt++;
         /* iodone error return */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                 &fc_msgBlk0710,                   /* ptr to msg structure */
                  fc_mes0710,                      /* ptr to msg */
                   fc_msgBlk0710.msgPreambleStr,   /* begin varargs */
                    (uint32)((cmd->target << 16) | cmd->lun),
                     (uint32)((cmd->retries << 16 ) | cmd->cmnd[0]),
                       cmd->result,
                        *iptr);                    /* end varargs */
      }

      lpfc_scsi_add_timer(cmd, cmd->timeout_per_command);
      cmd->scsi_done(cmd);
      iflg = 0;
      LPFC_LOCK_DRIVER(2);
    }
    else
      panic("Cmnd in done queue without scsi_done\n");
  }
  p_dev_ctl->iodone_head = 0;
  p_dev_ctl->iodone_list = 0;
  LPFC_UNLOCK_DRIVER;
  return;
}

/******************************************************************************
* Function name : fc_queue_done_cmd
*
* Description   : add done command to a queue to be flushed later
* 
******************************************************************************/
void fc_queue_done_cmd(fc_dev_ctl_t  *p_dev_ctl, 
                       struct buf    *sb)
{
  struct sc_buf     *sp;

  if(p_dev_ctl->iodone_head == NULL) {
    p_dev_ctl->iodone_head = sb;
    p_dev_ctl->iodone_list = sb;
  } else {
    p_dev_ctl->iodone_list->av_forw = sb;
    p_dev_ctl->iodone_list = sb;
  }
  sb->av_forw = NULL;

  sp = (struct sc_buf *)sb;
  if (sp->cmd_flag & FLAG_ABORT)
      sp->cmd_flag &= ~FLAG_ABORT;
}

/******************************************************************************
* Function name : remap_pci_mem
*
* Description   : remap pci memory, makes sure mapped memory is page-aligned
* 
******************************************************************************/
_local_ void * remap_pci_mem(u_long base, 
                                   u_long size)
{
#ifdef powerpc
  return (ioremap (base, size));
#else  
  u_long page_base = ((u_long) base)& PAGE_MASK;
  u_long page_offs = ((u_long) base) - page_base;
  u_long page_remapped  = (u_long) ioremap(page_base, page_offs+size);
  return (void *) (page_remapped? (page_remapped + page_offs) : ((ulong)-1));
#endif
}

/******************************************************************************
* Function name : unmap_pci_mem
*
* Description   : unmap pci memory
* 
******************************************************************************/
_local_ void  unmap_pci_mem(u_long vaddr)
{
  if (vaddr) {
  }
}

#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
/******************************************************************************
* Function name : pci_getadd
*
* Description   : get address from a pci register, accounts for 64 bit addresses
*                 returns next register 
* 
******************************************************************************/
_local_ int  pci_getadd(struct pci_dev *pdev, 
                              int             reg, 
                              u_long         *base)

{
    *base = pci_resource_start(pdev, reg);
        reg++;
    return ++reg;
}
#else
/******************************************************************************
* Function name : pci_getadd
*
* Description   : get address from a pci register, accounts for 64 bit addresses
*                 returns next register 
* 
******************************************************************************/
_local_ int  pci_getadd(struct pci_dev *pdev, 
                              int             reg, 
                              u_long         *base)
{
  *base = pdev->base_address[reg++];
  if ((*base & 0x7) == 0x4) { 
#if BITS_PER_LONG > 32
    *base |= (((u_long)pdev->base_address[reg]) << 32);
#endif
    ++reg;
  }
  return reg;
}
#endif

/******************************************************************************
* Function name : fc_DetectInstance 
*
* Description	:  
* 
******************************************************************************/
int fc_DetectInstance( int instance, 
		    struct pci_dev *pdev, 
		    uint type,
		    Scsi_Host_Template *tmpt)
{

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
   /* PCI_SUBSYSTEM_IDS supported */ 
   while ((pdev = pci_find_subsys(PCI_VENDOR_ID_EMULEX, type,
       PCI_ANY_ID, PCI_ANY_ID, pdev) )) 
   {
      if (pci_enable_device(pdev)) continue;
#else
   while ((pdev = pci_find_device(PCI_VENDOR_ID_EMULEX, type,
       pdev))) 
   {
#endif
        if(linux_attach(instance, tmpt, pdev) )
	  continue;
        instance++;
   }
  
  return(instance);
}

/******************************************************************************
* Function name : fc_detect
*
* Description   : Mid-level driver entry function for detecting the boards
*                 Also provides some initialization
* 
******************************************************************************/
int  fc_detect(Scsi_Host_Template *tmpt)
{
#define WAIT_4_FC_READY      200   /* Thats 200 * 25 ms = 5 sec */
#define MSEC_25_DELAY         25
#define PRE_FC_READY_DELAY    40
#define POST_FC_READY_DELAY   60

  int            instance = 0;
  struct pci_dev *pdev = NULL;
  fc_dev_ctl_t   *p_dev_ctl;
  FC_BRD_INFO    *binfo;
  int            i, j, cnt;
  /* To add another, add 1 to number of elements, add a line
   * sType[x] = id, leave last sType[x+1] = 0;
   */
  uint sType [8];

  sType[0] =  PCI_DEVICE_ID_THOR;
  sType[1] =  PCI_DEVICE_ID_SUPERFLY;
  sType[2] =  PCI_DEVICE_ID_PEGASUS;
  sType[3] =  PCI_DEVICE_ID_PFLY; 
  sType[4] =  PCI_DEVICE_ID_CENTAUR;
  sType[5] =  PCI_DEVICE_ID_DRAGONFLY;
  sType[6] =  PCI_DEVICE_ID_TFLY; 
  /* sType[x] = PCI_DEVICE_ID_XXX; */
  sType[7] = 0;

  /*
   * Intialization
   */
  lpfc_page_mask = ((unsigned int) ~(fcPAGESIZE - 1));
  fc_ticks_per_second = HZ;
  fc_scsi_abort_timeout_ticks = 300 * HZ /*CLOCK_TICK_RATE*/ ;
  fc_bzero(&DD_CTL, sizeof(fc_dd_ctl_t));
  for (i = 0; i < MAX_FC_BRDS; i++) {
    DD_CTL.p_dev[i] = NULL;
    DD_CTL.p_config[i] = NULL;
    fcinstance[i] = -1;
  }
  DD_CTL.num_devs = 0;

  fc_check_for_vpd = 1;   /* issue dump mbox command during HBA initialization
                           * to check VPD data (if any) for a Serial Number */
  fc_reset_on_attach = 0; /* Always reset HBA before initialization in attach */
  fc_fdmi_on = 0;         /* Enable FDMI */
  fc_max_ns_retry = 3;    /* max number of retries for NameServer CT requests
                           * during discovery. */

  fc_max_els_sent = 1;
  if(fc_max_els_sent > NLP_MAXREQ) 
     fc_max_els_sent = NLP_MAXREQ; 

#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
  tmpt->proc_dir = &proc_scsi_emulex;
#else
  tmpt->proc_name = NAMEEMULEX;
#endif

  printk("Emulex LightPulse FC SCSI/IP %s\n", lpfc_release_version);
  /*
   * the mid-level clears interrupts
   * no need to re-intialize pdev 
   */
  i = 0;
   while(sType[i])
   {
         instance = fc_DetectInstance(instance, pdev, sType[i], tmpt);
	 i++;
   }

  if(instance) {
    lpfcdfc_init();  /* Initialize diagnostic interface */
  }

  p_dev_ctl = (fc_dev_ctl_t *)NULL; /* Prevent compiler warning */
  if( (PRE_FC_READY_DELAY > 0) &&
        (instance > 0) &&
          (p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[0])) {
     binfo = &BINFO;
     for( i=0; i<PRE_FC_READY_DELAY; i++) {
        lpfc_DELAYMS( p_dev_ctl, MSEC_25_DELAY);  /* 25 millisec */
     }
  }

  for(j = 0;j < WAIT_4_FC_READY; j++) {
     cnt = 0;
     for (i = 0; i < instance; i++) {
        if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
           binfo = &BINFO;
           if((binfo->fc_ffstate >= FC_LINK_UP) && (binfo->fc_ffstate != FC_READY)) {
              cnt++;
           }
        }
     }
     if(cnt) {
        /* HBA(s) not FC_READY yet */
        lpfc_DELAYMS( p_dev_ctl, MSEC_25_DELAY);  /* 25 millisec */
        continue;
     }
     break;
  }

  /* There are cases where the HBAs are FC_READY but not all FC nodes 
   * have completed their FC PLOGI/PRLI sequence due to collisions. The
   * following delay loop provides a chance for these noded to complete
   * their FC PLOGI/PRLI sequence prior to allowing the SCSI layer to 
   * start up upon the return from this routine.
   */

  if( (POST_FC_READY_DELAY > 0) &&
        (instance > 0) &&
          (p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[0])) {
     binfo = &BINFO;
     for( i=0; i<POST_FC_READY_DELAY; i++) {
        lpfc_DELAYMS( p_dev_ctl, MSEC_25_DELAY);  /* 25 millisec */
     }
  }

#ifndef MODULE
  if(lpfc_detect_called == 2) {
     lpfc_detect_called = 1;
     lpfn_probe();
  }
  else
     lpfc_detect_called = 1;
#endif /* MODULE */
  if (instance == 0)
    printk ("This Open Source driver didn't detect any suitable Emulex Hardware\n");  
  return instance;
}

/******************************************************************************
* Function name : i_init
*
* Description   : Called from fc_attach to add interrupt vector for adapter 
* 
******************************************************************************/
int  i_init(struct intr *ihs)
{
  struct pci_dev  *pdev;
  fc_dev_ctl_t    *p_dev_ctl;

  p_dev_ctl = (fc_dev_ctl_t * )ihs; /* Since struct intr is at beginning */

  /*
   * Get PCI for this board
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev) 
    return(INTR_FAIL);
  if (request_irq(pdev->irq, do_fc_intr_handler, SA_INTERRUPT | SA_SHIRQ,
          "lpfcdd", (void *)ihs))
    return(INTR_FAIL);
  return(INTR_SUCC);
}

/******************************************************************************
* Function name : i_clear
*
* Description   : Called from fc_detach to remove interrupt vector for adapter
* 
******************************************************************************/
_static_ int i_clear(struct intr *ihs)
{
  struct pci_dev  *pdev;
  fc_dev_ctl_t    *p_dev_ctl;

  p_dev_ctl = (fc_dev_ctl_t * )ihs; /* Since struct intr is at beginning */

  /*
   * Get PCI for this board
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev) 
    return(1);
  free_irq(pdev->irq, p_dev_ctl);
  p_dev_ctl->intr_inited=0;
  return(0);
}

/******************************************************************************
* Function name : linux_attach
*
* Description   : LINUX initialization entry point, called from environment
*                 to attach to / initialize a specific adapter.
* 
******************************************************************************/
_local_  int  linux_attach(int                 instance, 
                                 Scsi_Host_Template *tmpt,
                                 struct pci_dev     *pdev)
{
  struct Scsi_Host *host;
  fc_dev_ctl_t *p_dev_ctl=NULL;
  FC_BRD_INFO  *binfo;
  FCCLOCK_INFO *clock_info;
  iCfgParam    *clp=NULL;
  int           initTimer = 0;
  ulong         iflg;
 
  /*
   * must have a valid pci_dev
   */
  if(!pdev)
    return (1);

  /* Allocate memory to manage HBA dma pool */
  fc_mem_dmapool[instance] = kmalloc((sizeof(struct fc_mem_pool) * FC_MAX_POOL),
    GFP_ATOMIC);
  if(fc_mem_dmapool[instance] == 0)
     return(1);

  fc_bzero((void *)fc_mem_dmapool[instance],
    (sizeof(struct fc_mem_pool) * FC_MAX_POOL));
  fc_idx_dmapool[instance] = 0;
  fc_size_dmapool[instance] = FC_MAX_POOL;

  /* 
   * Allocate space for adapter info structure
   */
  if (!(p_dev_ctl = (fc_dev_ctl_t *)fc_kmem_zalloc(sizeof(fc_dev_ctl_t)))) {
    return (1);
  }
  /* 
   * Allocate space for configuration parameters
   */
  if (!(clp = (iCfgParam *)fc_kmem_zalloc(sizeof(icfgparam)))) {
    goto fail1;
  }

  p_dev_ctl->pcidev = pdev;
  p_dev_ctl->sid_cnt = 0; /* Start scsid assignment at 1 */
  binfo = &BINFO;
  binfo->fc_brd_no = instance;
  spin_lock_init(&lpfc_q_lock[instance]);
  spin_lock_init(&lpfc_mempool_lock[instance]);

  if(lpfc_use_hostptr)
       binfo->fc_busflag = FC_HOSTPTR;       
#ifdef powerpc
  binfo->fc_busflag = FC_HOSTPTR;
#endif

  binfo->fc_p_dev_ctl = (uchar * )p_dev_ctl;
  DD_CTL.p_dev[instance] = p_dev_ctl;
  DD_CTL.p_config[instance] = clp;
  fcinstance[instance] = instance;

  /* 
   * Initialize config parameters
   */
  bcopy((void * )&icfgparam, (void *)clp, sizeof(icfgparam));

  /*
   * Initialize locks, and timeout functions
   */
  clock_info = &DD_CTL.fc_clock_info;
  CLOCKWDT = (void *)&lpfc_clktimer;
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
  init_waitqueue_head(&p_dev_ctl->linkwq);
  init_waitqueue_head(&p_dev_ctl->rscnwq);
  init_waitqueue_head(&p_dev_ctl->ctwq);
#endif


  initTimer = 0;
  if(lpfc_initTimer == 0) {
    LPFC_INIT_LOCK_DRIVER;  /* Just one global lock for driver */
    fc_clock_init();
    ((struct watchdog *)(CLOCKWDT))->func = fc_timer;
    ((struct watchdog *)(CLOCKWDT))->restart = 1;
    ((struct watchdog *)(CLOCKWDT))->count = 0;
    ((struct watchdog *)(CLOCKWDT))->stopping = 0;
    ((struct watchdog *)(CLOCKWDT))->timeout_id = 1;
    /* 
     * add our watchdog timer routine to kernel's list
     */
    ((struct watchdog *)(CLOCKWDT))->timer.expires = (uint32)(HZ + jiffies);
    ((struct watchdog *)(CLOCKWDT))->timer.function = local_timeout;
    ((struct watchdog *)(CLOCKWDT))->timer.data = (unsigned long)(CLOCKWDT);
    init_timer(&((struct watchdog *)(CLOCKWDT))->timer);
    add_timer(&((struct watchdog *)(CLOCKWDT))->timer);
    lpfc_initTimer = 1;
    initTimer = 1;
  }

  {
  struct lpfc_dpc *ldp;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
  struct semaphore sem = MUTEX_LOCKED;
#else
  DECLARE_MUTEX_LOCKED(sem);
#endif

  ldp = &lpfc_dpc[instance];
  ldp->dpc_notify = &sem;
  kernel_thread((int (*)(void *))lpfc_do_dpc, (void *) p_dev_ctl, 0);
  /*
   * Now wait for the kernel dpc thread to initialize and go to sleep.
   */
  down(&sem);
  ldp->dpc_notify = NULL;
  }

  p_dev_ctl->intr_inited = 0;
  fcinstcnt++;
  if (fc_attach(instance, (uint32 * )((ulong)instance))) {
    /* 
     * lower level routine will log error
     */
    fcinstcnt--;
    goto fail;
  }

  /* 
   * Register this board
   */
  host = scsi_register(tmpt, sizeof(unsigned long));
#ifdef FC_NEW_EH
  host->hostt->use_new_eh_code = 1;
#endif
  
  /*
   * Adjust the number of id's
   * Although max_id is an int, target id's are unsined chars
   * Do not exceed 255, otherwise the device scan will wrap around
   */
  host->max_id = MAX_FCP_TARGET;
  if(!lpfc_max_lun) {
     host->max_lun = MAX_FCP_LUN+1;
     lpfc_max_lun = MAX_FCP_LUN+1;
  }
  else {
     host->max_lun = lpfc_max_lun;
  }
  host->unique_id = instance;

  /* Adapter ID */
  host->this_id = MAX_FCP_TARGET - 1; 

  /*
   * Starting with 2.4.0 kernel, Linux can support commands longer
   * than 12 bytes. However, scsi_register() always sets it to 12.
   * For it to be useful to the midlayer, we have to set it here.
   */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
  host->max_cmd_len = 16;
#endif

  /*
   * Queue depths per lun
   */
  host->cmd_per_lun = 1;
  host->select_queue_depths = fc_select_queue_depth;

  /*
   * Save a pointer to device control in host and increment board
   */
  host->hostdata[0] = (unsigned long)p_dev_ctl;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,4)
  scsi_set_pci_device(host, pdev);
#endif
  p_dev_ctl->host = host;
  DD_CTL.num_devs++;

  iflg = 0;
  LPFC_LOCK_DRIVER(23);
  /* 
   * Need to start scsi timeout if FCP is turned on 
   * The SCSI timeout watch dog is for all adaptors, so do it once only
   */

  if((SCSI_TMO == 0) && clp[CFG_FCP_ON].a_current) {
    SCSI_TMO = fc_clk_set(0, 5, fc_scsi_timeout, 0, 0);
  }

  /* DQFULL */
  if ((clp[CFG_DQFULL_THROTTLE].a_current) &&
      (clp[CFG_DFT_LUN_Q_DEPTH].a_current > FC_MIN_QFULL)) {
      if ((clp[CFG_DQFULL_THROTTLE_UP_TIME].a_current) &&
          (clp[CFG_DQFULL_THROTTLE_UP_INC].a_current)) {
         fc_clk_set(p_dev_ctl, clp[CFG_DQFULL_THROTTLE_UP_TIME].a_current, 
                 fc_q_depth_up, 0, 0);
      }
  }
  LPFC_UNLOCK_DRIVER;
  return(0);

fail:
  if(initTimer) {
    if(SCSI_TMO) {
      fc_clk_can(0, SCSI_TMO);
      SCSI_TMO = 0;
    }
    clock_info = &DD_CTL.fc_clock_info;
    ((struct watchdog *)(CLOCKWDT))->stopping = 1;
    if (((struct watchdog *)(CLOCKWDT))->timer.function)
      del_timer(&((struct watchdog *)(CLOCKWDT))->timer);
    ((struct watchdog *)(CLOCKWDT))->timer.function=NULL;
    ((struct watchdog *)(CLOCKWDT))->timeout_id=0;
  }

  {
  struct lpfc_dpc   *ldp;
  ldp = &lpfc_dpc[instance];
  if(ldp->dpc_handler != NULL ) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
     struct semaphore sem = MUTEX_LOCKED;
#else
     DECLARE_MUTEX_LOCKED(sem);
#endif

    ldp->dpc_notify = &sem;
    send_sig(SIGKILL, ldp->dpc_handler, 1);
    down(&sem);
    ldp->dpc_notify = NULL;
  }
  }
  /* 
   * Free up any allocated resources 
   */
  fc_kmem_free(clp, sizeof(icfgparam));
 fail1:
  /*
   * Just in case the interrupt is still on
   */
  if(p_dev_ctl->intr_inited)
    i_clear((struct intr *)p_dev_ctl);
  fc_kmem_free(p_dev_ctl, sizeof(fc_dev_ctl_t));

  return(1);
}

/**************************************************************************
* Function name : fc_select_queue_depth
*
* Description:
*   Sets the queue depth for each SCSI device hanging off the input
**************************************************************************/
_static_ void fc_select_queue_depth(struct Scsi_Host *host, 
                                    Scsi_Device      *scsi_devs)
{
  Scsi_Device *device;
  fc_dev_ctl_t      *p_dev_ctl;

  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  for( device = scsi_devs; device != NULL; device = device->next ) {
     if( device->host == host )
        fc_device_queue_depth(p_dev_ctl, device);
  }
}

/******************************************************************************
* Function name : fc_device_queue_depth
*
* Description   : Determines the queue depth for a given device.  
*                 There are two ways
*                 a queue depth can be obtained for a tagged queueing device.  
*                 One way is the default queue depth which is determined by 
*                 whether if it is defined, then it is used as the default 
*                 queue depth.  
*                 Otherwise, we use either 4 or 8 as the default queue depth 
*                 (dependent on the number of hardware SCBs).
******************************************************************************/
int fc_device_queue_depth(fc_dev_ctl_t *p_dev_ctl, 
                          Scsi_Device  *device)
{
  iCfgParam  * clp;
  FC_BRD_INFO       *binfo;
         
  binfo = &p_dev_ctl->info;
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  if( device->tagged_supported ) {
      device->tagged_queue = 1;
      device->current_tag = 0;
      device->queue_depth = clp[CFG_DFT_LUN_Q_DEPTH].a_current;
  } else {
     device->queue_depth = 16;
  }
  return(device->queue_depth);
}

/******************************************************************************
* Function name : lpfc_do_dpc
*
* Description   : 
* 
******************************************************************************/
void lpfc_do_dpc(void *p) 
{
  fc_dev_ctl_t    * p_dev_ctl=(fc_dev_ctl_t*)p;
  FC_BRD_INFO     * binfo;
  FCCLOCK_INFO    * clock_info;
  iCfgParam   * clp;
  struct lpfc_dpc * ldp;
  void            * ioa;
  unsigned long secs;
  int  instance, ev;
  ulong  iflg;
  ulong  siflg;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
  struct fs_struct *fs;
  struct semaphore sem = MUTEX_LOCKED;
#else
  DECLARE_MUTEX_LOCKED(sem);
#endif

  lock_kernel();
  secs = 0;

  /*
   * If we were started as result of loading a module, close all of the
   * user space pages.  We don't need them, and if we didn't close them
   * they would be locked into memory.
   */
  exit_mm(current);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
  daemonize();
#else
  /* Become as one with the init task */
  exit_fs(current); /* current->fs->count--; */

  fs = init_task.fs;
  /*
   * Some kernels compiled for SMP, while actually running
   * on a uniproc machine, will return NULL for this call
   */
  if( fs) {
    current->fs = fs;
    atomic_inc(&fs->count);

     exit_files(current);
     current->files = init_task.files;
     atomic_inc(&current->files->count);
  }
#endif

  binfo = &BINFO;
  clock_info = &DD_CTL.fc_clock_info;
  instance = binfo->fc_brd_no ;
  clp = DD_CTL.p_config[instance];
  ldp = &lpfc_dpc[instance];

  /* Since this is a kernel process, lets be nice to it! */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#ifdef DEF_NICE
  current->nice = -20;
  current->processor = smp_processor_id();
#endif /* DEF_NICE */
  current->cpus_allowed = lpfc_one_cpu;


#else
  {
  int niceval;
  uint32 priority;

  niceval = -20;
  priority = niceval;
  if (niceval < 0)
    priority = -niceval;
  if (priority > 20)
    priority = 20;
  priority = (priority * DEF_PRIORITY + 10) / 20 + DEF_PRIORITY;

  if (niceval >= 0) {
    priority = 2*DEF_PRIORITY - priority;
    if (!priority)
       priority = 1;
  }
  current->priority = priority;
  }
#endif
  current->session = 1;
  current->pgrp = 1;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
  siginitsetinv(&current->blocked, SHUTDOWN_SIGS);
#else
  siginitsetinv(&current->blocked, sigmask(SIGKILL));  
#endif

  /*
   * Set the name of this process.
   */
  sprintf(current->comm, "lpfc_do_dpc_%d", instance);

  ldp->dpc_wait = &sem;
  ldp->dpc_handler = current;

  unlock_kernel();

  /*
   * Wake up the thread that created us.
   */
  if( ldp->dpc_notify != NULL )
     up(ldp->dpc_notify);
  ev = 0;

  while( 1 ) {
    /*
     * If we get a signal, it means we are supposed to go
     * away and die.  This typically happens if the user is
     * trying to unload a module.
     */
    if(ev == 0) {
       ldp->dpc_ticks = clock_info->ticks;

       if(clp[CFG_NETWORK_ON].a_current) {
#ifdef __ia64__
          if (bh_count(smp_processor_id()) > 0) {
             bh_count(smp_processor_id())--;
          }
#else
          if (local_bh_count(smp_processor_id()) > 0) {
             local_bh_count(smp_processor_id())--;
          }
#endif
       }

       /* Only wait if we go thru KP once with no work */
       down_interruptible(&sem);
       if( signal_pending(current) ) {

         iflg = 0;
#ifdef INIT_SIGHAND
         spin_lock_irqsave(&current->sighand->siglock, iflg);
         flush_signals(current);
         spin_unlock_irqrestore(&current->sighand->siglock, iflg);
#else
         spin_lock_irqsave(&current->sigmask_lock, iflg);
         flush_signals(current);
         spin_unlock_irqrestore(&current->sigmask_lock, iflg);
#endif

         /* Only allow our driver unload to kill the KP */
         if( ldp->dpc_notify != NULL )
             break;   /* get out */
       }
       ldp->dpc_ticks = clock_info->ticks;
       if(clp[CFG_NETWORK_ON].a_current) {
#ifdef __ia64__
          bh_count(smp_processor_id())++;
#else
          local_bh_count(smp_processor_id())++;
#endif
       }

    }
    ev = 0;

    siflg = 0;
    iflg = 0;
    LPFC_LOCK_SCSI_DONE;
    LPFC_LOCK_DRIVER(22);
    ldp->dpc_active = 1;

    p_dev_ctl->dpc_cnt++;
    p_dev_ctl->dev_flag &= ~FC_NEEDS_DPC;

    /* Handle timer interrupts */
    if(p_dev_ctl->qclk_head) {
       ev++;
       do_fc_timer(p_dev_ctl);
    }

    /* Handle adapter interrupts */
    if(p_dev_ctl->dpc_ha_copy) {
       ev++;
       do_fc_intr((struct intr *)p_dev_ctl);
    }

    if(p_dev_ctl->qcmd_head) {
       ev++;
       if(clp[CFG_CR_DELAY].a_current != 0) {
          ioa = FC_MAP_MEM(&binfo->fc_iomap_mem);  /* map in io registers */
         if ((uchar)READ_SLIM_ADDR(binfo, ((volatile uint32 *)ioa + (SLIMOFF+(FC_ELS_RING*2)+1))) !=
             ((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.port[FC_ELS_RING].rspPutInx) {
            handle_ring_event(p_dev_ctl, FC_ELS_RING, HA_R0CE_RSP);
         }
         if ((uchar)READ_SLIM_ADDR(binfo, ((volatile uint32 *)ioa + (SLIMOFF+(FC_FCP_RING*2)+1))) !=
             ((SLI2_SLIM * )binfo->fc_slim2.virt)->mbx.us.s2.port[FC_FCP_RING].rspPutInx) {
            handle_ring_event(p_dev_ctl, FC_FCP_RING, HA_R2CE_RSP);
         }
         FC_UNMAP_MEMIO(ioa);
       }
       do_fc_queuecommand(p_dev_ctl, siflg);
    }

    /* Handle SCSI layer aborts */
    if(p_dev_ctl->abort_head) {
       ev++;
       do_fc_abort(p_dev_ctl);
    }

    /* Handle SCSI layer device resets */
    if(p_dev_ctl->rdev_head) {
       ev++;
       do_fc_reset_device(p_dev_ctl);
    }

    /* Handle SCSI layer bus resets */
    if(p_dev_ctl->rbus_head) {
       ev++;
       do_fc_reset_bus(p_dev_ctl);
    }

    /* Handle SCSI layer host resets */
    if(p_dev_ctl->rhst_head) {
       ev++;
       do_fc_reset_host(p_dev_ctl);
    }

    /* Handle iodone processing */
    if(p_dev_ctl->iodone_head) {
       int count, first_inq;
       Scsi_Cmnd *cmd;
       struct buf * head;
       struct dev_info   *devp;
       struct sc_buf  *sp;
       uint32 *iptr;

       ev++;
       ldp->dpc_active = 0;

       head = p_dev_ctl->iodone_head;
       count = 0;
       while(head) {
         count++;
         cmd = head->cmnd;
         devp = ((struct sc_buf *)head)->current_devp;
         head=head->av_forw;

         if(devp)
           devp->iodonecnt++;
         else
           panic("NULL devp in flush_done\n");

         if(cmd && (cmd->scsi_done != NULL)) {
           sp = (struct sc_buf *)cmd->host_scribble;
           if (!sp) {
             /* NULL sp in DPC flush_done */
             fc_log_printf_msg_vargs( binfo->fc_brd_no,
                    &fc_msgBlk0709,                   /* ptr to msg structure */
                     fc_mes0709,                      /* ptr to msg */
                      fc_msgBlk0709.msgPreambleStr,   /* begin varargs */
                       cmd->cmnd[0],
                        cmd->serial_number,
                         cmd->retries,
                          cmd->result);               /* end varargs */
              continue;
           }
     
           FCSTATCTR.fcpRsvd1++;
     
           if(devp->scp) {
              sp->bufstruct.av_forw = devp->scp;
              devp->scp = sp;
           }
           else {
              devp->scp = sp;
              devp->scp->bufstruct.av_forw = 0;
           }
           devp->scpcnt++;
           cmd->host_scribble = 0;

           iptr = (uint32 *)&cmd->sense_buffer[0];
           if((cmd->result) || *iptr) {
              devp->errorcnt++;
              /* iodone error return */
              fc_log_printf_msg_vargs( binfo->fc_brd_no,
                     &fc_msgBlk0711,                   /* ptr to msg structure */
                      fc_mes0711,                      /* ptr to msg */
                       fc_msgBlk0711.msgPreambleStr,   /* begin varargs */
                        (uint32)((cmd->target << 16) | cmd->lun),
                         (uint32)((cmd->retries << 16 ) | cmd->cmnd[0]),
                          cmd->result,
                           *iptr);                     /* end varargs */
           }

           first_inq = 0;
           if(devp->first_check & FIRST_IO) {
              uchar *buf;
              if(cmd->cmnd[0] == FCP_SCSI_INQUIRY) {
                 buf = (uchar *)cmd->request_buffer;
                 if((cmd->result) ||
                    ((*buf & 0x70) != 0)) {          /* lun not there */
#ifdef FREE_LUN
                    deviFree(p_dev_ctl, devp, devp->nodep);
#else
                    devp->first_check &= ~FIRST_IO;
#endif
                 } else {
                    devp->first_check &= ~FIRST_IO;
                 }
                 first_inq = 1;
              }
           }

           LPFC_UNLOCK_DRIVER;
           lpfc_scsi_add_timer(cmd, cmd->timeout_per_command);
           cmd->scsi_done(cmd);
           iflg = 0;
           LPFC_LOCK_DRIVER(2);
         }
         else
           panic("Cmnd in done queue without scsi_done\n");
       }
       p_dev_ctl->iodone_head = 0;
       p_dev_ctl->iodone_list = 0;
       LPFC_UNLOCK_DRIVER;
    }
    else {
       ldp->dpc_active = 0;
       LPFC_UNLOCK_DRIVER;
    }
    LPFC_UNLOCK_SCSI_DONE;

    if(p_dev_ctl->dev_flag & FC_SCHED_CFG_INIT) {
       p_dev_ctl->dev_flag &= ~FC_SCHED_CFG_INIT;
       fc_cfg_init(p_dev_ctl);

       LPFC_LOCK_SCSI_DONE;
       LPFC_LOCK_DRIVER(27);
       if(p_dev_ctl->fc_estabtmo) {
          fc_clk_can(p_dev_ctl, p_dev_ctl->fc_estabtmo);
       }
       if (binfo->fc_ffstate != FC_READY) {
          p_dev_ctl->fc_estabtmo =
             fc_clk_set(p_dev_ctl, 60, fc_establish_link_tmo, 0, 0);
       }
       LPFC_UNLOCK_DRIVER;
       LPFC_UNLOCK_SCSI_DONE;
    }
  }

  /*
   * Make sure that nobody tries to wake us up again.
   */
  ldp->dpc_wait = NULL;
  ldp->dpc_handler = NULL;
  ldp->dpc_active = 0;

  /*
   * If anyone is waiting for us to exit (i.e. someone trying to unload
   * a driver), then wake up that process to let them know we are on
   * the way out the door.  This may be overkill - I *think* that we
   * could probably just unload the driver and send the signal, and when
   * the error handling thread wakes up that it would just exit without
   * needing to touch any memory associated with the driver itself.
   */
  if( ldp->dpc_notify != NULL )
    up(ldp->dpc_notify);
}

#ifdef MODULE
/******************************************************************************
* Function name : fc_release
*
* Description   : 
* 
******************************************************************************/
int fc_release(struct Scsi_Host *host)
{
  fc_dev_ctl_t      *p_dev_ctl;
  FC_BRD_INFO       *binfo;
  node_t            *node_ptr;
  struct dev_info   *dev_ptr;
  struct lpfc_dpc   *ldp;
  int               instance;
  int               dev_index,target;
  fc_lun_t          lun;
  ulong             iflg;

  /*
   * Indicate driver unloading so our interrupt handler can stop
   * accepting interrupts.
   */
   lpfc_driver_unloading = 1;

  /*
   * get dev control from host 
   */
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  binfo = &BINFO;
  instance = binfo->fc_brd_no ;

  if(lpfcdiag_cnt) {
     /* Cannot unload driver while lpfcdiag Interface is active */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk1200,                   /* ptr to msg structure */
             fc_mes1200,                      /* ptr to msg */
              fc_msgBlk1200.msgPreambleStr,   /* begin varargs */
               lpfcdiag_cnt,
                (uint32)instance);            /* end varargs */
  }

  iflg = 0;
  LPFC_LOCK_DRIVER(24);
  linux_detach(instance);
  /*
   *Clear all devi's
   *Although host_queue has all devices, its not a good idea to touch it!
   *instead we will loop on all possible targets and luns
   */
  for(target=0; target < host->max_id; target++) {
    dev_index = INDEX(ZERO_PAN, target);
    node_ptr = binfo->device_queue_hash[dev_index].node_ptr;
    if(!node_ptr)
      continue;
    for(lun=0; lun <= host->max_lun; lun++){
      dev_ptr = fc_find_lun(binfo, dev_index, lun); 
      if(!dev_ptr)
        continue;
      /*
       * Free this device
       */
      deviFree(p_dev_ctl, dev_ptr, node_ptr);
    }
    fc_kmem_free(node_ptr, sizeof(node_t));
    binfo->device_queue_hash[dev_index].node_ptr = 0;
  } 

  fcinstcnt--;
  DD_CTL.num_devs--;
  LPFC_UNLOCK_DRIVER;

  if(lpfc_major)
    unregister_chrdev(lpfc_major, "lpfcdfc");

  ldp = &lpfc_dpc[instance];
  if(ldp->dpc_handler != NULL ) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
     struct semaphore sem = MUTEX_LOCKED;
#else
     DECLARE_MUTEX_LOCKED(sem);
#endif

    ldp->dpc_notify = &sem;
    send_sig(SIGKILL, ldp->dpc_handler, 1);
    down(&sem);
    ldp->dpc_notify = NULL;
  }

  return 0;
}
#endif /* MODULE */

#ifdef MODULE
/******************************************************************************
* Function name : linux_detach
*
* Description   : LINUX deinitialization entry point, called from environment
*                 to detach from / free resources for  a specific adapter.
******************************************************************************/
_local_ int linux_detach( int instance)
{
  FC_BRD_INFO    * binfo;
  iCfgParam  * clp;
  fc_dev_ctl_t   * p_dev_ctl = (fc_dev_ctl_t * ) NULL;

  p_dev_ctl = DD_CTL.p_dev[instance];
  if (p_dev_ctl == NULL) {
    return(0);
  }
  binfo = &BINFO;
  clp = DD_CTL.p_config[instance];
  if (clp == NULL) {
    return(0);
  }

  /* 
   * Stop and free resources associated with scsi timeout timer
   */
  if(DD_CTL.num_devs == 1) {
    FCCLOCK_INFO    * clock_info;

    if(SCSI_TMO) {
      fc_clk_can(0, SCSI_TMO);
      SCSI_TMO = 0;
    }
    clock_info = &DD_CTL.fc_clock_info;
    ((struct watchdog *)(CLOCKWDT))->stopping = 1;
    if (((struct watchdog *)(CLOCKWDT))->timer.function)
      del_timer(&((struct watchdog *)(CLOCKWDT))->timer);
    ((struct watchdog *)(CLOCKWDT))->timer.function=NULL;
    ((struct watchdog *)(CLOCKWDT))->timeout_id=0;
  }
  fc_detach(instance);

  fc_kmem_free(DD_CTL.p_dev[instance], sizeof(fc_dev_ctl_t));
  DD_CTL.p_dev[instance] = 0;
  fc_kmem_free(DD_CTL.p_config[instance], sizeof(icfgparam));
  DD_CTL.p_config[instance] = 0;

  kfree(fc_mem_dmapool[instance]);
  return(0);
}
#endif /* MODULE */

/******************************************************************************
* Function name : fc_abort
*
* Description   : Linux mid-level command abort entry
*                 Note we are using the new error handling routines
******************************************************************************/
int fc_abort(Scsi_Cmnd *Cmnd)
{
  struct Scsi_Host  *host;
  fc_dev_ctl_t      *p_dev_ctl;
  FC_BRD_INFO       * binfo;
  ulong               iflg;
  struct lpfc_dpc *ldp;


  host = Cmnd->host;
  if(!host) {
#ifdef FC_NEW_EH
    return FAILED ;
#else
    return SCSI_ABORT_NOT_RUNNING ;
#endif
  }
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  if(!p_dev_ctl) {
#if FC_NEW_EH
    return FAILED ;
#else
    return SCSI_ABORT_NOT_RUNNING ;
#endif
  }
  binfo = &BINFO;

  iflg = 0;
  LPFC_LOCK_DRIVER(5);
  ldp = &lpfc_dpc[binfo->fc_brd_no];
  if (ldp->dpc_wait == NULL) {
    LPFC_UNLOCK_DRIVER;
#if FC_NEW_EH
    return SUCCESS;
#else
    return SCSI_ABORT_SUCCESS ;
#endif
  }

  fc_dpc_lstchk(p_dev_ctl, Cmnd);
  if(p_dev_ctl->abort_head == NULL) {
    p_dev_ctl->abort_head = (void *)Cmnd;
    p_dev_ctl->abort_list = (void *)Cmnd;
  } else {
    ((Scsi_Cmnd *)(p_dev_ctl->abort_list))->reset_chain = Cmnd;
    p_dev_ctl->abort_list = (void *)Cmnd;
  }
  Cmnd->reset_chain = NULL;


  if (ldp->dpc_active == 0) {
    LPFC_UNLOCK_DRIVER;
    up(ldp->dpc_wait);
  }
  else {
    LPFC_UNLOCK_DRIVER;
  }

#if FC_NEW_EH
  return SUCCESS;
#else
  return SCSI_ABORT_SUCCESS ;
#endif
}

/******************************************************************************
* Function name : do_fc_abort
*
* Description   : 
* 
******************************************************************************/
int do_fc_abort(fc_dev_ctl_t *p_dev_ctl)
{
  Scsi_Cmnd         * Cmnd;
  Scsi_Cmnd         * oCmnd;
  FC_BRD_INFO       * binfo;
  dvi_t             * dev_ptr;
  struct sc_buf     * sp;
  int               dev_index,target;
  fc_lun_t          lun;

  binfo = &BINFO;
  Cmnd = (Scsi_Cmnd *)p_dev_ctl->abort_head;
  while(Cmnd) {
     target = (int)Cmnd->target;
     lun = (fc_lun_t)Cmnd->lun;
     dev_index = INDEX(ZERO_PAN, target);

     dev_ptr = fc_find_lun(binfo, dev_index, lun); 
     /* SCSI layer issued abort device */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0712,                   /* ptr to msg structure */
             fc_mes0712,                      /* ptr to msg */
              fc_msgBlk0712.msgPreambleStr,   /* begin varargs */
               target,
                (uint32)lun,
                  Cmnd->cmnd[0],
                   Cmnd->serial_number);      /* end varargs */
     if(!dev_ptr || !(dev_ptr->nodep)) {
       goto done;
     }

     if (dev_ptr->flags & CHK_SCSI_ABDR) {
       goto done;
     }

     sp = (struct sc_buf *)Cmnd->host_scribble;
     if (lpfc_find_cmd(p_dev_ctl, Cmnd)) {
        FCSTATCTR.fcpRsvd2++;
     } else {
        if (fc_abort_clk_blk(p_dev_ctl, lpfc_scsi_selto_timeout, sp, 0)) {
            FCSTATCTR.fcpRsvd2++;
        }
     }
done:
     oCmnd = Cmnd;
     Cmnd = Cmnd->reset_chain;
     oCmnd->reset_chain = 0;
  }
  p_dev_ctl->abort_head = 0;
  p_dev_ctl->abort_list = 0;

  return(0);
}

#ifndef FC_NEW_EH
/******************************************************************************
* Function name : lpfc_reset
*
* Description   : 
* 
******************************************************************************/
int lpfc_reset(Scsi_Cmnd   *Cmnd, 
               unsigned int flags)
{
   int action;

   if( flags & SCSI_RESET_SUGGEST_HOST_RESET ) {
       if((action = fc_reset_host(Cmnd)) == FAILED)
           return(SCSI_RESET_ERROR);
       action = SCSI_RESET_SUCCESS | SCSI_RESET_HOST_RESET;
   }
   else if( flags & SCSI_RESET_SUGGEST_BUS_RESET ) {
       if((action = fc_reset_bus(Cmnd)) == FAILED)
           return(SCSI_RESET_ERROR);
       action = SCSI_RESET_SUCCESS | SCSI_RESET_BUS_RESET;
   }
   else {
       if((action = fc_reset_device(Cmnd)) == FAILED)
           return(SCSI_RESET_ERROR);
       action = SCSI_RESET_SUCCESS;
   }
   return(action);
}
#endif

/******************************************************************************
* Function name : fc_reset_device
*
* Description   : Linux mid-level reset device entry
*                 Note we are using the new error handling routines
*                 In the old handlers there is only one reset entry which has 
*                 two arguments
******************************************************************************/
int fc_reset_device(Scsi_Cmnd *Cmnd)
{
  struct Scsi_Host  *host;
  fc_dev_ctl_t      *p_dev_ctl;
  FC_BRD_INFO       *binfo;
  ulong              iflg;
  struct lpfc_dpc *ldp;
 
  host = Cmnd->host;
  if(!host) {
    return FAILED ;
  }
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  if(!p_dev_ctl) {
    return FAILED;
  }
  binfo = &BINFO;

  iflg = 0;
  LPFC_LOCK_DRIVER(6);
  ldp = &lpfc_dpc[binfo->fc_brd_no];
  if (ldp->dpc_wait == NULL) {
    LPFC_UNLOCK_DRIVER;
    return SUCCESS;
  }

  fc_dpc_lstchk(p_dev_ctl, Cmnd);
  if(p_dev_ctl->rdev_head == NULL) {
    p_dev_ctl->rdev_head = (void *)Cmnd;
    p_dev_ctl->rdev_list = (void *)Cmnd;
  } else {
    ((Scsi_Cmnd *)(p_dev_ctl->rdev_list))->reset_chain = Cmnd;
    p_dev_ctl->rdev_list = (void *)Cmnd;
  }
  Cmnd->reset_chain = NULL;

  if (ldp->dpc_active == 0) {
    LPFC_UNLOCK_DRIVER;
    up(ldp->dpc_wait);
  }
  else {
    LPFC_UNLOCK_DRIVER;
  }

  return SUCCESS;
}

/******************************************************************************
* Function name : do_fc_reset_device
*
* Description   : 
* 
******************************************************************************/
int do_fc_reset_device(fc_dev_ctl_t *p_dev_ctl)
{
  Scsi_Cmnd         * Cmnd;
  Scsi_Cmnd         * oCmnd;
  struct dev_info   * dev_ptr;
  FC_BRD_INFO       * binfo;
  int                 dev_index, target, j;
  fc_lun_t            lun;
 
  binfo = &BINFO;
  Cmnd = (Scsi_Cmnd *)p_dev_ctl->rdev_head;
  while(Cmnd) {
     target = (int)Cmnd->target;
     lun = (fc_lun_t)Cmnd->lun;
     dev_index = INDEX(ZERO_PAN, target);

     dev_ptr = fc_find_lun(binfo, dev_index, lun); 
     j = 0;
     /* SCSI layer issued target reset */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0713,                   /* ptr to msg structure */
             fc_mes0713,                      /* ptr to msg */
              fc_msgBlk0713.msgPreambleStr,   /* begin varargs */
               target,
                (uint32)lun,
                  dev_index);                 /* end varargs */
     if(dev_ptr == 0) {
       goto done;
     }
     if ((binfo->fc_ffstate != FC_READY) ||
        (!(dev_ptr->nodep)) ||
        (dev_ptr->nodep->rpi == 0xfffe)) {
        goto done;
     }
     fc_fcp_abort(p_dev_ctl, TARGET_RESET, dev_index, -1);


done:
     oCmnd = Cmnd;
     Cmnd = Cmnd->reset_chain;
     oCmnd->reset_chain = 0;
  }
  p_dev_ctl->rdev_head = 0;
  p_dev_ctl->rdev_list = 0;

  return(0);
}

/******************************************************************************
* Function name : fc_reset_bus
*
* Description   : Linux mid-level reset host/bus entry
* 
******************************************************************************/
int fc_reset_bus(Scsi_Cmnd *Cmnd)
{
  struct Scsi_Host  *host;
  fc_dev_ctl_t      *p_dev_ctl;
  FC_BRD_INFO       *binfo;
  ulong              iflg;
  struct lpfc_dpc *ldp;

  host = Cmnd->host;
  if(!host) {
    return FAILED;
  }
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  if(!p_dev_ctl) {
    return FAILED;
  }
  binfo = &p_dev_ctl->info;

  iflg = 0;
  LPFC_LOCK_DRIVER(8);
  ldp = &lpfc_dpc[binfo->fc_brd_no];
  if (ldp->dpc_wait == NULL) {
    LPFC_UNLOCK_DRIVER;
    return SUCCESS;
  }

  fc_dpc_lstchk(p_dev_ctl, Cmnd);
  if(p_dev_ctl->rbus_head == NULL) {
    p_dev_ctl->rbus_head = (void *)Cmnd;
    p_dev_ctl->rbus_list = (void *)Cmnd;
  } else {
    ((Scsi_Cmnd *)(p_dev_ctl->rbus_list))->reset_chain = Cmnd;
    p_dev_ctl->rbus_list = (void *)Cmnd;
  }
  Cmnd->reset_chain = NULL;

  if (ldp->dpc_active == 0) {
    LPFC_UNLOCK_DRIVER;
    up(ldp->dpc_wait);
  }
  else {
    LPFC_UNLOCK_DRIVER;
  }

  return SUCCESS;
}

/******************************************************************************
* Function name : do_fc_reset_bus
*
* Description   : 
* 
******************************************************************************/
int do_fc_reset_bus(fc_dev_ctl_t *p_dev_ctl)
{
  Scsi_Cmnd         * Cmnd;
  Scsi_Cmnd         * oCmnd;
  FC_BRD_INFO       *binfo;
  node_t * node_ptr;
  struct dev_info   * dev_ptr;
  NODELIST        * nlp;
  NODELIST        * new_nlp;
  iCfgParam     *clp;
  int               rets = FAILED; 

  binfo = &p_dev_ctl->info;
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  Cmnd = (Scsi_Cmnd *)p_dev_ctl->rbus_head;
  while(Cmnd) {
     /* SCSI layer issued Bus Reset */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0714,                   /* ptr to msg structure */
             fc_mes0714,                      /* ptr to msg */
              fc_msgBlk0714.msgPreambleStr,   /* begin varargs */
               Cmnd->target,
                (uint32)Cmnd->lun);           /* end varargs */
     /*
      * Tell them
      */  
     if (binfo->fc_ffstate == FC_READY) {
        rets =  SUCCESS;
        fc_fcp_abort(p_dev_ctl, TARGET_RESET, -1, -1);
     }
     else {
        /*
         * Check to see if we should wait for FC_READY
         */  
        if ((binfo->fc_ffstate < FC_LINK_DOWN) ||
           (binfo->fc_ffstate == FC_ERROR)) {
           rets =  FAILED;
        }
        else {
           rets =  SUCCESS;
        }
     }

     /* Reset first_check */
     nlp = binfo->fc_nlpmap_start;
     while(nlp != (NODELIST *)&binfo->fc_nlpmap_start) {
         new_nlp = (NODELIST *)nlp->nlp_listp_next;
         if (nlp->nlp_type & NLP_FCP_TARGET) {
            if(clp[CFG_FIRST_CHECK].a_current) {
               /* If we are an FCP node, update first_check flag for all LUNs */
               if ((node_ptr = (node_t * )nlp->nlp_targetp) != NULL) {
                  for (dev_ptr = node_ptr->lunlist; dev_ptr != NULL; 
                      dev_ptr = dev_ptr->next) {
                     dev_ptr->first_check = FIRST_CHECK_COND;
                  }
               }
            }
         }
         nlp = new_nlp;
     }
     oCmnd = Cmnd;
     Cmnd = Cmnd->reset_chain;
     oCmnd->reset_chain = 0;
  }
  p_dev_ctl->rbus_head = 0;
  p_dev_ctl->rbus_list = 0;

  return rets;
}

/******************************************************************************
* Function name : fc_reset_host
*
* Description   : 
* 
******************************************************************************/
int fc_reset_host(Scsi_Cmnd *Cmnd)
{
  struct Scsi_Host  *host;
  fc_dev_ctl_t      *p_dev_ctl;
  FC_BRD_INFO       *binfo;
  ulong              iflg;
  struct lpfc_dpc *ldp;

  host = Cmnd->host;
  if(!host) {
    return FAILED;
  }
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  if(!p_dev_ctl) {
    return FAILED;
  }
  binfo = &p_dev_ctl->info;

  iflg = 0;
  LPFC_LOCK_DRIVER(10);
  ldp = &lpfc_dpc[binfo->fc_brd_no];
  if (ldp->dpc_wait == NULL) {
    LPFC_UNLOCK_DRIVER;
    return SUCCESS;
  }

  fc_dpc_lstchk(p_dev_ctl, Cmnd);
  if(p_dev_ctl->rhst_head == NULL) {
    p_dev_ctl->rhst_head = (void *)Cmnd;
    p_dev_ctl->rhst_list = (void *)Cmnd;
  } else {
    ((Scsi_Cmnd *)(p_dev_ctl->rhst_list))->reset_chain = Cmnd;
    p_dev_ctl->rhst_list = (void *)Cmnd;
  }
  Cmnd->reset_chain = NULL;

  if (ldp->dpc_active == 0) {
    LPFC_UNLOCK_DRIVER;
    up(ldp->dpc_wait);
  }
  else {
    LPFC_UNLOCK_DRIVER;
  }

  return SUCCESS;
}

/******************************************************************************
* Function name : do_fc_reset_host
*
* Description   : 
* 
******************************************************************************/
int do_fc_reset_host(fc_dev_ctl_t *p_dev_ctl)
{
  Scsi_Cmnd         * Cmnd;
  Scsi_Cmnd         * oCmnd;
  FC_BRD_INFO       *binfo;
  int               rets = FAILED; 

  binfo = &p_dev_ctl->info;
  Cmnd = (Scsi_Cmnd *)p_dev_ctl->rhst_head;
  while(Cmnd) {
     /* SCSI layer issued Host Reset */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0715,                   /* ptr to msg structure */
             fc_mes0715,                      /* ptr to msg */
              fc_msgBlk0715.msgPreambleStr,   /* begin varargs */
               Cmnd->target,
                (uint32)Cmnd->lun);           /* end varargs */
     /*
      * Check to see if we should wait for FC_READY
      */  
     if ((binfo->fc_ffstate < FC_LINK_DOWN) || (binfo->fc_ffstate == FC_ERROR)) {
        rets =  FAILED;
     }
     else {
        rets =  SUCCESS;
     }
     oCmnd = Cmnd;
     Cmnd = Cmnd->reset_chain;
     oCmnd->reset_chain = 0;
  }
  p_dev_ctl->rhst_head = 0;
  p_dev_ctl->rhst_list = 0;

  return(rets);
}


static char addrStr[18];

/******************************************************************************
* Function name : addr_sprintf
*
* Description   : Used by fc_info for displaying WWNN / WWPNs
* 
******************************************************************************/
_static_ char * addr_sprintf(register uchar *ap)
{
   register int i;
   register char    *cp = addrStr;
   static char  digits[] = "0123456789abcdef";

   for (i = 0; i < 8; i++) {
      *cp++ = digits[*ap >> 4];
      *cp++ = digits[*ap++ & 0xf];
      *cp++ = ':';
   }
   *--cp = 0;
   return(addrStr);
}   /* End addr_sprintf */

/******************************************************************************
* Function name : fc_info
*
* Description   : Prepare host information for mid-level
* 
******************************************************************************/
const char *fc_info(struct Scsi_Host *host)
{
  static char     buf[4096];
  fc_dev_ctl_t    *p_dev_ctl;
  FC_BRD_INFO     * binfo;
  struct pci_dev  *pdev;
  char            *multip;
  int             idx, i, j, incr;
  char            hdw[9]; 
  NODELIST        *nlp;

  buf[0]='\0';
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  if(!p_dev_ctl)
    return buf;
  binfo = &BINFO;
  pdev = p_dev_ctl->pcidev ;

  for(idx=0; idx < MAX_FC_BRDS; idx++) {
     if(p_dev_ctl == DD_CTL.p_dev[idx])
        break;
  }

  multip = "LPFC";

  if (!(p_dev_ctl->dev_flag & FC_FULL_INFO_CALL)) {
    if(pdev != NULL) {
      switch(pdev->device){
      case PCI_DEVICE_ID_CENTAUR:
        if(FC_JEDEC_ID(VPD.rev.biuRev) == CENTAUR_2G_JEDEC_ID) {
           sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP9002 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        } else {
           sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP9000 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        }
        break;
      case PCI_DEVICE_ID_DRAGONFLY:
        sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP8000 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        break;
      case PCI_DEVICE_ID_PEGASUS:
        sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP9802 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        break;
      case PCI_DEVICE_ID_PFLY:
        sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP982 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        break;
      case PCI_DEVICE_ID_THOR:
        sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP10000 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        break;
      case PCI_DEVICE_ID_TFLY:
        sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse LP1050 on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
        break;
      default:
        sprintf(&buf[strlen(buf)],
          "HBA: Emulex LightPulse on PCI bus %02x device %02x irq %d",
          p_dev_ctl->pcidev->bus->number, p_dev_ctl->pcidev->devfn,
          p_dev_ctl->pcidev->irq);
      }
    } 
    p_dev_ctl->dev_flag |= FC_FULL_INFO_CALL;
    return(buf);
  }

  sprintf(buf, "Emulex LightPulse %s Driver Version: %s\n", 
          multip, lpfc_release_version); 

  if(pdev != NULL) {
    switch(pdev->device){
    case PCI_DEVICE_ID_CENTAUR:
      if(FC_JEDEC_ID(VPD.rev.biuRev) == CENTAUR_2G_JEDEC_ID) {
         sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP9002 2 Gigabit PCI Fibre Channel Adapter\n");
      } else {
         sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP9000 1 Gigabit PCI Fibre Channel Adapter\n");
      }
      break;
    case PCI_DEVICE_ID_DRAGONFLY:
      sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP8000 1 Gigabit PCI Fibre Channel Adapter\n");
      break;
    case PCI_DEVICE_ID_PEGASUS:
      sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP9802 2 Gigabit PCI Fibre Channel Adapter\n");
      break;
    case PCI_DEVICE_ID_PFLY:
      sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP982 2 Gigabit PCI Fibre Channel Adapter\n");
      break;
    case PCI_DEVICE_ID_THOR:
      sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP10000 2 Gigabit PCI Fibre Channel Adapter\n");
      break;
    case PCI_DEVICE_ID_TFLY:
      sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse LP1050 2 Gigabit PCI Fibre Channel Adapter\n");
      break;
    default:
      sprintf(&buf[strlen(buf)],
        "HBA: Emulex LightPulse PCI Fibre Channel Adapter\n");
    }
  } 

  sprintf(&buf[strlen(buf)], "SerialNum: %s\n", binfo->fc_SerialNumber);

  decode_firmware_rev(binfo, &VPD);
  sprintf(&buf[strlen(buf)], "Firmware Version: %s\n", fwrevision);

  sprintf(&buf[strlen(buf)], "Hdw: ");
  /* Convert JEDEC ID to ascii for hardware version */
  incr = VPD.rev.biuRev;
  for(i=0;i<8;i++) {
     j = (incr & 0xf);
     if(j <= 9)
        hdw[7-i] = (char)((uchar)0x30 + (uchar)j);
     else
        hdw[7-i] = (char)((uchar)0x61 + (uchar)(j-10));
     incr = (incr >> 4);
  }
  hdw[8] = 0; 
  strcat(buf, hdw);

  sprintf(&buf[strlen(buf)], "\nVendorId: 0x%x\n", 
          ((((uint32)pdev->device) << 16) | (uint32)(pdev->vendor)));

  sprintf(&buf[strlen(buf)], "Portname: ");
  strcat(buf, addr_sprintf((uchar *)&binfo->fc_portname));

  sprintf(&buf[strlen(buf)], "   Nodename: ");
  strcat(buf, addr_sprintf((uchar *)&binfo->fc_nodename));

  switch (binfo->fc_ffstate) {
     case FC_INIT_START:
     case FC_INIT_NVPARAMS:
     case FC_INIT_REV:
     case FC_INIT_PARTSLIM:
     case FC_INIT_CFGRING:
     case FC_INIT_INITLINK:
     case FC_LINK_DOWN:
        sprintf(&buf[strlen(buf)], "\n\nLink Down\n");
        break;
     case FC_LINK_UP:
     case FC_INIT_SPARAM:
     case FC_CFG_LINK:
        sprintf(&buf[strlen(buf)], "\n\nLink Up\n");
        break;
     case FC_FLOGI:
     case FC_LOOP_DISC:
     case FC_NS_REG:
     case FC_NS_QRY:
     case FC_NODE_DISC:
     case FC_REG_LOGIN:
     case FC_CLEAR_LA:
        sprintf(&buf[strlen(buf)], "\n\nLink Up - Discovery\n");
        break;
     case FC_READY:
        sprintf(&buf[strlen(buf)], "\n\nLink Up - Ready:\n");
        sprintf(&buf[strlen(buf)], "   PortID 0x%x\n", binfo->fc_myDID);
        if (binfo->fc_topology == TOPOLOGY_LOOP) {
           if(binfo->fc_flag & FC_PUBLIC_LOOP)
              sprintf(&buf[strlen(buf)], "   Public Loop\n");
           else
              sprintf(&buf[strlen(buf)], "   Private Loop\n");
        } else {
           if(binfo->fc_flag & FC_FABRIC) 
              sprintf(&buf[strlen(buf)], "   Fabric\n");
           else 
              sprintf(&buf[strlen(buf)], "   Point-2-Point\n");
        }

        if(binfo->fc_linkspeed == LA_2GHZ_LINK)
           sprintf(&buf[strlen(buf)], "   Current speed 2G\n");
        else
           sprintf(&buf[strlen(buf)], "   Current speed 1G\n");

        nlp = binfo->fc_nlpmap_start;
        while(nlp != (NODELIST *)&binfo->fc_nlpmap_start) {
           if (nlp->nlp_state == NLP_ALLOC) {
              sprintf(&buf[strlen(buf)], "\nlpfc%dt%02x DID %06x WWPN ",
                 idx, FC_SCSID(nlp->id.nlp_pan, nlp->id.nlp_sid), nlp->nlp_DID);
              strcat(buf, addr_sprintf((uchar *)&nlp->nlp_portname));
              strcat(buf, " WWNN ");
              strcat(buf, addr_sprintf((uchar *)&nlp->nlp_nodename));
           }
           if ((4096 - strlen(buf)) < 90)
              break;
           nlp = (NODELIST *)nlp->nlp_listp_next;
        }
        if(nlp != (NODELIST *)&binfo->fc_nlpmap_start)
           strcat(buf,"\n....\n");
  }

  return (buf);
}

/******************************************************************************
* Function name : fc_data_direction
*
* Description   : If we do not relay on Cmnd->sc_data_direction call this
*                 routine to determine if we are doing a read or write.
* 
******************************************************************************/
int fc_data_direction(Scsi_Cmnd *Cmnd)
{
   int ret_code;

   switch (Cmnd->cmnd[0]) {
   case WRITE_6:
   case WRITE_10:
   case WRITE_12:
   case CHANGE_DEFINITION:
   case LOG_SELECT:
   case MODE_SELECT:
   case MODE_SELECT_10:
   case WRITE_BUFFER:
   case VERIFY:
   case WRITE_VERIFY:
   case WRITE_VERIFY_12:
   case WRITE_LONG:
   case WRITE_LONG_2:
   case WRITE_SAME:
   case SEND_DIAGNOSTIC:
   case FORMAT_UNIT:
   case REASSIGN_BLOCKS: 
   case FCP_SCSI_RELEASE_LUNR:
   case FCP_SCSI_RELEASE_LUNV:
   case HPVA_SETPASSTHROUGHMODE:
   case HPVA_EXECUTEPASSTHROUGH:
   case HPVA_CREATELUN:
   case HPVA_SETLUNSECURITYLIST:
   case HPVA_SETCLOCK:
   case HPVA_RECOVER:
   case HPVA_GENERICSERVICEOUT:
   case DMEP_EXPORT_OUT:
     ret_code = B_WRITE; 
     break;
   case MDACIOCTL_DIRECT_CMD:
     switch (Cmnd->cmnd[2]) {
     case MDACIOCTL_STOREIMAGE:
     case MDACIOCTL_WRITESIGNATURE:
     case MDACIOCTL_SETREALTIMECLOCK:
     case MDACIOCTL_PASS_THRU_CDB:
     case MDACIOCTL_CREATENEWCONF:
     case MDACIOCTL_ADDNEWCONF:
     case MDACIOCTL_MORE:
     case MDACIOCTL_SETPHYSDEVPARAMETER:
     case MDACIOCTL_SETLOGDEVPARAMETER:
     case MDACIOCTL_SETCONTROLLERPARAMETER:
     case MDACIOCTL_WRITESANMAP:
     case MDACIOCTL_SETMACADDRESS:
        ret_code = B_WRITE;
        break;
     case MDACIOCTL_PASS_THRU_INITIATE:
        if (Cmnd->cmnd[3] & 0x80) {
           ret_code = B_WRITE;
        }
        else {
           ret_code = B_READ; 
        }
        break;
     default:
        ret_code = B_READ; 
     }
     break;
   default:
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
     if (Cmnd->sc_data_direction == SCSI_DATA_WRITE)
        ret_code = B_WRITE;
     else
#endif
        ret_code = B_READ;
   }
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
   if(ret_code == B_WRITE)
        Cmnd->sc_data_direction = SCSI_DATA_WRITE;
   else
        Cmnd->sc_data_direction = SCSI_DATA_READ;
#endif
   return ret_code;
}

int
chkLun(
node_t  *node_ptr,
fc_lun_t lun)
{
   uint32     rptLunLen;
   uint32    *datap32;
   uint32     lunvalue, i;

   if(node_ptr->virtRptLunData) {
      datap32 = (uint32 *)node_ptr->virtRptLunData;
      rptLunLen = SWAP_DATA(*datap32);
      for(i=0; i < rptLunLen; i+=8) {
         datap32 += 2;
         lunvalue = (((* datap32) >> FC_LUN_SHIFT) & 0xff);
         if (lunvalue == (uint32)lun)
            return 1;
      }
      return 0;
   }
   else {
      return 1;
   }
}
/******************************************************************************
* Function name : fc_queuecommand
*
* Description   : Linux queue command entry
* 
******************************************************************************/
int fc_queuecommand(Scsi_Cmnd  *Cmnd, 
                    void      (*done)(Scsi_Cmnd *))
{
  FC_BRD_INFO       * binfo;
  struct Scsi_Host  *host;
  fc_dev_ctl_t      *p_dev_ctl;
  iCfgParam     *clp;
  struct dev_info   *dev_ptr;
  node_t            *node_ptr;
  struct sc_buf     *sp;
  int               dev_index,target,retcod;
  fc_lun_t          lun;
  ulong             iflg;
  struct lpfc_dpc *ldp;


  host = Cmnd->host;
  fc_bzero(Cmnd->sense_buffer, 16);
  if(!host){
    retcod=DID_BAD_TARGET;
    Cmnd->result = ScsiResult(retcod, 0);
    done(Cmnd);
    return(0);
  }
  Cmnd->scsi_done = done;  /* Save done routine for this command */

  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];  
  if(p_dev_ctl == 0) {
    retcod=DID_BAD_TARGET;
    Cmnd->result = ScsiResult(retcod, 0);
    done(Cmnd);
    return(0);
  }


  retcod = 0;
  binfo = &BINFO;
  clp = DD_CTL.p_config[binfo->fc_brd_no];

  LPFC_LOCK_DRIVER(12);
  ldp = &lpfc_dpc[binfo->fc_brd_no];
  if (ldp->dpc_wait == NULL) {
    retcod=DID_NO_CONNECT;
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
    if(lpfc_use_removable) {
      Cmnd->sense_buffer[0] = 0x70;
      Cmnd->sense_buffer[2] = UNIT_ATTENTION;
      Cmnd->device->removable = 1;
    }
#endif
    Cmnd->result = ScsiResult(retcod, 0);
    FCSTATCTR.fcpRsvd8++;
    done(Cmnd);
    LPFC_UNLOCK_DRIVER;
    return(0);
  }

  target = (int)Cmnd->target;
  lun = (fc_lun_t)Cmnd->lun;

  if(lun > MAX_FCP_LUN) {
    retcod=DID_BAD_TARGET;
    Cmnd->result = ScsiResult(retcod, 0);
    LPFC_UNLOCK_DRIVER;
    done(Cmnd);
    return(0);
  }

  /*
   * Device for target/lun
   */
  dev_index = INDEX(ZERO_PAN, target);
  if (!(dev_ptr = fc_find_lun(binfo, dev_index, lun))) {
    if(!(dev_ptr=fc_getDVI(p_dev_ctl, target, lun))){
      retcod=DID_NO_CONNECT;
      Cmnd->result = ScsiResult(retcod, 0);
      FCSTATCTR.fcpRsvd3++;
      LPFC_UNLOCK_DRIVER;
      done(Cmnd);
      return(0);
    }
 }

  node_ptr =  binfo->device_queue_hash[dev_index].node_ptr;
  if((node_ptr) &&
     ((node_ptr->flags & FC_NODEV_TMO) || (lun >= node_ptr->max_lun))) {
    retcod=DID_NO_CONNECT;
    Cmnd->result = ScsiResult(retcod, 0);
    LPFC_UNLOCK_DRIVER;
    done(Cmnd);
    return(0);
  }

    if((node_ptr) && (Cmnd->cmnd[0] == 0x12) && (!chkLun(node_ptr, lun))) {
       retcod=DID_NO_CONNECT;
       Cmnd->result = ScsiResult(retcod, 0);
       LPFC_UNLOCK_DRIVER;
       done(Cmnd);
       return(0);
    }

    if(binfo->fc_flag & FC_LD_TIMEOUT) {
       retcod=DID_NO_CONNECT;
       Cmnd->result = ScsiResult(retcod, 0);
       LPFC_UNLOCK_DRIVER;
       done(Cmnd);
       return(0);
    }

  dev_ptr->qcmdcnt++;

  sp = dev_ptr->scp;
  if(!sp){
     retcod=DID_NO_CONNECT;
     Cmnd->result = ScsiResult(retcod, 0);
     dev_ptr->iodonecnt++;
     dev_ptr->errorcnt++;
     FCSTATCTR.fcpRsvd5++;
     LPFC_UNLOCK_DRIVER;
     done(Cmnd);
     return(0);
  }

  Cmnd->host_scribble = (void *)sp;
  dev_ptr->scp = sp->bufstruct.av_forw;
  dev_ptr->scpcnt--;
  fc_bzero(sp,sizeof(struct sc_buf));
  sp->bufstruct.cmnd = Cmnd;
  sp->current_devp = dev_ptr;
  FCSTATCTR.fcpRsvd0++;
  lpfc_scsi_delete_timer(Cmnd);

  /* Since we delete active timers, we can use eh_timeout.data as a linked
   * list ptr internally within the driver.
   */
  if(p_dev_ctl->qcmd_head == NULL) {
    p_dev_ctl->qcmd_head = (void *)Cmnd;
    p_dev_ctl->qcmd_list = (void *)Cmnd;
  } else {
    ((Scsi_Cmnd *)(p_dev_ctl->qcmd_list))->eh_timeout.data = (ulong)Cmnd;
    p_dev_ctl->qcmd_list = (void *)Cmnd;
  }
  Cmnd->eh_timeout.data = (unsigned long) NULL;

  if (ldp->dpc_active == 0) {
    LPFC_UNLOCK_DRIVER;
    up(ldp->dpc_wait);
  }
  else {
    LPFC_UNLOCK_DRIVER;
  }
  return 0;
}

/******************************************************************************
* Function name : do_fc_queuecommand
*
* Description   : 
* 
******************************************************************************/
int do_fc_queuecommand(fc_dev_ctl_t *p_dev_ctl,
                       ulong         siflg)
{
  FC_BRD_INFO       * binfo;
  iCfgParam     * clp;
  struct dev_info   * dev_ptr;
  struct sc_buf     * sp;
  struct buf        * bp;
  Scsi_Cmnd         * Cmnd;
  Scsi_Cmnd         * oCmnd;
  int                 i, retcod, firstin;

  binfo = &BINFO;
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  Cmnd = (Scsi_Cmnd *)p_dev_ctl->qcmd_head;
  firstin = 1;


  while(Cmnd) {
     sp =  (struct sc_buf *)(Cmnd->host_scribble);
     dev_ptr = sp->current_devp;

   sp->flags = SC_RESUME;


  IOcnt++;
  /*
   * Buffer count depends on whether scatter-gather is used or not
   */
  if(!Cmnd->use_sg){
    sp->bufstruct.b_bcount = (int)Cmnd->request_bufflen;
  }
  else {
    struct scatterlist *scatter = (struct scatterlist *)Cmnd->buffer;
    sp->bufstruct.b_bcount = 0;

    for(i=0; i < Cmnd->use_sg; i++)
      sp->bufstruct.b_bcount += scatter[i].length;
  }

  /*
   * Set read/write flag
   */
#if LINUX_VERSION_CODE > LinuxVersionCode(2,4,4)
  if(lpfc_use_data_direction) {
    if(Cmnd->sc_data_direction == SCSI_DATA_WRITE)
       sp->bufstruct.b_flags = B_WRITE;
    else
       sp->bufstruct.b_flags = B_READ;
  }
  else {
     sp->bufstruct.b_flags = fc_data_direction(Cmnd); 
  }
#else
  sp->bufstruct.b_flags = fc_data_direction(Cmnd); 
#endif

  if (Cmnd->cmnd[0] == TEST_UNIT_READY)
     sp->bufstruct.b_bcount = 0;

  /*
   * Fill in the sp struct
   */
  bcopy((void *)Cmnd->cmnd, (void *)&sp->scsi_command.scsi_cmd, 16);
        
  sp->scsi_command.scsi_length=Cmnd->cmd_len;
  sp->scsi_command.scsi_id=Cmnd->target;
  sp->scsi_command.scsi_lun=Cmnd->lun;
  if (Cmnd->device->tagged_supported) {
    switch (Cmnd->tag) {
    case HEAD_OF_QUEUE_TAG:
      sp->scsi_command.flags = HEAD_OF_Q;
      break;
    case ORDERED_QUEUE_TAG:
      sp->scsi_command.flags = ORDERED_Q;
      break;
    default:
      sp->scsi_command.flags = SIMPLE_Q;
    break;
    }
  }
  else
    sp->scsi_command.flags = 0;

  sp->timeout_value = Cmnd->timeout_per_command / fc_ticks_per_second;
  sp->adap_q_status = 0;
  sp->bufstruct.av_forw = NULL;

  retcod = 0;
  if(p_dev_ctl->device_state == DEAD) {
    retcod=DID_NO_CONNECT;
    Cmnd->result = ScsiResult(retcod, 0);
    FCSTATCTR.fcpRsvd8++;
    goto done;
  }

  /*
   * Is it a valid target?
   */
  if((dev_ptr == 0) || (dev_ptr->nodep == 0)) {
    retcod=DID_NO_CONNECT;
    Cmnd->result = ScsiResult(retcod, 0);
    FCSTATCTR.fcpRsvd4++;
    goto done;
  }

   if(dev_ptr->nodep == 0) {
      FCSTATCTR.fcpRsvd6++;
      retcod=DID_SOFT_ERROR;
   }
   else {
      if((clp[CFG_LINKDOWN_TMO].a_current == 0) || clp[CFG_HOLDIO].a_current) {
         retcod=0;
      } 
     else {
        retcod=0;
        if (binfo->fc_flag & FC_LD_TIMEOUT) {
           if(clp[CFG_NODEV_TMO].a_current == 0) {
               retcod=DID_SOFT_ERROR;
               FCSTATCTR.fcpRsvd7++;
           }
           else {
              if(dev_ptr->nodep->flags & FC_NODEV_TMO) {
                 retcod=DID_SOFT_ERROR;
                 FCSTATCTR.fcpRsvd7++;
              }
           }
        }
     }
   }
  if(retcod)
     goto done;
  retcod=DID_OK;


  if (dev_ptr->pend_head == NULL) {
    dev_ptr->pend_head = sp;
    dev_ptr->pend_tail = sp;
  } else {
    dev_ptr->pend_tail->bufstruct.av_forw = (struct buf *)sp;
    dev_ptr->pend_tail = sp;
  }    
  dev_ptr->pend_count++;

  /* 
   * put on the DEVICE_WAITING_head 
   */
  fc_enq_wait(dev_ptr);  

   /*
    * Send out the SCSI REPORT LUN command before sending the very
    * first SCSI command to that device.
    */
   if (dev_ptr->nodep->rptlunstate == REPORT_LUN_REQUIRED) {
      dev_ptr->nodep->rptlunstate = REPORT_LUN_ONGOING;
      issue_report_lun(p_dev_ctl, dev_ptr, 0);
   } else {
      if ( (dev_ptr->nodep->rptlunstate == REPORT_LUN_COMPLETE) &&
         !(dev_ptr->flags & CHK_SCSI_ABDR) && dev_ptr->numfcbufs)
         fc_issue_cmd(p_dev_ctl);
  }

  /*
   * Done
   */
done:
  if(retcod!=DID_OK) {
    dev_ptr->iodonecnt++;
    dev_ptr->errorcnt++;
    bp = (struct buf *) sp;
    bp->b_error = EIO;
    bp->b_flags |= B_ERROR;
    sp->status_validity = SC_ADAPTER_ERROR;
    sp->general_card_status = SC_SCSI_BUS_RESET;
    fc_delay_iodone(p_dev_ctl, sp);
  }
    oCmnd = Cmnd;
    Cmnd = (Scsi_Cmnd *)Cmnd->eh_timeout.data;
    oCmnd->eh_timeout.data = 0;
  }
  p_dev_ctl->qcmd_head = 0;
  p_dev_ctl->qcmd_list = 0;

  return 0;
}

/******************************************************************************
* Function name : fc_rtalloc
*
* Description   : 
* 
******************************************************************************/
_local_ int fc_rtalloc(fc_dev_ctl_t    *p_dev_ctl,
                       struct dev_info *dev_ptr)
{
    int i;
    unsigned int size;
    fc_buf_t       *fcptr;
    struct sc_buf  *sp;
    dma_addr_t      phys;
    FC_BRD_INFO       * binfo;
    MBUF_INFO * buf_info;
    MBUF_INFO bufinfo;

    binfo = &p_dev_ctl->info;
    for (i = 0; i < dev_ptr->fcp_lun_queue_depth+1 ; i++) {

      size = fc_po2(sizeof(fc_buf_t));
      phys = (dma_addr_t)((ulong)INVALID_PHYS);

      buf_info = &bufinfo;
      buf_info->size = size;
      buf_info->flags = FC_MBUF_DMA;
      buf_info->align = size;
      buf_info->phys = 0;
      buf_info->dma_handle = 0;
      buf_info->data_handle = 0;
      fc_malloc(p_dev_ctl, buf_info);
      fcptr = buf_info->virt;
      phys = (dma_addr_t)((ulong)buf_info->phys);
      if (!fcptr || is_invalid_phys((void *)((ulong)phys))) {
        return(0);
      }

      fc_bzero(fcptr, sizeof(fc_buf_t));

      fcptr->dev_ptr = dev_ptr;
      fcptr->phys_adr = (void *)((ulong)phys);

#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
      fcptr->fc_cmd_dma_handle = (ulong *)fcptr->phys_adr;
#endif
      fc_enq_fcbuf(fcptr);

      sp = (struct sc_buf *)fc_kmem_zalloc(sizeof(struct sc_buf));
      if (!sp) {
        return(0);
      }
      if(dev_ptr->scp) {
         sp->bufstruct.av_forw = dev_ptr->scp;
         dev_ptr->scp = sp;
      }
      else {
         dev_ptr->scp = sp;
         dev_ptr->scp->bufstruct.av_forw = 0;
      }
      dev_ptr->scpcnt++;
    } /* end for loop */
    return(1);
} /* end of fc_rtalloc */


/******************************************************************************
* Function name : fc_biosparam
*
* Description   : This is a weak form of biosparam
* 
******************************************************************************/
int fc_biosparam(Disk   *disk, 
                 kdev_t n, 
                 int    ip[])
{
  int size = disk->capacity;

  ip[0] = 64;
  ip[1] = 32;
  ip[2] = size >> 11;
  if (ip[2] > 1024) {
    ip[0] = 255;
    ip[1] = 63;
    ip[2] = size / (ip[0] * ip[1]);
#ifndef FC_EXTEND_TRANS_A
    if (ip[2] > 1023)
      ip[2] = 1023;
#endif
  }

  return 0;
}


/******************************************************************************
* Function name : do_fc_intr_handler
*
* Description   : Local interupt handler
* 
******************************************************************************/
void do_fc_intr_handler(int             irq, 
                        void           *dev_id, 
                        struct pt_regs *regs)
{
  struct intr *ihs;
  FC_BRD_INFO       * binfo;
  fc_dev_ctl_t * p_dev_ctl;
  void *ioa;
  volatile uint32      ha_copy;
  uint32               i;
  ulong                siflg;
  ulong                iflg;

  /*
   * If driver is unloading, we can stop processing interrupt.
   */
  if (lpfc_driver_unloading)
     return;

  ihs = (struct intr *)dev_id;
  p_dev_ctl = (fc_dev_ctl_t * )ihs;
  if(!p_dev_ctl){
    return;
  }
  
  for(i=0;i<fcinstcnt;i++) {
     if(DD_CTL.p_dev[i] == p_dev_ctl) {
        break;
     }
  }
  if(i == fcinstcnt) {
     return;
  }

  siflg = 0;
  iflg = 0;
  LPFC_LOCK_SCSI_DONE;
  LPFC_LOCK_DRIVER(13);
  binfo = &p_dev_ctl->info;
  /* Ignore all interrupts during initialization. */
  if(binfo->fc_ffstate < FC_LINK_DOWN) {
     LPFC_UNLOCK_DRIVER;
     LPFC_UNLOCK_SCSI_DONE;
     return;
  }

  ioa = FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */

  /* Read host attention register to determine interrupt source */
  ha_copy = READ_CSR_REG(binfo, FC_HA_REG(binfo, ioa));

  /* Clear Attention Sources, except ERROR (to preserve status) & LATT
   *    (ha_copy & ~HA_ERATT & ~HA_LATT);
   */
  WRITE_CSR_REG(binfo, FC_HA_REG(binfo,ioa), (ha_copy & ~(HA_LATT | HA_ERATT)));

  if (ha_copy & HA_ERATT) {     /* Link / board error */
      volatile uint32 status;

      /* do what needs to be done, get error from STATUS REGISTER */
      status = READ_CSR_REG(binfo, FC_STAT_REG(binfo, ioa));
      /* Clear Chip error bit */
      WRITE_CSR_REG(binfo, FC_HA_REG(binfo, ioa), HA_ERATT);
      if(p_dev_ctl->dpc_hstatus == 0)
         p_dev_ctl->dpc_hstatus = status;
  }

  if (ha_copy & HA_LATT) {   /* Link Attention interrupt */
      volatile uint32 control;

      if (binfo->fc_process_LA) {
         control = READ_CSR_REG(binfo, FC_HC_REG(binfo, ioa));
         control &= ~HC_LAINT_ENA;
         WRITE_CSR_REG(binfo, FC_HC_REG(binfo, ioa), control);
         /* Clear Link Attention in HA REG */
         WRITE_CSR_REG(binfo, FC_HA_REG(binfo,ioa), (volatile uint32)(HA_LATT));
      }
  }

  FC_UNMAP_MEMIO(ioa);


  p_dev_ctl->dpc_ha_copy |= ha_copy;

  {
  struct lpfc_dpc *ldp;
  ldp = &lpfc_dpc[binfo->fc_brd_no];
  if ((p_dev_ctl->power_up == 0) || (ldp->dpc_wait == NULL)) {
     do_fc_intr((struct intr *)p_dev_ctl);
     LPFC_UNLOCK_DRIVER;
     fc_flush_done_cmds(p_dev_ctl, siflg);
     LPFC_UNLOCK_SCSI_DONE;
  }
  else {
    if (ldp->dpc_active == 0) {
      LPFC_UNLOCK_DRIVER;
      LPFC_UNLOCK_SCSI_DONE;
      up(ldp->dpc_wait);
    }
    else {
       LPFC_UNLOCK_DRIVER;
       fc_flush_done_cmds(p_dev_ctl, siflg);
       LPFC_UNLOCK_SCSI_DONE;
    }
  }
  }
}

/******************************************************************************
* Function name : do_fc_intr
*
* Description   : 
*                 p_ihs  also points to device control area
******************************************************************************/
int do_fc_intr(struct intr *p_ihs)
{
   fc_dev_ctl_t * p_dev_ctl = (fc_dev_ctl_t * )p_ihs;
   volatile uint32      ha_copy;
   FC_BRD_INFO   * binfo;
   iCfgParam     * clp;
   fcipbuf_t     * mbp;
   MAILBOXQ      * mb;
   IOCBQ         * delayiocb;
   IOCBQ         * temp;
   IOCBQ         * processiocb;
   IOCBQ         * endiocb;
   int  ipri, rc;

   binfo = &BINFO;
   ipri = disable_lock(FC_LVL, &CMD_LOCK);
   binfo->fc_flag |= FC_INTR_THREAD;

   /* Read host attention register to determine interrupt source */
   ha_copy = p_dev_ctl->dpc_ha_copy;
   p_dev_ctl->dpc_ha_copy = 0;

   if (ha_copy) {
      rc = INTR_SUCC;
      binfo->fc_flag |= FC_INTR_WORK;
   } else {
      clp = DD_CTL.p_config[binfo->fc_brd_no];
      if (clp[CFG_INTR_ACK].a_current && (binfo->fc_flag&FC_INTR_WORK)) {
         rc = INTR_SUCC;  /* Just claim the first non-working interrupt */
         binfo->fc_flag &= ~FC_INTR_WORK;
      } else {
         if (clp[CFG_INTR_ACK].a_current == 2)
            rc = INTR_SUCC;  /* Always claim the interrupt */
         else
            rc = INTR_FAIL;
      }
   }

   if (binfo->fc_flag & FC_OFFLINE_MODE) {
      binfo->fc_flag &= ~FC_INTR_THREAD;
      unlock_enable(ipri, &CMD_LOCK);
      return(INTR_FAIL);
   }
   processiocb = 0;
   if(binfo->fc_delayxmit) {
      delayiocb = binfo->fc_delayxmit;
      binfo->fc_delayxmit = 0;
      endiocb = 0;
      while(delayiocb) {
         temp = delayiocb;
         delayiocb = (IOCBQ *)temp->q;
         temp->rsvd2--;
         /* If retry == 0, process IOCB */
         if(temp->rsvd2 == 0) {
            if(processiocb == 0) {
               processiocb = temp;
            }
            else {
               endiocb->q = (uchar *)temp;
            }
            endiocb = temp;
            temp->q = 0;
         }
         else {
            /* Make delayxmit point to first non-zero retry */
            if(binfo->fc_delayxmit == 0)
               binfo->fc_delayxmit = temp;
         }
      }
      if(processiocb) {
         /* Handle any delayed IOCBs */
         endiocb = processiocb;
         while(endiocb) {
            temp = endiocb;
            endiocb = (IOCBQ *)temp->q;
            temp->q = 0;
            issue_iocb_cmd(binfo, &binfo->fc_ring[FC_ELS_RING], temp);
         }
      }
   }

   if (ha_copy & HA_ERATT) {     /* Link / board error */
      unlock_enable(ipri, &CMD_LOCK);
      handle_ff_error(p_dev_ctl);
      return(rc);
   } else {
      if (ha_copy & HA_MBATT) { /* Mailbox interrupt */
         handle_mb_event(p_dev_ctl);
         if(binfo->fc_flag & FC_PENDING_RING0) {
            binfo->fc_flag &= ~FC_PENDING_RING0;
            ha_copy |= HA_R0ATT; /* event on ring 0 */
         }
      }

      if (ha_copy & HA_LATT) {   /* Link Attention interrupt */
         if (binfo->fc_process_LA) {
            handle_link_event(p_dev_ctl);
         }
      }

      if (ha_copy & HA_R0ATT) { /* event on ring 0 */
         if(binfo->fc_mbox_active == 0)
            handle_ring_event(p_dev_ctl, 0, (ha_copy & 0x0000000F));
         else
            binfo->fc_flag |= FC_PENDING_RING0;
      }

      if (ha_copy & HA_R1ATT) { /* event on ring 1 */
         /* This ring handles IP. Defer processing anything on this ring
        * till all FCP ELS traffic settles down.
        */
         if (binfo->fc_ffstate <= FC_NODE_DISC)
            binfo->fc_deferip |= (uchar)((ha_copy >> 4) & 0x0000000F);
         else
            handle_ring_event(p_dev_ctl, 1, ((ha_copy >> 4) & 0x0000000F));
      }

      if (ha_copy & HA_R2ATT) { /* event on ring 2 */
         handle_ring_event(p_dev_ctl, 2, ((ha_copy >> 8) & 0x0000000F));
      }

      if (ha_copy & HA_R3ATT) { /* event on ring 3 */
         handle_ring_event(p_dev_ctl, 3, ((ha_copy >> 12) & 0x0000000F));
      }
   }

   if((processiocb == 0) && (binfo->fc_delayxmit) &&
      (binfo->fc_mbox_active == 0)) {
      if ((mb = (MAILBOXQ * )fc_mem_get(binfo, MEM_MBOX))) {
         fc_read_rpi(binfo, (uint32)1, (MAILBOX * )mb, (uint32)0);
         if (issue_mb_cmd(binfo, (MAILBOX * )mb, MBX_NOWAIT) != MBX_BUSY) {
            fc_mem_put(binfo, MEM_MBOX, (uchar * )mb);
         }
      }
   }

   binfo->fc_flag &= ~FC_INTR_THREAD;

   while (p_dev_ctl->mbufl_head != 0) {
      binfo->fc_flag |= FC_INTR_WORK;
      mbp = (fcipbuf_t * )p_dev_ctl->mbufl_head;
      p_dev_ctl->mbufl_head = (uchar * )fcnextpkt(mbp);
      fcnextpkt(mbp) = 0;
      fc_xmit(p_dev_ctl, mbp);
   }
   p_dev_ctl->mbufl_tail = 0;
   unlock_enable(ipri, &CMD_LOCK);
   return(rc);
}       /* End do_fc_intr */

/******************************************************************************
* Function name : fc_memmap
*
* Description   : Called from fc_attach to map shared memory (SLIM and CSRs) 
*                 for adapter and to setup memory for SLI2 interface.
******************************************************************************/
int fc_memmap(fc_dev_ctl_t  *p_dev_ctl)
{
  FC_BRD_INFO    *binfo;
  struct pci_dev *pdev;
  int   reg;
  ulong            base;

  binfo = &BINFO;

  /*
   * Get PCI for board
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev){ 
    panic("no dev in pcimap\n");
    return(1);
  }
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,3)
  /* Configure DMA attributes. */
#if BITS_PER_LONG > 32
  if (pci_set_dma_mask(pdev, (uint64_t) 0xffffffffffffffff)) {
    printk("Cannot set dma mask\n");
    return(1);
  }
#else
  if (pci_set_dma_mask(pdev, (uint64_t) 0xffffffff)) {
    printk("Cannot set dma mask\n");
    return(1);
  }
#endif
#else
#if BITS_PER_LONG > 32
  pdev->dma_mask = 0xffffffffffffffff;
#endif
#endif

  /*
   * address in first register
   */
  reg = 0;
  reg = pci_getadd(pdev, reg, &base);

  /*
   * need to mask the value to get the physical address
   */
  base  &= PCI_BASE_ADDRESS_MEM_MASK; 
  DDS.bus_mem_addr = base;

  /*
   * next two registers are the control, get the first one, if doing direct io
   * if i/o port is to be used get the second 
   * Note that pci_getadd returns the correct next register
   */
  reg = pci_getadd(pdev, reg, &base);
  base  &= PCI_BASE_ADDRESS_MEM_MASK; 
  DDS.bus_io_addr =  base;
  /* 
   * Map adapter SLIM and Control Registers
   */
  binfo->fc_iomap_mem = remap_pci_mem((ulong)DDS.bus_mem_addr,FF_SLIM_SIZE);
  if(binfo->fc_iomap_mem == ((void *)(-1))){
    return (ENOMEM);  
  }

  binfo->fc_iomap_io =remap_pci_mem((ulong)DDS.bus_io_addr,FF_REG_AREA_SIZE);
  if(binfo->fc_iomap_io == ((void *)(-1))){
    unmap_pci_mem((ulong)binfo->fc_iomap_mem);    
    return (ENOMEM);  
  }


  /*
   * Setup SLI2 interface
   */
  if ((binfo->fc_sli == 2) && (binfo->fc_slim2.virt == 0)) {
    MBUF_INFO * buf_info;
    MBUF_INFO bufinfo;

    buf_info = &bufinfo;
    
    /*
     * Allocate memory for SLI-2 structures
     */
    buf_info->size = sizeof(SLI2_SLIM);
    buf_info->flags = FC_MBUF_DMA;
    buf_info->align = fcPAGESIZE;
    buf_info->dma_handle = 0;
    buf_info->data_handle = 0;
    fc_malloc(p_dev_ctl, buf_info);
    if (buf_info->virt == NULL) {

      /*
       * unmap adapter SLIM and Control Registers
       */
      unmap_pci_mem((ulong)binfo->fc_iomap_mem);
      unmap_pci_mem((ulong)binfo->fc_iomap_io);

      return (ENOMEM);
    }

    binfo->fc_slim2.virt = (uchar * )buf_info->virt;
    binfo->fc_slim2.phys = (uchar * )buf_info->phys;
    binfo->fc_slim2.data_handle = buf_info->data_handle;
    binfo->fc_slim2.dma_handle  = buf_info->dma_handle;
    fc_bzero((char *)binfo->fc_slim2.virt, sizeof(SLI2_SLIM));
  }
  return(0);
}

/******************************************************************************
* Function name : fc_unmemmap
*
* Description   : Called from fc_detach to unmap shared memory (SLIM and CSRs) 
*                 for adapter
* 
******************************************************************************/
int fc_unmemmap(fc_dev_ctl_t  *p_dev_ctl) 
{
  FC_BRD_INFO    *binfo;

  binfo = &BINFO;

  /* 
   * unmap adapter SLIM and Control Registers
   */
  unmap_pci_mem((ulong)binfo->fc_iomap_mem);
  unmap_pci_mem((ulong)binfo->fc_iomap_io);
  /*
   * Free resources associated with SLI2 interface
   */
  if (binfo->fc_slim2.virt) {
    MBUF_INFO * buf_info;
    MBUF_INFO bufinfo;

    buf_info = &bufinfo;
    buf_info->phys = (uint32 * )binfo->fc_slim2.phys;
    buf_info->data_handle = binfo->fc_slim2.data_handle;
    buf_info->dma_handle  = binfo->fc_slim2.dma_handle;
    buf_info->flags = FC_MBUF_DMA;

    buf_info->virt = (uint32 * )binfo->fc_slim2.virt;
    buf_info->size = sizeof(SLI2_SLIM);
    fc_free(p_dev_ctl, buf_info);
    binfo->fc_slim2.virt = 0;
    binfo->fc_slim2.phys = 0;
    binfo->fc_slim2.dma_handle = 0;
    binfo->fc_slim2.data_handle = 0;
  }
  return(0);
}

/******************************************************************************
* Function name : fc_pcimap
*
* Description   : Called from fc_attach to setup PCI configuration registers
* 
******************************************************************************/
int  fc_pcimap(fc_dev_ctl_t  *p_dev_ctl)
{
  FC_BRD_INFO    *binfo;
  iCfgParam      *clp;
  struct pci_dev *pdev;
  u16            cmd;

  binfo = &BINFO;
  clp = DD_CTL.p_config[binfo->fc_brd_no];

  /*
   * PCI for board
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev)
    return(1);

  /*
   * bus mastering and parity checking enabled
   */
  pci_read_config_word(pdev, PCI_COMMAND, &cmd);
  if(cmd & CMD_PARITY_CHK)  
    cmd = CMD_CFG_VALUE ;
  else
    cmd = (CMD_CFG_VALUE & ~(CMD_PARITY_CHK));


  pci_write_config_word(pdev, PCI_COMMAND, cmd);
  
  if(lpfc_pci_latency_clocks)
    pci_write_config_byte(pdev, PCI_LATENCY_TMR_REGISTER,(uchar)lpfc_pci_latency_clocks);

  if(lpfc_pci_cache_line)
    pci_write_config_byte(pdev, PCI_CACHE_LINE_REGISTER,(uchar)lpfc_pci_cache_line);

  /*
   * Get the irq from the pdev structure
   */
  DDS.bus_intr_lvl = (int)pdev->irq;
  
  return(0);
}

/******************************************************************************
* Function name : lpfc_cfg_init
*
* Description   : Called from handle_ff_error() to bring link back up
* 
******************************************************************************/
int lpfc_cfg_init(fc_dev_ctl_t  *p_dev_ctl) 
{
   FC_BRD_INFO  * binfo;
   struct lpfc_dpc *ldp;

   binfo = &BINFO;
   p_dev_ctl->dev_flag |= FC_SCHED_CFG_INIT;
   ldp = &lpfc_dpc[binfo->fc_brd_no];
   if ((ldp->dpc_active == 0) && ldp->dpc_wait)
      up(ldp->dpc_wait);
   return(0);
}

/******************************************************************************
* Function name : lpfc_kfree_skb
*
* Description   : This routine is only called by the IP portion of the driver 
*                 and the Fabric NameServer portion of the driver. It should 
*                 free a fcipbuf chain.
******************************************************************************/
int lpfc_kfree_skb(struct sk_buff *skb)
{
  struct sk_buff *sskb;

  while(skb->next) {
    sskb = skb;
    skb = skb->next;
    sskb->next = 0;
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
    if(in_interrupt()) {
       dev_kfree_skb_irq(sskb);
    }
    else {
      dev_kfree_skb(sskb);
    }
#else
   dev_kfree_skb(sskb);
#endif
  }
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
  if(in_interrupt()) {
     dev_kfree_skb_irq(skb);
  }
  else {
     dev_kfree_skb(skb);
  }
#else
  dev_kfree_skb(skb);
#endif
  return(0);
}

/******************************************************************************
* Function name : lpfc_alloc_skb
*
* Description   : 
* 
******************************************************************************/
struct sk_buff *lpfc_alloc_skb(unsigned int size)
{
  return(alloc_skb(size, GFP_ATOMIC));
}

/******************************************************************************
* Function name : fc_malloc
*
* Description   : fc_malloc  environment specific routine for memory 
*                 allocation / mapping
* The buf_info->flags field describes the memory operation requested.
*
* FC_MBUF_PHYSONLY set  requests a supplied virtual address be mapped for DMA
*     Virtual address is supplied in buf_info->virt
*     DMA mapping flag is in buf_info->align (DMA_READ, DMA_WRITE_ONLY, both)
*     The mapped physical address is returned buf_info->phys
* 
* FC_MBUF_PHYSONLY cleared requests memory be allocated for driver use and
* if FC_MBUF_DMA is set the memory is also mapped for DMA
*     The byte alignment of the memory request is supplied in buf_info->align
*     The byte size of the memory request is supplied in buf_info->size
*     The virtual address is returned buf_info->virt
*     The mapped physical address is returned buf_info->phys (for FC_MBUF_DMA)
*
******************************************************************************/
uchar *fc_malloc(fc_dev_ctl_t *p_dev_ctl, 
                 MBUF_INFO    *buf_info)
{
  FC_BRD_INFO  * binfo;
  unsigned int size;

  binfo = &BINFO;
  buf_info->phys = (void *)((ulong)INVALID_PHYS);
  buf_info->dma_handle = 0;
  if (buf_info->flags & FC_MBUF_PHYSONLY) {
    if(buf_info->virt == NULL)
      return NULL;
#if LINUX_VERSION_CODE <= LinuxVersionCode(2,4,12)
    buf_info->phys = (void *)((ulong)pci_map_single(p_dev_ctl->pcidev,
       buf_info->virt, buf_info->size, PCI_DMA_BIDIRECTIONAL));
#else
    {
    struct page *page = virt_to_page((ulong)(buf_info->virt));
    unsigned long offset = ((unsigned long)buf_info->virt & ~PAGE_MASK);

    buf_info->phys = (void *)((ulong)pci_map_page(p_dev_ctl->pcidev,
       page, offset, buf_info->size, PCI_DMA_BIDIRECTIONAL));
    }
#endif

#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
    buf_info->dma_handle = buf_info->phys;
#endif
    FCSTATCTR.fcMapCnt++;
    return((uchar * )buf_info->virt);
  }
  if((buf_info->flags & FC_MBUF_DMA))  {
    size = fc_po2(buf_info->size);
    buf_info->phys = (void *)((ulong)INVALID_PHYS);
    buf_info->virt = lpfc_kmalloc(size, GFP_ATOMIC, &buf_info->phys, p_dev_ctl);
    if (buf_info->virt) {
       if(is_invalid_phys(buf_info->phys)) {
          lpfc_kfree((unsigned int)buf_info->size, (void *)buf_info->virt, (void *)buf_info->phys, p_dev_ctl);
          buf_info->virt = 0;
       }
    }
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
    buf_info->dma_handle = buf_info->phys;
#endif
    if(buf_info->virt == 0) {
      buf_info->phys = (void *)((ulong)INVALID_PHYS);
      buf_info->dma_handle = 0;
    }
  }
  else {
    buf_info->size = ((buf_info->size + 7) & 0xfffffff8); 
    buf_info->virt = (uint32 * )lpfc_kmalloc((unsigned int)buf_info->size, GFP_ATOMIC, 0, 0); 
    if(buf_info->virt) 
      fc_bzero(buf_info->virt, buf_info->size);
    buf_info->phys = (void *)((ulong)INVALID_PHYS);
  }
  FCSTATCTR.fcMallocCnt++;
  FCSTATCTR.fcMallocByte += buf_info->size;
  return((uchar * )buf_info->virt);
}

/******************************************************************************
* Function name : fc_po2
*
* Description   : Convert size to next highest power of 2
* 
******************************************************************************/
ulong fc_po2(ulong size)
{
   ulong order;

   for (order = 1; order < size; order <<= 1);
   return(order);
}

void *lpfc_last_dma_page = 0;
int   lpfc_dma_page_offset = 0;

/******************************************************************************
* Function name : fc_free
*
* Description   : Environment specific routine for memory de-allocation/unmapping
* The buf_info->flags field describes the memory operation requested.
* FC_MBUF_PHYSONLY set  requests a supplied virtual address be unmapped
* for DMA, but not freed.
*     The mapped physical address to be unmapped is in buf_info->phys
* FC_MBUF_PHYSONLY cleared requests memory be freed and unmapped for DMA
* only if FC_MBUF_DMA is set.
*     The mapped physical address to be unmapped is in buf_info->phys
*     The virtual address to be freed is in buf_info->virt
******************************************************************************/
void fc_free(fc_dev_ctl_t *p_dev_ctl,
             MBUF_INFO    *buf_info)
{
  FC_BRD_INFO    * binfo;
  unsigned int     size;

  binfo = &BINFO;

  if (buf_info->flags & FC_MBUF_PHYSONLY) {
#if LINUX_VERSION_CODE <= LinuxVersionCode(2,4,12)
    pci_unmap_single(p_dev_ctl->pcidev,
      (ulong)(buf_info->phys), buf_info->size, PCI_DMA_BIDIRECTIONAL);
#else
    pci_unmap_page(p_dev_ctl->pcidev,
      (ulong)(buf_info->phys), buf_info->size, PCI_DMA_BIDIRECTIONAL);
#endif
    FCSTATCTR.fcUnMapCnt++;
  }
  else {
    if((buf_info->flags & FC_MBUF_DMA)) {
      size = fc_po2(buf_info->size);
      lpfc_kfree((unsigned int)buf_info->size, (void *)buf_info->virt, (void *)buf_info->phys, p_dev_ctl);
    }
    else {
      buf_info->size = ((buf_info->size + 7) & 0xfffffff8); 
      lpfc_kfree((unsigned int)buf_info->size, (void *)buf_info->virt, (void *)((ulong)INVALID_PHYS), 0);
    }
    FCSTATCTR.fcFreeCnt++;
    FCSTATCTR.fcFreeByte += buf_info->size;
  }
}

/******************************************************************************
* Function name : fc_rdpci_cmd
*
******************************************************************************/
ushort fc_rdpci_cmd(fc_dev_ctl_t  *p_dev_ctl)
{
  u16            cmd;
  struct pci_dev *pdev;

  /*
   * PCI device
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev){ 
    panic("no dev in fc_rdpci_cmd\n");
    return((ushort)0);
  }
  pci_read_config_word(pdev, PCI_COMMAND, &cmd);
  return((ushort)cmd);
}

/******************************************************************************
* Function name : fc_rdpci_32
*
******************************************************************************/
uint32 fc_rdpci_32(fc_dev_ctl_t *p_dev_ctl,
                   uint32        offset)
{
  uint32          cmd;
  struct pci_dev *pdev;

  /*
   * PCI device
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev){ 
    panic("no dev in fc_rdpci_32\n");
    return((ushort)0);
  }
  pci_read_config_dword(pdev, offset, &cmd);
  return(cmd);
}

/******************************************************************************
* Function name : fc_wrpci_cmd
*
******************************************************************************/
void fc_wrpci_cmd(fc_dev_ctl_t  *p_dev_ctl,
                  ushort         cfg_value)
{
  struct pci_dev *pdev;

  /*
   * PCI device
   */
  pdev = p_dev_ctl->pcidev;
  if(!pdev){ 
    panic("no dev in fc_wrpci_cmd\n");
    return;
  }
  pci_write_config_word(pdev, PCI_COMMAND, cfg_value);
}
/******************************************************************************
*
* Function name : lpfc_fcp_error()
*
* Description    : Handle an FCP response error
*
* Context    : called from handle_fcp_event
*          Can be called by interrupt thread.
******************************************************************************/
_static_ void lpfc_fcp_error(fc_buf_t * fcptr,
                             IOCB     * cmd)
{
   FCP_RSP       *fcpRsp = &fcptr->fcp_rsp;
   struct sc_buf *sp = fcptr->sc_bufp; 
   struct buf    *bp;
   Scsi_Cmnd     *Cmnd;

   bp = (struct buf *)sp;
   Cmnd = bp->cmnd;

   if (fcpRsp->rspStatus2 & RESID_UNDER) {
      uint32 len, resid, brd;

      if((fcptr->dev_ptr) && (fcptr->dev_ptr->nodep))
         brd = fcptr->dev_ptr->nodep->ap->info.fc_brd_no;
      else
         brd = 0;

      len = SWAP_DATA(fcptr->fcp_cmd.fcpDl);
      resid = SWAP_DATA(fcpRsp->rspResId);
      
      /* FCP residual underrun, expected <len>, residual <resid> */
      fc_log_printf_msg_vargs( brd,
             &fc_msgBlk0716,                   /* ptr to msg structure */
              fc_mes0716,                      /* ptr to msg */
               fc_msgBlk0716.msgPreambleStr,   /* begin varargs */
                len,
                 resid,
                  Cmnd->cmnd[0],
                   Cmnd->underflow);           /* end varargs */

      switch (Cmnd->cmnd[0]) {
      case TEST_UNIT_READY:
      case REQUEST_SENSE:
      case INQUIRY:
      case RECEIVE_DIAGNOSTIC:
      case READ_CAPACITY:
      case FCP_SCSI_READ_DEFECT_LIST: 
      case MDACIOCTL_DIRECT_CMD:
        break;
      default:
         if((!(fcpRsp->rspStatus2 & SNS_LEN_VALID)) &&
	    (len - resid < Cmnd->underflow)) {
            /* FCP command <cmd> residual underrun converted to error */
            fc_log_printf_msg_vargs( brd,
                   &fc_msgBlk0717,                   /* ptr to msg structure */
                    fc_mes0717,                      /* ptr to msg */
                     fc_msgBlk0717.msgPreambleStr,   /* begin varargs */
                      Cmnd->cmnd[0],
                       Cmnd->underflow,
                        len,
                         resid);                     /* end varargs */
            fcpRsp->rspStatus3 = SC_COMMAND_TERMINATED;
            fcpRsp->rspStatus2 &= ~RESID_UNDER;
            sp->scsi_status = 0;
         }
         break;
      }
   }
}
/******************************************************************************
* Function name : fc_do_iodone
*
* Description   : Return a SCSI initiated I/O to above layer
*                 when the I/O completes.
******************************************************************************/
int fc_do_iodone(struct buf *bp)
{
  Scsi_Cmnd *Cmnd;
  struct sc_buf  * sp = (struct sc_buf *) bp;
  FC_BRD_INFO       * binfo;
  iCfgParam     * clp;
  fc_dev_ctl_t      * p_dev_ctl;
  NODELIST          * nlp;
  node_t * node_ptr;
  struct Scsi_Host *host;
  struct dev_info   * dev_ptr;
  int dev_index;
  int host_status = DID_OK;

  IOcnt--;

  if(!bp) {
    return(1);
  }
  /*
   * Linux command from our buffer
   */
  Cmnd = bp->cmnd;

  /*
   * must have Cmnd and Linux completion functions
   */
  if(!Cmnd || !Cmnd->scsi_done){
    return (0);
  }

  /*
   * retrieve host adapter and device control
   */
  host = Cmnd->host;
  if(!host){
    return (0);
  }
  p_dev_ctl = (fc_dev_ctl_t *)host->hostdata[0];
  if(!p_dev_ctl){
    return (0);
  }
        
  fc_fcp_bufunmap(p_dev_ctl, sp);

  dev_index = INDEX(ZERO_PAN, Cmnd->target);
  binfo = &BINFO;
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  dev_ptr = sp->current_devp;

  if (!dev_ptr) {
    node_ptr = binfo->device_queue_hash[dev_index].node_ptr;
    goto qf;
  }

  if((node_ptr = dev_ptr->nodep) == 0) {
     node_ptr = binfo->device_queue_hash[dev_index].node_ptr;
     if(!node_ptr) {
       dev_ptr = 0;
       goto qf;
     }
  }

  if(node_ptr->rpi == 0xfffe) {
qf:
    if ((binfo->fc_ffstate > FC_LINK_DOWN) && (binfo->fc_ffstate < FC_READY))
       goto force_retry;

    if(node_ptr)
       nlp = node_ptr->nlp;
    else
       nlp = 0;
    if (nlp && 
       (binfo->fc_flag & FC_RSCN_MODE) && (binfo->fc_ffstate == FC_READY) &&
       (nlp->nlp_action & (NLP_DO_ADDR_AUTH | NLP_DO_DISC_START | NLP_DO_RSCN)))
       goto force_retry;

    if ((node_ptr) && (clp[CFG_NODEV_TMO].a_current)) {
       if(node_ptr->flags & FC_NODEV_TMO) {
#ifdef FC_NEW_EH
          Cmnd->retries = Cmnd->allowed; /* no more retries */
#endif
          host_status = DID_NO_CONNECT;
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
         if(lpfc_use_removable) {
            Cmnd->sense_buffer[0] = 0x70;
            Cmnd->sense_buffer[2] = UNIT_ATTENTION;
            Cmnd->device->removable = 1;
         }
#endif
          if(dev_ptr)
             dev_ptr->scsi_dev = (void *)Cmnd->device;
       }
       else {
#ifdef FC_NEW_EH
          Cmnd->retries = 0; /* Force retry */
#endif
          host_status = DID_BUS_BUSY;
       }
       Cmnd->result = ScsiResult(host_status, 0); 
    } 
    else  {
       if((clp[CFG_LINKDOWN_TMO].a_current)&&(binfo->fc_flag & FC_LD_TIMEOUT)) {
#ifdef FC_NEW_EH
          Cmnd->retries = Cmnd->allowed; /* no more retries */
#endif
          host_status = DID_NO_CONNECT;
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
          if(lpfc_use_removable) {
             Cmnd->sense_buffer[0] = 0x70;
             Cmnd->sense_buffer[2] = UNIT_ATTENTION;
             Cmnd->device->removable = 1;
          }
#endif
          if(dev_ptr)
             dev_ptr->scsi_dev = (void *)Cmnd->device;
       }
       else {
force_retry:
#ifdef FC_NEW_EH
          Cmnd->retries = 0; /* Force retry */
#endif
          host_status = DID_BUS_BUSY;
       }
       Cmnd->result = ScsiResult(host_status, 0); 
    }
    fc_queue_done_cmd(p_dev_ctl, bp);
    return (0);
  }

  /*
   * mark it as done, no longer required, but will leave for now
   */
  bp->isdone=1;
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
  Cmnd->resid = bp->b_resid;
#endif

  /*
   * First check if a scsi error, mid-level handles these only if DID_OK
   */
  if(sp->status_validity == SC_SCSI_ERROR){
    if(sp->scsi_status==SC_CHECK_CONDITION){
      lpfc_copy_sense(dev_ptr, bp);
    }
    else if (sp->scsi_status == SC_RESERVATION_CONFLICT) {
      host_status = DID_ERROR;
    }
    else if (sp->scsi_status == SC_BUSY_STATUS) {
#ifdef FC_NEW_EH
      Cmnd->retries = 0; /* Force retry */
#endif
      host_status = DID_BUS_BUSY;
    }
    else {
      host_status = DID_ERROR;
    }

    if((bp->b_flags & B_ERROR)) {
       if (bp->b_error == EBUSY){
          host_status = DID_OK;
          sp->scsi_status = SC_QUEUE_FULL;
       } else if (bp->b_error == EINVAL){
#ifdef FC_NEW_EH
          Cmnd->retries = 0; /* Force retry */
#endif
          host_status = DID_BUS_BUSY;
          sp->scsi_status = 0;
       }
    }

    Cmnd->result = ScsiResult(host_status,sp->scsi_status); 
    fc_queue_done_cmd(p_dev_ctl, bp);
    return (0);
  }

  /*
   * check error flag
   */
  if((bp->b_flags & B_ERROR))
    {
      switch(bp->b_error){
      case 0:
        host_status = DID_OK;
        sp->scsi_status = 0;
        break;
      case EBUSY:
        host_status = DID_BUS_BUSY;
        sp->scsi_status = 0;
        break;
      case EINVAL:
#ifdef FC_NEW_EH
        Cmnd->retries = 0; /* Force retry */
#endif
        host_status = DID_BUS_BUSY;
        sp->scsi_status = 0;
        break;
      default:
#ifdef FC_NEW_EH
        host_status = DID_ERROR;
#else
        host_status = DID_BUS_BUSY;
#endif
        break;
      }
    }

  /*
   * next hardware errors
   */
  if(sp->status_validity == SC_ADAPTER_ERROR){
#ifdef FC_NEW_EH
    host_status = DID_ERROR;
#else
    host_status = DID_BUS_BUSY;
#endif
    Cmnd->result = ScsiResult(host_status,0); 
    fc_queue_done_cmd(p_dev_ctl, bp);
    return (0);
  }

  /* 
   * if lun0_missing feature is turned on and it's inquiry to a missing
   * lun 0, then we will fake out LINUX scsi layer to allow scanning
   * of other luns.
   */
  if (lpfc_lun0_missing) { 
     if ((Cmnd->cmnd[0] == FCP_SCSI_INQUIRY) && (Cmnd->lun == 0)) {
        uchar *buf;
        buf = (uchar *)Cmnd->request_buffer;
        if( *buf == 0x7f) {
           /* Make lun unassigned and wrong type */
           *buf = 0x3;
        }
      }
   }

  if(lpfc_lun_skip) {
     /* If a LINUX OS patch to support, LUN skipping / no LUN 0, is not present,
      * this code will fake out the LINUX scsi layer to allow it to detect
      * all LUNs if there are LUN holes on a device.
      */
     if (Cmnd->cmnd[0] == FCP_SCSI_INQUIRY) {
         uchar *buf;
         buf = (uchar *)Cmnd->request_buffer;
         if(( *buf == 0x7f) || ((*buf & 0xE0) == 0x20))  {
            /* Make lun unassigned and wrong type */
            *buf = 0x3;
         }
      }
   }

  Cmnd->result = ScsiResult(host_status,sp->scsi_status); 
  fc_queue_done_cmd(p_dev_ctl, bp);

  return(0);
}

/******************************************************************************
* Function name : fc_fcp_bufunmap
*
* Description   : 
* 
******************************************************************************/
int fc_fcp_bufunmap(fc_dev_ctl_t   * p_dev_ctl,
                    struct sc_buf  * sp)
{
  struct buf *bp;
  Scsi_Cmnd   * Cmnd;
  FC_BRD_INFO       * binfo;

  binfo = &BINFO;
  bp = (struct buf *)sp;
  Cmnd = bp->cmnd;
  /* unmap DMA resources used */
  if(!(sp->flags & SC_MAPPED))
     return(0);
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
  {
  int rwflg;

  rwflg = Cmnd->sc_data_direction;

   if (Cmnd->use_sg) {
       pci_unmap_sg(p_dev_ctl->pcidev, Cmnd->request_buffer, Cmnd->use_sg, rwflg);
   }
   else if ((Cmnd->request_bufflen) && (bp->av_back)) {
#if LINUX_VERSION_CODE <= LinuxVersionCode(2,4,12)
       pci_unmap_single(p_dev_ctl->pcidev, (uint64_t)((ulong)(bp->av_back)), Cmnd->request_bufflen, rwflg);
#else
       pci_unmap_page(p_dev_ctl->pcidev, (uint64_t)((ulong)(bp->av_back)), Cmnd->request_bufflen, rwflg);
#endif
  }
  }
  
#endif 
   FCSTATCTR.fcUnMapCnt++;
   sp->flags &= ~SC_MAPPED;
   return(0);
}

/******************************************************************************
* Function name : fc_fcp_bufmap
*
* Description   : Called from issue_fcp_cmd, used to map addresses in sbp to
*                 physical addresses for the I/O.
******************************************************************************/
int fc_fcp_bufmap(fc_dev_ctl_t  * p_dev_ctl,
                  struct sc_buf * sbp,
                  fc_buf_t      * fcptr,
                  IOCBQ         * temp,
                  ULP_BDE64     * bpl,
                  dvi_t         * dev_ptr,
                  int             pend)
{
  uint32        seg_cnt, cnt, num_bmps, i, num_bde;
  int           rwflg;
  FC_BRD_INFO * binfo = &BINFO;
  iCfgParam   * clp;
  struct buf  * bp;
  RING        * rp;
  IOCB        * cmd;
  Scsi_Cmnd   * cmnd;
  ULP_BDE64   * topbpl;
  MATCHMAP    * bmp;
  MATCHMAP    * last_bmp;
  void        * physaddr;
  struct scatterlist *sgel_p;
#ifdef powerpc
  struct scatterlist *sgel_p_t0;
#endif /* endif powerpc */

  bp = (struct buf *)sbp;
  /*
    Linux command */
  cmnd = bp->cmnd;
  if(!cmnd)
    return(FCP_EXIT);
  rp = &binfo->fc_ring[FC_FCP_RING];
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  cmd = &temp->iocb;

  last_bmp = fcptr->bmp;
  num_bmps = 1;
  num_bde = 0;
  topbpl = 0;
  sgel_p = 0;

  fcptr->flags |= DATA_MAPPED;
  if (cmnd->use_sg) {
    sbp->bufstruct.av_back = 0;
    sgel_p =  (struct scatterlist *)cmnd->request_buffer;
#if  LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
    seg_cnt = cmnd->use_sg;
    rwflg = 0;
#else
    rwflg = cmnd->sc_data_direction;
 #ifdef powerpc  /* remap to get a different set of fysadds that xclud zro */
    remapsgl:
 #endif /* end if powerpc */
    seg_cnt = pci_map_sg(p_dev_ctl->pcidev, sgel_p, cmnd->use_sg, rwflg);
 #ifdef powerpc    /* check 4 zro phys address, then remap to get a diff 1 */
    for (sgel_p_t0=sgel_p, i=0; i<seg_cnt; sgel_p_t0++,i++) {
       if (!scsi_sg_dma_address(sgel_p_t0)) {
          goto remapsgl;
       }
    }
 #endif /* endif powerpc */
#endif

    FCSTATCTR.fcMapCnt++;
    cnt = 0;
    /* scatter-gather list case */
    for (i = 0; i < seg_cnt; i++) {
      if(num_bde == ((FCELSSIZE / sizeof(ULP_BDE64)) - 3)) {
         if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
            sbp->bufstruct.b_bcount = cnt;
            break;
         }
         /* Fill in continuation entry to next bpl */
         bpl->addrHigh = (uint32)putPaddrHigh(bmp->phys);
         bpl->addrHigh = PCIMEM_LONG(bpl->addrHigh);
         bpl->addrLow = (uint32)putPaddrLow(bmp->phys);
         bpl->addrLow = PCIMEM_LONG(bpl->addrLow);
         bpl->tus.f.bdeFlags = BPL64_SIZE_WORD;
         num_bde++;
         if (num_bmps == 1) {
            cmd->un.fcpi64.bdl.bdeSize += (num_bde * sizeof(ULP_BDE64));
         } else {
            topbpl->tus.f.bdeSize = (num_bde * sizeof(ULP_BDE64));
            topbpl->tus.w = PCIMEM_LONG(topbpl->tus.w);
         }
         topbpl = bpl;
         bpl = (ULP_BDE64 * )bmp->virt;
         last_bmp->fc_mptr = (void *)bmp;
         last_bmp = bmp;
         num_bde = 0;
         num_bmps++;
      }
#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
      rwflg = 0;
#else
      if(rwflg == B_WRITE)
        rwflg = SCSI_DATA_WRITE;
      else
        rwflg = SCSI_DATA_READ;
#endif

      physaddr = (void *)((ulong)scsi_sg_dma_address(sgel_p));

      bpl->addrLow = PCIMEM_LONG(putPaddrLow(physaddr));
      bpl->addrHigh = PCIMEM_LONG(putPaddrHigh(physaddr));
      bpl->tus.f.bdeSize = scsi_sg_dma_len(sgel_p);
      cnt += bpl->tus.f.bdeSize;
      if (cmd->ulpCommand == CMD_FCP_IREAD64_CR)
        bpl->tus.f.bdeFlags = BUFF_USE_RCV;
      else
        bpl->tus.f.bdeFlags = 0;
      bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
      bpl++;
      sgel_p++;
      num_bde++;
    } /* end for loop */

  }
  else {

#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
    rwflg = 0;
#else
    rwflg = cmnd->sc_data_direction;
#endif

#if LINUX_VERSION_CODE <= LinuxVersionCode(2,4,12)
    physaddr = (void *)((ulong)pci_map_single(p_dev_ctl->pcidev,
       cmnd->request_buffer, cmnd->request_bufflen, rwflg));
#else
    {
    struct page *page = virt_to_page((ulong)(cmnd->request_buffer));
    unsigned long offset = ((unsigned long)cmnd->request_buffer & ~PAGE_MASK);

 #ifdef powerpc
    remapnsg:
 #endif /* endif powerpc */
    physaddr = (void *)((ulong)pci_map_page(p_dev_ctl->pcidev,
       page, offset, cmnd->request_bufflen, rwflg));
 #ifdef powerpc
    if (!physaddr) {
       goto remapnsg;
    }
 #endif /* endif remapnsg */
    }
#endif
    FCSTATCTR.fcMapCnt++;
    sbp->bufstruct.av_back = (void *)physaddr;
    /* no scatter-gather list case */
    bpl->addrLow = PCIMEM_LONG(putPaddrLow(physaddr));
    bpl->addrHigh = PCIMEM_LONG(putPaddrHigh(physaddr));
    bpl->tus.f.bdeSize = sbp->bufstruct.b_bcount;
    if (cmd->ulpCommand == CMD_FCP_IREAD64_CR)
      bpl->tus.f.bdeFlags = BUFF_USE_RCV;
    else
      bpl->tus.f.bdeFlags = 0;
    bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
    num_bde = 1;
    bpl++;

  }

  bpl->addrHigh = 0;
  bpl->addrLow = 0;
  bpl->tus.w = 0;
  last_bmp->fc_mptr = 0;
  if (num_bmps == 1) {
     cmd->un.fcpi64.bdl.bdeSize += (num_bde * sizeof(ULP_BDE64));
  } else {
     topbpl->tus.f.bdeSize = (num_bde * sizeof(ULP_BDE64));
     topbpl->tus.w = PCIMEM_LONG(topbpl->tus.w);
  }
  cmd->ulpBdeCount = 1;
  cmd->ulpLe = 1; /* Set the LE bit in the last iocb */

  /* Queue cmd chain to last iocb entry in xmit queue */
  if (rp->fc_tx.q_first == NULL) {
    rp->fc_tx.q_first = (uchar * )temp;
  } else {
    ((IOCBQ * )(rp->fc_tx.q_last))->q  = (uchar * )temp;
  }
  rp->fc_tx.q_last = (uchar * )temp;
  rp->fc_tx.q_cnt++;

  sbp->flags |= SC_MAPPED;
  return(0);
}

/******************************************************************************
* Function name : local_timeout
*
* Description   : Local handler for watchdog timeouts
******************************************************************************/
void local_timeout(unsigned long data)
{
  struct watchdog *wdt = (struct watchdog *)data;
  fc_dev_ctl_t * p_dev_ctl;
  FC_BRD_INFO  * binfo;
  int    skip_intr, i;
  ulong  siflg;
  ulong  iflg;
  struct lpfc_dpc *ldp;

  siflg = 0;
  skip_intr = 0;
  LPFC_LOCK_SCSI_DONE;
  iflg = 0;
  LPFC_LOCK_DRIVER0;
  for (i = 0; i < MAX_FC_BRDS; i++) {
    if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
      if(p_dev_ctl->fc_ipri != 0) {
         printk("LOCK 14 failure %x %x\n",(uint32)p_dev_ctl->fc_ipri, (uint32)iflg);
      }
      p_dev_ctl->fc_ipri = 14;

      /* Check to see if the DPC was scheduled since the last clock interrupt */
      if(p_dev_ctl->dpc_cnt == p_dev_ctl->save_dpc_cnt) {
         volatile uint32      ha_copy;
         void               * ioa;

         binfo = &BINFO;
         ioa = FC_MAP_IO(&binfo->fc_iomap_io);  /* map in io registers */
         /* Read host attention register to determine interrupt source */
         ha_copy = READ_CSR_REG(binfo, FC_HA_REG(binfo, ioa));
         FC_UNMAP_MEMIO(ioa);
	 /* If there are any hardware interrupts to process, they better
	  * get done before the next clock interrupt.
	  */
         if(p_dev_ctl->dpc_ha_copy || (ha_copy & ~HA_LATT)) {
	    if(p_dev_ctl->dev_flag & FC_NEEDS_DPC) {
	       skip_intr = 1;
               /* Local_timeout Skipping clock tick */
               fc_log_printf_msg_vargs( binfo->fc_brd_no,
                      &fc_msgBlk0756,                   /* ptr to msg structure */
                       fc_mes0756,                      /* ptr to msg */
                        fc_msgBlk0756.msgPreambleStr,   /* begin varargs */
                         p_dev_ctl->dpc_ha_copy,
                          ha_copy,
                           p_dev_ctl->dpc_cnt,
                            binfo->fc_ffstate);         /* end varargs */
	        if(wdt)
                   del_timer(&wdt->timer);
	    }
	    else {
	       p_dev_ctl->dev_flag |= FC_NEEDS_DPC;
            }
         }
      }
      p_dev_ctl->save_dpc_cnt = p_dev_ctl->dpc_cnt;
    }
  }

  if(skip_intr || !wdt || !wdt->timeout_id) {
    fc_reset_timer();
    goto out;
  }
  del_timer(&wdt->timer);

  ldp = &lpfc_dpc[0];
  if (ldp->dpc_wait == NULL) {
     if(wdt->func)
       wdt->func(wdt);
  }
  else {
    lpfc_timer(0);
  }

out:
  for (i = 0; i < MAX_FC_BRDS; i++) {
    if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
      p_dev_ctl->fc_ipri = 0;
    }
  }
  LPFC_UNLOCK_DRIVER0;

  LPFC_UNLOCK_SCSI_DONE;
  for (i = 0; i < MAX_FC_BRDS; i++) {
    if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
       binfo = &BINFO;
       ldp = &lpfc_dpc[binfo->fc_brd_no];
       if ((ldp->dpc_active == 0) && ldp->dpc_wait)
          up(ldp->dpc_wait);
    }
  }
}

/******************************************************************************
* Function name : fc_reset_timer
*
* Description   : 
* 
******************************************************************************/
void fc_reset_timer(void)
{
   FCCLOCK_INFO          * clock_info;

   clock_info = &DD_CTL.fc_clock_info;
   ((struct watchdog *)(CLOCKWDT))->func = fc_timer;
   ((struct watchdog *)(CLOCKWDT))->restart = 1;
   ((struct watchdog *)(CLOCKWDT))->count = 0;
   ((struct watchdog *)(CLOCKWDT))->stopping = 0;
   /* 
    * add our watchdog timer routine to kernel's list
    */
   ((struct watchdog *)(CLOCKWDT))->timer.expires = (uint32)(HZ + jiffies);
   ((struct watchdog *)(CLOCKWDT))->timer.function = local_timeout;
   ((struct watchdog *)(CLOCKWDT))->timer.data = (unsigned long)(CLOCKWDT);
   init_timer(&((struct watchdog *)(CLOCKWDT))->timer);
   add_timer(&((struct watchdog *)(CLOCKWDT))->timer);
   return;
}

/******************************************************************************
* Function name : curtime
*
* Description   : Set memory pointed to by time, with the current time (LBOLT) 
* 
******************************************************************************/
void curtime(uint32 *time)
{
  *time = jiffies; 
}

/******************************************************************************
* Function name : fc_initpci
*
* Description   : Called by driver diagnostic interface to initialize dfc_info
* 
******************************************************************************/
int fc_initpci(struct dfc_info *di,
               fc_dev_ctl_t    *p_dev_ctl)
{
   FC_BRD_INFO   * binfo;       /* point to the binfo area */
   struct pci_dev *pdev;

   pdev = p_dev_ctl->pcidev; 
   /*
     must have the pci struct
   */
   if(!pdev) 
     return(1);
   binfo = &BINFO;

   di->fc_ba.a_onmask = (ONDI_MBOX | ONDI_RMEM | ONDI_RPCI | ONDI_RCTLREG |
      ONDI_IOINFO | ONDI_LNKINFO | ONDI_NODEINFO | ONDI_CFGPARAM |
      ONDI_CT | ONDI_HBAAPI);
   di->fc_ba.a_offmask = (OFFDI_MBOX | OFFDI_RMEM | OFFDI_WMEM | OFFDI_RPCI | 
      OFFDI_WPCI | OFFDI_RCTLREG | OFFDI_WCTLREG);

   if ((binfo->fc_flag & FC_SLI2) && (fc_diag_state == DDI_ONDI))
      di->fc_ba.a_onmask |= ONDI_SLI2;
   else
      di->fc_ba.a_onmask |= ONDI_SLI1;
#ifdef powerpc
   di->fc_ba.a_onmask |= ONDI_BIG_ENDIAN;
#else
   di->fc_ba.a_onmask |= ONDI_LTL_ENDIAN;
#endif
   di->fc_ba.a_pci=((((uint32)pdev->device) << 16) | (uint32)(pdev->vendor));
   di->fc_ba.a_pci = SWAP_LONG(di->fc_ba.a_pci);
   di->fc_ba.a_ddi = fcinstance[binfo->fc_brd_no];
   if(pdev->bus)
      di->fc_ba.a_busid = (uint32)(pdev->bus->number);
   else
      di->fc_ba.a_busid = 0;
   di->fc_ba.a_devid = (uint32)(pdev->devfn);

   bcopy((void *)lpfc_release_version, di->fc_ba.a_drvrid, 8);
   decode_firmware_rev(binfo, &VPD);
   bcopy((void *)fwrevision, di->fc_ba.a_fwname, 32);


   /* setup structures for I/O mapping */
   di->fc_iomap_io = binfo->fc_iomap_io;
   di->fc_iomap_mem = binfo->fc_iomap_mem;
   di->fc_hmap = (char *)pdev;
   return(0);
}

/******************************************************************************
* Function name : fc_readpci
*
* Description   : Called by driver diagnostic interface to copy cnt bytes from
*                 PCI configuration registers, at offset, into buf.
******************************************************************************/
int fc_readpci(struct dfc_info *di,
               uint32          offset,
               char            *buf,
               uint32          cnt)
{
  struct pci_dev *pdev;
  int i;
  uint32 *lp;
  uint32 ldata;

  if(!di->fc_hmap) return(1);
  pdev = (struct pci_dev *)di->fc_hmap;
  for(i=0; i < cnt; i++){
    lp = (uint32 *)buf;
    pci_read_config_dword(pdev,(u8)offset, (u32 *)buf);
    ldata = *lp;
    *lp = SWAP_LONG(ldata);
    buf+=4;
    offset+=4;
  }
  return(0);
}

/******************************************************************************
* Function name : fc_writepci
*
* Description   : Called by driver diagnostic interface to write cnt bytes from 
*                 buf into PCI configuration registers, starting at offset.
******************************************************************************/
int fc_writepci(struct dfc_info *di,
                uint32           offset,
                char            *buf,
                uint32           cnt)
{
  struct pci_dev *pdev;
  int i;
  uint32 *lp;
  uint32 ldata;

  if(!di->fc_hmap) return(1);
  pdev = (struct pci_dev *)di->fc_hmap;
  for(i=0; i < cnt; i++){
    lp = (uint32 *)buf;
    ldata = *lp;
    *lp = SWAP_LONG(ldata);
    pci_write_config_dword(pdev,(u8)offset, *buf); 
    buf+=4;
    offset+=4;
  }
  return(0);
}

/******************************************************************************
* Function name : copyout
*
* Description   : copy from kernel-space to a user-space
* 
******************************************************************************/
int copyout(uchar         *src,
            uchar         *dst,
            unsigned long size)
{
  copy_to_user(dst,src,size);
   return(0);
}

/******************************************************************************
* Function name : copyin
*
* Description   : copy from user-space to kernel-space
* 
******************************************************************************/
int copyin(uchar         *src,
           uchar         *dst,
           unsigned long size)
{
  copy_from_user(dst,src,size);
  return(0);
}

/******************************************************************************
* Function name : fc_getDVI
*
* Description   : get a dvi ptr from a Linux scsi cmnd
* 
******************************************************************************/
_local_ dvi_t *fc_getDVI(fc_dev_ctl_t * p_dev_ctl,
                         int            target,
                         fc_lun_t       lun)
{
  FC_BRD_INFO     * binfo;
  iCfgParam       * clp;
  struct dev_info * dev_ptr;
  struct dev_info * save_ptr;
  node_t          * node_ptr;
  NODELIST        * nlp;
  int               dev_index;
  int               report_device = 1;

  binfo = &p_dev_ctl->info;
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  dev_index = INDEX(ZERO_PAN, target);

  if (dev_index >= MAX_FC_TARGETS){
    return (0);
  }

  if (lun < 0) {
     /* LUN address out of range */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0718,                   /* ptr to msg structure */
             fc_mes0718,                      /* ptr to msg */
              fc_msgBlk0718.msgPreambleStr,   /* begin varargs */
               target,
                (uint32)lun);                 /* end varargs */
     return (0);
  }

  /*
   * Find the target from the nlplist based on SCSI ID
   */
  if((nlp = fc_findnode_scsid(binfo, NLP_SEARCH_MAPPED, target)) == 0) { 
    /* 
     * Device SCSI ID is not a valid FCP target
     */

    return (0);
  }

  /* Allocate SCSI node structure for each open FC node */
  node_ptr = binfo->device_queue_hash[dev_index].node_ptr;
  if (node_ptr == NULL) {
    if (!(node_ptr = (node_t * ) fc_kmem_zalloc(sizeof(node_t)))) {
      return (0);
    }

    /* Initialize the node ptr structure */
    node_ptr->ap  = p_dev_ctl;  /* point back to adapter struct */
    node_ptr->devno = target;
    node_ptr->lunlist = NULL;
    node_ptr->max_lun = lpfc_max_lun;

    node_ptr->last_dev = NULL;
    node_ptr->num_active_io = 0;
    node_ptr->virtRptLunData = 0;
    node_ptr->physRptLunData = 0;

    node_ptr->tgt_queue_depth = (u_int)clp[CFG_DFT_TGT_Q_DEPTH].a_current;

    node_ptr->nlp = nlp;
    node_ptr->rpi = nlp->nlp_Rpi;
    node_ptr->last_good_rpi = nlp->nlp_Rpi;
    node_ptr->scsi_id = target;
    nlp->nlp_targetp = (uchar * )node_ptr;
    binfo->device_queue_hash[dev_index].node_ptr = node_ptr;
  }

  dev_ptr = fc_find_lun(binfo, dev_index, lun);
  /* device queue structure doesn't exist yet */
  if ( dev_ptr == NULL ) {
    if (!(dev_ptr = (dvi_t * ) fc_kmem_zalloc(sizeof(dvi_t)))) {
      return (0);
    }

    /* Initialize the dev_info structure */
    dev_ptr->nodep = node_ptr;
    dev_ptr->lun_id = lun;
    dev_ptr->flags = 0;
    dev_ptr->sense_valid     = FALSE;

    /* Initialize device queues */
    dev_ptr->ABORT_BDR_fwd = NULL;
    dev_ptr->ABORT_BDR_bkwd = NULL;
    dev_ptr->DEVICE_WAITING_fwd = NULL;
    dev_ptr->pend_head = NULL;
    dev_ptr->pend_tail = NULL;
    dev_ptr->pend_count = 0;
    dev_ptr->clear_head = NULL;
    dev_ptr->clear_count = 0;
    dev_ptr->active_io_count = 0;
    dev_ptr->stop_send_io = 0;
    dev_ptr->ioctl_wakeup = 0;
    dev_ptr->qfull_retries = lpfc_qfull_retry;
    dev_ptr->first_check = FIRST_CHECK_COND | FIRST_IO;

    dev_ptr->fcp_lun_queue_depth = (u_int)clp[CFG_DFT_LUN_Q_DEPTH].a_current;
    if (dev_ptr->fcp_lun_queue_depth < 1)
      dev_ptr->fcp_lun_queue_depth = 1;

    dev_ptr->fcp_cur_queue_depth = dev_ptr->fcp_lun_queue_depth;

    /* init command state flags */
    dev_ptr->queue_state    = ACTIVE;
    dev_ptr->opened = TRUE;
    dev_ptr->ioctl_wakeup = FALSE;
    dev_ptr->ioctl_event    = EVENT_NULL;
    dev_ptr->stop_event = FALSE;

    /*
     * Create fc_bufs - allocate virtual and bus lists for use with FCP
     */
    if(fc_rtalloc(p_dev_ctl, dev_ptr) == 0) {
      fc_kmem_free(dev_ptr, sizeof(dvi_t));
      return (0);
    }

    /* Add dev_ptr to lunlist */
    if (node_ptr->lunlist == NULL) {
      node_ptr->lunlist = dev_ptr;
    } else {
      save_ptr = node_ptr->lunlist;
      while (save_ptr->next != NULL ) {
        save_ptr = save_ptr->next;
      }
      save_ptr->next = dev_ptr;
    }
    dev_ptr->next = NULL;
  }

  if(clp[CFG_DEVICE_REPORT].a_current
     && dev_ptr!=NULL && report_device &&
     (dev_ptr->nodep->nlp->nlp_type & NLP_FCP_TARGET)) {
    nlp = dev_ptr->nodep->nlp;
    printk(KERN_INFO"!lpfc%d: Acquired FCP/SCSI Target 0x%lx LUN 0x%lx , D_ID is (0x%lx)\n",
            binfo->fc_brd_no,
            (ulong)target,
            (ulong)lun,
            (ulong)(nlp->nlp_DID));
  }

  return (dev_ptr);
}

dvi_t *
fc_alloc_devp(
fc_dev_ctl_t * p_dev_ctl,
int            target,
fc_lun_t       lun)
{
   return fc_getDVI(p_dev_ctl, target, lun);
}

/******************************************************************************
* Function name : deviFree
*
* Description   : Free a dvi_t pointer
* 
******************************************************************************/
_local_ void deviFree(fc_dev_ctl_t *p_dev_ctl,
                      dvi_t        *dev_ptr,
                      node_t       *node_ptr)
{
  struct dev_info   *curDev, *prevDev;
  fc_buf_t          *curFcBuf, *tmpFcBuf;
  struct sc_buf  *sp;
  dma_addr_t     phys;
  unsigned int size;
  MBUF_INFO * buf_info;
  MBUF_INFO bufinfo;

  /*
   * First, we free up all fcbuf for this device.
   */
  curFcBuf = dev_ptr->fcbuf_head;
  while (curFcBuf != NULL) {
    tmpFcBuf = curFcBuf->fc_fwd;
    phys = (dma_addr_t)((ulong)curFcBuf->phys_adr);
    size = fc_po2(sizeof(fc_buf_t));

    buf_info = &bufinfo;
    buf_info->phys = (void *)((ulong)phys);
    buf_info->data_handle = 0;
    buf_info->dma_handle  = 0;
    buf_info->flags = FC_MBUF_DMA;
    buf_info->virt = (uint32 * )curFcBuf;
    buf_info->size = size;
    fc_free(p_dev_ctl, buf_info);
    curFcBuf = tmpFcBuf;
  } /* end while loop */
  while (dev_ptr->scp != NULL) {
    sp = dev_ptr->scp;
    dev_ptr->scp = sp->bufstruct.av_forw;
    dev_ptr->scpcnt--;
    fc_kmem_free((void *)sp, sizeof(struct sc_buf));
  }   
   
  /*
   * Next, we are going to remove this device-lun combination.
   * But we need to make sure the link list where this dev_ptr
   * belongs is not broken.
   */

  curDev = node_ptr->lunlist;
  prevDev = curDev;
  while (curDev != NULL) {
    if  (curDev == dev_ptr)
      break;
    else {
      prevDev = curDev;
      curDev = curDev->next;
    }
  } 

  if (curDev == dev_ptr) {            /* This should always pass */
    if (curDev == node_ptr->lunlist)
      node_ptr->lunlist = curDev->next;
    else
      prevDev->next = curDev->next;
  }
  fc_kmem_free((void *)dev_ptr, sizeof(dvi_t));
}
/******************************************************************************
* Function name : fc_print 
*
* Description   : 
* 
******************************************************************************/
int fc_print(  char *str,
               void *a1,
               void *a2)
{
   printk((const char *)str, a1, a2);
   return(1);
} /* fc_print */

/******************************************************************************
* Function name : log_printf_msgblk
*
* Description   : Called from common code function fc_log_printf_msg_vargs
* Note          : Input string 'str' is formatted (sprintf) by caller.
******************************************************************************/
int log_printf_msgblk(int         brdno,
                      msgLogDef * msg,
                      char      * str, /* String formatted by caller */
                      int         log_only)
{
   int            ddiinst;
   char         * mod;
   
   ddiinst = fcinstance[brdno];
   mod = "lpfc";
   if (log_only) {
      /* system buffer only */
      switch(msg->msgType) {
         case FC_LOG_MSG_TYPE_INFO:
         case FC_LOG_MSG_TYPE_WARN:
            /* These LOG messages appear in LOG file only */
            printk(KERN_INFO"!%s%d:%04d:%s\n", mod, ddiinst, msg->msgNum, str);
            break;
         case FC_LOG_MSG_TYPE_ERR_CFG:
         case FC_LOG_MSG_TYPE_ERR:
            /* These LOG messages appear on the monitor and in the LOG file */
            printk(KERN_WARNING"!%s%d:%04d:%s\n", mod, ddiinst, msg->msgNum, str);
            break;
         case FC_LOG_MSG_TYPE_PANIC:
            panic("!%s%d:%04d:%s\n", mod, ddiinst, msg->msgNum, str);
            break;
         default:
            return(0);
      }
   }
   else {
      switch(msg->msgType) {
         case FC_LOG_MSG_TYPE_INFO:
         case FC_LOG_MSG_TYPE_WARN:
            printk(KERN_INFO"!%s%d:%04d:%s\n", mod, ddiinst, msg->msgNum, str);
            break;
         case FC_LOG_MSG_TYPE_ERR_CFG:
         case FC_LOG_MSG_TYPE_ERR:
            printk(KERN_WARNING"!%s%d:%04d:%s\n", mod, ddiinst, msg->msgNum, str);
            break;
         case FC_LOG_MSG_TYPE_PANIC:
            panic("!%s%d:%04d:%s\n", mod, ddiinst, msg->msgNum, str);
            break;
         default:
            return(0);
      }
   }
   return(1);
} /* log_printf_msgblk */

/******************************************************************************
* Function name : fc_write_toio
*
******************************************************************************/
_static_ void fc_write_toio(uint32  *src, 
                            uint32  *dest_io, 
                            uint32  cnt)
{
   uint32 ldata;
   int  i;

   for (i = 0; i < (int)cnt; i += sizeof(uint32)) {
      ldata = *src++;
      writel(ldata, dest_io);
      dest_io++;
   }
   return;
}

/******************************************************************************
* Function name : fc_read_fromio
*
* Description   : 
* 
******************************************************************************/
_static_ void fc_read_fromio( uint32  *src_io, 
                              uint32  *dest, 
                              uint32  cnt)
{
   uint32 ldata;
   int  i;

   for (i = 0; i < (int)cnt; i += sizeof(uint32)) {
      ldata = readl(src_io);
      src_io++;
      *dest++ = ldata;
   }
   return;
}   

/******************************************************************************
* Function name : fc_writel
*
* Description   : 
* 
******************************************************************************/
_static_ void fc_writel(uint32 * src_io, 
                        uint32   ldata)
{
  writel(ldata, src_io);
  return;
}

/******************************************************************************
* Function name : fc_readl
*
* Description   : 
* 
******************************************************************************/
_static_ uint32 fc_readl(uint32  *src_io)
{
   uint32 ldata;

  ldata = readl(src_io);
  return(ldata);
}



#ifdef MODULE
//#include "lpfc.ver"
#endif /* MODULE */

#endif /* __GENKSYMS__ */

#ifdef MODULE

#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif

int lpfc_xmit(fc_dev_ctl_t *p_dev_ctl, struct sk_buff *skb);
int lpfc_ioctl(int cmd, void *s);

EXPORT_SYMBOL(lpfc_xmit); 
EXPORT_SYMBOL(lpfc_ioctl); 

#endif /* MODULE */

/******************************************************************************
* Function name : fc_ioctl
*
* Description   : ioctl interface to diagnostic utilities
*                 called by a special character device driver (fcLINUXdiag.c)
*                 fd is the board number (instance), and s is a cmninfo pointer
* 
******************************************************************************/
int fc_ioctl(int   cmd, 
             void *s)
{
  int rc, fd;
  fc_dev_ctl_t *p_dev_ctl;
  struct dfccmdinfo *cp = (struct dfccmdinfo *)s;
  struct cmd_input *ci = (struct cmd_input *)cp->c_datain;

  if(!cp || !ci)
    return EINVAL;
  fd = ci->c_brd;
  if(fd > DD_CTL.num_devs)
    return EINVAL;
  if(!(p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[fd]))
    return EINVAL;

  rc = dfc_ioctl(cp, ci);

  return(rc);
}

/******************************************************************************
* Function name : dfc_sleep
*
* Description   : 
* 
******************************************************************************/
int dfc_sleep(fc_dev_ctl_t   *p_dev_ctl, 
              fcEvent_header *ep)
{
   switch(ep->e_mask) {
   case FC_REG_LINK_EVENT:
      ep->e_mode |= E_SLEEPING_MODE;
      interruptible_sleep_on(&p_dev_ctl->linkwq);
      if (signal_pending (current))
         return(1);
      break;
   case FC_REG_RSCN_EVENT:
      ep->e_mode |= E_SLEEPING_MODE;
      interruptible_sleep_on(&p_dev_ctl->rscnwq);
      if (signal_pending (current))
         return(1);
      break;
   case FC_REG_CT_EVENT:
      ep->e_mode |= E_SLEEPING_MODE;
      interruptible_sleep_on(&p_dev_ctl->ctwq);
      if (signal_pending (current))
         return(1);
      break;
   }
   return(0);
}

/******************************************************************************
* Function name : dfc_wakeup
*
* Description   : 
* 
******************************************************************************/
int dfc_wakeup(fc_dev_ctl_t   *p_dev_ctl, 
               fcEvent_header *ep)
{
   switch(ep->e_mask) {
   case FC_REG_LINK_EVENT:
      ep->e_mode &= ~E_SLEEPING_MODE;
      wake_up_interruptible(&p_dev_ctl->linkwq);
      break;
   case FC_REG_RSCN_EVENT:
      ep->e_mode &= ~E_SLEEPING_MODE;
      wake_up_interruptible(&p_dev_ctl->rscnwq);
      break;
   case FC_REG_CT_EVENT:
      ep->e_mode &= ~E_SLEEPING_MODE;
      wake_up_interruptible(&p_dev_ctl->ctwq);
      break;
   }
   return(0);
}

/******************************************************************************
* Function name : lpfc_xmit
*
* Description   : 
* 
******************************************************************************/
int lpfc_xmit(fc_dev_ctl_t   *p_dev_ctl, 
              struct sk_buff *skb)
{
   int rc;
   ulong siflg, iflg;

   siflg = 0;
   LPFC_LOCK_SCSI_DONE;
   iflg = 0;
   LPFC_LOCK_DRIVER(15);
   rc = fc_xmit(p_dev_ctl, skb);
   LPFC_UNLOCK_DRIVER;
   LPFC_UNLOCK_SCSI_DONE;
   return(rc);
}

/******************************************************************************
* Function name : lpfc_ioctl
*
* Description   : 
* 
******************************************************************************/
int lpfc_ioctl(int  cmd, 
               void *s)
{
   int i, cnt, ipri;
   NETDEVICE *dev;
   fc_dev_ctl_t *p_dev_ctl;
   FC_BRD_INFO  * binfo;
   iCfgParam    * clp;
   struct lpfn_probe *lp;
   ndd_t         * p_ndd;

   cnt = 0;
   switch(cmd) {
   case LPFN_PROBE:
#ifndef MODULE
        if(lpfc_detect_called != 1) {
           lpfc_detect_called = 2; /* defer calling this till after fc_detect */
           return(1);
        }
#endif /* MODULE */

    for (i = 0; i < MAX_FC_BRDS; i++) {
           if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
              clp = DD_CTL.p_config[i];
              binfo = &BINFO;
              if(clp[CFG_NETWORK_ON].a_current == 0)
                continue;
                ipri = disable_lock(FC_LVL, &CMD_LOCK);
                if(p_dev_ctl->ihs.lpfn_dev == 0) {
                  unsigned int alloc_size;

              /* ensure 32-byte alignment of the private area */
              alloc_size = sizeof(NETDEVICE) + 31;

              dev = (NETDEVICE *) lpfc_kmalloc (alloc_size, GFP_KERNEL, 0, 0);
                  if (dev == NULL) {
                     unlock_enable(ipri, &CMD_LOCK);
                     continue;
                  }
              memset(dev, 0, alloc_size);
#if LINUX_VERSION_CODE < LinuxVersionCode(2,4,0)
                   dev->name = (char *)(dev + 1);  
                   sprintf(dev->name, "lpfn%d", binfo->fc_brd_no);

#else
                  rtnl_lock();
          strcpy(dev->name, "lpfn%d");
          if (dev_alloc_name(dev, "lpfn%d")<0) {
                     rtnl_unlock();
             lpfc_kfree(alloc_size, (void *)dev, (void *)((ulong)INVALID_PHYS), 0);
                     unlock_enable(ipri, &CMD_LOCK);
                     continue;
          }

#endif
           dev->priv = (void *)p_dev_ctl;
           p_dev_ctl->ihs.lpfn_dev = dev;

               lp = (struct lpfn_probe *)s;
               p_ndd = (ndd_t * ) & (NDD);
               /* Initialize the device structure. */
               dev->hard_start_xmit = lp->hard_start_xmit;
               dev->get_stats   = lp->get_stats;
               dev->open        = lp->open;
               dev->stop        = lp->stop;
               dev->hard_header = lp->hard_header;
               dev->rebuild_header  = lp->rebuild_header;
               dev->change_mtu  = lp->change_mtu;
                   p_ndd->nd_receive    = 
                   (void (*)(void *, struct sk_buff *, void *))(lp->receive);

               /* Assume fc header + LLC/SNAP  24 bytes */
                   dev->hard_header_len = 24;
                   dev->type        = ARPHRD_ETHER;
                   dev->mtu     = p_dev_ctl->ihs.lpfn_mtu;
                   dev->addr_len    = 6;
                   dev->tx_queue_len    = 100;

                   memset(dev->broadcast, 0xFF, 6);
                   bcopy(p_dev_ctl->phys_addr, dev->dev_addr, 6);

                   /* New-style flags */
                   dev->flags       = IFF_BROADCAST;
                   dev_init_buffers(dev);
                   register_netdevice(dev);  
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
                   rtnl_unlock();
#endif

                   cnt++;
               }
               unlock_enable(ipri, &CMD_LOCK);
            }
         }
         break;
   case LPFN_DETACH:
    for (i = 0; i < MAX_FC_BRDS; i++) {
           if((p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i])) {
          clp = DD_CTL.p_config[i];
          if(clp[CFG_NETWORK_ON].a_current == 0)
             continue;
          ipri = disable_lock(FC_LVL, &CMD_LOCK);
          if((dev=p_dev_ctl->ihs.lpfn_dev)) {
             unregister_netdev(dev);
             dev->priv = NULL;
             p_dev_ctl->ihs.lpfn_dev = 0;
             cnt++;
          }
          unlock_enable(ipri, &CMD_LOCK);
           }
    }
        break;
   case LPFN_DFC:
      break;
   default:
      return(0);
   }
   return(cnt);
}


/******************************************************************************
* Function name : lpfcdfc_init
*
* Description   : Register your major, and accept a dynamic number
* 
******************************************************************************/
int lpfcdfc_init(void)
{
    int result;
#ifdef powerpc
    fc_dev_ctl_t    *p_dev_ctl;
    MBUF_INFO       *buf_info;
    MBUF_INFO       bufinfo;
    int             i;
#endif

    result = register_chrdev(lpfc_major, "lpfcdfc", &lpfc_fops);
    if (result < 0) {
        printk(KERN_WARNING "lpfcdfc: can't get major %d\n",lpfc_major);
        return result;
    }
    if (lpfc_major == 0) lpfc_major = result; /* dynamic */

#ifdef powerpc
    for(i=0; i < MAX_FC_BRDS; i++) {
       p_dev_ctl = (fc_dev_ctl_t *)DD_CTL.p_dev[i];
       if(p_dev_ctl) {
          buf_info = &bufinfo;
          buf_info->virt = 0;
          buf_info->phys = 0;
          buf_info->flags  = (FC_MBUF_IOCTL | FC_MBUF_UNLOCK);
          buf_info->align = sizeof(void *);
          buf_info->size = 64 * 1024;
          buf_info->dma_handle = 0;

          fc_malloc(p_dev_ctl, buf_info);
          p_dev_ctl->dfc_kernel_buf = buf_info->virt;
       }
    }
#endif

    return 0;
}

/******************************************************************************
* Function name : lpfcdiag_ioctl
*
* Description   : caller must insure that cmd is the board number and arg is 
*                 the cmdinfo pointer
* 
******************************************************************************/
int lpfcdiag_ioctl(struct inode * inode, 
                   struct file  * file,
                   unsigned int   cmd, 
                   unsigned long  arg)
{
  return -fc_ioctl(cmd, (void *)arg);
}

/******************************************************************************
* Function name : lpfcdiag_open
*
* Description   : 
* 
******************************************************************************/
int lpfcdiag_open(struct inode * inode, 
                  struct file  * file)
{
   fc_dev_ctl_t *p_dev_ctl;
   struct Scsi_Host *host;

   if(((p_dev_ctl = DD_CTL.p_dev[0])) &&
      ((host = p_dev_ctl->host))) {
      lpfcdiag_cnt++;
#ifdef MODULE
      MOD_INC_USE_COUNT;
#endif /* MODULE */
   }
   return(0);
}

/******************************************************************************
* Function name : lpfcdiag_release
*
* Description   : 
* 
******************************************************************************/
int lpfcdiag_release(struct inode * inode, 
                     struct file  * file)
{
   fc_dev_ctl_t *p_dev_ctl;
   struct Scsi_Host *host;

   if(((p_dev_ctl = DD_CTL.p_dev[0])) &&
      ((host = p_dev_ctl->host))) {
      lpfcdiag_cnt--;
#ifdef MODULE
      MOD_DEC_USE_COUNT;
#endif /* MODULE */
   }
   return(0);
}

/******************************************************************************
* Function name : fc_get_dds_bind
*
* Description   : Called from fc_attach to setup binding parameters for adapter 
******************************************************************************/
int  fc_get_dds_bind(fc_dev_ctl_t *p_dev_ctl)
{
  FC_BRD_INFO    *binfo;
  iCfgParam     *clp;
  char          **arrayp;
  u_int         cnt; 

  /* 
   * Check if there are any WWNN / scsid bindings
   */
  binfo = &BINFO;
  clp = DD_CTL.p_config[binfo->fc_brd_no];
  cnt = lpfc_bind_entries;
  arrayp = lpfc_fcp_bind_WWNN;
  if(cnt && (*arrayp != 0)) {
    fc_bind_wwnn(p_dev_ctl, arrayp, cnt);
  } else {

    /* 
     * Check if there are any WWPN / scsid bindings
     */
    arrayp = lpfc_fcp_bind_WWPN;
    if(cnt && (*arrayp != 0)) {
      fc_bind_wwpn(p_dev_ctl, arrayp, cnt);
    } else {

      /*
       * Check if there are any NPortID / scsid bindings
       */
      arrayp = lpfc_fcp_bind_DID;
      if(cnt && (*arrayp != 0)) {
        fc_bind_did(p_dev_ctl, arrayp, cnt);
      } else {
         switch(clp[CFG_AUTOMAP].a_current) {
         case 2:
            p_dev_ctl->fcp_mapping = FCP_SEED_WWPN;
            break;
         case 3:
            p_dev_ctl->fcp_mapping = FCP_SEED_DID;
            break;
         default:
            p_dev_ctl->fcp_mapping = FCP_SEED_WWNN;
         }
      }
    }
  }

  clp[CFG_SCAN_DOWN].a_current = (uint32)lpfc_scandown;
   if(cnt && (*arrayp != 0) && (clp[CFG_SCAN_DOWN].a_current == 2)) {
      /* Scan-down is 2 with Persistent binding - ignoring scan-down */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0411,                   /* ptr to msg structure */
              fc_mes0411,                      /* ptr to msg */
               fc_msgBlk0411.msgPreambleStr,   /* begin varargs */
                clp[CFG_SCAN_DOWN].a_current,
                 p_dev_ctl->fcp_mapping);      /* end varargs */
      clp[CFG_SCAN_DOWN].a_current = 0;
   }
   if(clp[CFG_SCAN_DOWN].a_current > 2) {
      /* Scan-down is out of range - ignoring scan-down */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0412,                   /* ptr to msg structure */
              fc_mes0412,                      /* ptr to msg */
               fc_msgBlk0412.msgPreambleStr,   /* begin varargs */
                clp[CFG_SCAN_DOWN].a_current,
                 p_dev_ctl->fcp_mapping);      /* end varargs */
      clp[CFG_SCAN_DOWN].a_current = 0;
   }
  return(0);
}

/******************************************************************************
* Function name : fc_get_dds
*
* Description   : Called from fc_attach to setup configuration parameters for 
*                 adapter 
*                 The goal of this routine is to fill in all the a_current 
*                 members of the CfgParam structure for all configuration 
*                 parameters.
* Example:
* clp[CFG_XXX].a_current = (uint32)value;
* value might be a define, a global variable, clp[CFG_XXX].a_default,
* or some other enviroment specific way of initializing config parameters.
******************************************************************************/
int  fc_get_dds(fc_dev_ctl_t *p_dev_ctl, 
                      uint32       *pio)
{
  FC_BRD_INFO    *binfo;
  iCfgParam     *clp;
  int           i;
  int           brd;

  binfo = &BINFO;
  brd = binfo->fc_brd_no;
  clp = DD_CTL.p_config[brd];

   p_dev_ctl->open_state = NORMAL_OPEN;

  /*
   * Set everything to the defaults
   */
  for(i=0; i < NUM_CFG_PARAM; i++)
    clp[i].a_current = clp[i].a_default;

   /* Default values for I/O colaesing */
  clp[CFG_CR_DELAY].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_CR_DELAY));
  clp[CFG_CR_COUNT].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_CR_COUNT));

  clp[CFG_AUTOMAP].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_AUTOMAP));

  clp[CFG_LINK_SPEED].a_current = 
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_LINK_SPEED));

  bcopy((uchar * )"lpfc0", (uchar *)DDS.logical_name, 6);
  DDS.logical_name[4] += binfo->fc_brd_no; 
  DDS.logical_name[5] = 0;

  clp[CFG_INTR_ACK].a_current = (uint32)lpfc_intr_ack;
  clp[CFG_IDENTIFY_SELF].a_current = 0;
  clp[CFG_DEVICE_REPORT].a_current = 0;

  clp[CFG_LOG_VERBOSE].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_LOG_VERBOSE));
  clp[CFG_LOG_ONLY].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_LOG_ONLY));
 
  /* Can NOT log verbose messages until you read VERBOSE config param */  
  if((binfo->fc_flag & FC_2G_CAPABLE) == 0) {
     /* This HBA is NOT 2G_CAPABLE */ 
     if( clp[CFG_LINK_SPEED].a_current > 1) {
        /* Reset link speed to auto. 1G HBA cfg'd for 2G. */
        fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk1303,                   /* ptr to msg structure */
                 fc_mes1303,                      /* ptr to msg */
                  fc_msgBlk1303.msgPreambleStr);  /* begin & end varargs */
        clp[CFG_LINK_SPEED].a_current = LINK_SPEED_AUTO;
     }
  }
  
  clp[CFG_NUM_IOCBS].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_NUM_IOCBS));

  if (clp[CFG_NUM_IOCBS].a_current < LPFC_MIN_NUM_IOCBS) {
     /* Num-iocbs too low, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0413,                   /* ptr to msg structure */
             fc_mes0413,                      /* ptr to msg */
              fc_msgBlk0413.msgPreambleStr,   /* begin varargs */
               clp[CFG_NUM_IOCBS].a_current,
                LPFC_MIN_NUM_IOCBS);          /* end varargs */
     clp[CFG_NUM_IOCBS].a_current = LPFC_MIN_NUM_IOCBS;
  }
  if (clp[CFG_NUM_IOCBS].a_current > LPFC_MAX_NUM_IOCBS) {
     /* Num-iocbs too high, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0414,                   /* ptr to msg structure */
             fc_mes0414,                      /* ptr to msg */
              fc_msgBlk0414.msgPreambleStr,   /* begin varargs */
               clp[CFG_NUM_IOCBS].a_current,
                LPFC_MAX_NUM_IOCBS);          /* end varargs */
     clp[CFG_NUM_IOCBS].a_current = LPFC_MAX_NUM_IOCBS;
  }

  clp[CFG_NUM_BUFS].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_NUM_BUFS));

  if (clp[CFG_NUM_BUFS].a_current < LPFC_MIN_NUM_BUFS) {
     /* Num-bufs too low, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0415,                   /* ptr to msg structure */
             fc_mes0415,                      /* ptr to msg */
              fc_msgBlk0415.msgPreambleStr,   /* begin varargs */
               clp[CFG_NUM_BUFS].a_current,
                LPFC_MIN_NUM_BUFS);           /* end varargs */
     clp[CFG_NUM_BUFS].a_current = LPFC_MIN_NUM_BUFS;
  }
  if (clp[CFG_NUM_BUFS].a_current > LPFC_MAX_NUM_BUFS) {
     /* Num-bufs too high, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0416,                   /* ptr to msg structure */
             fc_mes0416,                      /* ptr to msg */
              fc_msgBlk0416.msgPreambleStr,   /* begin varargs */
               clp[CFG_NUM_BUFS].a_current,
                LPFC_MAX_NUM_BUFS);           /* end varargs */
     clp[CFG_NUM_BUFS].a_current = LPFC_MAX_NUM_BUFS;
  }

  clp[CFG_FCP_ON].a_current = 1;
  clp[CFG_DFT_TGT_Q_DEPTH].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_DFT_TGT_Q_DEPTH));
  clp[CFG_DFT_LUN_Q_DEPTH].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_DFT_LUN_Q_DEPTH));
  if (clp[CFG_DFT_TGT_Q_DEPTH].a_current > LPFC_MAX_TGT_Q_DEPTH ) {
     /* Target qdepth too high, resetting to max */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0417,                   /* ptr to msg structure */
             fc_mes0417,                      /* ptr to msg */
              fc_msgBlk0417.msgPreambleStr,   /* begin varargs */
               clp[CFG_DFT_TGT_Q_DEPTH].a_current,
                 LPFC_MAX_TGT_Q_DEPTH);       /* end varargs */
     clp[CFG_DFT_TGT_Q_DEPTH].a_current = LPFC_MAX_TGT_Q_DEPTH;
  }
  if (clp[CFG_DFT_LUN_Q_DEPTH].a_current > LPFC_MAX_LUN_Q_DEPTH ) {
     /* Lun qdepth too high, resetting to max */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0418,                   /* ptr to msg structure */
             fc_mes0418,                      /* ptr to msg */
              fc_msgBlk0418.msgPreambleStr,   /* begin varargs */
               clp[CFG_DFT_LUN_Q_DEPTH].a_current,
                LPFC_MAX_LUN_Q_DEPTH);        /* end varargs */
     clp[CFG_DFT_LUN_Q_DEPTH].a_current = LPFC_MAX_LUN_Q_DEPTH;
  }
  if (clp[CFG_DFT_LUN_Q_DEPTH].a_current == 0 ) {
     /* Lun qdepth cannot be <zero>, resetting to 1 */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0419,                   /* ptr to msg structure */
             fc_mes0419,                      /* ptr to msg */
              fc_msgBlk0419.msgPreambleStr,   /* begin varargs */
               clp[CFG_DFT_LUN_Q_DEPTH].a_current ); /* end varargs */
     clp[CFG_DFT_LUN_Q_DEPTH].a_current = 1;
  }

  clp[CFG_DQFULL_THROTTLE_UP_TIME].a_current = 
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_DQFULL_THROTTLE_UP_TIME));
  clp[CFG_DQFULL_THROTTLE_UP_INC].a_current = 
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_DQFULL_THROTTLE_UP_INC));

  clp[CFG_FIRST_CHECK].a_current = (uint32)lpfc_first_check;
  clp[CFG_FCPFABRIC_TMO].a_current = 
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_FCPFABRIC_TMO));
  if (clp[CFG_FCPFABRIC_TMO].a_current > LPFC_MAX_FABRIC_TIMEOUT) {
     /* Fcpfabric_tmo too high, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0420,                   /* ptr to msg structure */
             fc_mes0420,                      /* ptr to msg */
              fc_msgBlk0420.msgPreambleStr,   /* begin varargs */
               clp[CFG_FCPFABRIC_TMO].a_current,
                LPFC_MAX_FABRIC_TIMEOUT);     /* end varargs */
     clp[CFG_FCPFABRIC_TMO].a_current = LPFC_MAX_FABRIC_TIMEOUT;
  }

  clp[CFG_FCP_CLASS].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_FCP_CLASS));
  switch (clp[CFG_FCP_CLASS].a_current) {
  case 2:
     clp[CFG_FCP_CLASS].a_current = CLASS2;
     break;
  case 3:
     clp[CFG_FCP_CLASS].a_current = CLASS3;
     break;
  default:
     /* Fcp-class is illegal, resetting to default */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0421,                   /* ptr to msg structure */
             fc_mes0421,                      /* ptr to msg */
              fc_msgBlk0421.msgPreambleStr,   /* begin varargs */
               clp[CFG_FCP_CLASS].a_current,
                CLASS3);                      /* end varargs */
     clp[CFG_FCP_CLASS].a_current = CLASS3;
     break;
  }

  clp[CFG_USE_ADISC].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_USE_ADISC));

  clp[CFG_NO_DEVICE_DELAY].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_NO_DEVICE_DELAY));
  if (clp[CFG_NO_DEVICE_DELAY].a_current > LPFC_MAX_NO_DEVICE_DELAY) {
     /* No-device-delay too high, resetting to max */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0422,                   /* ptr to msg structure */
             fc_mes0422,                      /* ptr to msg */
              fc_msgBlk0422.msgPreambleStr,   /* begin varargs */
               clp[CFG_NO_DEVICE_DELAY].a_current,
                 LPFC_MAX_NO_DEVICE_DELAY);   /* end varargs */
     clp[CFG_NO_DEVICE_DELAY].a_current = LPFC_MAX_NO_DEVICE_DELAY;
  }

  clp[CFG_NETWORK_ON].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_NETWORK_ON));
  clp[CFG_POST_IP_BUF].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_POST_IP_BUF));

  if (clp[CFG_POST_IP_BUF].a_current < LPFC_MIN_POST_IP_BUF) {
     /* Post_ip_buf too low, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0423,                   /* ptr to msg structure */
             fc_mes0423,                      /* ptr to msg */
              fc_msgBlk0423.msgPreambleStr,   /* begin varargs */
               clp[CFG_POST_IP_BUF].a_current,
                 LPFC_MIN_POST_IP_BUF);       /* end varargs */
     clp[CFG_POST_IP_BUF].a_current = LPFC_MIN_POST_IP_BUF;
  }
  if (clp[CFG_POST_IP_BUF].a_current > LPFC_MAX_POST_IP_BUF) {
     /* Post_ip_buf too high, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0424,                   /* ptr to msg structure */
             fc_mes0424,                      /* ptr to msg */
              fc_msgBlk0424.msgPreambleStr,   /* begin varargs */
               clp[CFG_POST_IP_BUF].a_current,
                LPFC_MAX_POST_IP_BUF);        /* end varargs */
     clp[CFG_POST_IP_BUF].a_current = LPFC_MAX_POST_IP_BUF;
  }

  clp[CFG_XMT_Q_SIZE].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_XMT_Q_SIZE));
  if (clp[CFG_XMT_Q_SIZE].a_current < LPFC_MIN_XMT_QUE_SIZE) {
     /* Xmt-que_size too low, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0425,                   /* ptr to msg structure */
             fc_mes0425,                      /* ptr to msg */
              fc_msgBlk0425.msgPreambleStr,   /* begin varargs */
               clp[CFG_XMT_Q_SIZE].a_current,
                LPFC_MIN_XMT_QUE_SIZE);       /* end varargs */
     clp[CFG_XMT_Q_SIZE].a_current = LPFC_MIN_XMT_QUE_SIZE;
  }
  if (clp[CFG_XMT_Q_SIZE].a_current > LPFC_MAX_XMT_QUE_SIZE) {
     /* Xmt-que_size too high, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0426,                   /* ptr to msg structure */
             fc_mes0426,                      /* ptr to msg */
              fc_msgBlk0426.msgPreambleStr,   /* begin varargs */
               clp[CFG_XMT_Q_SIZE].a_current,
                LPFC_MAX_XMT_QUE_SIZE);       /* end varargs */
     clp[CFG_XMT_Q_SIZE].a_current = LPFC_MAX_XMT_QUE_SIZE;
  }
  binfo->fc_ring[FC_IP_RING].fc_tx.q_max = clp[CFG_XMT_Q_SIZE].a_current;

  clp[CFG_IP_CLASS].a_current = (uint32)((ulong)fc_get_cfg_param(brd, CFG_IP_CLASS));
  switch (clp[CFG_IP_CLASS].a_current) {
  case 2:
     clp[CFG_IP_CLASS].a_current = CLASS2;
     break;
  case 3:
     clp[CFG_IP_CLASS].a_current = CLASS3;
     break;
  default:
     /* Ip-class is illegal, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0427,                   /* ptr to msg structure */
             fc_mes0427,                      /* ptr to msg */
              fc_msgBlk0427.msgPreambleStr,   /* begin varargs */
               clp[CFG_IP_CLASS].a_current,
                CLASS3);                      /* end varargs */
     clp[CFG_IP_CLASS].a_current = CLASS3;
     break;
  }

  clp[CFG_ZONE_RSCN].a_current = (uint32)lpfc_zone_rscn;
  p_dev_ctl->vendor_flag = (uint32)lpfc_vendor;

  clp[CFG_HOLDIO].a_current = (uint32)((ulong)fc_get_cfg_param(brd, CFG_HOLDIO));
  clp[CFG_ACK0].a_current = (uint32)((ulong)fc_get_cfg_param(brd, CFG_ACK0));
  clp[CFG_TOPOLOGY].a_current = (uint32)((ulong)fc_get_cfg_param(brd, CFG_TOPOLOGY));

  switch (clp[CFG_TOPOLOGY].a_current) {
  case 0:
  case 1:
  case 2:
  case 4:
  case 6:
    /* topology is a valid choice */
    break;
  default:
    /* Topology is illegal, resetting */
    fc_log_printf_msg_vargs( binfo->fc_brd_no,
           &fc_msgBlk0428,                   /* ptr to msg structure */
            fc_mes0428,                      /* ptr to msg */
             fc_msgBlk0428.msgPreambleStr,   /* begin varargs */
              clp[CFG_TOPOLOGY].a_current,
               LPFC_DFT_TOPOLOGY);           /* end varargs */
    clp[CFG_TOPOLOGY].a_current = LPFC_DFT_TOPOLOGY;
    break;
  }

  clp[CFG_NODEV_TMO].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_NODEV_TMO));
  clp[CFG_DELAY_RSP_ERR].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_DELAY_RSP_ERR));
  clp[CFG_CHK_COND_ERR].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_CHK_COND_ERR));

  clp[CFG_LINKDOWN_TMO].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_LINKDOWN_TMO));
  if (clp[CFG_LINKDOWN_TMO].a_current > LPFC_MAX_LNKDWN_TIMEOUT) {
     /* Linkdown_tmo too high, resetting */
     fc_log_printf_msg_vargs( binfo->fc_brd_no,
            &fc_msgBlk0429,                   /* ptr to msg structure */
             fc_mes0429,                      /* ptr to msg */
              fc_msgBlk0429.msgPreambleStr,   /* begin varargs */
               clp[CFG_LINKDOWN_TMO].a_current,
                LPFC_MAX_LNKDWN_TIMEOUT);     /* end varargs */
     clp[CFG_LINKDOWN_TMO].a_current = LPFC_MAX_LNKDWN_TIMEOUT;
  }

  clp[CFG_LINK_SPEED].a_current =
     (uint32)((ulong)fc_get_cfg_param(brd, CFG_LINK_SPEED));

  p_dev_ctl->ihs.lpfn_mtu = lpfc_mtu;
  if((lpfc_mtu % PAGE_SIZE) == 0)
    p_dev_ctl->ihs.lpfn_rcv_buf_size = lpfc_mtu;
  else {
    p_dev_ctl->ihs.lpfn_rcv_buf_size = ((lpfc_mtu + PAGE_SIZE) & (PAGE_MASK));
    p_dev_ctl->ihs.lpfn_rcv_buf_size -= 16;
  }
  clp[CFG_NUM_NODES].a_current = clp[CFG_NUM_NODES].a_default;

   return(0);
} /* fc_get_dds */

/******************************************************************************
* Function name : fc_bind_wwpn
*
******************************************************************************/
_local_ int  fc_bind_wwpn(fc_dev_ctl_t  *p_dev_ctl, 
                                char         **arrayp, 
                                u_int          cnt)
{
  uchar        *datap, *np;
  NODELIST     *nlp;
  nodeh_t      *hp;
  NAME_TYPE     pn;
  int           i, dev_index, entry, lpfc_num, rstatus;
  unsigned int  sum;

  FC_BRD_INFO * binfo = &BINFO;
 
  p_dev_ctl->fcp_mapping = FCP_SEED_WWPN;
  np = (uchar *)&pn;

  for(entry = 0; entry < cnt; entry++) {
    datap = (uchar *)arrayp[entry];
    if(datap == 0)
       break;
    /* Determined the number of ASC hex chars in WWNN & WWPN */
    for( i = 0; i < FC_MAX_WW_NN_PN_STRING; i++) {
       if( fc_asc_to_hex( datap[i]) < 0)
          break;
    }
    if((rstatus = fc_parse_binding_entry( p_dev_ctl, datap, np, 
                   i, sizeof( NAME_TYPE),
                    FC_BIND_WW_NN_PN, &sum, entry, &lpfc_num)) > 0) {
      if( rstatus == FC_SYNTAX_OK_BUT_NOT_THIS_BRD)
         continue;

      /* WWPN binding entry <num>: Syntax error code <code> */ 
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0430,                  /* ptr to msg structure */
              fc_mes0430,                     /* ptr to msg */
               fc_msgBlk0430.msgPreambleStr,  /* begin varargs */
                entry,
                 rstatus);                    /* end varargs */
      goto out;
    }

    /* Loop through all NODELIST entries and find
     * the next available entry.
     */
    if((nlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP)) == 0) {
       /* WWPN binding entry: node table full */
       fc_log_printf_msg_vargs( binfo->fc_brd_no,
              &fc_msgBlk0432,                   /* ptr to msg structure */
               fc_mes0432,                      /* ptr to msg */
                fc_msgBlk0432.msgPreambleStr);  /* begin & end varargs */
       goto out;
    }
    fc_bzero((void *)nlp, sizeof(NODELIST));
    nlp->sync = binfo->fc_sync;
    nlp->capabilities = binfo->fc_capabilities;

    nlp->nlp_state = NLP_SEED;
    nlp->nlp_type  = NLP_SEED_WWPN | NLP_FCP_TARGET;

    nlp->id.nlp_sid = DEV_SID(sum);
    nlp->id.nlp_pan = DEV_PAN(sum);
    bcopy((uchar * )&pn, &nlp->nlp_portname, sizeof(NAME_TYPE));

    dev_index = INDEX(nlp->id.nlp_pan, nlp->id.nlp_sid);
    hp =  &binfo->device_queue_hash[dev_index];

    /* Claim SCSI ID by copying bind parameter to 
     * proper index in device_queue_hash.
     */
    bcopy(&nlp->nlp_portname, &hp->un.dev_portname, sizeof(NAME_TYPE));
    hp->node_flag = FCP_SEED_WWPN;

    fc_nlp_bind(binfo, nlp);

  out:
    np = (uchar *)&pn;
  }
  return (0);
} /* fc_bind_wwpn */

/******************************************************************************
* Function name : fc_bind_wwnn
*
* Description   : p_dev_ctl, pointer to the device control area 
* 
******************************************************************************/
_local_ int  fc_bind_wwnn(fc_dev_ctl_t  *p_dev_ctl,
                                char         **arrayp, 
                                u_int          cnt)
{
  uchar        *datap, *np;
  NODELIST     *nlp;
  nodeh_t      *hp;
  NAME_TYPE     pn;
  int           i, dev_index, entry, lpfc_num, rstatus;
  unsigned int  sum;
  
  FC_BRD_INFO * binfo = &BINFO;
 
  p_dev_ctl->fcp_mapping = FCP_SEED_WWNN;
  np = (uchar *)&pn;

  for(entry = 0; entry < cnt; entry++) {
    datap = (uchar *)arrayp[entry];
    if(datap == 0)
       break;
    /* Determined the number of ASC hex chars in WWNN & WWPN */
    for( i = 0; i < FC_MAX_WW_NN_PN_STRING; i++) {
       if( fc_asc_to_hex( datap[i]) < 0)
          break;
    }
    if((rstatus = fc_parse_binding_entry( p_dev_ctl, datap, np,
                   i, sizeof( NAME_TYPE),
                    FC_BIND_WW_NN_PN, &sum, entry, &lpfc_num)) > 0) {
      if( rstatus == FC_SYNTAX_OK_BUT_NOT_THIS_BRD)
         continue;

      /* WWNN binding entry <num>: Syntax error code <code> */ 
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0431,                  /* ptr to msg structure */
              fc_mes0431,                     /* ptr to msg */
               fc_msgBlk0431.msgPreambleStr,  /* begin varargs */
                entry,
                 rstatus);                    /* end varargs */
      goto out;
    }

    /* Loop through all NODELIST entries and find
     * the next available entry.
     */
    if((nlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP)) == 0) {
       /* WWNN binding entry: node table full */
       fc_log_printf_msg_vargs( binfo->fc_brd_no,
              &fc_msgBlk0433,                    /* ptr to msg structure */
               fc_mes0433,                       /* ptr to msg */
                fc_msgBlk0433.msgPreambleStr);   /* begin & end varargs */
       goto out;
    }
    fc_bzero((void *)nlp, sizeof(NODELIST));
    nlp->sync = binfo->fc_sync;
    nlp->capabilities = binfo->fc_capabilities;

    nlp->nlp_state = NLP_SEED;
    nlp->nlp_type  = NLP_SEED_WWNN | NLP_FCP_TARGET;
    nlp->id.nlp_sid = DEV_SID(sum);
    nlp->id.nlp_pan = DEV_PAN(sum);
    bcopy((uchar * )&pn, &nlp->nlp_nodename, sizeof(NAME_TYPE));

    dev_index = INDEX(nlp->id.nlp_pan, nlp->id.nlp_sid);
    hp =  &binfo->device_queue_hash[dev_index];

    /* Claim SCSI ID by copying bind parameter to 
     * proper index in device_queue_hash.
     */
    bcopy(&nlp->nlp_nodename, &hp->un.dev_nodename, sizeof(NAME_TYPE));
    hp->node_flag = FCP_SEED_WWNN;

    fc_nlp_bind(binfo, nlp);

  out:
    np = (uchar *)&pn;
  } /* for loop */
  return (0);
} /* fc_bind_wwnn */

/******************************************************************************
* Function name : fc_bind_did
*
* Description   : p_dev_ctl,  pointer to the device control area
* 
******************************************************************************/
_local_ int  fc_bind_did(fc_dev_ctl_t  *p_dev_ctl,
                               char         **arrayp, 
                               u_int          cnt)
{
  uchar        *datap, *np;
  NODELIST     *nlp;
  nodeh_t      *hp;
  FC_BRD_INFO  *binfo = &BINFO;
  D_ID          ndid;
  int           i, dev_index, entry, lpfc_num, rstatus;
  unsigned int  sum;

  p_dev_ctl->fcp_mapping = FCP_SEED_DID;
  ndid.un.word = 0;
  np = (uchar *)&ndid.un.word;

  for(entry = 0; entry < cnt; entry++) {
    datap = (uchar *)arrayp[entry];
    if(datap == 0)
       break;
    /* Determined the number of ASC hex chars in DID */
    for( i = 0; i < FC_MAX_DID_STRING; i++) {
       if( fc_asc_to_hex( datap[i]) < 0)
          break;
    }
    if((rstatus = fc_parse_binding_entry( p_dev_ctl, datap, np,
                   i, sizeof(D_ID),
                    FC_BIND_DID, &sum, entry, &lpfc_num)) > 0) {
      if( rstatus == FC_SYNTAX_OK_BUT_NOT_THIS_BRD)
         continue;

      /* DID binding entry <num>: Syntax error code <code> */ 
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0434,                  /* ptr to msg structure */
              fc_mes0434,                     /* ptr to msg */
               fc_msgBlk0434.msgPreambleStr,  /* begin varargs */
                entry,
                 rstatus);                    /* end varargs */
      goto out;
    }

    /* Loop through all NODELIST entries and find
     * the next available entry.
     */
    if((nlp = (NODELIST *)fc_mem_get(binfo, MEM_NLP)) == 0) {
       /* DID binding entry: node table full */ 
       fc_log_printf_msg_vargs( binfo->fc_brd_no,
              &fc_msgBlk0435,                  /* ptr to msg structure */
               fc_mes0435,                     /* ptr to msg */
                fc_msgBlk0435.msgPreambleStr); /* begin & end varargs */
       goto out;
    }
    fc_bzero((void *)nlp, sizeof(NODELIST));
    nlp->sync = binfo->fc_sync;
    nlp->capabilities = binfo->fc_capabilities;

    nlp->nlp_state = NLP_SEED;
    nlp->nlp_type  = NLP_SEED_DID | NLP_FCP_TARGET;
    nlp->id.nlp_sid = DEV_SID(sum);
    nlp->id.nlp_pan = DEV_PAN(sum);
    nlp->nlp_DID = SWAP_DATA(ndid.un.word);

    dev_index = INDEX(nlp->id.nlp_pan, nlp->id.nlp_sid);
    hp =  &binfo->device_queue_hash[dev_index];

    /* Claim SCSI ID by copying bind parameter to 
     * proper index in device_queue_hash.
     */
    hp->un.dev_did = nlp->nlp_DID;
    hp->node_flag = FCP_SEED_DID;

    fc_nlp_bind(binfo, nlp);

  out:

    np = (uchar *)&ndid.un.word;
  }
  return (0);
}

/******************************************************************************
* Function name : fc_bufmap
*
* Description   :  Maps in the specified chunk of memory, bp + len, and returns
*                  the number of mapped segments in the scatter list. Upon return
*                  phys will point to a list of physical addresses and cnt will
*                  point to a corresponding list of sizes. Handle will point to a
*                  dma handle for the I/O, if needed.
*                  This routine is only called by the IP portion of the driver to
*                  get a scatter / gather list for a specific IP packet before
*                  starting the I/O.
******************************************************************************/
int fc_bufmap(fc_dev_ctl_t *p_dev_ctl,
              uchar        *bp,
              uint32        len,
              void        **phys,
              uint32       *cnt,
              void        **handle)
{
  MBUF_INFO         * buf_info;
  MBUF_INFO        bufinfo;

  buf_info = &bufinfo;
  *handle = 0;
  buf_info->phys = (void *)((ulong)INVALID_PHYS);
  buf_info->virt = bp;
  buf_info->size = len;
  buf_info->flags  = (FC_MBUF_PHYSONLY | FC_MBUF_DMA);
  fc_malloc(p_dev_ctl, buf_info);

  if(is_invalid_phys(buf_info->phys))
    return(0);

  phys[0] = (void *) buf_info->phys;
  cnt[0] = (uint32) len;
  return(1);
}

/******************************************************************************
* Function name : fc_bufunmap
*
* Description   : This is called to unmap a piece of memory, mapped by fc_bufmap,
*                 and to free the asociated DMA handle, if needed.
******************************************************************************/
void fc_bufunmap(fc_dev_ctl_t *p_dev_ctl,
                 uchar        *addr,
                 uchar        *dmahandle,
                 uint32        len)
{
    MBUF_INFO * buf_info;
    MBUF_INFO bufinfo;

    buf_info = &bufinfo;
    buf_info->phys = (uint32 * )addr;
    buf_info->flags = (FC_MBUF_PHYSONLY | FC_MBUF_DMA);
    buf_info->size = len; 
    fc_free(p_dev_ctl, buf_info);
}

/******************************************************************************
* Function name : lpfc_scsi_selto_timeout
*
* Description   : call back function for scsi timeout
******************************************************************************/
void lpfc_scsi_selto_timeout(fc_dev_ctl_t  *p_dev_ctl,
                             void          *l1,
                             void          *l2)
{
  struct buf        *bp;

  bp = (struct buf *)l1;
  /* Set any necessary flags for buf error */
  if((bp->b_error != EBUSY) && (bp->b_error != EINVAL))
     bp->b_error = EIO;
  bp->b_flags |= B_ERROR;
  fc_do_iodone(bp);
}

/******************************************************************************
* Function name : lpfc_copy_sense
*
* Description   : call back function for scsi timeout
******************************************************************************/
int lpfc_copy_sense(dvi_t      * dev_ptr,
                    struct buf * bp)
{
   Scsi_Cmnd *cmnd;
   int sense_cnt;

   cmnd = bp->cmnd;
   if (dev_ptr->sense_length > SCSI_SENSE_BUFFERSIZE) {
      sense_cnt = SCSI_SENSE_BUFFERSIZE;
   }
   else {
      sense_cnt = dev_ptr->sense_length;
   }
   /* Copy sense data into SCSI buffer */
   bcopy(dev_ptr->sense, cmnd->sense_buffer, sense_cnt);
   dev_ptr->sense_valid = 0;
   return(0);
}

/******************************************************************************
* Function name : get_cmd_off_txq
*
* Description   : 
* 
******************************************************************************/
IOCBQ *get_cmd_off_txq(fc_dev_ctl_t *p_dev_ctl,
                       ushort        iotag)
{
   FC_BRD_INFO * binfo = &BINFO;
   IOCBQ * iocbq, *save;
   RING * rp;
   unsigned long iflag;

   iflag = lpfc_q_disable_lock(p_dev_ctl);
   rp = &binfo->fc_ring[FC_FCP_RING];
   iocbq = (IOCBQ * )(rp->fc_tx.q_first);
   save = 0;
   while (iocbq) {
      if(iocbq->iocb.ulpIoTag == iotag) {
         if(save) 
            save->q = iocbq->q;
         else
            rp->fc_tx.q_first = (uchar *)iocbq->q;

         if(rp->fc_tx.q_last == (uchar *)iocbq)
            rp->fc_tx.q_last = (uchar *)save;
          
         rp->fc_tx.q_cnt--;
         lpfc_q_unlock_enable(p_dev_ctl, iflag);
         return iocbq;
      }
      save = iocbq;
      iocbq = (IOCBQ * )iocbq->q;
   }

   lpfc_q_unlock_enable(p_dev_ctl, iflag);
   return 0;
}

/******************************************************************************
* Function name : find_cmd_in_txpq
*
* Description   : 
* 
******************************************************************************/
int find_cmd_in_txpq(fc_dev_ctl_t *p_dev_ctl,
                     Scsi_Cmnd    *cmnd)
{
   FC_BRD_INFO * binfo = &BINFO;
   struct fc_buf *fcptr, *savefc;
   dvi_t         * dev_ptr;
   IOCBQ *iocb_cmd, *iocb_cn_cmd;
   struct buf *bp;
   RING * rp;
   struct sc_buf *sp;
   int abort_stat;
   unsigned long iflag;

   iflag = lpfc_q_disable_lock(p_dev_ctl);
   rp = &binfo->fc_ring[FC_FCP_RING];
   fcptr = (struct fc_buf *)(rp->fc_txp.q_first);
   savefc = 0;
   while (fcptr) {
      if(((struct buf *)(fcptr->sc_bufp))->cmnd == cmnd) {
         dev_ptr = fcptr->dev_ptr;
         lpfc_q_unlock_enable(p_dev_ctl, iflag);
         iocb_cmd = get_cmd_off_txq(p_dev_ctl, fcptr->iotag);
         iflag = lpfc_q_disable_lock(p_dev_ctl);
         if (iocb_cmd) {
            fc_mem_put(binfo, MEM_IOCB, (uchar * )iocb_cmd);

            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            while ((iocb_cn_cmd = get_cmd_off_txq(p_dev_ctl, fcptr->iotag))) {
               fc_mem_put(binfo, MEM_IOCB, (uchar * )iocb_cn_cmd);
            }
            iflag = lpfc_q_disable_lock(p_dev_ctl);

            bp = (struct buf *)fcptr->sc_bufp;
            bp->b_error = ETIMEDOUT;
            bp->b_flags |= B_ERROR;
            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            fc_do_iodone(bp);
            iflag = lpfc_q_disable_lock(p_dev_ctl);

            if(savefc)  {
               savefc->fc_fwd = fcptr->fc_fwd; 
               if (fcptr->fc_fwd)
                  fcptr->fc_fwd->fc_bkwd = savefc;
            } else {
               rp->fc_txp.q_first = (uchar *)(fcptr->fc_fwd);
               if (fcptr->fc_fwd)
                  fcptr->fc_fwd->fc_bkwd = 0;
            }

            if(rp->fc_txp.q_last == (uchar *)fcptr) {
               rp->fc_txp.q_last = (uchar *)savefc; 
            }
          
            rp->fc_txp.q_cnt--;
            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            fc_enq_fcbuf(fcptr);
            iflag = lpfc_q_disable_lock(p_dev_ctl);
            dev_ptr->active_io_count--;
            if (dev_ptr->nodep)
               dev_ptr->nodep->num_active_io--;
            else
               panic ("abort in txp: dev_ptr->nodep is NULL\n");
         } else {
            sp = (struct sc_buf *)(fcptr->sc_bufp);
            sp->cmd_flag |=  FLAG_ABORT;
            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            abort_stat = fc_abort_xri(binfo, fcptr->dev_ptr, 
                                      fcptr->iotag, ABORT_TYPE_ABTS);
            iflag = lpfc_q_disable_lock(p_dev_ctl);
         }
         lpfc_q_unlock_enable(p_dev_ctl, iflag);
         return 1;
      } else {
         savefc = fcptr;
         fcptr = (struct fc_buf *)fcptr->fc_fwd;
      }
   }
   lpfc_q_unlock_enable(p_dev_ctl, iflag);
   return 0;
}

/******************************************************************************
* Function name : find_cmd_in_tmolist
*
* Description    : 
* 
******************************************************************************/
int find_cmd_in_tmolist(fc_dev_ctl_t *p_dev_ctl,
                        Scsi_Cmnd    *cmnd)
{
   struct buf *bp, *savebp;

   savebp = 0;
   for (bp = p_dev_ctl->timeout_head; bp != NULL; ) {
      if (bp->cmnd == cmnd) {
         if(savebp) 
            savebp->av_forw = bp->av_forw; 
         else
            p_dev_ctl->timeout_head = bp->av_forw; 

         p_dev_ctl->timeout_count--;
         bp->b_error = ETIMEDOUT;
         bp->b_flags |= B_ERROR;
         fc_do_iodone(bp);
         return 1;
      } else {
         savebp = bp;
         bp = bp->av_forw;
      }
   }

   return 0;
}

/******************************************************************************
* Function name : find_cmd_in_pendlist
*
* Description   : 
* 
******************************************************************************/
int find_cmd_in_pendlist(dvi_t     *dev_ptr,
                         Scsi_Cmnd *cmnd)
{
   struct buf *bp, *savebp;
   node_t        * nodep;

   bp = (struct buf *)dev_ptr->pend_head;
   savebp = 0;
   while (bp) {
      if (bp->cmnd == cmnd) {
         nodep = dev_ptr->nodep;
         if(savebp) 
            savebp->av_forw = bp->av_forw; 
         else
            dev_ptr->pend_head = (struct sc_buf *)(bp->av_forw); 

         if (dev_ptr->pend_tail == (struct sc_buf *)bp)
            dev_ptr->pend_tail = (struct sc_buf *)savebp;

         dev_ptr->pend_count--;
         bp->b_error = ETIMEDOUT;
         bp->b_flags |= B_ERROR;
         fc_do_iodone(bp);
         return 1;
      } else {
         savebp = bp;
         bp = bp->av_forw;
      }
   }

   return 0;
}

/******************************************************************************
* Function name : lpfc_find_cmd
*
* Description   : 
* 
******************************************************************************/
_local_ int lpfc_find_cmd(fc_dev_ctl_t *p_dev_ctl,
                          Scsi_Cmnd    *cmnd)
{
   dvi_t          * dev_ptr;
   struct sc_buf  * sp;

   sp = (struct sc_buf *)cmnd->host_scribble;
   if(sp == 0)
      return 1;
   dev_ptr = sp->current_devp;
   if(dev_ptr) {
      if (find_cmd_in_pendlist(dev_ptr, cmnd))
         goto err1;
   }

   if (find_cmd_in_txpq(p_dev_ctl, cmnd))
      goto err1;
   if (find_cmd_in_tmolist(p_dev_ctl, cmnd))
      goto err1;

   return 0;

err1:
   return 1;
}

/******************************************************************************
* Function name : lpfc_nodev
*
* Description   : 
* 
******************************************************************************/
void lpfc_nodev(unsigned long l)
{
  return;
}

/******************************************************************************
* Function name : lpfc_scsi_add_timer
*
* Description   : Copied from scsi_add_timer
******************************************************************************/
void lpfc_scsi_add_timer(Scsi_Cmnd * SCset, 
                         int         timeout)
{

    if( SCset->eh_timeout.function != NULL )
    {
        del_timer(&SCset->eh_timeout);
    }

    if(SCset->eh_timeout.data != (unsigned long) SCset) {
       SCset->eh_timeout.data = (unsigned long) SCset;
       SCset->eh_timeout.function = (void (*)(unsigned long))lpfc_nodev;
    }
    SCset->eh_timeout.expires = jiffies + timeout;

    add_timer(&SCset->eh_timeout);
}

/******************************************************************************
* Function name : lpfc_scsi_delete_timer()
*
* Purpose:        Delete/cancel timer for a given function.
*                 Copied from scsi_delete_timer()
*
* Arguments:      SCset   - command that we are canceling timer for.
*
* Returns:        Amount of time remaining before command would have timed out.
******************************************************************************/
int lpfc_scsi_delete_timer(Scsi_Cmnd * SCset)
{
  int rtn;

  rtn = jiffies - SCset->eh_timeout.expires;
  del_timer(&SCset->eh_timeout);
  SCset->eh_timeout.data = (unsigned long) NULL;
  SCset->eh_timeout.function = NULL;
  return rtn;
}

/******************************************************************************
* Function name : fc_device_changed
*
* Description    : 
* 
******************************************************************************/
int fc_device_changed(fc_dev_ctl_t    *p_dev_ctl, 
                      struct dev_info *dev_ptr)
{
   Scsi_Device *sd;

   if(lpfc_use_removable) {
      sd = (Scsi_Device *)dev_ptr->scsi_dev;
      if(sd) {
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
         sd->changed = 0;
         sd->removable = 0;
#else
         sd->online = 1;
#endif
      }
   }
   return(0);
}
/******************************************************************************
* Function name : fc_bcopy
*
* Description   : 
* 
******************************************************************************/
void fc_bcopy(void  *from, 
              void  *to, 
              ulong cnt)
{
   bcopy(from, to, cnt);
}

/******************************************************************************
* Function name : fc_bzero
*
* Description   : 
* 
******************************************************************************/
void fc_bzero(void  *from, 
              ulong cnt)
{
  memset(from,0,(size_t)cnt);
}

/******************************************************************************
* Function name : fc_copyout
*
* Description   : 
* 
******************************************************************************/
int fc_copyout(uchar *from, 
               uchar *to, 
               ulong cnt)
{
   return(copyout(from, to, cnt));
}

/******************************************************************************
* Function name : fc_copyin
*
* Description   : 
* 
******************************************************************************/
int fc_copyin(uchar *from, 
              uchar *to, 
              ulong  cnt)
{
   return(copyin(from, to, cnt));
}

/******************************************************************************
* Function name : lpfc_mpdata_sync
*
* Description   : 
* 
******************************************************************************/
void lpfc_mpdata_sync(fc_dev_ctl_t *p_dev_ctl, 
                      void         *h, 
                      int           a, 
                      int           b, 
                      int           c)
{
#if LINUX_VERSION_CODE >= LinuxVersionCode(2,4,0)
   if(c == 1)
      c = PCI_DMA_FROMDEVICE;
   else
      c = PCI_DMA_TODEVICE;
   if(b)
      fc_pci_dma_sync_single(p_dev_ctl->pcidev, (dma_addr_t)((ulong)h), b, c);
   else
      fc_pci_dma_sync_single(p_dev_ctl->pcidev, (dma_addr_t)((ulong)h), 4096, c);
#endif
}

/******************************************************************************
* Function name : lpfc_ip_rcvsz
*
* Description   : 
* 
******************************************************************************/
int lpfc_ip_rcvsz(fc_dev_ctl_t  *p_dev_ctl)
{
    return(p_dev_ctl->ihs.lpfn_rcv_buf_size);
}

/******************************************************************************
* Function name : fc_dpc_lstchk
*
* Description   : Since abort, device reset, bus reset, and host reset dpc lists 
*                 all use reset_chain for linking, double check to make sure 
*                 LINUX doesn't use the same Cmnd for multiple resets / aborts.
******************************************************************************/
int fc_dpc_lstchk(fc_dev_ctl_t  * p_dev_ctl,
                  Scsi_Cmnd     * Cmnd)
{
  Scsi_Cmnd         * aCmnd;
  Scsi_Cmnd         * bCmnd;

  aCmnd = (Scsi_Cmnd *)p_dev_ctl->abort_head;
  bCmnd = 0;
  while(aCmnd) {
     /* Check for duplicate on abort list */
     if(aCmnd == Cmnd) {
        if(bCmnd == 0) {
           p_dev_ctl->abort_head = (void *)Cmnd->reset_chain;
        }
        else {
           bCmnd->reset_chain = Cmnd->reset_chain;
        }
        if(Cmnd == (Scsi_Cmnd *)p_dev_ctl->abort_list)
           p_dev_ctl->abort_list = (void *)bCmnd;
        Cmnd->reset_chain = 0;
        return(1);
     }
     bCmnd = aCmnd;
     aCmnd = aCmnd->reset_chain;
  }
  aCmnd = (Scsi_Cmnd *)p_dev_ctl->rdev_head;
  bCmnd = 0;
  while(aCmnd) {
     /* Check for duplicate on device reset list */
     if(aCmnd == Cmnd) {
        if(bCmnd == 0) {
           p_dev_ctl->rdev_head = (void *)Cmnd->reset_chain;
        }
        else {
           bCmnd->reset_chain = Cmnd->reset_chain;
        }
        if(Cmnd == (Scsi_Cmnd *)p_dev_ctl->rdev_list)
           p_dev_ctl->rdev_list = (void *)bCmnd;
        Cmnd->reset_chain = 0;
        return(2);
     }
     bCmnd = aCmnd;
     aCmnd = aCmnd->reset_chain;
  }
  aCmnd = (Scsi_Cmnd *)p_dev_ctl->rbus_head;
  bCmnd = 0;
  while(aCmnd) {
     /* Check for duplicate on bus reset list */
     if(aCmnd == Cmnd) {
        if(bCmnd == 0) {
           p_dev_ctl->rbus_head = (void *)Cmnd->reset_chain;
        }
        else {
           bCmnd->reset_chain = Cmnd->reset_chain;
        }
        if(Cmnd == (Scsi_Cmnd *)p_dev_ctl->rbus_list)
           p_dev_ctl->rbus_list = (void *)bCmnd;
        Cmnd->reset_chain = 0;
        return(3);
     }
     bCmnd = aCmnd;
     aCmnd = aCmnd->reset_chain;
  }
  aCmnd = (Scsi_Cmnd *)p_dev_ctl->rhst_head;
  bCmnd = 0;
  while(aCmnd) {
     /* Check for duplicate on host reset list */
     if(aCmnd == Cmnd) {
        if(bCmnd == 0) {
           p_dev_ctl->rhst_head = (void *)Cmnd->reset_chain;
        }
        else {
           bCmnd->reset_chain = Cmnd->reset_chain;
        }
        if(Cmnd == (Scsi_Cmnd *)p_dev_ctl->rhst_list)
           p_dev_ctl->rhst_list = (void *)bCmnd;
        Cmnd->reset_chain = 0;
        return(4);
     }
     bCmnd = aCmnd;
     aCmnd = aCmnd->reset_chain;
  }
  return(0);
}

/******************************************************************************
* Function name : fc_timer
*
* Description   : This function will be called by the driver every second.
******************************************************************************/
_static_ void lpfc_timer(void *p)
{
   fc_dev_ctl_t * p_dev_ctl;
   FCCLOCK_INFO * clock_info;
   ulong tix;
   FCCLOCK * x;
   int  ipri;

   clock_info = &DD_CTL.fc_clock_info;
   ipri = disable_lock(CLK_LVL, &CLOCK_LOCK);

   /*
   ***  Increment time_sample value
   */
   clock_info->ticks++;

   x = clock_info->fc_clkhdr.cl_f;

   /* counter for propagating negative values */
   tix = 0;
   /* If there are expired clocks */
   if (x != (FCCLOCK * ) & clock_info->fc_clkhdr) {
      x->cl_tix = x->cl_tix - 1;
      if (x->cl_tix <= 0) {
         /* Loop thru all clock blocks */
         while (x != (FCCLOCK * ) & clock_info->fc_clkhdr) {
            x->cl_tix += tix;
            /* If # of ticks left > 0, break out of loop */
            if (x->cl_tix > 0) 
               break;
            tix = x->cl_tix;

            fc_deque(x);
            /* Decrement count of unexpired clocks */
            clock_info->fc_clkhdr.count--;

            unlock_enable(ipri, &CLOCK_LOCK);

            p_dev_ctl = x->cl_p_dev_ctl;

            if(p_dev_ctl) {
               /* Queue clock blk to appropriate dpc to be processed */
               if(p_dev_ctl->qclk_head == NULL) {
                 p_dev_ctl->qclk_head = (void *)x;
                 p_dev_ctl->qclk_list = (void *)x;
               } else {
                 ((FCCLOCK *)(p_dev_ctl->qclk_list))->cl_fw = x;
                 p_dev_ctl->qclk_list = (void *)x;
               }
               x->cl_fw = NULL;
            }
            else {
               /* Call timeout routine */
               (*x->cl_func) (p_dev_ctl, x->cl_arg1, x->cl_arg2);
               /* Release clock block */
               fc_clkrelb(p_dev_ctl, x);
            }

            ipri = disable_lock(CLK_LVL, &CLOCK_LOCK);

            x = clock_info->fc_clkhdr.cl_f;
         }
      }
   }
   unlock_enable(ipri, &CLOCK_LOCK);
   fc_reset_timer();
}

/******************************************************************************
* Function name : do_fc_timer
*
* Description   : 
* 
******************************************************************************/
int do_fc_timer(fc_dev_ctl_t *p_dev_ctl)
{
  FCCLOCK         *x;
  FCCLOCK         *cb;

  cb = (FCCLOCK *)p_dev_ctl->qclk_head;
  while(cb) {
     x = cb;
     cb = cb->cl_fw;
     /* Call timeout routine */
     (*x->cl_func) (p_dev_ctl, x->cl_arg1, x->cl_arg2);
     /* Release clock block */
     fc_clkrelb(p_dev_ctl, x);
  }
  p_dev_ctl->qclk_head = 0;
  p_dev_ctl->qclk_list = 0;
  return(0);
}

/******************************************************************************
* Function name : lpfc_kmalloc
*
* Description   : 
* 
******************************************************************************/
void * lpfc_kmalloc(unsigned int   size, 
                    unsigned int   type, 
                    void         **pphys,
                    fc_dev_ctl_t  *p_dev_ctl)
{
   FC_BRD_INFO        * binfo;
   void               * pcidev;
   void               * virt;
   struct fc_mem_pool * fmp;
   dma_addr_t phys;
   int i, instance;

/* printk("lpfc_kmalloc: %d %d %lx %lx\n", size, type, pphys, p_dev_ctl);
*/
   if(pphys == 0) {
      virt = (void *)kmalloc(size, type);
      return(virt);
   }
   if(p_dev_ctl == 0) {
      /* lpfc_kmalloc: Bad p_dev_ctl */
      fc_log_printf_msg_vargs( 0,             /* force brd 0, no p_dev_ctl */
             &fc_msgBlk1201,                  /* ptr to msg structure */
              fc_mes1201,                     /* ptr to msg */
               fc_msgBlk1201.msgPreambleStr,  /* begin varargs */
                size,
                 type,
                  fc_idx_dmapool[0]);         /* end varargs */
      return(0);
   }
   instance = p_dev_ctl->info.fc_brd_no;
   pcidev = p_dev_ctl->pcidev;
   binfo = &BINFO;

   if(size > FC_MAX_SEGSZ) {
      /* lpfc_kmalloc: Bad size */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk1202,                      /* ptr to msg structure */
              fc_mes1202,                         /* ptr to msg */
               fc_msgBlk1202.msgPreambleStr,      /* begin varargs */
                size,
                 type,
                  fc_idx_dmapool[instance]);      /* end varargs */
      return(0);
   }
top:
   fmp = fc_mem_dmapool[instance];
   for(i=0;i<=fc_idx_dmapool[instance];i++) {
      fmp = (fc_mem_dmapool[instance] + i);
      if((fmp->p_virt == 0) || (fmp->p_left >= size))
         break;
   }
   if(i == (fc_size_dmapool[instance] - 2)) {
      /* Lets make it bigger */
     fc_size_dmapool[instance] += FC_MAX_POOL;
     fmp = kmalloc((sizeof(struct fc_mem_pool) * fc_size_dmapool[instance]),
       GFP_ATOMIC);
     if(fmp) {
       fc_bzero((void *)fmp,
         (sizeof(struct fc_mem_pool) * fc_size_dmapool[instance]));
       fc_bcopy((void *)fc_mem_dmapool[instance], fmp, 
         (sizeof(struct fc_mem_pool) * (fc_size_dmapool[instance]-FC_MAX_POOL)));
       kfree(fc_mem_dmapool[instance]);
       fc_mem_dmapool[instance] = fmp;
       goto top;
     }
     goto out;
   }

   if(fmp->p_virt == 0) {
      virt = pci_alloc_consistent(pcidev, FC_MAX_SEGSZ, &phys);
      if(virt) {
         fmp->p_phys = (void *)((ulong)phys);
         fmp->p_virt = virt;
         fmp->p_refcnt = 0;
         fmp->p_left = (ushort)FC_MAX_SEGSZ;
         if(i == fc_idx_dmapool[instance])
            if(i < (fc_size_dmapool[instance] - 2))
               fc_idx_dmapool[instance]++;
      }
      else {
         /* lpfc_kmalloc: Bad virtual addr */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk1204,                   /* ptr to msg structure */
                 fc_mes1204,                      /* ptr to msg */
                  fc_msgBlk1204.msgPreambleStr,   /* begin varargs */
                   i,
                    size,
                     type,
                      fc_idx_dmapool[instance]);  /* end varargs */
         return(0);
      }
   }

   if(fmp->p_left >= size) {
      fmp->p_refcnt++;
      virt = (void *)((uchar *)fmp->p_virt + FC_MAX_SEGSZ - fmp->p_left);
      phys = (dma_addr_t)(ulong)((uchar *)fmp->p_phys + FC_MAX_SEGSZ - fmp->p_left);
      *pphys = (void *)((ulong)phys);
      fmp->p_left -= size;
      return(virt);
   }
out:
   /* lpfc_kmalloc: dmapool FULL */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk1205,                         /* ptr to msg structure */
           fc_mes1205,                            /* ptr to msg */
            fc_msgBlk1205.msgPreambleStr,         /* begin varargs */
             i,
              size,
               type,
                fc_idx_dmapool[instance]);        /* end varargs */
   return(0);
}

/******************************************************************************
* Function name : lpfc_kfree
*
* Description   : 
* 
******************************************************************************/
void lpfc_kfree(unsigned int  size, 
                void         *virt,
                void         *phys,
                fc_dev_ctl_t *p_dev_ctl)
{
   FC_BRD_INFO        * binfo;
   struct fc_mem_pool * fmp;
   void * pcidev;
   int i, instance;

   if(is_invalid_phys(phys)) {
      kfree(virt);
      return;
   }

   if(p_dev_ctl == 0) {
      /* lpfc_kfree: Bad p_dev_ctl */
      fc_log_printf_msg_vargs( 0,                /* force brd 0, no p_dev_ctl */
             &fc_msgBlk1206,                     /* ptr to msg structure */
              fc_mes1206,                        /* ptr to msg */
               fc_msgBlk1206.msgPreambleStr,     /* begin varargs */
                size,
                 fc_idx_dmapool[0]);             /* end varargs */
      return;
   }

   instance = p_dev_ctl->info.fc_brd_no;
   pcidev = p_dev_ctl->pcidev;
   binfo = &BINFO;


   for(i=0;i<fc_idx_dmapool[instance];i++) {
      fmp = (fc_mem_dmapool[instance] + i);
      if((virt >= fmp->p_virt) &&
         (virt < (void *)((uchar *)fmp->p_virt + FC_MAX_SEGSZ))) {
         fmp->p_refcnt--;
         if(fmp->p_refcnt == 0) {
            pci_free_consistent(pcidev, FC_MAX_SEGSZ,
               fmp->p_virt, (dma_addr_t)((ulong)fmp->p_phys));
            fc_bzero((void *)fmp, sizeof(struct fc_mem_pool));
         }
         return;
      }
   }
   /* lpfc_kfree: NOT in dmapool */
   fc_log_printf_msg_vargs( binfo->fc_brd_no,
          &fc_msgBlk1207,                         /* ptr to msg structure */
           fc_mes1207,                            /* ptr to msg */
            fc_msgBlk1207.msgPreambleStr,         /* begin varargs */
             (uint32)((ulong)virt),
               size,
                fc_idx_dmapool[instance]);        /* end varargs */
   return;
} /* lpfc_kfree */

MODULE_LICENSE("GPL");
