/*************************************************** */
/* Rule Set Based Access Control                     */
/* Implementation of the Access Control Decision     */
/* Facility (ADF) - Malware Scan                     */
/* File: rsbac/adf/ms/main.c                         */
/*                                                   */
/* Author and (c) 1999-2003: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 11/Mar/2003                        */
/*************************************************** */

#include <linux/string.h>
#include <linux/module.h>
#include <linux/types.h>
#include <asm/uaccess.h>
#include <rsbac/types.h>
#include <rsbac/aci.h>
#include <rsbac/adf.h>
#include <rsbac/adf_main.h>
#include <rsbac/debug.h>
#include <rsbac/error.h>
#include <rsbac/ms_strings.h>
#include <rsbac/helpers.h>
#include <rsbac/getname.h>
#include <rsbac/net_getname.h>
#include <rsbac/rkmem.h>
#include <rsbac/proc_fs.h>
#include <linux/in.h>

/************************************************* */
/*           Global Variables                      */
/************************************************* */

static rsbac_ms_all_malware_t all_malware = RSBAC_MS_ALL_STRINGS;

#ifdef CONFIG_RSBAC_MS_EXT
EXPORT_SYMBOL(rsbac_ms_do_scan);
EXPORT_SYMBOL(rsbac_ms_scan_level);
#endif
rsbac_ms_do_scan_t * rsbac_ms_do_scan = NULL;
#if defined(CONFIG_RSBAC_MS_EXT_FPROTD) || defined(CONFIG_RSBAC_MS_EXT_CLAMD)
int rsbac_ms_scan_level = 0;
#else
int rsbac_ms_scan_level = RSBAC_MS_LEVEL;
#endif

#if defined(CONFIG_RSBAC_MS_EXT_FPROTD) || defined(CONFIG_RSBAC_MS_EXT_CLAMD)
#ifdef CONFIG_RSBAC_MS_EXT_FPROTD
/*** Network settings ***/
#define FPROTD_LOCAL_PORT 0
#define FPROTD_LOCAL_ADDR "127.0.0.1"
#define FPROT_REMOTE_PORT_MIN 10200
#define FPROT_REMOTE_PORT_MAX 10204
#define FPROT_REMOTE_ADDR "127.0.0.1"
rsbac_ms_do_scan_t * rsbac_ms_do_scan_fprotd = NULL;
#endif
#ifdef CONFIG_RSBAC_MS_EXT_CLAMD
#define CLAMD_LOCAL_PORT 0
#define CLAMD_LOCAL_ADDR "127.0.0.1"
#define CLAM_REMOTE_PORT_MIN CONFIG_RSBAC_MS_EXT_CLAMD_PORT
#define CLAM_REMOTE_PORT_MAX CONFIG_RSBAC_MS_EXT_CLAMD_PORT
#define CLAM_REMOTE_ADDR "127.0.0.1"
rsbac_ms_do_scan_t * rsbac_ms_do_scan_clamd = NULL;
#endif

static u_long ms_nr_calls = 0;

#if defined(CONFIG_RSBAC_PROC)
#define MS_LEVEL_PROC_NAME "ms_level"
static struct proc_dir_entry * ms_proc_info_p;
#endif
#endif

/************************************************* */
/*          Internal Help functions                */
/************************************************* */

#ifndef CONFIG_RSBAC_MS_EXT
/* open_by_dentry */
/* This is done by hand (copy from rsbac_read_open), because system calls */
/* are currently blocked by rsbac semaphores */

static int open_by_dentry(struct dentry * file_dentry_p, struct file * file_p)
  {
    int tmperr;

    if ( !(S_ISREG(file_dentry_p->d_inode->i_mode)) )
      { /* this is not a file! -> error! */
        printk(KERN_WARNING
               "open_by_dentry(): expected file is not a file!\n");
        return (-RSBAC_EREADFAILED);
      }
    if (!MAJOR(file_dentry_p->d_inode->i_dev))
      { /* ignore files on non-physical devices */
        printk(KERN_WARNING
               "open_by_dentry(): ignored file on a non-device!\n");
        return (-RSBAC_EREADFAILED);
      }
    /* Now we fill the file structure, and */
    /* if there is an open func, use it, otherwise ignore */
    if ((tmperr = init_private_file(file_p, file_dentry_p, O_RDONLY)))
      {
        printk(KERN_WARNING
               "open_by_dentry(): could not open file!\n");
        return (-RSBAC_EREADFAILED);
      }
    /* Without a read function we get into troubles -> error */
    if ((!file_p->f_op) || (!file_p->f_op->read))
      {
        printk(KERN_WARNING
               "open_by_dentry(): file read function missing!\n");
        if (file_p->f_op && file_p->f_op->release)
          file_p->f_op->release(file_dentry_p->d_inode,file_p);
        return(-RSBAC_EREADFAILED);
      }
    return 0;
  }

/* do_scan() */
/* This function scans the given file for malware, returning a result value */
/* whether the file is accepted or not. */

static int do_scan(struct dentry * dentry_p)
  {
    struct file  file;
    int i, j, len, c, tmperr=1;
    mm_segment_t oldfs;
    int retval = rsbac_ms_scan_level;
    u_long chunk = 0;
    char * kbuf;
    u_int count = 0;
    int str_nr[RSBAC_MS_NR_MALWARE];
    int str_offset[RSBAC_MS_NR_MALWARE];
    char * pos;
    char * bufend = NULL;
    boolean buffer_end;

    if(!dentry_p)
      {
        printk(KERN_WARNING
               "do_scan(): called with NULL dentry_p!\n");
        return -RSBAC_EINVALIDPOINTER;
      }
    if(!dentry_p->d_inode)
      {
        printk(KERN_WARNING
               "do_scan(): called with NULL dentry_p->d_inode!\n");
        return -RSBAC_EINVALIDPOINTER;
      }
    /* open */
#ifdef CONFIG_RSBAC_DEBUG
    if(rsbac_debug_adf_ms)
      printk(KERN_DEBUG "do_scan(): opening file (dev %u, inode %lu)\n",
                          dentry_p->d_inode->i_dev,
                          dentry_p->d_inode->i_ino);
#endif
    if(open_by_dentry(dentry_p,&file))
      {
        printk(KERN_WARNING
               "do_scan(): file open failed!\n");
        return -RSBAC_EREADFAILED;
      }
    /* allocate help buffer */
    kbuf = kmalloc(RSBAC_MS_MAX_STR_LEN + RSBAC_MS_CHUNK_SIZE, GFP_KERNEL);
    if(!kbuf)
      {
        printk(KERN_WARNING "do_scan(): could not allocate buffer memory!\n");
        return -RSBAC_ENOMEM;
      }
    /* clear buffer start / step back part */
    memset(kbuf,0,RSBAC_MS_MAX_STR_LEN);

    /* OK, now we can start reading */
    /* There is a read function for this file, so fill buffer as far */
    /* as possible. A positive return value means a read success,   */
    /* 0 end of file and a negative value an error.                 */

    /* Set current user space to kernel space, because read() writes */
    /* to user space */
    oldfs = get_fs();
    set_fs(KERNEL_DS);

#ifdef CONFIG_RSBAC_DEBUG
    if(rsbac_debug_adf_ms)
      printk(KERN_DEBUG "do_scan(): starting to read\n");
#endif
    for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
      {
        str_nr[i] = 0;
        str_offset[i] = 0;
      }
    while(tmperr > 0)
      {
        /* concat new data from file */
        tmperr = file.f_op->read(&file,
                                 kbuf + RSBAC_MS_MAX_STR_LEN,
                                 RSBAC_MS_CHUNK_SIZE,
                                 &file.f_pos);
        /* if none read, end of file is reached -> file OK -> return TRUE */
        if(!tmperr)
          {
            goto out;
          }
        if (tmperr < 0)
          {
            printk(KERN_WARNING "do_scan: read error from file!\n");
            retval = -RSBAC_EREADFAILED;
            goto out;
          }
        /* begin scan */
#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_adf_ms)
          printk(KERN_DEBUG
                 "do_scan(): starting to scan chunk %lu, %i bytes\n",
                 chunk, tmperr);
#endif
        bufend = kbuf + RSBAC_MS_MAX_STR_LEN + tmperr;
        j=0;
        while((j<RSBAC_MS_NR_MALWARE) && (retval > 0))
          {
            /* entry is first new char */
            pos = kbuf + RSBAC_MS_MAX_STR_LEN;
            buffer_end = FALSE;
            count = 0;

            /* in current string str_nr[j] we were at position str_offset[j] */
            c = str_offset[j];
            for(i = str_nr[j]; i<RSBAC_MS_NR_STRINGS; i++)
              {
                len = strlen(all_malware[j].string[i]);
                do
                  {
                    /* if next char in string is read char or '.', advance */
                    if (   (*pos == all_malware[j].string[i][c])
                        || (all_malware[j].string[i][c] == '.')
                       )
                      {
                        c++;
                      }
                    /* else: back to string pos c=0 and back with buffer pos */
                    else
                      { /* but only, if c > 0 */
                        if(c)
                          {
                            /* go back in buffer */
                            pos -= c;
                            count -= c;
                            /* reset c */
                            c=0;
                          }
                      }
                    /* advance... */
                    pos++;
                    count++;
                  }
                while (   (c < len)
                       && (count < tmperr)); /* end of do */
                /* buffer end reached? -> store values and go on with next entry */
                if(count >= tmperr)
                  {
                    /* check for last char match */
                    if(c == len)
                      {
                        c = 0;
                        i++;
                      }
                    /* Has last char match triggered full match? If not, go on here */
                    if(i<RSBAC_MS_NR_STRINGS)
                      {
                        /* set string no. */
                        str_nr[j] = i;
                        /* set string offset */
                        str_offset[j] = c;
                        /* note buffer end for this malware entry */
                        buffer_end = TRUE;
                        break;
                      }
                  }
                /* no segment end or last char full match -> string was matched */
#ifdef CONFIG_RSBAC_DEBUG
                if(rsbac_debug_adf_ms)
                  printk(KERN_DEBUG "do_scan(): string %i for %s matched at byte %u\n",
                         i, all_malware[j].name, count);
#endif
                /* reset str position counter for next string */
                c = 0;
              } /* end of for (running through all search strings) */

            if(!buffer_end)
              {
                /* OK, all strings were found -> file is probably infected */
                printk(KERN_WARNING
                       "do_scan(): file (dev %u, inode %lu) is probably infected by %s!\n",
                       dentry_p->d_inode->i_dev,
                       dentry_p->d_inode->i_ino,
                       all_malware[j].name);
                retval = 0;
                goto out;
              }
            j++;
          } /* end of while loop for all malware entries */

        /* all entries reached buffer end: continue */
        /* fill backbuf (start of buffer) */
        memcpy(kbuf, bufend - RSBAC_MS_MAX_STR_LEN, RSBAC_MS_MAX_STR_LEN);
#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_adf_ms)
          printk(KERN_DEBUG
                 "do_scan(): finished scan for chunk %lu\n",
                 chunk);
#endif
        chunk++;
      }

out:
    /* Set current user space back to user space, because read() writes */
    /* to user space */
    set_fs(oldfs);

    kfree(kbuf);
/* do we need this here? */
    if (file.f_op->release)
      file.f_op->release(dentry_p->d_inode,&file);
    /* return 0 = rejected, rsbac_ms_scan_level = accepted, negative value = error */
    return(retval);
  }
#endif

#if defined(CONFIG_RSBAC_MS_EXT_FPROTD) || defined(CONFIG_RSBAC_MS_EXT_CLAMD)
/* declare net functions */
long sys_socket(int family, int type, int protocol);
long sys_bind(int fd, struct sockaddr *umyaddr, int addrlen);
long sys_connect(int fd, struct sockaddr *uservaddr, int addrlen);
//ssize_t sys_read(unsigned int fd, char * buf, size_t count);
//ssize_t sys_write(unsigned int fd, const char * buf, size_t count);
long sys_send(int fd, void * buff, size_t len, unsigned flags);
long sys_recv(int fd, void * ubuf, size_t size, unsigned flags);
long sys_shutdown(int fd, int how);

#ifdef CONFIG_RSBAC_MS_EXT_FPROTD
/**** Scanning Function ****/

static int ms_fprotd_do_scan(struct dentry * dentry_p)
  {
    u_int i;
    mm_segment_t oldfs;
    int sock_fd;
    struct sockaddr_in addr;
    int err;
    u_int nr_restart = 0;

    /* create a socket */
    sock_fd = sys_socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sock_fd < 0)
      {
        printk(KERN_WARNING
               "ms_fprotd_do_scan(): creating local log socket failed with error %u, exiting!\n",
               sock_fd);
        return -RSBAC_EWRITEFAILED;
      }
    /* bind local address */
    addr.sin_family = PF_INET;
    addr.sin_port = htons(FPROTD_LOCAL_PORT);
    err = rsbac_net_str_to_inet(FPROTD_LOCAL_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "ms_fprotd_do_scan(): converting local socket address %s failed with error %u, exiting!\n",
               FPROTD_LOCAL_ADDR,
               err);
        sys_close(sock_fd);
        return -RSBAC_EINVALIDVALUE;
      }
    /* change data segment - sys_bind reads address from user space */
    oldfs = get_fs();
    set_fs(KERNEL_DS);
    err = sys_bind(sock_fd, (struct sockaddr *)&addr, sizeof(addr));
    set_fs(oldfs);
    if(err < 0)
      {
        printk(KERN_WARNING
               "ms_fprotd_do_scan(): binding local socket address %u.%u.%u.%u:%u failed with error %u, exiting!\n",
               NIPQUAD(addr.sin_addr.s_addr),
               FPROTD_LOCAL_PORT,
               err);
        sys_close(sock_fd);
        return -RSBAC_EWRITEFAILED;
      }

    /* convert remote address */
    addr.sin_family = PF_INET;
    err = rsbac_net_str_to_inet(FPROT_REMOTE_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "ms_fprotd_do_scan(): converting remote socket address %s failed with error %u, exiting!\n",
               FPROT_REMOTE_ADDR,
               err);
        sys_close(sock_fd);
        return -RSBAC_EINVALIDVALUE;
      }

restart:

    for(i = FPROT_REMOTE_PORT_MIN; i <= FPROT_REMOTE_PORT_MAX ; i++)
      {
        addr.sin_port = htons(i);
        oldfs = get_fs();
        set_fs(KERNEL_DS);
        err = sys_connect(sock_fd,
                          (struct sockaddr *)&addr,
                          sizeof(addr));
        set_fs(oldfs);
        if(err >=0)
          {
            char * sendbuf;
            char * filename;
            int size;
            int len;

#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_fprotd_do_scan(): successfully connected to scanner port %u.\n",
                     i);
#endif
            sendbuf = rsbac_kmalloc(PAGE_SIZE + 1024);
            if(!sendbuf)
              {
                sys_close(sock_fd);
                return -ENOMEM;
              }
            filename = rsbac_kmalloc(PAGE_SIZE);
            if(!filename)
              {
                rsbac_kfree(sendbuf);
                sys_close(sock_fd);
                return -ENOMEM;
              }
            err = rsbac_get_full_path(dentry_p, filename, PAGE_SIZE-1);
            if(err < 0)
              {
                rsbac_kfree(sendbuf);
                rsbac_kfree(filename);
                sys_close(sock_fd);
                return -RSBAC_EREADFAILED;
              }
            size = sprintf(sendbuf, "GET %s?%s HTTP/1.0\r\n\r\n",
                           filename,
                           CONFIG_RSBAC_MS_EXT_FPROTD_SW);
#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_fprotd_do_scan(): sending request %s",
                     sendbuf);
#endif
            len = 0;
            err = 1;
            while((err > 0) && (len < size))
              {
                oldfs = get_fs();
                set_fs(KERNEL_DS);
                err = sys_send(sock_fd, sendbuf + len, size - len, MSG_DONTWAIT);
                set_fs(oldfs);
                if(err > 0)
                  len += err;
              }
            if(err < 0)
              {
                rsbac_kfree(sendbuf);
                rsbac_kfree(filename);
                sys_close(sock_fd);
                return -RSBAC_EWRITEFAILED;
              }
#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_fprotd_do_scan(): sent %u bytes\n",
                     len);
#endif
            sys_shutdown(sock_fd, 1);
            len = 0;
            err = 1;
            memset(sendbuf, 0, PAGE_SIZE);
            while(err > 0)
              {
                oldfs = get_fs();
                set_fs(KERNEL_DS);
                err = sys_recv(sock_fd, sendbuf + len, PAGE_SIZE - len, 0);
                set_fs(oldfs);
                if(err > 0)
                  {
                    len += err;
                    if(len > (PAGE_SIZE - 512))
                      {
                        memcpy(sendbuf, sendbuf + PAGE_SIZE - 1024, len - (PAGE_SIZE - 1024));
                        len -= (PAGE_SIZE - 1024);
                      }
                  }
              }
            if(err >= 0)
              {
                sendbuf[len] = 0;
#ifdef CONFIG_RSBAC_DEBUG
                if(rsbac_debug_adf_ms)
                  printk(KERN_DEBUG
                         "ms_fprotd_do_scan(): got reply %s\n",
                         sendbuf);
#endif
                if(strstr(sendbuf, "infected</summary>"))
                  err = 0;
                else
                if(   strstr(sendbuf, "clean</summary>")
//                   || strstr(sendbuf, "was not scanned")
                   || strstr(sendbuf, "unknown</summary>")
                  )
                  err = rsbac_ms_scan_level;
                else
                  err = -RSBAC_EREADFAILED;
              }
            else
              {
#ifdef CONFIG_RSBAC_DEBUG
                if(rsbac_debug_adf_ms)
                  printk(KERN_DEBUG
                         "ms_fprotd_do_scan(): got reply error %i\n",
                         err);
#endif
              }
            rsbac_kfree(sendbuf);
            rsbac_kfree(filename);
            sys_close(sock_fd);
            if(   (err == -ERESTARTSYS)
               && (nr_restart < 10)
              )
              {
                nr_restart++;
                printk(KERN_DEBUG
                       "ms_fprotd_do_scan(): scanner connection on port %u failed with ERESTARTSYS, starting reconnection cycle %u.\n",
                       i,
                       nr_restart);
                goto restart;
              }
            return err;
          }
#ifdef CONFIG_RSBAC_DEBUG
        else
          {
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_fprotd_do_scan(): connection to scanner port %u failed with error %i.\n",
                     i,
                     err);
          }
#endif
      }

    printk(KERN_WARNING
           "ms_fprotd_do_scan(): connecting to scanner failed!\n");
    sys_close(sock_fd);
    return -RSBAC_EWRITEFAILED;
  }
#endif

#ifdef CONFIG_RSBAC_MS_EXT_CLAMD
/**** Scanning Function ****/

static int ms_clamd_do_scan(struct dentry * dentry_p)
  {
    u_int i;
    mm_segment_t oldfs;
    int sock_fd;
    struct sockaddr_in addr;
    int err;
    u_int nr_restart = 0;

    /* create a socket */
    sock_fd = sys_socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sock_fd < 0)
      {
        printk(KERN_WARNING
               "ms_clamd_do_scan(): creating local socket failed with error %u, exiting!\n",
               sock_fd);
        return -RSBAC_EWRITEFAILED;
      }
    /* bind local address */
    addr.sin_family = PF_INET;
    addr.sin_port = htons(CLAMD_LOCAL_PORT);
    err = rsbac_net_str_to_inet(CLAMD_LOCAL_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "ms_clamd_do_scan(): converting local socket address %s failed with error %u, exiting!\n",
               CLAMD_LOCAL_ADDR,
               err);
        sys_close(sock_fd);
        return -RSBAC_EINVALIDVALUE;
      }
    /* change data segment - sys_bind reads address from user space */
    oldfs = get_fs();
    set_fs(KERNEL_DS);
    err = sys_bind(sock_fd, (struct sockaddr *)&addr, sizeof(addr));
    set_fs(oldfs);
    if(err < 0)
      {
        printk(KERN_WARNING
               "ms_clamd_do_scan(): binding local socket address %u.%u.%u.%u:%u failed with error %u, exiting!\n",
               NIPQUAD(addr.sin_addr.s_addr),
               CLAMD_LOCAL_PORT,
               err);
        sys_close(sock_fd);
        return -RSBAC_EWRITEFAILED;
      }

    /* convert remote address */
    addr.sin_family = PF_INET;
    err = rsbac_net_str_to_inet(CLAM_REMOTE_ADDR,
                                &addr.sin_addr.s_addr);
    if(err < 0)
      {
        printk(KERN_WARNING
               "ms_clamd_do_scan(): converting remote socket address %s failed with error %u, exiting!\n",
               CLAM_REMOTE_ADDR,
               err);
        sys_close(sock_fd);
        return -RSBAC_EINVALIDVALUE;
      }

restart:

    for(i = CLAM_REMOTE_PORT_MIN; i <= CLAM_REMOTE_PORT_MAX ; i++)
      {
        addr.sin_port = htons(i);
        oldfs = get_fs();
        set_fs(KERNEL_DS);
        err = sys_connect(sock_fd,
                          (struct sockaddr *)&addr,
                          sizeof(addr));
        set_fs(oldfs);
        if(err >=0)
          {
            char * sendbuf;
            char * filename;
            int size;
            int len;

#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_clamd_do_scan(): successfully connected to scanner port %u.\n",
                     i);
#endif
            sendbuf = rsbac_kmalloc(PAGE_SIZE + 1024);
            if(!sendbuf)
              {
                sys_close(sock_fd);
                return -ENOMEM;
              }
            filename = rsbac_kmalloc(PAGE_SIZE);
            if(!filename)
              {
                rsbac_kfree(sendbuf);
                sys_close(sock_fd);
                return -ENOMEM;
              }
            err = rsbac_get_full_path(dentry_p, filename, PAGE_SIZE-1);
            if(err < 0)
              {
                rsbac_kfree(sendbuf);
                rsbac_kfree(filename);
                sys_close(sock_fd);
                return -RSBAC_EREADFAILED;
              }
            size = sprintf(sendbuf, "%s %s\n",
                           CONFIG_RSBAC_MS_EXT_CLAMD_ACTION,
                           filename);
#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_clamd_do_scan(): sending request %s",
                     sendbuf);
#endif
            len = 0;
            err = 1;
            while((err > 0) && (len < size))
              {
                oldfs = get_fs();
                set_fs(KERNEL_DS);
                err = sys_send(sock_fd, sendbuf + len, size - len, MSG_DONTWAIT);
                set_fs(oldfs);
                if(err > 0)
                  len += err;
              }
            if(err < 0)
              {
                rsbac_kfree(sendbuf);
                rsbac_kfree(filename);
                sys_close(sock_fd);
                return -RSBAC_EWRITEFAILED;
              }
#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_clamd_do_scan(): sent %u bytes\n",
                     len);
#endif
            sys_shutdown(sock_fd, 1);
            len = 0;
            err = 1;
            memset(sendbuf, 0, PAGE_SIZE);
            while(err > 0)
              {
                oldfs = get_fs();
                set_fs(KERNEL_DS);
                err = sys_recv(sock_fd, sendbuf + len, PAGE_SIZE - len, 0);
                set_fs(oldfs);
                if(err > 0)
                  {
                    len += err;
                    if(len > (PAGE_SIZE - 512))
                      {
                        memcpy(sendbuf, sendbuf + PAGE_SIZE - 1024, len - (PAGE_SIZE - 1024));
                        len -= (PAGE_SIZE - 1024);
                      }
                  }
              }
            if(err >= 0)
              {
                sendbuf[len] = 0;
#ifdef CONFIG_RSBAC_DEBUG
                if(rsbac_debug_adf_ms)
                  printk(KERN_DEBUG
                         "ms_clamd_do_scan(): got reply %s\n",
                         sendbuf);
#endif
                if(strstr(sendbuf, "FOUND\n"))
                  err = 0;
                else
                if(   strstr(sendbuf, "OK\n")
//                   || strstr(sendbuf, "was not scanned")
//                   || strstr(sendbuf, "unknown</summary>")
                  )
                  err = rsbac_ms_scan_level;
                else
                  err = -RSBAC_EREADFAILED;
              }
            else
              {
#ifdef CONFIG_RSBAC_DEBUG
                if(rsbac_debug_adf_ms)
                  printk(KERN_DEBUG
                         "ms_clamd_do_scan(): got reply error %i\n",
                         err);
#endif
              }
            rsbac_kfree(sendbuf);
            rsbac_kfree(filename);
            sys_close(sock_fd);
            if(   (err == -ERESTARTSYS)
               && (nr_restart < 10)
              )
              {
                nr_restart++;
                printk(KERN_DEBUG
                       "ms_clamd_do_scan(): scanner connection on port %u failed with ERESTARTSYS, starting reconnection cycle %u.\n",
                       i,
                       nr_restart);
                goto restart;
              }
            return err;
          }
#ifdef CONFIG_RSBAC_DEBUG
        else
          {
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG
                     "ms_clamd_do_scan(): connection to scanner port %u failed with error %i.\n",
                     i,
                     err);
          }
#endif
      }

    printk(KERN_WARNING
           "ms_clamd_do_scan(): connecting to scanner failed!\n");
    sys_close(sock_fd);
    return -RSBAC_EWRITEFAILED;
  }
#endif

/* PROC interface */
#ifdef CONFIG_RSBAC_PROC
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
static int
ms_proc_info(char *buffer, char **start, off_t offset, int length, int dummy)
#else
static int
ms_proc_info(char *buffer, char **start, off_t offset, int length)
#endif
{
  int len = 0;
  off_t pos   = 0;
  off_t begin = 0;

  union rsbac_target_id_t       rsbac_target_id;
  union rsbac_attribute_value_t rsbac_attribute_value;

  if (!rsbac_is_initialized())
    return (-ENOSYS);

  rsbac_target_id.scd = ST_rsbac;
  rsbac_attribute_value.dummy = 0;
  if (!rsbac_adf_request(R_GET_STATUS_DATA,
                         current->pid,
                         T_SCD,
                         rsbac_target_id,
                         A_none,
                         rsbac_attribute_value))
    {
      return -EPERM;
    }
  len += sprintf(buffer, "MS scanning levels\n------------------\n");
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

#ifdef CONFIG_RSBAC_MS_EXT_FPROTD
  len += sprintf(buffer + len, "F-Protd support included.\n");
#endif
#ifdef CONFIG_RSBAC_MS_EXT_CLAMD
  len += sprintf(buffer + len, ">Clamd support included.\n");
#endif
  len += sprintf(buffer + len, "%lu calls to do_scan function,\n",
                 ms_nr_calls);
  len += sprintf(buffer + len, "scan level is %u.\n",
                 rsbac_ms_scan_level);
  if(!rsbac_ms_scan_level)
    len += sprintf(buffer + len, "(scanning is off)\n");
  pos = begin + len;
  if (pos < offset)
    {
      len = 0;
      begin = pos;
    }
  if (pos > offset+length)
    goto out;

out:
  *start = buffer + (offset - begin);
  len -= (offset - begin);
  
  if (len > length)
    len = length;
  return len;
}

static ssize_t ms_proc_write(struct file * file, const char * buf,
                                     u_long count, void *ppos)
{
  ssize_t err = -EINVAL;
  char * k_buf;
  char * p;
  unsigned int new_level;

  union rsbac_attribute_value_t i_attr_val1;
  union rsbac_target_id_t       i_tid;

  if(count > PROC_BLOCK_SIZE)
    return(-EOVERFLOW);


  k_buf = (char *) __get_free_page(GFP_KERNEL);
  if(!k_buf)
    return(-ENOMEM);
  copy_from_user(k_buf, buf, count);

  if(count < 8 || strncmp("level", k_buf, 5))
    {
      goto out;
    }
  if (!rsbac_is_initialized())
    {
      err=-ENOSYS;
      goto out;
    }

    /*
     * Usage: echo "level #N" > /proc/rsbac_info/ms_fprotd
     *   to set scanning level to given level. Use level 0 to switch scanning off (default).
     */
    p = k_buf + 6;

    if( *p == '\0' )
      goto out;

    if(!strncmp(p, "inc", 3))
      {
        if(!rsbac_ms_scan_level)
          new_level = 20;
        else
          new_level = rsbac_ms_scan_level + 1;
      }
    else
      {
        new_level = simple_strtoul(p, NULL, 0);
      }

#ifdef CONFIG_RSBAC_SOFTMODE
    if(   !rsbac_softmode
#ifdef CONFIG_RSBAC_SOFTMODE_IND
       && !rsbac_ind_softmode[MS]
#endif
      )
#endif
      {
        /* Administrator or secoff? */
        i_tid.user = current->uid;
        if (rsbac_get_attr(MS,
                           T_USER,
                           i_tid,
                           A_ms_role,
                           &i_attr_val1,
                           TRUE))
          {
            printk(KERN_WARNING
                   "ms_proc_write(): rsbac_get_attr() returned error!\n");
                   return(-EPERM);
          }
        /* only allow, if secoff or (admin && level raised) */
        if (   (i_attr_val1.system_role != SR_security_officer)
            && (   (i_attr_val1.system_role != SR_administrator)
                || (new_level < rsbac_ms_scan_level)
               )
           )
          return -EPERM;
      }

    if(new_level)
      {
        if(!rsbac_ms_scan_level)
          {
            #ifdef CONFIG_RSBAC_MS_EXT_FPROTD
            rsbac_ms_do_scan_fprotd = ms_fprotd_do_scan;
            #endif
            #ifdef CONFIG_RSBAC_MS_EXT_CLAMD
            rsbac_ms_do_scan_clamd = ms_clamd_do_scan;
            #endif
            printk(KERN_INFO
                   "ms_proc_write(): activated scanner on scan level %u\n",
                   new_level);
          }
        rsbac_ms_scan_level = new_level;
      }
    else
      {
        if(rsbac_ms_scan_level)
          {
            #ifdef CONFIG_RSBAC_MS_EXT_FPROTD
            rsbac_ms_do_scan_fprotd = NULL;
            #endif
            #ifdef CONFIG_RSBAC_MS_EXT_CLAMD
            rsbac_ms_do_scan_clamd = NULL;
            #endif
            printk(KERN_INFO
                   "ms_proc_write(): deactivated scanner\n");
          }
        rsbac_ms_scan_level = new_level;
      }
    err = count;

out:
  free_page((ulong) k_buf);
  return(err);
}
#endif /* CONFIG_RSBAC_PROC */
#endif
  
/* scan() */
/* This function checks the scanned status for given file, calls do_scan, if */
/* unscanned, sets the scanned status and returns whether file is accepted or */
/* not. */

static boolean scan(struct rsbac_fs_file_t file,
                    rsbac_pid_t caller_pid,
                    boolean execute)
  {
    union rsbac_attribute_value_t i_attr_val1;
    union rsbac_target_id_t       i_tid;
    struct dentry * dentry_p = file.dentry_p;
    int err = 1;

    /* just in case... */
    if(!dentry_p || !dentry_p->d_inode)
      {
#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_adf_ms)
          printk(KERN_DEBUG "scan(): refused scan for file without proper dentry and inode\n");
#endif
        return(TRUE);
      }
    /* currently no check or scanned status change for non-regular files */
    if(!S_ISREG(dentry_p->d_inode->i_mode))
      {
#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_adf_ms)
          printk(KERN_DEBUG "scan(): refused scan for non-regular file\n");
#endif
        return(TRUE);
      }
    /* no check or scanned status change for procfs files */
    if(dentry_p->d_inode->i_sb->s_magic == PROC_SUPER_MAGIC)
      {
#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_adf_ms)
          printk(KERN_DEBUG "scan(): refused scan for procfs file\n");
#endif
        return(TRUE);
      }
    /* get ms_trusted for process */
    i_tid.process = caller_pid;
    if (rsbac_get_attr(MS,
                       T_PROCESS,
                       i_tid,
                       A_ms_trusted,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "scan(): rsbac_get_attr() returned error!\n");
        return(FALSE);
      }
    /* no check or scanned status change for trusted processes */
    if(   (i_attr_val1.ms_trusted == MT_full)
       || (   (i_attr_val1.ms_trusted == MT_read)
           && !execute
          )
      )
      return(TRUE);

    /* only scan, if requested for this target and access type */
    i_tid.file = file;
    if (rsbac_get_attr(MS,
                       T_FILE,
                       i_tid,
                       A_ms_need_scan,
                       &i_attr_val1,
                       TRUE))
      {
        printk(KERN_WARNING
               "scan(): rsbac_get_attr() returned error!\n");
        return(FALSE);
      }
    if(   (!execute && (i_attr_val1.ms_need_scan != MS_need_scan_full))
       || (execute && (i_attr_val1.ms_need_scan == MS_need_scan_no))
      )
      return TRUE;

    /* get scanned status for file */
    if (rsbac_get_attr(MS,
                       T_FILE,
                       i_tid,
                       A_ms_scanned,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "scan(): rsbac_get_attr() returned error!\n");
        return(FALSE);
      }
    /* accepted with current or higher level? */
    if(i_attr_val1.ms_scanned >= rsbac_ms_scan_level)
      return(TRUE);
    /* rejected with any level? */
    if(i_attr_val1.ms_scanned == MS_rejected)
      return(FALSE);

    /* Always allow, if no scanning engine */
    if(rsbac_ms_do_scan)
      err = rsbac_ms_do_scan(dentry_p);
    #ifdef CONFIG_RSBAC_MS_EXT_FPROTD
    if(   (err > 0)
       && rsbac_ms_do_scan_fprotd
      )
      err = rsbac_ms_do_scan_fprotd(dentry_p);
    #endif
    #ifdef CONFIG_RSBAC_MS_EXT_CLAMD
    if(   (err > 0)
       && rsbac_ms_do_scan_clamd
      )
      err = rsbac_ms_do_scan_clamd(dentry_p);
    #endif
    ms_nr_calls++;
    if(err > 0)
      {
        i_attr_val1.ms_scanned = rsbac_ms_scan_level;
        if (rsbac_set_attr(MS,
                       T_FILE,
                           i_tid,
                           A_ms_scanned,
                           i_attr_val1))
          {
            printk(KERN_WARNING
                   "scan(): rsbac_set_attr() returned error!\n");
            return(FALSE);
          }
        return(TRUE);
      }
    else
    if(!err)
      {
        i_attr_val1.ms_scanned = MS_rejected;
        if (rsbac_set_attr(MS,
                       T_FILE,
                           i_tid,
                           A_ms_scanned,
                           i_attr_val1))
          {
            printk(KERN_WARNING
                   "scan(): rsbac_set_attr() returned error!\n");
          }
        return(FALSE);
      }
    else
      {
        printk(KERN_WARNING
               "scan(): rsbac_ms_do_scan function returned error %i!\n",
               err);
        return(FALSE);
      }
  }; /* end of scan() */

static int reset_scanned(struct rsbac_fs_file_t file)
  {
    union rsbac_attribute_value_t i_attr_val1;
    union rsbac_target_id_t       i_tid;

    /* get scanned status for file */
    i_tid.file=file;
    if (rsbac_get_attr(MS,
                       T_FILE,
                       i_tid,
                       A_ms_scanned,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "reset_scanned(): rsbac_get_attr() returned error!\n");
        return(-RSBAC_EREADFAILED);
      }
    if(i_attr_val1.ms_scanned != MS_unscanned)
      {
        i_attr_val1.ms_scanned = MS_unscanned;
        if(rsbac_set_attr(MS,
                       T_FILE,
                          i_tid,
                          A_ms_scanned,
                          i_attr_val1))
          {
            printk(KERN_WARNING "reset_scanned(): rsbac_set_attr() returned error!\n");
            return(-RSBAC_EWRITEFAILED);
          }
      }
    return(0);
  };

/* do_scan_sock() */
/* This function scans the given segment for malware, returning a decision */
/* whether the segment is accepted or not. */
/* The scanning state is first retrieved from IPC ACI and later saved there */
/* before returning. */

#ifdef CONFIG_RSBAC_MS_SOCK
static enum rsbac_adf_req_ret_t
  do_scan_sock(union  rsbac_target_id_t tid,
                      rsbac_pid_t       caller_pid,
               struct ms_segment_t      segment)
  {
    char * kbuf;
    int oldbuflen;
    int i=0,c=0,len=0,j=0;
    int count = 0;
    int str_nr[RSBAC_MS_NR_MALWARE];
    int str_offset[RSBAC_MS_NR_MALWARE];
    char * pos;
    char * bufend;
    char * acibuf;
    boolean is_infected = FALSE;
    boolean buffer_end;
//    boolean vmalloc_used;
    enum  rsbac_attribute_t       i_attr;
    union rsbac_attribute_value_t i_attr_val1;
    union rsbac_target_id_t       i_tid;

    /* get ms_scanned for socket */
    if (rsbac_get_attr(MS,
                       T_NETOBJ,
                       tid,
                       A_ms_scanned,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "do_scan_sock(): rsbac_get_attr() returned error!\n");
        return(NOT_GRANTED);
      }
    /* reject, if previously active rejected */
    if(i_attr_val1.ms_scanned == MS_active_rejected)
      {
        /* save segment length */
        i_attr_val1.ms_buflen = segment.len;
        if(rsbac_set_attr(MS,
                          T_NETOBJ,
                          tid,
                          A_ms_buflen,
                          i_attr_val1))
          {
            printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
          }
        /* return not_granted */
        return(NOT_GRANTED);
      }
    /* reject and blank, if previously rejected */
    if(i_attr_val1.ms_scanned == MS_rejected)
      {
        if(segment.len)
          clear_user_buf(segment.ubuf,segment.len);
        return(NOT_GRANTED);
      }
    /* grant access for empty segment */
    if(!segment.len)
      return(GRANTED);

    /* get ms_buflen for sock */
    if (rsbac_get_attr(MS,
                       T_NETOBJ,
                       tid,
                       A_ms_buflen,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "do_scan_sock(): rsbac_get_attr() returned error!\n");
        return(NOT_GRANTED);
      }
    oldbuflen = i_attr_val1.ms_buflen;
    /* get old buffer */
    if (rsbac_get_attr(MS,
                       T_NETOBJ,
                       tid,
                       A_ms_backbuf,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "do_scan_sock(): rsbac_get_attr() returned error!\n");
        return(NOT_GRANTED);
      }
    /* allocate help buffer */
//    kbuf = rsbac_vkmalloc(oldbuflen + segment.len + 1, &vmalloc_used);
    kbuf = kmalloc(oldbuflen + segment.len + 1, GFP_KERNEL);
    if(!kbuf)
      {
        printk(KERN_WARNING "do_scan_sock(): could not allocate buffer memory!\n");
        return(NOT_GRANTED);
      }
    acibuf = i_attr_val1.ms_backbuf;
    if(oldbuflen)
      {
        memcpy(kbuf,acibuf,oldbuflen);
      }
    /* get ms_str_nr for sock */
    if (rsbac_get_attr(MS,
                       T_NETOBJ,
                       tid,
                       A_ms_str_nr,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "do_scan_sock(): rsbac_get_attr() returned error!\n");
        kfree(kbuf);
//        rsbac_vkfree(kbuf, vmalloc_used);
        return(NOT_GRANTED);
      }
    for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
      str_nr[i] = i_attr_val1.ms_str_nr[i];
    /* get ms_str_offset for sock */
    if (rsbac_get_attr(MS,
                       T_NETOBJ,
                       tid,
                       A_ms_str_offset,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "do_scan_sock(): rsbac_get_attr() returned error!\n");
        kfree(kbuf);
//        rsbac_vkfree(kbuf, vmalloc_used);
        return(NOT_GRANTED);
      }
    for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
      str_offset[i] = i_attr_val1.ms_str_offset[i];
    /* concat new data from user space */
    rsbac_get_user(kbuf+oldbuflen,segment.ubuf,segment.len);
    /* begin scan */
#ifdef CONFIG_RSBAC_DEBUG
    if(rsbac_debug_adf_ms)
      printk(KERN_DEBUG
             "do_scan_sock(): starting to scan on socket %p, %i bytes\n",
             tid.netobj.sock_p, segment.len);
#endif
    bufend = kbuf + (oldbuflen + segment.len);
    j=0;
    while((j<RSBAC_MS_NR_MALWARE) && (!is_infected))
      {
        /* entry is first new char */
        pos = kbuf + oldbuflen;
        buffer_end = FALSE;
        count = 0;

        /* in current string str_nr[j] we were at position str_offset[j] */
        c = str_offset[j];
        for(i = str_nr[j]; i<RSBAC_MS_NR_STRINGS; i++)
          {
            len = strlen(all_malware[j].string[i]);
            do
              {
                /* if next char in string is read char or '.', advance */
                if (   (*pos == all_malware[j].string[i][c])
                    || (all_malware[j].string[i][c] == '.')
                   )
                  {
                    c++;
                  }
                /* else: back to string pos c=0 and back with buffer pos */
                else
                  { /* but only, if c > 0 */
                    if(c)
                      {
                        /* go back in buffer */
                        pos -= c;
                        count -= c;
                        /* reset c */
                        c=0;
                      }
                  }
                /* advance... */
                pos++;
                count++;
              }
            while (   (c < len)
                   && (count < segment.len)); /* end of do */
            /* buffer end reached? -> store values and go on with next entry */
            if(count >= segment.len)
              {
                /* check for last char match */
                if(c == len)
                  {
                    c = 0;
                    i++;
                  }
                /* Has last char match triggered full match? If not, go on here */
                if(i<RSBAC_MS_NR_STRINGS)
                  {
                    /* set string no. */
                    str_nr[j] = i;
                    /* set string offset */
                    str_offset[j] = c;
                    /* note buffer end for this malware entry */
                    buffer_end = TRUE;
                    break;
                  }
              }
            /* no segment end or last char full match -> string was matched */
#ifdef CONFIG_RSBAC_DEBUG
            if(rsbac_debug_adf_ms)
              printk(KERN_DEBUG "do_scan_sock(): string %i for %s matched at byte %i\n",
                     i, all_malware[j].name, count);
#endif
            /* reset str position counter for next string */
            c = 0;
          } /* end of for (running through all search strings) */

        if(!buffer_end)
          {
            /* OK, all strings were found -> stream is probably infected */
            printk(KERN_WARNING
                   "do_scan_sock(): stream for process %i is probably infected by %s\n",
                   caller_pid, all_malware[j].name);
            is_infected = TRUE;
            break;
          }
        j++;
      } /* end of while loop for all malware entries */

    kfree(kbuf);
    //rsbac_vkfree(kbuf, vmalloc_used);

    /* if no infection, all entries reached buffer end: clean up and return */
    if(!is_infected)
      {
        /* set new str_nr[i] */
        for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
          i_attr_val1.ms_str_nr[i] = str_nr[i];
        if(rsbac_set_attr(MS,
                       T_NETOBJ,
                          tid,
                          A_ms_str_nr,
                          i_attr_val1))
          {
            printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
            return(NOT_GRANTED);
          }
        /* set new str_offset[i] */
        for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
          i_attr_val1.ms_str_offset[i] = str_offset[i];
        if(rsbac_set_attr(MS,
                       T_NETOBJ,
                          tid,
                          A_ms_str_offset,
                          i_attr_val1))
          {
            printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
            return(NOT_GRANTED);
          }
        /* set new backbuflen */
        if(oldbuflen + segment.len >= RSBAC_MS_MAX_STR_LEN)
          i_attr_val1.ms_buflen = RSBAC_MS_MAX_STR_LEN;
        else
          i_attr_val1.ms_buflen = oldbuflen+segment.len;
        if(rsbac_set_attr(MS,
                       T_NETOBJ,
                          tid,
                          A_ms_buflen,
                          i_attr_val1))
          {
            printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
            return(NOT_GRANTED);
          }
        /* fill backbuf (memory position is hold in acibuf) */
        memcpy(acibuf, bufend - i_attr_val1.ms_buflen, i_attr_val1.ms_buflen);
#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_adf_ms)
          printk(KERN_DEBUG
                 "do_scan_sock(): finished scan on sock %p\n",
                 tid.netobj.sock_p);
#endif
        return(GRANTED);
      }

    /* OK, stream is infected -> block depending on process attributes */
    /* get ms_sock_trusted_tcp/udp for process */
    i_tid.process = caller_pid;
    if(segment.prot == MP_TCP)
      i_attr = A_ms_sock_trusted_tcp;
    else
      i_attr = A_ms_sock_trusted_udp;
    if (rsbac_get_attr(MS,
                       T_PROCESS,
                       i_tid,
                       i_attr,
                       &i_attr_val1,
                       FALSE))
      {
        printk(KERN_WARNING
               "do_scan_sock(): rsbac_get_attr() returned error!\n");
        return(NOT_GRANTED);
      }
    /* no check or scanned status change for trusted processes */
    switch(i_attr_val1.ms_sock_trusted_tcp)
      {
        case MS_not_trusted:
          /* erase buffer contents */
          clear_user_buf(segment.ubuf,segment.len);
          /* set scan result */
          i_attr_val1.ms_scanned = MS_rejected;
          if(rsbac_set_attr(MS,
                       T_NETOBJ,
                            tid,
                            A_ms_scanned,
                            i_attr_val1))
            {
              printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
            }
          return(NOT_GRANTED);

        case MS_active:
          /* set scan result */
          i_attr_val1.ms_scanned = MS_active_rejected;
          if(rsbac_set_attr(MS,
                       T_NETOBJ,
                            tid,
                            A_ms_scanned,
                            i_attr_val1))
            {
              printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
            }
          /* save segment length */
          i_attr_val1.ms_buflen = segment.len;
          if(rsbac_set_attr(MS,
                       T_NETOBJ,
                            tid,
                            A_ms_buflen,
                            i_attr_val1))
            {
              printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
            }
          return(NOT_GRANTED);

        case MS_full:
          /* reset all ACI values to avoid problems */
          i_attr_val1.ms_buflen = 0;
          if(rsbac_set_attr(MS,
                       T_NETOBJ,
                            tid,
                            A_ms_buflen,
                            i_attr_val1))
            {
              printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
              return(NOT_GRANTED);
            }
          /* set string no. */
          for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
            i_attr_val1.ms_str_nr[i] = 0;
          if(rsbac_set_attr(MS,
                       T_NETOBJ,
                            tid,
                            A_ms_str_nr,
                            i_attr_val1))
            {
              printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
              return(NOT_GRANTED);
            }
          /* set string offset */
          for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
            i_attr_val1.ms_str_offset[i] = 0;
          if(rsbac_set_attr(MS,
                       T_NETOBJ,
                            tid,
                            A_ms_str_offset,
                            i_attr_val1))
            {
              printk(KERN_WARNING "do_scan_sock(): rsbac_set_attr() returned error!\n");
              return(NOT_GRANTED);
            }
          return(GRANTED);

        default:
          printk(KERN_WARNING "do_scan_sock(): invalid value for MS_sock_trusted_tcp/udp!\n");
          return(NOT_GRANTED);
      }
  }
#endif /* CONFIG_RSBAC_MS_SOCK */

/************************************************* */
/*          Externally visible functions           */
/************************************************* */

#ifdef CONFIG_RSBAC_INIT_DELAY
int rsbac_init_ms(void)
#else
int __init rsbac_init_ms(void)
#endif
  {
    int i;

    if (rsbac_is_initialized())
      {
        printk(KERN_WARNING "rsbac_init_ms(): RSBAC already initialized\n");
        return(-RSBAC_EREINIT);
      }

    /* init data structures */
    printk(KERN_INFO "rsbac_init_ms(): Initializing RSBAC: MS subsystem\n");

    for(i=0; i<RSBAC_MS_NR_MALWARE; i++)
      all_malware[i].string[0][0]++;
#ifndef CONFIG_RSBAC_MS_EXT
    rsbac_ms_do_scan = do_scan;
#endif
#ifdef CONFIG_RSBAC_MS_EXT
    #if defined(CONFIG_RSBAC_PROC)
    ms_proc_info_p = create_proc_entry(MS_LEVEL_PROC_NAME,
                                              S_IFREG | S_IRUGO | S_IWUGO,
                                              proc_rsbac_root_p);
    if(!ms_proc_info_p)
      {
        printk(KERN_WARNING "rsbac_init_ms: proc entry registration for %s failed.\n",
               MS_LEVEL_PROC_NAME);
      }
    else
      {
        ms_proc_info_p->get_info = ms_proc_info;
        ms_proc_info_p->write_proc = ms_proc_write;
      }
    #endif 
#endif

    return 0;
  }


enum rsbac_adf_req_ret_t
   rsbac_adf_request_ms  (enum  rsbac_adf_request_t     request,
                                rsbac_pid_t             caller_pid,
                          enum  rsbac_target_t          target,
                          union rsbac_target_id_t       tid,
                          enum  rsbac_attribute_t       attr,
                          union rsbac_attribute_value_t attr_val,
                                rsbac_uid_t             owner)
  {
/*    enum  rsbac_target_t          i_target;
*/
    union rsbac_target_id_t       i_tid;
/*    enum  rsbac_attribute_t       i_attr;
*/
    union rsbac_attribute_value_t i_attr_val1;
#ifdef CONFIG_RSBAC_MS_ROLE_PROT
    union rsbac_attribute_value_t i_attr_val2;
#endif

    switch (request)
      {
        case R_CHANGE_OWNER:
            switch(target)
              {
                case T_PROCESS:
                  #ifdef CONFIG_RSBAC_MS_ROLE_PROT
                  if(attr != A_owner)
                    return(UNDEFINED);
                  /* Administrator or secoff? */
                  i_tid.user = owner;
                  if (rsbac_get_attr(MS,
                                     T_USER,
                                     i_tid,
                                     A_ms_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request_ms(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* if general user or secoff, then grant */
                  if (   (i_attr_val1.system_role == SR_user)
                      || (i_attr_val1.system_role == SR_security_officer)
                     )
                    return(GRANTED);
                  /* old owner is sys-admin */
                  /* get target user's role */
                  i_tid.user = attr_val.owner;
                  if (rsbac_get_attr(MS,
                                     T_USER,
                                     i_tid,
                                     A_ms_role,
                                     &i_attr_val2,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request_ms(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* if target is security officer -> deny */
                  if(i_attr_val2.system_role == SR_security_officer)
                    return(NOT_GRANTED);
                  else
                    return(GRANTED);
                  #endif /* ROLE_PROT */

                  /* fall through */
                  /* all other cases */
                default:
                  return(DO_NOT_CARE);
              }

        case R_EXECUTE:
            switch(target)
              {
                case T_FILE:
                  /* scan file as execute */
                  if (scan(tid.file,caller_pid,TRUE))
                    { /* passed scanner: grant */
                      return(GRANTED);
                    }
                  else
                    return(NOT_GRANTED);

                /* all other cases are undefined */
                default:
                  return(DO_NOT_CARE);
              }

        case R_MODIFY_ATTRIBUTE:
            switch(attr)
              {
                case A_ms_backbuf:
                case A_ms_buflen:
                case A_ms_str_nr:
                case A_ms_str_offset:
                  return(NOT_GRANTED);
                case A_ms_scanned:
                case A_ms_trusted:
                case A_system_role:
                case A_ms_role:
                case A_ms_need_scan:
                #ifdef CONFIG_RSBAC_MS_AUTH_PROT
                case A_auth_may_setuid:
                case A_auth_may_set_cap:
                case A_auth_add_f_cap:
                case A_auth_remove_f_cap:
                #endif
                /* All attributes (remove target!) */
                case A_none:
                  /* Security Officer? */
                  i_tid.user = owner;
                  if (rsbac_get_attr(MS,
                                     T_USER,
                                     i_tid,
                                     A_ms_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request_ms(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* if sec_officer, then grant */
                  if (i_attr_val1.system_role == SR_security_officer)
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                default:
                  return(DO_NOT_CARE);
              }

        case R_READ_ATTRIBUTE:
            switch(attr)
              {
                /* every user may see scan status of files and sockets */
                case A_ms_scanned:
                  return(GRANTED);
                /* ...but only security officers may see other attributes */
                case A_system_role:
                case A_ms_role:
                case A_ms_trusted:
                case A_ms_backbuf:
                case A_ms_buflen:
                case A_ms_str_nr:
                case A_ms_str_offset:
                  /* Security Officer? */
                  i_tid.user = owner;
                  if (rsbac_get_attr(MS,
                                     T_USER,
                                     i_tid,
                                     A_ms_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request_ms(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* if sec_officer, then grant */
                  if (i_attr_val1.system_role == SR_security_officer)
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                default:
                  return(DO_NOT_CARE);
              }

#ifdef CONFIG_RSBAC_MS_SOCK
        case R_READ:
            switch(target)
              {
                case T_NETOBJ:
                  if(attr != A_ms_segment)
                    return(DO_NOT_CARE);
                  if(!tid.netobj.sock_p)
                    return(DO_NOT_CARE);
                  /* Always allow, if no scanning engine */
                  return(do_scan_sock(tid,
                                      caller_pid,
                                      attr_val.ms_segment));

                default:
                  return(DO_NOT_CARE);
              }
#endif

        case R_READ_OPEN:
        case R_READ_WRITE_OPEN:
            switch(target)
              {
                case T_FILE:
                  if(   (attr == A_new_object)
                     && (attr_val.new_object)
                    )
                    return DO_NOT_CARE;
                  /* scan file as non-execute */
                  if (scan(tid.file, caller_pid, FALSE))
                    { /* passed scanner: grant */
                      return(GRANTED);
                    }
                  else
                    return(NOT_GRANTED);

                /* all other cases */
                default:
                  return(DO_NOT_CARE);
              }

        case R_SWITCH_MODULE:
            switch(target)
              {
                case T_NONE:
                  /* we need the switch_target */
                  if(attr != A_switch_target)
                    return(UNDEFINED);
                  /* do not care for other modules */
                  if(   (attr_val.switch_target != MS)
                     #ifdef CONFIG_RSBAC_SOFTMODE
                     && (attr_val.switch_target != SOFTMODE)
                     #endif
                    )
                    return(DO_NOT_CARE);
                  /* test owner's ms_role */
                  i_tid.user = owner;
                  if (rsbac_get_attr(MS,
                                     T_USER,
                                     i_tid,
                                     A_ms_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING "rsbac_adf_request_ms(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* security officer? -> grant  */
                  if (i_attr_val1.system_role == SR_security_officer)
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                /* all other cases are undefined */
                default: return(DO_NOT_CARE);
              }
              
/*********************/
        default: return DO_NOT_CARE;
      }

    return(DO_NOT_CARE);
  }; /* end of rsbac_adf_request_ms() */


/*****************************************************************************/
/* If the request returned granted and the operation is performed,           */
/* the following function can be called by the AEF to get all aci set        */
/* correctly. For write accesses that are performed fully within the kernel, */
/* this is usually not done to prevent extra calls, including R_CLOSE for    */
/* cleaning up. Because of this, the write boundary is not adjusted - there  */
/* is no user-level writing anyway...                                        */
/* The second instance of target specification is the new target, if one has */
/* been created, otherwise its values are ignored.                           */
/* On success, 0 is returned, and an error from rsbac/error.h otherwise.     */

int  rsbac_adf_set_attr_ms(
                      enum  rsbac_adf_request_t     request,
                            rsbac_pid_t             caller_pid,
                      enum  rsbac_target_t          target,
                      union rsbac_target_id_t       tid,
                      enum  rsbac_target_t          new_target,
                      union rsbac_target_id_t       new_tid,
                      enum  rsbac_attribute_t       attr,
                      union rsbac_attribute_value_t attr_val,
                            rsbac_uid_t             owner)
  {
    union rsbac_target_id_t       i_tid;
    union rsbac_attribute_value_t i_attr_val1;
    union rsbac_attribute_value_t i_attr_val2;
    union rsbac_attribute_value_t i_attr_val3;

    switch (request)
      {
        case R_APPEND_OPEN:
        case R_READ_WRITE_OPEN:
            switch(target)
              {
                case T_FILE:
                  reset_scanned(tid.file);

                /* all other cases */
                default:
                  return(0);
              }

        case R_CLONE:
            if (target == T_PROCESS)
              {
                /* Get ms_trusted from first process */
                if (rsbac_get_attr(MS,
                                   T_PROCESS,
                                   tid,
                                   A_ms_trusted,
                                   &i_attr_val1,
                                   FALSE))
                  {
                    printk(KERN_WARNING
                           "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                    return(-RSBAC_EREADFAILED);
                  }
                /* Get ms_sock_trusted_tcp from first process */
                if (rsbac_get_attr(MS,
                                   T_PROCESS,
                                   tid,
                                   A_ms_sock_trusted_tcp,
                                   &i_attr_val2,
                                   FALSE))
                  {
                    printk(KERN_WARNING
                           "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                    return(-RSBAC_EREADFAILED);
                  }
                /* Get ms_sock_trusted_udp from first process */
                if (rsbac_get_attr(MS,
                                   T_PROCESS,
                                   tid,
                                   A_ms_sock_trusted_udp,
                                   &i_attr_val3,
                                   FALSE))
                  {
                    printk(KERN_WARNING
                           "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                    return(-RSBAC_EREADFAILED);
                  }
                /* Set ms_trusted for new process, if set for first */
                if (   i_attr_val1.ms_trusted
                    && (rsbac_set_attr(MS,
                                       T_PROCESS,
                                       new_tid,
                                       A_ms_trusted,
                                       i_attr_val1)) )
                  {
                    printk(KERN_WARNING "rsbac_adf_set_attr_ms(): rsbac_set_attr() returned error!\n");
                    return(-RSBAC_EWRITEFAILED);
                  }
                /* Set ms_sock_trusted_tcp for new process, if set for first */
                if (   i_attr_val2.ms_sock_trusted_tcp
                    && (rsbac_set_attr(MS,
                                       T_PROCESS,
                                       new_tid,
                                       A_ms_sock_trusted_tcp,
                                       i_attr_val2)) )
                  {
                    printk(KERN_WARNING "rsbac_adf_set_attr_ms(): rsbac_set_attr() returned error!\n");
                    return(-RSBAC_EWRITEFAILED);
                  }
                /* Set ms_sock_trusted_udp for new process, if set for first */
                if (   i_attr_val3.ms_sock_trusted_udp
                    && (rsbac_set_attr(MS,
                                       T_PROCESS,
                                       new_tid,
                                       A_ms_sock_trusted_udp,
                                       i_attr_val3)) )
                  {
                    printk(KERN_WARNING "rsbac_adf_set_attr_ms(): rsbac_set_attr() returned error!\n");
                    return(-RSBAC_EWRITEFAILED);
                  }
                return(0);
              }
            else
              return(0);

        case R_EXECUTE:
            switch(target)
              {
                case T_FILE:
                  /* get ms_trusted for file */
                  if (rsbac_get_attr(MS,
                                     T_FILE,
                                     tid,
                                     A_ms_trusted,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                      return(-RSBAC_EREADFAILED);
                    }
                  /* get for process */
                  i_tid.process = caller_pid;
                  if (rsbac_get_attr(MS,
                                     T_PROCESS,
                                     i_tid,
                                     A_ms_trusted,
                                     &i_attr_val2,
                                     FALSE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                      return(-RSBAC_EREADFAILED);
                    }
                  /* and set for process, if different */
                  #ifdef CONFIG_RSBAC_MS_PROP_TRUSTED
                  if(i_attr_val1.ms_trusted > i_attr_val2.ms_trusted)
                  #else
                  if(i_attr_val1.ms_trusted != i_attr_val2.ms_trusted)
                  #endif
                    if (rsbac_set_attr(MS,
                                       T_PROCESS,
                                       i_tid,
                                       A_ms_trusted,
                                       i_attr_val1))
                      {
                        printk(KERN_WARNING "rsbac_adf_set_attr_ms(): rsbac_set_attr() returned error!\n");
                        return(-RSBAC_EWRITEFAILED);
                      }

                  /* get ms_sock_trusted_tcp for file */
                  if (rsbac_get_attr(MS,
                                     T_FILE,
                                     tid,
                                     A_ms_sock_trusted_tcp,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                      return(-RSBAC_EREADFAILED);
                    }
                  /* get for process */
                  i_tid.process = caller_pid;
                  if (rsbac_get_attr(MS,
                                     T_PROCESS,
                                     i_tid,
                                     A_ms_sock_trusted_tcp,
                                     &i_attr_val2,
                                     FALSE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                      return(-RSBAC_EREADFAILED);
                    }
                  /* and set for process, if different */
                  if(i_attr_val1.ms_sock_trusted_tcp 
                      != i_attr_val2.ms_sock_trusted_tcp)
                    if (rsbac_set_attr(MS,
                                       T_PROCESS,
                                       i_tid,
                                       A_ms_sock_trusted_tcp,
                                       i_attr_val1))
                      {
                        printk(KERN_WARNING "rsbac_adf_set_attr_ms(): rsbac_set_attr() returned error!\n");
                        return(-RSBAC_EWRITEFAILED);
                      }

                  /* get ms_sock_trusted_udp for file */
                  if (rsbac_get_attr(MS,
                                     T_FILE,
                                     tid,
                                     A_ms_sock_trusted_udp,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                      return(-RSBAC_EREADFAILED);
                    }
                  /* get for process */
                  i_tid.process = caller_pid;
                  if (rsbac_get_attr(MS,
                                     T_PROCESS,
                                     i_tid,
                                     A_ms_sock_trusted_udp,
                                     &i_attr_val2,
                                     FALSE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_set_attr_ms(): rsbac_get_attr() returned error!\n");
                      return(-RSBAC_EREADFAILED);
                    }
                  /* and set for process, if different */
                  if(i_attr_val1.ms_sock_trusted_udp 
                      != i_attr_val2.ms_sock_trusted_udp)
                    if (rsbac_set_attr(MS,
                                       T_PROCESS,
                                       i_tid,
                                       A_ms_sock_trusted_udp,
                                       i_attr_val1))
                      {
                        printk(KERN_WARNING "rsbac_adf_set_attr_ms(): rsbac_set_attr() returned error!\n");
                        return(-RSBAC_EWRITEFAILED);
                      }
                  return(0);

                /* all other cases */
                default:
                  return(0);
              }

        case R_WRITE_OPEN:
            switch(target)
              {
                case T_FILE:
                  reset_scanned(tid.file);
                  return(0);

                /* all other cases */
                default:
                  return(0);
              }


/*********************/
        default: return(0);
      }

    return(0);
  }; /* end of rsbac_adf_set_attr_ms() */

/* end of rsbac/adf/ms/main.c */
