/*
 * driver/usb/usb.c
 *
 * (C) Copyright Linus Torvalds 1999
 *
 * NOTE! This is not actually a driver at all, rather this is
 * just a collection of helper routines that implement the
 * generic USB things that the real drivers can use..
 *
 * Think of this as a "USB library" rather than anything else.
 * It should be considered a slave, with no callbacks. Callbacks
 * are evil.
 */

#include <linux/string.h>
#include <linux/bitops.h>
#include <linux/malloc.h>

#include "usb.h"

/*
 * We have a per-interface "registered driver" list.
 */
static LIST_HEAD(usb_driver_list);

int usb_register(struct usb_driver *new_driver)
{
	/* Add it to the list of known drivers */
	list_add(&new_driver->driver_list, &usb_driver_list);

	/*
	 * We should go through all existing devices, and see if any of
	 * them would be acceptable to the new driver.. Let's do that
	 * in version 2.0.
	 */
	return 0;
}

void usb_deregister(struct usb_driver *driver)
{
	list_del(&driver->driver_list);
}

/*
 * This entrypoint gets called for each new device.
 *
 * We now walk the list of registered USB drivers,
 * looking for one that will accept this device as
 * his..
 */
void usb_device_descriptor(struct usb_device *dev)
{
	struct list_head *tmp = usb_driver_list.next;

	while (tmp != &usb_driver_list) {
		struct usb_driver *driver = list_entry(tmp, struct usb_driver, driver_list);
		tmp = tmp->next;
		if (driver->probe(dev))
			continue;
		dev->driver = driver;
		return;
	}

	/*
	 * Ok, no driver accepted the device, so show the info
	 * for debugging..
	 */
	printk("Unknown new USB device:\n");
	usb_show_device(dev);
}

/*
 * Parse the fairly incomprehensible output of
 * the USB configuration data, and build up the
 * USB device database.
 */
static int usb_expect_descriptor(unsigned char *ptr, int len, unsigned short desc)
{
	int desclen = desc & 0xff;
	int parsed = 0;

	for (;;) {
		unsigned short n_desc;
		int n_len, i;

		if (len < desclen)
			return -1;
		n_desc = *(unsigned short *)ptr;
		if (n_desc == desc)
			return parsed;
		printk("Expected descriptor %04x, got %04x - skipping\n", desc, n_desc);
		n_len = n_desc & 0xff;
		if (n_len < 2 || n_len > len)
			return -1;
		for (i = 0 ; i < n_len; i++)
			printk("   %d %02x\n", i, ptr[i]);
		len -= n_len;
		ptr += n_len;
		parsed += n_len;
	}
}

static int usb_parse_endpoint(struct usb_endpoint_descriptor *endpoint, unsigned char *ptr, int len)
{
	int parsed = usb_expect_descriptor(ptr, len, USB_ENDPOINT_DESC | 7);

	if (parsed < 0)
		return parsed;

	memcpy(endpoint, ptr + parsed, 7);
	return parsed + 7;
}

static int usb_parse_interface(struct usb_interface_descriptor *interface, unsigned char *ptr, int len)
{
	int i;
	int parsed = usb_expect_descriptor(ptr, len, USB_INTERFACE_DESC | 9);

	if (parsed < 0)
		return parsed;

	memcpy(interface, ptr + parsed, 9);
	len -= 9;
	parsed += 9;

	if (interface->bNumEndpoints > USB_MAXENDPOINTS)
		return -1;

	for (i = 0; i < interface->bNumEndpoints; i++) {
		int retval = usb_parse_endpoint(interface->endpoint + i, ptr + parsed, len);
		if (retval < 0)
			return retval;
		parsed += retval;
		len -= retval;
	}
	return parsed;
}

static int usb_parse_config(struct usb_config_descriptor *config, unsigned char *ptr, int len)
{
	int i;
	int parsed = usb_expect_descriptor(ptr, len, USB_CONFIG_DESC | 9);

	if (parsed < 0)
		return parsed;

	memcpy(config, ptr + parsed, 9);
	len -= 9;
	parsed += 9;

	if (config->bNumInterfaces > USB_MAXINTERFACES)
		return -1;

	for (i = 0; i < config->bNumInterfaces; i++) {
		int retval = usb_parse_interface(config->interface + i, ptr + parsed, len);
		if (retval < 0)
			return retval;
		parsed += retval;
		len -= retval;
	}
	return parsed;
}

int usb_parse_configuration(struct usb_device *dev, void *__buf, int bytes)
{
	int i;
	unsigned char *ptr = __buf;

	if (dev->descriptor.bNumConfigurations > USB_MAXCONFIG)
		return -1;

	for (i = 0; i < dev->descriptor.bNumConfigurations; i++) {
		int retval = usb_parse_config(dev->config + i, ptr, bytes);
		if (retval < 0)
			return retval;
		ptr += retval;
		bytes += retval;
	}
	return 0;
}

void usb_init_root_hub(struct usb_device *dev)
{
	dev->devnum = -1;
	dev->slow = 0;
}

/*
 * Something got disconnected. Get rid of it, and all of its children.
 */
void usb_disconnect(struct usb_device **pdev)
{
	struct usb_device * dev = *pdev;

	if (dev) {
		int i;

		*pdev = NULL;

		printk("USB disconnect on device %d\n", dev->devnum);
		/* Free up all the children.. */
		for (i = 0; i < USB_MAXCHILDREN; i++) {
			struct usb_device **child = dev->children + i;
			usb_disconnect(child);
		}

		/* Free up the device itself, including its device number */
		if (dev->devnum > 0)
			clear_bit(dev->devnum, &dev->devmap->devicemap);
		dev->op->deallocate(dev);
	}
}


/*
 * Connect a new USB device. This basically just initializes
 * the USB device information and sets up the topology - it's
 * up to the low-level driver to reset the port and actually
 * do the setup (the upper levels don't know how to do that).
 */
void usb_connect(struct usb_device *dev, struct usb_devmap *devmap)
{
	int devnum;

	memset(dev, 0, sizeof(*dev));
	dev->descriptor.bMaxPacketSize0 = 8;
	dev->devmap = devmap;

	devnum = find_next_zero_bit(devmap->devicemap, 128, 1);
	if (devnum < 128) {
		set_bit(devnum, devmap->devicemap);
		dev->devnum = devnum;
	}
}

/*
 * These are the actual routines to send
 * and receive control messages.
 */
int usb_set_address(struct usb_device *dev)
{
	unsigned short cmd[4];

	cmd[0] = SET_ADDRESS;
	cmd[1] = dev->devnum;
	cmd[2] = 0x0000;
	cmd[3] = 0x0000;

	return dev->op->control_msg(dev, usb_snddefctrl(dev), cmd, NULL, 0);
}

int usb_get_descriptor(struct usb_device *dev, unsigned short type, void *buf, int size)
{
	unsigned short cmd[4];

	cmd[0] = GET_DESCRIPTOR;
	cmd[1] = type;
	cmd[2] = 0x0000;			/* zero wIndex */
	cmd[3] = size;

	return dev->op->control_msg(dev, usb_rcvctrlpipe(dev,0), cmd, buf, size);
}

int usb_get_device_descriptor(struct usb_device *dev)
{
	return usb_get_descriptor(dev, USB_DEVICE_DESC, &dev->descriptor, sizeof(dev->descriptor));
}

int usb_get_hub_descriptor(struct usb_device *dev, void *data, int size)
{
	unsigned short cmd[4];

	cmd[0] = GET_HUB_DESCRIPTOR;
	cmd[1] = USB_HUB_DESC;
	cmd[2] = 0x0000;			/* zero wIndex */
	cmd[3] = size;

	return dev->op->control_msg(dev, usb_rcvctrlpipe(dev,0), cmd, data, size);
}

int usb_set_port_feature(struct usb_device *dev, int port, int feature)
{
	unsigned short cmd[4];

	cmd[0] = SET_PORT_FEATURE;
	cmd[1] = feature;			/* wValue: feature selector */
	cmd[2] =    port;			/* wIndex: port number */
	cmd[3] =  0x0000;			/* zero wLength */

	return dev->op->control_msg(dev, usb_sndctrlpipe(dev,0), cmd, NULL, 0);
}

int usb_get_port_status(struct usb_device *dev, int port, void *data)
{
	unsigned short cmd[4];

	cmd[0] = GET_PORT_STATUS;
	cmd[1] = 0x0000;			/* wValue = 0 */
	cmd[2] =   port;			/* wIndex = port */
	cmd[3] =      4;			/* wLength = 4 */

	return dev->op->control_msg(dev, usb_rcvctrlpipe(dev,0), cmd, data, 4);
}

int usb_get_protocol(struct usb_device *dev)
{
	unsigned short cmd[4];
	unsigned char buf[8];

	cmd[0] = GET_PROTOCOL;
	cmd[1] = 0x0;
	cmd[2] = 1;
	cmd[3] = 1;
	if (dev->op->control_msg(dev, usb_rcvctrlpipe(dev, 0), cmd, buf, 1))
		return -1;
	return buf[0];
}

int usb_set_protocol(struct usb_device *dev, int protocol)
{
	unsigned short cmd[4];

	cmd[0] = GET_PROTOCOL;
	cmd[1] = protocol;
	cmd[2] = 1;
	cmd[3] = 0;
	if (dev->op->control_msg(dev, usb_sndctrlpipe(dev, 0), cmd, NULL, 0))
		return -1;
	return 0;
}

int usb_set_idle(struct usb_device *dev)
{
	unsigned short cmd[4];

	cmd[0] = SET_IDLE;
	cmd[1] = 0;
	cmd[2] = 1;
	cmd[3] = 0;
	if (dev->op->control_msg(dev, usb_sndctrlpipe(dev, 0), cmd, NULL, 0))
		return -1;
	return 0;
}

int usb_set_configuration(struct usb_device *dev, int configuration)
{
	unsigned short cmd[4];

	cmd[0] = SET_CONFIGURATION;
	cmd[1] = configuration;
	cmd[2] = 0;
	cmd[3] = 0;
	if (dev->op->control_msg(dev, usb_sndctrlpipe(dev, 0), cmd, NULL, 0))
		return -1;
	return 0;
}

int usb_get_report(struct usb_device *dev)
{
	unsigned short cmd[4];
	unsigned char buf[8];

	cmd[0] = GET_REPORT;
	cmd[1] = 0x0100;
	cmd[2] = 1;
	cmd[3] = 3;
	if (dev->op->control_msg(dev, usb_rcvctrlpipe(dev, 0), cmd, buf, 3))
		return -1;
	return buf[0];
}

/*
 * Yee-haa, we found a hub.
 *
 * Let's look at it some more..
 */
void usb_new_hub(struct usb_device *dev)
{
	unsigned char hubdescriptor[8];

	if (usb_get_hub_descriptor(dev, hubdescriptor, 8))
		return;

	dev->maxchild = hubdescriptor[2];
	printk("%d-port Hub detected\n", dev->maxchild);

	/*
	 * Here we should add an interrupt TD to keep track of status
	 * changes etc..
	 */
}

int usb_get_configuration(struct usb_device *dev)
{
	unsigned int size;
	unsigned char buffer[128];

	/* Get the first 8 bytes - guaranteed */
	if (usb_get_descriptor(dev, USB_CONFIG_DESC, buffer, 8))
		return -1;

	/* Get the full buffer */
	size = *(unsigned short *)(buffer+2);
	if (size > sizeof(buffer))
		size = sizeof(buffer);

	if (usb_get_descriptor(dev, USB_CONFIG_DESC, buffer, size))
		return -1;

	return usb_parse_configuration(dev, buffer, size);
}

/*
 * By the time we get here, the device has gotten a new device ID
 * and is in the default state. We need to identify the thing and
 * get the ball rolling..
 */
void usb_new_device(struct usb_device *dev)
{
	printk("USB new device connect, device number %d\n", dev->devnum);
	if (usb_set_address(dev))
		return;
	if (usb_get_device_descriptor(dev))
		return;
	if (usb_get_configuration(dev))
		return;

	usb_device_descriptor(dev);

	/* Do we have a hub? */
	if (dev->descriptor.bDeviceClass == USB_HUB_CLASS)
		usb_new_hub(dev);
}

int usb_request_irq(struct usb_device *dev, unsigned int pipe, int (*handler)(int, void *), int period)
{
	return dev->op->request_irq(dev, pipe, handler, period);
}
