/* sonetdiag.h - SONET diagnostics */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <linux/atm.h>
#include <linux/sonet.h>


struct opts {
    const char *name;
    int value;
} options[] = {
    { "sbip", SONET_INS_SBIP  }, { "lbip", SONET_INS_LBIP  },
    { "pbip", SONET_INS_PBIP  }, { "frame",SONET_INS_FRAME },
    { "los",  SONET_INS_LOS   }, { "lais", SONET_INS_LAIS  },
    { "pais", SONET_INS_PAIS  }, { "hcs",  SONET_INS_HCS   },
    { NULL,   0 }
};


int main(int argc,char **argv)
{
    struct sockaddr_atmpvc addr;
    struct sonet_stats stats;
    struct opts *walk;
    char *opt;
    int zero,s,set,clear,error,minus;

    if (argc < 2 || ((zero = !strcmp(argv[1],"-z")) && argc < 3 )) {
	fprintf(stderr,"usage: %s [ -z ] itf [ [-]error ...]\n",argv[0]);
	return 1;
    }
    if (zero) {
	argc--;
	argv++;
    }
    if ((s = socket(PF_ATMPVC,SOCK_DGRAM,ATM_AAL5)) < 0) {
	perror("socket");
	return 1;
    }
    memset(&addr,0,sizeof(addr));
    addr.sap_family = AF_ATMPVC;
    addr.sap_addr.itf = atoi(argv[1]);
    argv += 2;
    argc -= 2;
    set = clear = error = 0;
    while (argc--) {
	if (minus = *(opt = *argv++) == '-') opt++;
	for (walk = options; walk->name; walk++)
	    if (!strcmp(walk->name,opt)) break;
	if (walk->name)
	    if (minus) clear |= walk->value;
	    else set |= walk->value;
	else {
	    fprintf(stderr,"unrecognized option: %s\n",opt);
	    error = 1;
	}
    }
    if (error) return 1;
    if (connect(s,(struct sockaddr *) &addr,sizeof(addr)) < 0) {
        perror("bind");
        return 1;
    }
    if (!set && !clear) {
	if (ioctl(s,zero ? SONET_GETSTATZ : SONET_GETSTAT,&stats) < 0) {
	    perror(zero ? "ioctl SONET_GETSTATZ" : "ioctl SONET_GETSTAT");
	    return 1;
	}
	if (ioctl(s,SONET_GETDIAG,&set) < 0)
	    perror("ioctl SONET_GETDIAG");
	printf("Section BIP errors: %10ld\n",stats.section_bip);
	printf("Line BIP errors:    %10ld\n",stats.line_bip);
	printf("Path BIP errors:    %10ld\n",stats.path_bip);
	printf("Line FEBE:          %10ld\n",stats.line_febe);
	printf("Path FEBE:          %10ld\n",stats.path_febe);
	printf("Correctable HCS:    %10ld\n",stats.corr_hcs);
	printf("Uncorrectable HCS:  %10ld\n",stats.uncorr_hcs);
	printf("TX cells:           %10ld\n",stats.tx_cells);
	printf("RX cells:           %10ld\n",stats.rx_cells);
	if (set) {
	    int i;

	    printf("\nDiagnostics:");
	    for (i = 0; options[i].name; i++)
		if (set & options[i].value) printf(" %s",options[i].name);
	    putchar('\n');
	}
    }
    else {
	if (set) {
	    if (ioctl(s,SONET_SETDIAG,&set) < 0)
		perror("ioctl SONET_SETDIAG");
	}
	if (clear) {
	    if (ioctl(s,SONET_CLRDIAG,&clear) < 0)
		perror("ioctl SONET_SETDIAG");
	}
    }
    return 0;
}
